// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
// Copyright (C) 1989 University of Colorado, Boulder, Colorado
// Copyright (C) 1990 University of Colorado, Boulder, Colorado
//
// written by Dirk Grunwald (grunwald@foobar.colorado.edu)
//
#ifndef SpinBarrier_h
#define SpinBarrier_h
#pragma once

//
//	Implement a UNIX process barrier
//

#include <SpinLock.h>
#include <AwesimeConfig.h>
#include <assert.h>
#include <Pragma.h>

class SpinBarrier : SpinLock {
    short pHeight;
    VOLATILE short pCount;
    VOLATILE short generation;
    short pLoops;
    int pMaxLoops;

public:
    SpinBarrier( int h, int loops = DefaultSpinBarrierLoops, int max = 0 );
    ~SpinBarrier();

    //
    // Normal entry. Everyone calls rendezvous()
    //
    int rendezvous();
    //
    // lowerBarrier forces all waiting jobs to continue
    //
    void lowerBarrier();

    //
    // Changing the height of the barrier can cause people to continue
    // if it is lowered below the current pCount.
    //
    int height();
    void height(int h);

    //
    // loops() is the numbr of times a barrier spins before doing a getpid()
    // to release the UNIX cpu. maxLoopss() the maximum number of getpids()
    // to do before returning from the barrier (i.e. for barrier with timeout)
    //
    int loops();
    void loops(int h);
    int maxLoops();
    void maxLoops(int h);

    int count();
};

inline
SpinBarrier::SpinBarrier( int h, int l, int m )
{
    pHeight = h;
    pLoops = l;
    pMaxLoops = m;
    pCount = 0;
}

inline int
SpinBarrier::height()
{
    return(pHeight);
}

inline int
SpinBarrier::loops()
{
    return(pLoops);
}

inline void
SpinBarrier::loops(int h)
{
    pLoops = h;
}

inline int
SpinBarrier::maxLoops()
{
    return(pMaxLoops);
}

inline void
SpinBarrier::maxLoops(int h)
{
    pMaxLoops = h;
}

inline int
SpinBarrier::count()
{
    return(pCount);
}

inline
SpinBarrier::~SpinBarrier()
{
    assert( pCount == 0 );
}

#ifndef USE_SHARED_MEMORY
inline int
SpinBarrier::rendezvous()
{
    assert( (pCount+1) == pHeight );
    generation++;
    pCount = 0;
    return(1);
}

inline void
SpinBarrier::lowerBarrier()
{
    generation++;
    pCount = 0;
}
#endif /* USE_SHARED_MEMORY */

#endif /* SpinBarrier */
