/* Scheme implementation intended for JACAL.
   Copyright (C) 1989, 1990 Aubrey Jaffer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

The author can be reached at jaffer@ai.mit.edu or
Aubrey Jaffer, 84 Pleasant St., Wakefield MA 01880
*/

#include <stdio.h>
#include <signal.h>
#include <setjmp.h>
#include "scm.h"

#ifdef unix
#include <sys/types.h>
#include <sys/times.h>
struct tms time_buffer;
#define TIME_IN_MSEC(x) (((x)*50)/3)
long myruntime()
{
	times(&time_buffer);
	return time_buffer.tms_utime;
}
#else
#ifdef vms
#include <time.h>
#define TIME_IN_MSEC(x) ((x)*10)
long myruntime()
{
	return clock();
}
#else
#ifdef MSDOS
#include <stdlib.h>
#include <sys\types.h>
#include <sys\timeb.h>
#include <time.h>
#define TIME_IN_MSEC(x) ((x))
struct timeb time_buffer;
long myruntime()
{
	ftime(&time_buffer);
	return (long) time_buffer.time*1000 + time_buffer.millitm;
}
#else
#include <stdlib.h>
#define TIME_IN_MSEC(x) ((x)*1000)
long myruntime()
{
	long x;
	long time();
	time(&x);
	return (long) x;
}
#endif
#endif
#endif

jmp_buf errjmp;
int errjmp_ok = 0, sig_disabled = 1,  sig_deferred = 0;
SCM s_errobj = BOOL_F;
SCM s_load;
long line_num = 1;
char *load_name = NULL;
char filextension[] = FILE_EXTENSION;

#ifndef verbose
int verbose = 0;
#endif
long gc_cells_allocated = 0, gc_rt, gc_time_taken;
long gc_cells_collected, gc_malloc_collected, gc_ports_collected;

char upcase[char_code_limit];
char downcase[char_code_limit];
char lowers[]="abcdefghijklmnopqrstuvwxyz";
char uppers[]="ABCDEFGHIJKLMNOPQRSTUVWXYZ";
init_tables()
{
	int i;
	for(i=0;i<char_code_limit;i++) upcase[i]=downcase[i]=i;
	for(i=0;i<26;i++) {
		upcase[lowers[i]]=uppers[i];
		downcase[uppers[i]]=lowers[i];
	}
}

main(argc,argv)
int argc; 
char **argv;
{
  if (argc <= 1) {
#ifndef verbose
    verbose=1;
#endif
    puts("SCM version 1.2, Copyright (C) 1989, 1990 Aubrey Jaffer.");
    puts("SCM comes with ABSOLUTELY NO WARRANTY; for details type `(warranty)'.");
    puts("This is free software, and you are welcome to redistribute it");
    puts("under certain conditions; type `(terms)' for details.");
  }
  init_tables();
  init_storage();
  init_subrs();
  init_eval();
  init_io();
  init_scl();
  init_scm();
  if (verbose)
  {
    fputs(";heap_size = ",stdout);
    iprint(heap_size,10,stdout);
    fputs(" cells, ",stdout);
    iprint(heap_size*sizeof(cell),10,stdout);
    puts(" bytes.");
  }
  repl_driver(argc,argv,1);
  if (verbose) puts(";EXIT");
  exit(0);
}

#ifdef unix
void handle_sigbus(sig)
int sig;
{
	signal(SIGBUS,handle_sigbus);
	err("bus error",BOOL_F);
}
#endif
void handle_sigsegv(sig)
int sig;
{
	signal(SIGSEGV,handle_sigsegv);
	err("segment violation",BOOL_F);
}
void handle_sigint(sig)
int sig;
{
	signal(SIGINT,handle_sigint);
	if (sig_disabled) sig_deferred = 1;
	else err_ctrl_c();
}

repl_driver(argc,argv,want_sigint)
int argc;
char **argv;
int want_sigint;
{
	long i;
	stack_start_ptr = &i;
	switch (setjmp(errjmp)) {
	case 2: return;
	case 0:
		if (want_sigint) signal(SIGINT,handle_sigint);
#ifdef unix
		signal(SIGBUS,handle_sigbus);
#endif
		signal(SIGSEGV,handle_sigsegv);
		errjmp_ok = 1;
		sig_deferred = 0;
		sig_disabled = 0;
		for(i=1;i<argc;++i)if(argv[i][0] != '-') iload(argv[i]);
	default:
		errjmp_ok = 1;
		sig_deferred = 0;
		sig_disabled = 0;
		load_name = NULL;
		repl();
	}
}

growth_mon(obj, size, units)
char *obj;
long size;
char *units;
{
	if (!verbose) return;
	DEFER_SIGINT;
	fputs("; grew ",stdout);
	fputs(obj,stdout);
	fputs(" to ",stdout);
	iprint(size,10,stdout);
	putc(' ',stdout);
	puts(units);
	ALLOW_SIGINT;
}

gc_ms_stats_start()
{
	gc_rt = myruntime();
	gc_cells_collected = 0;
	gc_malloc_collected = 0;
	gc_ports_collected = 0;
	if (!verbose) return;
	fputs(";GC ",stdout);
}
gc_ms_stats_end()
{
	gc_rt = myruntime() - gc_rt;
	gc_time_taken = gc_time_taken + gc_rt;
	if (verbose) {
		iprint(TIME_IN_MSEC(gc_rt),10,stdout);
		fputs(" cpu mSec, ",stdout);
		iprint(gc_cells_collected,10,stdout);
		fputs(" cells, ",stdout);
		iprint(gc_malloc_collected,10,stdout);
		fputs(" malloc, ",stdout);
		iprint(gc_ports_collected,10,stdout);
		puts(" ports collected");
	}
}

SCM gc_status(args)
SCM args;
{
  register SCM l;
  register long n;
  DEFER_SIGINT;
  if NNULLP(args)
    if FALSEP(CAR(args)) verbose = 0;
    else verbose = 1;
  if (verbose) puts(";garbage collection verbose");
  else
    puts(";garbage collection silent");
  {
    for(n=0,l=freelist;NNULLP(l); ++n) l = CDR(l);
    putc(';',stdout);
    iprint(heap_size-n,10,stdout);
    fputs(" allocated ",stdout);
    iprint(n,10,stdout);
    puts(" free");
  }
  ALLOW_SIGINT;
  return BOOL_F;
}

repl()
{
  SCM x;
  long rt;
  DEFER_SIGINT;
  while(1) {
    fputs("> ",stdout);
    fflush(stdout);
    ALLOW_SIGINT;
    x = lread(cur_inp);
    if (x == EOF_VAL) break;
    rt = myruntime();
    gc_cells_allocated = 0;
    gc_time_taken = 0;
    x = EVAL(x,EOL);
    {
      DEFER_SIGINT;
      fputs(";Evaluation took ",stdout);
      iprint(TIME_IN_MSEC(myruntime()-rt),10,stdout);
      fputs(" mSec (",stdout);
      iprint(TIME_IN_MSEC(gc_time_taken),10,stdout);
      fputs(" in gc) ",stdout);
      iprint(gc_cells_allocated,10,stdout);
      puts(" cons work");
    }
    lprin1f(x,stdout,1);
    putc('\n',stdout);
  }

}
SCM quit()
{
	longjmp(errjmp,2);
}

err_head()
{
	fflush(stdout);
	putc('\n',stderr);
	if(load_name) {
		putc('"',stderr);
		fputs(load_name,stderr);
		fputs("\", line ",stderr);
		iprint((long)line_num,10,stderr);
		fputs(": ",stderr);
	}
}
wta(arg,pos,s_subr)
SCM arg,s_subr;
long pos;
{
	DEFER_SIGINT;
	err_head();
	fputs("ERROR: ",stderr);
	switch (pos) {
	case FUN:
		fputs("Wrong type to apply ",stderr);
		goto eobj;
	case ARG1:
	case ARG2:
	case ARG3:
		fputs("Wrong type in arg ",stderr);
		iprint(pos,10,stderr);
		fputs(" to ",stderr);
eputsname:
		fputs(ISYMP(s_subr)?ISYMCHARS(s_subr):CHARS(SNAME(s_subr)),
		      stderr);
		goto eobj;
	case WNA:
		fputs("Wrong number of args to ",stderr);
		goto putsname;
	case OUTOFRANGE:
		fputs("Argument out of range to ",stderr);
		goto eputsname;
	case OVERFLOW:
		fputs("numerical overflow in ",stderr);
		goto putsname;
	case NALLOC:
		fputs("could not allocate ",stderr);
		fputs((char *)s_subr,stderr);
		putc(' ',stderr);
		iprint(INUM(arg),10,stderr);
		goto newline;
	case NOFILE:
		perror((char *)arg);
		fputs("could not open file in ",stderr);
putsname:
		fputs(ISYMP(s_subr)?ISYMCHARS(s_subr):CHARS(SNAME(s_subr)),
		      stderr);
		goto newline;
	case ENDFILE:
		fputs("end of file in ",stderr);
		fputs((char *)arg,stderr);
		goto newline;
	default:
		fputs((char *)pos,stderr);
		if NFALSEP(arg)
eobj:
			fputs(": (see errobj)",stderr);
newline:
		putc('\n',stderr);
		break;
	}
	fflush(stderr);
	if (errjmp_ok == 1) {
		VCELL(s_errobj)=arg;
		longjmp(errjmp,1);
	}
	lprin1f(arg,stderr,1);
	putc('\n',stderr);
	fputs("FATAL ERROR DURING CRITICAL CODE SECTION\n",stderr);
	exit(1);
}
err(message,x)
char *message;
SCM x;
{
	wta(x,message,BOOL_F);
}
err_ctrl_c()
{
	err("control-c interrupt",BOOL_F);
}

SCM iiopen(filename)
char *filename;
{
	FILE *f = fopen(filename,"r");
	if (!f) {
		if (filextension[0]) {
			char buf[64];
			register int i,j;
			for(i=0;filename[i];i++) buf[i] = filename[i];
			for(j=0;filextension[j];j++)buf[i++] = filextension[j];
			buf[i] = 0;
			f = fopen(buf,"r");
		}
		if (!f) wta(filename,NOFILE,s_load);
	}
	return makport(f,tc_inport);
}
iload(filename)
char *filename;
{
	char *olonam = load_name;
	register long olninum = line_num;
	SCM form,port = iiopen(filename);
	load_name = filename;
	line_num = 1;
	DEFER_SIGINT;
	fputs(";loading ",stdout);
	fputs(filename,stdout);
	ALLOW_SIGINT;
	while(1) {
		form = lread(port);
		if (EOF_VAL == form) break;
		SIDEVAL(form,EOL);
	}
	close_port(port);
	DEFER_SIGINT;
	fputs(" ;done loading ",stdout);
	ALLOW_SIGINT;
	puts(filename);
	line_num = olninum;
	load_name = olonam;
}
SCM load(filename)
SCM filename;
{
	ASSERT(NIMP(filename) && STRINGP(filename),filename,ARG1,s_load);
	iload(CHARS(filename));
	return UNSPECIFIED;
}
icat(in,out)
SCM in, out;
{
  FILE *fin = STREAM(in);
  FILE *fout = STREAM(out);
  int c;
  DEFER_SIGINT;
  while (EOF != (c = getc(fin))) {
    CHECK_SIGINT;
    putc(c, fout);
  }
  ALLOW_SIGINT;
}
SCM warranty()
{
  SCM port = iiopen(WARRANTY);
  icat(port,cur_outp);
  return close_port(port);
}
list_file(file)
SCM file;
{
  SCM port = open_input_file(file);
  icat(port,cur_outp);
  return close_port(port);
}
/* This is cfib, for compiled fib. Test to see what the overhead
   of interpretation actually is in a given implementation
;(define (fib x)
;  (if (< x 2)
;      x
;    (+ (fib (- x 1))
;       (fib (- x 2))))) */

SCM cfib(x)
SCM x;
{
  if NFALSEP(lessp(x,MAKINUM(2L),EOL)) return(x);
  else return(sum(cfib(difference(x,MAKINUM(1L))),
		  cfib(difference(x,MAKINUM(2L)))));
}

#ifdef vms
#include <descrip.h>
#include <ssdef.h>
long no_interrupt();
SCM s_sys_edit;
SCM sys_edit(fname)
SCM fname;
{
	struct dsc$descriptor_s d;
	long iflag;
	ASSERT(STRINGP(fname),fname,ARG1,s_sys_edit);
	d.dsc$b_dtype = DSC$K_DTYPE_T;
	d.dsc$b_class = DSC$K_CLASS_S;
	d.dsc$w_length = LENGTH(fname);
	d.dsc$a_pointer = CHARS(fname);
	iflag = no_interrupt(1);
	edt$edit(&d);
	no_interrupt(iflag);
	return(fname);
}

SCM vms_debug(v)
SCM v;
{
	lib$signal(SS$_DEBUG);
	return(v);
}

#endif

init_scm()
{
	s_errobj=intern("errobj",-6L);
	VCELL(s_errobj)=BOOL_F;
	init_subr("quit",tc6_subr_0,quit);
	s_load=init_subr("load",tc6_subr_1,load);
	init_subr("list-file",tc6_subr_1,list_file);
	init_subr("warranty",tc6_subr_0,warranty);
	init_subr("terms",tc6_subr_0,warranty);
	init_subr("verbose",tc6_lsubr,gc_status);
	init_subr("cfib",tc6_subr_1,cfib);
#ifdef vms
	s_sys_edit = init_subr("edit",tc6_subr_1,sys_edit);
	init_subr("vms-debug",tc6_subr_1,vms_debug);
#endif
}
