/* Scheme implementation intended for JACAL.
   Copyright (C) 1990, 1991 Aubrey Jaffer.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

The author can be reached at jaffer@ai.mit.edu or
Aubrey Jaffer, 84 Pleasant St., Wakefield MA 01880
*/

#include <stdio.h>
#ifdef vms
# ifndef CHEAP_CONTINUATIONS
#  include "setjump.h"
# else
#  include <setjmp.h>
# endif
#else
# include <setjmp.h>
#endif
#include "scm.h"

#ifdef MSDOS
# include <stdlib.h>
# include <sys\types.h>
# include <sys\timeb.h>
# include <time.h>
# define HAVE_CLOCK
# define HAVE_FTIME
#else
# ifdef unix
#  include <sys/types.h>
#  include <sys/times.h>
#  include <sys/timeb.h>
#  ifndef CLK_TCK
#   define CLK_TCK (60)
#  endif
#  define HAVE_TIMES
#  define HAVE_FTIME
# else
#  include <time.h>
#  ifdef vms
#   ifndef CLK_TCK
#    define CLK_TCK (100)
#   endif
#   define HAVE_CLOCK
#   define HAVE_FTIME
#  else
#   ifdef __STDC__
#    define HAVE_CLOCK
#   endif /* __STDC__ */
#  endif /* !vms */
# endif /* !unix */
#endif /* !MSDOS */

#ifndef CLK_TCK
#define CLK_TCK 1000
#endif

#if CLK_TCK==60
#define TIME_IN_MSEC(x) (((x)*50)/3)
#else
#define TIME_IN_MSEC(x) ((x)*(long)(1000/CLK_TCK))
#endif

#ifdef HAVE_FTIME
struct timeb real_base;
SCM real_time()
{
	struct timeb time_buffer;
	long tmp;
	ftime(&time_buffer);
	time_buffer.time -= real_base.time;
	tmp = time_buffer.millitm - real_base.millitm;
	tmp = time_buffer.time*1000L + tmp;
	tmp *= CLK_TCK;
	tmp /= 1000;
	return MAKINUM(tmp);
}
#else /* !HAVE_FTIME */
long real_base;
SCM real_time()
{
	return MAKINUM((time(0) - real_base) * CLK_TCK);
}
#endif /* !HAVE_FTIME */

#ifndef HAVE_CLOCK
#ifdef HAVE_TIMES
long clock()
{
  struct tms time_buffer;
  times(&time_buffer);
  return time_buffer.tms_utime;
}
#else /* !HAVE_TIMES */
#define clock ((time(0) - real_base) * CLK_TCK)
#endif /* !HAVE_TIMES */
#endif /* !run_time */

long run_base=0;
SCM run_time()
{
  return MAKINUM(clock()-run_base);
}

static iproc subr0s[]={
	{"get-internal-run-time",run_time},
	{"get-internal-real-time",real_time},
	{0,0}};
SCM s_itups = BOOL_F;

void init_time()
{
	s_itups=intern("internal-time-units-per-second",-30L);
	VCELL(s_itups)=MAKINUM((long)CLK_TCK);
#ifdef HAVE_FTIME
	ftime(&real_base);
#else
	time(&real_base);
#endif
	run_base = clock();
}

SCM sym_features=EOL;
extern char s_getenv[], s_system[];
static char s_tryload[]="try-load";
#define s_load (&s_tryload[4])

char *features[] = {
  "p1178","rev4-report",
#ifdef IO_EXTENSIONS
  "i/o-extensions",
#endif
#ifdef STR_EXTENSIONS
  "substring-move",
#endif
#ifdef BIGDIG
  "bignum",
#endif
#ifdef FLOATS
  "real", "complex",
#ifdef ENGNOT
  "engnot",
#endif
#endif
  "integer-limits",
#ifndef CHEAP_CONTINUATIONS
  "full-continuation",
#endif
#ifdef RECKLESS
  "reckless",
#endif
#ifdef vms
  "ed",
#endif
  s_getenv,s_system,
  0};

void init_features()
{
  char **feats = features;
  sym_features = intern("*features*",-10L);
  VCELL(sym_features) = EOL;
  for(;*feats;feats++) {
      VCELL(sym_features) =
      cons(intern(*feats,-(long)strlen(*feats)),
	   VCELL(sym_features));
  }    
}

jmp_buf errjmp;
int errjmp_ok = 0, sig_disabled = 1,  sig_deferred = 0;
SCM sym_errobj = BOOL_F;
SCM sym_loadpath = BOOL_F;
long line_num = 1;

int verbose = 0;
long cells_allocated = 0, gc_rt, gc_time_taken;
long gc_cells_collected, gc_malloc_collected, gc_ports_collected;
SCM repl_driver()
{
	long i;
	stack_start_ptr = &i;
	sym_errobj=intern("errobj",-6L);
	VCELL(sym_errobj)=UNDEFINED;
	switch (setjmp(errjmp)) {
	case 0:
		errjmp_ok = 1;
		sig_deferred = 0;
		sig_disabled = 0;
		{
		  SCM name = lgetenv(makfromstr("SCM_INIT_PATH",
						 sizeof "SCM_INIT_PATH"-1));
		  if FALSEP(name)
#ifdef IMPLINIT
		    name = makfromstr(IMPLINIT, sizeof IMPLINIT-1);
#else
		    goto noname;
#endif
		  if (BOOL_T != tryload(name))
noname:		    wta((SCM)CHARS(name),(char *)NOFILE,s_load);
		}
	default:
		errjmp_ok = 1;
		sig_deferred = 0;
		sig_disabled = 0;
		VCELL(sym_loadpath) = BOOL_F;
		repl();
	case 3:
		throwval = MAKINUM(1);
	case 2:
		return throwval;
	}
}

void growth_mon(obj, size, units)
char *obj;
long size;
char *units;
{
	if (verbose>1)
	{
		DEFER_SIGINT;
		fputs("; grew ",stdout);
		fputs(obj,stdout);
		fputs(" to ",stdout);
		iprint(size,10,stdout);
		putc(' ',stdout);
		puts(units);
		ALLOW_SIGINT;
	}
}

void gc_start()
{
	gc_rt = clock();
	gc_cells_collected = 0;
	gc_malloc_collected = 0;
	gc_ports_collected = 0;
	if (verbose>2) fputs(";GC ",stdout);
}
void gc_end()
{
	gc_rt = clock() - gc_rt;
	gc_time_taken = gc_time_taken + gc_rt;
	if (verbose>2) {
		iprint(TIME_IN_MSEC(gc_rt),10,stdout);
		fputs(" cpu mSec, ",stdout);
		iprint(gc_cells_collected,10,stdout);
		fputs(" cells, ",stdout);
		iprint(gc_malloc_collected,10,stdout);
		fputs(" malloc, ",stdout);
		iprint(gc_ports_collected,10,stdout);
		puts(" ports collected");
	}
}

SCM prolixity(arg)
SCM arg;
{
  int old = verbose;
  if (!UNBNDP(arg)) {
    if FALSEP(arg) verbose = 0;
    else verbose = INUM(arg);
  }
  return MAKINUM(old);
}

void repl()
{
  SCM x;
  long rt;
  DEFER_SIGINT;
  while(1) {
    fputs("> ",stdout);
    fflush(stdout);
    ALLOW_SIGINT;
    x = lread(cur_inp);
    if (x == EOF_VAL) break;
    rt = clock();
    cells_allocated = 0;
    gc_time_taken = 0;
    x = EVAL(x,EOL);
    DEFER_SIGINT;
    if (verbose) {
      fputs(";Evaluation took ",stdout);
      iprint(TIME_IN_MSEC(clock()-rt),10,stdout);
      fputs(" mSec (",stdout);
      iprint(TIME_IN_MSEC(gc_time_taken),10,stdout);
      fputs(" in gc) ",stdout);
      iprint(cells_allocated,10,stdout);
      puts(" cons work");
    }
    iprin1(x,stdout,1);
    putc('\n',stdout);
  }
}
SCM quit(n)
SCM n;
{
  if UNBNDP(n) n=INUM0;
  throwval = n;
  longjmp(errjmp,2);
}
void err_ctrl_c()
{
	wta(UNDEFINED,"control-c interrupt","");
}

SCM tryload(filename)
SCM filename;
{
  ASSERT(NIMP(filename) && STRINGP(filename),filename,ARG1,s_load);
  {
    SCM oloadpath = VCELL(sym_loadpath);
    long olninum = line_num;
    SCM form,port;
    FILE *f = fopen(CHARS(filename),"r");
    if (!f) return BOOL_F;
    port = makport(f,tc_inport);
    VCELL(sym_loadpath) = filename;
    line_num = 1;
    while(1) {
      form = lread(port);
      if (EOF_VAL == form) break;
      SIDEVAL(form,EOL);
    }
    close_port(port);
    line_num = olninum;
    VCELL(sym_loadpath) = oloadpath;
  }
  return BOOL_T;
}

void err_head(str)
char *str;
{
	fflush(stdout);
	putc('\n',stderr);
	if(BOOL_F != VCELL(sym_loadpath)) {
		iprin1(VCELL(sym_loadpath),stderr,1);
		fputs(", line ",stderr);
		iprint((long)line_num,10,stderr);
		fputs(": ",stderr);
	}
	fputs(str,stderr);
}
void warn(str1,str2)
char *str1,*str2;
{
	DEFER_SIGINT;
	err_head("WARNING: ");
	fputs(str1,stderr);
	fputs(str2,stderr);
	putc('\n',stderr);
	fflush(stderr);
	ALLOW_SIGINT;
}
extern cell dummy_cell;
void everr(sexp,env,arg,pos,s_subr)
SCM sexp,env,arg;
char *pos, *s_subr;
{
	DEFER_SIGINT;
	err_head("ERROR: ");
	if ((~0xfL) & (long)pos) goto stringpos;
	switch ((int)pos) {
	case ARG1:
	case ARG2:
	case ARG3:
	case ARG4:
	case ARG5:
		fputs("Wrong type in arg ",stderr);
		iprint((long)pos,10,stderr);
		fputs(" to ",stderr);
		goto putsname;
	case WNA:
		fputs("Wrong number of args to ",stderr);
		iprin1(arg,stderr,1);
		if (s_subr && *s_subr)
			fputs(" from ",stderr);
		arg = UNDEFINED;
		goto putsname;
	case NALLOC:
		fputs("could not allocate ",stderr);
		iprint(INUM(arg),10,stderr);
		putc(' ',stderr);
		goto putsname;
	case NOFILE:
		perror((char *)arg);
		fputs("could not open file in ",stderr);
		fputs(s_subr,stderr);
		arg = UNDEFINED;
		break;
	default:
	stringpos:
		fputs(pos,stderr);
		if INUMP(arg) {
		  fputs(s_subr,stderr);
		  fputs(": ",stderr);
		  iprin1(arg,stderr,1);
		  arg = UNDEFINED;
		  break;
		}
putsname:
		fputs(s_subr,stderr);
		if UNBNDP(sexp) break;
		if NIMP(sexp) {
		  fputs("\n; in expression: ",stderr);
		  if (sexp == (SCM)&dummy_cell) iprin1(CAR(sexp),stderr,1);
		  if ECONSP(sexp)
		    iprlist("(... ",CAR(sexp),CDR(sexp),')',stderr,1);
		  else iprin1(sexp,stderr,1);
		}
		if NULLP(env) {
		  fputs("\n; in top level environment.",stderr);
		  break;
		}
		fputs("\n; in scope:",stderr);
		while NNULLP(env) {
		  putc('\n',stderr);
		  fputs(";   ",stderr);
		  iprin1(CAR(CAR(env)),stderr,1);
		  env = CDR(env);
		}
	}
	if (errjmp_ok && !UNBNDP(arg))
		fputs("\n; see errobj",stderr);
	putc('\n',stderr);
	fflush(stderr);
	if (errjmp_ok) {
		VCELL(sym_errobj)=arg;
		longjmp(errjmp,1);
	}
	iprin1(arg,stderr,1);
	fputs("\nFATAL ERROR DURING CRITICAL CODE SECTION\n",stderr);
	longjmp(errjmp,3);
}
SCM userr(nam,str,args)
SCM nam,str,args;
{
	DEFER_SIGINT;
	err_head("ERROR : ");
	iprin1(nam,stderr,0);
	iprlist(" ",str,args,'\n',stderr,1);
	fflush(stderr);
	longjmp(errjmp,1);
}
void wta(arg,pos,s_subr)
SCM arg;
char *pos, *s_subr;
{
  everr(UNDEFINED,EOL,arg,pos,s_subr);
}

static iproc subr1s[]={
	{s_load,tryload},
	{s_tryload,tryload},
	{0,0}};

static iproc subr1os[]={
	{"quit",quit},
	{"verbose",prolixity},
	{0,0}};

static iproc lsubr2s[]={
	{"error",userr},
	{0,0}};

void init_repl()
{
	sym_loadpath=intern("*load-pathname*",-15L);
	VCELL(sym_loadpath)=BOOL_F;
	init_iprocs(subr0s, tc7_subr_0);
	init_iprocs(subr1os, tc7_subr_1o);
	init_iprocs(subr1s, tc7_subr_1);
	init_iprocs(lsubr2s, tc7_lsubr_2);
}
