;;;; modular.scm, modular fixnum arithmetic for Scheme
;;; Copyright (C) 1991 Aubrey Jaffer.
;
;  (extended-euclid n1 n2)				procedure
;
;Returns a list of 3 integers (d x y) such that d=gcd(n1,n2)=n1*x+n2*y.
;
;   For all of these procedures all arguments should be exact
;   non-negative integers such that k1 > k2 and k1 > k3.  The returned
;   value will be an exact non-negative integer less than k1.  If all
;   the arguments are fixnums the compuation will use only fixnums.
;
;  (modular:invert k1 k2)				procedure
;
;Returns an integer n such that 1 = (n * k2) mod k1.  If k2 has no
;inverse mod k1 an error is signaled.
;
;  (modular:negate k1 k2)				procedure
;
;Returns (-k2) mod k1.
;
;  (modular:+ k1 k2 k3)					procedure
;
;Returns (k2 + k3) mod k1.
;
;  (modular:- k1 k2 k3)					procedure
;
;Returns (k2 - k3) mod k1.
;
;  (modular:* k1 k2 k3)					procedure
;
;Returns (k2 * k3) mod k1.
;
;  (modular:expt k1 k2 k3)				procedure
;
;Returns (k2 ^ k3) mod k1.
;
;;;;--------------------------------------------------------------
(require 'logical)

;;; from:
;;; Introduction to Algorithms by T. Cormen, C. Leiserson, R. Rivest.
;;; 1989 MIT Press.
;;; (extended-euclid a b) returns a list (d x y) such that
;;; d=gcd(a,b)=a*x+b*y.
(define (modular:extended-euclid a b)
  (if (zero? b)
      (list a 1 0)
      (let ((res (modular:extended-euclid b (modulo a b))))
	(list (car res)
	      (caddr res)
	      (- (cadr res) (* (quotient a b) (caddr res)))))))

(define (modular:invert m a)
  (let ((d (modular:extended-euclid a m)))
    (if (= 1 (car d))
	(modulo (cadr d) m)
	(slib:error "modular:invert can't invert" m a))))

(define modular:negate -)

(define (modular:+ m a b) (modulo (+ (- a m) b) m))

(define (modular:- m a b) (modulo (- a b) m))

;;; See: L'Ecuyer, P. and Cote, S. "Implementing a Random Number Package
;;; with Splitting Facilities." ACM Transactions on Mathematical
;;; Software, 17:98-111 (1991)

;;; modular:r = 2**((nb-2)/2) where nb = number of bits in a word.
(define modular:r
  (ash 1 (quotient (integer-length most-positive-fixnum) 2)))
(define (modular:* m a b)
  (let ((a0 a)
	(p 0))
    (cond ((< a modular:r))
	  ((< b modular:r) (set! a b) (set! b a0) (set! a0 a))
	  (else
	   (set! a0 (modulo a modular:r))
	   (let ((a1 (quotient a modular:r))
		 (qh (quotient m modular:r))
		 (rh (modulo m modular:r)))
	     (cond ((>= a1 modular:r)
		    (set! a1 (- a1 modular:r))
		    (set! p (modulo (- (* modular:r (modulo b qh))
				       (* (quotient b qh) rh)) m))))
	     (cond ((not (zero? a1))
		    (let ((q (quotient m a1)))
		      (set! p (- p (* (quotient b q) (modulo m a1))))
		      (set! p (modulo (+ (if (positive? p) (- p m) p)
					 (* a1 (modulo b q))) m)))))
	     (set! p (modulo (- (* modular:r (modulo p qh))
				(* (quotient p qh) rh)) m)))))
    (if (zero? a0)
	p
	(let ((q (quotient m a0)))
	  (set! p (- p (* (quotient b q) (modulo m a0))))
	  (modulo (+ (if (positive? p) (- p m) p)
		     (* a0 (modulo b q))) m)))))

(define (modular:expt m a b)
  (cond ((= a 1) 1)
	((= a (- m 1)) (if (odd? b) a 1))
	((zero? a) 0)
	(else
					;Fermat's theorem
	 (logical:ipow-by-squaring a (modulo b (- m 1)) 1
				   (lambda (c d) (modular:* m c d))))))

(define extended-euclid modular:extended-euclid)
