;;;; Pseudo-Random number generator for scheme.
;;; Copyright (C) 1991 Aubrey Jaffer.
;
;  (random n)						procedure
;  (random n state)					procedure
;
;Accepts a positive integer or real n and returns a number of the
;same type between zero (inclusive) and n (exclusive).  The values
;returned have a uniform distribution.
;
;The optional argument state must be of the type produced by
;(make-random-state).  It defaults to the value of the variable
;*random-state*.  This object is used to maintain the state of the
;pseudo-random-number generator and is altered as a side effect of the
;RANDOM operation.
;
;  *random-state*					variable
;
;Holds a data structure that encodes the internal state of the
;random-number generator that RANDOM uses by default.  The nature of
;this data structure is implementation-dependent.  It may be printed
;out and successfully read back in, but may or may not function
;correclty as a random-number state object in another implementation.
;
;  (make-random-state)					procedure
;  (make-random-state state)				procedure
;
;Returns a new object of type suitable for use as the value of the
;variable *random-state* and as second argument to RANDOM.  If argument
;state is given, a copy of it is returned.  Otherwise a copy of
;*random-state* is returned.
;
;If inexaxt numbers are support by the Scheme implementation,
;randinex.scm will be loaded as well.  Randinex.scm contains
;procedures for generating inexact distributions.
;;;;------------------------------------------------------------------

(require 'logical)

(define random:tap-1 24)
(define random:size 55)

(define (random:size-int l)
  (if (exact? (string->number (make-string l #\f) 16))
      l
      (random:size-int (- l 1))))
(define random:chunk-size (* 4 (random:size-int 8)))

(define random:MASK
  (string->number (make-string (quotient random:chunk-size 4) #\f) 16))

(define *random-state*
  '#(
 "d909ef3e" "fd330ab3" "e33f7843" "76783fbd" "f3675fb3"
 "b54ef879" "0be45590" "a6794679" "0bcd56d3" "fabcdef8"
 "9cbd3efd" "3fd3efcd" "e064ef27" "dddecc08" "34444292"
 "85444454" "4c519210" "c0366273" "54734567" "70abcddc"
 "1bbdac53" "616c5a86" "a982efa9" "105996a0" "5f0cccba"
 "1ea055e1" "fe2acd8d" "1891c1d4" "e6690270" "6912bccc"
 "2678e141" "61222224" "907abcbb" "4ad6829b" "9cdd1404"
 "57798841" "5b892496" "871c9cd1" "d1e67bda" "8b0a3233"
 "578ef23f" "28274ef6" "823ef5ef" "845678c5" "e67890a5"
 "5890abcb" "851fa9ab" "13efa13a" "b12278d6" "daf805ab"
 "a0befc36" "0068a7b5" "e024fd90" "a7b690e2" "27f3571a"
 0))

(let ((random-strings *random-state*))
  (set! *random-state* (make-vector (+ random:size 1) 0))
  (let ((nibbles (quotient random:chunk-size 4)))
    (do ((i 0 (+ i 1)))
	((= i random:size))
      (vector-set!
       *random-state* i
       (string->number (substring (vector-ref random-strings i)
				  0 nibbles)
		       16)))))

;;; random:chunk returns an integer in the range of
;;; 0 to (- (expt 2 random:chunk-size) 1)
(define (random:chunk v)
  (let* ((p (vector-ref v random:size))
	 (i (modulo (- p random:tap-1) random:size))
	 (ans (vector-ref v p)))
    (vector-set! v p (logical:logxor (vector-ref v i) ans))
    (vector-set! v random:size (modulo (- p 1) random:size))
    ans))

(define (random:bits n state)
  (cond ((= n random:chunk-size) (random:chunk state))
	((< n random:chunk-size)
	 (logical:logand (random:chunk state) (- (logical:ash 1 n) 1)))
	(else
	 (+ (* (random:bits (- n random:chunk-size) state)
	       (+ 1 random:MASK))
	    (random:chunk state)))))

(define (random:random modu . args)
  (let ((state (if (null? args) *random-state* (car args))))
    (if (exact? modu)
	(let ((ilen (integer-length modu)))
	  (do ((r (random:bits ilen state)
		  (random:bits ilen state))) ;this could be improved.
	      ((< r modu) r)))
	(* (random:uniform state) modu))))
;;;random:uniform is in randinex.scm.  It is needed only if inexact is
;;;supported.

(define (random:make-random-state . args)
  (let ((state (if (null? args) *random-state* (car args))))
    (list->vector (vector->list state))))

(define random random:random)
(define make-random-state random:make-random-state)

(provide 'random)			;to prevent loops
(if (provided? 'inexact) (require 'random-inexact))
