/*

  Linux Driver for BusLogic MultiMaster and FlashPoint SCSI Host Adapters

  Copyright 1995-1998 by Leonard N. Zubkoff <lnz@dandelion.com>

  This program is free software; you may redistribute and/or modify it under
  the terms of the GNU General Public License Version 2 as published by the
  Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY, without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
  for complete details.

  The author respectfully requests that any modifications to this software be
  sent directly to him for evaluation and testing.

  Special thanks to Wayne Yen, Jin-Lon Hon, and Alex Win of BusLogic, whose
  advice has been invaluable, to David Gentzel, for writing the original Linux
  BusLogic driver, and to Paul Gortmaker, for being such a dedicated test site.

  Finally, special thanks to Mylex/BusLogic for making the FlashPoint SCCB
  Manager available as freely redistributable source code.

*/


#include <linux/config.h>


/*
  Define types for some of the structures that interface with the rest
  of the Linux Kernel and SCSI Subsystem.
*/

typedef unsigned long ProcessorFlags_T;
typedef struct pt_regs Registers_T;
typedef struct partition PartitionTable_T;
typedef struct pci_dev PCI_Device_T;
typedef Scsi_Host_Template SCSI_Host_Template_T;
typedef struct Scsi_Host SCSI_Host_T;
typedef struct scsi_device SCSI_Device_T;
typedef struct scsi_cmnd SCSI_Command_T;
typedef struct scatterlist SCSI_ScatterList_T;


/*
  Define prototypes for the BusLogic Driver Interface Functions.
*/

extern const char *BusLogic_DriverInfo(SCSI_Host_T *);
extern int BusLogic_DetectHostAdapter(SCSI_Host_Template_T *);
extern int BusLogic_ReleaseHostAdapter(SCSI_Host_T *);
extern int BusLogic_QueueCommand(SCSI_Command_T *,
				 void (*CompletionRoutine)(SCSI_Command_T *));
extern int BusLogic_BIOSDiskParameters(struct scsi_device *,
		struct block_device *, sector_t, int *);
extern int BusLogic_ProcDirectoryInfo(char *, char **, off_t, int, int, int);
extern int BusLogic_SlaveConfigure(SCSI_Device_T *);

#ifdef BusLogic_DriverVersion


/*
  FlashPoint support is only available for the Intel x86 Architecture with
  CONFIG_PCI set.
*/

#ifndef __i386__
#undef CONFIG_SCSI_OMIT_FLASHPOINT
#define CONFIG_SCSI_OMIT_FLASHPOINT
#endif

#ifndef CONFIG_PCI
#undef CONFIG_SCSI_OMIT_FLASHPOINT
#define CONFIG_SCSI_OMIT_FLASHPOINT
#define BusLogic_InitializeProbeInfoListISA \
  BusLogic_InitializeProbeInfoList
#endif


/*
  Define the maximum number of BusLogic Host Adapters supported by this driver.
*/

#define BusLogic_MaxHostAdapters		16


/*
  Define the maximum number of Target Devices supported by this driver.
*/

#define BusLogic_MaxTargetDevices		16


/*
  Define the maximum number of Scatter/Gather Segments used by this driver.
  For optimal performance, it is important that this limit be at least as
  large as the largest single request generated by the I/O Subsystem.
*/

#define BusLogic_ScatterGatherLimit		128


/*
  Define the maximum, maximum automatic, minimum automatic, and default Queue
  Depth to allow for Target Devices depending on whether or not they support
  Tagged Queuing and whether or not ISA Bounce Buffers are required.
*/

#define BusLogic_MaxTaggedQueueDepth		64
#define BusLogic_MaxAutomaticTaggedQueueDepth	28
#define BusLogic_MinAutomaticTaggedQueueDepth	7
#define BusLogic_TaggedQueueDepthBB		3
#define BusLogic_UntaggedQueueDepth		3
#define BusLogic_UntaggedQueueDepthBB		2


/*
  Define the default amount of time in seconds to wait between a Host Adapter
  Hard Reset which initiates a SCSI Bus Reset and issuing any SCSI commands.
  Some SCSI devices get confused if they receive SCSI commands too soon after
  a SCSI Bus Reset.
*/

#define BusLogic_DefaultBusSettleTime		2


/*
  Define the maximum number of Mailboxes that should be used for MultiMaster
  Host Adapters.  This number is chosen to be larger than the maximum Host
  Adapter Queue Depth and small enough so that the Host Adapter structure
  does not cross an allocation block size boundary.
*/

#define BusLogic_MaxMailboxes			211


/*
  Define the number of CCBs that should be allocated as a group to optimize
  Kernel memory allocation.
*/

#define BusLogic_CCB_AllocationGroupSize	7


/*
  Define the Host Adapter Line and Message Buffer Sizes.
*/

#define BusLogic_LineBufferSize			100
#define BusLogic_MessageBufferSize		9700


/*
  Define the Driver Message Levels.
*/

typedef enum BusLogic_MessageLevel
{
  BusLogic_AnnounceLevel =			0,
  BusLogic_InfoLevel =				1,
  BusLogic_NoticeLevel =			2,
  BusLogic_WarningLevel =			3,
  BusLogic_ErrorLevel =				4
}
BusLogic_MessageLevel_T;

static char
  *BusLogic_MessageLevelMap[] =
    { KERN_NOTICE, KERN_NOTICE, KERN_NOTICE, KERN_WARNING, KERN_ERR };


/*
  Define Driver Message macros.
*/

#define BusLogic_Announce(Format, Arguments...) \
  BusLogic_Message(BusLogic_AnnounceLevel, Format, ##Arguments)

#define BusLogic_Info(Format, Arguments...) \
  BusLogic_Message(BusLogic_InfoLevel, Format, ##Arguments)

#define BusLogic_Notice(Format, Arguments...) \
  BusLogic_Message(BusLogic_NoticeLevel, Format, ##Arguments)

#define BusLogic_Warning(Format, Arguments...) \
  BusLogic_Message(BusLogic_WarningLevel, Format, ##Arguments)

#define BusLogic_Error(Format, Arguments...) \
  BusLogic_Message(BusLogic_ErrorLevel, Format, ##Arguments)


/*
  Define the types of BusLogic Host Adapters that are supported and the number
  of I/O Addresses required by each type.
*/

typedef enum
{
  BusLogic_MultiMaster =			1,
  BusLogic_FlashPoint =				2
}
__attribute__ ((packed))
BusLogic_HostAdapterType_T;

#define BusLogic_MultiMasterAddressCount	4
#define BusLogic_FlashPointAddressCount		256

static int
  BusLogic_HostAdapterAddressCount[3] =
    { 0, BusLogic_MultiMasterAddressCount, BusLogic_FlashPointAddressCount };


/*
  Define macros for testing the Host Adapter Type.
*/

#ifndef CONFIG_SCSI_OMIT_FLASHPOINT

#define BusLogic_MultiMasterHostAdapterP(HostAdapter) \
  (HostAdapter->HostAdapterType == BusLogic_MultiMaster)

#define BusLogic_FlashPointHostAdapterP(HostAdapter) \
  (HostAdapter->HostAdapterType == BusLogic_FlashPoint)

#else

#define BusLogic_MultiMasterHostAdapterP(HostAdapter) \
  (true)

#define BusLogic_FlashPointHostAdapterP(HostAdapter) \
  (false)

#endif


/*
  Define the possible Host Adapter Bus Types.
*/

typedef enum
{
  BusLogic_Unknown_Bus =			0,
  BusLogic_ISA_Bus =				1,
  BusLogic_EISA_Bus =				2,
  BusLogic_PCI_Bus =				3,
  BusLogic_VESA_Bus =				4,
  BusLogic_MCA_Bus =				5
}
__attribute__ ((packed))
BusLogic_HostAdapterBusType_T;

static char
  *BusLogic_HostAdapterBusNames[] =
    { "Unknown", "ISA", "EISA", "PCI", "VESA", "MCA" };

static BusLogic_HostAdapterBusType_T
  BusLogic_HostAdapterBusTypes[] =
    { BusLogic_VESA_Bus,				/* BT-4xx */
      BusLogic_ISA_Bus,					/* BT-5xx */
      BusLogic_MCA_Bus,					/* BT-6xx */
      BusLogic_EISA_Bus,				/* BT-7xx */
      BusLogic_Unknown_Bus,				/* BT-8xx */
      BusLogic_PCI_Bus };				/* BT-9xx */


/*
  Define the possible Host Adapter BIOS Disk Geometry Translations.
*/

typedef enum BusLogic_BIOS_DiskGeometryTranslation
{
  BusLogic_BIOS_Disk_Not_Installed =		0,
  BusLogic_BIOS_Disk_Installed_64x32 =		1,
  BusLogic_BIOS_Disk_Installed_128x32 =		2,
  BusLogic_BIOS_Disk_Installed_255x63 =		3
}
__attribute__ ((packed))
BusLogic_BIOS_DiskGeometryTranslation_T;


/*
  Define a Boolean data type.
*/

typedef enum { false, true } __attribute__ ((packed)) boolean;


/*
  Define a 32 bit I/O Address data type.
*/

typedef unsigned int BusLogic_IO_Address_T;


/*
  Define a 32 bit PCI Bus Address data type.
*/

typedef unsigned int BusLogic_PCI_Address_T;


/*
  Define a 32 bit Base Address data type.
*/

typedef unsigned int BusLogic_Base_Address_T;


/*
  Define a 32 bit Bus Address data type.
*/

typedef unsigned int BusLogic_BusAddress_T;


/*
  Define a 32 bit Byte Count data type.
*/

typedef unsigned int BusLogic_ByteCount_T;


/*
  Define a 10^18 Statistics Byte Counter data type.
*/

typedef struct BusLogic_ByteCounter
{
  unsigned int Units;
  unsigned int Billions;
}
BusLogic_ByteCounter_T;


/*
  Define the structure for I/O Address and Bus Probing Information.
*/

typedef struct BusLogic_ProbeInfo
{
  BusLogic_HostAdapterType_T HostAdapterType;
  BusLogic_HostAdapterBusType_T HostAdapterBusType;
  BusLogic_IO_Address_T IO_Address;
  BusLogic_PCI_Address_T PCI_Address;
  PCI_Device_T *PCI_Device;
  unsigned char Bus;
  unsigned char Device;
  unsigned char IRQ_Channel;
}
BusLogic_ProbeInfo_T;


/*
  Define the Probe Options.
*/

typedef struct BusLogic_ProbeOptions
{
  boolean NoProbe:1;					/* Bit 0 */
  boolean NoProbeISA:1;					/* Bit 1 */
  boolean NoProbePCI:1;					/* Bit 2 */
  boolean NoSortPCI:1;					/* Bit 3 */
  boolean MultiMasterFirst:1;				/* Bit 4 */
  boolean FlashPointFirst:1;				/* Bit 5 */
  boolean LimitedProbeISA:1;				/* Bit 6 */
  boolean Probe330:1;					/* Bit 7 */
  boolean Probe334:1;					/* Bit 8 */
  boolean Probe230:1;					/* Bit 9 */
  boolean Probe234:1;					/* Bit 10 */
  boolean Probe130:1;					/* Bit 11 */
  boolean Probe134:1;					/* Bit 12 */
}
BusLogic_ProbeOptions_T;


/*
  Define the Global Options.
*/

typedef struct BusLogic_GlobalOptions
{
  boolean TraceProbe:1;					/* Bit 0 */
  boolean TraceHardwareReset:1;				/* Bit 1 */
  boolean TraceConfiguration:1;				/* Bit 2 */
  boolean TraceErrors:1;				/* Bit 3 */
}
BusLogic_GlobalOptions_T;


/*
  Define the Local Options.
*/

typedef struct BusLogic_LocalOptions
{
  boolean InhibitTargetInquiry:1;			/* Bit 0 */
}
BusLogic_LocalOptions_T;


/*
  Define the Error Recovery Strategy Options.
*/

typedef enum
{
  BusLogic_ErrorRecovery_Default =		0,
  BusLogic_ErrorRecovery_BusDeviceReset =	1,
  BusLogic_ErrorRecovery_HardReset =		2,
  BusLogic_ErrorRecovery_None =			3
}
__attribute__ ((packed))
BusLogic_ErrorRecoveryStrategy_T;

static char
  *BusLogic_ErrorRecoveryStrategyNames[] =
    { "Default", "Bus Device Reset", "Hard Reset", "None" },
  BusLogic_ErrorRecoveryStrategyLetters[] =
    { 'D', 'B', 'H', 'N' };


/*
  Define the BusLogic SCSI Host Adapter I/O Register Offsets.
*/

#define BusLogic_ControlRegisterOffset		0	/* WO register */
#define BusLogic_StatusRegisterOffset		0	/* RO register */
#define BusLogic_CommandParameterRegisterOffset	1	/* WO register */
#define BusLogic_DataInRegisterOffset		1	/* RO register */
#define BusLogic_InterruptRegisterOffset	2	/* RO register */
#define BusLogic_GeometryRegisterOffset		3	/* RO register */


/*
  Define the structure of the write-only Control Register.
*/

typedef union BusLogic_ControlRegister
{
  unsigned char All;
  struct {
    unsigned char :4;					/* Bits 0-3 */
    boolean SCSIBusReset:1;				/* Bit 4 */
    boolean InterruptReset:1;				/* Bit 5 */
    boolean SoftReset:1;				/* Bit 6 */
    boolean HardReset:1;				/* Bit 7 */
  } Bits;
}
BusLogic_ControlRegister_T;


/*
  Define the structure of the read-only Status Register.
*/

typedef union BusLogic_StatusRegister
{
  unsigned char All;
  struct {
    boolean CommandInvalid:1;				/* Bit 0 */
    boolean Reserved:1;					/* Bit 1 */
    boolean DataInRegisterReady:1;			/* Bit 2 */
    boolean CommandParameterRegisterBusy:1;		/* Bit 3 */
    boolean HostAdapterReady:1;				/* Bit 4 */
    boolean InitializationRequired:1;			/* Bit 5 */
    boolean DiagnosticFailure:1;			/* Bit 6 */
    boolean DiagnosticActive:1;				/* Bit 7 */
  } Bits;
}
BusLogic_StatusRegister_T;


/*
  Define the structure of the read-only Interrupt Register.
*/

typedef union BusLogic_InterruptRegister
{
  unsigned char All;
  struct {
    boolean IncomingMailboxLoaded:1;			/* Bit 0 */
    boolean OutgoingMailboxAvailable:1;			/* Bit 1 */
    boolean CommandComplete:1;				/* Bit 2 */
    boolean ExternalBusReset:1;				/* Bit 3 */
    unsigned char Reserved:3;				/* Bits 4-6 */
    boolean InterruptValid:1;				/* Bit 7 */
  } Bits;
}
BusLogic_InterruptRegister_T;


/*
  Define the structure of the read-only Geometry Register.
*/

typedef union BusLogic_GeometryRegister
{
  unsigned char All;
  struct {
    BusLogic_BIOS_DiskGeometryTranslation_T Drive0Geometry:2; /* Bits 0-1 */
    BusLogic_BIOS_DiskGeometryTranslation_T Drive1Geometry:2; /* Bits 2-3 */
    unsigned char :3;					/* Bits 4-6 */
    boolean ExtendedTranslationEnabled:1;		/* Bit 7 */
  } Bits;
}
BusLogic_GeometryRegister_T;


/*
  Define the BusLogic SCSI Host Adapter Command Register Operation Codes.
*/

typedef enum
{
  BusLogic_TestCommandCompleteInterrupt =	0x00,
  BusLogic_InitializeMailbox =			0x01,
  BusLogic_ExecuteMailboxCommand =		0x02,
  BusLogic_ExecuteBIOSCommand =			0x03,
  BusLogic_InquireBoardID =			0x04,
  BusLogic_EnableOutgoingMailboxAvailableInt =	0x05,
  BusLogic_SetSCSISelectionTimeout =		0x06,
  BusLogic_SetPreemptTimeOnBus =		0x07,
  BusLogic_SetTimeOffBus =			0x08,
  BusLogic_SetBusTransferRate =			0x09,
  BusLogic_InquireInstalledDevicesID0to7 =	0x0A,
  BusLogic_InquireConfiguration =		0x0B,
  BusLogic_EnableTargetMode =			0x0C,
  BusLogic_InquireSetupInformation =		0x0D,
  BusLogic_WriteAdapterLocalRAM =		0x1A,
  BusLogic_ReadAdapterLocalRAM =		0x1B,
  BusLogic_WriteBusMasterChipFIFO =		0x1C,
  BusLogic_ReadBusMasterChipFIFO =		0x1D,
  BusLogic_EchoCommandData =			0x1F,
  BusLogic_HostAdapterDiagnostic =		0x20,
  BusLogic_SetAdapterOptions =			0x21,
  BusLogic_InquireInstalledDevicesID8to15 =	0x23,
  BusLogic_InquireTargetDevices =		0x24,
  BusLogic_DisableHostAdapterInterrupt =	0x25,
  BusLogic_InitializeExtendedMailbox =		0x81,
  BusLogic_ExecuteSCSICommand =			0x83,
  BusLogic_InquireFirmwareVersion3rdDigit =	0x84,
  BusLogic_InquireFirmwareVersionLetter =	0x85,
  BusLogic_InquirePCIHostAdapterInformation =	0x86,
  BusLogic_InquireHostAdapterModelNumber =	0x8B,
  BusLogic_InquireSynchronousPeriod =		0x8C,
  BusLogic_InquireExtendedSetupInformation =	0x8D,
  BusLogic_EnableStrictRoundRobinMode =		0x8F,
  BusLogic_StoreHostAdapterLocalRAM =		0x90,
  BusLogic_FetchHostAdapterLocalRAM =		0x91,
  BusLogic_StoreLocalDataInEEPROM =		0x92,
  BusLogic_UploadAutoSCSICode =			0x94,
  BusLogic_ModifyIOAddress =			0x95,
  BusLogic_SetCCBFormat =			0x96,
  BusLogic_WriteInquiryBuffer =			0x9A,
  BusLogic_ReadInquiryBuffer =			0x9B,
  BusLogic_FlashROMUploadDownload =		0xA7,
  BusLogic_ReadSCAMData =			0xA8,
  BusLogic_WriteSCAMData =			0xA9
}
BusLogic_OperationCode_T;


/*
  Define the Inquire Board ID reply structure.
*/

typedef struct BusLogic_BoardID
{
  unsigned char BoardType;				/* Byte 0 */
  unsigned char CustomFeatures;				/* Byte 1 */
  unsigned char FirmwareVersion1stDigit;		/* Byte 2 */
  unsigned char FirmwareVersion2ndDigit;		/* Byte 3 */
}
BusLogic_BoardID_T;


/*
  Define the Inquire Installed Devices ID 0 to 7 and Inquire Installed
  Devices ID 8 to 15 reply type.  For each Target Device, a byte is returned
  where bit 0 set indicates that Logical Unit 0 exists, bit 1 set indicates
  that Logical Unit 1 exists, and so on.
*/

typedef unsigned char BusLogic_InstalledDevices8_T[8];


/*
  Define the Inquire Target Devices reply type.  Inquire Target Devices only
  tests Logical Unit 0 of each Target Device unlike the Inquire Installed
  Devices commands which test Logical Units 0 - 7.  Two bytes are returned,
  where byte 0 bit 0 set indicates that Target Device 0 exists, and so on.
*/

typedef unsigned short BusLogic_InstalledDevices_T;


/*
  Define the Inquire Configuration reply structure.
*/

typedef struct BusLogic_Configuration
{
  unsigned char :5;					/* Byte 0 Bits 0-4 */
  boolean DMA_Channel5:1;				/* Byte 0 Bit 5 */
  boolean DMA_Channel6:1;				/* Byte 0 Bit 6 */
  boolean DMA_Channel7:1;				/* Byte 0 Bit 7 */
  boolean IRQ_Channel9:1;				/* Byte 1 Bit 0 */
  boolean IRQ_Channel10:1;				/* Byte 1 Bit 1 */
  boolean IRQ_Channel11:1;				/* Byte 1 Bit 2 */
  boolean IRQ_Channel12:1;				/* Byte 1 Bit 3 */
  unsigned char :1;					/* Byte 1 Bit 4 */
  boolean IRQ_Channel14:1;				/* Byte 1 Bit 5 */
  boolean IRQ_Channel15:1;				/* Byte 1 Bit 6 */
  unsigned char :1;					/* Byte 1 Bit 7 */
  unsigned char HostAdapterID:4;			/* Byte 2 Bits 0-3 */
  unsigned char :4;					/* Byte 2 Bits 4-7 */
}
BusLogic_Configuration_T;


/*
  Define the Inquire Setup Information reply structure.
*/

typedef struct BusLogic_SynchronousValue
{
  unsigned char Offset:4;				/* Bits 0-3 */
  unsigned char TransferPeriod:3;			/* Bits 4-6 */
  boolean Synchronous:1;				/* Bit 7 */
}
BusLogic_SynchronousValue_T;

typedef BusLogic_SynchronousValue_T
  BusLogic_SynchronousValues8_T[8];

typedef BusLogic_SynchronousValue_T
  BusLogic_SynchronousValues_T[BusLogic_MaxTargetDevices];

typedef struct BusLogic_SetupInformation
{
  boolean SynchronousInitiationEnabled:1;		/* Byte 0 Bit 0 */
  boolean ParityCheckingEnabled:1;			/* Byte 0 Bit 1 */
  unsigned char :6;					/* Byte 0 Bits 2-7 */
  unsigned char BusTransferRate;			/* Byte 1 */
  unsigned char PreemptTimeOnBus;			/* Byte 2 */
  unsigned char TimeOffBus;				/* Byte 3 */
  unsigned char MailboxCount;				/* Byte 4 */
  unsigned char MailboxAddress[3];			/* Bytes 5-7 */
  BusLogic_SynchronousValues8_T SynchronousValuesID0to7; /* Bytes 8-15 */
  unsigned char DisconnectPermittedID0to7;		/* Byte 16 */
  unsigned char Signature;				/* Byte 17 */
  unsigned char CharacterD;				/* Byte 18 */
  unsigned char HostBusType;				/* Byte 19 */
  unsigned char WideTransfersPermittedID0to7;		/* Byte 20 */
  unsigned char WideTransfersActiveID0to7;		/* Byte 21 */
  BusLogic_SynchronousValues8_T SynchronousValuesID8to15; /* Bytes 22-29 */
  unsigned char DisconnectPermittedID8to15;		/* Byte 30 */
  unsigned char :8;					/* Byte 31 */
  unsigned char WideTransfersPermittedID8to15;		/* Byte 32 */
  unsigned char WideTransfersActiveID8to15;		/* Byte 33 */
}
BusLogic_SetupInformation_T;


/*
  Define the Initialize Extended Mailbox request structure.
*/

typedef struct BusLogic_ExtendedMailboxRequest
{
  unsigned char MailboxCount;				/* Byte 0 */
  BusLogic_BusAddress_T BaseMailboxAddress;		/* Bytes 1-4 */
}
__attribute__ ((packed))
BusLogic_ExtendedMailboxRequest_T;


/*
  Define the Inquire Firmware Version 3rd Digit reply type.
*/

typedef unsigned char BusLogic_FirmwareVersion3rdDigit_T;


/*
  Define the Inquire Firmware Version Letter reply type.
*/

typedef unsigned char BusLogic_FirmwareVersionLetter_T;


/*
  Define the Inquire PCI Host Adapter Information reply type.  The ISA
  Compatible I/O Port values are defined here and are also used with
  the Modify I/O Address command.
*/

typedef enum BusLogic_ISACompatibleIOPort
{
  BusLogic_IO_330 =				0,
  BusLogic_IO_334 =				1,
  BusLogic_IO_230 =				2,
  BusLogic_IO_234 =				3,
  BusLogic_IO_130 =				4,
  BusLogic_IO_134 =				5,
  BusLogic_IO_Disable =				6,
  BusLogic_IO_Disable2 =			7
}
__attribute__ ((packed))
BusLogic_ISACompatibleIOPort_T;

typedef struct BusLogic_PCIHostAdapterInformation
{
  BusLogic_ISACompatibleIOPort_T ISACompatibleIOPort;	/* Byte 0 */
  unsigned char PCIAssignedIRQChannel;			/* Byte 1 */
  boolean LowByteTerminated:1;				/* Byte 2 Bit 0 */
  boolean HighByteTerminated:1;				/* Byte 2 Bit 1 */
  unsigned char :2;					/* Byte 2 Bits 2-3 */
  boolean JP1:1;					/* Byte 2 Bit 4 */
  boolean JP2:1;					/* Byte 2 Bit 5 */
  boolean JP3:1;					/* Byte 2 Bit 6 */
  boolean GenericInfoValid:1;				/* Byte 2 Bit 7 */
  unsigned char :8;					/* Byte 3 */
}
BusLogic_PCIHostAdapterInformation_T;


/*
  Define the Inquire Host Adapter Model Number reply type.
*/

typedef unsigned char BusLogic_HostAdapterModelNumber_T[5];


/*
  Define the Inquire Synchronous Period reply type.  For each Target Device,
  a byte is returned which represents the Synchronous Transfer Period in units
  of 10 nanoseconds.
*/

typedef unsigned char BusLogic_SynchronousPeriod_T[BusLogic_MaxTargetDevices];


/*
  Define the Inquire Extended Setup Information reply structure.
*/

typedef struct BusLogic_ExtendedSetupInformation
{
  unsigned char BusType;				/* Byte 0 */
  unsigned char BIOS_Address;				/* Byte 1 */
  unsigned short ScatterGatherLimit;			/* Bytes 2-3 */
  unsigned char MailboxCount;				/* Byte 4 */
  BusLogic_BusAddress_T BaseMailboxAddress;		/* Bytes 5-8 */
  struct { unsigned char :2;				/* Byte 9 Bits 0-1 */
	   boolean FastOnEISA:1;			/* Byte 9 Bit 2 */
	   unsigned char :3;				/* Byte 9 Bits 3-5 */
	   boolean LevelSensitiveInterrupt:1;		/* Byte 9 Bit 6 */
	   unsigned char :1; } Misc;			/* Byte 9 Bit 7 */
  unsigned char FirmwareRevision[3];			/* Bytes 10-12 */
  boolean HostWideSCSI:1;				/* Byte 13 Bit 0 */
  boolean HostDifferentialSCSI:1;			/* Byte 13 Bit 1 */
  boolean HostSupportsSCAM:1;				/* Byte 13 Bit 2 */
  boolean HostUltraSCSI:1;				/* Byte 13 Bit 3 */
  boolean HostSmartTermination:1;			/* Byte 13 Bit 4 */
  unsigned char :3;					/* Byte 13 Bits 5-7 */
}
__attribute__ ((packed))
BusLogic_ExtendedSetupInformation_T;


/*
  Define the Enable Strict Round Robin Mode request type.
*/

typedef enum BusLogic_RoundRobinModeRequest
{
  BusLogic_AggressiveRoundRobinMode =		0,
  BusLogic_StrictRoundRobinMode =		1
}
__attribute__ ((packed))
BusLogic_RoundRobinModeRequest_T;


/*
  Define the Fetch Host Adapter Local RAM request type.
*/

#define BusLogic_BIOS_BaseOffset		0
#define BusLogic_AutoSCSI_BaseOffset		64

typedef struct BusLogic_FetchHostAdapterLocalRAMRequest
{
  unsigned char ByteOffset;				/* Byte 0 */
  unsigned char ByteCount;				/* Byte 1 */
}
BusLogic_FetchHostAdapterLocalRAMRequest_T;


/*
  Define the Host Adapter Local RAM AutoSCSI structure.
*/

typedef struct BusLogic_AutoSCSIData
{
  unsigned char InternalFactorySignature[2];		/* Bytes 0-1 */
  unsigned char InformationByteCount;			/* Byte 2 */
  unsigned char HostAdapterType[6];			/* Bytes 3-8 */
  unsigned char :8;					/* Byte 9 */
  boolean FloppyEnabled:1;				/* Byte 10 Bit 0 */
  boolean FloppySecondary:1;				/* Byte 10 Bit 1 */
  boolean LevelSensitiveInterrupt:1;			/* Byte 10 Bit 2 */
  unsigned char :2;					/* Byte 10 Bits 3-4 */
  unsigned char SystemRAMAreaForBIOS:3;			/* Byte 10 Bits 5-7 */
  unsigned char DMA_Channel:7;				/* Byte 11 Bits 0-6 */
  boolean DMA_AutoConfiguration:1;			/* Byte 11 Bit 7 */
  unsigned char IRQ_Channel:7;				/* Byte 12 Bits 0-6 */
  boolean IRQ_AutoConfiguration:1;			/* Byte 12 Bit 7 */
  unsigned char DMA_TransferRate;			/* Byte 13 */
  unsigned char SCSI_ID;				/* Byte 14 */
  boolean LowByteTerminated:1;				/* Byte 15 Bit 0 */
  boolean ParityCheckingEnabled:1;			/* Byte 15 Bit 1 */
  boolean HighByteTerminated:1;				/* Byte 15 Bit 2 */
  boolean NoisyCablingEnvironment:1;			/* Byte 15 Bit 3 */
  boolean FastSynchronousNegotiation:1;			/* Byte 15 Bit 4 */
  boolean BusResetEnabled:1;				/* Byte 15 Bit 5 */
  boolean :1;						/* Byte 15 Bit 6 */
  boolean ActiveNegationEnabled:1;			/* Byte 15 Bit 7 */
  unsigned char BusOnDelay;				/* Byte 16 */
  unsigned char BusOffDelay;				/* Byte 17 */
  boolean HostAdapterBIOSEnabled:1;			/* Byte 18 Bit 0 */
  boolean BIOSRedirectionOfINT19Enabled:1;		/* Byte 18 Bit 1 */
  boolean ExtendedTranslationEnabled:1;			/* Byte 18 Bit 2 */
  boolean MapRemovableAsFixedEnabled:1;			/* Byte 18 Bit 3 */
  boolean :1;						/* Byte 18 Bit 4 */
  boolean BIOSSupportsMoreThan2DrivesEnabled:1;		/* Byte 18 Bit 5 */
  boolean BIOSInterruptModeEnabled:1;			/* Byte 18 Bit 6 */
  boolean FlopticalSupportEnabled:1;			/* Byte 19 Bit 7 */
  unsigned short DeviceEnabled;				/* Bytes 19-20 */
  unsigned short WidePermitted;				/* Bytes 21-22 */
  unsigned short FastPermitted;				/* Bytes 23-24 */
  unsigned short SynchronousPermitted;			/* Bytes 25-26 */
  unsigned short DisconnectPermitted;			/* Bytes 27-28 */
  unsigned short SendStartUnitCommand;			/* Bytes 29-30 */
  unsigned short IgnoreInBIOSScan;			/* Bytes 31-32 */
  unsigned char PCIInterruptPin:2;			/* Byte 33 Bits 0-1 */
  unsigned char HostAdapterIOPortAddress:2;		/* Byte 33 Bits 2-3 */
  boolean StrictRoundRobinModeEnabled:1;		/* Byte 33 Bit 4 */
  boolean VESABusSpeedGreaterThan33MHz:1;		/* Byte 33 Bit 5 */
  boolean VESABurstWriteEnabled:1;			/* Byte 33 Bit 6 */
  boolean VESABurstReadEnabled:1;			/* Byte 33 Bit 7 */
  unsigned short UltraPermitted;			/* Bytes 34-35 */
  unsigned int :32;					/* Bytes 36-39 */
  unsigned char :8;					/* Byte 40 */
  unsigned char AutoSCSIMaximumLUN;			/* Byte 41 */
  boolean :1;						/* Byte 42 Bit 0 */
  boolean SCAM_Dominant:1;				/* Byte 42 Bit 1 */
  boolean SCAM_Enabled:1;				/* Byte 42 Bit 2 */
  boolean SCAM_Level2:1;				/* Byte 42 Bit 3 */
  unsigned char :4;					/* Byte 42 Bits 4-7 */
  boolean INT13ExtensionEnabled:1;			/* Byte 43 Bit 0 */
  boolean :1;						/* Byte 43 Bit 1 */
  boolean CDROMBootEnabled:1;				/* Byte 43 Bit 2 */
  unsigned char :5;					/* Byte 43 Bits 3-7 */
  unsigned char BootTargetID:4;				/* Byte 44 Bits 0-3 */
  unsigned char BootChannel:4;				/* Byte 44 Bits 4-7 */
  unsigned char ForceBusDeviceScanningOrder:1;		/* Byte 45 Bit 0 */
  unsigned char :7;					/* Byte 45 Bits 1-7 */
  unsigned short NonTaggedToAlternateLUNPermitted;	/* Bytes 46-47 */
  unsigned short RenegotiateSyncAfterCheckCondition;	/* Bytes 48-49 */
  unsigned char Reserved[10];				/* Bytes 50-59 */
  unsigned char ManufacturingDiagnostic[2];		/* Bytes 60-61 */
  unsigned short Checksum;				/* Bytes 62-63 */
}
__attribute__ ((packed))
BusLogic_AutoSCSIData_T;


/*
  Define the Host Adapter Local RAM Auto SCSI Byte 45 structure.
*/

typedef struct BusLogic_AutoSCSIByte45
{
  unsigned char ForceBusDeviceScanningOrder:1;		/* Bit 0 */
  unsigned char :7;					/* Bits 1-7 */
}
BusLogic_AutoSCSIByte45_T;


/*
  Define the Host Adapter Local RAM BIOS Drive Map Byte structure.
*/

#define BusLogic_BIOS_DriveMapOffset		17

typedef struct BusLogic_BIOSDriveMapByte
{
  unsigned char TargetIDBit3:1;				/* Bit 0 */
  unsigned char :2;					/* Bits 1-2 */
  BusLogic_BIOS_DiskGeometryTranslation_T DiskGeometry:2; /* Bits 3-4 */
  unsigned char TargetID:3;				/* Bits 5-7 */
}
BusLogic_BIOSDriveMapByte_T;


/*
  Define the Modify I/O Address request type.  On PCI Host Adapters, the
  Modify I/O Address command allows modification of the ISA compatible I/O
  Address that the Host Adapter responds to; it does not affect the PCI
  compliant I/O Address assigned at system initialization.
*/

typedef BusLogic_ISACompatibleIOPort_T BusLogic_ModifyIOAddressRequest_T;


/*
  Define the Set CCB Format request type.  Extended LUN Format CCBs are
  necessary to support more than 8 Logical Units per Target Device.
*/

typedef enum BusLogic_SetCCBFormatRequest
{
  BusLogic_LegacyLUNFormatCCB =			0,
  BusLogic_ExtendedLUNFormatCCB =		1
}
__attribute__ ((packed))
BusLogic_SetCCBFormatRequest_T;


/*
  Define the Requested Reply Length type used by the Inquire Setup Information,
  Inquire Host Adapter Model Number, Inquire Synchronous Period, and Inquire
  Extended Setup Information commands.
*/

typedef unsigned char BusLogic_RequestedReplyLength_T;


/*
  Define the Outgoing Mailbox Action Codes.
*/

typedef enum
{
  BusLogic_OutgoingMailboxFree =		0x00,
  BusLogic_MailboxStartCommand =		0x01,
  BusLogic_MailboxAbortCommand =		0x02
}
__attribute__ ((packed))
BusLogic_ActionCode_T;


/*
  Define the Incoming Mailbox Completion Codes.  The MultiMaster Firmware
  only uses codes 0 - 4.  The FlashPoint SCCB Manager has no mailboxes, so
  completion codes are stored in the CCB; it only uses codes 1, 2, 4, and 5.
*/

typedef enum
{
  BusLogic_IncomingMailboxFree =		0x00,
  BusLogic_CommandCompletedWithoutError =	0x01,
  BusLogic_CommandAbortedAtHostRequest =	0x02,
  BusLogic_AbortedCommandNotFound =		0x03,
  BusLogic_CommandCompletedWithError =		0x04,
  BusLogic_InvalidCCB =				0x05
}
__attribute__ ((packed))
BusLogic_CompletionCode_T;


/*
  Define the Command Control Block (CCB) Opcodes.
*/

typedef enum
{
  BusLogic_InitiatorCCB =			0x00,
  BusLogic_TargetCCB =				0x01,
  BusLogic_InitiatorCCB_ScatterGather =		0x02,
  BusLogic_InitiatorCCB_ResidualDataLength =	0x03,
  BusLogic_InitiatorCCB_ScatterGatherResidual =	0x04,
  BusLogic_BusDeviceReset =			0x81
}
__attribute__ ((packed))
BusLogic_CCB_Opcode_T;


/*
  Define the CCB Data Direction Codes.
*/

typedef enum
{
  BusLogic_UncheckedDataTransfer =		0,
  BusLogic_DataInLengthChecked =		1,
  BusLogic_DataOutLengthChecked =		2,
  BusLogic_NoDataTransfer =			3
}
BusLogic_DataDirection_T;


/*
  Define the Host Adapter Status Codes.  The MultiMaster Firmware does not
  return status code 0x0C; it uses 0x12 for both overruns and underruns.
*/

typedef enum
{
  BusLogic_CommandCompletedNormally =		0x00,
  BusLogic_LinkedCommandCompleted =		0x0A,
  BusLogic_LinkedCommandCompletedWithFlag =	0x0B,
  BusLogic_DataUnderRun =			0x0C,
  BusLogic_SCSISelectionTimeout =		0x11,
  BusLogic_DataOverRun =			0x12,
  BusLogic_UnexpectedBusFree =			0x13,
  BusLogic_InvalidBusPhaseRequested =		0x14,
  BusLogic_InvalidOutgoingMailboxActionCode =	0x15,
  BusLogic_InvalidCommandOperationCode =	0x16,
  BusLogic_LinkedCCBhasInvalidLUN =		0x17,
  BusLogic_InvalidCommandParameter =		0x1A,
  BusLogic_AutoRequestSenseFailed =		0x1B,
  BusLogic_TaggedQueuingMessageRejected =	0x1C,
  BusLogic_UnsupportedMessageReceived =		0x1D,
  BusLogic_HostAdapterHardwareFailed =		0x20,
  BusLogic_TargetFailedResponseToATN =		0x21,
  BusLogic_HostAdapterAssertedRST =		0x22,
  BusLogic_OtherDeviceAssertedRST =		0x23,
  BusLogic_TargetDeviceReconnectedImproperly =	0x24,
  BusLogic_HostAdapterAssertedBusDeviceReset =	0x25,
  BusLogic_AbortQueueGenerated =		0x26,
  BusLogic_HostAdapterSoftwareError =		0x27,
  BusLogic_HostAdapterHardwareTimeoutError =	0x30,
  BusLogic_SCSIParityErrorDetected =		0x34
}
__attribute__ ((packed))
BusLogic_HostAdapterStatus_T;


/*
  Define the SCSI Target Device Status Codes.
*/

typedef enum
{
  BusLogic_OperationGood =			0x00,
  BusLogic_CheckCondition =			0x02,
  BusLogic_DeviceBusy =				0x08
}
__attribute__ ((packed))
BusLogic_TargetDeviceStatus_T;


/*
  Define the Queue Tag Codes.
*/

typedef enum
{
  BusLogic_SimpleQueueTag =			0,
  BusLogic_HeadOfQueueTag =			1,
  BusLogic_OrderedQueueTag =			2,
  BusLogic_ReservedQT =				3
}
BusLogic_QueueTag_T;


/*
  Define the SCSI Command Descriptor Block (CDB).
*/

#define BusLogic_CDB_MaxLength			12

typedef unsigned char SCSI_CDB_T[BusLogic_CDB_MaxLength];


/*
  Define the Scatter/Gather Segment structure required by the MultiMaster
  Firmware Interface and the FlashPoint SCCB Manager.
*/

typedef struct BusLogic_ScatterGatherSegment
{
  BusLogic_ByteCount_T SegmentByteCount;		/* Bytes 0-3 */
  BusLogic_BusAddress_T SegmentDataPointer;		/* Bytes 4-7 */
}
BusLogic_ScatterGatherSegment_T;


/*
  Define the Driver CCB Status Codes.
*/

typedef enum
{
  BusLogic_CCB_Free =				0,
  BusLogic_CCB_Active =				1,
  BusLogic_CCB_Completed =			2,
  BusLogic_CCB_Reset =				3
}
__attribute__ ((packed))
BusLogic_CCB_Status_T;


/*
  Define the 32 Bit Mode Command Control Block (CCB) structure.  The first 40
  bytes are defined by and common to both the MultiMaster Firmware and the
  FlashPoint SCCB Manager.  The next 60 bytes are defined by the FlashPoint
  SCCB Manager.  The remaining components are defined by the Linux BusLogic
  Driver.  Extended LUN Format CCBs differ from Legacy LUN Format 32 Bit Mode
  CCBs only in having the TagEnable and QueueTag fields moved from byte 17 to
  byte 1, and the Logical Unit field in byte 17 expanded to 6 bits.  In theory,
  Extended LUN Format CCBs can support up to 64 Logical Units, but in practice
  many devices will respond improperly to Logical Units between 32 and 63, and
  the SCSI-2 specification defines Bit 5 as LUNTAR.  Extended LUN Format CCBs
  are used by recent versions of the MultiMaster Firmware, as well as by the
  FlashPoint SCCB Manager; the FlashPoint SCCB Manager only supports 32 Logical
  Units.  Since 64 Logical Units are unlikely to be needed in practice, and
  since they are problematic for the above reasons, and since limiting them to
  5 bits simplifies the CCB structure definition, this driver only supports
  32 Logical Units per Target Device.
*/

typedef struct BusLogic_CCB
{
  /*
    MultiMaster Firmware and FlashPoint SCCB Manager Common Portion.
  */
  BusLogic_CCB_Opcode_T Opcode;				/* Byte 0 */
  unsigned char :3;					/* Byte 1 Bits 0-2 */
  BusLogic_DataDirection_T DataDirection:2;		/* Byte 1 Bits 3-4 */
  boolean TagEnable:1;					/* Byte 1 Bit 5 */
  BusLogic_QueueTag_T QueueTag:2;			/* Byte 1 Bits 6-7 */
  unsigned char CDB_Length;				/* Byte 2 */
  unsigned char SenseDataLength;			/* Byte 3 */
  BusLogic_ByteCount_T DataLength;			/* Bytes 4-7 */
  BusLogic_BusAddress_T DataPointer;			/* Bytes 8-11 */
  unsigned char :8;					/* Byte 12 */
  unsigned char :8;					/* Byte 13 */
  BusLogic_HostAdapterStatus_T HostAdapterStatus;	/* Byte 14 */
  BusLogic_TargetDeviceStatus_T TargetDeviceStatus;	/* Byte 15 */
  unsigned char TargetID;				/* Byte 16 */
  unsigned char LogicalUnit:5;				/* Byte 17 Bits 0-4 */
  boolean LegacyTagEnable:1;				/* Byte 17 Bit 5 */
  BusLogic_QueueTag_T LegacyQueueTag:2;			/* Byte 17 Bits 6-7 */
  SCSI_CDB_T CDB;					/* Bytes 18-29 */
  unsigned char :8;					/* Byte 30 */
  unsigned char :8;					/* Byte 31 */
  unsigned int :32;					/* Bytes 32-35 */
  BusLogic_BusAddress_T SenseDataPointer;		/* Bytes 36-39 */
  /*
    FlashPoint SCCB Manager Defined Portion.
  */
  void (*CallbackFunction)(struct BusLogic_CCB *);	/* Bytes 40-43 */
  BusLogic_Base_Address_T BaseAddress;			/* Bytes 44-47 */
  BusLogic_CompletionCode_T CompletionCode;		/* Byte 48 */
#ifndef CONFIG_SCSI_OMIT_FLASHPOINT
  unsigned char :8;					/* Byte 49 */
  unsigned short OS_Flags;				/* Bytes 50-51 */
  unsigned char Private[48];				/* Bytes 52-99 */
#endif
  /*
    BusLogic Linux Driver Defined Portion.
  */
  dma_addr_t AllocationGroupHead;
  unsigned int AllocationGroupSize;
  BusLogic_BusAddress_T DMA_Handle;
  BusLogic_CCB_Status_T Status;
  unsigned long SerialNumber;
  SCSI_Command_T *Command;
  struct BusLogic_HostAdapter *HostAdapter;
  struct BusLogic_CCB *Next;
  struct BusLogic_CCB *NextAll;
  BusLogic_ScatterGatherSegment_T
    ScatterGatherList[BusLogic_ScatterGatherLimit];
}
BusLogic_CCB_T;


/*
  Define the 32 Bit Mode Outgoing Mailbox structure.
*/

typedef struct BusLogic_OutgoingMailbox
{
  BusLogic_BusAddress_T CCB;				/* Bytes 0-3 */
  unsigned int :24;					/* Bytes 4-6 */
  BusLogic_ActionCode_T ActionCode;			/* Byte 7 */
}
BusLogic_OutgoingMailbox_T;


/*
  Define the 32 Bit Mode Incoming Mailbox structure.
*/

typedef struct BusLogic_IncomingMailbox
{
  BusLogic_BusAddress_T CCB;				/* Bytes 0-3 */
  BusLogic_HostAdapterStatus_T HostAdapterStatus;	/* Byte 4 */
  BusLogic_TargetDeviceStatus_T TargetDeviceStatus;	/* Byte 5 */
  unsigned char :8;					/* Byte 6 */
  BusLogic_CompletionCode_T CompletionCode;		/* Byte 7 */
}
BusLogic_IncomingMailbox_T;


/*
  Define the BusLogic Driver Options structure.
*/

typedef struct BusLogic_DriverOptions
{
  unsigned short TaggedQueuingPermitted;
  unsigned short TaggedQueuingPermittedMask;
  unsigned short BusSettleTime;
  BusLogic_LocalOptions_T LocalOptions;
  unsigned char CommonQueueDepth;
  unsigned char QueueDepth[BusLogic_MaxTargetDevices];
  BusLogic_ErrorRecoveryStrategy_T
    ErrorRecoveryStrategy[BusLogic_MaxTargetDevices];
}
BusLogic_DriverOptions_T;


/*
  Define the Host Adapter Target Flags structure.
*/

typedef struct BusLogic_TargetFlags
{
  boolean TargetExists:1;
  boolean TaggedQueuingSupported:1;
  boolean WideTransfersSupported:1;
  boolean TaggedQueuingActive:1;
  boolean WideTransfersActive:1;
  boolean CommandSuccessfulFlag:1;
  boolean TargetInfoReported:1;
}
BusLogic_TargetFlags_T;


/*
  Define the Host Adapter Target Statistics structure.
*/

#define BusLogic_SizeBuckets			10

typedef unsigned int BusLogic_CommandSizeBuckets_T[BusLogic_SizeBuckets];

typedef struct BusLogic_TargetStatistics
{
  unsigned int CommandsAttempted;
  unsigned int CommandsCompleted;
  unsigned int ReadCommands;
  unsigned int WriteCommands;
  BusLogic_ByteCounter_T TotalBytesRead;
  BusLogic_ByteCounter_T TotalBytesWritten;
  BusLogic_CommandSizeBuckets_T ReadCommandSizeBuckets;
  BusLogic_CommandSizeBuckets_T WriteCommandSizeBuckets;
  unsigned short CommandAbortsRequested;
  unsigned short CommandAbortsAttempted;
  unsigned short CommandAbortsCompleted;
  unsigned short BusDeviceResetsRequested;
  unsigned short BusDeviceResetsAttempted;
  unsigned short BusDeviceResetsCompleted;
  unsigned short HostAdapterResetsRequested;
  unsigned short HostAdapterResetsAttempted;
  unsigned short HostAdapterResetsCompleted;
}
BusLogic_TargetStatistics_T;


/*
  Define the FlashPoint Card Handle data type.
*/

#define FlashPoint_BadCardHandle		0xFFFFFFFF

typedef unsigned int FlashPoint_CardHandle_T;


/*
  Define the FlashPoint Information structure.  This structure is defined
  by the FlashPoint SCCB Manager.
*/

typedef struct FlashPoint_Info
{
  BusLogic_Base_Address_T BaseAddress;			/* Bytes 0-3 */
  boolean Present;					/* Byte 4 */
  unsigned char IRQ_Channel;				/* Byte 5 */
  unsigned char SCSI_ID;				/* Byte 6 */
  unsigned char SCSI_LUN;				/* Byte 7 */
  unsigned short FirmwareRevision;			/* Bytes 8-9 */
  unsigned short SynchronousPermitted;			/* Bytes 10-11 */
  unsigned short FastPermitted;				/* Bytes 12-13 */
  unsigned short UltraPermitted;			/* Bytes 14-15 */
  unsigned short DisconnectPermitted;			/* Bytes 16-17 */
  unsigned short WidePermitted;				/* Bytes 18-19 */
  boolean ParityCheckingEnabled:1;			/* Byte 20 Bit 0 */
  boolean HostWideSCSI:1;				/* Byte 20 Bit 1 */
  boolean HostSoftReset:1;				/* Byte 20 Bit 2 */
  boolean ExtendedTranslationEnabled:1;			/* Byte 20 Bit 3 */
  boolean LowByteTerminated:1;				/* Byte 20 Bit 4 */
  boolean HighByteTerminated:1;				/* Byte 20 Bit 5 */
  boolean ReportDataUnderrun:1;				/* Byte 20 Bit 6 */
  boolean SCAM_Enabled:1;				/* Byte 20 Bit 7 */
  boolean SCAM_Level2:1;				/* Byte 21 Bit 0 */
  unsigned char :7;					/* Byte 21 Bits 1-7 */
  unsigned char Family;					/* Byte 22 */
  unsigned char BusType;				/* Byte 23 */
  unsigned char ModelNumber[3];				/* Bytes 24-26 */
  unsigned char RelativeCardNumber;			/* Byte 27 */
  unsigned char Reserved[4];				/* Bytes 28-31 */
  unsigned int OS_Reserved;				/* Bytes 32-35 */
  unsigned char TranslationInfo[4];			/* Bytes 36-39 */
  unsigned int Reserved2[5];				/* Bytes 40-59 */
  unsigned int SecondaryRange;				/* Bytes 60-63 */
}
FlashPoint_Info_T;


/*
  Define the BusLogic Driver Host Adapter structure.
*/

typedef struct BusLogic_HostAdapter
{
  SCSI_Host_T *SCSI_Host;
  PCI_Device_T *PCI_Device;
  BusLogic_HostAdapterType_T HostAdapterType;
  BusLogic_HostAdapterBusType_T HostAdapterBusType;
  BusLogic_IO_Address_T IO_Address;
  BusLogic_PCI_Address_T PCI_Address;
  unsigned short AddressCount;
  unsigned char HostNumber;
  unsigned char ModelName[9];
  unsigned char FirmwareVersion[6];
  unsigned char FullModelName[18];
  unsigned char Bus;
  unsigned char Device;
  unsigned char IRQ_Channel;
  unsigned char DMA_Channel;
  unsigned char SCSI_ID;
  boolean IRQ_ChannelAcquired:1;
  boolean DMA_ChannelAcquired:1;
  boolean ExtendedTranslationEnabled:1;
  boolean ParityCheckingEnabled:1;
  boolean BusResetEnabled:1;
  boolean LevelSensitiveInterrupt:1;
  boolean HostWideSCSI:1;
  boolean HostDifferentialSCSI:1;
  boolean HostSupportsSCAM:1;
  boolean HostUltraSCSI:1;
  boolean ExtendedLUNSupport:1;
  boolean TerminationInfoValid:1;
  boolean LowByteTerminated:1;
  boolean HighByteTerminated:1;
  boolean BounceBuffersRequired:1;
  boolean StrictRoundRobinModeSupport:1;
  boolean SCAM_Enabled:1;
  boolean SCAM_Level2:1;
  boolean HostAdapterInitialized:1;
  boolean HostAdapterExternalReset:1;
  boolean HostAdapterInternalError:1;
  boolean ProcessCompletedCCBsActive;
  volatile boolean HostAdapterCommandCompleted;
  unsigned short HostAdapterScatterGatherLimit;
  unsigned short DriverScatterGatherLimit;
  unsigned short MaxTargetDevices;
  unsigned short MaxLogicalUnits;
  unsigned short MailboxCount;
  unsigned short InitialCCBs;
  unsigned short IncrementalCCBs;
  unsigned short AllocatedCCBs;
  unsigned short DriverQueueDepth;
  unsigned short HostAdapterQueueDepth;
  unsigned short UntaggedQueueDepth;
  unsigned short CommonQueueDepth;
  unsigned short BusSettleTime;
  unsigned short SynchronousPermitted;
  unsigned short FastPermitted;
  unsigned short UltraPermitted;
  unsigned short WidePermitted;
  unsigned short DisconnectPermitted;
  unsigned short TaggedQueuingPermitted;
  unsigned short ExternalHostAdapterResets;
  unsigned short HostAdapterInternalErrors;
  unsigned short TargetDeviceCount;
  unsigned short MessageBufferLength;
  BusLogic_BusAddress_T BIOS_Address;
  BusLogic_DriverOptions_T *DriverOptions;
  FlashPoint_Info_T FlashPointInfo;
  FlashPoint_CardHandle_T CardHandle;
  struct BusLogic_HostAdapter *Next;
  BusLogic_CCB_T *All_CCBs;
  BusLogic_CCB_T *Free_CCBs;
  BusLogic_CCB_T *FirstCompletedCCB;
  BusLogic_CCB_T *LastCompletedCCB;
  BusLogic_CCB_T *BusDeviceResetPendingCCB[BusLogic_MaxTargetDevices];
  BusLogic_ErrorRecoveryStrategy_T
    ErrorRecoveryStrategy[BusLogic_MaxTargetDevices];
  BusLogic_TargetFlags_T TargetFlags[BusLogic_MaxTargetDevices];
  unsigned char QueueDepth[BusLogic_MaxTargetDevices];
  unsigned char SynchronousPeriod[BusLogic_MaxTargetDevices];
  unsigned char SynchronousOffset[BusLogic_MaxTargetDevices];
  unsigned char ActiveCommands[BusLogic_MaxTargetDevices];
  unsigned int CommandsSinceReset[BusLogic_MaxTargetDevices];
  unsigned long LastSequencePoint[BusLogic_MaxTargetDevices];
  unsigned long LastResetAttempted[BusLogic_MaxTargetDevices];
  unsigned long LastResetCompleted[BusLogic_MaxTargetDevices];
  BusLogic_OutgoingMailbox_T *FirstOutgoingMailbox;
  BusLogic_OutgoingMailbox_T *LastOutgoingMailbox;
  BusLogic_OutgoingMailbox_T *NextOutgoingMailbox;
  BusLogic_IncomingMailbox_T *FirstIncomingMailbox;
  BusLogic_IncomingMailbox_T *LastIncomingMailbox;
  BusLogic_IncomingMailbox_T *NextIncomingMailbox;
  BusLogic_TargetStatistics_T TargetStatistics[BusLogic_MaxTargetDevices];
  unsigned char *MailboxSpace;
  dma_addr_t	MailboxSpaceHandle;
  unsigned int MailboxSize;
  unsigned long CCB_Offset;
/* [BusLogic_MaxMailboxes
			     * (sizeof(BusLogic_OutgoingMailbox_T)
				+ sizeof(BusLogic_IncomingMailbox_T))]; */
  char MessageBuffer[BusLogic_MessageBufferSize];
}
BusLogic_HostAdapter_T;


/*
  Define a structure for the BIOS Disk Parameters.
*/

typedef struct BIOS_DiskParameters
{
  int Heads;
  int Sectors;
  int Cylinders;
}
BIOS_DiskParameters_T;


/*
  Define a structure for the SCSI Inquiry command results.
*/

typedef struct SCSI_Inquiry
{
  unsigned char PeripheralDeviceType:5;			/* Byte 0 Bits 0-4 */
  unsigned char PeripheralQualifier:3;			/* Byte 0 Bits 5-7 */
  unsigned char DeviceTypeModifier:7;			/* Byte 1 Bits 0-6 */
  boolean RMB:1;					/* Byte 1 Bit 7 */
  unsigned char ANSI_ApprovedVersion:3;			/* Byte 2 Bits 0-2 */
  unsigned char ECMA_Version:3;				/* Byte 2 Bits 3-5 */
  unsigned char ISO_Version:2;				/* Byte 2 Bits 6-7 */
  unsigned char ResponseDataFormat:4;			/* Byte 3 Bits 0-3 */
  unsigned char :2;					/* Byte 3 Bits 4-5 */
  boolean TrmIOP:1;					/* Byte 3 Bit 6 */
  boolean AENC:1;					/* Byte 3 Bit 7 */
  unsigned char AdditionalLength;			/* Byte 4 */
  unsigned char :8;					/* Byte 5 */
  unsigned char :8;					/* Byte 6 */
  boolean SftRe:1;					/* Byte 7 Bit 0 */
  boolean CmdQue:1;					/* Byte 7 Bit 1 */
  boolean :1;						/* Byte 7 Bit 2 */
  boolean Linked:1;					/* Byte 7 Bit 3 */
  boolean Sync:1;					/* Byte 7 Bit 4 */
  boolean WBus16:1;					/* Byte 7 Bit 5 */
  boolean WBus32:1;					/* Byte 7 Bit 6 */
  boolean RelAdr:1;					/* Byte 7 Bit 7 */
  unsigned char VendorIdentification[8];		/* Bytes 8-15 */
  unsigned char ProductIdentification[16];		/* Bytes 16-31 */
  unsigned char ProductRevisionLevel[4];		/* Bytes 32-35 */
}
SCSI_Inquiry_T;


/*
  BusLogic_AcquireHostAdapterLock acquires exclusive access to Host Adapter.
*/

static inline
void BusLogic_AcquireHostAdapterLock(BusLogic_HostAdapter_T *HostAdapter)
{
  spin_lock_irq(HostAdapter->SCSI_Host->host_lock);
}


/*
  BusLogic_ReleaseHostAdapterLock releases exclusive access to Host Adapter.
*/

static inline
void BusLogic_ReleaseHostAdapterLock(BusLogic_HostAdapter_T *HostAdapter)
{
  spin_unlock_irq(HostAdapter->SCSI_Host->host_lock);
}


/*
  BusLogic_AcquireHostAdapterLockIH acquires exclusive access to Host Adapter,
  but is only called from the interrupt handler.
*/

static inline
void BusLogic_AcquireHostAdapterLockIH(BusLogic_HostAdapter_T *HostAdapter,
				       ProcessorFlags_T *ProcessorFlags)
{
  spin_lock_irqsave(HostAdapter->SCSI_Host->host_lock, *ProcessorFlags);
}


/*
  BusLogic_ReleaseHostAdapterLockIH releases exclusive access to Host Adapter,
  but is only called from the interrupt handler.
*/

static inline
void BusLogic_ReleaseHostAdapterLockIH(BusLogic_HostAdapter_T *HostAdapter,
				       ProcessorFlags_T *ProcessorFlags)
{
  spin_unlock_irqrestore(HostAdapter->SCSI_Host->host_lock, *ProcessorFlags);
}


/*
  Define functions to provide an abstraction for reading and writing the
  Host Adapter I/O Registers.
*/

static inline
void BusLogic_SCSIBusReset(BusLogic_HostAdapter_T *HostAdapter)
{
  BusLogic_ControlRegister_T ControlRegister;
  ControlRegister.All = 0;
  ControlRegister.Bits.SCSIBusReset = true;
  outb(ControlRegister.All,
       HostAdapter->IO_Address + BusLogic_ControlRegisterOffset);
}

static inline
void BusLogic_InterruptReset(BusLogic_HostAdapter_T *HostAdapter)
{
  BusLogic_ControlRegister_T ControlRegister;
  ControlRegister.All = 0;
  ControlRegister.Bits.InterruptReset = true;
  outb(ControlRegister.All,
       HostAdapter->IO_Address + BusLogic_ControlRegisterOffset);
}

static inline
void BusLogic_SoftReset(BusLogic_HostAdapter_T *HostAdapter)
{
  BusLogic_ControlRegister_T ControlRegister;
  ControlRegister.All = 0;
  ControlRegister.Bits.SoftReset = true;
  outb(ControlRegister.All,
       HostAdapter->IO_Address + BusLogic_ControlRegisterOffset);
}

static inline
void BusLogic_HardReset(BusLogic_HostAdapter_T *HostAdapter)
{
  BusLogic_ControlRegister_T ControlRegister;
  ControlRegister.All = 0;
  ControlRegister.Bits.HardReset = true;
  outb(ControlRegister.All,
       HostAdapter->IO_Address + BusLogic_ControlRegisterOffset);
}

static inline
unsigned char BusLogic_ReadStatusRegister(BusLogic_HostAdapter_T *HostAdapter)
{
  return inb(HostAdapter->IO_Address + BusLogic_StatusRegisterOffset);
}

static inline
void BusLogic_WriteCommandParameterRegister(BusLogic_HostAdapter_T
					      *HostAdapter,
					    unsigned char Value)
{
  outb(Value,
       HostAdapter->IO_Address + BusLogic_CommandParameterRegisterOffset);
}

static inline
unsigned char BusLogic_ReadDataInRegister(BusLogic_HostAdapter_T *HostAdapter)
{
  return inb(HostAdapter->IO_Address + BusLogic_DataInRegisterOffset);
}

static inline
unsigned char BusLogic_ReadInterruptRegister(BusLogic_HostAdapter_T
					     *HostAdapter)
{
  return inb(HostAdapter->IO_Address + BusLogic_InterruptRegisterOffset);
}

static inline
unsigned char BusLogic_ReadGeometryRegister(BusLogic_HostAdapter_T
					    *HostAdapter)
{
  return inb(HostAdapter->IO_Address + BusLogic_GeometryRegisterOffset);
}


/*
  BusLogic_StartMailboxCommand issues an Execute Mailbox Command, which
  notifies the Host Adapter that an entry has been made in an Outgoing
  Mailbox.
*/

static inline
void BusLogic_StartMailboxCommand(BusLogic_HostAdapter_T *HostAdapter)
{
  BusLogic_WriteCommandParameterRegister(HostAdapter,
					 BusLogic_ExecuteMailboxCommand);
}


/*
  BusLogic_Delay waits for Seconds to elapse.
*/

static inline void BusLogic_Delay(int Seconds)
{
  mdelay(1000 * Seconds);
}


/*
  Virtual_to_Bus and Bus_to_Virtual map between Kernel Virtual Addresses
  and PCI/VLB/EISA/ISA Bus Addresses.
*/

static inline BusLogic_BusAddress_T Virtual_to_Bus(void *VirtualAddress)
{
  return (BusLogic_BusAddress_T) virt_to_bus(VirtualAddress);
}

static inline void *Bus_to_Virtual(BusLogic_BusAddress_T BusAddress)
{
  return (void *) bus_to_virt(BusAddress);
}


/*
  Virtual_to_32Bit_Virtual maps between Kernel Virtual Addresses and
  32 bit Kernel Virtual Addresses.  This avoids compilation warnings
  on 64 bit architectures.
*/

static inline
BusLogic_BusAddress_T Virtual_to_32Bit_Virtual(void *VirtualAddress)
{
  return (BusLogic_BusAddress_T) (unsigned long) VirtualAddress;
}


/*
  BusLogic_IncrementErrorCounter increments Error Counter by 1, stopping at
  65535 rather than wrapping around to 0.
*/

static inline void BusLogic_IncrementErrorCounter(unsigned short *ErrorCounter)
{
  if (*ErrorCounter < 65535) (*ErrorCounter)++;
}


/*
  BusLogic_IncrementByteCounter increments Byte Counter by Amount.
*/

static inline void BusLogic_IncrementByteCounter(BusLogic_ByteCounter_T
						   *ByteCounter,
						 unsigned int Amount)
{
  ByteCounter->Units += Amount;
  if (ByteCounter->Units > 999999999)
    {
      ByteCounter->Units -= 1000000000;
      ByteCounter->Billions++;
    }
}


/*
  BusLogic_IncrementSizeBucket increments the Bucket for Amount.
*/

static inline void BusLogic_IncrementSizeBucket(BusLogic_CommandSizeBuckets_T
						  CommandSizeBuckets,
						unsigned int Amount)
{
  int Index = 0;
  if (Amount < 8*1024)
    {
      if (Amount < 2*1024)
	Index = (Amount < 1*1024 ? 0 : 1);
      else Index = (Amount < 4*1024 ? 2 : 3);
    }
  else if (Amount < 128*1024)
    {
      if (Amount < 32*1024)
	Index = (Amount < 16*1024 ? 4 : 5);
      else Index = (Amount < 64*1024 ? 6 : 7);
    }
  else Index = (Amount < 256*1024 ? 8 : 9);
  CommandSizeBuckets[Index]++;
}


/*
  Define the version number of the FlashPoint Firmware (SCCB Manager).
*/

#define FlashPoint_FirmwareVersion		"5.02"


/*
  Define the possible return values from FlashPoint_HandleInterrupt.
*/

#define FlashPoint_NormalInterrupt		0x00
#define FlashPoint_InternalError		0xFE
#define FlashPoint_ExternalBusReset		0xFF


/*
  Define prototypes for the forward referenced BusLogic Driver
  Internal Functions.
*/

static void BusLogic_QueueCompletedCCB(BusLogic_CCB_T *);
static irqreturn_t BusLogic_InterruptHandler(int, void *, Registers_T *);
static int BusLogic_ResetHostAdapter(BusLogic_HostAdapter_T *,
				     SCSI_Command_T *, unsigned int);
static void BusLogic_Message(BusLogic_MessageLevel_T, char *,
			     BusLogic_HostAdapter_T *, ...);

/*
  Declare the Initialization Functions.
*/

static void BusLogic_AnnounceDriver(BusLogic_HostAdapter_T *) __init;
static void BusLogic_RegisterHostAdapter(BusLogic_HostAdapter_T *) __init;
static void BusLogic_UnregisterHostAdapter(BusLogic_HostAdapter_T *) __init;
static boolean BusLogic_CreateInitialCCBs(BusLogic_HostAdapter_T *) __init;
static void BusLogic_DestroyCCBs(BusLogic_HostAdapter_T *) __init;
static void BusLogic_AppendProbeAddressISA(BusLogic_IO_Address_T) __init;
static void
BusLogic_InitializeProbeInfoListISA(BusLogic_HostAdapter_T *) __init;
static void BusLogic_SortProbeInfo(BusLogic_ProbeInfo_T *, int) __init;
static int
BusLogic_InitializeMultiMasterProbeInfo(BusLogic_HostAdapter_T *) __init;
static int
BusLogic_InitializeFlashPointProbeInfo(BusLogic_HostAdapter_T *) __init;
static void BusLogic_InitializeProbeInfoList(BusLogic_HostAdapter_T *) __init;
static boolean BusLogic_Failure(BusLogic_HostAdapter_T *, char *) __init;
static boolean BusLogic_ProbeHostAdapter(BusLogic_HostAdapter_T *) __init;
static boolean BusLogic_CheckHostAdapter(BusLogic_HostAdapter_T *) __init;
static boolean
BusLogic_ReadHostAdapterConfiguration(BusLogic_HostAdapter_T *) __init;
static boolean
BusLogic_ReportHostAdapterConfiguration(BusLogic_HostAdapter_T *) __init;
static boolean BusLogic_AcquireResources(BusLogic_HostAdapter_T *) __init;
static void BusLogic_ReleaseResources(BusLogic_HostAdapter_T *) __init;
static boolean BusLogic_TargetDeviceInquiry(BusLogic_HostAdapter_T *) __init;
static void BusLogic_InitializeHostStructure(BusLogic_HostAdapter_T *,
					     SCSI_Host_T *) __init;
int BusLogic_DetectHostAdapter(SCSI_Host_Template_T *) __init;
int BusLogic_ReleaseHostAdapter(SCSI_Host_T *) __init;
static boolean BusLogic_ParseKeyword(char **, char *) __init;
static int BusLogic_ParseDriverOptions(char *) __init;
static int BusLogic_Setup(char *) __init;


#endif /* BusLogic_DriverVersion */
