/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/signrank/cdf' );
import pdf = require( './../../../../../base/dists/signrank/pdf' );
import quantile = require( './../../../../../base/dists/signrank/quantile' );

/**
* Interface describing the `signrank` namespace.
*/
interface Namespace {
	/**
	* Wilcoxon signed rank test statistic cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param n - number of observations
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 7.0, 9 );
	* // returns ~0.037
	*
	* var mycdf = ns.cdf.factory( 8 );
	* var y = mycdf( 3.9 );
	* // returns ~0.027
	*/
	cdf: typeof cdf;

	/**
	* Wilcoxon signed rank test statistic probability density function (PDF).
	*
	* @param x - input value
	* @param n - number of observations
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 7.0, 9 );
	* // returns ~0.01
	*
	* var mypdf = ns.pdf.factory( 8 );
	* var y = mypdf( 4.0 );
	* // returns ~0.008
	*/
	pdf: typeof pdf;

	/**
	* Wilcoxon signed rank test statistic quantile function.
	*
	* @param p - input value
	* @param n - number of observations
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.5, 4 );
	* // returns 5
	*
	* var myQuantile = ns.quantile.factory( 10 );
	*
	* y = myQuantile( 0.5 );
	* // returns 27
	*
	* y = myQuantile( 0.8 );
	* // returns 36
	*/
	quantile: typeof quantile;
}

/**
* Distribution of Wilcoxon signed rank test statistic.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
