/***   sort.c - for sorting   ***/

#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include "CNdata.h"
#include "CNproperty.h"
#include "CNdatatypes.h"
#include "CNround.h"
#include "CNplot3D.h"

/*****************************************************
*          Implementation of Quicksort algorithm     *
*          By Edward W. Scheckler   August 3, 1990   *
*	            EECS Dept.  UC-Berkeley	     *
*	   					     *
*	   For complicated plots, this algorithm     *
* 	   runs in approx. 2NlnN time, which is      *
*	   orders of magnitude faster then bubble    *
*	   sort methods running as N**2              *
*	   This algorithm is not very efficient for  *
*	   lists that are already mostly in order.   *
*	   In those cases, bubble sort is best       *
*	   so do_quick_sort first checks to see      *
*	   if the data is almost ordered.            *
*****************************************************/

void   CNdo_quick_sort_curves(); 
static CNcurveptr *allocate_quicksort_curve_array();
static void       quicksort_curves();
void   CNbubble_sort_curves();

void   CNdo_quick_sort_trias(); 
static CNtriaptr *allocate_quicksort_tria_array();
static void       quicksort_trias();
void   CNbubble_sort_trias();

void   CNdo_quick_sort_rects(); 
static CNrectptr *allocate_quicksort_rect_array();
static void       quicksort_rects();
void   CNbubble_sort_rects();


/*
 * SORT CURVES
 */
void CNdo_quick_sort_curves(curvehead, curvetail, view_transfo, xlog,ylog,zlog) 
CNcurveptr *curvehead, *curvetail;
CNmatrix   view_transfo;
short      xlog, ylog, zlog;
{
   CNcurveptr S,A,B,C;
   CNcoord    point,newpt;
   CNpointptr P;
   CNcurveptr *a;
   double     ztot;
   int        narray,in,sort_times;
   int        npts, sorted, longline;

   /* allocate an array for holding pointers to the curves */
   narray = CNcount_curves(*curvehead, *curvetail);
   a = allocate_quicksort_curve_array(narray);

   /* first calculate the z-ave value of all the points in the list */
   /* also set the elements of the curveptr array a                 */
   in = 0;
   for (S=(*curvehead); S!=NULL; S=S->next) {
      *(a + in) = S;
      in ++;
      ztot = 0.0;
      npts = 0;
      if (S->pointhead == NULL) continue;

      /* Check to see if the curve is joined */
      longline = CNlongline(S->pointhead, S->pointtail, 1.0e-5);

      /* Calculate the z-value */
      for (P=S->pointhead; P!=NULL; P=P->next) {

         /* If the last point is same as the first point, don't count it */
         if ((P->next == NULL) && !longline) continue;

         /* Transform the point, taking into account logarithms */
         point.x = (xlog) ? CNlog10(P->x) : P->x;
         point.y = (ylog) ? CNlog10(P->y) : P->y;
         point.z = (zlog) ? CNlog10(P->z) : P->z;
         newpt = CNtransform_point(&point,view_transfo);
         ztot = ztot + newpt.z;
         npts++;
      }
      if (npts>0) S->zave = ztot/(double)npts;
   }

   /*first check if already sorted*/
   sort_times = 0;
   do {
      sorted = CN_TRUE;
      for (S=(*curvehead); S->next!=NULL; S=S->next) {
         /* compare each adjacent pair of elements */
         B = S->next;
         if (S->zave > B->zave) {
            A = S->prev;
            C = B->next;
            if (S== *curvehead) *curvehead = B;
            if (B== *curvetail) *curvetail = S;
            if (A!=NULL) A->next = B;
            B->next = S;
            B->prev = A;
            S->next = C;
            S->prev = B;
            if (C!=NULL) C->prev = S;
            S = S->prev;
            sorted = CN_FALSE;
         }
      }
      sort_times++;
   } while (!sorted && sort_times < 5);

   /* If it's sorted, return */
   if (sorted){
      free((char *)a);
      return;
   }

   /*do the quicksort*/
   quicksort_curves(a,0,narray-1,narray-1);

   /*put the sorted array elements back into the curve list*/
   
   *curvehead = a[0];
   *curvetail = a[narray-1];
   for(in = 0; in < narray; in++){
	if(in+1 < narray){
	a[in]->next = a[in+1];
	} else {
	a[in]->next = NULL;
	}
	if(in-1 >= 0 ){
	a[in]->prev = a[in-1];
	}else {
	a[in]->prev= NULL;
	}
   }

   free((char *)a);
   
}

/* Allocate room for a bunch of curves */
static CNcurveptr *allocate_quicksort_curve_array(narr)
int narr;
{
   CNcurveptr *newptr;
   unsigned int arr_size;
   arr_size = (unsigned)(narr)*sizeof(CNcurve);
   if ((newptr = (CNcurveptr *)malloc(arr_size))==NULL) {
      (void) fprintf(stderr,"Out of memory\n");
      exit(-1);
   }

   return(newptr);
}

/* Sort out curves */
static void quicksort_curves(a,l,r,narr)
CNcurveptr *a;
int l,r,narr;
{
	int i,j;
	int p;
	int sortflag;
	CNcurveptr v,t,ta;
	/*if the list is small, do an insertion sort for speed*/
	if(r-l > 5){
		v = a[r];
		i = l - 1;
		j = r;
		do{
			do{
				i++;
			} while(i< narr &&!(a[i]->zave >= v->zave));
			do{
				j--;
			}while( j > 0 && !(a[j]->zave <= v->zave));
			t = a[i];
			a[i] = a[j];
			a[j] = t;
		}while(!(j <= i));
			a[j] = a[i];
			a[i] = a[r];
			a[r] = t;
			quicksort_curves(a,l,i-1,narr);
			quicksort_curves(a,i+1,r,narr);
	} else {
		do{
		   sortflag = 0;
		   for(p=l;p<r;p++){
		      if(a[p]->zave > a[p+1]->zave){
			  ta = a[p];
			  a[p] = a[p+1];
			  a[p+1] = ta;
			  sortflag = 1;
		      }
		   }
	        }while(sortflag != 0);
	}

}	


/*
 * Use a bubble sort to sort out curves
 */
void CNbubble_sort_curves(curvehead, curvetail, view_transfo, xlog,ylog,zlog) 
CNcurveptr *curvehead, *curvetail;
CNmatrix   view_transfo;
short      xlog, ylog, zlog;
{
   CNcurveptr S,A,B,C;
   CNcoord    point,newpt;
   CNpointptr P;
   double     ztot;
   int        npts, sorted, longline;

   /* first calculate the z-ave value of all the points in the list */
   for (S=(*curvehead); S!=NULL; S=S->next) {
      ztot = 0.0;
      npts = 0;
      if (S->pointhead == NULL) continue;

      /* Check to see if the curve is joined */
      longline = CNlongline(S->pointhead, S->pointtail, 1.0e-5);

      /* Calculate the z-value */
      for (P=S->pointhead; P!=NULL; P=P->next) {

         /* If the last point is same as the first point, don't count it */
         if ((P->next == NULL) && !longline) continue;

         /* Transform the point, taking into account logarithms */
         point.x = (xlog) ? CNlog10(P->x) : P->x;
         point.y = (ylog) ? CNlog10(P->y) : P->y;
         point.z = (zlog) ? CNlog10(P->z) : P->z;
         newpt = CNtransform_point(&point,view_transfo);
         ztot = ztot + newpt.z;
         npts++;
      }
      if (npts>0) S->zave = ztot/(double)npts;
   }

   /* 
    * now sort according to average z value 
    * the first structure should have the smallest z-value 
    */
   do {
      sorted = CN_TRUE;
      for (S=(*curvehead); S->next!=NULL; S=S->next) {
         /* compare each adjacent pair of elements */
         B = S->next;
         if (S->zave > B->zave) {
            /* A -> S -> B -> C */
            /* A -> B -> S -> C */
            A = S->prev;
            C = B->next;
            if (S== *curvehead) *curvehead = B;
            if (B== *curvetail) *curvetail = S;
            if (A!=NULL) A->next = B;
            B->next = S;
            B->prev = A;
            S->next = C;
            S->prev = B;
            if (C!=NULL) C->prev = S;
            S = S->prev;
            sorted = CN_FALSE;
         }
      }
   } while (!sorted);
}



/*
 * SORT TRIANGLES
 */
void CNdo_quick_sort_trias(triahead, triatail, view_transfo, xlog,ylog,zlog) 
CNtriaptr *triahead, *triatail;
CNmatrix   view_transfo;
short      xlog, ylog, zlog;
{
   CNtriaptr  S,A,B,C;
   CNcoord    point,newpt;
   CNtriaptr *a;
   double     ztot;
   int        narray,in,sort_times;
   int        npts, sorted;

   /* allocate an array for holding pointers to the trias */
   narray = CNcount_trias(*triahead, *triatail);
   a = allocate_quicksort_tria_array(narray);

   /* first calculate the z-ave value of all the points in the list */
   /* also set the elements of the triaptr array a                 */
   in = 0;
   for (S=(*triahead); S!=NULL; S=S->next) {
      *(a + in) = S;
      in ++;
      ztot = 0.0;
      npts = 0;

      /* Calculate the z-value of the triangle center */

      /* Transform each point, taking into account logarithms */
      point.x = (xlog) ? CNlog10(S->n1->coord->x) : S->n1->coord->x;
      point.y = (ylog) ? CNlog10(S->n1->coord->y) : S->n1->coord->y;
      point.z = (zlog) ? CNlog10(S->n1->t       ) : S->n1->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n2->coord->x) : S->n2->coord->x;
      point.y = (ylog) ? CNlog10(S->n2->coord->y) : S->n2->coord->y;
      point.z = (zlog) ? CNlog10(S->n2->t       ) : S->n2->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n3->coord->x) : S->n3->coord->x;
      point.y = (ylog) ? CNlog10(S->n3->coord->y) : S->n3->coord->y;
      point.z = (zlog) ? CNlog10(S->n3->t       ) : S->n3->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      if (npts>0) S->zave = ztot/(double)npts;
   }

   /*first check if already sorted*/
   sort_times = 0;
   do {
      sorted = CN_TRUE;
      for (S=(*triahead); S->next!=NULL; S=S->next) {
         /* compare each adjacent pair of elements */
         B = S->next;
         if (S->zave > B->zave) {
            A = S->prev;
            C = B->next;
            if (S== *triahead) *triahead = B;
            if (B== *triatail) *triatail = S;
            if (A!=NULL) A->next = B;
            B->next = S;
            B->prev = A;
            S->next = C;
            S->prev = B;
            if (C!=NULL) C->prev = S;
            S = S->prev;
            sorted = CN_FALSE;
         }
      }
      sort_times++;
   } while (!sorted && sort_times < 5);

   /* If it's sorted, return */
   if (sorted){
      free((char *)a);
      return;
   }

   /*do the quicksort*/
   quicksort_trias(a,0,narray-1,narray-1);

   /*put the sorted array elements back into the triangle list*/
   
   *triahead = a[0];
   *triatail = a[narray-1];
   for(in = 0; in < narray; in++){
	if(in+1 < narray){
	a[in]->next = a[in+1];
	} else {
	a[in]->next = NULL;
	}
	if(in-1 >= 0 ){
	a[in]->prev = a[in-1];
	}else {
	a[in]->prev= NULL;
	}
   }

   free((char *)a);
   
}

/* Allocate room for a bunch of triangles */
static CNtriaptr *allocate_quicksort_tria_array(narr)
int narr;
{
   CNtriaptr *newptr;
   unsigned int arr_size;
   arr_size = (unsigned)(narr)*sizeof(CNtria);
   if ((newptr = (CNtriaptr *)malloc(arr_size))==NULL) {
      (void) fprintf(stderr,"Out of memory\n");
      exit(-1);
   }
   return(newptr);
}

/* Sort out trias */
static void quicksort_trias(a,l,r,narr)
CNtriaptr *a;
int l,r,narr;
{
	int i,j;
	int p;
	int sortflag;
	CNtriaptr v,t,ta;
	/*if the list is small, do an insertion sort for speed*/
	if(r-l > 5){
		v = a[r];
		i = l - 1;
		j = r;
		do{
			do{
				i++;
			} while(i< narr &&!(a[i]->zave >= v->zave));
			do{
				j--;
			}while( j > 0 && !(a[j]->zave <= v->zave));
			t = a[i];
			a[i] = a[j];
			a[j] = t;
		}while(!(j <= i));
			a[j] = a[i];
			a[i] = a[r];
			a[r] = t;
			quicksort_trias(a,l,i-1,narr);
			quicksort_trias(a,i+1,r,narr);
	} else {
		do{
		   sortflag = 0;
		   for(p=l;p<r;p++){
		      if(a[p]->zave > a[p+1]->zave){
			  ta = a[p];
			  a[p] = a[p+1];
			  a[p+1] = ta;
			  sortflag = 1;
		      }
		   }
	        }while(sortflag != 0);
	}

}	


/*
 * Use a bubble sort to sort out trias
 */
void CNbubble_sort_trias(triahead, triatail, view_transfo, xlog,ylog,zlog) 
CNtriaptr *triahead, *triatail;
CNmatrix   view_transfo;
short      xlog, ylog, zlog;
{
   CNtriaptr  S,A,B,C;
   CNcoord    point,newpt;
   double     ztot;
   int        npts, sorted;

   /* first calculate the z-ave value of all the points in the list */
   for (S=(*triahead); S!=NULL; S=S->next) {
      ztot = 0.0;
      npts = 0;

      /* Calculate the z-value of the triangle center */

      /* Transform each point, taking into account logarithms */
      point.x = (xlog) ? CNlog10(S->n1->coord->x) : S->n1->coord->x;
      point.y = (ylog) ? CNlog10(S->n1->coord->y) : S->n1->coord->y;
      point.z = (zlog) ? CNlog10(S->n1->t       ) : S->n1->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n2->coord->x) : S->n2->coord->x;
      point.y = (ylog) ? CNlog10(S->n2->coord->y) : S->n2->coord->y;
      point.z = (zlog) ? CNlog10(S->n2->t       ) : S->n2->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n3->coord->x) : S->n3->coord->x;
      point.y = (ylog) ? CNlog10(S->n3->coord->y) : S->n3->coord->y;
      point.z = (zlog) ? CNlog10(S->n3->t       ) : S->n3->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      if (npts>0) S->zave = ztot/(double)npts;
   }

   /* 
    * now sort according to average z value 
    * the first structure should have the smallest z-value 
    */
   do {
      sorted = CN_TRUE;
      for (S=(*triahead); S->next!=NULL; S=S->next) {
         /* compare each adjacent pair of elements */
         B = S->next;
         if (S->zave > B->zave) {
            /* A -> S -> B -> C */
            /* A -> B -> S -> C */
            A = S->prev;
            C = B->next;
            if (S== *triahead) *triahead = B;
            if (B== *triatail) *triatail = S;
            if (A!=NULL) A->next = B;
            B->next = S;
            B->prev = A;
            S->next = C;
            S->prev = B;
            if (C!=NULL) C->prev = S;
            S = S->prev;
            sorted = CN_FALSE;
         }
      }
   } while (!sorted);
}



/*
 * SORT RECTANGLES
 */
void CNdo_quick_sort_rects(recthead, recttail, view_transfo, xlog,ylog,zlog) 
CNrectptr *recthead, *recttail;
CNmatrix   view_transfo;
short      xlog, ylog, zlog;
{
   CNrectptr  S,A,B,C;
   CNcoord    point,newpt;
   CNrectptr *a;
   double     ztot;
   int        narray,in,sort_times;
   int        npts, sorted;

   /* allocate an array for holding pointers to the rects */
   narray = CNcount_rects(*recthead, *recttail);
   a = allocate_quicksort_rect_array(narray);

   /* first calculate the z-ave value of all the points in the list */
   /* also set the elements of the rectptr array a                 */
   in = 0;
   for (S=(*recthead); S!=NULL; S=S->next) {
      *(a + in) = S;
      in ++;
      ztot = 0.0;
      npts = 0;

      /* Calculate the z-value of the rectangle center */

      /* Transform each point, taking into account logarithms */
      point.x = (xlog) ? CNlog10(S->n1->coord->x) : S->n1->coord->x;
      point.y = (ylog) ? CNlog10(S->n1->coord->y) : S->n1->coord->y;
      point.z = (zlog) ? CNlog10(S->n1->t       ) : S->n1->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n2->coord->x) : S->n2->coord->x;
      point.y = (ylog) ? CNlog10(S->n2->coord->y) : S->n2->coord->y;
      point.z = (zlog) ? CNlog10(S->n2->t       ) : S->n2->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n3->coord->x) : S->n3->coord->x;
      point.y = (ylog) ? CNlog10(S->n3->coord->y) : S->n3->coord->y;
      point.z = (zlog) ? CNlog10(S->n3->t       ) : S->n3->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n4->coord->x) : S->n4->coord->x;
      point.y = (ylog) ? CNlog10(S->n4->coord->y) : S->n4->coord->y;
      point.z = (zlog) ? CNlog10(S->n4->t       ) : S->n4->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      if (npts>0) S->zave = ztot/(double)npts;
   }

   /*first check if already sorted*/
   sort_times = 0;
   do {
      sorted = CN_TRUE;
      for (S=(*recthead); S->next!=NULL; S=S->next) {
         /* compare each adjacent pair of elements */
         B = S->next;
         if (S->zave > B->zave) {
            A = S->prev;
            C = B->next;
            if (S== *recthead) *recthead = B;
            if (B== *recttail) *recttail = S;
            if (A!=NULL) A->next = B;
            B->next = S;
            B->prev = A;
            S->next = C;
            S->prev = B;
            if (C!=NULL) C->prev = S;
            S = S->prev;
            sorted = CN_FALSE;
         }
      }
      sort_times++;
   } while (!sorted && sort_times < 5);

   /* If it's sorted, return */
   if (sorted){
      free((char *)a);
      return;
   }

   /*do the quicksort*/
   quicksort_rects(a,0,narray-1,narray-1);

   /*put the sorted array elements back into the rectangle list*/
   
   *recthead = a[0];
   *recttail = a[narray-1];
   for(in = 0; in < narray; in++){
	if(in+1 < narray){
	a[in]->next = a[in+1];
	} else {
	a[in]->next = NULL;
	}
	if(in-1 >= 0 ){
	a[in]->prev = a[in-1];
	}else {
	a[in]->prev= NULL;
	}
   }

   free((char *)a);
   
}

/* Allocate room for a bunch of rectangles */
static CNrectptr *allocate_quicksort_rect_array(narr)
int narr;
{
   CNrectptr *newptr;
   unsigned int arr_size;
   arr_size = (unsigned)(narr)*sizeof(CNrect);
   if ((newptr = (CNrectptr *)malloc(arr_size))==NULL) {
      (void) fprintf(stderr,"Out of memory\n");
      exit(-1);
   }
   return(newptr);
}

/* Sort out rects */
static void quicksort_rects(a,l,r,narr)
CNrectptr *a;
int l,r,narr;
{
	int i,j;
	int p;
	int sortflag;
	CNrectptr v,t,ta;
	/*if the list is small, do an insertion sort for speed*/
	if(r-l > 5){
		v = a[r];
		i = l - 1;
		j = r;
		do{
			do{
				i++;
			} while(i< narr &&!(a[i]->zave >= v->zave));
			do{
				j--;
			}while( j > 0 && !(a[j]->zave <= v->zave));
			t = a[i];
			a[i] = a[j];
			a[j] = t;
		}while(!(j <= i));
			a[j] = a[i];
			a[i] = a[r];
			a[r] = t;
			quicksort_rects(a,l,i-1,narr);
			quicksort_rects(a,i+1,r,narr);
	} else {
		do{
		   sortflag = 0;
		   for(p=l;p<r;p++){
		      if(a[p]->zave > a[p+1]->zave){
			  ta = a[p];
			  a[p] = a[p+1];
			  a[p+1] = ta;
			  sortflag = 1;
		      }
		   }
	        }while(sortflag != 0);
	}

}	


/*
 * Use a bubble sort to sort out rects
 */
void CNbubble_sort_rects(recthead, recttail, view_transfo, xlog,ylog,zlog) 
CNrectptr *recthead, *recttail;
CNmatrix   view_transfo;
short      xlog, ylog, zlog;
{
   CNrectptr  S,A,B,C;
   CNcoord    point,newpt;
   double     ztot;
   int        npts, sorted;

   /* first calculate the z-ave value of all the points in the list */
   for (S=(*recthead); S!=NULL; S=S->next) {
      ztot = 0.0;
      npts = 0;

      /* Transform each point, taking into account logarithms */
      point.x = (xlog) ? CNlog10(S->n1->coord->x) : S->n1->coord->x;
      point.y = (ylog) ? CNlog10(S->n1->coord->y) : S->n1->coord->y;
      point.z = (zlog) ? CNlog10(S->n1->t       ) : S->n1->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n2->coord->x) : S->n2->coord->x;
      point.y = (ylog) ? CNlog10(S->n2->coord->y) : S->n2->coord->y;
      point.z = (zlog) ? CNlog10(S->n2->t       ) : S->n2->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n3->coord->x) : S->n3->coord->x;
      point.y = (ylog) ? CNlog10(S->n3->coord->y) : S->n3->coord->y;
      point.z = (zlog) ? CNlog10(S->n3->t       ) : S->n3->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      point.x = (xlog) ? CNlog10(S->n4->coord->x) : S->n4->coord->x;
      point.y = (ylog) ? CNlog10(S->n4->coord->y) : S->n4->coord->y;
      point.z = (zlog) ? CNlog10(S->n4->t       ) : S->n4->t;
      newpt = CNtransform_point(&point,view_transfo);
      ztot = ztot + newpt.z;
      npts++;

      if (npts>0) S->zave = ztot/(double)npts;
   }

   /* 
    * now sort according to average z value 
    * the first structure should have the smallest z-value 
    */
   do {
      sorted = CN_TRUE;
      for (S=(*recthead); S->next!=NULL; S=S->next) {
         /* compare each adjacent pair of elements */
         B = S->next;
         if (S->zave > B->zave) {
            /* A -> S -> B -> C */
            /* A -> B -> S -> C */
            A = S->prev;
            C = B->next;
            if (S== *recthead) *recthead = B;
            if (B== *recttail) *recttail = S;
            if (A!=NULL) A->next = B;
            B->next = S;
            B->prev = A;
            S->next = C;
            S->prev = B;
            if (C!=NULL) C->prev = S;
            S = S->prev;
            sorted = CN_FALSE;
         }
      }
   } while (!sorted);
}

