/* ZGV v2.2 - (c) 1993,1994 Russell Marks for improbabledesigns.
 * See README for license details.
 *
 * zgv.c - This provides the zgv file selector, and interfaces to the
 *         vga display routines (vgadisp.c)
 */


#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <setjmp.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <vga.h>

#include "zgv.h"
#include "gifeng.h"
#include "vgadisp.h"
#include "readnbkey.h"
#include "font.h"
#include "3deffects.h"
#include "handlevt.h"
#include "helppage.h"
#include "rc_config.h"
#include "rcfile.h"
#include "usejpeg.h"  /* we only use *this* for my_error_exit() */
#include "readpnm.h"	/* for the dithering - should be moved! */
#include "resizepic.h"

#include "zgvlogopck.h"


char zgvhelp[][80]={
  " ", " ",
  "? (question mark) \t this help page",
  " ",
  "up (also k or q) \t move file selection cursor up",
  "down (also j or a) \t move file selection cursor down",
  "left (also h or o) \t move file selection cursor left",
  "right (also l or p) \t move file selection cursor right",
  " ",
  "Enter \t\t display file or change directory",
  " ",
  "v \t\t\t visual selection mode on/off",
  "u \t\t\t create/update thumbnails in visual mode",
  "Ctrl-R \t\t update directory listing / redraw screen",
  "D or Delete \t\t delete file",
  " ",
#ifdef HANDS_ON_VC_CODE
  "F1 to F8 \t\t change virtual console",
  " ",
#endif
  "Esc or x \t\t exit zgv",
  ""
  };


#define fwinxpos(f) (40+(((f)-1)/YSIZ)*BARWIDTH)
#define fwinypos(f) (180+barheight*(((f)-1)%YSIZ))

/* from 18-bit RGB (as used by VGA palette) to 3:3:2 palette index */
#define MAKE332COL(r,g,b) (((r)>>3)*32+((g)>>3)*4+((b)>>4))


#define LIGHT 2
#define DARK 1
#define BLACK 15
#define MIDGREY 0

#define GDFSIZ 3
#define BARWIDTH 138
#define BAR_RESTRICT_WIDTH  BARWIDTH-6

#define COLS 600
#define MDIRSIZ 1000
#define YSIZ (285/barheight)


int idx_light,idx_medium,idx_dark,idx_black;
int updating_index=0;

int barheight;


int gcompare(void *,void *);
void showhowfar(int,int);
void exitproperly();

struct GDIR {
  char name[256];
  char isdir;            /* 0=file, 1=dir. */
  int xvw,xvh;		/* xvpic width and height, zero if none */
  } gifdir[MDIRSIZ];
int gifdirsiz;

int zgv_ttyfd,howfar_upto;
jmp_buf setjmpbuf;   /* in case someone aborts decompressing a file */
static int xv332_how_far_xpos,xv332_how_far_ypos;
static int one_file_only=0;


main(argc,argv)
int argc;
char **argv;
{
int argsleft;

vga_disabledriverreport();    /* quieten svgalib */
vga_init();
/* root permissions should now have been ditched */

pixelsize=1;
getconfig();				/* see rcfile.c... */
argsleft=parsecommandline(argc,argv);	/* ...for these two. */
if(argsleft>1) usage_help();
copyfromconfig();

cleartext();

/* do one-file-only if have arg. */
if(argsleft==1)
  load_one_file(argv[argc-1]);

screenon();
if((zgv_ttyfd=open("/dev/tty",O_RDONLY|O_NONBLOCK))>=0)
  {
  mainloop();
  close(zgv_ttyfd);
  }
screenoff();
exit(0);
}


load_one_file(filename)
char *filename;
{
hffunc hf;
  
one_file_only=1;
if(cfg.onefile_progress)
  {
  screenon();
  inithowfar();
  hf=showhowfar;
  }
else 
  {
  hf=NULL;
  fprintf(stderr,"Loading...");
  }
  
if(readpicture(filename,hf,1)!=0)
  {
  fprintf(stderr,"\rError loading file.\n");
  exit(1);
  }
else
  if(hf==NULL)
    fprintf(stderr,"\r          \r");
      
screenoff();
exit(0);
}


copyfromconfig()
{
curvgamode=cfg.videomode;
zoom=cfg.zoom;
vkludge=cfg.vkludge;
brightness=cfg.brightness;
contrast=cfg.contrast;
if((curvgamode==G320x400x256)||(curvgamode==G360x480x256))
  virtual=1;
else
  virtual=0;
}


mainloop()
{
int quit,key,curent; /* quit, key pressed, current entry */
int oldent,startfrom,oldstart,f,tmp;
char cline[100],buf[1024];

quit=0; curent=1; oldent=1;
startfrom=1;

readgifdir();
showgifdir(startfrom,0);
showbar(curent,1,startfrom);
while(!quit)
  {
  oldent=curent;
  usleep(10000);
  key=readnbkey(zgv_ttyfd);

/* for consistency between zgv.c and vgadisp.c, we don't use the old VC
 * switching code unless...
 */
#ifdef HANDS_ON_VC_CODE
  /* if it's from F1 to F8, we switch consoles */
  if((key>=RK_F1)&&(key<=RK_F8))
    if(handlevt(zgv_ttyfd,key-RK_F1+1))
      redrawall(curent,startfrom);
#endif
  
  switch(key)
    {
    case 'u':
      if(cfg.xvpic_index)
        {
        updating_index=1;
        update_xvpics();
        updating_index=0;
        redrawall(curent,startfrom);
        }
      break;
    case 'v':
      cfg.xvpic_index=!cfg.xvpic_index;
      redrawall(curent,startfrom);
      break;
    case '?':
      showhelp(zgv_ttyfd,"- KEYS FOR FILE SELECT SCREEN -",zgvhelp);
      redrawall(curent,startfrom);
      break;
    case 18: case 12:   /* ^R and ^L */
      readgifdir();
      if(curent>gifdirsiz) curent=gifdirsiz;
      oldent=curent;
      redrawall(curent,startfrom);
      break;
    case 'D': case RK_DELETE:   /* shift-D or the 'del' key */
      if(gifdir[curent].isdir) break;
      if(delete_file(gifdir[curent].name))
        {
        readgifdir();
        if(curent>gifdirsiz) curent=gifdirsiz;
        oldent=curent;
        }
      redrawall(curent,startfrom);
      break;
    case 'q': case 'k': case RK_CURSOR_UP:
      if(curent>1) curent--; break;
    case 'a': case 'j': case RK_CURSOR_DOWN:
      if(curent<gifdirsiz) curent++; break;
    case 'o': case 'h': case RK_CURSOR_LEFT:
      curent-=YSIZ;
      if(curent<1) curent=1;
      break;
    case 'p': case 'l': case RK_CURSOR_RIGHT:
      curent+=YSIZ;
      if(curent>gifdirsiz) curent=gifdirsiz;
      break;
    case RK_ENTER:
      if(permissiondenied(gifdir[curent].name))
        {
        msgbox(zgv_ttyfd,"Permission denied or file not found",
        	MSGBOXTYPE_OK,idx_light,idx_dark,idx_black);
        redrawall(curent,startfrom);
        break;
        }
      if(gifdir[curent].isdir)
        {
        showbar(curent,0,startfrom);
        showgifdir(startfrom,1);
        chdir(gifdir[curent].name);
        readgifdir();
        showgifdir(1,0); curent=oldent=startfrom=1;
        showbar(curent,1,startfrom);
        }
      else
        {
        inithowfar();
        /* save context for possible abort */
        if(setjmp(setjmpbuf))
          /* if we get here, someone aborted loading a file. */
          msgbox(zgv_ttyfd,"File view aborted",MSGBOXTYPE_OK,
			          	idx_light,idx_dark,idx_black);
        else
          {
          if((tmp=readpicture(gifdir[curent].name,showhowfar,1))!=_PIC_OK)
            showerrmessage(tmp);
          }
        redrawall(curent,startfrom);
        }
      break;
#if 0
    case 26:   /* ^Z */
      screenoff();
      raise(SIGTSTP);
      redrawall(curent,startfrom);
      break;
#endif
    case 'x': case RK_ESC:
      quit=1;
    }
    
  oldstart=startfrom;
  while(curent<startfrom)
    startfrom-=YSIZ;
  while(fwinxpos(curent-startfrom+1)+BARWIDTH>COLS)
    startfrom+=YSIZ;
  if(startfrom<1) startfrom=1;
  if(startfrom!=oldstart)
    {
    showbar(oldent,0,oldstart);
    showgifdir(oldstart,1);
    showgifdir(startfrom,0);
    showbar(curent,1,startfrom);
    }
  else  
    if(curent!=oldent)
      {
      showbar(oldent,0,startfrom);
      showbar(curent,1,startfrom);
      }
  }
}


/* this could also be file not found of course */
int permissiondenied(fname)
char *fname;
{
FILE *junk;
if((junk=fopen(fname,"rb"))==NULL)
  return(1);
fclose(junk);
return(0);
}


redrawall(curent,startfrom)
int curent,startfrom;
{
screenon();

showgifdir(startfrom,0);
showbar(curent,1,startfrom);
}

inithowfar()
{
int f;

vga_setcolor(idx_medium);
for(f=220;f<=260;f++)
  vga_drawline(100,f,540,f);
draw3dbox(100,220,540,260,2,1, idx_light,idx_dark);
draw3dbox(109,229,531,251,1,0, idx_light,idx_dark);

howfar_upto=0;
drawtext3d(250,235,2,"Decompressing - please wait",0,
					idx_light,idx_dark,idx_black);
}


void showhowfar(sofar,total)
int sofar,total;
{
int f,d;

/* we jump back to an abort message if Esc was pressed */
if(!one_file_only)
  if(readnbkey(zgv_ttyfd)==RK_ESC)
    {
    /* these routines blast the malloc'ed stuff, which has *for sure*
     * been allocated by now, because we must already be reading the file
     * in for us to get here!
     */
    aborted_file_cleanup();
    longjmp(setjmpbuf,1);
    }

if(((sofar%10)==0)||(sofar==total))
  {
  vga_setcolor(idx_light); /* we set this always in case of a VC switch */
  d=(420*sofar)/total;
  if(d>howfar_upto)
    {
    for(f=howfar_upto;f<=d;f++)
      vga_drawline(110+f,230,110+f,250);
    howfar_upto=f;
    }
  }
}


showbar(entnum,selected,startfrom)
int entnum,selected;
int startfrom;
{
char ctmp[100];
int xpos,ypos;
int siz,xt;

xpos=fwinxpos(entnum-startfrom+1);
if((xpos<1)||(xpos+BARWIDTH>COLS)) return(0);
ypos=fwinypos(entnum-startfrom+1);
prettyfile(&ctmp,&(gifdir[entnum]));

set_max_text_width(BAR_RESTRICT_WIDTH);
if(cfg.blockcursor)
  {
  /* new block-style cursor - hopefully easier to read/see. */
  if(selected)
    draw3dbox(xpos-2,ypos-2,xpos+BARWIDTH+1,ypos+barheight+1,4,1,
    	idx_dark,idx_dark);
  else
    undraw3dbox(xpos-2,ypos-2,xpos+BARWIDTH+1,ypos+barheight+1,4);
  }
else
  {
  if(selected)
    {
    draw3dbox(xpos,ypos,xpos+BARWIDTH-1,ypos+barheight-1,1,1,
    	idx_light,idx_dark);
    xt=cfg.xvpic_index?centreseltxt(xpos,GDFSIZ,ctmp):xpos+3;
    drawtext3d(xt,ypos+3,GDFSIZ,ctmp,0, idx_light,idx_dark,idx_black);
    /* box if cfg.xvpic_index and is an xvpic being used */
    if(cfg.xvpic_index && gifdir[entnum].xvw!=0)
      draw3dbox(xpos+(BARWIDTH-gifdir[entnum].xvw)/2-2,
                ypos+GDFSIZ*6+39-gifdir[entnum].xvh/2-2,
                xpos+(BARWIDTH-gifdir[entnum].xvw)/2+gifdir[entnum].xvw+1,
                ypos+GDFSIZ*6+39-gifdir[entnum].xvh/2+gifdir[entnum].xvh+1,
                1,0, idx_light,idx_dark);
    }
  else
    {
    undraw3dbox(xpos,ypos,xpos+BARWIDTH-1,ypos+barheight-1,1);
    xt=cfg.xvpic_index?centreseltxt(xpos,GDFSIZ,ctmp):xpos+3;
    undrawtext3d(xt,ypos+3,GDFSIZ,ctmp);
    vga_setcolor(idx_black);
    vgadrawtext(xt,ypos+3,GDFSIZ,ctmp);
    /* undraw box if cfg.xvpic_index and is an xvpic being used */
    if(cfg.xvpic_index && gifdir[entnum].xvw!=0)
      undraw3dbox(xpos+(BARWIDTH-gifdir[entnum].xvw)/2-2,
                  ypos+GDFSIZ*6+39-gifdir[entnum].xvh/2-2,
                  xpos+(BARWIDTH-gifdir[entnum].xvw)/2+gifdir[entnum].xvw+1,
                  ypos+GDFSIZ*6+39-gifdir[entnum].xvh/2+gifdir[entnum].xvh+1,
                  1);
    }
  }
set_max_text_width(NO_CLIP_FONT);
}


int centreseltxt(x,fsiz,str)
int x,fsiz;
char *str;
{
int a;

a=vgatextsize(fsiz,str);
return(x+(BARWIDTH-a)/2);
}


showgifdir(startfrom,unshow)
int startfrom,unshow;
{
char cdir[MAXPATHLEN+1];
char ctmp[MAXPATHLEN+100];
int f,ypos,xpos,w,h,y,xt;
unsigned char *image;

getcwd(cdir,MAXPATHLEN);
if(updating_index)
  sprintf(ctmp,"Updating index of %s",cdir);
else
  sprintf(ctmp,"Directory of %s",cdir);

set_max_text_width(560);
if(unshow)
  undrawtext3d(40,135,4,ctmp);
else
  drawtext3d(40,135,4,ctmp,1, idx_light,idx_dark,idx_black);

set_max_text_width(BAR_RESTRICT_WIDTH);
for(f=startfrom;f<=gifdirsiz;f++)
  {
  vga_setcolor(unshow?idx_medium:idx_black);
  xpos=fwinxpos(f-startfrom+1);
  if(xpos+BARWIDTH>COLS) break;

  ypos=fwinypos(f-startfrom+1);
  prettyfile(&ctmp,&(gifdir[f]));
  xt=cfg.xvpic_index?centreseltxt(xpos,GDFSIZ,ctmp):xpos+3;
  vgadrawtext(xt,ypos+3,GDFSIZ,ctmp);
  if(cfg.thicktext && cfg.blockcursor)
    vgadrawtext(xt+1,ypos+3,GDFSIZ,ctmp);
  if(cfg.xvpic_index)
    {
    /* load and draw index file (or undraw it) */
    if(unshow)
      {
      image=malloc(84);
      if(image!=NULL)
        {
        memset(image,idx_medium,84);
        for(y=-2;y<62;y++)
          vga_drawscansegment(image,
          	xpos+(BARWIDTH-80)/2-2,ypos+y+GDFSIZ*6+9,84);
        free(image);
        }
      }
    else
      {
      sprintf(ctmp,".xvpics/%s",gifdir[f].name);
      gifdir[f].xvw=gifdir[f].xvh=0;
      if(gifdir[f].isdir)
        {
        /* 'folder' icon, as usual for these type of things */
        vga_setcolor(unshow?idx_medium:idx_black);
        xt=xpos+(BARWIDTH-80)/2;
        ypos+=GDFSIZ*6+9;
        
        /* main bit */
        vga_drawline(xt+10,ypos+50,xt+70,ypos+50);
        vga_drawline(xt+70,ypos+50,xt+70,ypos+20);
        vga_drawline(xt+70,ypos+20,xt+65,ypos+15);
        vga_drawline(xt+65,ypos+15,xt+15,ypos+15);
        vga_drawline(xt+15,ypos+15,xt+10,ypos+20);
        vga_drawline(xt+10,ypos+20,xt+10,ypos+50);
        
        /* top bit */
        vga_drawline(xt+15,ypos+15,xt+20,ypos+10);
        vga_drawline(xt+20,ypos+10,xt+35,ypos+10);
        vga_drawline(xt+35,ypos+10,xt+40,ypos+15);
        ypos-=GDFSIZ*6+9;
        
        gifdir[f].xvw=w=80; gifdir[f].xvh=h=60;
        }
      else
        if(read_xv332(ctmp,&image,&w,&h)==_PIC_OK)
          {
          gifdir[f].xvw=w; gifdir[f].xvh=h;
          for(y=0;y<h;y++)
            vga_drawscansegment(image+y*w,
            	xpos+(BARWIDTH-w)/2,ypos+y+GDFSIZ*6+39-h/2,w);
          free(image);
          }
        else
          {
          /* a default icon-type-thing here */
          vga_setcolor(unshow?idx_medium:idx_black);
          xt=xpos+(BARWIDTH-80)/2;
          ypos+=GDFSIZ*6+9;
          
          /* main bit */
          vga_drawline(xt+20,ypos+50,xt+60,ypos+50);
          vga_drawline(xt+60,ypos+50,xt+60,ypos+20);
          vga_drawline(xt+60,ypos+20,xt+50,ypos+10);
          vga_drawline(xt+50,ypos+10,xt+20,ypos+10);
          vga_drawline(xt+20,ypos+10,xt+20,ypos+50);
          
          /* 'folded' bit */
          vga_drawline(xt+50,ypos+10,xt+50,ypos+20);
          vga_drawline(xt+50,ypos+20,xt+60,ypos+20);
          
          ypos-=GDFSIZ*6+9;
          gifdir[f].xvw=w=80; gifdir[f].xvh=h=60;
          }
      
      if(gifdir[f].xvw!=0)
        {
        draw3dbox(xpos+(BARWIDTH-w)/2-1,ypos+GDFSIZ*6+38-h/2,
                  xpos+(BARWIDTH-w)/2+w,ypos+GDFSIZ*6+39-h/2+h,1,0,
                  idx_light,idx_dark);
        }
      }
    }
  }
set_max_text_width(NO_CLIP_FONT);
}

readgifdir()
{
DIR *dirfile;
struct dirent *anentry;
struct stat buf;
char cdir[MAXPATHLEN+1];
int entnum,l,isdir;

getcwd(cdir,MAXPATHLEN);
dirfile=opendir(".");
entnum=0;
while((anentry=readdir(dirfile))!=NULL)
  {
  if(anentry->d_name[0]=='.' && anentry->d_name[1]!='.')
    continue;	/* skip 'dot' files */
  if((strcmp(anentry->d_name,".")!=0)&&       /* no . and no .. if at root */
     (!((strcmp(cdir,"/")==0)&&(strcmp(anentry->d_name,"..")==0))))
    {
    if((stat(anentry->d_name,&buf))==-1)
      buf.st_mode=0;
    if((buf.st_mode&0x7000)==0x4000) isdir=1; else isdir=0;
    /* directories, GIF/JPG/PBM/PGM/PPM tested here */
    if(((l=strlen(anentry->d_name))>4) || isdir)
      if((!strcasecmp(anentry->d_name+l-4,".gif")) ||
         (!strcasecmp(anentry->d_name+l-4,".jpg")) ||
         (!strcasecmp(anentry->d_name+l-4,".pbm")) ||
         (!strcasecmp(anentry->d_name+l-4,".pgm")) ||
         (!strcasecmp(anentry->d_name+l-4,".ppm")) || isdir)
        {
        entnum++;
        gifdir[entnum].isdir=isdir;
        strcpy(gifdir[entnum].name,anentry->d_name);
        }
    }
  }
closedir(dirfile);
gifdirsiz=entnum;
qsort(&(gifdir[1]),gifdirsiz,sizeof(struct GDIR),(void *)gcompare);
}


int gcompare(gn1,gn2)
void *gn1,*gn2;
{
struct GDIR *g1,*g2;
char buf1[80],buf2[80];

g1=(struct GDIR *)gn1; g2=(struct GDIR *)gn2;
prettyfile(buf1,g1); prettyfile(buf2,g2);
return(strcmp(buf1,buf2));
}


prettyfile(buf,gifdptr)
char *buf;
struct GDIR *gifdptr;
{
if(gifdptr->isdir)
  {
  buf[0]='(';
  strcpy(buf+1,gifdptr->name);
  strcat(buf,")");
  }
else
  strcpy(buf,gifdptr->name);
}


screenon()
{
int r,g,b,n;
unsigned char *tmp;

if(cfg.xvpic_index && vga_hasmode(G640x480x256))
  {
  vga_setmode(G640x480x256);
  /* construct 3:3:2 palette */
  for(r=n=0;r<8;r++)
    for(g=0;g<8;g++)
      for(b=0;b<4;b++,n++)
        vga_setpalette(n,r*63/7,g*63/7,b*63/3);
  /* fix bar height */
  barheight=GDFSIZ*6+6+70;
  /* find approximations to file selector colours.
   * these are then blasted with the *real* file selector colours
   * unless 'perfectindex' is set.
   */
  idx_medium=MAKE332COL(cfg.medium_r,cfg.medium_g,cfg.medium_b);
  idx_dark  =MAKE332COL(cfg.dark_r  ,cfg.dark_g  ,cfg.dark_b  );
  idx_light =MAKE332COL(cfg.light_r ,cfg.light_g ,cfg.light_b );
  idx_black =MAKE332COL(cfg.black_r ,cfg.black_g ,cfg.black_b );
  }
else
  {
  /* even for the normal file selector we use an 8-bit mode if possible,
   * since it's much faster.
   */
  vga_setmode(vga_hasmode(G640x480x256)?G640x480x256:G640x480x16);
  cfg.xvpic_index=0;
  barheight=GDFSIZ*6+6;
  idx_medium=MIDGREY; idx_dark=DARK; idx_light=LIGHT; idx_black=BLACK;
  if(one_file_only && vga_hasmode(G640x480x256))
    idx_medium=255;	/* well, not 0 at any rate */
  }

if((cfg.xvpic_index && cfg.perfectindex==0) || cfg.xvpic_index==0)
  {
  vga_setpalette(idx_medium,cfg.medium_r,cfg.medium_g,cfg.medium_b);
  vga_setpalette(idx_dark  ,cfg.dark_r,cfg.dark_g,cfg.dark_b);
  vga_setpalette(idx_light ,cfg.light_r,cfg.light_g,cfg.light_b);
  vga_setpalette(idx_black ,cfg.black_r,cfg.black_g,cfg.black_b);
  }

if(one_file_only) return(0);

if(cfg.xvpic_index)
  {
  /* clear screen with 'medium' (i.e. background) colour. */
  tmp=malloc(640);
  if(tmp!=NULL)
    {
    memset(tmp,idx_medium,640);
    for(n=0;n<480;n++)
      vga_drawscanline(n,tmp);
    free(tmp);
    }
  }

draw3dbox(0,0,639,99,2,1,	idx_light,idx_dark);
draw3dbox(10,10,629,89,1,0,	idx_light,idx_dark);
draw3dbox(0,100,639,479,2,1,	idx_light,idx_dark);
draw3dbox(10,110,629,469,1,0,	idx_light,idx_dark);

drawzgvlogo(10,10);
}


screenoff()
{
vga_setmode(TEXT);
cleartext();
}


drawzgvlogo(a,b)
int a,b;
{
int x,y,c,bw;
byte *ptr;

ptr=zgvlogo;
bw=logow>>3;
if((logow%8)>0) bw+=1;
vga_setcolor(idx_black);
for(y=0;y<logoh;y++)
  {
  ptr=zgvlogo+y*bw;
  for(x=0;x<logow;x++)
    {
    if((x%8)==0)
      c=*ptr;
    if((c&128)==0) vga_drawpixel(a+x,b+y);
    c<<=1;
    c&=0xff;
    if((x%8)==7)
      ptr++;
    }
  }
}


cleartext()
{
if(cfg.cleartext)
  fprintf(stderr,"\x1b[H\x1b[J");
}


/* this shows any error message from GIF reading.
 * Notice that JPEG errors have already been reported!
 */
showerrmessage(errnumber)
int errnumber;
{
char buf[256];

strcpy(buf,"Error reading file - ");
switch(errnumber)
  {
  case _PICERR_NOFILE:
    strcat(buf,"file not found"); break;
  case _PICERR_NOMEM:
    strcat(buf,"out of memory"); break;
  case _PICERR_BADMAGIC:
    strcat(buf,"not a GIF/JPG/PNM or bad extension"); break;
  case _PICERR_NOCOLOURMAP:
    strcat(buf,"no global colour map"); break;
  case _PICERR_NOIMAGE:
    strcat(buf,"no image block found"); break;
  case _PICERR_LOCALMAP:
    strcat(buf,"can't handle local colour maps"); break;
  case _PICERR_CORRUPT:
    strcat(buf,"corrupt or short file"); break;
  case _PICERR_SHOWN_ALREADY:
    /* this only happens with JPEGs */
    return;
  default:
    strcat(buf,"unknown error (ulp!)");
  }
msgbox(zgv_ttyfd,buf,MSGBOXTYPE_OK, idx_light,idx_dark,idx_black);
}


/* ok, a couple of people want move, copy, and (especially) delete,
 * and it sounds like a good idea, so here goes.
 *
 * [ move and copy aren't done yet :( ]
 */

/* delete is the easiest.
 * we also delete any matching thumbnail file in .xvpics, and
 * attempt to remove the .xvpics directory also - relying on the
 * OS to do the Right Thing if other thumbnail files remain
 * (which it does, of course).
 */
delete_file(filename)
char *filename;
{
char buf[270];
int retn;
int poserr;

if(cfg.nodelprompt==0)
  {
  sprintf(buf,"Really delete %s?",filename);
  retn=msgbox(zgv_ttyfd,buf,MSGBOXTYPE_YESNO, idx_light,idx_dark,idx_black);
  }

if(retn==1 || cfg.nodelprompt)
  {
  if(remove(filename)==-1)
    msgbox(zgv_ttyfd,"Unable to delete file!",MSGBOXTYPE_OK,
    				idx_light,idx_dark,idx_black);
  else
    {
    sprintf(buf,".xvpics/%s",filename);
    remove(buf);		/* don't care if it fails */
    rmdir(".xvpics");	/* same here */
    }
  }

return(retn);
}


void xv332_how_far(sofar,total)
int sofar,total;
{
char tmp[128];
int done;

done=sofar*100/total;

if((done%5)==0 || sofar==total)
  {
  clear_xvpic(xv332_how_far_xpos,xv332_how_far_ypos);

  if(sofar!=total)
    {
    vga_setcolor(idx_black);
    sprintf(tmp,"Reading - %2d%%",done);
    vgadrawtext(xv332_how_far_xpos+(BARWIDTH-70)/2,
                xv332_how_far_ypos+GDFSIZ*6+39-4,2,tmp);
    }
  }
}


clear_xvpic(xpos,ypos)
int xpos,ypos;
{
unsigned char tmp[80];
int y;

memset(tmp,idx_medium,80);
for(y=0;y<60;y++)
  vga_drawscansegment(tmp,xpos+(BARWIDTH-80)/2,ypos+y+GDFSIZ*6+9,80);
}


/* if howfar equals 0, no progress report is done.
 * if it is >0, then the high 16 bits are 'xpos' and low 16 are 'ypos'.
 */
int makexv332(filename,howfar)
char *filename;
unsigned int howfar;	/* not that signedness should really matter */
{
FILE *out;
int tmp;
int w,h,y;
char buf[270];
unsigned char *smallpic;

pixelsize=1;		/* ouch */

if(howfar)
  {
  /* given the way the progress reporting from readpicture() works,
   * I have to use some global variables here. :( 
   */
  xv332_how_far_xpos=howfar>>16;
  xv332_how_far_ypos=howfar&0xFFFF;
  }

if((tmp=readpicture(filename,howfar?xv332_how_far:NULL,0))!=_PIC_OK)
  return(tmp);

/* image is pointed to by theimage, image_palette */

clear_xvpic(xv332_how_far_xpos,xv332_how_far_ypos);
vgadrawtext(xv332_how_far_xpos+(BARWIDTH-62)/2,
            xv332_how_far_ypos+GDFSIZ*6+39-4,2,"Resampling...");

/* resize */
w=80; h=60;
smallpic=resizepic(theimage,image_palette+512,image_palette+256,image_palette,
		width,height,&w,&h);
free(theimage); free(image_palette);	/* finished with these */

clear_xvpic(xv332_how_far_xpos,xv332_how_far_ypos);
vga_setcolor(idx_black);
vgadrawtext(xv332_how_far_xpos+(BARWIDTH-55)/2,
            xv332_how_far_ypos+GDFSIZ*6+39-4,2,"Dithering...");

/* dither */
ditherinit(w);
for(y=0;y<h;y++)
  ditherline(smallpic+y*80*3,y,w);
ditherfinish();

/* write */
sprintf(buf,".xvpics/%s",filename);
if((out=fopen(buf,"wb"))==NULL)
  return(_PICERR_NOFILE);		/* well, kind of */

fprintf(out,"P7 332\n");
fprintf(out,"# xv-compatible thumbnail file produced by zgv v%s\n",ZGV_VER);
fprintf(out,"#END_OF_COMMENTS\n");
fprintf(out,"%d %d 255\n",w,h);

for(y=0;y<h;y++)
  fwrite(smallpic+y*80*3,1,w,out);

fclose(out);
free(smallpic);

return(_PIC_OK);
}


/* update indexes (xvpics) of current directory.
 * we draw the stuff as we go along, and allow Esc to abort
 * the process. Checking for Esc happens between creating xvpics.
 * the directory '.xvpics' is created if needed.
 */
update_xvpics()
{
int f;
int curent,oldstart,oldent,startfrom;
int need_mkdir;
struct stat realpic,xvpic;
char buf[270];


/* for each GIF/JPG/PNM in the current directory, we check to see if
 * a file exists in the .xvpics directory with the same filename.
 * If not, it is created. If a file does exist, we check the
 * modification times. If the picture file is newer than the index
 * file, a new one gets created. Hope that's clear now. :)
 */

redrawall(1,1);
curent=startfrom=1;
need_mkdir=1;

for(f=1;f<=gifdirsiz;f++)
  {
  if(gifdir[f].isdir==0)
    {
    if(stat(gifdir[f].name,&realpic)==-1)
      continue;			/* the picture file doesn't exist */

    sprintf(buf,".xvpics/%s",gifdir[f].name);
    if(stat(buf,&xvpic)==-1 || realpic.st_mtime>xvpic.st_mtime)
      {
      if(need_mkdir)
        {
        /* it may already exist, but that's no great problem.
         * we let the umask sort out the permissions.
         */
        if(mkdir(".xvpics",0777)==-1)
          {
          if(errno!=EEXIST && errno!=EACCES)
            {
            msgbox(zgv_ttyfd,"Unable to create .xvpics directory",MSGBOXTYPE_OK,
            			idx_light,idx_dark,idx_black);
            return(0);
            }
          }
        need_mkdir=0;
        }
  
      makexv332(gifdir[f].name,(fwinxpos(f-startfrom+1)<<16) |
      				fwinypos(f-startfrom+1));
      /* redraw without mode change */
      showbar(curent,0,startfrom);
      showgifdir(startfrom,1);
      showgifdir(startfrom,0);
      showbar(curent,1,startfrom);
      }
    }
    
  /* move down one if possible */
  oldent=curent; oldstart=startfrom;
  if(curent<gifdirsiz) curent++;
  
  /* move right if needed */
  while(fwinxpos(curent-startfrom+1)+BARWIDTH>COLS)
    startfrom+=YSIZ;

  /* redraw */
  if(startfrom!=oldstart)
    {
    showbar(oldent,0,oldstart);
    showgifdir(oldstart,1);
    showgifdir(startfrom,0);
    showbar(curent,1,startfrom);
    }
  else  
    if(curent!=oldent)
      {
      showbar(oldent,0,startfrom);
      showbar(curent,1,startfrom);
      }
      
  /* check for Esc */
  if(readnbkey(zgv_ttyfd)==RK_ESC)
    return(0);
  }

usleep(400000);	/* for effect :) */
}
