/* This file contains the cont() method, which is a standard part of the
   libplot.  It continues a line from the current position of the graphics
   cursor to the point specified by x and y.

   This method is used in the construction of paths.  By repeatedly
   invoking cont(), the user may construct a polyline of arbitrary length.
   arc() and ellarc() may also be invoked, to add circular or elliptic arc
   elements to the path.  The path will terminate when the user either

     (1) explicitly invokes the endpath() method, or 
     (2) changes the value of one of the relevant drawing attributes, 
          e.g. by invoking move(), linemod(), linewidth(), pencolor(), 
	  fillcolor(), or filltype(), or 
     (3) draws some non-path object, by invoking box(), 
           circle(), point(), label(), alabel(), etc., or 
     (4) invokes restorestate() to restore an earlier drawing state. */

#include "sys-defines.h"
#include "extern.h"

/* initial size */
#define DATAPOINTS_BUFSIZ MAX_UNFILLED_POLYLINE_LENGTH

int
#ifdef _HAVE_PROTOS
_g_fcont (R___(Plotter *_plotter) double x, double y)
#else
_g_fcont (R___(_plotter) x, y)
     S___(Plotter *_plotter;) 
     double x, y;
#endif
{
  plGeneralizedPoint newpoint;

  if (!_plotter->open)
    {
      _plotter->error (R___(_plotter) 
		       "fcont: invalid operation");
      return -1;
    }

  /* if path buffer exists and is occupied by a single arc, replace arc by
     a polyline if that's called for */
  if (_plotter->have_mixed_paths == false
      && _plotter->drawstate->points_in_path == 2)
    _maybe_replace_arc (S___(_plotter));

  /* create or adjust size of path buffer, as needed */
  if (_plotter->drawstate->datapoints_len == 0)
    {
      _plotter->drawstate->datapoints = (plGeneralizedPoint *) 
	_plot_xmalloc (DATAPOINTS_BUFSIZ * sizeof(plGeneralizedPoint));
      _plotter->drawstate->datapoints_len = DATAPOINTS_BUFSIZ;
    }
  if (_plotter->drawstate->points_in_path == _plotter->drawstate->datapoints_len)
    {
      _plotter->drawstate->datapoints = (plGeneralizedPoint *) 
	_plot_xrealloc (_plotter->drawstate->datapoints, 
			2 * _plotter->drawstate->datapoints_len * sizeof(plGeneralizedPoint));
      _plotter->drawstate->datapoints_len *= 2;
    }
  
  /* analyse the present situation */

  if (_plotter->drawstate->points_in_path == 0)
    /* no path in progress, so begin one (at current position) */
    {
      newpoint.x = _plotter->drawstate->pos.x;
      newpoint.y = _plotter->drawstate->pos.y;
      _plotter->drawstate->datapoints[0] = newpoint;
      _plotter->drawstate->points_in_path++;
    }

  /* add new point to path buffer, so that points_in_path >=2 */
  newpoint.type = S_LINE;
  newpoint.x = x;
  newpoint.y = y;
  _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path++]
    = newpoint;

  /* update our notion of position */
  _plotter->drawstate->pos.x = x;
  _plotter->drawstate->pos.y = y;

  /* now conduct some tests on length; may invoke endpath() method */

  /* Provided that the Plotter supports the flushing of too-long polylines,
     if the path is getting too long (and it doesn't have to be filled),
     flush it to output and begin a new one.  `Too long' is
     Plotter-dependent.  The `suppress_polyline_flushout' flag is set
     during the drawing of polygonal approximations to ellipses
     (incl. circles), elliptic arcs, and circular arcs; see g_arc.c.  */
  if (_plotter->flush_long_polylines
      && (_plotter->drawstate->points_in_path 
	  >= _plotter->max_unfilled_polyline_length)
      && !_plotter->drawstate->suppress_polyline_flushout
      && (_plotter->drawstate->fill_type == 0))
    _plotter->endpath (S___(_plotter));
  
  /* Check whether we're about to violate the hard length limit on all
     polylines.  (Such a limit is imposed is imposed in an XPlotter,
     because of the X protocol restrictions; for other Plotters the length
     limit is typically INT_MAX.) */
  if (_plotter->drawstate->points_in_path >= _plotter->hard_polyline_length_limit)
    {
      _plotter->warning (R___(_plotter) 
			 "breaking an overly long path");
      _plotter->endpath (S___(_plotter));
    }

  return 0;
}

/* Some Plotters don't allow mixed paths to appear in the path storage
   buffer, because endpath() doesn't know how to handle them.  `Mixed
   paths' are arcs interspersed with line segments, or a path consisting of
   more than a single arc.  For such Plotters, this may be invoked by
   fcont() or farc() or fellarc() or fbezier2() or fbezier3(), to delete a
   single arc from the buffer and replace it by its polygonal
   approximation, i.e. by a polyline.  The replacement is generated by
   repeatedly calling fcont(). */
void
#ifdef _HAVE_PROTOS
_maybe_replace_arc (S___(Plotter *_plotter))
#else
_maybe_replace_arc (S___(_plotter))
     S___(Plotter *_plotter;) 
#endif
{
  if (_plotter->have_mixed_paths == false
      && _plotter->drawstate->points_in_path >= 2)
    switch (_plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].type)
      {
	double ax0, ay0, axc, ayc, ax1, ay1;
	plPoint pc, p0, p1;

      case S_ARC:
	/* path buffer contains a circular arc segment, so replace it */
	ax0 = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 2].x;
	ay0 = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 2].y;
	axc = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].xc;
	ayc = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].yc;
	ax1 = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].x;
	ay1 = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].y;
	
	_plotter->drawstate->points_in_path -= 1;
	/* back up (i.e. adjust our notion of position) */
	_plotter->drawstate->pos.x = ax0;
	_plotter->drawstate->pos.y = ay0;
	/* add polygonal approximation to circular arc to the path buffer, by
	   invoking _fakearc(), i.e., by invoking fcont() repeatedly */
	p0.x = ax0; p0.y = ay0;
	p1.x = ax1; p1.y = ay1;      
	pc.x = axc; pc.y = ayc;      
	_draw_circular_arc (R___(_plotter) p0, p1, pc);
	break;
      case S_ELLARC:
	/* path buffer contains an elliptic arc segment, so replace it */
	ax0 = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 2].x;
	ay0 = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 2].y;
	axc = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].xc;
	ayc = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].yc;
	ax1 = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].x;
	ay1 = _plotter->drawstate->datapoints[_plotter->drawstate->points_in_path - 1].y;
	
	_plotter->drawstate->points_in_path -= 1;
	/* back up (i.e. adjust our notion of position) */
	_plotter->drawstate->pos.x = ax0;
	_plotter->drawstate->pos.y = ay0;
	/* add polygonal approximation to elliptic arc to the path buffer, by
	   invoking _fakearc(), i.e., by invoking fcont() repeatedly */
	p0.x = ax0; p0.y = ay0;
	p1.x = ax1; p1.y = ay1;      
	pc.x = axc; pc.y = ayc;      
	_draw_elliptic_arc (R___(_plotter) p0, p1, pc);
	break;
      default:
	/* other segment type, OK */
	break;
      }
  
  /* Provided that the Plotter supports the flushing of too-long polylines,
     if the path is getting too long (and it doesn't have to be filled),
     flush it to output and begin a new one.  `Too long' is
     Plotter-dependent.  The `suppress_polyline_flushout' flag is set
     during the drawing of polygonal approximations to ellipses
     (incl. circles), elliptic arcs, and circular arcs; see g_arc.c.  */
  if (_plotter->flush_long_polylines
      && (_plotter->drawstate->points_in_path 
	  >= _plotter->max_unfilled_polyline_length)
      && !_plotter->drawstate->suppress_polyline_flushout
      && (_plotter->drawstate->fill_type == 0))
    _plotter->endpath (S___(_plotter));
  
  return;
}
