#ifndef lint
static char sccsid[] = "@(#)fax2tiff.c	1.9 5/22/90";
#endif

/*
 * Copyright (c) 1990 by Sam Leffler.
 * All rights reserved.
 *
 * This file is provided for unrestricted use provided that this
 * legend is included on all tape media and as a part of the
 * software program in whole or part.  Users may copy, modify or
 * distribute this file at will.
 */

/* 
 * Convert a CCITT Group 3 FAX file to TIFF Group 3 format.
 * Based on Paul Haeberli's fromfax program (with help from Jef Poskanzer).
 */
#include <stdio.h>
#include "tiffio.h"
#include "fax.h"

#define TABSIZE(tab)	(sizeof(tab)/sizeof(struct tableentry))
#define DEFXSIZE	1728
#define DEFYSIZE	(2*1075)

#define MAXWIDTH	2048

#define	howmany(x, y)	(((x)+((y)-1))/(y))

int	eofflag, eols;
int	rawzeros;
int	badfaxlines;
int	badfaxrun;
int	maxbadfaxrun;
int	shdata;
int	shbit;
int	verbose;
int	stretch;
int	reversebits = 0;
int	warnings = 0;
char	*filename;

#define WHASHA	3510
#define WHASHB	1178

#define BHASHA	293
#define BHASHB	2695

#define HASHSIZE	1021
tableentry *whash[HASHSIZE];
tableentry *bhash[HASHSIZE];

usage()
{
	fprintf(stderr, "usage: fax2tiff [-kprsvwO] [-o image.tif] faxfile ...\n");
	exit(-1);
}

int	kludge = 0;

main(argc, argv)
	int argc;
	char *argv[];
{
	FILE *in;
	TIFF *out = NULL;
	int compression = COMPRESSION_CCITTFAX3;
	int fillorder = FILLORDER_LSB2MSB;
	int group3options = 0;
	int photometric = PHOTOMETRIC_MINISBLACK;
	int isClassF = 1;
	int rows;
	int c;
	int pn, npages;
	extern int optind;
	extern char *optarg;

	while ((c = getopt(argc, argv, "o:bcfkOprsvwz")) != -1)
		switch (c) {
		case 'b':		/* undocumented -- for testing */
			photometric = PHOTOMETRIC_MINISWHITE;
			break;
		case 'c':		/* generate "classic" g3 format */
			isClassF = 0;
			break;
		case 'f':		/* generate Class F format */
			isClassF = 1;
			break;
		case 'k':		/* kludge to skip over input trash */
			kludge = 1;
			break;
		case 'O':		/* reverse bit ordering in output */
			fillorder = FILLORDER_MSB2LSB;
			break;
		case 'o':
			out = TIFFOpen(optarg, "w");
			if (out == NULL)
				exit(-2);
			break;
		case 'p':		/* zero pad scanlines before EOL */
			group3options = GROUP3OPT_FILLBITS;
			break;
		case 'r':		/* input file has reversed fillorder */
			reversebits = 1;
			break;
		case 's':		/* stretch image by dup'ng scanlines */
			stretch = 1;
			break;
		case 'w':		/* enable warnings about short lines */
			warnings = 1;
			break;
		case 'v':		/* -v for info, -vv for debugging */
			verbose++;
			break;
		case 'z':		/* undocumented -- for testing */
			compression = COMPRESSION_LZW;
			break;
		case '?':
			usage();
			/*NOTREACHED*/
		}
	if (out == NULL) {
		out = TIFFOpen("fax.tif", "w");
		if (out == NULL)
			exit(-2);
	}
	npages = argc - optind;
	if (npages < 1)
		usage();
	bzero(whash, HASHSIZE*sizeof(tableentry *));
	addtohash(whash, twtable, TABSIZE(twtable), WHASHA, WHASHB);
	addtohash(whash, mwtable, TABSIZE(mwtable), WHASHA, WHASHB);
	addtohash(whash, extable, TABSIZE(extable), WHASHA, WHASHB);
	bzero(bhash, HASHSIZE*sizeof(tableentry *));
	addtohash(bhash, tbtable, TABSIZE(tbtable), BHASHA, BHASHB);
	addtohash(bhash, mbtable, TABSIZE(mbtable), BHASHA, BHASHB);
	addtohash(bhash, extable, TABSIZE(extable), BHASHA, BHASHB);
	for (pn = 0; optind < argc; pn++, optind++) {
		in = fopen(filename = argv[optind], "r");
		if (in == NULL) {
			fprintf(stderr,
			    "%s: %s: Can not open\n", argv[0], filename);
			continue;
		}
		TIFFSetField(out, TIFFTAG_IMAGEWIDTH, DEFXSIZE);
		TIFFSetField(out, TIFFTAG_BITSPERSAMPLE, 1);
		TIFFSetField(out, TIFFTAG_GROUP3OPTIONS, 0);
		TIFFSetField(out, TIFFTAG_COMPRESSION, compression);
		TIFFSetField(out, TIFFTAG_PHOTOMETRIC, photometric);
		TIFFSetField(out, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
		TIFFSetField(out, TIFFTAG_SAMPLESPERPIXEL, 1);
		if (compression == COMPRESSION_CCITTFAX3) {
			TIFFSetField(out, TIFFTAG_GROUP3OPTIONS, group3options);
			TIFFSetField(out, TIFFTAG_ROWSPERSTRIP, -1);
			TIFFModeCCITTFax3(out, isClassF);
		} else
			TIFFSetField(out, TIFFTAG_ROWSPERSTRIP,
			    (8*1024)/TIFFScanlineSize(out));
		TIFFSetField(out, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
		TIFFSetField(out, TIFFTAG_FILLORDER, fillorder);
		TIFFSetField(out, TIFFTAG_SOFTWARE, "fax2tiff");
		TIFFSetField(out, TIFFTAG_XRESOLUTION, 200.0);
		TIFFSetField(out, TIFFTAG_YRESOLUTION, 200.0);
		TIFFSetField(out, TIFFTAG_RESOLUTIONUNIT, RESUNIT_INCH);
		TIFFSetField(out, TIFFTAG_PAGENUMBER, pn, npages);

		rows = copyFaxFile(in, out);
		fclose(in);

		if (verbose) {
			fprintf(stderr, "%s:\n", argv[optind]);
			fprintf(stderr, "%d rows in input\n", rows);
			fprintf(stderr, "%d total bad rows\n", badfaxlines);
			fprintf(stderr, "%d max consecutive bad rows\n",
			    maxbadfaxrun);
		}
		if (compression == COMPRESSION_CCITTFAX3 && isClassF) {
			TIFFSetField(out, TIFFTAG_BADFAXLINES, badfaxlines);
			TIFFSetField(out, TIFFTAG_CLEANFAXDATA,
			    badfaxlines ? 2 : 0);
			TIFFSetField(out, TIFFTAG_CONSECUTIVEBADFAXLINES,
			    maxbadfaxrun);
		}
		TIFFWriteDirectory(out);
	}
	TIFFClose(out);
	fclose(in);
	exit(0);
}

copyFaxFile(in, out)
	FILE *in;
	TIFF *out;
{
	u_char *buf;
	int row;
	int bsize;

	eofflag = 0;
	eols = 0;
	if (kludge) {
		/* XXX skip over some strange stuff at the beginning */
		skiptoeol(in);
		skiptoeol(in);
		skiptoeol(in);
	}
	skiptoeol(in);

	bsize = howmany(MAXWIDTH, 8) * 8;
	buf = (u_char *)malloc(bsize);

	row = 0;
	badfaxlines = 0;
	badfaxrun = 0;
	maxbadfaxrun = 0;
	while (!eofflag) {
		if (!getfaxrow(in, row, buf, bsize))
			break;
		if (TIFFWriteScanline(out, buf, row, 0) < 0) {
			fprintf(stderr, "%s: Problem at row %d.\n",
			    filename, row);
			break;
		}
		row++;
		if (stretch) {
			if (TIFFWriteScanline(out, buf, row, 0) < 0) {
				fprintf(stderr, "%s: Problem at row %d.\n",
				    filename, row);
				break;
			}
			row++;
		}
	}
	if (badfaxrun > maxbadfaxrun)
		maxbadfaxrun = badfaxrun;
	return (row);
}

setrow(sbuf, val, n)
	u_char *sbuf;
	int val, n;
{
	while (n--)
		*sbuf++ = val;
}

addtohash(hash, te, n, a, b)
	tableentry *hash[];
	tableentry *te;
	int n, a, b;
{
	unsigned int pos;

	while (n--) {
		pos = ((te->length+a)*(te->code+b))%HASHSIZE;
		if (hash[pos] != 0) {
			fprintf(stderr,
			   "Internal error: addtohash fatal hash collision\n");
			exit(1);
		}
		hash[pos] = te;
		te++;
	}
}

tableentry *
hashfind(hash, length, code, a, b)
	tableentry *hash[];
	int length, code;
	int a, b;
{
	unsigned int pos;
	tableentry *te;

	pos = ((length+a)*(code+b))%HASHSIZE;
	if (pos < 0 || pos >= HASHSIZE) {
		fprintf(stderr,
	"Internal error: bad hash position, length %d code %d pos %d\n",
		    length, code, pos);
		exit(1);
	}
	te = hash[pos];
	return ((te && te->length == length && te->code == code) ? te : 0);
}

/*
 * Fill a span with ones.
 */
static void
fillspan(cp, x, count)
	register u_char *cp;
	register int x, count;
{
	static unsigned char masks[] =
	    { 0, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfc, 0xfe, 0xff };

	cp += x>>3;
	if (x &= 7) {
		if (count < 8 - x) {
			*cp++ |= masks[count] >> x;
			return;
		}
		*cp++ |= 0xff >> x;
		count -= 8 - x;
	}
	while (count >= 8) {
		*cp++ = 0xff;
		count -= 8;
	}
	*cp |= masks[count];
}

#include <setjmp.h>

jmp_buf	EOFbuf;

getfaxrow(inf, row, sbuf, n)
	FILE *inf;
	int row;
	u_char *sbuf;
	int n;
{
	int x;
	int curlen, curcode, nextbit;
	int count, color;
	tableentry *te;

	bzero(sbuf, n);
	x = 0;
	rawzeros = 0;
	curlen = 0;
	curcode = 0;
	color = 1;
	count = 0;
	if (setjmp(EOFbuf))
		return (0);
	while (!eofflag) {
		if (x >= MAXWIDTH) {
			if (badfaxrun > maxbadfaxrun)
				maxbadfaxrun = badfaxrun;
			badfaxrun = 0;
			skiptoeol(inf);
			return (!eofflag); 
		}
		do {
			if (verbose > 1)
				fprintf(stderr, "[l %d] ", x);
			if (rawzeros >= 11) {
				nextbit = rawgetbit(inf);
				if (nextbit) {
					if (x == 0)
						/* XXX should be 6 */
						eofflag = (++eols == 3);
					else
						eols = 0;
					if (warnings && x && x < 1728)
						fprintf(stderr,
				"%s: Warning, row %d short (len %d).\n",
						    filename, row, x);
					if (badfaxrun > maxbadfaxrun)
						maxbadfaxrun = badfaxrun;
					badfaxrun = 0;
					return (!eofflag); 
				}
			} else
				nextbit = rawgetbit(inf);
			curcode = (curcode<<1) + nextbit;
			curlen++;
		} while (curcode <= 0);
		if (curlen > 13) {
			fprintf(stderr,
   "%s: Bad code word at row %d, x %d (len %d code 0x%x), skipping to EOL\n",
			    filename,
			    row, x,
			    curlen, curcode);
			badfaxlines++;
			badfaxrun++;
			skiptoeol(inf);
			return (!eofflag);
		}
		if (color) {
			if (curlen < 4)
				continue;
			te = hashfind(whash, curlen, curcode, WHASHA, WHASHB);
		} else {
			if (curlen < 2)
				continue;
			te = hashfind(bhash, curlen, curcode, BHASHA, BHASHB);
		}
		if (!te)
			continue;
		switch (te->tabid) {
		case TWTABLE:
		case TBTABLE:
			count += te->count;
			if (x+count > MAXWIDTH) 
				count = MAXWIDTH-x;
			if (count > 0) {
				if (color)
					fillspan(sbuf, x, count);
				x += count;
			}
			count = 0;
			color = !color;
			break;
		case MWTABLE:
		case MBTABLE:
		case EXTABLE:
			count += te->count;
			break;
		default:
			fprintf(stderr, "fax2tiff: Internal bad poop\n");
			exit(1);
		}
		curcode = curlen = 0;
	}
	return (0);
}

skiptoeol(file)
	FILE *file;
{

	while (rawzeros < 11)
		(void) rawgetbit(file);
	while (!rawgetbit(file))
		;
}

int
rawgetbit(file)
	FILE *file;
{
	if ((shbit&0xff) == 0) {
		shdata = getc(file);
		if (shdata == EOF) {
			fprintf(stderr, "%s: Premature EOF\n", filename);
			eofflag++;
			longjmp(EOFbuf, 1);
		}
		shbit = reversebits ? 0x01 : 0x80; 
	}
	if (shdata&shbit) {
		rawzeros = 0;
		if (reversebits)
			shbit <<= 1; 
		else
			shbit >>= 1;
		return (1);
	} else {
		rawzeros++;
		if (reversebits)
			shbit <<= 1; 
		else
			shbit >>= 1;
		return (0);
	}
}
