/*
	PalmHugo.c
	
	The initialization and entry points for Hugo for Palm OS

	Copyright (c) 2002-2006 by Kent Tessman
	The General Coffee Company Film Productions
*/

#include "PalmHugo.h"
#include "PalmHugoRsc.h"
#include "heheader.h"

#include "FldGlue.h"

#define TYPED_STORY_COMMANDS

void RestoreWindow();
void UpdatePleaseWait(int percent);
Boolean SaveSetup(char *name);
Boolean LoadSetup(char *name);
char *GetScrollbackWord(int x, int y);
void ExitDialog(Boolean ok);


/***********************************************************************
 *
 *   Internal Constants
 *
 ***********************************************************************/
#define appFileCreator		 PH_CREATOR_TYPE
#define appVersionNum            0x01
#define appPrefID                0x00
#define appPrefVersionNum        0x01

// Define the minimum OS version we support
// (3.1 for now, although really 3.5)
#define ourMinVersion	sysMakeROMVersion(3,1,0,sysROMStageRelease,0)


/***********************************************************************
 *
 *  Various globals
 *
 ***********************************************************************/
/* A debugger would be easier, but since we don't have one, we can
   set this for "breakpoints": */
char ph_debug = false;

Boolean game_started = false, got_first_input = false;
int processed_shortcut = false;
Int16 palm_screen_width = 0, palm_screen_height = 0;
Int16 menu_line_height;
int palm_display_depth = 1;
Boolean override_app_exit = false;
Boolean promptmore_waiting;
RGBColorType field_text, field_back, field_caret,
	default_field_text, default_field_back, default_field_caret,
	default_real_field_back;
	
#ifndef TEXT_PROMPTMORE
WinHandle promptmore_winhandle = NULL;
RectangleType promptmore_rect;
#endif

Boolean override_window_restore = false;
Boolean override_window_save = false;
// from hebuffer.c:
extern int tb_first_used;
extern int tb_first_unused;
extern int tb_last_used;
extern int tb_last_unused;
extern int tb_used;
extern int tb_unused;
typedef struct
{
	char *data;
	int left, top, right, bottom;
	int prev, next;
#ifdef TEXTBUFFER_FORMATTING
	int font, fcolor, bgcolor;
#endif
} tb_list_struct;
extern tb_list_struct tb_list[MAX_TEXTBUFFER_COUNT];

// State-saving:
Boolean delete_snapshot = false;
Boolean auto_resuming = false;
char saved_state_game_name[MAXPATH+1] = "";

// For RunDialog():
Boolean dialog_ok = false;
char file_selected[MAXPATH];
UInt32 file_select_type;
int file_select_error;

// from herun.c:
extern int lowest_windowbottom, physical_lowest_windowbottom;


/***********************************************************************
 *
 *  Various functions
 *
 ***********************************************************************/

int he_main(int argc, char **argv);	// engine entry point

void TypeCommand(char *cmd, char clear, char newline)
{
}

void PrintFatalError(char *a)
{
	// Hacky because we apparently can't have an \n at the end of
	// the alert text; we'll always have a stack-based array for an
	// arg, anyway (coming from FatalError() in hemisc.c.
	if (a[StrLen(a)]=='\n') a[StrLen(a)] = '\0';
	
	FrmCustomAlert(FatalAlert, "FATAL ERROR", a, "");
	
	delete_snapshot = true;
}

void PalmHugoExit(int n)
{
	static Boolean exit_called = false;
	
	if (!exit_called)
	{
		exit_called = true;
		ErrThrow(n);
	}
}

void PopupMessage(char *title, char *msg)
{
	FrmCustomAlert(GeneralAlert, title, msg, "");
}

void DeleteFile(char *name)
{
	UInt16 card;
	LocalID db;
	
	for (card=0; card<MemNumCards(); card++)
	{
		if ((db = DmFindDatabase(card, name)))
		{
			DmDeleteDatabase(card, db);
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:    RomVersionCompatible
 *
 * DESCRIPTION: This routine checks that a ROM version meets your
 *              minimum requirement.
 *
 * PARAMETERS:  requiredVersion - minimum rom version required
 *                                (see sysFtrNumROMVersion in SystemMgr.h 
 *                                for format)
 *              launchFlags     - flags that indicate if the application 
 *                                UI is initialized.
 *
 * RETURNED:    error code or zero if rom is compatible
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
UInt32 romVersion;

void GetRomVersion()
{
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
}	

static Err RomVersionCompatible(UInt32 requiredVersion, UInt16 launchFlags)
{
	UInt32 romVer;
	
	// See if we're on in minimum required version of the ROM or later.
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVer);
	if (romVer < requiredVersion)
		{
		if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
			(sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
			{
			FrmAlert (RomIncompatibleAlert);
		
			// Palm OS 1.0 will continuously relaunch this app unless we switch to 
			// another safe one.
			if (romVer < ourMinVersion)
				{
				AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
				}
			}
		
		return sysErrRomIncompatible;
		}

	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:    GetObjectPtr
 *
 * DESCRIPTION: This routine returns a pointer to an object in the current
 *              form.
 *
 * PARAMETERS:  formId - id of the form to display
 *
 * RETURNED:    void *
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
void *GetObjectPtr(UInt16 objectID)
{
	FormPtr frmP;

	frmP = FrmGetActiveForm();
	return FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, objectID));
}


/***********************************************************************
 *
 *   Memory Management Functions
 *
 *   We read static memory from the gamefile, and read/write dynamic
 *   memory from the dynamic storage file.
 *
 ***********************************************************************/

#pragma mark ----- Memory management -----

PH_FILE *db_cache;
typedef struct
{
	MemHandle handle;
	MemPtr ptr;
} CACHED_RECORD;
CACHED_RECORD *cached_record;
int static_records = 0;
int dynamic_records = 0;

unsigned char GetMemValue(long addr)
{
	// If addr is below the object table, it's static data
	if (addr < objtable*16L)
	{
		return *((unsigned char *)cached_record[addr/game->blocksize].ptr+addr%game->blocksize);
	}
	else if (addr < codeend)
	{
		addr -= objtable*16L;
		return *((unsigned char *)cached_record[static_records+addr/db_cache->blocksize].ptr+addr%db_cache->blocksize);
	}
	
	return 0;	// bad read addr
}

void SetMemValue(long addr, unsigned char val)
{
	// If addr is below the object table, it's static data, so
	// we can't write to it (and anyway, we should never be here
	// in that case)
	if (addr >= objtable*16L && addr < codeend)
	{
		addr -= objtable*16L;
		DmWrite(cached_record[static_records+addr/db_cache->blocksize].ptr, (UInt32)(addr%db_cache->blocksize), &val, 1);
	}
}

// SetupGameDataCache() caches MemPtrs for both dynamic and static memory
// to avoid expensive record-swapping.  Note that all handles will need to
// be freed and records released.

Boolean SetupGameDataCache(void)
{
	UInt16 i, record_count = 0, total_records;

	// Count how many records of each type we'll need
	static_records = (int)((double)(objtable*16L) / (double)(game->blocksize) + 1);
	dynamic_records = (int)((double)(codeend - objtable*16L) / (double)(db_cache->blocksize) + 1);
	total_records = static_records + dynamic_records;
	
	cached_record = MemPtrNew(total_records * sizeof(CACHED_RECORD));
	if (!cached_record) goto Abort;
	MemSet(cached_record, total_records * sizeof(CACHED_RECORD), 0);

	// Get the actual record data
	for (i=0; i<total_records; i++)
	{
		if (i < static_records)
			cached_record[i].handle = DmGetRecord(game->ref, i);
		else
			cached_record[i].handle = DmGetRecord(db_cache->ref, i-static_records);
		if (cached_record[i].handle==NULL) goto Abort;
		record_count++;
		cached_record[i].ptr = MemHandleLock(cached_record[i].handle);
		if (cached_record[i].ptr==NULL) goto Abort;
	}
	
	return true;
	
Abort:
	if (cached_record)
	{
		for (i=0; i<record_count; i++)
		{
			if (cached_record[i].handle)
			{
				MemHandleUnlock(cached_record[i].handle);
				if (i < static_records)
					DmReleaseRecord(game->ref, i, false);
				else
					DmReleaseRecord(db_cache->ref, i-static_records, false);
			}
			if (i < static_records)
				DmReleaseRecord(game->ref, i, false);
			else
				DmReleaseRecord(db_cache->ref, i-static_records, false);
		}
		MemPtrFree(cached_record);
		cached_record = NULL;
	}
	static_records = 0;
	dynamic_records = 0;
	return false;
}

void FreeGameDataCache(void)
{
	UInt16 i;
	
	// Free cached records
	if (cached_record)
	{
		for (i=0; i<static_records + dynamic_records; i++)
		{
			if (cached_record[i].handle)
			{
				MemHandleUnlock(cached_record[i].handle);
				if (i < static_records)
					DmReleaseRecord(game->ref, i, false);
				else
					DmReleaseRecord(db_cache->ref, i-static_records, true);
			}
		}
		MemPtrFree(cached_record);
	}
	cached_record = NULL;
	static_records = 0;
	dynamic_records = 0;
}

// LoadGameData() basically caches the dynamic data area in
// the database called CURRENT_GAME_DATA

int LoadGameData(char reload)
{
	int i;
	long c;
	UInt32 count;
	
	// Just in case this is a restart instead of a cold load
	FreeGameDataCache();

	// Won't be loading the text bank
	loaded_in_memory = false;

	if (db_cache) fclose(db_cache);
	db_cache = NULL;
	
	fseek(game, 0, SEEK_SET);
	// Enough memory for the header
	mem = MemPtrNew(256);
	if (!mem) FatalError(MEMORY_E);
	for (i=0; i<256; i++)
		mem[i] = fgetc(game);

	objtable = mem[H_OBJTABLE] + mem[H_OBJTABLE+1]*256;
	MemPtrFree(mem);
	
	// If we're autoresuming, we try to use the existing game data
	if (auto_resuming)
	{
		fseek(game, 0, SEEK_END);
		fgetc(game);
	
		if ((db_cache = fopen(CURRENT_GAME_DATA, "rb+"))==NULL)
		{
			return false;
		}
		
		// Because the engine is going to free it when it shuts down:
		mem = NULL;
		
		if (!SetupGameDataCache())
		{
			return false;
		}
		
		return true;
	}
	
	// Otherwise, we start from scratch and (re)create CURRENT_GAME_DATA
	DeleteFile(CURRENT_GAME_DATA);
	
	file_select_type = PH_INTERNAL_TYPE;
	if ((db_cache = fopen(CURRENT_GAME_DATA, "wb+"))==NULL)
	{
		FrmCustomAlert(FatalAlert, "FATAL ERROR",
			"Dynamic cache creation failure", "");
		exit(-1);
	}
	
	// Match the blocksize of db_cache to the game
	db_cache->blocksize = game->blocksize;
	
#ifdef USE_TEXTBUFFER
	// Clear textbuffer
	TB_Clear(0, 0, palm_screen_width, palm_screen_height);
#endif

	ShowPleaseWait(true);

	// Read in blocks of game->blocksize
	mem = MemPtrNew(game->blocksize);
	if (!mem) FatalError(MEMORY_E);
	
	// Load the game data from the object table to the end of
	// code (and start of the textbank)
	fseek(game, objtable*16L, SEEK_SET);
	c = objtable*16L;
	while (c < codeend)
	{
		if (codeend-c < (long)game->blocksize)
			count = codeend-c;
		else
			count = game->blocksize;
			
		fread(mem, sizeof(unsigned char), count, game);
			
		db_cache->open_record = dmMaxRecordIndex;
		db_cache->record_handle = DmNewRecord(db_cache->ref, &db_cache->open_record, count);
		db_cache->record_data = MemHandleLock(db_cache->record_handle);
		if (!db_cache->record_data) FatalError(MEMORY_E);
		
		DmWrite(db_cache->record_data, 0, mem, count);
		
		MemHandleUnlock(db_cache->record_handle);
		DmReleaseRecord(db_cache->ref, db_cache->open_record, true);
		db_cache->open_record = EOF;

		c += count;
		db_cache->length += count;

		UpdatePleaseWait((c-objtable*16L)*100L/(codeend-objtable*16L));
	}
/*
	if (ferror(db_cache))
	{
		FrmCustomAlert(FatalAlert, "FATAL ERROR", 
			"Dynamic cache write failure", "");
		exit(-1);
	}
*/	
	MemPtrFree(mem);
	mem = NULL;

	if (!SetupGameDataCache())
	{
		return false;
	}

	ShowPleaseWait(false);
	
	return true;
}

void FreeGameData(void)
{
	FreeGameDataCache();

	// Close the main gamefile
	if (game)
	{
		ph_fclose(game);
		game = NULL;
	}
	
#ifdef USE_TEXTBUFFER
	// Clear textbuffer
	TB_Clear(0, 0, palm_screen_width, palm_screen_height);
#endif

	if (promptmore_winhandle) WinRestoreBits(promptmore_winhandle, promptmore_rect.topLeft.x, promptmore_rect.topLeft.y);

	// Free the parsing prelist
	if (obj_parselist)
	{
		hugo_blockfree(obj_parselist);
		obj_parselist = NULL;
	}
	
	// Close various files
	if (db_cache) ph_fclose(db_cache);
	db_cache = NULL;
	if (record) ph_fclose(record);
	record = NULL;
	if (script) ph_fclose(script);
	script = NULL;
	if (io) ph_fclose(io);
	io = NULL;
}


#pragma mark ----- Save/restore -----

/***********************************************************************
 *
 *   Save/restore functions
 *
 *   Replaced from herun.c so that we can display a progress bar--at
 *   least until these aren't so painfully slow.
 *
 ***********************************************************************/

int RestoreGameData(void)
{
	char testid[3], testserial[9];
	int lbyte, hbyte;
	int j;
	unsigned int k, undosize;
	long i;

	/* Check ID */
	testid[0] = (char)fgetc(save);
	testid[1] = (char)fgetc(save);
	testid[2] = '\0';
	if (ferror(save)) goto RestoreError;

	if (strcmp(testid, id))
	{
		AP("Incorrect save file.");
		if (fclose(save)) FatalError(READ_E);
		save = NULL;
		return 0;
	}

	/* Check serial number */
	if (!fgets(testserial, 9, save)) goto RestoreError;
	if (strcmp(testserial, serial))
	{
		AP("Save file created by different version.");
		if (fclose(save)) FatalError(READ_E);
		save = NULL;
		return 0;
	}

	/* Restore variables */
	for (k=0; k<MAXGLOBALS+MAXLOCALS; k++)
	{
		if ((lbyte = fgetc(save))==EOF || (hbyte = fgetc(save))==EOF)
			goto RestoreError;
		var[k] = lbyte + hbyte * 256;
	}

	/* Restore objtable and above */

	if (fseek(game, objtable*16L, SEEK_SET)) goto RestoreError;
	i = 0;

	while (i<codeend-(long)(objtable*16L))
	{
		int updatecount = 0;
		
		if ((hbyte = fgetc(save))==EOF && ferror(save)) goto RestoreError;

		if (hbyte==0)
		{
			if ((lbyte = fgetc(save))==EOF && ferror(save)) goto RestoreError;
			SETMEM(objtable*16L+i, (unsigned char)lbyte);
			i++;

			/* Skip byte in game file */
			if (fgetc(game)==EOF) goto RestoreError;
		}
		else
		{
			while (hbyte--)
			{
				/* Get unchanged game file byte */
				if ((lbyte = fgetc(game))==EOF) goto RestoreError;
				SETMEM(objtable*16L+i, (unsigned char)lbyte);
				i++;
			}
		}
		
		if (updatecount%1000==0) UpdatePleaseWait(i*100L/(codeend-(long)objtable*16L));
		updatecount++;
	}

	/* Restore undo data */
	if ((lbyte = fgetc(save))==EOF || (hbyte = fgetc(save))==EOF)
		goto RestoreError;
	undosize = lbyte + hbyte*256;

	/* We can only restore undo data if it was saved by a port with
	   the same MAXUNDO as us */
	if (undosize==MAXUNDO)
	{
		for (k=0; k<MAXUNDO; k++)
		{
			for (j=0; j<5; j++)
			{
				if ((lbyte = fgetc(save))==EOF || (hbyte = fgetc(save))==EOF)
					goto RestoreError;
				undostack[k][j] = lbyte + hbyte*256;
			}
		}
		if ((lbyte = fgetc(save))==EOF || (hbyte = fgetc(save))==EOF) goto RestoreError;
		undoptr = lbyte + hbyte*256;
		if ((lbyte = fgetc(save))==EOF || (hbyte = fgetc(save))==EOF) goto RestoreError;
		undoturn = lbyte + hbyte*256;
		if ((lbyte = fgetc(save))==EOF || (hbyte = fgetc(save))==EOF) goto RestoreError;
		undoinvalid = (unsigned char)lbyte, undorecord = (unsigned char)hbyte;
	}
	else undoinvalid = true;

	return true;
	
RestoreError:
	return false;
}

int SaveGameData(void)
{
	int c, j;
	int lbyte, hbyte;
	long i;
	int samecount = 0;

	/* Write ID */
	if (fputc(id[0], save)==EOF || fputc(id[1], save)==EOF) goto SaveError;

	/* Write serial number */
	if (fputs(serial, save)==EOF) goto SaveError;

	/* Save variables */
	for (c=0; c<MAXGLOBALS+MAXLOCALS; c++)
	{
		hbyte = (unsigned int)var[c] / 256;
		lbyte = (unsigned int)var[c] - hbyte * 256;
		if (fputc(lbyte, save)==EOF || fputc(hbyte, save)==EOF) goto SaveError;
	}

	/* Save objtable to end of code space */

	if (fseek(game, objtable*16L, SEEK_SET)) goto SaveError;

	for (i=0; i<=codeend-(long)(objtable*16L); i++)
	{
		if ((lbyte = fgetc(game))==EOF) goto SaveError;
		hbyte = MEM(objtable*16L+i);

		/* If memory same as original game file */
		if (lbyte==hbyte && samecount<255) samecount++;

		/* If memory differs (or samecount exceeds 1 byte) */
		else
		{
			if (samecount)
				if (fputc(samecount, save)==EOF) goto SaveError;

			if (lbyte!=hbyte)
			{
				if (fputc(0, save)==EOF) goto SaveError;
				if (fputc(hbyte, save)==EOF) goto SaveError;
				samecount = 0;
			}
			else samecount = 1;
		}

		if (i%1000==0) UpdatePleaseWait(i*100L/(codeend-objtable*16L));
	}
	if (samecount)
		if (fputc(samecount, save)==EOF) goto SaveError;

	/* Save undo data */
	
	/* Save the number of turns in this port's undo stack */
	hbyte = (unsigned int)MAXUNDO / 256;
	lbyte = (unsigned int)MAXUNDO - hbyte*256;
	if (fputc(lbyte, save)==EOF || fputc(hbyte, save)==EOF)
		goto SaveError;
	for (c=0; c<MAXUNDO; c++)
	{
		for (j=0; j<5; j++)
		{
			hbyte = (unsigned int)undostack[c][j] / 256;
			lbyte = (unsigned int)undostack[c][j] - hbyte*256;
			if (fputc(lbyte, save)==EOF || fputc(hbyte, save)==EOF)
				goto SaveError;
		}
	}
	if (fputc(undoptr-(undoptr/256)*256, save)==EOF || fputc(undoptr/256, save)==EOF)
		goto SaveError;
	if (fputc(undoturn-(undoturn/256)*256, save)==EOF || fputc(undoturn/256, save)==EOF)
		goto SaveError;
	if (fputc(undoinvalid, save)==EOF || fputc(undorecord, save)==EOF)
		goto SaveError;
		
	return true;
		
SaveError:
	return false;
}


#pragma mark ----- Preferences, setup, and state -----

/***********************************************************************
 *
 * Preference and setup management
 *
 ***********************************************************************/
PalmHugoPreferenceType ph_prefs;

dir_list_struct default_dir_list[NUM_DIRECTIONS] = {
	{"North", "N "}, {"Northeast", "NE"}, {"East", "E "}, {"Southeast", "SE"},
	{"South", "S "}, {"Southwest", "SW"}, {"West", "W "}, {"Northwest", "NW"},
	{"Up", "U "}, {"Down", "D "}, {"In", "In"}, {"Out", "Ou"}
};

command_list_struct default_command_list[NUM_DIRECTIONS] = {
	{"Examine", true}, {"Get", true}, {"Drop", true}, {"Ask ^ about", false},
	{"", false}, {"", false}, {"", false}, {"", false},
	{"", false}, {"", false}, {"", false}, {"", false}
};

word_list_struct default_word_list[NUM_WORDS] = {
	{"Look", true}, {"Inventory", true}, {"Undo", true}, {"", false},
	{"", false}, {"", false}, {"", false}, {"", false}, 
	{"", false}, {"", false}, {"", false}, {"", false}, 
};

void SetupDirectionButtons(void)
{
	UInt16 i;
	int left;
	FormPtr frmP;
	ControlPtr ctrlP;
	
	frmP = FrmGetActiveForm();

	left = 11;	// of first direction button

	// Add the '...' button
	ctrlP = CtlNewControl((void **)&frmP, WORD_BUTTON, pushButtonCtl, "...",
		1, DIRBUTTON_TOP_ROW, left-2, DIRBUTTON_HEIGHT, stdFont, 0, true);
// Only need to do this if it's a buttonCtl:
//	ctrlP->attr.frame = rectangleButtonFrame;
	FrmShowObject(frmP, FrmGetObjectIndex(frmP, WORD_BUTTON));
	
	// Add each direction button
	for (i=0; i<NUM_DIRECTIONS; i++)
	{
		Int16 width = FntCharsWidth(ph_prefs.dir_list[i].button, StrLen(ph_prefs.dir_list[i].button)) + 1;
		ctrlP = CtlNewControl((void **)&frmP, WORD_BUTTON+i+1, pushButtonCtl,
			ph_prefs.dir_list[i].button,
			left, DIRBUTTON_TOP_ROW, width, DIRBUTTON_HEIGHT,
			stdFont, 0, true);
// Only need to do this if it's a buttonCtl:
//		ctrlP->attr.frame = rectangleButtonFrame;
		
		if (StrCompare(ph_prefs.dir_list[i].button, ""))
		{
			FrmShowObject(frmP, FrmGetObjectIndex(frmP, WORD_BUTTON+i+1));
			left += width + 1;
		}
		else
			FrmHideObject(frmP, FrmGetObjectIndex(frmP, WORD_BUTTON+i+1));
	}
}

void ResetDirectionButtons(void)
{
	int i;
	FormPtr frmP = FrmGetActiveForm();
	for (i=WORD_BUTTON+NUM_DIRECTIONS; i>=WORD_BUTTON; i--)
	{
		FrmHideObject(frmP, FrmGetObjectIndex(frmP, i));
		FrmRemoveObject(&frmP, FrmGetObjectIndex(frmP, i));
	}
	SetupDirectionButtons();
}

void ResetButtonList(void)
{
	int i;
	for (i=0; i<NUM_DIRECTIONS; i++)
	{
		StrCopy(ph_prefs.dir_list[i].cmd, default_dir_list[i].cmd);
		StrCopy(ph_prefs.dir_list[i].button, default_dir_list[i].button);
	}
}

void ResetCommandList(void)
{
	int i;
	for (i=0; i<NUM_COMMANDS; i++)
	{
		StrCopy(ph_prefs.command_list[i].cmd, default_command_list[i].cmd);
		ph_prefs.command_list[i].linefeed = default_command_list[i].linefeed;
	}
}

void ResetWordList(void)
{
	int i;
	for (i=0; i<NUM_WORDS; i++)
	{
		StrCopy(ph_prefs.word_list[i].word, default_word_list[i].word);
		ph_prefs.word_list[i].linefeed = default_word_list[i].linefeed;
	}
}


/***********************************************************************
 *
 * Setup management
 *
 ***********************************************************************/
Boolean SaveSetup(char *name)
{
	UInt16 card, new_record = 0;
	LocalID db = 0;
	DmOpenRef ref = NULL;
	MemHandle memH = NULL;
	MemPtr memP = NULL;
	Boolean result = false;
	
	for (card=0; card<MemNumCards(); card++)
	{
		if ((db = DmFindDatabase(card, name)))
		{
			sprintf(line, "Overwrite \"%s\" settings?", name);
			if (FrmCustomAlert(ConfirmAlert, line, "", "")==0)
			{
				DmDeleteDatabase(card, db);
				break;
			}
			else
				return false;
		}
	}
	if (db==0) card = 0;

	if ((DmCreateDatabase(card, name,
		PH_CREATOR_TYPE, PH_PREFS_TYPE, false))!=errNone)
	{
		return false;
	}
	if (!(db = DmFindDatabase(card, name)))
		return false;
		
	ref = DmOpenDatabase(card, db, dmModeWrite);
	if (!ref) goto SaveCleanup;		
	new_record = dmMaxRecordIndex;
	memH = DmNewRecord(ref, &new_record, sizeof(PalmHugoPreferenceType));
	if (!memH) goto SaveCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto SaveCleanup;
	
	ph_prefs.smartformatting = smartformatting;
	DmWrite(memP, 0, &ph_prefs, sizeof(PalmHugoPreferenceType));
	
	result = true;
	
SaveCleanup:
	if (memP) MemHandleUnlock(memH);
	if (memH) DmReleaseRecord(ref, new_record, true);
	if (ref) DmCloseDatabase(ref);
	
	if (result==false) DmDeleteDatabase(card, db);

	return result;
}

Boolean LoadSetup(char *name)
{
	int i;
	UInt16 card, record = 0;
	LocalID db = 0;
	DmOpenRef ref = NULL;
	MemHandle memH = NULL;
	MemPtr memP = NULL;
	Boolean result = false;
	
	for (card=0; card<MemNumCards(); card++)
	{
		if ((db = DmFindDatabase(card, name)))
		{
			break;
		}
	}
	if (db==0) return false;

	ref = DmOpenDatabase(card, db, dmModeReadOnly);
	if (!ref) goto LoadCleanup;		
	record = 0;
	memH = DmGetRecord(ref, record);
	if (!memH) goto LoadCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto LoadCleanup;

	for (i=0; i<sizeof(PalmHugoPreferenceType); i++)
		*((char *)&ph_prefs+i) = *((char *)memP + i);
		
	smartformatting = ph_prefs.smartformatting;	
	
	result = true;
	
LoadCleanup:
	if (memP) MemHandleUnlock(memH);
	if (memH) DmReleaseRecord(ref, record, true);
	if (ref) DmCloseDatabase(ref);
	
	if (result==false) DmDeleteDatabase(card, db);

	return result;
}


/***********************************************************************
 *
 * Saved state management
 *
 ***********************************************************************/
typedef struct
{
	// General engine stuff:
	long codeend;
	// Engine display state:
	int last_window_top, last_window_bottom, last_window_left, last_window_right;
	int lowest_windowbottom, physical_lowest_windowbottom;
	int physical_windowwidth, physical_windowheight,
		physical_windowtop, physical_windowleft,
		physical_windowbottom, physical_windowright;
	int currentpos, currentline;
	int currentfont;
	int charwidth, lineheight, FIXEDCHARWIDTH, FIXEDLINEHEIGHT;
	int current_text_x, current_text_y;
	int fcolor, bgcolor, icolor, default_bgcolor;
	int current_input_font;
	// Text buffer state:
	int tb_first_used, tb_first_unused, tb_last_used, tb_last_unused,
		tb_used, tb_unused;
} saved_state_struct;

Boolean SaveState()
{
	UInt16 card, record = 0;
	LocalID db = 0;
	DmOpenRef ref = NULL;
	MemHandle memH = NULL;
	MemPtr memP = NULL;
	Boolean result = false;
	saved_state_struct state;
	int i, len;
	UInt16 tb_len, cm_len;
	
	// If we haven't even started the game properly, we won't bother
	// saving it
	if (!got_first_input)
	{
		DeleteFile(CURRENT_GAME_SNAP);
		return false;
	}
	
	// We will only resume to a command input prompt, so if an input
	// isn't active, we won't restore the text buffer
	if (!during_player_input)
	{
		TB_Clear(0, 0, palm_screen_width, palm_screen_height);	
	}
	
	for (card=0; card<MemNumCards(); card++)
	{
		if ((db = DmFindDatabase(card, CURRENT_GAME_SNAP)))
		{
			DmDeleteDatabase(card, db);
			break;
		}
	}
	if (db==0) card = 0;
	
	if ((DmCreateDatabase(card, CURRENT_GAME_SNAP,
		PH_CREATOR_TYPE, PH_INTERNAL_TYPE, false))!=errNone)
	{
		return false;
	}
	if (!(db = DmFindDatabase(card, CURRENT_GAME_SNAP)))
		return false;

	ref = DmOpenDatabase(card, db, dmModeWrite);
	if (!ref) goto SaveCleanup;		

	// Write the game name (record 0)
	record = dmMaxRecordIndex;
	memH = DmNewRecord(ref, &record, MAXPATH+1);
	if (!memH) goto SaveCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto SaveCleanup;
	DmStrCopy(memP, 0, saved_state_game_name);
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);

	// Fill in the saved_state_struct
	state.codeend = codeend;
	state.last_window_top = last_window_top;
	state.last_window_bottom = last_window_bottom;
	state.last_window_left = last_window_left;
	state.last_window_right = last_window_right;
	state.lowest_windowbottom = lowest_windowbottom;
	state.physical_lowest_windowbottom = physical_lowest_windowbottom;
	state.physical_windowwidth = physical_windowwidth;
	state.physical_windowheight = physical_windowheight;
	state.physical_windowtop = physical_windowtop;
	state.physical_windowleft = physical_windowleft;
	state.physical_windowbottom = physical_windowbottom;
	state.physical_windowright = physical_windowright;
	state.currentpos = currentpos;
	state.currentline = currentline;
	state.currentfont = currentfont;
	state.charwidth = charwidth;
	state.lineheight = lineheight;
	state.FIXEDCHARWIDTH = FIXEDCHARWIDTH;
	state.FIXEDLINEHEIGHT = FIXEDLINEHEIGHT;
	state.current_text_x = current_text_x;
	state.current_text_y = current_text_y;
	state.fcolor = fcolor;
	state.bgcolor = bgcolor;
	state.icolor = icolor;
	state.default_bgcolor = default_bgcolor;
	state.current_input_font = current_input_font;
	// tb_list
	state.tb_first_used = tb_first_used;
	state.tb_first_unused = tb_first_unused;
	state.tb_last_used = tb_last_used;
	state.tb_last_unused = tb_last_unused;
	state.tb_used = tb_used;
	state.tb_unused = tb_unused;
	
	// Write global variable information (record 1)
	record = dmMaxRecordIndex;
	memH = DmNewRecord(ref, &record, sizeof(int)*(MAXGLOBALS+MAXLOCALS));
	if (!memH) goto SaveCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto SaveCleanup;
	DmWrite(memP, 0, var, sizeof(int)*(MAXGLOBALS+MAXLOCALS));
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);

	// Write the saved_state_struct	(record 2)
	record = dmMaxRecordIndex;
	memH = DmNewRecord(ref, &record, sizeof(saved_state_struct));
	if (!memH) goto SaveCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto SaveCleanup;
	DmWrite(memP, 0, &state, sizeof(saved_state_struct));
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);

	// Write tb_list (record 3)
	record = dmMaxRecordIndex;
	memH = DmNewRecord(ref, &record, sizeof(tb_list));
	if (!memH) goto SaveCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto SaveCleanup;
	DmWrite(memP, 0, tb_list, sizeof(tb_list));
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);
	
	// Figure out the total length of textbuffer data
	tb_len = 0;
	for (i=0; i<MAX_TEXTBUFFER_COUNT; i++)
	{
		if (tb_list[i].data)
			tb_len += StrLen(tb_list[i].data);
		tb_len++;
	}

	// Write the textbuffer data (record 4)
	record = dmMaxRecordIndex;
	memH = DmNewRecord(ref, &record, tb_len);
	if (!memH) goto SaveCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto SaveCleanup;
	tb_len = 0;
	for (i=0; i<MAX_TEXTBUFFER_COUNT; i++)
	{
		if (tb_list[i].data==NULL)
		{
			DmSet(memP, tb_len, 1, 0);
			tb_len++;
		}
		else
		{
			len = StrLen(tb_list[i].data);
			DmSet(memP, tb_len, 1, len);
			tb_len++;
			DmWrite(memP, tb_len, tb_list[i].data, len);
			tb_len+=len;
		}
	}
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);
	
	// Figure out the total length of context menu data
	cm_len = 1;	// for context_commands global
	for (i=0; i<context_commands; i++)
	{
		cm_len += StrLen(context_command[i]);
		cm_len++;
	}
	
	// Write the context menu data (record 5)
	record = dmMaxRecordIndex;
	memH = DmNewRecord(ref, &record, cm_len);
	if (!memH) goto SaveCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto SaveCleanup;
	cm_len = 0;
	DmSet(memP, cm_len, 1, context_commands);
	cm_len++;
	for (i=0; i<context_commands; i++)
	{
		int len;
		len = StrLen(context_command[i]);
		DmSet(memP, cm_len, 1, len);
		cm_len++;
		DmWrite(memP, cm_len, context_command[i], len);
		cm_len+=len;
	}
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);
	
	// Write the scrollback data (record 6)
#ifdef SCROLLBACK_DEFINED
	record = dmMaxRecordIndex;
	memH = DmNewRecord(ref, &record, StrLen(scrollback_buffer)+sizeof(int)*2);
	if (!memH) goto SaveCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto SaveCleanup;
	DmWrite(memP, 0, &scrollback_pos, sizeof(int));
	DmWrite(memP, sizeof(int), scrollback_buffer, StrLen(scrollback_buffer));
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);
#endif
	
	memP = NULL;
	memH = NULL;
	
	result = true;
		
SaveCleanup:
	if (memP) MemHandleUnlock(memH);
	if (memH) DmReleaseRecord(ref, record, true);
	if (ref) DmCloseDatabase(ref);
	
	if (result==false) DmDeleteDatabase(card, db);

	return result;
}

Boolean LoadState()
{
	/* Note that we don't actually restore the state here; we only
	   see if a state was saved.  We actually restore it in
	   AutoResume().
	*/
	
	UInt16 card, record = 0;
	LocalID db = 0;
	DmOpenRef ref = NULL;
	MemHandle memH = NULL;
	MemPtr memP = NULL;
	Boolean result = false;
	
	for (card=0; card<MemNumCards(); card++)
	{
		if ((db = DmFindDatabase(card, CURRENT_GAME_SNAP)))
		{
			break;
		}
	}
	if (db==0) return false;

	ref = DmOpenDatabase(card, db, dmModeReadOnly);
	if (!ref) goto LoadCleanup;
	
	// Read the game name
	record = 0;
	memH = DmGetRecord(ref, record);
	if (!memH) goto LoadCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto LoadCleanup;
	if (StrLen(memP) > MAXPATH) goto LoadCleanup;
	StrCopy(saved_state_game_name, memP);
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);

	memP = NULL;
	memH = NULL;

	auto_resuming = true;

	result = true;
	
LoadCleanup:
	if (memP) MemHandleUnlock(memH);
	if (memH) DmReleaseRecord(ref, record, true);
	if (ref) DmCloseDatabase(ref);
	
	if (result==false) DmDeleteDatabase(card, db);
	
	return result;
} 

int AutoResume(void)
{
	UInt16 card, record = 0;
	LocalID db = 0;
	DmOpenRef ref = NULL;
	MemHandle memH = NULL;
	MemPtr memP = NULL;
	int result = false;
	int i, len;
	saved_state_struct state;
	UInt16 tb_len, cm_len;
	Boolean tb_restore_error = false;
	
	for (card=0; card<MemNumCards(); card++)
	{
		if ((db = DmFindDatabase(card, CURRENT_GAME_SNAP)))
		{
			break;
		}
	}
	if (db==0) return false;

	ref = DmOpenDatabase(card, db, dmModeReadOnly);
	if (!ref) goto ResumeCleanup;
	
	// Read global variable information (record 1)
	record = 1;
	memH = DmGetRecord(ref, record);
	if (!memH) goto ResumeCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto ResumeCleanup;
	for (i=0; i<sizeof(int)*(MAXGLOBALS+MAXLOCALS); i++)
		*((char *)var+i) = *((char *)memP + i);	
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);

	// Read the saved_state_struct (record 2)
	record = 2;
	memH = DmGetRecord(ref, record);
	if (!memH) goto ResumeCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto ResumeCleanup;
	for (i=0; i<sizeof(saved_state_struct); i++)
		*((char *)&state+i) = *((char *)memP + i);	
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);

	// Extract state info from the saved_state_struct
	codeend = state.codeend;
	last_window_top = state.last_window_top;
	last_window_bottom = state.last_window_bottom;
	last_window_left = state.last_window_left;
	last_window_right = state.last_window_right;
	lowest_windowbottom = state.lowest_windowbottom;
	physical_lowest_windowbottom = state.physical_lowest_windowbottom;
	physical_windowwidth = state.physical_windowwidth;
	physical_windowheight = state.physical_windowheight;
	physical_windowtop = state.physical_windowtop;
	physical_windowleft = state.physical_windowleft;
	physical_windowbottom = state.physical_windowbottom;
	physical_windowright = state.physical_windowright;
	currentpos = state.currentpos;
	currentline = state.currentline;
//	currentfont = state.currentfont;
	charwidth = state.charwidth;
	lineheight = state.lineheight;
	FIXEDCHARWIDTH = state.FIXEDCHARWIDTH;
	FIXEDLINEHEIGHT = state.FIXEDLINEHEIGHT;
	current_text_x = state.current_text_x;
	current_text_y = state.current_text_y;
	fcolor = state.fcolor;
	bgcolor = state.bgcolor;
	icolor = state.icolor;
	default_bgcolor = state.default_bgcolor;
//	current_input_font = state.current_input_font;
	currentfont = state.current_input_font;		// since we'll be resuming to player input
	// tb_list
	tb_first_used = state.tb_first_used;
	tb_first_unused = state.tb_first_unused;
	tb_last_used = state.tb_last_used;
	tb_last_unused = state.tb_last_unused;
	tb_used = state.tb_used;
	tb_unused = state.tb_unused;
	
	// Read tb_list (record 3)
	record = 3;
	memH = DmGetRecord(ref, record);
	if (!memH) goto ResumeCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto ResumeCleanup;
	for (i=0; i<sizeof(tb_list); i++)
		*((char *)&tb_list+i) = *((char *)memP + i);	
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);

	// Read the textbuffer data (record 4)
	record = 4;
	memH = DmGetRecord(ref, record);
	if (!memH) goto ResumeCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto ResumeCleanup;
	tb_len = 0;
	for (i=0; i<MAX_TEXTBUFFER_COUNT; i++)
	{
		if (((char *)memP)[tb_len]==0)
		{
			tb_list[i].data = NULL;
			tb_len++;
		}
		else
		{
			len = *((char *)memP + tb_len);
			tb_len++;
			tb_list[i].data = hugo_blockalloc(len+1);
			if (tb_list[i].data)
			{
				StrNCopy(tb_list[i].data, (char *)((char *)memP+tb_len), len+1);
				tb_list[i].data[len] = '\0';
			}
			else
				tb_restore_error = true;
			tb_len+=len;
		}
	}
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);
	
	// Read the context menu data (record 5)
	record = 5;
	memH = DmGetRecord(ref, record);
	if (!memH) goto ResumeCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto ResumeCleanup;
	cm_len = 0;
	context_commands = *((char *)memP);
	cm_len++;
	for (i=0; i<context_commands; i++)
	{
		len = *((char *)memP + cm_len);
		cm_len++;
		StrNCopy(context_command[i], (char *)((char *)memP+cm_len), len);
		context_command[i][len] = '\0';
		cm_len+=len;
	}
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);
	
	// Read the scrollback data (record 6)
#ifdef SCROLLBACK_DEFINED
	record = 6;
	memH = DmGetRecord(ref, record);
	if (!memH) goto ResumeCleanup;
	memP = MemHandleLock(memH);
	if (!memP) goto ResumeCleanup;
	scrollback_pos = *((int *)memP)-1;
	StrCopy(scrollback_buffer, (char *)memP+sizeof(int));
	MemHandleUnlock(memH);
	DmReleaseRecord(ref, record, true);
#endif
	
	// Successful resume; do necessary setup here
	
	if (!tb_restore_error)
		RestoreWindow();
	else
	{
		hugo_clearfullscreen();
		auto_resuming = false;
	}
	
	hugo_font(currentfont);
	
	memP = NULL;
	memH = NULL;

	result = true;
	
ResumeCleanup:
	if (memP) MemHandleUnlock(memH);
	if (memH) DmReleaseRecord(ref, record, true);
	if (ref) DmCloseDatabase(ref);
	
	if (result==false) DmDeleteDatabase(card, db);
	
	if (!result)
	{
		FrmCustomAlert(FatalAlert, "FATAL ERROR",
			"Autoresume failure", "");
		exit(-1);		
	}
	
	return true;
}


#pragma mark ----- Window/display management -----

/***********************************************************************
 *
 * FUNCTION:    RestoreWindow
 *
 ***********************************************************************/
void RestoreWindow(void)
{
	Boolean inspt;
	FormPtr frmP = FrmGetActiveForm();

	int i, temp_x, temp_y, temp_font;
   	RectangleType rect;

	if (WinGetDrawWindow()!=FrmGetWindowHandle(frmP)) return;

	if (!override_window_restore)
	{
		inspt = InsPtEnabled();
		InsPtEnable(false);

		temp_x = current_text_x;
		temp_y = current_text_y;
		temp_font = currentfont;
		i = tb_first_used;
		
		WinPushDrawState();

		if (default_bgcolor!=DEF_BGCOLOR && ph_prefs.use_colors)
		{
			RctSetRectangle(&rect, 0, 0,
				palm_screen_width, palm_screen_height);
			if (palm_display_depth < 8 || !ph_prefs.use_colors)
			{
				if (default_bgcolor==7 || default_bgcolor==15)
				{
					WinSetBackColor(hugo_color(DEF_BGCOLOR));
				}
				else
				{
					WinSetBackColor(hugo_color(DEF_FCOLOR));
				}
			}
			else
				WinSetBackColor(hugo_color(default_bgcolor));
			WinEraseRectangle(&rect, 0);
		}

		while (i!=-1 && tb_list[i].data)
		{
			hugo_settextcolor(tb_list[i].fcolor);
			hugo_setbackcolor(tb_list[i].bgcolor);
			
			// Window area
			if (!StrCompare(tb_list[i].data, " _win"))
			{
			   	RctSetRectangle(&rect, tb_list[i].left, tb_list[i].top,
			   		tb_list[i].right-tb_list[i].left+1,
			   		tb_list[i].bottom-tb_list[i].top+1);
			   	SetWindowColors();
				WinEraseRectangle(&rect, 0);
				RestoreWindowColors();
			}
			
			// Word cell
			else
			{
				hugo_font(currentfont = tb_list[i].font);
				current_text_x = tb_list[i].left;
				current_text_y = tb_list[i].top;
				hugo_print(tb_list[i].data);
			}
			i = tb_list[i].next;
		}
		
		hugo_settextcolor(fcolor);
		hugo_setbackcolor(bgcolor);
		hugo_font(currentfont = temp_font);
		current_text_x = temp_x;
		current_text_y = temp_y;
		
		WinPopDrawState();

		if (getline_active && frmP==FrmGetFormPtr(MainForm))
		{
			UInt16 fieldindex;
			RctSetRectangle(&rect, getline_x, getline_y,
				physical_windowright-current_text_x+1, lineheight);
			FldSetBounds(GetObjectPtr(MainInputField), &rect);
			fieldindex = FrmGetObjectIndex(frmP, MainInputField);
			FrmShowObject(frmP, fieldindex);
			FrmSetFocus(frmP, fieldindex);
		}
		
		if (inspt) InsPtEnable(true);
	}
	override_window_restore = false;

	if (promptmore_waiting)
#ifdef TEXT_PROMPTMORE
		DrawPromptMore();
#else
		FrmShowObject(frmP, FrmGetObjectIndex(frmP, MainMoreButton));
#endif
}

#ifdef PROMPTMORE_REPLACED

#ifdef TEXT_PROMPTMORE
int pm_x, pm_y;
void DrawPromptMore(void)
{
	FontID f = FntGetFont();
	WinPushDrawState();
	FntSetFont(boldFont);
	WinSetTextColor(UIColorGetTableEntryIndex(UIFormFill));
	WinSetBackColor(UIColorGetTableEntryIndex(UIObjectForeground));
	WinDrawChars(" [MORE...] ", 11, pm_x, pm_y);
	FntSetFont(f);
	WinPopDrawState();
}
#endif

void PromptMore(void)
{
#ifdef TEXT_PROMPTMORE
/* For some reason, using the UI button is causing formatting problems with text
   immediately following the MORE prompt:
*/
	char buf[32];
	promptmore_waiting = true;
	pm_x = current_text_x;
	pm_y = current_text_y;
	DrawPromptMore();
	while (promptmore_waiting)
	{
		ProcessPalmOSEvents();
	}
	MemSet(buf, 32, ' ');
	SetWindowColors();
	WinDrawChars(buf, 32, pm_x, pm_y);
	RestoreWindowColors();
	full = 0;
#else
	FormPtr frmP;
	RectangleType promptmore_rect;
	UInt16 err;
	ControlPtr ctrlP = (ControlPtr)GetObjectPtr(MainMoreButton);

	frmP = FrmGetActiveForm();

	// Save the area behind the button
	FrmGetObjectBounds(frmP, FrmGetObjectIndex(frmP, MainMoreButton), &promptmore_rect);
	// In another brilliant API feature, FrmGetObjectBounds() won't actually
	// give you the bounds of the entire button:
	promptmore_rect.topLeft.x--;
	promptmore_rect.topLeft.y--;
	promptmore_rect.extent.x+=2;
	promptmore_rect.extent.y+=2;
	
	promptmore_winhandle = WinSaveBits(&promptmore_rect, &err);
	FrmShowObject(frmP, FrmGetObjectIndex(frmP, MainMoreButton));

	// Eat up any pending events
	ProcessPendingPalmOSEvents();
	while (hugo_iskeywaiting()) hugo_getkey();
	
	promptmore_waiting = true;
	while (promptmore_waiting)
	{
		ProcessPalmOSEvents();
	}
		
	full = 0;
	
	FrmHideObject(frmP, FrmGetObjectIndex(frmP, MainMoreButton));
	if (promptmore_winhandle) WinRestoreBits(promptmore_winhandle, promptmore_rect.topLeft.x, promptmore_rect.topLeft.y);
	promptmore_winhandle = NULL;

#endif	// !TEXT_PROMPTMORE
}
#endif	// PROMPTMORE_REPLACED


// Call SetWindowColors() before doing any drawing in the window, and
// be sure to call RestoreWindowColors() after.

void SetWindowColors(void)
{
	if (romVersion < 0x03503000) return;

	WinPushDrawState();
	
	if (ph_prefs.use_colors)
	{
		if (palm_display_depth < 8)
		{
			if (current_back_color==DEF_BGCOLOR || current_back_color==7 ||
				current_back_color==15)
			{
SetNormalColors:
				// Normal
				WinSetTextColor(hugo_color(DEF_FCOLOR));
				WinSetForeColor(hugo_color(DEF_FCOLOR));
				WinSetBackColor(hugo_color(DEF_BGCOLOR));
			}
			else
			{
				if (current_back_color!=default_bgcolor &&
					default_bgcolor!=DEF_BGCOLOR && default_bgcolor!=7
					&& default_bgcolor!=15)
				{
					goto SetNormalColors;
				}
				
				// Inverse
				WinSetTextColor(hugo_color(DEF_BGCOLOR));
				WinSetForeColor(hugo_color(DEF_BGCOLOR));
				WinSetBackColor(hugo_color(DEF_FCOLOR));
			}		
		}
		else
		{
			WinSetTextColor(hugo_color(current_text_color));
			WinSetForeColor(hugo_color(current_text_color));
			WinSetBackColor(hugo_color(current_back_color));
		}
	}
}

void RestoreWindowColors(void)
{
	if (romVersion >= 0x03503000)
	{
		WinPopDrawState();
	}
}


#pragma mark ----- Popup commands -----

/***********************************************************************
 *
 * Popup command management
 *
 ***********************************************************************/
char *MakePopupCommand(char *cmd, char *word)
{
	static char *p = NULL;
	int len;
	if (!StrCompare(cmd, "")) return NULL;
	
	len = StrLen(cmd) + StrLen(word) + 1;	// "cmd" + " " + "word"
		
	p = MemPtrNew(len+1);
	if (p==NULL) return NULL;
	
	StrCopy(p, cmd);
	if (StrStr(p, "^"))
	{
		StrCopy(StrStr(p, "^"), word);
		StrCat(p, StrStr(cmd, "^")+1);
	}
	else
	{
		StrCat(p, " ");
		StrCat(p, word);
	}
	
	return p;
}

void CheckForTrailingSpace(FieldPtr fieldP)
{
	UInt16 start, end;
	char *text = FldGetTextPtr(fieldP);
	
	// Don't add a space if there's an active selection
	FldGetSelection(fieldP, &start, &end);
	if (start!=end) return;
	
	end = FldGetInsPtPosition(fieldP);
	
	if (end)
	{
		if (text[end-1]!=' ')
			FldInsert(fieldP, " ", 1);				
	}
}

Boolean HandlePopupTap(char *w, int x, int y)
{
	char *entry[NUM_COMMANDS+1];
	Boolean linefeed[NUM_COMMANDS+1];
	int i, count, num = 0, selected, width = 40, prev;
	ListPtr listP;
	FormPtr frmP;
	FieldPtr fieldP;
	UInt16 list_id;
	
	if (!w) return false;

	frmP = FrmGetFormPtr(MainForm);
	fieldP = FrmGetObjectPtr(frmP, FrmGetObjectIndex(frmP, MainInputField));
	
	if (frmP==FrmGetActiveForm())
		list_id = MainPopupList;
	else
		list_id = ScrollbackPopupList;

	// Figure out how many commands we're going to display
	for (i=0; i<NUM_COMMANDS; i++)
	{
		if (StrCompare(ph_prefs.command_list[i].cmd, ""))
			num++;
	}
	if (!num) return false;
	
	// Figure out how many, if any, entries will come before
	// the word alone in the list
	if (y+num*menu_line_height > (palm_screen_height-menu_line_height))
		prev = ((y+num*menu_line_height)-(palm_screen_height-menu_line_height))/menu_line_height;
	else
		prev = 0;
		
	count = 0;
	for (i=0; i<prev; i++)
	{
		if (StrCompare(ph_prefs.command_list[i].cmd, ""))
		{
			entry[count] = MakePopupCommand(ph_prefs.command_list[i].cmd, w);
			if (!entry[count]) break;
			if (ph_prefs.command_list[i].linefeed)
				linefeed[count] = true;
			else
				linefeed[count] = false;

			if (width < FntCharsWidth(entry[count], StrLen(entry[count]))+4)
				width = FntCharsWidth(entry[count], StrLen(entry[count]))+4;
			count++;
		}
	}
	
	// If we had previous entries but didn't add them, there
	// was an error, so bail
	if (prev && count==0) return false;
	
	// Now add the word alone
	entry[count] = MemPtrNew(StrLen(w)+1);
	if (entry[count])
	{
		StrCopy(entry[count], w);
		linefeed[count] = false;
		count++;
	}
	
	// And add the remaining commands
	for (i=prev; i<NUM_COMMANDS; i++)
	{
		if (StrCompare(ph_prefs.command_list[i].cmd, ""))
		{
			entry[count] = MakePopupCommand(ph_prefs.command_list[i].cmd, w);
			if (!entry[count]) break;
			if (ph_prefs.command_list[i].linefeed)
				linefeed[count] = true;
			else
				linefeed[count] = false;
			if (width < FntCharsWidth(entry[count], StrLen(entry[count]))+4)
				width = FntCharsWidth(entry[count], StrLen(entry[count]))+4;
			count++;
		}
		ProcessPendingPalmOSEvents();
	}
	
	if (count)
	{
		Boolean ret = false;
		int visible = count;
		RectangleType rect;
		
		x-=10;
		y-=6;
		if (width > palm_screen_width-10)
			width = palm_screen_width-10;
		if (x > palm_screen_width - width)
			x = palm_screen_width - width;
		listP = GetObjectPtr(list_id);
		LstSetListChoices(listP, (char **)&entry, count);
		LstSetSelection(listP, prev);
		LstSetPosition(listP, x, y-prev*menu_line_height);
		LstSetHeight(listP, visible);
		//listP->bounds.extent.x = width;
		FrmGetObjectBounds(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(), list_id), &rect);
		rect.extent.x = width;
		FrmSetObjectBounds(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(), list_id), &rect);
		
		selected = LstPopupList(listP);
		if (selected>=0)
		{
			//if (selected!=prev) 
			//	FldDelete(fieldP, 0, FldGetTextLength(fieldP));
			CheckForTrailingSpace(fieldP);
			FldInsert(fieldP, entry[selected], StrLen(entry[selected]));
			if (linefeed[selected])
				EvtEnqueueKey(10, 0, 0);
			else
				FldInsert(fieldP, " ", 1);
			ret = true;
		}
		
		while (count)
		{
			MemPtrFree(entry[--count]);
		}
		
		return ret;
	}
	
	return false;
}
				

#pragma mark ----- Form handlers -----

/***********************************************************************
 *
 * FUNCTION:    MainFormDoCommand
 *
 * DESCRIPTION: This routine performs the menu command specified.
 *
 * PARAMETERS:  command  - menu item id
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean MainFormDoCommand(UInt16 command)
{
	FormPtr frmP;
	
	MenuEraseStatus(NULL);

	switch (command)
	{
#ifdef TYPED_STORY_COMMANDS
		case StorySaveStory:
		case StoryRestoreStory:
		case EditUndo:
		case sysEditMenuUndoCmd:
		{
			FieldPtr fieldP;
			char *cmd;
			
			if (!during_player_input)
			{
				FrmAlert(NotInputAlert);
				return true;
			}
			if (command==StorySaveStory)
				cmd = "Save";
			else if (command==StoryRestoreStory)
				cmd = "Restore";
			else
				cmd = "Undo";
				
			fieldP = GetObjectPtr(MainInputField);
			FldDelete(fieldP, 0, FldGetTextLength(fieldP));
			CheckForTrailingSpace(fieldP);
			FldInsert(fieldP, cmd, StrLen(cmd));
			EvtEnqueueKey(10, 0, 0);
			
			return true;
		}
#else
		case StorySaveStory:
		case StoryRestoreStory:
#endif
		case StoryRestartStory:
			if (during_player_input)
			{
				getline_active = false;
				processed_shortcut = command;
			}
			else
				FrmAlert(NotInputAlert);
			return true;
			
		case StoryQuitStory:
			if (FrmCustomAlert(ConfirmAlert, "Abandon the current story?", "", "")==0)
			{
				override_window_restore = true;
				var[endflag] = -1;
				exit(-1);
			}
			return true;

		case StoryAboutHugo:
			frmP = FrmInitForm(AboutForm);
			FrmDoDialog(frmP);		// Display the About Box
			FrmDeleteForm(frmP);
			break;
			
#ifndef TYPED_STORY_COMMANDS
		case EditUndo:
		case sysEditMenuUndoCmd:
			if (during_player_input)
			{
				getline_active = false;
				processed_shortcut = EditUndo;
			}
			else
				FrmAlert(NotInputAlert);
			return true;
#endif			
		case EditCopy:
		case EditCut:
		case EditPaste:
		case EditSelectAll:
		case sysEditMenuCopyCmd:
		case sysEditMenuCutCmd:
		case sysEditMenuPasteCmd:
		case sysEditMenuSelectAllCmd:
			if (getline_active)
			{
				FieldPtr fieldP = GetObjectPtr(MainInputField);
				if (command==EditCopy || command==sysEditMenuCopyCmd)
				{
					FldCopy(fieldP);
					break;
				}
				else if (command==EditCut || command==sysEditMenuCutCmd)
				{
					FldCut(fieldP);
					break;
				}
				else if (command==EditPaste || command==sysEditMenuPasteCmd)
				{
					FldPaste(fieldP);
					break;
				}
				else if (command==EditSelectAll || command==sysEditMenuSelectAllCmd)
				{
					FldSetSelection(fieldP, 0, StrLen(FldGetTextPtr(fieldP)));
					break;
				}
			}
			return true;
			
		case EditShowScrollback:
			RunDialog(ScrollbackForm);
			return true;

		case EditGraffitiReference:
		case sysEditMenuGraffitiCmd:
			SysGraffitiReferenceDialog(referenceDefault);
			break;
			
		case EditKeyboard:
		case sysEditMenuKeyboardCmd:
			SysKeyboardDialog(kbdDefault);
			break;
			
		case SetupManageCommands:
			RunDialog(CommandsForm);
			break;
			
		case SetupManageButtons:
			RunDialog(ButtonsForm);
			if (dialog_ok) ResetDirectionButtons();
			break;
			
		case SetupManageWordList:
			RunDialog(WordsForm);
			break;
			
		case SetupOptions:
			RunDialog(OptionsForm);
			break;
			
		case SetupSaveSetup:
			file_select_type = PH_PREFS_TYPE;
			RunDialog(FileSaveForm);
			if (dialog_ok)
			{
				if (!SaveSetup(file_selected))
					PopupMessage("Unable to save settings", file_selected);
			}
			return true;
			
		case SetupLoadSetup:
			file_select_type = PH_PREFS_TYPE;
			RunDialog(FileOpenForm);
			if (file_select_error==PH_NO_FILES)
			{
				PopupMessage("Saved settings available", "None");
				break;
			}
			if (dialog_ok)
			{
				if (!LoadSetup(file_selected))
					PopupMessage("Unable to load settings", file_selected);
				else
					ResetDirectionButtons();
			}
			break;
			
		default:
			return false;
	}
	
	return true;
}


/***********************************************************************
 *
 * FUNCTION:    MainFormHandleEvent
 *
 * DESCRIPTION: This routine is the event handler for the 
 *              "MainForm" of this application.
 *
 * PARAMETERS:  eventP  - a pointer to an EventType structure
 *
 * RETURNED:    true if the event has handle and should not be passed
 *              to a higher level handler.
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean MainFormHandleEvent(EventPtr eventP)
{
	FormPtr frmP;
	static char found_word[12] = "";	// good enough precision
	static RectangleType found_rect;
	static Boolean found_rect_flag = false;
	
	switch (eventP->eType) 
	{
		case menuCmdBarOpenEvent:
			override_window_save = true;
			break;
		
		case menuEvent:
			MainFormDoCommand(eventP->data.menu.itemID);
			return true;

		case frmOpenEvent:
		{
			// Display the main form
			frmP = FrmGetActiveForm();
			FrmDrawForm(frmP);
			
			if (!game_started) return true;
			
			RestoreWindow();
			if (getline_active)
			{
				UInt16 fieldindex;
				RectangleType rect;
				RctSetRectangle(&rect, getline_x, getline_y,
					physical_windowright-current_text_x+1, lineheight);
				FldSetBounds(GetObjectPtr(MainInputField), &rect);
				fieldindex = FrmGetObjectIndex(frmP, MainInputField);
				FrmShowObject(frmP, fieldindex);
				FrmSetFocus(frmP, fieldindex);
			}
			return true;
		}

		case frmUpdateEvent:
			frmP = FrmGetActiveForm();
			FrmDrawForm(frmP);
			RestoreWindow();
			return true;

		case keyDownEvent:
			if (!promptmore_waiting)
			{
				if (!getline_active)
				{
					PushKeypress(eventP->data.keyDown.chr);
					return true;
				}
				
				// Enter
				else if (eventP->data.keyDown.chr==10)
				{
					getline_active = false;
					return true;
				}
				
				// Up/down keys
				else if (eventP->data.keyDown.chr==11 || eventP->data.keyDown.chr==12)
				{
					RunDialog(ScrollbackForm);
					return true;
				}
			}
			else	// promptmore_waiting
			{
				if (eventP->data.keyDown.chr==' ' || eventP->data.keyDown.chr==10)
				{
					promptmore_waiting = false;
					return true;
				}
				else if (eventP->data.keyDown.chr==11 || eventP->data.keyDown.chr==12)
				{
					RunDialog(ScrollbackForm);
					return true;
				}
			}
			break;
			
		case penDownEvent:
			if (getline_active && TB_FindWord(eventP->screenX, eventP->screenY))
			{
				StrNCopy(found_word, TB_FindWord(eventP->screenX, eventP->screenY), 12);
				RctSetRectangle(&found_rect, tb_list[tb_selected].left, tb_list[tb_selected].top,
					tb_list[tb_selected].right-tb_list[tb_selected].left,
					tb_list[tb_selected].bottom-tb_list[tb_selected].top);
				WinInvertRectangle(&found_rect, 0);
				found_rect_flag = true;
			}
			else if (!getline_active && !promptmore_waiting)
			{
				// Send a "mouse click" keypress
				PushKeypress(1);
				PushKeypress((eventP->screenX-physical_windowleft)/FIXEDCHARWIDTH + 1);
				PushKeypress((eventP->screenY-physical_windowtop)/FIXEDLINEHEIGHT + 1);
				return true;
			}
			break;
			
		case penUpEvent:
			if (found_rect_flag)
			{
				WinInvertRectangle(&found_rect, 0);
				found_rect_flag = false;
			}
			if (getline_active && TB_FindWord(eventP->screenX, eventP->screenY))
			{
				if (!StrNCompare(found_word, TB_FindWord(eventP->screenX, eventP->screenY), 11) &&
					StrCompare(found_word, ""))
				{
					override_window_restore = true;
					HandlePopupTap(TB_FindWord(eventP->screenX, eventP->screenY),
						eventP->screenX, eventP->screenY);
				}
				StrCopy(found_word, "");
			}
			return true;

		case ctlSelectEvent:
		{
			UInt16 control = eventP->data.ctlSelect.controlID;
			
			if (control==MainMoreButton)
			{
				promptmore_waiting = false;
				return true;
			}
			
			if (control==WORD_BUTTON && getline_active)
			{
				char *entry[16];
				char free_last = false;
				Boolean linefeed[16];
				int i, count = 0, visible, selected = 0, width;
				ListPtr listP;
				FieldPtr fieldP = GetObjectPtr(MainInputField);
				FormPtr frmP = FrmGetActiveForm();
				RectangleType rect;
				
				CtlSetValue(GetObjectPtr(control), 0);

				override_window_restore = true;
			
				// First priority is to display context commands, if any
				if (context_commands && getline_active)
				{
					width = 40;
							
					for (i=0; i<context_commands; i++)
					{
						// Omit separators ("-")
						if (StrCompare(context_command[i], "") && StrCompare(context_command[i], "-"))
						{
							// Then add the context commands
							entry[count] = context_command[i];
							if (StrStr(entry[count], "..."))
								linefeed[count] = false;
							else
								linefeed[count] = true;
							if (width < FntCharsWidth(entry[count], StrLen(entry[count]))+4)
								width = FntCharsWidth(entry[count], StrLen(entry[count]))+4;
							count++;
						}
					}
					
					// First add the '...' option
					entry[count] = MemPtrNew(4);
					if (entry[count]!=NULL)
					{
						StrCopy(entry[count], "...");
						count++;
						free_last = true;
					}

					if (count)
					{
						visible = count;
						if (count > 12) visible = 12;
						if (width>palm_screen_width-10)
							width = palm_screen_width-10;
						listP = GetObjectPtr(MainPopupList);
						LstSetListChoices(listP, (char **)&entry, count);
						LstSetPosition(listP, 0, palm_screen_height);
						LstSetHeight(listP, visible);
						//listP->bounds.extent.x = listP->bounds.topLeft.x+width;
						FrmGetObjectBounds(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(), MainPopupList), &rect);
						rect.extent.x = width;
						FrmSetObjectBounds(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(), MainPopupList), &rect);
						
						selected = LstPopupList(listP);
						
						// count-1 is '...', handled below
						if (selected>=0 && selected<count-1)
						{
							//FldDelete(fieldP, 0, FldGetTextLength(fieldP));
							CheckForTrailingSpace(fieldP);
							FldInsert(fieldP, entry[selected], StrLen(entry[selected])-(StrStr(entry[selected], "...")!=NULL?3:0));
							if (linefeed[selected])
								EvtEnqueueKey(10, 0, 0);
							else
								FldInsert(fieldP, " ", 1);
						}
					}
					
					if (free_last)
						// Free the '...' option
						MemPtrFree(entry[count-1]);
				}

				// count-1 is '...' (or no context commands at all)
				if (context_commands && (selected!=count-1 || count==0))
					return true;
				
				// User word list
				count = 0;
				width = 40;
				for (i=0; i<NUM_WORDS; i++)
				{
					if (StrCompare(ph_prefs.word_list[i].word, ""))
					{
						if (ph_prefs.word_list[i].linefeed)
							linefeed[count] = true;
						else
							linefeed[count] = false;
						entry[count] = ph_prefs.word_list[i].word;
						if (width < FntCharsWidth(entry[count], StrLen(entry[count]))+4)
							width = FntCharsWidth(entry[count], StrLen(entry[count]))+4;
						count++;
					}
				}
				
				// Draw the word list
				if (count)
				{
					visible = count;
					if (count > 12) visible = 12;
					listP = GetObjectPtr(MainPopupList);
					LstSetListChoices(listP, (char **)&entry, count);
					LstSetPosition(listP, 0, palm_screen_height-visible*FntCharHeight());
					LstSetHeight(listP, visible);
					//listP->bounds.extent.x = listP->bounds.topLeft.x+width;
					FrmGetObjectBounds(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(), MainPopupList), &rect);
					rect.extent.x = width;
					FrmSetObjectBounds(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(), MainPopupList), &rect);
					
					LstSetSelection(listP, -1);	// no selection
					selected = LstPopupList(listP);
					if (selected>=0)
					{
						CheckForTrailingSpace(fieldP);
						FldInsert(fieldP, entry[selected], StrLen(entry[selected]));
						if (linefeed[selected])
							EvtEnqueueKey(10, 0, 0);
						else
							FldInsert(fieldP, " ", 1);
					}
				}
				
				return true;
			}
			else if (control>WORD_BUTTON && control<=WORD_BUTTON+NUM_DIRECTIONS && getline_active)
			{
				CtlSetValue(GetObjectPtr(control), 0);
				
				if (StrCompare(ph_prefs.dir_list[control-WORD_BUTTON-1].cmd, ""))
				{
					FieldPtr fieldP = GetObjectPtr(MainInputField);
					//FldDelete(fieldP, 0, FldGetTextLength(fieldP));
					CheckForTrailingSpace(fieldP);
					FldInsert(fieldP, ph_prefs.dir_list[control-WORD_BUTTON-1].cmd,
						StrLen(ph_prefs.dir_list[control-WORD_BUTTON-1].cmd));
					EvtEnqueueKey(10, 0, 0);
				}
				return true;
			}
			break;
		}

		case winEnterEvent:
		{
			if (romVersion < 0x03503000) return true;
			
			if (eventP->data.winEnter.enterWindow==(WindowType *)FrmGetFormPtr(MainForm))
			{
				UIColorSetTableEntry(UIFieldText, &field_text);
				UIColorSetTableEntry(UIFieldBackground, &field_back);
				UIColorSetTableEntry(UIFieldCaret, &field_caret);
				RestoreWindow();
			}
			return true;
		}
		
		case winExitEvent:
		{
			if (romVersion < 0x03503000) return true;
			
			if (eventP->data.winExit.exitWindow==(WindowType *)FrmGetFormPtr(MainForm))
			{
				UIColorGetTableEntryRGB(UIFieldText, &field_text);
				UIColorGetTableEntryRGB(UIFieldBackground, &field_back);
				UIColorGetTableEntryRGB(UIFieldCaret, &field_caret);
				UIColorSetTableEntry(UIFieldText, &default_field_text);
				UIColorSetTableEntry(UIFieldBackground, &default_field_back);
				UIColorSetTableEntry(UIFieldCaret, &default_field_caret);
			}
			break;
		}
	}
	
	return false;
}


/***********************************************************************
 *
 * FUNCTION:    FileFormHandleEvent
 *
 * Used for both FileOpen and FileSave forms.
 *
 ***********************************************************************/
static Boolean FileFormHandleEvent(EventPtr eventP)
{
	static UInt16 formtype;
	static int file_count;
	static char *file[MAX_FILELIST] = {};
	static UInt16 file_card[MAX_FILELIST];
	static ListType *listP;

	FormPtr frmP = NULL;
	int i;

	switch (eventP->eType) 
	{
		case frmOpenEvent:
		{
			UInt16 card;
			
			StrCopy(file_selected, "");
			file_select_error = PH_NO_ERROR;
			
			frmP = FrmGetActiveForm();
			formtype = FrmGetFormId(frmP);
				
			// Determine the available PDBs
			file_count = 0;
			for (card=0; card<=MemNumCards(); card++)
			{
				for (i=0; i<DmNumDatabases(card); i++)
				{
					char name[32];
					UInt32 typeP, creatorP;
					if (DmDatabaseInfo(card, DmGetDatabase(card, i), name,
						NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
						&typeP, &creatorP)==errNone)
					{
						if (typeP==file_select_type && creatorP==PH_CREATOR_TYPE)
						{
							if ((file[file_count] = (char *)MemPtrNew(StrLen(name)+6+1)))
							{
								StrCopy(file[file_count], name);
								file_card[file_count] = card;
								if (++file_count==MAX_FILELIST)
									break;
							}
							else
								break;
						}
					}				
				}
			}
			
			// If no PDBs are available
			if (file_count==0 && formtype==FileOpenForm)
			{
				file_select_error = PH_NO_FILES;
				ExitDialog(false);
				return true;
			}
						
			// Load the form list with the file titles
			listP = (ListType *)GetObjectPtr(
				(formtype==FileOpenForm)?FileOpenListList:FileSaveListList);
			LstSetListChoices(listP, file, file_count);
			
			// Set the appropriate title, if required
			if (formtype==FileOpenForm)
			{
				if (file_select_type==PH_GAME_TYPE)
					FrmSetTitle(frmP, "Choose a Hugo Game");
				else if (file_select_type==PH_SAVE_TYPE)
					FrmSetTitle(frmP, "Restore a Saved Game");
				else
					FrmSetTitle(frmP, "Hugo Open");
			}
			else if (formtype==FileSaveForm)
			{
				if (file_select_type==PH_SAVE_TYPE)
					FrmSetTitle(frmP, "Save Game");
				else
					FrmSetTitle(frmP, "Hugo Save");
			}
			
			// Display the file form
			FrmDrawForm(frmP);
			
			if (formtype!=FileOpenForm)
				FrmSetFocus(frmP, FrmGetObjectIndex(frmP, FileSaveInputField));
			
			return true;
		}
		
		case frmCloseEvent:
		{
// If we popped up this form from MainForm instead of displaying it at
// the start, we have to jump here manually because frmCloseEvent doesn't
// get sent.
FreeFilenames:
			for (i=0; i<file_count; i++)
			{
				MemPtrFree(file[i]);
			}
			file_count = 0;
			break;
		}
		
		case lstSelectEvent:
		{
			FieldPtr fieldP;
			char *sel;
			
			if (formtype==FileOpenForm)
				break;
				
			fieldP = (FieldType *)GetObjectPtr(FileSaveInputField);
			sel = LstGetSelectionText(listP, LstGetSelection(listP));
			FldDelete(fieldP, 0, FldGetTextLength(fieldP));
			FldInsert(fieldP, sel, StrLen(sel));
			break;
		}
		
		case ctlSelectEvent:
		{
			int selection = LstGetSelection(listP);
				
			switch (eventP->data.ctlEnter.controlID)
			{
				case FileOpenDeleteButton:
				case FileSaveDeleteButton:
				{
					if (FrmCustomAlert(DeleteAlert, file[selection], "", "")==0)
					{
						DmDeleteDatabase(file_card[selection], 
							DmFindDatabase(file_card[selection], file[selection]));
						MemPtrFree(file[selection]);
						for (i=selection; i<file_count-1; i++)
						{
							file[i] = file[i+1];
							file_card[i] = file_card[i+1];
						}
						file_count--;
						
						if (file_count)
						{
							LstSetListChoices(listP, file, file_count);
							LstDrawList(listP);
							break;
						}
						else if (formtype!=FileOpenForm)
						{
							break;
						}
						// else fall through to Cancel if not FileSave
					}
					else
						break;
				}
				case FileOpenCancelButton:
				case FileSaveCancelButton:
				{
					file_select_error = PH_NO_SELECTION;
					StrCopy(file_selected, "");
					ExitDialog(false);
					goto FreeFilenames;
					break;
				}
				case FileOpenOKButton:
				{
					StrCopy(file_selected, file[selection]);
					ExitDialog(true);
					goto FreeFilenames;
					break;
				}
				case FileSaveOKButton:
				{
					StrCopy(file_selected, FldGetTextPtr(GetObjectPtr(FileSaveInputField)));
					ExitDialog(true);
					goto FreeFilenames;
					break;
				}
			}
			return true;					
		}
	}
	
	return false;
}


/***********************************************************************
 *
 * FUNCTION:    EditFormHandleEvent
 *
 * Used for Word, Command, and Direction management forms.
 *
 ***********************************************************************/
static Boolean EditFormHandleEvent(EventPtr eventP)
{
	static UInt16 formtype;
	FormPtr frmP;
	FieldPtr fieldP;
	int i;
	
	switch (eventP->eType) 
	{
		case frmOpenEvent:
		{
			frmP = FrmGetActiveForm();
			formtype = FrmGetFormId(frmP);
			
RedrawForm:
			switch (formtype)
			{
				case CommandsForm:
				{
					for (i=0; i<NUM_COMMANDS; i++)
					{
						ProcessPendingPalmOSEvents();
						fieldP = GetObjectPtr(CommandsLineField+i*2);
						FldDelete(fieldP, 0, FldGetTextLength(fieldP));
						FldInsert(fieldP, ph_prefs.command_list[i].cmd, StrLen(ph_prefs.command_list[i].cmd));
						CtlSetValue(GetObjectPtr(CommandsLFCheckbox+i*2), ph_prefs.command_list[i].linefeed);
					}
					break;
				}
					
				case ButtonsForm:
				{
					for (i=0; i<NUM_DIRECTIONS; i++)
					{
						ProcessPendingPalmOSEvents();
						fieldP = GetObjectPtr(ButtonsNorthField+i*2);
						FldDelete(fieldP, 0, FldGetTextLength(fieldP));
						FldInsert(fieldP, ph_prefs.dir_list[i].cmd, StrLen(ph_prefs.dir_list[i].cmd));
						fieldP = GetObjectPtr(ButtonsNField+i*2);
						FldDelete(fieldP, 0, FldGetTextLength(fieldP));
						FldInsert(fieldP, ph_prefs.dir_list[i].button, StrLen(ph_prefs.dir_list[i].button));
					}				
					break;
				}
					
				case WordsForm:
				{
					for (i=0; i<NUM_WORDS; i++)
					{
						ProcessPendingPalmOSEvents();
						fieldP = GetObjectPtr(WordsEntryField+i*2);
						FldDelete(fieldP, 0, FldGetTextLength(fieldP));
						FldInsert(fieldP, ph_prefs.word_list[i].word, StrLen(ph_prefs.word_list[i].word));
						CtlSetValue(GetObjectPtr(WordsLFCheckbox+i*2), ph_prefs.word_list[i].linefeed);
					}
					break;
				}
			}
					
			FrmDrawForm(frmP);
			return true;
		}
		
		case ctlSelectEvent:
		{
			switch (eventP->data.ctlEnter.controlID)
			{
				case CommandsCancelButton:
				case ButtonsCancelButton:
				case WordsCancelButton:
					ExitDialog(false);
					return true;
					
				case CommandsOKButton:
				{
					for (i=0; i<NUM_COMMANDS; i++)
					{
						fieldP = GetObjectPtr(CommandsLineField+i*2);
						if (FldGetTextPtr(fieldP))
							StrNCopy(ph_prefs.command_list[i].cmd, FldGetTextPtr(fieldP), COMMAND_LENGTH);
						else
							StrCopy(ph_prefs.command_list[i].cmd, "");
						ph_prefs.command_list[i].linefeed = CtlGetValue(GetObjectPtr(CommandsLFCheckbox+i*2));
					}
					ExitDialog(true);
					return true;
				}
					
				case ButtonsOKButton:
				{
					for (i=0; i<NUM_DIRECTIONS; i++)
					{
						fieldP = GetObjectPtr(ButtonsNorthField+i*2);
						if (FldGetTextPtr(fieldP))
							StrNCopy(ph_prefs.dir_list[i].cmd, FldGetTextPtr(fieldP), DIRECTION_LENGTH);
						else
							StrCopy(ph_prefs.dir_list[i].cmd, "");
						fieldP = GetObjectPtr(ButtonsNField+i*2);
						if (FldGetTextPtr(fieldP))
							StrNCopy(ph_prefs.dir_list[i].button, FldGetTextPtr(fieldP), DIRBUTTON_LENGTH);
						else
							StrCopy(ph_prefs.dir_list[i].button, "");
					}
					ExitDialog(true);
					return true;
				}
				
				case WordsOKButton:
				{
					for (i=0; i<NUM_WORDS; i++)
					{
						fieldP = GetObjectPtr(WordsEntryField+i*2);
						if (FldGetTextPtr(fieldP))
							StrNCopy(ph_prefs.word_list[i].word, FldGetTextPtr(fieldP), WORD_LENGTH);
						else
							StrCopy(ph_prefs.word_list[i].word, "");
						ph_prefs.word_list[i].linefeed = CtlGetValue(GetObjectPtr(WordsLFCheckbox+i*2));
					}
					ExitDialog(true);
					return true;
				}
				
				case CommandsResetButton:
				{
					if (FrmCustomAlert(ResetListAlert, "commands", "", "")==0)
					{
						ResetCommandList();
						goto RedrawForm;
					}
				}
				
				case ButtonsResetButton:
				{
					if (FrmCustomAlert(ResetListAlert, "buttons", "", "")==0)
					{
						ResetButtonList();
						goto RedrawForm;
					}
				}
				
				case WordsResetButton:
				{
					if (FrmCustomAlert(ResetListAlert, "words", "", "")==0)
					{
						ResetWordList();
						goto RedrawForm;
					}
				}
			}
			break;
		}
		
		case menuEvent:
		{
			// There's only one menu in these forms, and that's
			// the Edit menu
			UInt16 focus;
			frmP = FrmGetActiveForm();
			focus = FrmGetFocus(frmP);
			if (FrmGetObjectType(frmP, focus)!=frmFieldObj)
				return true;
			fieldP = (FieldPtr)FrmGetObjectPtr(frmP, focus);
			switch (eventP->data.menu.itemID)
			{
				case EditFormCopy:
				case sysEditMenuCopyCmd:
				{
					FldCopy(fieldP);
					break;
				}
				case EditFormCut:
				case sysEditMenuCutCmd:
				{
					FldCut(fieldP);
					break;
				}
				case EditFormPaste:
				case sysEditMenuPasteCmd:
				{
					FldPaste(fieldP);
					break;
				}
				case EditFormSelectAll:
				case sysEditMenuSelectAllCmd:
				{
					FldSetSelection(fieldP, 0, StrLen(FldGetTextPtr(fieldP)));
					break;
				}
			}
			return true;
		}
	}
	
	return false;
}
 
 
/***********************************************************************
 *
 * FUNCTION:    ScrollbackFormHandleEvent
 *
 ***********************************************************************/
#define max(a,b) (((a)>=(b))?(a):(b))
#define min(a,b) (((a)<=(b))?(a):(b))
 
static UInt16 GetPosOfPoint(FieldType *fldP, Coord x, Coord y, Boolean *leftSideP)
{
	UInt16 pos;
	UInt16 yPos;
	UInt16 width;
	UInt16 lastWidth;
	UInt16 charSize;
	UInt16 charsPerLine;
	Char *chars;
	RectangleType rect;
	FieldAttrType attr;
	
	*leftSideP = false;

	if (!FldGetTextLength(fldP))
		return (0);
		
	FldGetBounds(fldP, &rect);
	FldGetAttributes(fldP, &attr);

	// Compute the x position of the insertion point.  First, adjust the 
	// x coordinate such that it is within the bounds of field.
	x = max(0, x - rect.topLeft.x);
	
/* This will never be called for a single-line field
	if (attr.singleLine)
	{
		pos = 0;
		charsPerLine = FldGetTextLength(fldP);

		if (attr.justification == rightAlign)
		{
			if (x < rect.extent.x - FntCharsWidth(FldGetTextPtr(fldP), charsPerLine))
			{
				*leftSideP = true;
				return (0);
			}
		}
	}

	else 
*/
	{
		// Compute the line that the y-coordinate is on.
		yPos = max (((y - rect.topLeft.y) / FntLineHeight()), 0);
		yPos = min (yPos, rect.extent.y / FntLineHeight() - 1);

		// Accessing these members behaves badly on OS 5; to be honest, everything
		// behaves badly on OS 5:  just try watching fldP and marvel at how it's
		// an invalid field structure, yet you can get the bounds rect from it that
		// in no way resembles the field's data.  PalmOSGlue.lib to the rescue.
		//pos = fldP->lines[yPos].start;
		//charsPerLine = fldP->lines[yPos].length;
		FldGlueGetLineInfo(fldP, yPos, &pos, &charsPerLine);		

		if (charsPerLine && (FldGetTextPtr(fldP)[pos + charsPerLine - 1] == linefeedChr))
			charsPerLine--;
	}

	// Determine where in the line (the character position) the x coordinate
	// passed is. 
	if (charsPerLine)
	{
		width = 0;
		lastWidth = 0;
		while (charsPerLine) 
		{
			chars = &FldGetTextPtr(fldP)[pos];
			if (*chars != tabChr)
			{
				charSize = TxtNextCharSize(chars, 0);
				width += FntCharsWidth(chars, charSize);
			}
			else
			{
				width += fntTabChrWidth - (width % fntTabChrWidth);
				charSize = 1;
			}
			
			if (width >= x)
			{
				if ((x - lastWidth) >= ((width - lastWidth) >> 1))
				{
					pos += charSize;
					*leftSideP = true;
				}
				break;
			}

			pos += charSize;
			charsPerLine -= charSize;
			lastWidth = width;
		}
		if (width < x)
			*leftSideP = true;
	}

	return (pos);
}

char *GetScrollbackWord(int x, int y)
{
	FieldPtr fieldP;
	Boolean left;
	int pos, count;
	
	fieldP = GetObjectPtr(ScrollbackTextField);
	
	pos = GetPosOfPoint(fieldP, x, y, &left);
	
	if (scrollback_buffer[pos]=='\n') return NULL;	// for TB_FindWord() consistency
	
	// Find the start of the tapped word
	while (pos > 0)
	{
		//if ((scrollback_buffer[pos-1]>='0' && scrollback_buffer[pos-1]<='9') || scrollback_buffer[pos-1]>='A')
		if ((unsigned char)scrollback_buffer[pos-1]>' ')
			pos--;
		else
			break;
	}
	// Make sure it starts on an alphanumeric
	while (scrollback_buffer[pos]<'0' || (scrollback_buffer[pos]>'9' && (unsigned char)scrollback_buffer[pos]<'A'))
	{
		if ((unsigned char)scrollback_buffer[++pos]<' ') return NULL;
	}
	
	// Find the end of the tapped word
	count=0;
	while (true)
	{
		line[count] = scrollback_buffer[pos+count];
		if ((unsigned char)line[count]<=' ')
			break;
		count++;
	}
	while (count > 0)
	{
		if (line[count]>='0' && line[count]<='9' || (unsigned char)line[count]>='A')
			break;

		line[count]='\0';
		count--;
	}
	
	if (!StrCompare(line, "")) return NULL;		// for TB_FindWord() consistency
	if (line[0]==' ') return NULL;

	return line;
}

static Boolean ScrollbackFormHandleEvent(EventPtr eventP)
{
	static char found_word[12] = "";		// good enough precision
	FormPtr frmP;
	FieldPtr fieldP;
	static int selection_pos;
	
	switch (eventP->eType) 
	{
		case frmOpenEvent:
		{
			RectangleType rect;
			frmP = FrmGetActiveForm();
			fieldP = GetObjectPtr(ScrollbackTextField);
			FldSetTextPtr(fieldP, scrollback_buffer);
			FldRecalculateField(fieldP, true);
			FldSetScrollPosition(fieldP, StrLen(scrollback_buffer));
			FrmDrawForm(frmP);
			FldScrollField(fieldP, 10, winUp);
			FldGetBounds(fieldP, &rect);
			WinDrawLine(0, rect.extent.y, palm_screen_width, rect.extent.y);
			WinDrawLine(0, rect.extent.y+1, palm_screen_width, rect.extent.y+1);
			
			return true;
		}
		case ctlSelectEvent:
		{
			switch (eventP->data.ctlEnter.controlID)
			{
				case ScrollbackOKButton:
					ExitDialog(true);
					return true;
			}
			break;
		}
		case penDownEvent:
		{
			StrCopy(found_word, "");
			break;
		}
		case keyDownEvent:
		{
			fieldP = GetObjectPtr(ScrollbackTextField);
			if (eventP->data.keyDown.chr==11)
			{
				FldScrollField(fieldP, 10, winUp);
			}
			else if (eventP->data.keyDown.chr==12)
			{
				if (!FldGetNumberOfBlankLines(fieldP))
					FldScrollField(fieldP, 10, winDown);
			}
			return true;
		}
		case fldEnterEvent:
		{
			Boolean left;
			int x = eventP->screenX, y = eventP->screenY;
			fieldP = GetObjectPtr(ScrollbackTextField);
			
			if (!FldGetTextPtr(fieldP))
			{
				StrCopy(found_word, "");
				return true;
			}
			
			if (GetScrollbackWord(x, y))
			{
				// Don't allow selection of window breaks
				if (StrStr(GetScrollbackWord(x, y), "_____"))
					return true;
					
				// Highlight the selected word
				selection_pos = GetPosOfPoint(fieldP, x, y, &left);
				StrCopy(line, GetScrollbackWord(x, y));
				while (StrNCompare(line, FldGetTextPtr(fieldP)+selection_pos, StrLen(line)))
				{
					if (--selection_pos<0) break;
				}
				if (selection_pos)
					FldSetSelection(fieldP, selection_pos, selection_pos+StrLen(line));

				// And remember it
				StrNCopy(found_word, GetScrollbackWord(eventP->screenX, eventP->screenY), 12);
			}
			else selection_pos = -1;

			return true;
		}
		case penUpEvent:
		{
			// Unhighlight the selected word
			fieldP = GetObjectPtr(ScrollbackTextField);
			FldSetSelection(fieldP, 0, 0);
				
			if (!StrCompare(found_word, "") || !getline_active)
				return true;
				
			if ((GetScrollbackWord(eventP->screenX, eventP->screenY)) &&
				!StrNCompare(found_word, GetScrollbackWord(eventP->screenX, eventP->screenY), 11))
			{
				Boolean ret;
				
				ret = HandlePopupTap(GetScrollbackWord(eventP->screenX, eventP->screenY),
					eventP->screenX, eventP->screenY);
				StrCopy(found_word, "");
				if (ret) ExitDialog(true);
				return true;
			}
			break;
		}
		case ctlRepeatEvent:
		{
			fieldP = GetObjectPtr(ScrollbackTextField);
			if (eventP->data.ctlRepeat.controlID==ScrollbackUpGraphicRepeatingButton)
			{
				FldScrollField(fieldP, 1, winUp);
			}
			else if (eventP->data.ctlRepeat.controlID==ScrollbackDownGraphicRepeatingButton)
			{
				if (!FldGetNumberOfBlankLines(fieldP))
					FldScrollField(fieldP, 1, winDown);
			}
			break;
		}
	}
	
	return false;
}


/***********************************************************************
 *
 * FUNCTION:    OptionsFormHandleEvent
 *
 ***********************************************************************/
static Boolean OptionsFormHandleEvent(EventPtr eventP)
{
	FormPtr frmP;
	ControlPtr ctrlP;
	
	switch (eventP->eType) 
	{
		case frmOpenEvent:
		{
			frmP = FrmGetActiveForm();
			ctrlP = GetObjectPtr(OptionsBlankLinesCheckbox);
			CtlSetValue(ctrlP, ph_prefs.no_blank_lines);
			ctrlP = GetObjectPtr(OptionsMinimalWindowsCheckbox);
			CtlSetValue(ctrlP, ph_prefs.minimal_windows);
			ctrlP = GetObjectPtr(OptionsUseColorsCheckbox);
			CtlSetValue(ctrlP, ph_prefs.use_colors);
			ctrlP = GetObjectPtr(OptionsSmartFormattingCheckbox);
			CtlSetValue(ctrlP, smartformatting);
			FrmDrawForm(frmP);
			return true;
		}
		case ctlSelectEvent:
		{
			switch (eventP->data.ctlEnter.controlID)
			{
				case OptionsOKButton:
					ctrlP = GetObjectPtr(OptionsBlankLinesCheckbox);
					ph_prefs.no_blank_lines = CtlGetValue(ctrlP);
					ctrlP = GetObjectPtr(OptionsMinimalWindowsCheckbox);
					ph_prefs.minimal_windows = CtlGetValue(ctrlP);
					ctrlP = GetObjectPtr(OptionsUseColorsCheckbox);
					ph_prefs.use_colors = CtlGetValue(ctrlP);
					ctrlP = GetObjectPtr(OptionsSmartFormattingCheckbox);
					smartformatting = CtlGetValue(ctrlP);
					
					ExitDialog(true);
					
					return true;
					
				case OptionsCancelButton:
					ExitDialog(false);
					return true;
			}
			break;
		}
	}
	return false;
}


/***********************************************************************
 *
 * FUNCTION:    PleaseWaitFormHandleEvent
 *
 ***********************************************************************/
void ShowPleaseWait(Boolean show)
{
	static Boolean shown = false;
	
	if (show)
	{
		shown = true;
		override_app_exit = true;
		FrmPopupForm(PleaseWaitForm);
	}
	else
	{
		if (shown)
			FrmReturnToForm(MainForm);
		override_app_exit = false;
		shown = false;
	}
	
	ProcessPendingPalmOSEvents();

	if (show)
	{
		FormPtr fp;
		RectangleType rect;
		
		fp = FrmGetFormPtr(PleaseWaitForm);
		if (fp)
		{
			WinSetDrawWindow(FrmGetWindowHandle(fp));
			RctSetRectangle(&rect, 10, 28, 80, 2);
			WinDrawRectangleFrame(rectangleFrame, &rect);
			WinSetDrawWindow(WinGetDisplayWindow());
		}
	}
}

void UpdatePleaseWait(int percent)
{
	FormPtr fp;
	RectangleType rect;
	
	fp = FrmGetFormPtr(PleaseWaitForm);
	if (fp)
	{
		WinSetDrawWindow(FrmGetWindowHandle(fp));
		RctSetRectangle(&rect, 10, 28, 80*percent/100, 2);
		WinDrawRectangle(&rect, 0);
		WinSetDrawWindow(WinGetDisplayWindow());
	}
}

static Boolean PleaseWaitFormHandleEvent(EventPtr eventP)
{
	FormPtr frmP;
	
	switch (eventP->eType) 
	{
		case frmOpenEvent:
		{
			frmP = FrmGetActiveForm();
			FrmDrawForm(frmP);
			return true;
		}
	}
	return false;
}


#pragma mark ----- Higher-level app stuff -----

/***********************************************************************
 *
 * FUNCTION:    AppHandleEvent
 *
 * DESCRIPTION: This routine loads form resources and set the event
 *              handler for the form loaded.
 *
 * PARAMETERS:  event  - a pointer to an EventType structure
 *
 * RETURNED:    true if the event has handle and should not be passed
 *              to a higher level handler.
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Boolean AppHandleEvent(EventPtr eventP)
{
	UInt16 formId;
	FormPtr frmP;
	
	if (eventP->eType == frmLoadEvent)
		{
		// Load the form resource.
		formId = eventP->data.frmLoad.formID;
		frmP = FrmInitForm(formId);
		FrmSetActiveForm(frmP);

		// Set the event handler for the form.  The handler of the currently
		// active form is called by FrmHandleEvent each time is receives an
		// event.
		switch (formId)
		{
			case MainForm:
			{
				if (romVersion < 0x03503000)
				{
					palm_screen_width = 160, palm_screen_height = 160;
				}
				else
				{
					RectangleType rect;
					
					FrmGetFormBounds(frmP, &rect);
					palm_screen_width = rect.extent.x;
					palm_screen_height = rect.extent.y;
					
					menu_line_height = FntLineHeight();
					
					if (romVersion >= 0x04003000)
					{
						BitmapPtr bmpP;
						bmpP = WinGetBitmap(WinGetDisplayWindow());
						palm_display_depth = BmpGetBitDepth(bmpP);
					}
					else
						palm_display_depth = WinGetBitmap(WinGetDisplayWindow())->pixelSize;
					
					UIColorGetTableEntryRGB(UIObjectForeground, &field_text);
					UIColorGetTableEntryRGB(UIFormFill, &field_back);
					UIColorGetTableEntryRGB(UIFieldCaret, &field_caret);
					UIColorGetTableEntryRGB(UIFieldText, &default_field_text);
					UIColorGetTableEntryRGB(UIFieldBackground, &default_field_back);
					UIColorGetTableEntryRGB(UIFieldCaret, &default_field_caret);
				}
				palm_screen_height-=PH_BOTTOM_ROW;

				FrmSetEventHandler(frmP, MainFormHandleEvent);
				break;
			}
				
			case FileOpenForm:
			case FileSaveForm:
				FrmSetEventHandler(frmP, FileFormHandleEvent);
				break;
				
			case CommandsForm:
			case ButtonsForm:
			case WordsForm:
				FrmSetEventHandler(frmP, EditFormHandleEvent);
				break;
				
			case ScrollbackForm:
				FrmSetEventHandler(frmP, ScrollbackFormHandleEvent);
				break;
				
			case OptionsForm:
				FrmSetEventHandler(frmP, OptionsFormHandleEvent);
				break;
				
			case PleaseWaitForm:
				FrmSetEventHandler(frmP, PleaseWaitFormHandleEvent);
				break;
				
			default:
				ErrFatalDisplay("Invalid Form Load Event");
				break;

		}
		return true;
	}
	
	return false;
}


/***********************************************************************
 *
 * FUNCTION:    AppEventLoop
 *
 * DESCRIPTION: This routine is the event loop for the application.  
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void AppEventLoop(void)
{
	UInt16 error;
	EventType event;

	do {
		EvtGetEvent(&event, evtWaitForever);

		if (! SysHandleEvent(&event))
			if (! MenuHandleEvent(0, &event, &error))
				if (! AppHandleEvent(&event))
					FrmDispatchEvent(&event);
	} while (event.eType != appStopEvent);
}


/***********************************************************************
 *
 * FUNCTION:     AppStart
 *
 * DESCRIPTION:  Get the current application's preferences.
 *
 * PARAMETERS:   nothing
 *
 * RETURNED:     Err value 0 if nothing went wrong
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static Err AppStart(void)
{
	UInt16 prefsSize;
	
	ResetCommandList();
	ResetButtonList();
	ResetWordList();

	ph_prefs.no_blank_lines = false;
	ph_prefs.minimal_windows = false;
	ph_prefs.use_colors = true;
	ph_prefs.smartformatting = smartformatting;

	// Read the saved preferences / saved-state information.
	prefsSize = sizeof(PalmHugoPreferenceType);
	if (PrefGetAppPreferences(appFileCreator, appPrefID, &ph_prefs, &prefsSize, true) != 
		noPreferenceFound)
	{
	}
	
	if (prefsSize != sizeof(PalmHugoPreferenceType))
	{
		ResetCommandList();
		ResetButtonList();
		ResetWordList();
	
		ph_prefs.no_blank_lines = false;
		ph_prefs.minimal_windows = false;
		ph_prefs.use_colors = true;
		ph_prefs.smartformatting = smartformatting;
	}
	
	smartformatting = ph_prefs.smartformatting;
	
	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:    AppStop
 *
 * DESCRIPTION: Save the current state of the application.
 *
 * PARAMETERS:  nothing
 *
 * RETURNED:    nothing
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
static void AppStop(void)
{
	// Write the saved preferences / saved-state information.  This data 
	// will be backed up during a HotSync.
	ph_prefs.smartformatting = smartformatting;
	PrefSetAppPreferences (appFileCreator, appPrefID, appPrefVersionNum, 
		&ph_prefs, sizeof (ph_prefs), true);
		
	// Save the state if we didn't quit but are instead switching away
	if (game_started && !var[endflag] && !delete_snapshot)
	{
		SaveState();
	}

	FreeGameData();
	
	// If we did quit, delete any saved gamestate
	if (var[endflag] || delete_snapshot)
	{
		DeleteFile(CURRENT_GAME_SNAP);
		DeleteFile(CURRENT_GAME_DATA);
	}

	// Free text buffer
	TB_Clear(0, 0, palm_screen_width, palm_screen_height);
	
	// Clear any remaining allocated memory
	if (mem) hugo_blockfree(mem);
	
	// Close all the open forms.
	FrmCloseAllForms();
}


/***********************************************************************
 *
 * FUNCTION:    PalmHugoMain
 *
 * DESCRIPTION: This is the main entry point for the application.
 *
 * PARAMETERS:  cmd - word value specifying the launch code. 
 *              cmdPB - pointer to a structure that is associated with the launch code. 
 *              launchFlags -  word value providing extra information about the launch.
 *
 * RETURNED:    Result of launch
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/

static UInt32 PalmHugoMain(UInt16 cmd, MemPtr /*cmdPBP*/, UInt16 launchFlags)
{
	Err error;
	error = RomVersionCompatible (ourMinVersion, launchFlags);
	if (error) return (error);
	
	switch (cmd)
	{
		case sysAppLaunchCmdNormalLaunch:
		{
			GetRomVersion();
		
			ErrTry
			{
				error = AppStart();
				if (error) 
					return error;
			}
			ErrCatch(err)
			{
			} ErrEndCatch

			ErrTry
			{
				LocalID db = 0;
				UInt16 card = 0;
				Boolean current_game_available = true;
				
				// Check for CURRENT_GAME_DATA
				for (card=0; card<MemNumCards(); card++)
				{
					db = DmFindDatabase(card, CURRENT_GAME_DATA);
					if (db) break;
				}
				if (card>=MemNumCards()) current_game_available = false;
				
				for (card=0; card<MemNumCards(); card++)
				{
					db = DmFindDatabase(card, CURRENT_GAME_SNAP);
					if (db) break;
				}
				
				// If there's a state to restore, restore it
				if (db && current_game_available)
				{
					if (LoadState())
					{
						char *argv[2] = {"", saved_state_game_name};
						
						FrmGotoForm(MainForm);
						ProcessPendingPalmOSEvents();
						
						SetupDirectionButtons();
						
						game_started = true;
						loaded_in_memory = false;

	 					ErrTry
	 					{
							he_main(2, argv);
						}
						ErrCatch(err)
						{
						} ErrEndCatch
					}
					else
						db = 0;
				}
				
				if (!db)
				{
NoSavedState:
					TB_Clear(0, 0, palm_screen_width, palm_screen_height);
					
					// Run the FileOpen form
					file_select_type = PH_GAME_TYPE;
					if (RunDialog(FileOpenForm))
					{
						if (file_select_error==PH_NO_FILES)
						{
							FormPtr frmP;
							frmP = FrmInitForm(NoGamesForm);
							FrmDoDialog(frmP);
							FrmDeleteForm(frmP);
						}
						else if (StrCompare(file_selected, ""))
						{
							char *argv[2] = {"", file_selected};
							StrCopy(saved_state_game_name, file_selected);
							
							SetupDirectionButtons();

							game_started = true;
		 					loaded_in_memory = false;
							
		 					//AppEventLoop();
		 					ErrTry
		 					{
								he_main(2, argv);
							}
							ErrCatch(err)
							{
							} ErrEndCatch
						}
					}
				}
			}
			ErrCatch(err)
			{
			} ErrEndCatch
			
			AppStop();
			break;
		}

		default:
			break;

	}
	
	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:    PilotMain
 *
 * DESCRIPTION: This is the main entry point for the application.
 *
 * PARAMETERS:  cmd - word value specifying the launch code. 
 *              cmdPB - pointer to a structure that is associated with the launch code. 
 *              launchFlags -  word value providing extra information about the launch.
 * RETURNED:    Result of launch
 *
 * REVISION HISTORY:
 *
 *
 ***********************************************************************/
UInt32 PilotMain( UInt16 cmd, MemPtr cmdPBP, UInt16 launchFlags)
{
    return PalmHugoMain(cmd, cmdPBP, launchFlags);
}


/***********************************************************************
 *
 * FUNCTION:   ProcessPalmOSEvents
 *
 * DESCRIPTION: The main poll while the Hugo Engine is running.
 *
 ***********************************************************************/
void ProcessPalmOSEvents(void)
{
	UInt16 error;
	EventType event;
	
	EvtGetEvent(&event, SysTicksPerSecond()/10);

	if (! SysHandleEvent(&event))
		if (! MenuHandleEvent(0, &event, &error))
			if (! AppHandleEvent(&event))
				FrmDispatchEvent(&event);
				
	if (event.eType==appStopEvent && !override_app_exit) exit(0);
}

void ProcessPendingPalmOSEvents(void)
{
	while (EvtEventAvail()) ProcessPalmOSEvents();
}


/***********************************************************************
 *
 * FUNCTION:   RunDialog
 *
 * DESCRIPTION: The runner for standard PalmHugo dialogs.
 *
 * A PalmHugo dialog must call ExitDialog() for any control that's
 * supposed to close it.
 *
 ***********************************************************************/
static Boolean dialog_is_active = false;

Boolean RunDialog(UInt16 formID)
{
	EventType event;
	UInt16 error;
	WinHandle wh;
	Boolean popup = 0;
	
	dialog_is_active = true;
	dialog_ok = false;

	if (romVersion >= 0x03503000)
	{
		wh = WinGetDrawWindow();
		WinPushDrawState();
	}

	if (game_started)
	{
		FrmPopupForm(formID);
		popup = true;
	}
	else
	{
		FrmGotoForm(formID);
	}
		
	do {
		EvtGetEvent(&event, evtWaitForever);

		if (! SysHandleEvent(&event))
			if (! MenuHandleEvent(0, &event, &error))
				if (! AppHandleEvent(&event))
					FrmDispatchEvent(&event);

	} while (event.eType != appStopEvent && dialog_is_active);
	
	if (romVersion >= 0x03503000)
	{
		WinSetDrawWindow(wh);
		WinPopDrawState();
	}

	if (popup)
	{
		FrmReturnToForm(MainForm);
	}
	else
	{
		FrmGotoForm(MainForm);
	}
	ProcessPendingPalmOSEvents();
	
	if (event.eType==appStopEvent)
		exit(0);
		
	return true;
}

void ExitDialog(Boolean ok)
{
	dialog_is_active = false;
	dialog_ok = ok;
}
