/*
	palm_stdio.c
	
	Because Palm OS doesn't use stdio, and we have to work around the stdio
	file-management calls in the Hugo Engine.
*/

#include "palm_stdio.h"
#include "PalmHugo.h"

PH_FILE *ph_fopen(char *filename, char *mode)
{
	UInt16 card, m = dmModeReadOnly;
	LocalID db = 0;
	PH_FILE *file;
	
#ifdef DEBUG_STDIO
	char buf[64];
	StrPrintF(buf, "fopen(%s, %s)", filename, mode);
	Printout(buf);
#endif
	// Figure out the mode
	switch (mode[0])
	{
		case 'r':  m = dmModeReadOnly; break;
		case 'w':  m = dmModeWrite; break;
		
		default:   return NULL;
	}

	// Try to find the filename as a database
	for (card=0; card<MemNumCards(); card++)
	{
		if ((db = DmFindDatabase(card, filename)))
			break;
	}
	if (db==0 && m==dmModeReadOnly)
		return NULL;
	else if (m==dmModeWrite)
	{
		if (db)
			DmDeleteDatabase(card, db);
		else
			card = 0;
		if ((DmCreateDatabase(card, filename,
			PH_CREATOR_TYPE, file_select_type, false))!=errNone)
		{
			return NULL;
		}
	}
	if (!(db = DmFindDatabase(card, filename)))
			return NULL;
			
	if (StrStr(mode, "+")) m = dmModeReadWrite;
	
	// Try to allocate the PH_FILE
	file = MemPtrNew(sizeof(PH_FILE));
	if (file==NULL)
		return NULL;

	// Actually try to open the db in the given mode
	if (!(file->ref = DmOpenDatabase(card, db, m)))
	{
		MemPtrFree(file);
		return NULL;
	}
	
	// Figure out the length of the entire 'file'
	if ((DmDatabaseSize(card, db, NULL, NULL, &file->length))!=errNone)
	{
		MemPtrFree(file);
		return NULL;
	}

	// Setup the PH_FILE structure
#ifdef DEBUG_STDIO
	StrNCopy(file->name, filename, 32);
#endif
	file->blocksize = PH_DB_DEFAULT_BLOCKSIZE;
	file->error = 0;
	file->pos = 0;
	file->open_record = EOF;
	file->record_data = NULL;
	file->mode = m;

	// Figure out the blocksize (if not the default)
	if (DmNumRecords(file->ref))
	{
		file->record_handle = DmGetRecord(file->ref, 0);
		if (file->record_handle!=NULL)
		{
			file->blocksize = MemHandleSize(file->record_handle);
			DmReleaseRecord(file->ref, 0, false);
		}
	}
	
	if (file_select_type==PH_SAVE_TYPE) ShowPleaseWait(true);

	return file;
}

int ph_fclose(PH_FILE *file)
{
#ifdef DEBUG_STDIO
	char buf[64];
	StrPrintF(buf, "fclose(%s)", file->name);
	Printout(buf);
#endif
	ShowPleaseWait(false);

	if (file->open_record != EOF)
	{
		MemHandleUnlock(file->record_handle);
		if (file->mode!=dmModeReadOnly)
			DmReleaseRecord(file->ref, file->open_record, true);
	}
	DmCloseDatabase(file->ref);
	MemPtrFree(file);
	
	return 0;
}

int ph_fseek(PH_FILE *file, long offset, int whence)
{
#ifdef DEBUG_STDIO
	if (ph_debug)
	{
		char buf[64];
		StrPrintF(buf, "fseek(%s, %d, %d)", file->name, offset, whence);
		Printout(buf);
	}
#endif
	switch (whence)
	{
		case SEEK_SET:
			file->pos = offset;
			break;
		case SEEK_CUR:
			file->pos += offset;
			break;
		case SEEK_END:
			file->pos = file->length - offset - 1;
	}
	
	if (file->pos < 0) file->pos = 0;
	if (file->pos > file->length-1) file->pos = file->length-1;

	file->error = 0;
	
	return 0;
}

static UInt16 FixupDBRecord(PH_FILE *file, UInt16 this_record)
{
	if (this_record != file->open_record)
	{
		if (file->open_record!=EOF)
		{
			MemHandleUnlock(file->record_handle);
			if (file->mode!=dmModeReadOnly)
				DmReleaseRecord(file->ref, file->open_record, true);
		}
		if (file->mode==dmModeReadOnly || this_record<DmNumRecords(file->ref))
		{
			if (file->mode==dmModeReadOnly)
				file->record_handle = DmQueryRecord(file->ref, this_record);
			else
				file->record_handle = DmGetRecord(file->ref, this_record);
		}
		else
		{
			UInt16 new_record = dmMaxRecordIndex;
			file->record_handle = DmNewRecord(file->ref, &new_record, file->blocksize);
		}
			
		if (file->record_handle==NULL)
		{
			file->error = true;
			return EOF;
		}
		file->open_record = this_record;
		file->record_data = MemHandleLock(file->record_handle);
		if (file->record_data==NULL)
		{
			file->error = true;
			if (file->mode!=dmModeReadOnly)
				DmReleaseRecord(file->ref, file->open_record, true);
			file->open_record = EOF;
			return EOF;
		}
	}
	return this_record;
}

int ph_fgetc(PH_FILE *file)
{
	UInt16 this_record = file->pos/file->blocksize;
	
#ifdef DEBUG_STDIO
	if (ph_debug)
	{
		char buf[64];
		StrPrintF(buf, "fgetc(%s)", file->name);
		Printout(buf);
	}
#endif
	if (file->pos >= file->length)
	{
		return EOF;
	}

	if (FixupDBRecord(file, this_record)==EOF)
		return EOF;

	file->error = 0;
	
	return (*((unsigned char *)file->record_data + (file->pos++)%file->blocksize));
}

int ph_fputc(int c, PH_FILE *file)
{
	unsigned char data;
	UInt16 this_record = file->pos/file->blocksize;

#ifdef DEBUG_STDIO
	if (ph_debug)
	{
		char buf[64];
		StrPrintF(buf, "fputc(%s)", file->name);
		Printout(buf);
	}
#endif
	if (FixupDBRecord(file, this_record)==EOF)
		return EOF;
		
	data = (char)c;
	if ((DmWrite(file->record_data, (file->pos++)%file->blocksize, &data, 1))!=errNone)
		return EOF;
	if (file->pos > file->length)
		file->length = file->pos;
		
	file->error = 0;
	
	return data;
}

size_t ph_fread(void *ptr, size_t size, size_t nmemb, PH_FILE *file)
{
	size_t total, read;
	char *p = (char *)ptr;
	
#ifdef DEBUG_STDIO
	char buf[64];
	StrPrintF(buf, "fread(%s)", file->name);
	Printout(buf);
#endif
	file->error = 0;
	
	total = size*nmemb;

	for (read=0; read<total; read++)
	{
		int c = ph_fgetc(file);
		if (c==EOF)
			break;
		p[read] = (char)c;
	}
		
	return read;
}

char *ph_fgets (char *s, int n, PH_FILE *file)
{
	int count = 0;
	
#ifdef DEBUG_STDIO
	char buf[64];
	StrPrintF(buf, "fgets(buf, %d, %s)", n, file->name);
	Printout(buf);
#endif
	file->error = 0;
	
	while (count < n-1)
	{
		int c = fgetc(file);
		if (c==EOF)
		{
			if (!count)
				return NULL;
			else
			{
				break;
			}
		}
		s[count++] = c;
		if (c=='\n') break;
	}
	s[count] = '\0';
	
	return s;
}

int ph_fputs(const char *s, PH_FILE *file)
{
	int i;
	
#ifdef DEBUG_STDIO
	char buf[256];
	StrPrintF(buf, "fputs(s, %s)", file->name);
	Printout(buf);
#endif
	file->error = 0;
	
	for (i=0; i<StrLen(s); i++)
	{
		if (fputc(s[i], file)==EOF)
			return EOF;
	}
	return 0;
}
