/*
	HESOUND.C

	DirectSound/DirectShow playback for sound (RIFF WAVE samples),
	music (MIDI and MP3), and music modules (MOD/S3M/XM, using
	a DirectSound MikMod driver).

		hugo_playmusic
		hugo_stopmusic
		hugo_playsample
		hugo_stopsample

	for the Hugo Engine

	Copyright (c) 1995-2004 by Kent Tessman

	Both hugo_playmusic() and hugo_playsample() are called with the
	file hot--i.e., opened and positioned to the start of the resource.
	It must be closed before returning.

	A little tricky because hugo_playmusic() must choose between
	calling PlayMusic_MikMod() and PlayMusic_DirectShow();
	hugo_playsample() operates exclusively via DirectSound.

	Not just for this reason, hugo_stop*() must be called before
	hugo_play*() can play a new instance of sound/music.

	NOTE:  The version of the MikMod library used here is 2.10
	(or 2.09d).
*/

#include <windows.h>
#undef GetProp		// from windows.h
#include <mmsystem.h>
#include <math.h>
#ifndef USE_NT_SOUND_ONLY
#include <dsound.h>
#include <control.h>
#include <strmif.h>
#include <uuids.h>
#include <evcode.h>
#endif
#undef FindResource

// Async file reader stuff:
#ifndef USE_NT_SOUND_ONLY
#include <stdio.h>
#include <streams.h>
#include <asyncio.h>
#include <asyncrdr.h>
#include "memfile.h"
#endif

extern "C"
{

#include "heheader.h"
#include "mikmod.h"
#include "hewin32.h"

// Don't need DirectSound and friends if we're only v2.5
#if defined (COMPILE_V25)
#define USE_NT_SOUND_ONLY
#endif

int hugo_playmusic(FILE *infile, long reslength, char loop_flag);
void hugo_musicvolume(int vol);
void hugo_stopmusic(void);
int hugo_playsample(FILE *infile, long reslength, char loop_flag);
void hugo_samplevolume(int vol);
void hugo_stopsample(void);

int InitPlayer(void);
static int SoundDriverError(int retry);
void SuspendAudio(void);
void ResumeAudio(void);
int PlayMusic_MikMod(FILE *infile, char loop_flag);
void StopMusic_MikMod(void);
int PlayMusic_DirectShow(FILE *infile, long length, char loop_flag);
void StopMusic_DirectShow(void);
void SuspendAudio_DirectShow(void);
void ResumeAudio_DirectShow(void);
int PlaySample_DirectSound(FILE *infile, long length, char loop_flag);
void StopSample_DirectSound(void);

/* From hesound_nt.c */
int NT_hugo_playmusic(HUGO_FILE infile, long reslength, char loop_flag);
void NT_hugo_musicvolume(int vol);
void NT_hugo_stopmusic(void);
int NT_hugo_playsample(HUGO_FILE infile, long reslength, char loop_flag);
void NT_hugo_stopsample(void);
void NT_hugo_samplevolume(int vol);
int NT_InitPlayer(void);
void NT_SuspendAudio(void);
void NT_ResumeAudio(void);

char NT_sound_system = false;
char music_is_playing = false;
int music_volume = 100;
char sample_is_playing = false;
int sample_volume = 100;
char sound_driver_failed = 0, tried_to_play_sound = false;

/* DirectSound stuff: */
static LPDIRECTSOUND lpDirectSound;
static LPDIRECTSOUNDBUFFER lpDirectSoundBuffer;

/* For audio suspension: */
struct
{
	char filename[MAXPATH];
	char resname[MAXPATH];
	char loop_flag;
} resume_sample, resume_music;
char audio_suspended = false;
int suspended_mp_volume;

#define MUSIC_MIKMOD     1
#define MUSIC_DIRECTSHOW 2


/* SoundDriverError

	Returns true to keep waiting for an audio device.
*/

static int SoundDriverError(int retry)
{
	char *title = "Hugo Engine - Sound Trouble";

	/* If the game hasn't tried to play a sound yet, don't even bother
	   with an error, just disable sound because it (probably) doesn't
	   need it
	*/
	if (!tried_to_play_sound) return false;

	switch (retry)
	{
		case 0:
			MessageBox(wndMain,
				"Unable to initialize the audio device.  \
Another application may be using the device, or it may not be configured \
properly.",
				title,
				MB_ICONEXCLAMATION);
			return 0;

		case 1:
			if (MessageBox(wndMain,
				"Unable to restart the audio device.  \
Another application may be using the device.  Keep waiting?",
				title,
				MB_APPLMODAL |
				MB_ICONEXCLAMATION |
				MB_YESNO)==IDYES)
			{
				return true;
			}
			else
				return false;
	}
	return false;	// default to prevent warning
}


/* hugo_playmusic

	Return false if it fails because of an ERROR.  Calls
	either PlayMusic_MikMod or PlayMusic_DirectShow, depending
	on what sound system is being used.
*/

int hugo_playmusic(FILE *f, long reslength, char loop_flag)
{
#ifdef USE_NT_SOUND_ONLY
	return NT_hugo_playmusic(f, reslength, loop_flag);
#else
	if (NT_sound_system) return NT_hugo_playmusic(f, reslength, loop_flag);

	tried_to_play_sound = true;

	if (disable_sounds)
	{
		/* Wait until we've actually tried to play a sound before
		   saying we couldn't initialize the device
		*/
		if (sound_driver_failed) SoundDriverError(0);
		sound_driver_failed = 0;
		fclose(f);
		return true;	/* not an error */
	}

	if (music_is_playing)
	{
		hugo_stopmusic();
	}

	switch (resource_type)
	{
		case MOD_R:
		case S3M_R:
		case XM_R:
			return PlayMusic_MikMod(f, loop_flag);
#if !defined (COMPILE_V25)
		case MIDI_R:
		case MP3_R:
			return PlayMusic_DirectShow(f, reslength, loop_flag);
#endif
		default:
#if defined (DEBUGGER)
			SwitchtoDebugger();
			DebugMessageBox("Music Loading Error", "Unrecognized music format.");
			SwitchtoGame();
#endif
			fclose(f);
			return false;	/* error */
	}

#endif	// #ifndef USE_NT_SOUND_ONLY
}


/* hugo_musicvolume */

void hugo_musicvolume(int vol)
{
#ifdef USE_NT_SOUND_ONLY
	if (NT_sound_system)
	{
		NT_hugo_musicvolume(vol);
		return;
	}
#else
	music_volume = vol;
#endif
}


/* hugo_stopmusic */

void hugo_stopmusic(void)
{
#ifdef USE_NT_SOUND_ONLY
	if (NT_sound_system)
	{
		NT_hugo_stopmusic();
		return;
	}
#else
	if (music_is_playing==MUSIC_MIKMOD)
		StopMusic_MikMod();
	else if (music_is_playing==MUSIC_DIRECTSHOW)
		StopMusic_DirectShow();
#endif
}


/* hugo_playsample

	Return false if it fails because of an ERROR.
*/

int hugo_playsample(FILE *f, long reslength, char loop_flag)
{
#ifdef USE_NT_SOUND_ONLY
	return NT_hugo_playsample(f, reslength, loop_flag);
#else
	if (NT_sound_system) return NT_hugo_playsample(f, reslength, loop_flag);

	tried_to_play_sound = true;

	if (sample_is_playing) hugo_stopsample();

	sample_is_playing = PlaySample_DirectSound(f, reslength, loop_flag);

#if defined (DEBUGGER)
	if (!sample_is_playing)
		DebugMessageBox("Sound Library Error",
			"Error playing sample via DirectSound.");
#endif
	return sample_is_playing;

#endif	// #ifndef USE_NT_SOUND_ONLY
}


/* hugo_samplevolume */

void hugo_samplevolume(int vol)
{
#ifdef USE_NT_SOUND_ONLY
	if (NT_sound_system)
	{
		NT_hugo_samplevolume(vol);
		return;
	}
#else
	sample_volume = vol;
#endif
}


/* hugo_stopsample */

void hugo_stopsample(void)
{
#ifdef USE_NT_SOUND_ONLY
	if (NT_sound_system)
	{
		NT_hugo_stopsample();
		return;
	}
#else
	if (sample_is_playing)
	{
		StopSample_DirectSound();
		sample_is_playing = false;
	}
#endif	// #ifndef USE_NT_SOUND_ONLY
}


/*
 * Audio suspend/resume:
 *
 */

void SuspendAudio(void)
{
#ifdef USE_NT_SOUND_ONLY
	if (NT_sound_system)
	{
		NT_SuspendAudio();
		return;
	}
#else
	if (audio_suspended) return;

	/* suspend DirectSound sample */
	if (sample_is_playing and lpDirectSoundBuffer)
	{
		IDirectSoundBuffer_SetVolume(lpDirectSoundBuffer, -10000);
	}

	/* suspend MikMod */
	suspended_mp_volume = mp_volume;
	mp_volume = 0;

	if (music_is_playing==MUSIC_DIRECTSHOW)
		SuspendAudio_DirectShow();

	audio_suspended = true;

#endif	// #ifndef USE_NT_SOUND_ONLY
}


void ResumeAudio(void)
{
#ifdef USE_NT_SOUND_ONLY
	if (NT_sound_system)
	{
		NT_ResumeAudio();
		return;
	}
#else
	if (!audio_suspended) return;

	audio_suspended = false;

	/* resume DirectSound sample */
	if (sample_is_playing and lpDirectSoundBuffer)
	{
		IDirectSoundBuffer_SetVolume(lpDirectSoundBuffer,
//			sample_volume?(long)((log10(sample_volume*0.85)-2)*5000):-10000);
			sample_volume?(long)((log10(sample_volume)-2)*5000):-10000);
	}

	/* resume MikMod */
	mp_volume = suspended_mp_volume;

	if (music_is_playing==MUSIC_DIRECTSHOW)
		ResumeAudio_DirectShow();

#endif	// #ifndef USE_NT_SOUND_ONLY
}


/*
 * MikMod management:
 *
 */

UNIMOD *modfile = NULL;

// From MikMod mplayer.c:
extern int forbid;

// From MikMod drv_ds.c:
extern CRITICAL_SECTION win32_mikmod_cs;
extern DRIVER drv_ds;


/* tickhandler() is used to manage MikMod's MOD-playing loop */

static void tickhandler(void)
{
	if (disable_sounds) return;

	/* If we don't do this check, MikMod will crash us if there's a
	   sample but no module playing
	*/
	if (!music_is_playing) return;

	MP_HandleTick();    	/* play 1 tick of the module */
	MD_SetBPM(mp_bpm);
}


/* InitPlayer */

int InitPlayer(void)
{
#ifdef USE_NT_SOUND_ONLY
	NT_sound_system = true;
	return NT_InitPlayer();
#else
	DWORD dwVersion = GetVersion();
	DWORD dwWindowsMajorVersion = (DWORD)(LOBYTE(LOWORD(dwVersion)));
	//DWORD dwWindowsMinorVersion = (DWORD)(HIBYTE(LOWORD(dwVersion)));
	if (dwVersion < 0x80000000)	// Windows NT/2000/XP
	{
		if (dwWindowsMajorVersion < 5)
		{
			NT_sound_system = true;
			return NT_InitPlayer();
		}
	}

	ML_RegisterLoader(&load_mod);	/* module loaders */
	ML_RegisterLoader(&load_s3m);
	ML_RegisterLoader(&load_xm);

//	MD_RegisterDriver(&drv_nos);
	MD_RegisterDriver(&drv_ds);	/* DirectSound driver */

	MD_RegisterPlayer(tickhandler);

	md_mixfreq = 44100;			/* standard mixing frequency */
	md_mode = DMODE_16BITS|DMODE_STEREO;	/* standard mixing mode */
	md_dmabufsize = 2000;			/* standard DMA buffer size */
	md_device = 0;				/* standard device: autodetect */

	if (!MD_Init())			/* initialize driver */
	{
		/* We'll later check sound_driver_failed when we actually
		   try to play a sample or song--don't bother printing any
		   error message until it we've tried to play a sound.
		*/
		sound_driver_failed = true;
		return false;
	}

	return true;

#endif	// #ifndef USE_NT_SOUND_ONLY
}


#ifndef USE_NT_SOUND_ONLY

/* PlayMusic_MikMod

	Called by hugo_playmusic for music modules (MOD/S3M/XM).
*/

int PlayMusic_MikMod(FILE *f, char loop_flag)
{
	/* Set the rewind/SEEK_SET reference used by MikMod to the
	   start of the resource
	*/
	_mm_setiobase(ftell(f));

	modfile = ML_LoadFP(f);
	fclose(f);

	if (modfile)
	{
		/* Initialize the player and start the DirectSound
		   driver
		*/
		MP_Init(modfile);
		md_numchn = modfile->numchn;
		MD_PlayStart();

		if (loop_flag)
		{
			forbid = 1;		/* suspend interrupts */
			modfile->reppos = 1;
			mp_loop = 1;
			forbid = 0;		/* restore interrupts */
		}
		else mp_loop = 0;

		mp_volume = music_volume;	/* 0 - 100 */
		if (!reg_PlaySounds) mp_volume = 0;

		music_is_playing = MUSIC_MIKMOD;

		return true;
	}
#if defined (DEBUGGER)
	else
		DebugMessageBox("Sound Library Error", myerr);
#endif
	return false;
}


void StopMusic_MikMod(void)
{
	EnterCriticalSection(&win32_mikmod_cs);
	music_is_playing = false;
	modfile->numpos = 0;

	Sleep(20);		/* wait for the player to stop */

	MD_PlayStop();
	ML_Free(modfile);
	modfile = NULL;

	LeaveCriticalSection(&win32_mikmod_cs);
}


/*
 * WAVE file reading:
 *
 */

#define ER_MEM			0xe000
#define ER_CANNOTOPEN		0xe100
#define ER_NOTWAVEFILE		0xe101
#define ER_CANNOTREAD		0xe102
#define ER_CORRUPTWAVEFILE	0xe103

static WAVEFORMATEX *pwf;
static HMMIO hm;
static MMCKINFO mmci, mmciRIFF;

/* WaveOpenFile

	Returns 0 if successful, the error code if not.

	pszFileName - Input filename to load
	phmmioIn    - Pointer to handle which will be used
		for further mmio routines
	ppwfxInfo   - Ptr to ptr to WaveFormatEx structure
		with all info about the file
*/

int WaveOpenFile(char *pszFileName, long pos, HMMIO *phmmioIn,
	WAVEFORMATEX **ppwfxInfo, MMCKINFO *pckInRIFF)
{
	HMMIO hmmioIn;
	MMCKINFO ckIn;			// chunk info for general use
	PCMWAVEFORMAT pcmWaveFormat;	// temp PCM structure to load in
	WORD cbExtraAlloc;		// extra bytes for waveformatex
	int nError;			// return value.

	char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILENAME], ext[MAXEXT];
	char envvar[32];
	char temppath[MAXPATH];

	// Initialization
	*ppwfxInfo = NULL;
	nError = 0;
	hmmioIn = NULL;
	
	/* Try to open the given, vanilla filename */
	if ((hmmioIn = mmioOpen(pszFileName, NULL, MMIO_ALLOCBUF | MMIO_READ)) == NULL)
	{
		hugo_splitpath(pszFileName, drive, dir, fname, ext);      /* file to open */

		/* If the given filename doesn't already specify where to find it */
		if (!strcmp(drive, "") and !strcmp(dir, ""))
		{
			/* Check gamefile directory */
			hugo_makepath(temppath, "", gamepath, fname, ext);

			if ((hmmioIn = mmioOpen(temppath, NULL, MMIO_ALLOCBUF | MMIO_READ)) == NULL)
			{
				/* Check environment variables */

				strcpy(envvar, "hugo_games");
				if (getenv(strupr(envvar)))
				{
					hugo_makepath(temppath, "", getenv(strupr(envvar)), fname, ext);

					if ((hmmioIn = mmioOpen(temppath, NULL, MMIO_ALLOCBUF | MMIO_READ)) != NULL)
					{
						goto OpenedFile;
					}
				}

				strcpy(envvar, "hugo_object");
				if (getenv(strupr(envvar)))
				{
					hugo_makepath(temppath, "", getenv(strupr(envvar)), fname, ext);

					if ((hmmioIn = mmioOpen(temppath, NULL, MMIO_ALLOCBUF | MMIO_READ)) == NULL)
					{
						return -1;
					}
				}
			}
		}
	}

/*
	if ((hmmioIn = mmioOpen(pszFileName, NULL, MMIO_ALLOCBUF | MMIO_READ)) == NULL)
	{
		nError = ER_CANNOTOPEN;
		goto ErrorReadingWave;
	}
*/
OpenedFile:
	mmioSeek(hmmioIn, pos, SEEK_SET);

	if ((nError = (int)mmioDescend(hmmioIn, pckInRIFF, NULL, 0)) != 0)
	{
		goto ErrorReadingWave;
	}

	if ((pckInRIFF->ckid != FOURCC_RIFF) || (pckInRIFF->fccType != mmioFOURCC('W', 'A', 'V', 'E')))
	{
		nError = ER_NOTWAVEFILE;
		goto ErrorReadingWave;
	}
		
	// Search the input file for for the 'fmt ' chunk
	ckIn.ckid = mmioFOURCC('f', 'm', 't', ' ');
	if ((nError = (int)mmioDescend(hmmioIn, &ckIn, pckInRIFF, MMIO_FINDCHUNK)) != 0)
	{
		goto ErrorReadingWave;                
	}
					
	/* Expect the 'fmt' chunk to be at least as large as PCMWAVEFORMAT;
	   if there are extra parameters at the end, we'll ignore them
	*/
	if (ckIn.cksize < (long) sizeof(PCMWAVEFORMAT))
	{
		nError = ER_NOTWAVEFILE;
		goto ErrorReadingWave;
	}
															
	// Read the 'fmt ' chunk into pcmWaveFormat
	if (mmioRead(hmmioIn, (HPSTR) &pcmWaveFormat, (long) sizeof(pcmWaveFormat)) != (long) sizeof(pcmWaveFormat))
	{
		nError = ER_CANNOTREAD;
		goto ErrorReadingWave;
	}

	// Allocate the waveformatex, but if it's not PCM
	// format, read the next word, and that's how many extra
	// bytes to allocate
	if (pcmWaveFormat.wf.wFormatTag == WAVE_FORMAT_PCM)
		cbExtraAlloc = 0;                               
	else
	{
		// Read in length of extra bytes
		if (mmioRead(hmmioIn, (LPSTR) &cbExtraAlloc,
			(long) sizeof(cbExtraAlloc)) != (long) sizeof(cbExtraAlloc))
		{
			nError = ER_CANNOTREAD;
			goto ErrorReadingWave;
		}
	}
							
	// Now allocate the waveformatex structure.
	if ((*ppwfxInfo = (WAVEFORMATEX *)GlobalAlloc(GMEM_FIXED, sizeof(WAVEFORMATEX)+cbExtraAlloc)) == NULL)
	{
		nError = ER_MEM;
		goto ErrorReadingWave;
	}

	// Copy the bytes from the pcm structure to the waveformatex structure
	memcpy(*ppwfxInfo, &pcmWaveFormat, sizeof(pcmWaveFormat));
	(*ppwfxInfo)->cbSize = cbExtraAlloc;

	// Now, read those extra bytes into the structure, if cbExtraAlloc != 0.
	if (cbExtraAlloc != 0)
	{
		if (mmioRead(hmmioIn, (LPSTR) (((BYTE*)&((*ppwfxInfo)->cbSize))+sizeof(cbExtraAlloc)),
			(long) (cbExtraAlloc)) != (long) (cbExtraAlloc))
		{
			nError = ER_NOTWAVEFILE;
			goto ErrorReadingWave;
		}
	}

	// Ascend the input file out of the 'fmt ' chunk
	if ((nError = mmioAscend(hmmioIn, &ckIn, 0)) != 0)
	{
		goto ErrorReadingWave;
	}
	
	goto TempCleanup;               

ErrorReadingWave:
	if (*ppwfxInfo != NULL)
	{
		GlobalFree(*ppwfxInfo);
		*ppwfxInfo = NULL;
	}               

	if (hmmioIn != NULL)
	{
		mmioClose(hmmioIn, 0);
		hmmioIn = NULL;
	}
	
TempCleanup:
	*phmmioIn = hmmioIn;

	return(nError);
}


/* WaveStartDataRead

	Has to be called before WaveReadFile as it searches for the 
	chunk to descend into for reading, i.e., the 'data' chunk.  
	For simplicity, this used to be in the open routine, but was
	taken out and moved to a separate routine so there was more 
	control on the chunks that are before the data chunk, such 
	as 'fact', etc.
*/

int WaveStartDataRead(HMMIO *phmmioIn, MMCKINFO *pckIn, MMCKINFO *pckInRIFF)
{
	if (mmioSeek(*phmmioIn, pckInRIFF->dwDataOffset + sizeof(FOURCC), SEEK_SET)==-1)
	{
		return FALSE;
	}

	// Search the input file for for the 'data' chunk
	pckIn->ckid = mmioFOURCC('d', 'a', 't', 'a');
	
	return mmioDescend(*phmmioIn, pckIn, NULL, MMIO_FINDCHUNK);
}


/* WaveReadFile

      This will read wave data from the wave file.  Make sure we're 
      descended into the data chunk, else this will fail.

	hmmioIn		- Handle to mmio
	cbRead		- # of bytes to read
	pbDest		- Destination buffer to put bytes
	cbActualRead	- # of bytes actually read
*/

int WaveReadFile(HMMIO hmmioIn, UINT cbRead, BYTE *pbDest,
	MMCKINFO *pckIn, UINT *cbActualRead)
{

	MMIOINFO mmioinfoIn;	// current status of hmmioIn
	int nError;
	UINT cT, cbDataIn, uCopyLength;

	nError = 0;

	if (nError = mmioGetInfo(hmmioIn, &mmioinfoIn, 0) != 0)
	{
		goto ErrorCannotRead;
	}
				
	cbDataIn = cbRead;
	if (cbDataIn > pckIn->cksize) cbDataIn = pckIn->cksize;       

	pckIn->cksize -= cbDataIn;
	
	for (cT = 0; cT < cbDataIn; )
	{
		/* Copy the bytes from the io to the buffer */
		if (mmioinfoIn.pchNext == mmioinfoIn.pchEndRead)
		{
			if ((nError = mmioAdvance(hmmioIn, &mmioinfoIn, MMIO_READ)) != 0)
			{
				goto ErrorCannotRead;
			} 
			if (mmioinfoIn.pchNext == mmioinfoIn.pchEndRead)
			{
				nError = ER_CORRUPTWAVEFILE;
				goto ErrorCannotRead;
			}
		}
			
		// Actual copy
		uCopyLength = (UINT)(mmioinfoIn.pchEndRead - mmioinfoIn.pchNext);
		if ((cbDataIn - cT) < uCopyLength) uCopyLength = cbDataIn - cT;
		memcpy((BYTE *)(pbDest+cT), (BYTE *)mmioinfoIn.pchNext, uCopyLength);
		cT += uCopyLength;
		mmioinfoIn.pchNext += uCopyLength;
	}

	if ((nError = mmioSetInfo(hmmioIn, &mmioinfoIn, 0)) != 0)
	{
		goto ErrorCannotRead;
	}

	*cbActualRead = cbDataIn;
	goto FinishedReading;

ErrorCannotRead:
	*cbActualRead = 0;

FinishedReading:
	return (nError);
}


/* WaveCloseReadFile

	This will close the wave file openned with WaveOpenFile.  
	Returns 0 if successful, non-zero if there was a warning.
	
	phmmioIn - Pointer to the handle to input MMIO
	ppwfxSrc - Pointer to pointer to WaveFormatEx structure
*/

int WaveCloseReadFile(HMMIO *phmmio, WAVEFORMATEX **ppwfxSrc)
{

	if (*ppwfxSrc != NULL)
	{
		GlobalFree(*ppwfxSrc);
		*ppwfxSrc = NULL;
	}

	if (*phmmio != NULL)
	{
		mmioClose(*phmmio, 0);
		*phmmio = NULL;
	}

	return(0);
}

/*
 * DirectSound/DirectShow management:
 *
 */

// DirectShow stuff:
static IGraphBuilder *pigb;
static IMediaControl *pimc;
static IMediaEvent *pime;
static IMediaPosition *pimp;
static IBasicAudio *piba;
static UINT gwID;		/* timer handle */
static char dshow_is_looping = 0;
static long initial_dshow_volume;

// Async reader stuff:
static CMemReader *rdr = NULL;
static PBYTE pbMem = NULL;
static CMemStream *Stream = NULL;

static void CALLBACK DirectShowManager(
	UINT IDEvent,		/* identifies timer event */
	UINT uReserved,		/* not used */
	DWORD dwUser,		/* application-defined instance data */
	DWORD dwReserved1,	/* not used */
	DWORD dwReserved2)	/* not used */
{
	long lEventCode;
	long lParam1;
	long lParam2;
	static char suspended = false;

	/* Silence this baby if the main window loses focus */
	if (GetForegroundWindow()!=wndMain)
	{
		pimc->Pause();
		suspended = true;
	}
	else if (suspended)
	{
		pimc->Run();
		suspended = false;
	}

	/* Check to see if the current music resource is finished */
	pime->GetEvent(&lEventCode, &lParam1, &lParam2, 20);
	if (lEventCode==EC_COMPLETE)
	{
		if (dshow_is_looping)
		{
			pimp->put_CurrentPosition(0);
			pimc->Run();
		}
		else
			StopMusic_DirectShow();
	}
	pime->FreeEventParams(lEventCode, lParam1, lParam2);
}


/* PlayMusic_DirectShow

	For playing MIDI and MP3 resources.  Called by hugo_playmusic 
	for MIDI/MP3.  Returns true on success, false on failure.
*/

int PlayMusic_DirectShow(FILE *infile, long length, char loop_flag)
{
	HRESULT hr;
	long vol;
	WCHAR wFile[MAX_PATH];

	// async reader stuff:
	long pos;
	CMediaType mt;

	piba = NULL;

	pos = ftell(infile);
	fclose(infile);

	HANDLE hFile;
	if ((hFile = TryToOpenHandle(loaded_filename, "games"))==INVALID_HANDLE_VALUE)
	{
		if ((hFile = TryToOpenHandle(loaded_filename, "object"))==INVALID_HANDLE_VALUE)
		{
			goto DirectShowError;
		}
	}
	SetFilePointer(hFile, pos, NULL, FILE_BEGIN);

	pbMem = new BYTE[length];
	if (pbMem == NULL)
	{
//		printf("Could not allocate memory\n");
		CloseHandle(hFile);
		goto DirectShowError;
	}

	DWORD dwBytesRead;
	if (!ReadFile(hFile, (LPVOID)pbMem, length,
			&dwBytesRead, NULL) 
		|| dwBytesRead!=(unsigned)length)
	{
//		printf("Could not read file\n");
		CloseHandle(hFile);
		goto DirectShowError;
	}
	CloseHandle(hFile);

	if (resource_type==MIDI_R)
	{
//		mt.majortype = MEDIATYPE_Midi;
		mt.subtype = MEDIASUBTYPE_None;
	}
	else	// MP3_R
	{
		mt.majortype = MEDIATYPE_Stream;
		mt.subtype = MEDIASUBTYPE_MPEG1Audio;
	}

	Stream = new CMemStream(pbMem, length);
	rdr = new CMemReader(Stream, &mt, &hr);
	if (FAILED(hr) || rdr == NULL)
	{
//		printf("Could not create filter HRESULT 0x%8.8X\n", hr);
		goto DirectShowError;
	}
	// MS's async filter sample says to do this so we don't
	// "accidentally go away"--not sure why, but:
	rdr->AddRef();

	/* DirectShow speaks Unicode */
	MultiByteToWideChar(CP_ACP, 0, loaded_resname, -1, wFile, MAX_PATH);

	/* Instantiate the filter graph manager, asking for the 
	   IGraphBuilder interface
	*/
	hr = CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC_SERVER, IID_IGraphBuilder, (void **)&pigb);
	if (FAILED(hr)) goto DirectShowError;

	// Add the async filter
	pigb->AddFilter(rdr, NULL);

	/* Get the IMediaControl interface */
	pigb->QueryInterface(IID_IMediaControl, (void **)&pimc);

	/* Get the IMediaEvent interface */
	pigb->QueryInterface(IID_IMediaEvent, (void **)&pime);

	/* Get the IMediaPosition interface */
	pigb->QueryInterface(IID_IMediaPosition, (void **)&pimp);

	/* Render the file */
//	hr = pigb->RenderFile(wFile, NULL);
	hr = pigb->Render(rdr->GetPin(0));
	if (FAILED(hr)) goto ReleaseAll;

	/* Get the IBasicAudio interface and set the volume */
	pigb->QueryInterface(IID_IBasicAudio, (void **)&piba);
	/* Because MIDI isn't played through the waveform device,
	   but via synthesizer */
	if (resource_type==MIDI_R)
		vol = music_volume*9/10;
	else
		vol = music_volume;
	piba->put_Volume(vol?(long)((log10(vol)-2)*5000):-10000);
	piba->get_Volume(&initial_dshow_volume);
	if (!reg_PlaySounds) SuspendAudio_DirectShow();

	/* Play the music resource */
	hr = pimc->Run();
	if (FAILED(hr)) goto ReleaseAll;

	/* Now set up the control timer */
	timeBeginPeriod(20);		// set the minimum resolution
	gwID = timeSetEvent(40,		// how often
		  40,			// timer resolution
		  DirectShowManager,
		  0,			// info to pass to callback
		  TIME_PERIODIC);	// oneshot or periodic

	music_is_playing = MUSIC_DIRECTSHOW;
	dshow_is_looping = loop_flag;
  
	return true;			// success

ReleaseAll:
	pigb->Release();
	pimc->Release();
	pime->Release();
	pimp->Release();
	if (piba) piba->Release();

DirectShowError:

	// Release async filter stuff:
	if (rdr) rdr->Release();
	if (Stream) delete Stream;
	if (pbMem) delete pbMem;

#if defined (DEBUGGER)
		DebugMessageBox("Music Playing Error",
			"Unable to play music via DirectShow.");
#endif
	return false;
}


void StopMusic_DirectShow(void)
{
	pimc->Stop();

	/* Stop the timer */
	timeKillEvent(gwID);
	timeEndPeriod(20);

	pigb->Release();
	pimc->Release();
	pime->Release();
	pimp->Release();
	piba->Release();

	// Release async filter stuff:
	if (rdr) rdr->Release();
	if (Stream) delete Stream;
	if (pbMem) delete pbMem;

	music_is_playing = false;
}


void SuspendAudio_DirectShow(void)
{
	piba->put_Volume(-10000);

}


void ResumeAudio_DirectShow(void)
{
	piba->put_Volume(initial_dshow_volume);
}


/* PlaySample_DirectSound

	Returns true on success, false on failure.
*/

int PlaySample_DirectSound(FILE *infile, long length, char loop_flag)
{
	int r;
	unsigned int bytesread;
	DSCAPS dsc;
	DSBUFFERDESC dsbdesc;
	LPVOID lpvPtr1;
	DWORD dwBytes1;
	LPVOID lpvPtr2;
	DWORD dwBytes2;
	HRESULT hr;

	long pos;
	pos = ftell(infile);
	fclose(infile);

	/* WaveOpenFile returns non-zero on failure */
	if ((r = WaveOpenFile(loaded_filename, pos, &hm, &pwf, &mmciRIFF))!=0)
	{
		return false;
	}

	if ((r = WaveStartDataRead(&hm, &mmci, &mmciRIFF))!=0)
        {
		WaveCloseReadFile(&hm, &pwf);
		return false;
	}

	/* Create DirectSound object */
	hr = CoCreateInstance(CLSID_DirectSound, NULL, CLSCTX_INPROC_SERVER,
		  IID_IDirectSound, (void**)&lpDirectSound);
	if (FAILED(hr))
	{
DirectSoundFailed:
		WaveCloseReadFile(&hm, &pwf);
		if (lpDirectSoundBuffer) IDirectSoundBuffer_Release(lpDirectSoundBuffer);
		lpDirectSoundBuffer = NULL;
		if (lpDirectSound) IDirectSound_Release(lpDirectSound);
		lpDirectSound = NULL;

		return false;
	}

	/* Init DirectSound */
	hr = IDirectSound_Initialize(lpDirectSound, NULL);
	if (FAILED(hr)) goto DirectSoundFailed;

  	/* Get DS driver caps */
	IDirectSound_GetCaps(lpDirectSound, &dsc);

	/* Set up DSBUFFERDESC structure */
	memset(&dsbdesc, 0, sizeof(DSBUFFERDESC));
	dsbdesc.dwSize = sizeof(DSBUFFERDESC);
	dsbdesc.dwFlags = DSBCAPS_CTRLVOLUME | DSBCAPS_GETCURRENTPOSITION2;
//	dsbdesc.dwBufferBytes = mmciRIFF.cksize;
	dsbdesc.dwBufferBytes = mmciRIFF.cksize + 16;
	dsbdesc.lpwfxFormat = pwf;

	/* Set cooperative level */
	hr = IDirectSound_SetCooperativeLevel(lpDirectSound, wndMain, DSSCL_PRIORITY);
	if (hr==DS_OK)
	{
		// Succeeded--try to create buffer
		hr = IDirectSound_CreateSoundBuffer(lpDirectSound,
			&dsbdesc, &lpDirectSoundBuffer, NULL);
	}
	if (FAILED(hr)) goto DirectSoundFailed;

	/* Get the address of the buffer */
	hr = IDirectSoundBuffer_Lock(lpDirectSoundBuffer, 0, 0, 
		&lpvPtr1, &dwBytes1, &lpvPtr2, &dwBytes2, DSBLOCK_ENTIREBUFFER);
	if (FAILED(hr)) goto DirectSoundFailed;

	if ((r = WaveReadFile(hm, mmci.cksize,
		(BYTE *)lpvPtr1, &mmci, &bytesread)!=0))
	{
		goto DirectSoundFailed;
	}

	/* Give the buffer back to DirectSound */
	hr = IDirectSoundBuffer_Unlock(lpDirectSoundBuffer,
		lpvPtr1, dwBytes1, lpvPtr2, dwBytes2);
	if (FAILED(hr)) goto DirectSoundFailed;

	/* If audio is active, play the sound buffer */
	if (!audio_suspended)
	{
		IDirectSoundBuffer_SetVolume(lpDirectSoundBuffer,
//			sample_volume?(long)((log10(sample_volume*0.85)-2)*5000):-10000);
			sample_volume?(long)((log10(sample_volume)-2)*5000):-10000);
	}
	else	/* muted */
	{
		IDirectSoundBuffer_SetVolume(lpDirectSoundBuffer, -10000);
	}
	IDirectSoundBuffer_Play(lpDirectSoundBuffer, 0, 0,
		loop_flag ? DSBPLAY_LOOPING : 0);


	WaveCloseReadFile(&hm, &pwf);

	// Set up the resume_sample struct
	strcpy(resume_sample.filename, loaded_filename);
	strcpy(resume_sample.resname, loaded_resname);
	resume_sample.loop_flag = loop_flag;

	return true;
}


void StopSample_DirectSound(void)
{
	if (lpDirectSoundBuffer)
	{
		IDirectSoundBuffer_Stop(lpDirectSoundBuffer);
		IDirectSoundBuffer_Release(lpDirectSoundBuffer);
	}
	lpDirectSoundBuffer = NULL;
	if (lpDirectSound) IDirectSound_Release(lpDirectSound);
	lpDirectSound = NULL;
}

#endif	// #ifndef USE_NT_SOUND_ONLY

}	// extern "C"
