/*
	hewince.c

	WinCE-specific functions for Hugo for WinCE

	Copyright (c) 2002-2006 by Kent Tessman
*/

#include <windows.h>
#include <aygshell.h>
#include <sipapi.h>
#include <commctrl.h>
#include <commdlg.h>
#ifndef IDM_NEWMENUMAX
#include <newmenu.h>	// Thanks for changing the header contents
#endif

#include <time.h>

#include "heheader.h"
#include "hewin32.h"
#include "hemenu.h"
#include "hewince.h"
#include "ceres.h"


HWND menubar = NULL;		// CE's menu bar
int reg_MinimalWindows = false, reg_NoBlankLines = false, reg_UseColors = true;
unsigned char illegal_window = false, minimal_windowing = false;

int system_font_height;
#define MENUITEM_HEIGHT (system_font_height+5)

#ifdef PROMPTMORE_REPLACED
char promptmore_active = false;
#endif


// prototypes
void CheckCEMenuItems(void);


// from hewin.c
extern int reg_FixedPoint, reg_PropPoint;
extern TCHAR reg_FixedFont[], reg_PropFont[];
extern WCHAR unicode_buffer[];
void CheckMenuItems(void);
void ResetCharDimensions(int);

extern HMENU menuContext;
int BuildContextMenu(HMENU hmenu);

// from hemsvc.c
extern int last_hugo_font;


//---------------------------------------------------------------------------
// Popup commands, directions, etc.
//---------------------------------------------------------------------------

#define COMMAND_LENGTH 24
typedef struct
{
	TCHAR cmd[COMMAND_LENGTH+1];
	BOOL linefeed;
} list_struct;

#define ID_DIRECTION1 (WM_APP)
#define NUM_DIRECTIONS 12
list_struct default_dir_list[NUM_DIRECTIONS] = {
	{_T("North"), false}, {_T("Northeast"), false}, {_T("East"), false}, {_T("Southeast"), false},
	{_T("South"), false}, {_T("Southwest"), false}, {_T("West"), false}, {_T("Northwest"), false},
	{_T("Up"), false}, {_T("Down"), false}, {_T("In"), false}, {_T("Out"), false}
};

#define ID_COMMAND1 (ID_DIRECTION1 + NUM_DIRECTIONS)
#define NUM_COMMANDS 12
list_struct default_command_list[NUM_COMMANDS] = {
	{_T("Examine"), true}, {_T("Get"), true}, {_T("Drop"), true}, {_T("Ask ^ about"), false},
	{_T(""), false}, {_T(""), false}, {_T(""), false}, {_T(""), false},
	{_T(""), false}, {_T(""), false}, {_T(""), false}, {_T(""), false}
};

#define ID_WORD1 (ID_COMMAND1 + NUM_COMMANDS)
#define NUM_WORDS 12
list_struct default_word_list[NUM_WORDS] = {
	{_T("Look"), true}, {_T("Inventory"), true}, {_T("Undo"), true}, {_T(""), false},
	{_T(""), false}, {_T(""), false}, {_T(""), false}, {_T(""), false},
	{_T(""), false}, {_T(""), false}, {_T(""), false}, {_T(""), false}
};

typedef struct 
{
	list_struct dir_list[NUM_DIRECTIONS];
	list_struct command_list[NUM_COMMANDS];
	list_struct word_list[NUM_WORDS];
} PreferenceType;

PreferenceType prefs;

void ResetDirectionList(void)
{
	int i;
	for (i=0; i<NUM_DIRECTIONS; i++)
	{
		_tcscpy(prefs.dir_list[i].cmd, default_dir_list[i].cmd);
		prefs.dir_list[i].linefeed = false;
	}
}

void ResetCommandList(void)
{
	int i;
	for (i=0; i<NUM_COMMANDS; i++)
	{
		_tcscpy(prefs.command_list[i].cmd, default_command_list[i].cmd);
		prefs.command_list[i].linefeed = default_command_list[i].linefeed;
	}
}

void ResetWordList(void)
{
	int i;
	for (i=0; i<NUM_WORDS; i++)
	{
		_tcscpy(prefs.word_list[i].cmd, default_word_list[i].cmd);
		prefs.word_list[i].linefeed = default_word_list[i].linefeed;
	}
}

int wce_GetMenuItemCount(HMENU hMenu)	// from wcealt.cpp
{
	const int MAX_NUM_ITEMS = 256;
	int  iPos, iCount;

	MENUITEMINFO mii;
	memset((char *)&mii, 0, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);

	iCount = 0;
	for (iPos = 0; iPos < MAX_NUM_ITEMS; iPos++)
	{
		if(!GetMenuItemInfo(hMenu, (UINT)iPos, TRUE, &mii))
			break;
		iCount++;
	}

	return iCount;
}

int RebuildPopupMenuFromList(HMENU hmenu, int firstid, list_struct *list, int num)
{
	int i, count = 0;

	if (!hmenu || !list) return 0;

	for (i=0; i<num; i++)
	{
		if (_tcscmp(list[i].cmd, _T("")))
		{
			AppendMenu(hmenu, MF_STRING, firstid+i, list[i].cmd);
			count++;
		}
	}

	return count;
}

HMENU word_menu = NULL;

void ResetPopupMenus(void)
{
	HMENU hmenu;

	// These are by-position, unfortunately

	// Reset direction menu
	hmenu = GetSubMenu(menuMain, 2);
	while (wce_GetMenuItemCount(hmenu))
		DeleteMenu(hmenu, 0, MF_BYPOSITION);
	RebuildPopupMenuFromList(hmenu, ID_DIRECTION1, prefs.dir_list, NUM_DIRECTIONS);

	// Reset word/context menu
	hmenu = GetSubMenu(menuMain, 3);
	if (word_menu)
	{
		while (wce_GetMenuItemCount(word_menu))
			DeleteMenu(word_menu, 0, MF_BYPOSITION);
	}
	while (wce_GetMenuItemCount(hmenu))
		DeleteMenu(hmenu, 0, MF_BYPOSITION);
	if (context_commands)
	{
		BuildContextMenu(hmenu);

		word_menu = CreatePopupMenu();
		if (word_menu)
		{
			AppendMenu(hmenu, MF_POPUP, (UINT)word_menu, _T("..."));
			hmenu = word_menu;
		}
	}
	RebuildPopupMenuFromList(hmenu, ID_WORD1, prefs.word_list, NUM_WORDS);
}


//---------------------------------------------------------------------------
// Popup list management
//---------------------------------------------------------------------------

list_struct *edit_list;
int edit_num;
BOOL edit_allow_linefeed;

BOOL CALLBACK EditListDialogProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			int i;
			for (i=0; i<edit_num; i++)
			{
				SetWindowText(GetDlgItem(hwndDlg, IDC_EDITITEM1+i), edit_list[i].cmd);
				if (edit_allow_linefeed)
					SendMessage(GetDlgItem(hwndDlg, IDC_CHECKITEM1+i), BM_SETCHECK, edit_list[i].linefeed, 0);
				else
					EnableWindow(GetDlgItem(hwndDlg, IDC_CHECKITEM1+i), FALSE);
			}
			return TRUE;
		}

		case WM_COMMAND:
		{
			switch (LOWORD(wParam))
			{
				case IDOK:
				{
					int i;
					for (i=0; i<edit_num; i++)
					{
						GetWindowText(GetDlgItem(hwndDlg, IDC_EDITITEM1+i), edit_list[i].cmd, COMMAND_LENGTH);
						if (SendMessage(GetDlgItem(hwndDlg, IDC_CHECKITEM1+i), BM_GETCHECK, 0, 0)==BST_CHECKED)
							edit_list[i].linefeed = TRUE;
						else
							edit_list[i].linefeed = FALSE;
					}
					EndDialog(hwndDlg, 1);
					break;
				}

				case IDRESET:
				{
					if (MessageBox(hwndDlg,
						_T("This will reset all items to their default values.  Continue?"),
						_T("Reset List Items"),
						MB_ICONQUESTION | MB_OKCANCEL )==IDOK)
					{
						if (edit_list==prefs.command_list)
							ResetCommandList();
						else if (edit_list==prefs.dir_list)
							ResetDirectionList();
						else if (edit_list==prefs.word_list)
							ResetWordList();
						else
							break;

						SendMessage(hwndDlg, WM_INITDIALOG, 0, 0);
					}
					break;
				}

				case IDCANCEL:
					EndDialog(hwndDlg, 0);
					break;
			}
		}
	}

	return FALSE;
}

void EditPopupList(list_struct *list, int num, BOOL allow_linefeed)
{
	edit_list = list;
	edit_num = num;
	edit_allow_linefeed = allow_linefeed;

	if (DialogBox(AppInstance, MAKEINTRESOURCE(IDD_EDITLIST), wndMain, EditListDialogProc)==IDOK)
		ResetPopupMenus();
}


//---------------------------------------------------------------------------
// MakePopupCommand
//---------------------------------------------------------------------------

TCHAR *MakePopupCommand(TCHAR *cmd, char *word)
{
	_tcscpy(unicode_buffer, cmd);
	if (_tcsstr(unicode_buffer, _T("^")))
	{
		strcpy_AtoU(_tcsstr(unicode_buffer, _T("^")), word);
		_tcscat(unicode_buffer, _tcsstr(cmd, _T("^"))+1);
	}
	else
	{
		_tcscat(unicode_buffer, _T(" "));
		strcpy_AtoU(unicode_buffer+_tcslen(unicode_buffer), word);
	}
	
	return unicode_buffer;
}


//---------------------------------------------------------------------------
// FontSizeDialogProc
//---------------------------------------------------------------------------

int *edit_fontsize;

BOOL CALLBACK FontSizeDialogProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	static HFONT hfont = NULL;

	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			// Point-size range 2-12
			SendMessage(GetDlgItem(hwndDlg, IDC_FONTSIZE_SLIDER), TBM_SETRANGE, TRUE, 
				(LPARAM)MAKELONG(2, 12));
			SendMessage(GetDlgItem(hwndDlg, IDC_FONTSIZE_SLIDER), TBM_SETPOS, TRUE, *edit_fontsize);

			// Update the text size
			SendMessage(hwndDlg, WM_NOTIFY, IDC_FONTSIZE_SLIDER, 0);

			return TRUE;
		}
		case WM_DESTROY:
		{
			if (hfont) DeleteObject(hfont);
			hfont = NULL;
			break;
		}
		case WM_NOTIFY:
		{
			if (wParam==IDC_FONTSIZE_SLIDER)
			{
				int size;
				LOGFONT lf;

				memset(&lf, 0, sizeof(LOGFONT));
				lf.lfWeight = FW_NORMAL;
				lf.lfStrikeOut = 0;
				
				size = SendMessage(GetDlgItem(hwndDlg, IDC_FONTSIZE_SLIDER), TBM_GETPOS, 0, 0);
				// Convert point to logical height
				lf.lfHeight = -(long)(MulDiv(size, GetDeviceCaps(dcMem, LOGPIXELSY), 72));
				
				if (edit_fontsize==&reg_FixedPoint)
				{
					lf.lfPitchAndFamily = FIXED_PITCH;
					_tcscpy(lf.lfFaceName, reg_FixedFont);
				}
				else
				{
					lf.lfPitchAndFamily = VARIABLE_PITCH;
					_tcscpy(lf.lfFaceName, reg_PropFont);
				}

				// Get rid of the old font
				if (hfont) DeleteObject(hfont);

				hfont = CreateFontIndirect(&lf);
				if (hfont)
					SendMessage(GetDlgItem(hwndDlg, IDC_FONTSIZE_EXAMPLE), WM_SETFONT, (WPARAM)hfont, TRUE);
			}
			break;
		}
		case WM_COMMAND:
		{
			switch (LOWORD(wParam))
			{
				case IDOK:
				{
					if (hfont)
					{
						int tempfont = currentfont;

						// Change the font size variable
						*edit_fontsize = SendMessage(GetDlgItem(hwndDlg, IDC_FONTSIZE_SLIDER), TBM_GETPOS, 0, 0);

						// Need to do this here to make sure hugo_font() sets the font
						last_hugo_font = -1;
						if (edit_fontsize==&reg_FixedPoint)
						{
							hugo_font(currentfont = NORMAL_FONT);
							ResetCharDimensions(NORMAL_FONT);
						}
						else
						{
							hugo_font(currentfont = PROP_FONT);
							ResetCharDimensions(PROP_FONT);
						}
						hugo_font(currentfont = tempfont);
					}
					EndDialog(hwndDlg, 1);
					break;
				}

				case IDCANCEL:
					EndDialog(hwndDlg, 0);
					break;
			}
			break;
		}
	}

	return false;
}


//---------------------------------------------------------------------------
// WinCEWndProc
//
// Return 0 to allow WndProc's normal processing; return 1 to override.
//---------------------------------------------------------------------------

LRESULT CALLBACK WinCEWndProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
		case WM_ACTIVATE:
		case WM_KILLFOCUS:
			CheckSIPStatus();
			break;

		case WM_SETTINGCHANGE:
		{
			if (wParam==SPI_SETSIPINFO)
			{
				CheckSIPStatus();
				// We want to avoid spurious "resizes" due to the SIP
				override_resize_handling = true;
			}
			break;
		}
		case WM_INITMENUPOPUP:
		case WM_EXITMENULOOP:
		{
			CheckMenuItems();
			ResetPopupMenus();
			break;
		}
		case WM_KEYDOWN:
		{
			if (promptmore_active)
			{
				full = 0;
				return -1;
			}
			break;
		}
		case WM_COMMAND:
		{
			switch LOWORD(wParam)
			{
				case ID_MORE:
					full = 0;
					break;
			}

			if (!getline_active)
				break;
				
			// Handle direction menu
			if (LOWORD(wParam)>=ID_DIRECTION1 && LOWORD(wParam)<ID_DIRECTION1+NUM_DIRECTIONS)
			{
				unsigned int i, dir;
				PushKeypress(OVERRIDE_UPDATE_CLIENT);
				PushKeypress(27);	// Esc
				dir = LOWORD(wParam)-ID_DIRECTION1;
				for (i=0; i<_tcslen(prefs.dir_list[dir].cmd); i++)
				{
					PushKeypress(prefs.dir_list[dir].cmd[i]);
				}
				PushKeypress(UPDATE_CLIENT);
				PushKeypress(13);	// Enter
			}

			// Handle word menu
			else if (LOWORD(wParam)>=ID_WORD1 && LOWORD(wParam)<ID_WORD1+NUM_WORDS)
			{
				unsigned int i, w;
				PushKeypress(OVERRIDE_UPDATE_CLIENT);
				//PushKeypress(27);	// Esc
				w = LOWORD(wParam)-ID_WORD1;
				for (i=0; i<_tcslen(prefs.word_list[w].cmd); i++)
				{
					PushKeypress(prefs.word_list[w].cmd[i]);
				}
				PushKeypress(UPDATE_CLIENT);
				if (prefs.word_list[w].linefeed)
					PushKeypress(13);  // Enter
			}

			else switch (LOWORD(wParam))
			{
				// Lists

				case HE_OPTIONS_EDITCOMMANDS:
				{
					EditPopupList(prefs.command_list, NUM_COMMANDS, true);
					return true;
				}
				case HE_OPTIONS_EDITDIRECTIONS:
				{
					EditPopupList(prefs.dir_list, NUM_DIRECTIONS, false);
					return true;
				}
				case HE_OPTIONS_EDITWORDS:
				{
					EditPopupList(prefs.word_list, NUM_WORDS, true);
					return true;
				}
				case HE_OPTIONS_LOADLISTS:
				{
					TCHAR fn[MAXPATH] = _T("");
					char afn[MAXPATH];
					OPENFILENAME ofn;

					memset(&ofn, 0, sizeof(OPENFILENAME));
					ofn.lStructSize = sizeof(OPENFILENAME);
					ofn.hwndOwner = wndMain;
					ofn.lpstrFilter = _T("Settings files\0*.settings\0All files\0*.*\0");
					ofn.lpstrCustomFilter = NULL;
					ofn.nFilterIndex = 0;
					ofn.lpstrFile = fn;
					ofn.lpstrDefExt = _T("settings");
					ofn.nMaxFile = MAXPATH;
					ofn.lpstrFileTitle = NULL;
					ofn.lpstrInitialDir = NULL;
					ofn.lpstrTitle = _T("Load Lists");
					ofn.Flags = OFN_PATHMUSTEXIST | OFN_EXPLORER |
						OFN_HIDEREADONLY | OFN_ENABLEHOOK;

					if (!GetOpenFileName(&ofn))
						return true;

					strcpy_UtoA(afn, fn);

					if (!LoadCESettingsFile(afn))
					{
						MessageBox(wndMain,
							_T("Unable to load settings file."),
							_T("Hugo Engine"),
							MB_ICONERROR | MB_OK);
					}

					return true;
				}
				case HE_OPTIONS_SAVELISTS:
				{
					TCHAR fn[MAXPATH] = _T("");
					char afn[MAXPATH];
					OPENFILENAME ofn;

					memset(&ofn, 0, sizeof(OPENFILENAME));
					ofn.lStructSize = sizeof(OPENFILENAME);
					ofn.hwndOwner = wndMain;
					ofn.lpstrFilter = _T("Settings files\0*.settings\0All files\0*.*\0");
					ofn.lpstrCustomFilter = NULL;
					ofn.nFilterIndex = 0;
					ofn.lpstrFile = fn;
					ofn.lpstrDefExt = _T("settings");
					ofn.nMaxFile = MAXPATH;
					ofn.lpstrFileTitle = NULL;
					ofn.lpstrInitialDir = NULL;
					ofn.lpstrTitle = _T("Load Lists");
					ofn.Flags = OFN_PATHMUSTEXIST | OFN_EXPLORER |
						OFN_HIDEREADONLY | OFN_ENABLEHOOK;

					if (!GetSaveFileName(&ofn))
						return true;

					strcpy_UtoA(afn, fn);

					if (!SaveCESettingsFile(afn))
					{
						MessageBox(wndMain,
							_T("Unable to save settings file."),
							_T("Hugo Engine"),
							MB_ICONERROR | MB_OK);
					}

					return true;
				}

				// Fonts

				case HE_OPTIONS_FONTSIZE_FIXED:
				{
					edit_fontsize = &reg_FixedPoint;
					DialogBox(AppInstance, MAKEINTRESOURCE(IDD_FONTSIZE), wndMain, FontSizeDialogProc);
					return true;
				}
				case HE_OPTIONS_FONTSIZE_PROP:
				{
					edit_fontsize = &reg_PropPoint;
					DialogBox(AppInstance, MAKEINTRESOURCE(IDD_FONTSIZE), wndMain, FontSizeDialogProc);
					return true;
				}

				// Display

				case HE_OPTIONS_MINIMALWINDOWS:
				{
					reg_MinimalWindows = !reg_MinimalWindows;
					CheckCEMenuItems();
					return true;
				}
				case HE_OPTIONS_NOBLANKLINES:
				{
					reg_NoBlankLines = !reg_NoBlankLines;
					CheckCEMenuItems();
					return true;
				}
				case HE_OPTIONS_USECOLORS:
				{
					reg_UseColors = !reg_UseColors;
					CheckCEMenuItems();
					return true;
				}
				case ID_CONTEXT_COMMANDS:
				{
					RECT rect;
					GetClientRect(wndMain, &rect);
					HandleContextCommands(rect.right/2, rect.bottom);
					return true;
				}

				// Other

				case HE_OPTIONS_CLEARFILES:
				{
					if (MessageBox(wndMain,
						_T("Clear recently used files list?"),
						_T("Clear Recent Files"),
						MB_ICONQUESTION | MB_OKCANCEL )==IDOK)
					{
						int i;
						for (i=0; i<MRU_FILE_COUNT; i++)
							_tcscpy(MRU_filename[i], _T(""));
					}
					return true;
				}
			}

			break;
		}
		case WM_LBUTTONDOWN:
		{
			char *word;

			if (promptmore_active)
			{
				full = 0;
				return -1;
			}
			
			word = TB_FindWord(LOWORD(lParam), HIWORD(lParam));
			if (during_player_input && word)
			{
				POINT pt;
				int i, num, prev, count;
				HMENU hmenu;
				RECT rect;
				BOOL linefeed[NUM_COMMANDS+1];
				MSG msg;
				int result;

				pt.x = LOWORD(lParam);
				pt.y = HIWORD(lParam);


				// Figure out how many commands we're going to display
				if (!(hmenu = CreatePopupMenu()))
					break;

				num = 0;
				for (i=0; i<NUM_COMMANDS; i++)
				{
					if (_tcscmp(prefs.command_list[i].cmd, _T("")))
						num++;
				}
				if (!num)
					break;
		
				// Figure out how many, if any, entries will come before
				// the word alone in the list
				ClientToScreen(wndMain, &pt);
				GetClientRect(wndMain, &rect);
				if (pt.y+(num+1)*MENUITEM_HEIGHT > rect.bottom)
					prev = ((pt.y+(num+1)*MENUITEM_HEIGHT)-rect.bottom)/MENUITEM_HEIGHT;
				else
					prev = 0;

				count = 0;
				for (i=0; i<prev; i++)
				{
					if (_tcscmp(prefs.command_list[i].cmd, _T("")))
					{
						AppendMenu(hmenu, 0, ID_COMMAND1+count, MakePopupCommand(prefs.command_list[i].cmd, word));
						if (prefs.command_list[i].linefeed)
							linefeed[count] = true;
						else
							linefeed[count] = false;
						count++;
					}
				}

				// Now add the word alone
				strcpy_AtoU(unicode_buffer, word);
				AppendMenu(hmenu, 0, ID_COMMAND1+count, unicode_buffer);
				linefeed[count] = false;
				count++;

				// And add the remaining commands
				for (i=prev; i<NUM_COMMANDS; i++)
				{
					if (_tcscmp(prefs.command_list[i].cmd, _T("")))
					{
						AppendMenu(hmenu, 0, ID_COMMAND1+count, MakePopupCommand(prefs.command_list[i].cmd, word));
						if (prefs.command_list[i].linefeed)
							linefeed[count] = true;
						else
							linefeed[count] = false;
						count++;
					}
				}

				result = TrackPopupMenu(hmenu,
						TPM_RIGHTBUTTON | TPM_NONOTIFY | TPM_RETURNCMD,
						pt.x-20, pt.y-MENUITEM_HEIGHT/3 - prev*MENUITEM_HEIGHT - 2,
						0, wndMain, NULL);

				// Without this, we won't be able to dismiss the
				// popup menu by clicking outside it--it'll just
				// call up another popup menu
				PeekMessage(&msg, wndMain,
					WM_LBUTTONDOWN, WM_LBUTTONDOWN, PM_REMOVE);

				if (result)
				{
					MENUITEMINFO mi;
					mi.cbSize = sizeof(MENUITEMINFO);
					mi.fMask = MIIM_TYPE;
					mi.dwTypeData = unicode_buffer;
					mi.cch = MAXBUFFER;
					GetMenuItemInfo(hmenu, result, FALSE, &mi);
					PushKeypress(OVERRIDE_UPDATE_CLIENT);
					for (i=0; i<(signed)_tcslen(mi.dwTypeData); i++)
					{
						PushKeypress(mi.dwTypeData[i]);
					}
					if (!linefeed[result-ID_COMMAND1])
						PushKeypress(' ');
					PushKeypress(UPDATE_CLIENT);
					if (linefeed[result-ID_COMMAND1])
						PushKeypress(13);
				}

				DestroyMenu(hmenu);

				return (1);
			}

			break;
		}
	}

	return 0;
}


//---------------------------------------------------------------------------
// CreateCEMenubar
//---------------------------------------------------------------------------

void CreateCEMenubar(HINSTANCE hInstance, HWND wnd)
{
	if (wnd)
	{
		// Create a menubar
		SHMENUBARINFO mbi;
		memset(&mbi, 0, sizeof(SHMENUBARINFO));
		mbi.cbSize     = sizeof(SHMENUBARINFO);
		mbi.hwndParent = wnd;
		mbi.dwFlags = 0;
		mbi.nToolBarId = HEWINCE_MENUBAR;
		mbi.hInstRes   = hInstance;
		mbi.nBmpId     = IDR_TOOLBAR1;
		mbi.cBmpImages = 2;
        
		if (SHCreateMenuBar(&mbi))
		{
			menubar = mbi.hwndMB;
			return;
		}
	}

	MessageBox(NULL, _T("Error creating CE menubar."), _T("Hugo Engine"), MB_ICONEXCLAMATION);
	exit(0);
}


//---------------------------------------------------------------------------
// Menu bar
//---------------------------------------------------------------------------

void EnableMenuButton(int id, BOOL on)
{
	TBBUTTONINFO bi;

	bi.cbSize = sizeof(TBBUTTONINFO);
	bi.dwMask = TBIF_STATE;

	if (SendMessage(menubar, TB_GETBUTTONINFO, id, (LPARAM)&bi))
	{
		if (on)
			bi.fsState |= TBSTATE_ENABLED;
		else
			bi.fsState &= ~TBSTATE_ENABLED;
		SendMessage(menubar, TB_SETBUTTONINFO, id, (LPARAM)&bi);
	}
}

void EnableInputButtons(BOOL on)
{
	EnableMenuButton(ID_MOVE, on);
	EnableMenuButton(ID_WORDS, on);

	if (on) ResetPopupMenus();
	
	DrawMenuBar(wndMain);
}

void EnableMenuBar(BOOL on)
{
//	EnableMenuButton(ID_FILE, on);
	EnableMenuButton(ID_OPTIONS, on);
	if (getline_active)
	{
		EnableMenuButton(ID_MOVE, on);
		EnableMenuButton(ID_WORDS, on);
	}
}

void CheckCEMenuItems(void)
{
	CheckMenuItem(menuMain, HE_OPTIONS_NOBLANKLINES,
		reg_NoBlankLines?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem(menuMain, HE_OPTIONS_MINIMALWINDOWS,
		reg_MinimalWindows?MF_CHECKED:MF_UNCHECKED);
	CheckMenuItem(menuMain, HE_OPTIONS_USECOLORS,
		reg_UseColors?MF_CHECKED:MF_UNCHECKED);

	DrawMenuBar(wndMain);
}


//---------------------------------------------------------------------------
// SetupCEWindow
//---------------------------------------------------------------------------

void SetupCEWindow(HINSTANCE hInstance, HWND wnd)
{
	TBBUTTONINFO bi;
	NMNEWMENU nm;
	HDC dc;
	HFONT font;
	TEXTMETRIC tm;

	// Window-shifting screws up if SIP is showing at game start
	SipShowIM(SIPF_OFF);
	CheckSIPStatus();

	// Part of figuring out how big a given popup menu line will be
	dc = GetDC(wndMain);
	font = SelectObject(dc, GetStockObject(SYSTEM_FONT));
	GetTextMetrics(dc, &tm);
	system_font_height = tm.tmHeight;
	SelectObject(dc, font);
	DeleteDC(dc);

	CreateCEMenubar(hInstance, wnd);

	menuMain = (HMENU)SendMessage(menubar, SHCMBM_GETMENU, 0, (LPARAM)&nm);

	CheckCEMenuItems();

	bi.cbSize = sizeof(TBBUTTONINFO);
	bi.dwMask = TBIF_STATE;
	if (SendMessage(menubar, TB_GETBUTTONINFO, ID_MORE, (LPARAM)&bi))
	{
		bi.fsState |= TBSTATE_HIDDEN;
		SendMessage(menubar, TB_SETBUTTONINFO, ID_MORE, (LPARAM)&bi);
		DrawMenuBar(wndMain);
	}

#ifdef NO_GRAPHICS
	EnableMenuItem(menuMain, HE_OPTIONS_SHOWGRAPHICS, MF_GRAYED);
#endif

#ifdef NO_SOUND
	EnableMenuItem(menuMain, HE_OPTIONS_PLAYSOUNDS, MF_GRAYED);
#endif

#if defined (NO_GRAPHICS) && defined(NO_SOUND)
#endif

	ResetDirectionList();
	ResetCommandList();
	ResetWordList();

#ifdef UNDER_CE
	LoadDefaultCESettingsFile();
#endif

	ResetPopupMenus();

	EnableInputButtons(false);

// EnableInputButtons() does this
//	 DrawMenuBar(menubar);
}


//---------------------------------------------------------------------------
// CheckSIPStatus
//---------------------------------------------------------------------------

static char sip_shown = 0;

void CheckSIPStatus(void)
{
	static int sip_height = 0;
	SIPINFO si;
	RECT win_rect, visible_rect;
	int left, top, right, bottom;

	si.cbSize = sizeof(SIPINFO);
	si.dwImDataSize = 0;
	SipGetInfo(&si);

	if (si.fdwFlags & SIPF_ON && sip_shown)
		return;
	else if (!(si.fdwFlags & SIPF_ON) && sip_shown==0)
		return;
	
	if (si.fdwFlags & SIPF_ON)
		sip_shown = 1;
	else
		sip_shown = 0;
	
	GetWindowRect(wndMain, &win_rect);
	visible_rect = si.rcVisibleDesktop;

	if (sip_shown)
		sip_height = win_rect.bottom - visible_rect.bottom - visible_rect.top;

	left = win_rect.left;
	right = win_rect.right;
	top = win_rect.top + sip_height*(sip_shown?-1:1);
	bottom = win_rect.bottom - win_rect.top;

	MoveWindow(wndMain, left, top, right, bottom, FALSE);
}


//---------------------------------------------------------------------------
// CE settings file
//---------------------------------------------------------------------------

// Increment this every time we change the settings file format or
// prefs structure:
#define SETTINGS_VERSION_CHECK 1

BOOL LoadDefaultCESettingsFile(void)
{
	char settings_file[MAXPATH];
	char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILENAME], ext[MAXEXT];

	// Put the settings file in the application directory
	strcpy_UtoA(settings_file, executable_module);
	hugo_splitpath(settings_file, drive, dir, fname, ext);
	hugo_makepath(settings_file, drive, dir, CE_SETTINGS_FILE, "");

	return LoadCESettingsFile(settings_file);
}

BOOL LoadCESettingsFile(char *filename)
{
	FILE *file;
	PreferenceType loadprefs;

	file = fopen(filename, "rb");
	if (!file) return false;

	// Read crude header
	if (fgetc(file)!='P' || fgetc(file)!='H' || fgetc(file)!='S' ||
		 fgetc(file)!=SETTINGS_VERSION_CHECK)
	{
		fclose(file);
		return false;
	}

	if (fread(&loadprefs, sizeof(PreferenceType), 1, file) < 1)
	{
		fclose(file);
		return false;
	}
	
	fclose(file);

	memcpy(&prefs, &loadprefs, sizeof(PreferenceType));

	return true;
}

BOOL SaveDefaultCESettingsFile(void)
{
	char settings_file[MAXPATH];
	char drive[MAXDRIVE], dir[MAXDIR], fname[MAXFILENAME], ext[MAXEXT];

	// Put the settings file in the application directory
	strcpy_UtoA(settings_file, executable_module);
	hugo_splitpath(settings_file, drive, dir, fname, ext);
	hugo_makepath(settings_file, drive, dir, CE_SETTINGS_FILE, "");

	return SaveCESettingsFile(settings_file);
}

BOOL SaveCESettingsFile(char *filename)
{
	FILE *file;

	file = fopen(filename, "wb");
	if (!file) return false;

	// Write crude header
	fputc('P', file);
	fputc('H', file);
	fputc('S', file);
	fputc(SETTINGS_VERSION_CHECK, file);

	if (fwrite(&prefs, sizeof(PreferenceType), 1, file) < 1)
	{
		fclose(file);
		return false;
	}

	fclose(file);

	return true;
}


//---------------------------------------------------------------------------
// PromptMore
//---------------------------------------------------------------------------

//#define HIDE_SIP_DURING_PROMPTMORE

#ifdef PROMPTMORE_REPLACED
void PromptMore(void)
{
	TBBUTTONINFO bi;
	MSG msg;
	BOOL result;
	int old_sip_shown = sip_shown;

	promptmore_active = true;

#ifdef HIDE_SIP_DURING_PROMPTMORE
	if (sip_shown)
	{
		SipShowIM(SIPF_OFF);
		CheckSIPStatus();
	}
#endif

	// Show the "More" button
	bi.cbSize = sizeof(TBBUTTONINFO);
	bi.dwMask = TBIF_STATE;
	if (SendMessage(menubar, TB_GETBUTTONINFO, ID_MORE, (LPARAM)&bi))
	{
		bi.fsState &= ~TBSTATE_HIDDEN;
		SendMessage(menubar, TB_SETBUTTONINFO, ID_MORE, (LPARAM)&bi);
	}

	EnableMenuBar(false);
	DrawMenuBar(wndMain);

	UpdateClient(true);

	while ((result = GetMessage(&msg, NULL, 0, 0)) > 0 && full)
	{
		TranslateMessage(&msg);

		if (msg.message!=WM_CHAR)
		{
			DispatchMessage(&msg);
		}
		if (msg.message==WM_QUIT)
		{
			break;
		}
	}
	if (result <= 0)
	{
		exit(msg.wParam);
	}

	// Hide the "More" button
	if (SendMessage(menubar, TB_GETBUTTONINFO, ID_MORE, (LPARAM)&bi))
	{
		bi.fsState |= TBSTATE_HIDDEN;
		SendMessage(menubar, TB_SETBUTTONINFO, ID_MORE, (LPARAM)&bi);
	}

	EnableMenuBar(true);

#ifdef HIDE_SIP_DURING_PROMPTMORE
	if (old_sip_shown)
	{
		SipShowIM(SIPF_ON);
		CheckSIPStatus();
	}
#endif

	promptmore_active = false;
	DrawMenuBar(wndMain);
}
#endif


//---------------------------------------------------------------------------
// Time support
//---------------------------------------------------------------------------

time_t time(time_t *t)
{
	// NOTE:  This is NOT a proper time_t result, but
	// since we're only using it for seeding random numbers,
	// the result will do.
	SYSTEMTIME st;
	GetLocalTime(&st);
	return st.wDay*24*3600 + st.wHour*3600 + st.wMinute*60 + st.wSecond;
}

void hugo_gettimeformatted(char *a)
{
	SYSTEMTIME st;

	GetLocalTime(&st);
	_stprintf(unicode_buffer, _T("%04d-%02d-%02d %02d:%02d:%02d"),
		st.wYear, st.wMonth, st.wDay, st.wHour, st.wMinute, st.wSecond);
	strcpy_UtoA(a, unicode_buffer);
}

