/* --------------------------------------------------------------------

outputFilter.t

This module defines an output filter function which makes it easier
to use some fancy HTML-TADS formatting tricks. I haven't come across
any other sample output filter code, so I thought people might find
this useful.

To use it, first make sure you're in HTML mode (by printing "\H+").
The commonInit() function is a good place to do this. Then you just
need to add the following call:

    setOutputFilter (&outputFilter);

(Note that _all_ TADS interpreters since version 2.3 understand HTML,
not just multimedia interpreters, although text-only terps will simply
ignore most tags. So there's no need to avoid HTML just because you're
using a text-only interpreter.)

Once installed, the filter modifies your game's output as follows:
  
 *  Apostrophes are changed to '&rsquo;', the curly-right-quote entity.
    This makes words like "isn't" look much nicer.
    
    (If you actually want to put quotes around a word, you should use
    the <Q> tag, <Q>like this</Q>. Exercise for the reader: extend this
    filter to modify, say, {curly brackets} into <Q> tags.)
  
 *  The underscore character toggles the '<em>' tag, so you can write
    stuff _like this_ to highlight a word or phrase.
  
 *  '--' is changed to '&endash;' and '---' is changed to '&emdash;'.
    This is stolen from LaTeX, where I find it a very handy feature.
  
 *  The string '...' is expanded with hard spaces, into '\ .\ .\ .'.
    This is obviously a matter of taste, but I think 'this . . .'
    looks nicer than 'this...', particularly in a proportional font.
    
    (Why not just print '. . .' in the first place? Well, the problem
    is that TADS will change this to '.  .  .' unless the user turns
    off the 'two spaces after each sentence' option. The use of hard
    spaces prevents this, but they're fiddly to type and rather ugly;
    so let's just do it automatically with a filter function.)
  
 *  '^' is expanded to '\b' and '~' is expanded to '\t'. The point of
    this is that you can easily change this behaviour at runtime, as
    in the Adventions games: I've supplied two sysverbs, 'space' and
    'indent', to demonstrate this.
    
    \b
    ^Now you can start blocks of text like this.
    ^~And start each new paragraph like this.
    ^~And the user can change adjust the formatting to his/her taste.
    
    The Adventions games seem to use functions called 'P()' and 'I()',
    which strikes me as being a bit cumbersome.

Escape sequences, HTML tags and format strings are not modified (but
format strings will be passed through the filter after evaluation).
The filter will get confused if you print out a lengthy HTML tag in
small chunks, however; you may want to turn it off in these cases,
with setOutputFilter(nil).

You might not want to use all of these features -- you might need to
use '_' or '^' as ordinary characters, for instance -- but it's easy
enough to turn some of them off: just follow the comments and delete
the relevant chunk of code.

This module was written by Iain Merrick (im@cs.york.ac.uk). I didn't
know anything about output filters until Stephen Granade sent me some
code for changing apostrophes into &rsquo;'s, but this code isn't
derived from his.

-------------------------------------------------------------------- */

// This module should work unchanged if you #include it after <std.t>.
// Alternatively, just carve it up and paste it into your game wherever
// you like. Depending on the order in which you do things, you might
// need to add the following prototype before calling setOutputFilter:

outputFilter: function;

// I like C-style syntax, so...

#pragma C+

// Finally, the main feature:

outputFilter: function (str)
{
	// All the work is done here. TADS will call this function with a
	// string which is due to be printed. If we return nil, the string
	// will be printed unchanged. If we return another string, on the
	// other hand, TADS will print that string instead. And that's what
	// we're going to do.

	// Here's the string we're going to return (initially empty):

	local result = '' ;

	// We also need some flags to keep track of what we're doing:

	local inFmt = nil ;	 // TRUE when we're inside a format string.
	local inTag = nil ;  // TRUE when we're inside an HTML tag.
	local inEmph = nil ; // TRUE when we're using _emphasis_.
	
	// Now we're ready to go. We just eat characters from 'str',
	// modify them, and append the result to 'result'. We keep doing
	// this until 'str' is empty, then return 'result.
	
	while (str != '')
	{
		// Split off the first char of the string.
		
		local c = substr (str, 1, 1) ;
		str = substr (str, 2, length (str) - 1) ;
		
		// '<' starts a tag and '>' finishes it;
		// '%' both starts and finishes a format string.
		
		if (c == '<' && !inTag) inTag = true ;
		if (c == '>' && inTag) inTag = nil ;
		
		if (c == '%') inFmt = !inFmt ;
		
		// If we're inside a tag or a format string,
		// we shouldn't modify this character.
		
		if (inTag || inFmt)
		{
			result += c ; // Append 'c' to 'result'.
			continue ; // Jump back to the top of the while loop.
		}
		
		// If we find a backslash, print both this character
		// and the next one unchanged (apart from the special
		// case of \').
		
		if (c == '\\')
		{
			// Peek at the next character.
			c = substr (str, 1, 1) ;
			
			if (c != '\'')
			{
				// If we have "\'", just drop the "\".
				// The "'" will be changed to "&rsquo;"
				// next time round the loop. (We could
				// print the "&rsquo;" here, but that
				// duplicates code unnecessarily.)
				continue ;
			}
			else
			{
				// Print the escape sequence unchanged.
				result += '\\' ;
				result += c ;
				str = substr (str, 2, length (str) - 1) ;
				continue ;
			}
		}
		
		// If we find an apostrophe, convert it to '&rsquo;'.
		
		if (c == '\'')
		{
			result += '&rsquo;' ;
			continue ;
		}
		
		// If we find an underscore, output an <em> or </em> tag.
		
		if (c == '_')
		{
			if (!inEmph)
			{
				result += '<em>' ;
				inEmph = true ;
			}
			else
			{
				result += '</em>' ;
				inEmph = nil ;
			}

			continue ;
		}
		
		// If we find a hyphen, look for more hyphens.
		// '--' becomes '&endash' and '---' becomes '&emdash'.
		
		if (c == '-')
		{
			if (substr (str, 1, 2) == '--')
			{
				str = substr (str, 3, length (str) - 2) ;
				result += '&emdash;' ;
				continue ;
			}
			
			if (substr (str, 1, 1) == '-')
			{
				str = substr (str, 2, length (str) - 1) ;
				result += '&endash;' ;
				continue ;
			}
		}
		
		// If we find '...', convert it to '\ .\ .\ .'
		
		if (c == '.')
		{
			if (substr (str, 1, 2) == '..')
			{
				str = substr (str, 3, length (str) - 2) ;
				result += '\ .\ .\ .' ;
				continue ;
			}
		}
		
		// '^' is a line break and '~' is an indent, used for spacing
		// out paragraphs of text. The precise behaviour is controlled
		// by the global variables 'doublespace' and 'indent'.
		
		if (c == '^')
		{
			if (global.doublespace)
				result += '\b' ;
			else
				result += '\n' ;
			continue ;
		}
		
		if (c == '~')
		{
			if (global.indent)
				result += '\t' ;
			continue ;
		}
		
		// If we didn't find anything special, just
		// output whatever character we found.
		
		result += c ;
		
		// Note: a 'switch' statement might be faster than all those
		// 'if's, particularly if you add more modifiers, but I decided
		// that using 'switch' would make the code less readable.
		
		// As it is, this filter doesn't slow things down noticeably on
		// any machine on which I've tried it. Your mileage may vary,
		// however, particularly if you're printing out large blocks of
		// text on a slow machine.
		
		// If you need an order-of-magnitude speedup, you might want to
		// try locating the special characters with find() or a regexp,
		// instead of just reading the string char by char.
	}
	
	return result ;
}

// Now we add some verbs to control the paragraph spacing. These are
// lifted almost directly from Dave Bagget's 'Colossal Cave Revisited'.

spaceVerb: sysverb
	sdesc = "space"
	verb = 'space'
	action(actor) = {
		if (global.doublespace) {
			"Now single-spacing text.";
			global.doublespace = nil;
		}
		else {
			"Now double-spacing text.";
			global.doublespace = true;
		}

		abort;	/* doesn't count as a turn */
 	}
;
indentVerb: sysverb
	sdesc = "indent"
	verb = 'indent'
	action(actor) = {
		if (global.indent) {
			"Paragraph indentation now off.";
			global.indent = nil;
		}
		else {
			"Paragraph indentation now on.";
			global.indent = true;
		}
		
		abort;	/* doesn't count as a turn */
 	}
;

// 'global.doublespace' and 'global.indent' don't actually
// exist yet, of course, but that's easily fixed:

modify global
	doublespace = true
	indent = true
;
