/*****************************************************************************
 *  Functions.t.  This TADS include file has all the neccessary functions
 *  in it, listed in alphabetical order.  I recommend that you store a copy
 *  of it away somewhere and simply modify it directly, adding new functions
 *  in the appropriate places.  While TADS lets you have a lot of include
 *  statements, that number is not unlimited, as I have seen.
 ****************************************************************************/

#pragma C-

/*
 *  addbulk: function( list )
 *
 *  This function returns the sum of the bulks (given by the bulk
 *  property) of each object in list.  The value returned includes
 *  only the bulk of each object in the list, and not of the contents
 *  of the objects, as it is assumed that an object does not change in
 *  size when something is put inside it.  You can easily change this
 *  assumption for special objects (such as a bag that stretches as
 *  things are put inside) by writing an appropriate bulk method
 *  for that object.
 */

//ABULK
addbulk: function( list )
{
    local i, tot, totbulk, rem, cur;

    tot := length( list );
    i := 1;
    totbulk := 0;
    while( i <= tot )
    {
        cur := list[i];
        if ( not cur.isworn )
            totbulk := totbulk + cur.bulk;
        i := i + 1;
    }
    return( totbulk );
}

/*
 *  addweight: function( list )
 *
 *  Adds the weights of the objects in list and returns the sum.
 *  The weight of an object is given by its weight property.  This
 *  routine includes the weights of all of the contents of each object,
 *  and the weights of their contents, and so forth.
 */

//AWEIT
addweight: function( l )
{
    local tot, i, c, totweight;

    tot := length( l );
    i := 1;
    totweight := 0;
    while ( i <= tot )
    {
        c := l[i];
        totweight := totweight + c.weight;
        if (length( c.contents ))
            totweight := totweight + addweight( c.contents );
        i := i + 1;
    }
    return( totweight );
}

/*
 *   checkDoor:  if the door d is open, this function silently returns
 *   the room r.  Otherwise, print a message ("The door is closed.") and
 *   return nil.
 */

//CKDOR
checkDoor: function( d, r )
{
    if ( d.isopen ) return( r );
    else
    {
	setit( d );
	caps(); d.thedesc; " is closed. ";
	return( nil );
    }
}

/*
 *   checkReach: determines whether the object obj can be reached by
 *   actor in location loc, using the verb v.  This routine returns true
 *   if obj is a special object (numObj or strObj), if obj is in actor's
 *   inventory or actor's location, or if it's in the 'reachable' list for
 *   loc.  
 */

//CKRCH
checkReach: function( loc, actor, v, obj )
{
    if ( obj=numObj or obj=strObj ) return;
    if ( not ( actor.isCarrying( obj ) or obj.isIn( actor.location )))
    {
	if (find( loc.reachable, obj ) <> nil ) return;
	"%You% can't reach "; obj.thedesc; " from "; loc.thedesc; ". ";
	exit;
    }
}

/*
 *   darktravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */

//DRKTR
darkTravel: function
{
    "You stumble around in the dark, and don't get anywhere. ";
}

/*
 *   The die() function is called when the player dies.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).
 */

//DIEEE
die: function
{
    "\b*** You have died ***\b";
    scoreRank();
    "\bYou may restore a saved game, start over, quit, or undo
    the current command.\n";
    while ( 1 )
    {
        local resp;

	"\nPlease enter RESTORE, RESTART, QUIT, or UNDO: >";
        resp := upper(input());
        if ( resp = 'RESTORE' )
	{
	    resp := askfile( 'File to restore' );
	    if ( resp = nil ) "Restore failed. ";
	    else if ( restore( resp )) "Restore failed. ";
	    else
	    {
	        Me.location.lookAround(true);
	        scoreStatus();
		abort;
	    }
	}
        else if ( resp = 'RESTART' )
	{
	    scoreStatus();
            restart();
	}
	else if (resp = 'UNDO')
	{
	    if (undo())
	    {
		"(Undoing one command)\b";
		Me.location.lookAround(true);
	        scoreStatus();
		abort;
	    }
	    else
		"Sorry, no undo information is available. ";
	}
	else if ( resp = 'QUIT' )
        {
	    terminate();
            quit();
	    abort;
        }
    }
}

/*
 *   The init() function is run at the very beginning of the game.
 *   This newer version of init() only starts daemons.  However, it calls
 *   userinit() which is a function that you can either replace or
 *   just edit directly.  You'll want to move the player's actor ("Me")
 *   to the initial room, which is defined in global.start (default is
 *   startroom, just as before.), display any introductory text, and have
 *   the player look.  In the default userinit(), the player is moved to
 *   global.start and looks around.  It also displays the sdesc of the
 *   version object.
 */

//EENIT
init: function
{
    setdaemon( turncount, nil );         // start the turn counter daemon
#ifdef AUTO_SAVE_REMINDER
    notify(global, #reminder, AUTO_SAVE_REMINDER*1);
#endif
#ifdef MUST_EAT
    setdaemon( eatDaemon, nil );         // start the hunger daemon
#endif
#ifdef MUST_SLEEP
    setdaemon( sleepDaemon, nil );       // start the hunger daemon
#endif
    userinit();
}

/*
 *   initRestart - flag when a restart has occurred by setting a flag
 *   in global.
 */

//IREST
initRestart: function(parm)
{
    global.restarting := true;
}

/*
 *  initSearch: function
 *
 *  Initializes the containers of objects with a searchLoc, underLoc,
 *  and behindLoc by setting up searchCont, underCont, and
 *  behindCont lists, respectively.  You should call this function once in
 *  your preinit (or init, if you prefer) function to ensure that
 *  the underable, behindable, and searchable objects are set up correctly.
 *  
 *  As a bonus, we take this opportunity to initialize global.floatingList
 *  with a list of all objects of class floatingItem.  It is necessary to
 *  initialize this list, so that validDoList and validIoList include objects
 *  with variable location properties.  Note that, for this to work,
 *  all objects with variable location properties must be declared
 *  to be of class floatingItem.
 */

//ISRCH
initSearch: function
{
    local o;
    
    o := firstobj(hiddenItem);
    while (o <> nil)
    {
	if (o.searchLoc)
	    o.searchLoc.searchCont := o.searchLoc.searchCont + o;
	else if (o.underLoc)
	    o.underLoc.underCont := o.underLoc.underCont + o;
	else if (o.behindLoc)
	    o.behindLoc.behindCont := o.behindLoc.behindCont + o;
	o := nextobj(o, hiddenItem);
    }
    
    global.floatingList := [];
    for (o := firstobj(floatingItem) ; o ; o := nextobj(o, floatingItem))
	global.floatingList += o;
}

/*
 *  isIndistinguishable: function(obj1, obj2)
 *
 *  Returns true if the two objects are indistinguishable for the purposes
 *  of listing.  The two objects are equivalent if they both have the
 *  isEquivalent property set to true, they both have the same immediate
 *  superclass, and their other listing properties match (in particular,
 *  isworn and (islamp and islit) match for both objects).
 */

//IIDIS
isIndistinguishable: function(obj1, obj2)
{
    return (firstsc(obj1) = firstsc(obj2)
            and obj1.isworn = obj2.isworn
            and ((obj1.islamp and obj1.islit)
                 = (obj2.islamp and obj2.islit)));
}

/*
 *  itemcnt: function( list )
 *
 *  Returns a count of the "listable" objects in list.  An
 *  object is listable (that is, it shows up in a room's description)
 *  if its isListed property is true.  This function is
 *  useful for determining how many objects (if any) will be listed
 *  in a room's description.  Indistinguishable items are counted as
 *  a single item (two items are indistinguishable if they both have
 *  the same immediate superclass, and their isEquivalent properties
 *  are both true.
 */

//ITMCT
itemcnt: function( list )
{
    local cnt, tot, i, obj, j;
    tot := length(list);
    cnt := 0;
    i := 1;
    for (i := 1, cnt := 0 ; i <= tot ; ++i)
    {
	/* only consider this item if it's to be listed */
	obj := list[i];
	if (obj.isListed)
	{
	    /*
	     *   see if there are other equivalent items later in the
	     *   list - if so, don't count it (this ensures that each such
	     *   item is counted only once, since only the last such item
	     *   in the list will be counted) 
	     */
	    if (obj.isEquivalent)
	    {
		local sc;
		
		sc := firstsc(obj);
		for (j := i + 1 ; j <= tot ; ++j)
		{
		    if (isIndistinguishable(obj, list[j]))
			goto skip_this_item;
		}
	    }
	    
	    /* count this item */
	    ++cnt;

	skip_this_item: ;
	}
    }
    return cnt;
}

/*
 *  listcont: function( obj )
 *
 *  This function displays the contents of an object, separated by
 *  commas.  The thedesc properties of the contents are used.
 *  It is up to the caller to provide the introduction to the list
 *  (usually something to the effect of "The box contains" is
 *  displayed before calling listcont) and finishing the
 *  sentence (usually by displaying a period).  An object is listed
 *  only if its isListed property is true.  If there are
 *  multiple indistinguishable items in the list, the items are
 *  listed only once (with the number of the items).
 */

//LCONT
listcont: function( obj )
{
    local i, count, tot, list, cur, disptot, prefix_count;

    list := obj.contents;
    tot := length( list );
    count := 0;
    disptot := itemcnt( list );
    for (i := 1 ; i <= tot ; ++i)
    {
        cur := list[i];
        if ( cur.isListed )
        {
	    /* presume there is only one such object */
	    prefix_count := 1;
	    
	    /*
	     *   if this is one of more than one equivalent items, list
	     *   it only if it's the first one, and show the number of
	     *   such items along with the first one 
	     */
	    if (cur.isEquivalent)
	    {
		local before, after;
		local j;
		local sc;

		sc := firstsc(cur);
		for (before := after := 0, j := 1 ; j <= tot ; ++j)
		{
		    if (isIndistinguishable(cur, list[j]))
		    {
			if (j < i)
			    ++before;
			else
			    ++after;
		    }
		}

		/*
		 *   if there are multiple such objects, and this is the
		 *   first such object, list it with the count prefixed;
		 *   if there are multiple and this isn't the first one,
		 *   skip it; otherwise, go on as normal 
		 */
		if (before = 0)
		    prefix_count := after;
		else
		    continue;
	    }

            if ( count > 0 )
            {
                if ( count+1 < disptot )
                    ", ";
                else if (count = 1)
                    " and ";
                else
                    ", and ";
            }

	    /* list the object, along with the number of such items */
	    if (prefix_count = 1)
		cur.adesc;
	    else
	    {
		sayPrefixCount(prefix_count); " ";
		cur.pluraldesc;
	    }

	    /* show any additional information about the item */
            if ( cur.isworn ) " (being worn)";
#ifdef BITTY_COMBAT
	    if (cur = Me.ready) " (being wielded)";
#endif
            if ( cur.islamp and cur.islit ) " (providing light)";
            count := count + 1;
        }
    }
}

/*
 *  listcontcont: function( obj )
 *
 *  This function lists the contents of the contents of an object.
 *  It displays full sentences, so no introductory or closing text
 *  is required.  Any item in the contents list of the object
 *  obj whose contentsVisible property is true has
 *  its contents listed.  An Object whose isqcontainer or
 *  isqsurface property is true will not have its
 *  contents listed.
 */

//LCNCN
listcontcont: function( obj )
{
    local list, i, tot;
    list := obj.contents;
    tot := length( list );
    i := 1;
    while ( i <= tot )
    {
        showcontcont( list[i] );
	i := i + 1;
    }
}

/*
 *  listfixedcontcont: function( obj )
 *
 *  List the contents of the contents of any fixeditem objects
 *  in the contents list of the object obj.  This routine
 *  makes sure that all objects that can be taken are listed somewhere
 *  in a room's description.  This routine recurses down the contents
 *  tree, following each branch until either something has been listed
 *  or the branch ends without anything being listable.  This routine
 *  displays a complete sentence, so no introductory or closing text
 *  is needed.
 */

//LFXCN
listfixedcontcont: function( obj )
{
    local list, i, tot, thisobj;

    list := obj.contents;
    tot := length( list );
    i := 1;
    while ( i <= tot )
    {
        thisobj := list[i];
        if ( thisobj.isfixed and thisobj.contentsVisible and
	  not thisobj.isqcontainer )
            showcontcont( thisobj );
	i := i + 1;
    }
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  This default version just says "I beg your pardon?"
 */

//PARDN
pardon: function
{
    "I beg your pardon? ";
}

/*
 *   preinit() is called after compiling the game, before it is written
 *   to the binary game file.  It performs all the initialization that can
 *   be done statically before storing the game in the file, which speeds
 *   loading the game, since all this work has been done ahead of time.
 *
 *   This routine puts all lamp objects (those objects with islamp = true) into
 *   the list global.lamplist.  This list is consulted when determining whether
 *   a dark room contains any light sources.
 */

//PRENT
preinit: function
{
    local o;
    
    global.lamplist := [];
    o := firstobj();
    while( o <> nil )
    {
        if ( o.islamp ) global.lamplist := global.lamplist + o;
        o := nextobj( o );
    }
    initSearch();
}

/*
 *  reachableList: function
 *
 *  Returns a list of all the objects reachable from a given object.
 *  That is, if the object is open or is not an openable, it returns the
 *  contents of the object, plus the reachableList result of each object;
 *  if the object is closed, it returns an empty list.
 */

//RCHLI
reachableList: function(obj)
{
    local ret := [];
    local i, lst, len;
    
    if (not isclass(obj, openable)
	or (isclass(obj, openable) and obj.isopen))
    {
	lst := obj.contents;
	len := length(lst);
	ret += lst;
	for (i := 1 ; i <= len ; ++i)
	    ret += reachableList(lst[i]);
    }

    return(ret);
}

/*
 *  sayPrefixCount: function( cnt )
 *
 *  This function displays a count (suitable for use in listcont when
 *  showing the number of equivalent items.  We display the count spelled out
 *  if it's a small number, otherwise we just display the digits of the
 *  number.
 */

//SPFXC
sayPrefixCount: function(cnt)
{
    if (cnt <= 20)
	say(['one' 'two' 'three' 'four' 'five'
	     'six' 'seven' 'eight' 'nine' 'ten'
	     'eleven' 'twelve' 'thirteen' 'fourteen' 'fifteen'
	     'sixteen' 'seventeen' 'eighteen' 'nineteen' 'twenty'][cnt]);
    else
	say(cnt);
}

/*
 *   showcontcont:  list the contents of the object, plus the contents of
 *   an fixeditem's contained by the object.  A complete sentence is shown.
 *   This is an internal routine used by listcontcont and listfixedcontcont.
 */

//SCNCN
showcontcont: function( obj )
{
    if (itemcnt( obj.contents ))
    {
        if ( obj.issurface and not obj.isqsurface )
        {
            "Sitting on "; obj.thedesc;" is "; listcont( obj );
            ". ";
        }
        else if ( obj.contentsVisible and not obj.isqcontainer )
        {
            caps();
            obj.thedesc; " seems to contain ";
            listcont( obj );
            ". ";
        }
    }
    if ( obj.contentsVisible and not obj.isqcontainer )
        listfixedcontcont( obj );
}

/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */

//TRMN8
terminate: function
{
}

/*
 *  turncount: function( parm )
 *
 *  This function can be used as a daemon (normally set up in the init
 *  function) to update the turn counter after each turn.  This routine
 *  increments global.turnsofar, and then calls scorestatus to
 *  update the status line with the new turn count.
 */

//TRNCT
turncount: function( parm )
{
    incturn();
    global.turnsofar += 1;
    scoreStatus();
}

/*
 *  userinit: function
 *	Please see init() for a description of this function.
 */

//USRNT
userinit: function(parm)
{
    version.sdesc;
    Me.location := global.start;          // move player to initial location
    global.start.lookAround( true );      // show player where he is
    global.start.isseen := true;          // note that we've seen the room
}
;

/*
 *  visibleList: function
 *
 *  This function is similar to reachableList, but returns the
 *  list of objects visible within a given object.
 */

//VLIST
visibleList: function(obj)
{
    local ret := [];
    local i, lst, len;
    
    if (not isclass(obj, openable)
	or (isclass(obj, openable) and obj.isopen)
	or obj.contentsVisible)
    {
	lst := obj.contents;
	len := length(lst);
	ret += lst;
	for (i := 1 ; i <= len ; ++i)
	    ret += visibleList(lst[i]);
    }

    return(ret);
}

