/*  Copyright (c) 1990, 1999 by Michael J. Roberts.  All Rights Reserved. */
/*
    Ditch Day Drifter
    Interactive Fiction by Michael J. Roberts.
    
    Developed with TADS: The Text Adventure Development System.
    
    This game is a sample TADS source file.  It demonstrates many of the
    features of TADS.  The TADS language is fully documented in the TADS
    Author's Manual, which is freely available in electronic format;
    you can probably get a copy of the Author's Manual from the same
    place you obtained this file.
    
    Please read LICENSE.DOC for information about using and copying this
    file, and about registering your copy of TADS.
*/

/*
 *   First, #include the Alt library.
 */

#include <stdif.h>
 
#pragma C+

modify story
    startingLocation = room3
    title = "Ditch Day Drifter"
    headline = "Interactive Fiction by Michael J.\ Roberts
        \bRelease 1.0
        \nCopyright (c) 1990 by Michael J.\ Roberts. All Rights Reserved.
        \nDeveloped with TADS: The Text Adventure Development System. "
    introduction = "\tYou wake up to the sound of voices in the hall.
        You are confused for a moment; it's only 8 AM, far too early 
        for anyone to be getting up. Then, it dawns on you: it's ditch 
        day here at the fictitious California Institute of Technology 
        in the mythical city of Pasadena, California. \(Ditch Day\), 
        that strange tradition wherein seniors bar their doors with
        various devices and underclassmen attempt to defeat these 
        devices (for no other apparent reason than that the devices 
        are there), has arrived.\b"
    initDaemon = {
        notify(queue, &processQueue, 0);    
        setdaemon(turnCount, nil);                 // start the turn counter daemon
        setdaemon(sleepDaemon, nil);                      // start the sleep daemon
        setdaemon(eatDaemon, nil);                       // start the hunger daemon
        gameClock.setClock('8:00 am', 1, 2000, 5, 01);        
    }
    /*
     *   The terminate() function is called just before the game ends.  We
     *   just display a good-bye message.
     */
    terminate = {
        "\bThanks for participating in \(Ditch Day!\)\n";
    }
;

modify global
    maxScore = 80                                     // maximum possible score
    sleepTime = 600     // interval between sleeping times (longest time awake)
    eatTime = 250         // interval between meals (longest time without food)
;

/* 
 *   Likewise, provide a parseErrorParam function that doesn't do
 *   anything.  This function could customize the messages that the parser
 *   generates when parser errors occur.  By returning nil, we simply let
 *   the parser use the default messages.  (We'd achieve the same effect
 *   by simply omitting this function entirely.)  
 */
parseErrorParam: function(num, str, ...) {
    return nil;
}


/*
 *   The die() function is called when the player dies.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).
 */
replace die: function {
    "\b*** You have died ***\b";
    scoreRank();

    /*
     *   Check if the player (Me) purchased insurance.  If so, issue
     *   an appropriate message.
     */
    if (parserGetMe().isInsured)
        "\bI'm sure that Lloyd is even now paying out a big lump sum for
        your early demise. However, that is of little use to you now. ";

    /*
     *   Tell the user the options available, and then ask for some
     *   input.  Keep asking until we get something we recognize.
     */
    "\bYou may restore a saved game, start over, undo the
        last move, or quit. ";
    while ( 1 ) {
        local resp;

        "\nPlease enter RESTORE, RESTART, UNDO, or QUIT: >";
        resp = upper(inputLine());     /* get input, convert to uppercase */
        if (resp == 'RESTORE') {
            resp = askfile( 'File to restore' );       /* find filename */
            if (resp == nil) 
                "Restore failed. ";
            else if (restore(resp)) 
                "Restore failed. ";
            else {
                /*
                 *   We've successfully restored a game.  Reset the status
                 *   line's score/turn counter, and resume play.
                 */
                scoreStatus(global.score, global.turnSoFar);
                abort;
            }
        } else if (resp == 'RESTART') {
            /*
             *   We're going to start over.  Reset the status line's
             *   score/turn counter and start from the beginning.
             */
            setscore( 0, 0 );
            restart();
        } else if (resp == 'UNDO') {
            if (undo()) {
                "(Undoing one command)\b";
                Me.location.lookAround(true);
                setscore(global.score, global.turnsofar);
                abort;
            } else
                "Sorry, no undo information is available.";
        } else if (resp == 'QUIT') {
            /*
             *   We're quitting the game.  Do any final activity necessary,
             *   and exit.
             */
            story.terminate;
            quitGame();
            abort;
        }
    }
}

/*
 *   The scoreRank() function displays how well the player is doing.
 *   In addition to displaying the numerical score and number of turns
 *   played so far, we'll classify the score with a rank such as "sophomore."
 *
 *   Note that "global.maxscore" defines the maximum number of points
 *   possible in the game.
 */
replace scoreRank: function {
    local s;
    
    s = global.score;
    
    "In a total of "; say(global.turnSoFar );
    " turns, you have achieved a score of ";
    say( s ); " points out of a possible "; say(global.maxScore);
    ", which gives you a rank of ";
    
    if ( s < 10 ) "high-school hopeful";
    else if ( s < 25 ) "freshman";
    else if ( s < 40 ) "sophomore";
    else if ( s < 60 ) "junior";
    else if ( s < global.maxScore ) "senior";
    else "graduate";
    
    ". ";
}



room3: Room
    sDesc = "Room 3"
    lDesc = "This is your room. You live a fairly austere life, being a 
        poor college student. The only notable features are the bed 
        (unmade, of course) and a small wooden desk.  An exit is west. "
    west = alley1                                   // room that lies west
    out = alley1                                    // the exit
;

/*
 *   The desk is a surface, which means you can put stuff on top of it.
 *   Note that the drawer is a separate object.
 */
room3Desk: Fixture, Surface
    location = room3
    noun = 'desk'
    adjective = 'small' 'wooden' 'wood'
    sDesc = "small wooden desk"
    lDesc = {
        "It's the small desk that comes with all of the rooms in the house.
        The desktop is pitifully small, especially considering that you often
        need to have several physics texts and tables of integrals open
        simultaneously. The desk has a small drawer (";
        if (room3Drawer.isOpen) 
            "open"; 
        else "closed";
        "). ";
    }
    /*
     *   For convenience, redirect any open/close activity to the
     *   drawer.  Since the desk can't be opened and closed, we can
     *   reasonably expect that the player is really referring to the
     *   drawer if he tries to open or close the desk.
     */
    verDoOpen(actor) = { room3Drawer.verDoOpen(actor); }
    doOpen(actor) = { room3Drawer.doOpen(actor); }
    verDoClose(actor) = { room3Drawer.verDoClose(actor); }
    doClose(actor) = { room3Drawer.doClose(actor); }
;

/*
 *   A container can contain stuff.  An openable is a special type of container
 *   that can be opened and closed.  Though it's technically part of the desk,
 *   it's a Fixture (==> it can't be taken), so we can just as well make it
 *   part of room3; note that this is in fact necessary, since if it were
 *   located in the desk, it would appear to be ON the desk (since the desk is
 *   a surface).
 */
room3Drawer: Fixture, OpenableContainer
    location = room3
    noun = 'drawer'
    isOpen = nil
    sDesc = "drawer"
;

/*
 *   A qcontainer is a "quiet container."  It acts like a container in all
 *   ways, except that its contents aren't displayed in a room's message.
 *   We want the player to have to think to examine the wasteBasket more
 *   carefully to find out what's in it, so we'll make it a qcontainer.
 *   Which is fairly natural:  when looking around a room, you don't usually
 *   notice what's in a waste basket, even though you may notice the waste
 *   basket itself.
 *
 *   The sDesc is just the name of the object, as displayed by the game
 *   (as in "You see a wasteBasket here").  The noun and adjective lists
 *   specify how the user can refer to the object; only enough need be
 *   specified by the user to uniquely identify the object for the purpose
 *   of the command.  Hence, you can specify as many words as you want without
 *   adding any burden to the user---the more words the better.  The location
 *   specifies the object (a room in this case) where the object is
 *   to be found.
 */
wasteBasket: Qcontainer
    location = room3
    noun = 'basket' 'wasteBasket'
    adjective = 'waste'
    sDesc = "waste basket"
    moveInto(obj) = {
        /*
         *   If this object is ever removed from its original location,
         *   turn off the "quiet" attribute.
         */
        self.isQcontainer = nil;
        pass moveInto;
    }
;

/*
 *   A BedItem is something you can lie down on.
 */
bed: BedItem
    location = room3
    noun = 'bed'
    sDesc = "bed"
    lDesc = "It's a perfectly ordinary bed. It's particularly ordinary
        (for around here, anyway) in that it hasn't been made in a very
        long time. "
    
    /*
     *   verDoLookunder is called when the player types "look under bed"
     *   to verify that this object can be used as a direct object (Do) for
     *   that verb (Lookunder).  If no message is printed, it means that
     *   it can be used.
     */
    verDoLookunder(actor) = {}
    
    /*
     *   ...and then, if verification succeeded, doLookunder is called to
     *   actually apply the verb (Lookunder) to this direct object (do).
     */
    doLookunder(actor) = {
        if (dollar.isFound)
            "You don't find anything of interest. ";
        else {
            dollar.isFound = true;
            "You find a dollar bill! You pocket the bill.
            (Okay, so it's an obvious adventure game puzzle, but I'm sure
            you would have been disappointed if nothing had been there.) ";
            dollar.moveInto(actor);
        }
    }
    
    /*
     *   Verification and action for the "Make" verb.
     */
    verDoMake(actor) = {}
    doMake(actor) = {
        "It was a nice thought, but you suddenly realize that you never
        learned how. ";
    }
;

/*
 *   An "item" is the most basic class of objects that a user can carry
 *   around.  An item has no special properties.  The "isCrawlable"
 *   attribute that we're setting here is used in the north-south crawl
 *   in the steam tunnels (later in the game); setting it to "true"
 *   means that the player can carry this object through the crawl.
 */
dollar: Item
    noun = 'bill'
    adjective = 'dollar' 'one' '1'
    sDesc = "one dollar bill"
    isCrawlable = true
;

room4: Room
    sDesc = "Room 4"
    lDesc = "This is room 4, where the weird senior across the hall
        lives. An exit is to the east, and a strange passage leads down. "
    east = alley1
    out = alley1
    down = {
        /*
         *   This is a bit of code attached to a direction.  We can do
         *   anything we want in code like this, so long as we return a
         *   room (or nil) when we're done.  In this case, we just want
         *   to display a message when the player travels this way.
         */
        "The passage takes you down a winding stairway to a previously
        unseen entrance to...\b";
        return shipRoom;
    }
;

/*
 *   A ReadableItem is an object that can be read, such as a note, a sign, a
 *   book, or a memo.  By default, reading the object displays its lDesc.
 *   However, if a separate readDesc is specified, "look at note" and
 *   "read note" could display different messages.
 */
winNote: ReadableItem
    location = room4
    noun = 'note'
    isCrawlable = true
    sDesc = "note"
    lDesc = "Congratulations! You've broken the stack! Please take this
        fine WarpMaster 2000(tm) warp motivator, with my compliments. I'll
        see you in \"Deep Space Drifter\"! "
;

shipRoom: Room
    sDesc = "Spaceship Room"
    lDesc = "This is a very large cave dominated by a tall spaceship.
        High above, a vertical tunnel leads upward; it is evidently a 
        launch tube for the spaceship. The exit is north. "
    north = chuteRoom
    in = shipInterior
    up = {
        "The tunnel is far too high above to climb. ";
        return nil;
    }
;

/*
 *   The receptacle is both a Fixture (something that can't be taken and
 *   carried around) and a container, so both superclasses are specified.
 */
shipRecept: Fixture, Container
    location = shipRoom
    noun = 'socket'
    sDesc = "socket"
    ioPutIn(actor, dobj) = {
        /*
         *   We only want to allow the warp motivator to be put in here.
         *   Check any object going in and disallow it if it's anything else.
         */
        if (dobj == motivator) {
            "It fits snugly. ";
            dobj.moveInto(self);
        } else "It doesn't fit. ";
    }
;

spaceship: FixedItem
    location = shipRoom
    noun = 'spaceship' 'ship'
    adjective = 'space'
    sDesc = "spaceship"
    lDesc = {
        "The spaceship is a tall gray metal cylinder with a pointed nosecone
        high above, and three large fins at the bottom. A large socket ";
        if (motivator.location == shipRecept)
            "on the side contains a warp motivator. ";
        else
            "is on the side (the socket is currently empty). The socket
            is labelled \"insert warp motivator here.\" ";
        "You can enter the spaceship through an open door. ";
    }
    verDoEnter(actor) = {}
    doEnter(actor) = { self.doBoard(actor); }
    verDoBoard(actor) = {}
    doBoard(actor) = {
        actor.travelTo(shipInterior);
    }
;

shipInterior: Room
    sDesc = "Spaceship"
    lDesc = "You are in the cockpit of the spaceship. The control panel is
        quite simple; the only feature that interests you at the moment is
        a button labelled \"Launch.\" "
    out = shipRoom
;

/*
 *   This is a fake "ship" that the player can refer to while inside the
 *   spaceship.  This allows commands such as "look at ship" and "get out"
 *   to work properly while within the ship.
 */
fakeShip: FixedItem
    location = shipInterior
    noun = 'ship' 'spaceship'
    adjective = 'space'
    sDesc = "spaceship"
    lDesc = {shipInterior.lDesc;}
    verDoUnboard(actor) = {}
    doUnboard(actor) = {
        actor.travelTo(shipInterior.out);
    }
;

/*
 *   A ButtonItem can be pushed.  It's automatically a FixedItem, and
 *   always defines the noun 'button'.  The doPush method specifies what
 *   happens when the button is pushed.
 */
launchButton: ButtonItem
    location = shipInterior
    adjective = 'launch'
    sDesc = "launch button"
    doPush(actor) = {
        if (motivator.location == shipRecept) {
            incScore(10);
            "The ship's engines start to come to life. \"Launch sequence
            engaged,\" the mechanical computer voice announces. ";
            if (lloyd.location == parserGetMe().location)
                "\n\tLloyd heads for the door. \"Sorry,\" he says, \"I can't
                go with you. Your policy specifically excludes coverage
                during missions in deep space, and, frankly, it's too risky
                for my tastes. Besides, I don't appear in 'Deep Space
                Drifter.'\" Lloyd waves goodbye, and rolls out of the
                spaceship.\n\t";
            "The hatch closes automatically, sealing you into the
            space vessel. The engines become louder and louder.
            The computer voice announces, \"Launch
            in five... four... three... two... one... liftoff!\"
            The engines blast the ship into orbit.
            \n\tYou realize that the time has come to set course for
            \"Deep Space Drifter,\" another fine TADS adventure from
            High Energy Software.\b";
            
            scoreRank();
            story.terminate;
            quitGame();
            abort;
        } else {
            "The ship's computer voice announces from a hidden speaker,
            \"Error: no warp motivator installed.\" ";
        }
    }
;

motivator: Treasure
    location = room4
    noun = 'motivator' 'warpmaster'
    adjective = 'warp'
    sDesc = "warp motivator"
    takeValue = 20
    lDesc = "It's a WarpMaster 2000(tm), the top-of-the-line model. "
;

foodThing: FoodItem
    location = room3Drawer
    noun = 'food'
    sDesc = "food"
    aDesc = "some food"
    lDesc = "It's a non-descript food item of the type the Food Service
        typically prepares. "
;

/*
 *   An openable is a container, with the additional property that it can
 *   be opened and closed.  To simplify things, we don't have a separate
 *   cap; use of the cap is implied in the "open" and "close" commands.
 */
bottle: OpenableContainer
    location = wasteBasket
    noun = 'bottle'
    adjective = 'two-liter' 'plastic' 'two' 'liter'
    isOpen = nil
    sDesc = "two-liter plastic bottle"
    lDesc = {
        "The bottle is ";
        if (self.isFull) 
            "full of liquid nitrogen. ";
        else "empty. ";

        "It's "; 
        if (self.isOpen) 
            "open. "; 
        else "closed. ";
        
        if (funnel.location == self)
            "There's a funnel in the bottle's mouth. ";
    }
    ioPutIn(actor, dobj) = {
        if (!self.isOpen) {
            "It might help to open the bottle first. ";
        } else if (dobj == ln2) {
            if (funnel.location == self) {
                "You manage to get some liquid nitrogen into the bottle. ";
                bottle.isFull = true;
            } else {
                "You can't manage to get any liquid nitrogen into the
                 tiny opening. ";
            }
        } else if (dobj == funnel) {
            "A perfect fit! ";
            funnel.moveInto(self);
        }
        else "That won't fit in the bottle. ";
    }
    verIoPourIn(actor) = {}
    ioPourIn(actor, dobj) = {self.ioPutIn(actor, dobj);}
    doClose(actor) = {
        if (funnel.location == self)
            "You'll have to take the funnel out first. ";
        else if (self.isFull) {
            "It takes some effort to close the bottle, since the rapidly
            evaporating nitrogen occupies much more volume as a gas than
            as a liquid. However, you manage to close it. ";
            notify(self, #explodeWarning, 3);
            self.isOpen = nil;
        } else {
            "Okay, it's closed. ";
            self.isOpen = nil;
        }
    }
    
    /*
     *   explodeWarning is sent to the bottle as a "notification," which is
     *   a message that's scheduled to occur some number of turns in the
     *   future.  In this case, closing the bottle while it has liquid
     *   nitrogen inside will set the explodeWarning notification.
     */
    explodeWarning = {
        if (!self.isOpen) {
            if (self.isIn(parserGetMe().location)) {
                "\bThe bottle is starting to make lots of noise, as though
                the plastic were being stretched to its limit. ";
            }
            notify(self, #explode, 3);
        }
    }
    
    /*
     *   explode is set as a notification by explodeWarning.  This routine
     *   actually causes the explosion.  Since the bottle explodes, we will
     *   remove it from the game (by moving it into "nil").  If the bottle
     *   is in the right place, we'll also do some useful things.
     */
    explode = {
        if (!self.isOpen) {
            "\b";
            if (self.location == bankSafe) {
                if (parserGetMe().location == bankVault) {
                    "There is a terrible explosion from within the safe.
                     The door blasts open with a clang and a huge cloud of
                     water vapor. ";
                     
                    if (lloyd.location == bankVault)
                        "\n\tLloyd throws himself between you and the
                        vault. \"Please be careful!\" he admonishes.
                        \"The payment for Accidental Death due to Explosion
                        is enormous!\" ";
                } else
                    "You hear a distant, muffled explosion. ";
                
                bankSafe.isOpen = true;
                bankSafe.isBlasted = true;
            } else if (self.isIn(parserGetMe().location)) {
                "The bottle explodes with a deafening boom and a huge
                cloud of water vapor. As with most explosions, standing
                in such close proximity was not advisable; it was, in
                fact, fatal. ";
                die();
                abort;
            } else {
                "You hear a distant explosion. ";
            }
            
            self.moveInto(nil);
        }
    }
;

/*
 *   We're defining our own "class" of objects here.  Up to now, we've been
 *   using classes defined in "adv.t", which you will recall we included at
 *   the top of the file.  This class has some useful properties.  For one,
 *   it has an attribute (istreasure) that tells us that the object is
 *   indeed a treasure (used in the slot in room 4's door to ensure that
 *   an object can be put in the slot).  In addition, it supplements the
 *   doTake routine:  when a treasure is taken for the first time, we'll
 *   add the object's "takeValue" to the overall score.
 */
class Treasure: Item
    isTreasure = true
    takeValue = 5       // default point value when object is taken
    putValue = 5        // default point value when object is put in slot
    doTake(actor) = {
        if (!self.hasScored) {
            incScore(self.takeValue); // add our "takeValue" to the score
            self.hasScored = true;     // note that we have scored
        }
        pass doTake;        // continue with the normal doTake from "item"
    }
;

alley1: Room
    sDesc = "Alley One"
    lDesc = {
        "You are in the eternal twilight of Alley One, one of the twisty
        little passages (all different) making up the student house in
        which you live. Your room (room 3) is to the east. To the west
        is a door (";
        alley1Door.isOpen ? "open" : "closed";
        "), affixed to which is a large sign. An exit is south, and the
        hallway continues north. ";
    }
    east = room3
    north = alley1North
    west = {
        /*
         *   Sometimes, we'll want to run some code when the player walks
         *   in a particular direction rather than just go to a new room.
         *   This is how.  Returning "nil" means that the player can't go
         *   this way.
         */
        if (alley1Door.isOpen) {
            return room4;
        } else {
            "The door is closed and locked. You might read the
            sign on the door for more information. ";
            return nil;
        }
    }
    south = breezeway
    out = breezeway
;

alley1North: Room
    sDesc = "Alley One"
    lDesc = "You are at the north end of alley 1.  A small room is
        to the west. "
    south = alley1
    west = alley1Comp
;

alley1Comp: Room
    sDesc = "Computer Room"
    lDesc =  {
        "You are in a small computer room. Not surprisingly, the room
         contains a personal computer. The exit is east. ";
        if (!self.isSeen) notify(compStudents, #converse, 0);
    }
    east = alley1North
;

alley1PC: FixedItem
    location = alley1Comp
    noun = 'computer'
    adjective = 'personal'
    sDesc = "personal computer"
    lDesc = "The computer is in use by a couple of your fellow undergraduates.
        Closer inspection of the screen shows that they seem to be playing a
        text adventure. You've never really understood the appeal of those games
        yourself, but you quickly surmise that the game is part of one of the
        seniors' stacks. "
    verIoTypeOn(actor) = {}
    ioTypeOn(actor, dobj) = {
        "The computer is already in use. Common courtesy demands that you
        wait your turn. ";
    }
    verDoTurnoff(actor) = {}
    doTurnoff(actor) = {
        "The students won't let you, since they're busy using the computer. ";
    }
;

compStudents: Actor
    location = alley1Comp
    noun = 'students' 'undergraduates'
    state = 0
    sDesc = "students"
    aDesc = "a couple of students"
    lDesc = "The students are busy using the computer. "
    actorDesc = "A couple of your fellow undergraduates are here, using
        the computer. They seem quite absorbed in what they're doing. "
    actorAction( v, d, p, i ) = {
        "They're too wrapped up in what they're doing. ";
        exit;
    }
    doAskAbout(actor, iobj) = {
        "They're too busy to answer. ";
    }
    converse = {
        if (parserGetMe().location == self.location) {
            "\b";
            switch(self.state) {
                case 0:
                    "\"Where are we going to find the dollar bill?\" one
                    of the students asks the other. They sit back and stare
                    at the screen, lost in thought. ";
                    break;
                case 1:
                    "\"Hey!\" says one of the students. \"Did you look under
                    the bed?\" The other student shakes his head. \"No way,
                    that would be a stupid puzzle!\" ";
                    break;
                case 2:
                    "One of the students using the computer types a long
                    string of commands, and finally types \"look under bed.\"
                    \"Wow! The dollar bill actually was under the bed! How
                    lame!\" ";
                    break;
                default:
                    "The students continue to play with the computer. ";
            }
            ++self.state;
        }
    }
;

alley1Door: LockableDoorway
    location = alley1
    lDesc = {
        if (self.isOpen) 
            "It's open. ";
        else
            "The door is closed and locked. There is a slot in the door,
            and above that, a large sign is affixed to the door. ";
    }
;

alley1Sign: Fixture, ReadableItem
    location = alley1
    noun = 'sign'
    sDesc = "sign"
    lDesc = "The sign says:
        \b\t\tWelcome to Ditch Day!
        \bThis stack is a treasure hunt. Gather all of the treasures, and you
        break the stack.
        To satisfy the requirements of this stack, you must find the
        items listed below, and deposit them in the slot in the door. When
        all items have been put in the slot, the stack will be solved and
        the door will open automatically. The items to find are:
        \b\tThe Great Seal of the Omega
        \n\tMr.\ Happy Gear
        \n\tA Million Random Digits
        \n\tA DarbCard
        \bThese items are hidden amongst the expanses of the Great
        Undergraduate Excavation project. Happy hunting!
        \bFor first-time participants, please note that
        this is a \"finesse stack.\" You are not permitted to attempt to break
        the stack by brute force. Instead, you must follow the rules above. "
;

alley1Slot: Fixture, Container
    location = alley1
    noun = 'slot'
    sDesc = "slot"
    count = 0
    ioPutIn(actor, dobj) = {
        if (dobj.isTreasure) {
            "\^<<dobj.theDesc>> disappears into the slot. ";
            dobj.moveInto(nil);
            incScore(dobj.putValue);
            ++self.count;
            if (self.count == 4) {
                "As the treasure disappears into the slot, you hear a
                klaxon from the other side of the door. An elaborate
                series of clicks and clanks follows, then the door swings
                open. ";
                alley1Door.isLocked = nil;
                alley1Door.isOpen = true;
            }
        } else {
            "The slot will only accept items on the treasure list. ";
        }
    }
;

breezeway: Room
    sDesc = "Breezeway"
    lDesc = "You are in a short passage that connects a courtyard,
        to the east, to the outside of the building, which lies to the
        west. A hallway leads north. "
    north = alley1
    east = courtyard
    west = orangeWalk1
;

courtyard: Room
    sDesc = "Courtyard"
    lDesc = "You are in a large outdoor courtyard.
        An arched passage is to the west.
        A passage leads east, and a stairway leads down. "
    east = lounge
    down = hall1
    west = breezeway
;

lounge: Room
    sDesc = "Lounge"
    lDesc = "You are in the lounge. A passage leads west, and a dining
        room lies to the north. "
    north = diningRoom
    west = courtyard
    out = courtyard
;

diningRoom: Room
    sDesc = "Dining Room"
    lDesc = "You are in the dining room. There is a wooden table in
        the center of the room. The lounge lies to the south,
        and a passage to the east leads into the kitchen. "
    east = kitchen
    south = lounge
;

diningTable: Fixture, Surface
    location = diningRoom
    noun = 'table'
    adjective = 'wooden'
    sDesc = "wooden table"
;

fishfood: FoodItem
    location = diningTable
    noun = 'module'
    adjective = 'fish' 'protein'
    sDesc = "fish protein module"
    lDesc = "It's a small pyramid-shaped white object, which is widely
        considered to consist primarily of fish protein. The food service
        typically resorts to such unappetizing fare toward the end of the
        year. "
;

kitchen: Room
    sDesc = "Kitchen"
    lDesc = "You are in the kitchen.  A ToxiCola(tm) machine is here.
        A passage leads into the dining room to the west. "
    west = diningRoom
    out = diningRoom
;

toxicolaMachine: Fixture, Container
    location = kitchen
    noun = 'machine' 'compartment'
    adjective = 'toxicola'
    sDesc = "ToxiCola machine"
    lDesc = {
        "The machine dispenses ToxiCola, one of the big losers in the
         Cola Wars.  But, hey, it's cheap, so the food service installed it.
         The machine consists of a compartment large enough for a cup,
         and a button for dispensing ToxiCola into the cup. ";
         if (cup.location == self)
            "The compartment contains a cup. ";
    }
    ioPutIn(actor, dobj) = {
        if (dobj != cup)
            "That won't fit in the compartment. ";
        else pass ioPutIn;
    }
;

cup: Container
    location = diningTable
    noun = 'cup'
    adjective = 'coffee'
    isFull = nil
    sDesc = "coffee cup"
    lDesc = {
        if (self.isFull) 
            "It's full of a viscous brown fluid. ";
        else "It's empty. ";
    }
    ioPutIn(actor, dobj) = {
        if (dobj == ln2) 
            "The liquid nitrogen evaporates on contact. ";
        else "It won't fit in the cup. ";
    }
;

toxicola: FixedItem
    noun = 'toxicola' 'cola'
    sDesc = "toxicola"
    lDesc = "It's a thick brown fluid. It appears to be quite flat. "
    doTake(actor) = {
        "You'll have to leave it in the cup. ";
    }
    verDoDrink(actor) = {}
    doDrink(actor) = {
        "You drink the ToxiCola, despite your better judgment. It
        initially sparks a sugar and caffeine rush, but that rapidly
        fades, to be replaced by a strange dull throbbing. You enter
        a semi-conscious state for several hours. It finally passes,
        but you're not sure if it's been hours, weeks, or years. ";
        cup.isFull = nil;
        self.moveInto(nil);
    }
;

toxicolaButton: ButtonItem
    location = kitchen
    sDesc = "button"
    doPush(actor) = {
        if (cup.location == toxicolaMachine) {
            if (cup.isFull)
                "ToxiCola spills over the already full cup, and drains
                away. ";
            else {
                "The horrible brown viscous fluid you have come to
                know as ToxiCola fills the cup. ";
                cup.isFull = true;
                toxicola.moveInto(cup);
            }
        } else {
            "Horrible viscous brown fluid spills into the empty
            compartment, and drains away. ";
        }
    }
;

orangeWalk1: Room
    sDesc = "Orange Walk"
    lDesc = "You are on a walkway lined with orange trees. The walkway
        continues to the north, and an arched passage leads into a building
        to the east. "
    east = breezeway
    north = orangeWalk2
;

/*
 *   We want a "class" of objects for the orange trees lining the
 *   orange walk.  They don't do anything, so they're "decoration."
 */
class OrangeTree: Decoration
    noun = 'tree' 'trees'
    adjective = 'orange'
    sDesc = "orange trees"
    aDesc = "an orange tree"
    lDesc = "The orange trees are perfectly ordinary. "
    verDoClimb(actor) = {}
    doClimb(actor) = {
        "You climb into one of the orange trees, and quickly find the
        view from the few feet higher to be highly uninteresting. You
        soon climb back down. ";
    }
;

orangeTree1: OrangeTree
    location = orangeWalk1
;

orangeWalk2: Room
    sDesc = "Orange Walk"
    lDesc = "You are on a walkway lined with orange trees. The walkway
        continues to the south, and leads into a large grassy square to the
        north. "
    north = quad
    south = orangeWalk1
;

orangeTree2: OrangeTree
    location = orangeWalk2
;

quad: Room
    sDesc = "Quad"
    lDesc = "You are on the quad, a large grassy square in the
        center of campus.  The bookstore lies to the northwest; the
        health center lies to the northeast; Buildings and Grounds
        lies to the north; and walkways lead west and south.
        \n\tSome students dressed in radiation suits are staging a bogus
        toxic leak, undoubtedly to fulfill the requirements of one of the
        stacks.  They are wandering around, looking very busy.  Many reporters
        are standing at a safe distance, looking terrified.
        The supposed clean-up crew is pouring lots of liquid nitrogen onto
        the ground, resulting in huge clouds of water vapor. "
    south = orangeWalk2
    nw = bookstore
    west = walkway
    ne = healthCenter
    north = BandG
;

BandG: Room
    sDesc = "B&G"
    lDesc = "You are in the Buildings and Grounds office. The exit is to
        the south. "
    south = quad
    out = quad
;

bngmemo: ReadableItem
    location = BandG
    noun = 'scrap' 'paper'
    isCrawlable = true
    sDesc = "scrap of paper"
    lDesc = "Most of the paper has been torn away; the part that remains
        seems to have a list of numerical codes of some kind on it:
        \b\t293 -- north tunnel lighting
        \n\t322 -- station 2 lighting
        \n\t612 -- behavior lab
        \bThe rest is missing. "
;

healthCenter: Room
    sDesc = "Health Center"
    lDesc = "You are in the health center. Like the rest of campus, this
        place is deserted because of Ditch Day. The large desk would normally
        have a receptionist behind it, but today, no one is in sight. "
    sw = quad
    out = quad
;

healthDesk: Fixture, Surface
    location = healthCenter
    noun = 'desk'
    adjective = 'large'
    sDesc = "desk"
;

healthMemo: ReadableItem
    location = healthDesk
    noun = 'memo'
    adjective = 'health'
    plural = 'memos'
    isCrawlable = true
    sDesc = "health memo"
    lDesc = "From:\t\tDirector of the Health Center
        \nTo:\t\tAll Health Center Personnel
        \nSubject:\tToxiCola toxicity
        \bMany students have visited the Health Center recently, complaining
        that the ToxiCola that is served in the student houses has not been
        properly caffeinated.  The students are complaining of drowsiness,
        dizziness, and other symptoms that are normally associated with an
        insufficient intake of caffeine.
        \n\tUpon investigation, we have learned that the ToxiCola dispensers
        in the student houses have become contaminated with a substance that
        induces these effects. The substance has not yet been identified, but
        the concentration seems to be increasing. All Health Center personnel
        are urgently directed to advise students to avoid the ToxiCola if at
        all possible. Students should be informed that their student health
        insurance coverage will cover any purchases of other caffeinated
        beverages that they need for their studies. "
;

students: Decoration
    location = quad
    noun = 'student' 'students'
    sDesc = "students"
    aDesc = "a group of students"
    lDesc = "The students are making quite a good show of the simulated
        nuclear waste spill. They're all wearing white clean-room suits with
        official-looking \"Radiation Control District\" badges. They're
        scampering about purposefully, keeping the crowd of reporters back
        at a safe distance. "
;

presscorp: Decoration
    location = quad
    noun = 'reporter' 'reporters'
    sDesc = "reporters"
    aDesc = "a group of reporters"
;

flask: Container
    location = quad
    noun = 'flask'
    sDesc = "flask"
    lDesc = "The flask appears to have lots of liquid nitrogen in it;
        it's hard to tell just how much, since the opening is perpetually
        clouded over with thick plumes of water vapor. "
    ioPutIn(actor, dobj) = {
        "You know from experience that it wouldn't be a good idea to
        put "; dobj.theDesc; " into the liquid nitrogen, since the LN2 is
        really, really cold. ";
    }
;

ln2: Item
    location = flask
    noun = 'nitrogen' 'ln2'
    adjective = 'liquid'
    sDesc = "liquid nitrogen"
    aDesc = "some liquid nitrogen"      // "a liquid nitrogen" doesn't compute
    lDesc = "You can't see much, thanks to the thick clouds of water
        vapor that inevitably form over such a cold substance. "
    doTake(actor) = {
        "You're better off leaving the liquid nitrogen in the flask. ";
    }
    verDoPour(actor) = {}
    verDoPourIn(actor, iobj) = {}
    doPour(actor) = {
        askForIobj(inPrep);
    }
;

walkway: Room
    sDesc = "Walkway"
    lDesc = "You are on a walkway.  A large grassy square lies to the east;
        buildings lie to the north and south. The walkway continues west. "
    east = quad
    north = behaviorLab
    south = security
    west = walkway2
;

walkway2: Room
    sDesc = "Walkway"
    lDesc = "You are on a walkway, which continues to the east. Buildings
        are to the north and south. "
    east = walkway
    south = explosiveLab
    north = biobuilding
;

biobuilding: Room
    sDesc = "Biology Building"
    lDesc = "You are in the biology building. The exit is south. "
    south = walkway2
    out = walkway2
;

bionotes: ReadableItem
    location = biobuilding
    noun = 'notebook'
    sDesc = "notebook"
    lDesc = "The notebook explains various lab techniques used in cloning
        organisms. Since the invention of the CloneMaster, which requires only
        a sample of genetic material from a subject (such as some blood, or a
        bit of skin, or the like) and the basic skills required to operate a
        household blender, most of the techniques are obsolete. Some of the data,
        however, are interesting. For example, the notebook outlines the procedure
        for reversing the sex of a clone; the introduction of chemicals identified
        herein only as Genetic Factor XQ3, Polymerase Blue, and Compound T99 at
        the start of the cloning process aparently does the trick. Most of the
        rest of the document is a discussion of the human immune system; the
        author comes to the conclusion that the human immune system, though a
        novel idea, is far too improbable to ever actually be implemented. "
;

explosiveLab: Room
    sDesc = "Explosive Lab"
    lDesc = "It's not that the lab itself is explosive (though it has blown
        up a couple times in the past); rather, they study explosives here.
        Unfortunately, all the good stuff has been removed by other Ditch Day
        participants who got up earlier than you did. The exit is north. "
    north = walkway2
    out = walkway2
;

ln2doc: ReadableItem
    location = explosiveLab
    noun = 'thesis'
    sDesc = "thesis"
    lDesc = "The thesis is about Thermal Expansion Devices. It explains about
        a new class of explosives that are made possible by low-temperature
        fluid technology (i.e., liquid nitrogen) and high-tension polymer
        containment vessels (i.e., plastic bottles). A great deal of jargon and
        complicated theories are presented, presumably to fool the faculty
        advisor into thinking the author actually did something useful with his
        research funds; after wading through the nonsense, you find that the
        paper is merely talking about putting liquid nitrogen into a plastic
        soft-drink bottle, closing the bottle, then letting nature take its
        course. Since the nitrogen will tend to evaporate at room temperature,
        but will have no place to go, the bottle will eventually explode.
        \bOn the cover page, you notice this important warning: \"Kids! Don't
        try this at home! Liquid nitrogen is extremely cold, and can cause severe
        injuries; it should only be handled by trained professionals. Never
        put liquid nitrogen in a closed container.\" "
;

bookstore: Room
    sDesc = "Bookstore"
    lDesc = "You are in the bookstore. The shelves are quite empty; no doubt
        everything has been bought up by seniors in attempts to build their
        stacks and underclassmen in attempts to break them. The exit is to the
        southeast. "
    out = {return self.se;}
    se = {
        /*
         *   We need to check that the battery has been paid for, if it's
         *   been taken.  If not, don't let 'em out.
         */
        if (battery.isIn(parserGetMe()) && !battery.isPaid) {
            "The clerk notices that you want to buy the battery. \"That'll be
            five dollars,\" she says, waiting patiently. ";
            return nil;
        } else return quad;
    }
;

battery: Item
    location = bookstore
    noun = 'battery'
    isPaid = nil
    sDesc = "battery"
    verDoPayfor(actor) = {
        if (actor.location != bookstore)
            "I don't see anyone to pay! ";
    }
    doPayfor(actor) = {
        clerk.doPay(actor);
    }
    verIoPayFor(actor) = {}
    ioPayFor(actor, dobj) = {
        if ( dobj != clerk ) {
            "You can't pay "; dobj.theDesc; " for that! ";
        }
        else clerk.doPay(actor);
    }
;

money: Item
    noun = 'bill' 'money'
    adjective = 'five' 'dollar' '5'
    isCrawlable = true
    sDesc = "five dollar bill"
;

clerk: Actor
    location = bookstore
    noun = 'clerk'
    sDesc = "Clerk"
    isHer = true
    lDesc = "The clerk is a kindly lady to whom you have paid many hundreds
        of dollars for books and other college necessities. "
    actorDesc = "A clerk is near the exit, prepared to ring up any purchases
        you might want to make (not that there's much left here to buy). "
    verIoGiveTo(actor) = {}
    ioGiveTo(actor, dobj) = {
        if (dobj == money || dobj == dollar) 
            self.doPay(actor);
        else if (dobj == darbCard) {
            "The clerk shakes her head. \"Sorry,\" she says, \"we only
            accept cash.\" ";
        } else if (dobj == cup && cup.isFull) {
            "\"I never drink that stuff,\" the clerk says, \"and neither
            should you! Do you have any idea what's in there? I probably
            don't know half as much chemistry as you, but I know better
            than to drink that.\" ";
        } else {
            "She doesn't appear interested. ";
        }
    }
    verDoPay(actor) = {}
    doPay(actor) = {
        if (!battery.isIn(parserGetMe())) {
            "The clerk looks confused. \"I don't see what you're
            paying for,\" she says. She then looks amused, realizing
            that the students here somtimes get a little ahead of
            themselves. ";
        } else if (battery.isPaid) {
            "The clerk looks confused and says, \"You've already paid!\"
            She then looks amused, realizing that the students here can
            be a bit absent-minded at times. ";
        } else if (!money.isIn(parserGetMe())) {
            if (dollar.isIn(parserGetMe())) {
                "The clerk reminds you that the battery is five dollars.
                All you have is one dollar. She looks amused; \"They can
                do calculus in their sleep but they can't add,\" she jokes. ";
            } else {
                "You unfortunately don't have anything with which to pay the
                clerk. ";
            }
        } else {
            "The clerk accepts your money and says \"Thank you, have a nice
            day.\"";
            battery.isPaid = true;
            money.moveInto(nil);
        }
    }
;

behaviorLab: Room
    sDesc = "Behavior Lab"
    lDesc = "You are in the behavior lab.  The exit is to the south,
        and a locked door is to the north. The door is labelled \"Maze -
        Experimental Subjects Only.\" In addition, a passage labelled
        \"Viewing Room\" leads east. "
    south = walkway
    out = walkway
    east = mazeView
    north = {
        "The door is closed and locked. Besides, do you really want to
        be an \"Experimental Subject\"?";
        return nil;
    }
;

mazeView: Room
    sDesc = "Maze Viewing Room"
    lDesc = {
        "The entire north wall of this room is occupied by a window
         overlooking a vast human-sized labyrinth. No experimental subjects
         (i.e., students) seem to be wandering through the maze at the
         moment, ";
         
        if (!mazeStart.isSeen)
            "but you have the strange feeling that you will have to find
            your way through the maze some day. You read with a sense of
            foreboding the plaque affixed to the wall:\b";
        else
            "so your attention wanders to the plaque on the wall:\b";
            
        mazePlaque.lDesc;
    }
    west = behaviorLab
;

mazeWindow: FixedItem
    location = mazeView
    noun = 'window'
    sDesc = "window"
    verDoLookthru(actor) = {}
    doLookthru(actor) = { self.lDesc; }
    lDesc = {
        "The window looks out onto a vast human-sized labyrinth.
         The maze is currently devoid of experimental subjects";
        if (!mazeStart.isSeen)
            ", but you have the strange feeling that you'll have to find
            your way through the maze someday";
        ". ";
    }
;

mazePlaque: Fixture, ReadableItem
    location = mazeView
    noun = 'plaque'
    sDesc = "plaque"
    lDesc = "*** The Behavioral Biology Laboratory Psycho-Magnetic Maze ***
        \bThe Psycho-Magnetic Maze that is visible through the window has been
        constructed to determine how the human directional sense interacts with
        strong electromagnetic and nuclear fields. Through careful tuning of
        these fields, we have found that human subjects often become completely
        disoriented in the maze, resulting in hours of random and
        desperate wandering, and much amusement to those of us in the observation
        room. "
;

behaviorDoor: LockableDoorway
    location = behaviorLab
    lDesc = "The door is closed and locked, and labelled \"Experimental
        Subjects Only.\""
;

security: Room
    sDesc = "Security Office"
    lDesc = "You are in the campus security office, the very nerve-center of
        the elite Kaltech Kops.  The officers all
        appear to be absent; undoubtedly they are all scurrying hither and
        yon trying to deal with the ditch-day festivities.  There is a desk
        in the center of the room.  The exit is to
        the north. "
    north = walkway
    out = walkway
;

securityDesk: Fixture, Surface
    location = security
    noun = 'desk'
    sDesc = "desk"
;

securityMemo: ReadableItem
    location = securityDesk
    noun = 'memo'
    adjective = 'security'
    plural = 'memos'
    isCrawlable = true
    sDesc = "security memo"
    lDesc = "From:\t\tDirector of Security
        \nTo:\t\tAll Security Personnel
        \nSubject:\tGreat Undergraduate Excavation
        \bIt has come to the attention of the Kaltech Kops that the student
        activity in the steam tunnels, known as the \"Great Undergraduate
        Excavation\" project, has escalated vastly in recent years. Due to
        objections from city officials about noise from underground and the
        fear local residents have expressed that their property will be
        undermined, this office has taken action to halt all GUE activity.
        Effective immediately, a security officer will be posted at all
        known steam tunnel entrances, and shall under no circumstances allow
        students to enter. "
;

flashlight: Container, Lightsource
    location = security
    noun = 'flashlight' 'light'
    adjective = 'flash'
    sDesc = "flashlight"
    lDesc = {
        if (battery.location == self) {
            if (self.lightsOn)
                "The flashlight (which contains a battery) is turned on
                and is providing a warm, reassuring beam of light. ";
            else
                "The flashlight (which contains a battery) is currently off. ";
        } else {
            "The flashlight is off. It seems to be missing a battery. ";
        }
    }
    grab(obj) = {
        /*
         *   grab( obj ) is invoked whenever an object 'obj' that was
         *   previously located within me is removed.  If the battery is
         *   removed, the flashlight turns off.
         */
        if (obj == battery) self.lightsOn = nil;
    }
    ioPutIn(actor, dobj) = {
        if (dobj != battery) {
            "You can't put "; dobj.theDesc; " into the flashlight. ";
        }
        else pass ioPutIn;
    }
    verDoTurnon(actor) = {
        if (self.lightsOn) "It's already on! ";
    }
    doTurnon(actor) = {
        if (battery.location == self) {
            "The flashlight is now on. ";
            self.lightsOn = true;
        }
        else "The flashlight won't turn on without a battery. ";
    }
    verDoTurnoff(actor) = {
        if (!self.lightsOn) "It's not on. ";
    }
    doTurnoff(actor) = {
        "Okay, the flashlight is now turned off. ";
        self.lightsOn = nil;
    }
;

hall1: Room
    sDesc = "Hallway"
    lDesc = "You are at the west end of a basement hallway. A stairway
        leads up. "
    up = courtyard
    east = hall2
;

hall2: Room
    sDesc = "Hallway"
    lDesc = "You are in an east-west hallway in the basement.  Another hallway
        goes off to the north. "
    west = hall1
    east = hall3
    north = hall4
;

hall3: Room
    sDesc = "Hallway"
    lDesc = "You are at the east end of a hallway in the basement.
        A passage leads north. "
    west = hall2
    north = laundry
;

laundry: Room
    sDesc = "Laundry Room"
    lDesc = "You are in the laundry room. There is a washing machine
        against one wall. The exit is to the south. "
    south = hall3
    out = hall3
;

washingMachine: Fixture, OpenableContainer
    location = laundry
    noun = 'machine' 'washer'
    adjective = 'washing'
    isOpen = nil
    sDesc = "washing machine"
;

jeans: Item
    location = washingMachine
    noun = 'jeans' 'pants'
    adjective = 'blue'
    sDesc = "blue jeans"
    lDesc = {
        if (self.isSeen) 
            "It's an ordinary pair of jeans. ";
        else {
            "It looks like an ordinary pair of jeans, though not
            your size. As you inspect them, you notice a key fall
            out of them and to the ground. ";
            masterKey.moveInto(parserGetMe().location);
            self.isSeen = true;
        }
    }
    verDoLookin(actor) = {}
    doLookin(actor) = {self.lDesc;}
    verDoWear(actor) = {"They're not your size. ";}
;

masterKey: KeyItem
    noun = 'key'
    adjective = 'master'
    isCrawlable = true
    sDesc = "master key"
    doTake(actor) = {
        if (!self.isSeen) {
            "*Some* adventure games would try to impose their authors'
            misguided sense of ethics on you at this point, telling you
            that you don't feel like picking up the key, or you don't have
            time to do that, or that it's against the rules to even possess
            a master key, much less steal one from some other student's
            pants that you happened to find in a laundry, or even more
            likely that you are unable to take the key while wearing that
            dress. However, you're the player, and you're in charge around
            here, so I'll let you make your own judgments about what's
            ethical and proper here... ";
            self.isSeen = true;
        }
        pass doTake;
    }
;

hall4: Room
    sDesc = "Hallway"
    lDesc = "You are in a north-south hallway in the basement. "
    south = hall2
    north = hall5
;

hall5: Room
    sDesc = "Hallway"
    lDesc = "You are at a corner in the basement hallway. You can go east
        or south. "
    south = hall4
    east = hall6
;

hall6: Room
    sDesc = "Hallway"
    lDesc = "You are at the east end of a basement hallway. To the north is
        a \"storage room,\" which everyone knows is actually an entrance
        to the steam tunnels. "
    west = hall5
    north = storage
;

storage: Room
    sDesc = "Storage Room"
    lDesc = {
        "You are in a large storage room. There really hasn't been
         anything stored here for a long time (at least, not anything that
         anybody wants to ever see again). The exit is to the south. To
         the north lies a door, which is ";
         tunnelDoor.isOpen ? "open" : "closed"; ". A small card
         table is sitting in front of the door. ";
    }
    south = hall6
    north = {
        if (guard.isActive) {
            "The guard won't let you enter the tunnel. ";
            return nil;
        } else if (!tunnelDoor.isOpen) {
            "A closed door stands in your way. ";
            setit(tunnelDoor);       /* "it" now refers to the closed door */
            return nil;
        } else return tunnel1;
    }
    enterRoom(actor) = {
        if (guard.isActive) {
            notify(guard, #patrol, 0);
        }
        pass enterRoom;
    }
    leaveRoom(actor) = {
        if (guard.isActive) {
            unnotify(guard, #patrol);
        }
        pass leaveRoom;
    }
;

tunnelDoor: LockableDoorway
    location = storage
    noun = 'door'
    adjective = 'tunnel'
    myKey = masterKey
;

guard: Actor
    location = storage
    noun = 'guard' 'him'
    adjective = 'security'
    isHim = true
    isActive = true
    sDesc = "guard"
    actorDesc = {
        if (self.isActive)
            "A guard is sitting at the card table in front of the door.
            He watches you carefully, evidently thinking that you might
            be planning to try to go through the door. ";
        else
            "A guard is slumped over a small card table in front of the
            steam tunnel entrance, evidently fast asleep. How typical. ";
    }
    lDesc = {
        "The guard is a member of the Kaltech Kops, the elite corps of
        dedicated men and women that keeps the campus safe from undesirables
        (i.e., the students). ";
        if (!self.isActive)
            "Currently, the guard is fast asleep, which is quite typical. ";
    }
    verIoGiveTo(actor) = {
        if (!self.isActive)
            "The guard appears to be fast asleep. ";
    }
    ioGiveTo(actor, dobj) = {
        if (dobj == dollar || dobj == money)
            "The guard looks at you sternly. \"You should be ashamed of
            yourself, trying to bribe a member of the elite Kaltech Kops!\"
            he admonishes you, refusing your offer. ";
        else if (dobj != cup || !cup.isFull)
            "The guard doesn't appear interested. ";
        else {
            self.isActive = nil;
            unnotify(self, #patrol);
            "The guard happily accepts your offer; \"ToxiCola! My favorite!\"
            he says appreciatively, not knowing the evil deed
            that you have in mind. He quickly drinks the entire cup of
            ToxiCola. \"Wow! Just the caffeine pickup I needed,\" he says
            happily.
            \n\tAfter a few moments, though, he looks rather queasy. \"That
            caffeine just doesn't last long enough,\" he says just before
            he passes out, slumping over the card table. ";
            cup.isFull = nil;
            toxicola.moveInto(nil);
            cup.moveInto(cardTable);
            incScore(10);
        }
    }
    patrolMessage = [
        // random message 1
        'The guard eyes you warily.'
        // random message 2
        'The guard looks at his empty glass, probably wishing he had
        something to drink.'
        // random message 3
        'The guard flips purposefully through the pages of his memo pad.'
        // random message 4
        'The guard writes something down on his memo pad, glancing up from
        time to time to eye you suspiciously.'
        // random message 5
        'The guard picks up his empty glass and starts to drink, then
        realizes it is empty and puts it back down.'
    ]
    patrol = {
        if (self.location == parserGetMe().location) {
            "\b";
            say(self.patrolMessage[rand( 5 )]);
        }
    }
;

cardTable: Fixture, Surface
    location = storage
    noun = 'table'
    adjective = 'card'
    sDesc = "card table"
;

emptyGlass: Container
    location = cardTable
    noun = 'glass'
    adjective = 'empty'
    sDesc = "empty glass"
    aDesc = "an empty glass"
    doTake(actor) = {
        if (guard.isActive)
            "The guard won't let you take the glass. \"Get your own,\"
            he says. ";
        else pass doTake;
    }
    ioPutIn(actor, dobj) = {
        if (dobj == ln2 ) 
            "The liquid nitrogen evaporates on contact. ";
        else "It won't fit in the glass. ";
    }
;

tunnelSounds: function(parm) {
    if (parserGetMe().location.isTunnel) {
        "\b";
        say(TunnelRoom.randomsound[rand( 5 )]);
    }
    setfuse(tunnelSounds, rand( 5 ), nil);
}

class TunnelRoom: Room
    isTunnel = true
    sDesc = "Steam Tunnel"
    randomsound = [
    // random sound 1
      'The rumbling sound suddenly becomes very loud, then, after
       a few moments, dies down to background levels again.'
    // random sound 2
      'A series of clanking noises, like marbles rolling through the
      steam pipes, starts in the distance, then comes
      closer and closer, until it seems to pass right overhead. It disappears
      into the distance.'
    // random sound 3
      'A very loud bang suddenly reverberates through the tunnel.'
    // random sound 4
      'One of the pipes starts to hiss wildly. After a few moments, it
      fades back into the background sounds.'
    // random sound 5
      'A loud buzzing sound, like an overloaded electrical circuit,
      emanates from somewhere nearby. After a few moments it is gone.'
    ]
;

tunnelPipes: FixedItem
    location = {
        if (parserGetMe().location.isTunnel) 
            return parserGetMe().location;
        else return nil;
    }
    locationOK = true
    noun = 'pipe' 'pipes'
    sDesc = "pipes"
    lDesc = "The pipes range from very small copper tubes only an inch around
        to huge asbestos-covered cylinders over two feet in diameter.  Many of
        the larger pipes are marked \"STEAM - 300 PSI.\" "
;

tunnel1: TunnelRoom
    lDesc = "You are in a steam tunnel. It is very hot and dry in here.
        The place has a strange musty odor; the air is very still, but there
        are distant sounds of all sorts that vibrate through the pipes. The
        pipes all seem to be hissing quietly, and a low rumbling sound constantly
        reverberates through the tunnel. Occasionally a distant clang or thud
        or crack emanates from the pipes.
        \n\tThe steam tunnel runs east and west. A small passage leads south. "
    east = tunnel2
    west = tunnel3
    south = storage
    enterRoom(actor) = {
        if (!self.isSeen) 
            setfuse(tunnelSounds, rand( 5 ), nil);
        pass enterRoom;
    }
;

tunnel2: TunnelRoom
    lDesc = "You are at the east end of the steam tunnel. "
    west = tunnel1
;

tunnelStorage: Room
    sDesc = "Storage Room"
    lDesc = "You are in a small storage room. The exit lies north. "
    north = tunnel13
;

tunnel3: TunnelRoom
    lDesc = "You are in an east-west section of the steam tunnels. "
    east = tunnel1
    west = tunnel4
;

tunnel4: TunnelRoom
    lDesc = "You are at a T-intersection of two sections of steam
        tunnel; tunnels go off to the north, south, and east. "
    east = tunnel3
    north = tunnel5
    south = tunnel6
;

class DarkTunnel: DarkRoom, TunnelRoom
    controlOn = nil
    isLit(vantage) = {
        if (self.controlOn) 
            return true;
        else pass isLit;
    }
;

tunnel5: DarkTunnel
    lDesc = "You are at a corner in the steam tunnel: you can go
        west and south. "
    west = tunnel7
    south = tunnel4
;

tunnel6: TunnelRoom
    lDesc = "You are at a corner in the steam tunnel: you can go
        west and north. "
    west = tunnel8
    north = tunnel4
;

tunnel7: DarkTunnel
    lDesc = "You are in an east-west section of the steam tunnel.
        A very small passage between some steam pipes leads north, but
        it would be a tight squeeze. "
    east = tunnel5
    west = tunnel9
    north = {
        return crawlTest(tunnel12,
            'You lay down on one of the pipes and start to snake
            through the passage. For a moment, you think you\'re
            stuck, but you manage to wriggle your way through. You
            emerge on the north side of the narrow crawl.');
    }
;

tunnel8: TunnelRoom
    controlOn = true
    lDesc = {
        "You are in an east-west section of the steam tunnel. On the
         wall is a control unit. ";
        if (!self.controlOn) {
            "It is quite dark in this section of the tunnel; the only
             illumination is coming from the control unit's display";
             
            if ((flashlight.isIn(parserGetMe()) || flashlight.isIn(tunnel8))
            && flashlight.lightsOn )
                " (and from the flashlight, of course)";
        
            ". ";
        }
    }
    east = tunnel6
    west = tunnel10
;

controlUnit: FixedItem
    sDesc = "control unit"
    noun = 'unit'
    adjective = 'control'
    location = tunnel8
    lDesc = {
        "The control unit is quite modern and high-tech looking, in
         stark contrast to the tunnels around it. It consists of a keypad
         which allows entry of arbitrary numbers, a large green button,
         and a display screen. The unit is labelled \"Station 2.\" ";
        controlDisp.lDesc;
    }
    objRef = {
        local l;
        
        l = self.value;
        
        if (l == 322) 
            return tunnel8;
        else if (l == 612) 
            return MazeRoom;
        else if (l == 293) 
            return DarkTunnel;
        else return nil;
    }
    value = 322
;

controlPad: FixedItem
    location = tunnel8
    noun = 'keypad' 'pad'
    adjective = 'key'
    sDesc = "keypad"
    lDesc = "It's one of those cheesy membrane keypads, like on a microwave
        oven or the new Enterprise's control panels. It allows you to type numbers
        made up of digits from 0 to 9. "
    verIoTypeOn(actor) = {}
    ioTypeOn(actor, dobj) = {
        if (dobj != numObj)
            "The keypad only allows entry of numbers. ";
        else {
            "As you type the number sequence, the display screen is
            updated. ";
            controlUnit.value = numObj.value;
            controlDisp.lDesc;
        }
    }
;

controlButton: ButtonItem
    location = tunnel8
    adjective = 'green'
    sDesc = "button"
    doPush(actor) = {
        local r;
        
        r = controlUnit.objRef;
        if (r) {
            "The display is updated as you press the button. ";
            r.controlOn = not r.controlOn;
            controlDisp.lDesc;
            if (r == tunnel8) {
                "The lights in the tunnel ";
                if (r.controlOn)
                    "come on. ";
                else {
                    "go out, leaving only the display screen's light ";
                    if ((flashlight.isIn(parserGetMe()) 
                    || flashlight.isIn(tunnel8))
                    && flashlight.lightsOn)
                        "(and the flashlight, of course) ";
                    "for illumination. ";
                }
            }
        }
        else "The unit beeps; nothing else appears to happen. ";
    }
;

controlDisp: FixedItem
    location = tunnel8
    noun = 'screen'
    adjective = 'display'
    sDesc = "display screen"
    lDesc = {
        local l, r;
        
        l = controlUnit.value;
        r = controlUnit.objRef;
        "The screen is currently displaying: \""; say( l ); ": ";       
        if (r) {
            r.controlOn == true ? "ON" : "OFF";
        }
        else "???";
        
        "\". ";
    }
;

tunnel9: DarkTunnel
    lDesc = "You are at a corner in the steam tunnel.  You can go east
        or south. "
    east = tunnel7
    south = tunnel11
;

tunnel10: TunnelRoom
    lDesc = "You are at a corner in the steam tunnel. You can go east
        or north. "
    east = tunnel8
    north = tunnel11
;

tunnel11: TunnelRoom
    lDesc = "You are in a north-south section of the steam tunnels. Set
        into one wall is a large chute. "
    north = tunnel9
    south = tunnel10
;

chute: Fixture, Container
    location = tunnel11
    noun = 'chute'
    sDesc = "chute"
    lDesc = "The chute is large enough for anything you're carrying,
        but not nearly big enough for you. You can't tell where it goes,
        except down. "
    ioPutIn(actor, dobj) = {
        "You put "; dobj.theDesc; " into the chute, and it slides away
        into the darkness. After a few moments, you hear a soft thud. ";
        dobj.moveInto(chuteRoom);
    }
;

crawlTest: function(rm, msg) {
    local i;
    
    i = 1;
    while ( i <= length(parserGetMe().contents)) {
        if (!parserGetMe().contents[i].isCrawlable) {
            "You'll never get through carrying ";
            parserGetMe().contents[i].theDesc; ". ";
            return nil;
        }
        ++i;
    }
    say(msg); "\b";
    return rm;
}

tunnel12: TunnelRoom
    lDesc = {
        if (self.isSeen)
            "You are in an east-west section of the steam tunnels. A narrow
            passage between some steam pipes might allow you to go south. ";
        else
            "You are in another steam tunnel, but this one is substantially
            different from the tunnels you have been in so far. This tunnel
            is much wider, less cluttered; it looks like it was built more
            recently than the south tunnels. The tunnel itself runs east
            and west, and passing back to the south is evidently possible,
            given your presence here. ";
    }
    east = tunnel13
    west = pitTop
    south = {
        return crawlTest(tunnel7, 'Going through the crawl
            southbound is just as difficult as it was coming north,
            you observe.');
    }
;

tunnel13: TunnelRoom
    lDesc = "You are at the east end of a steam tunnel. A passage leads
        north, and another one leads south. "
    west = tunnel12
    north = maze0
    south = tunnelStorage
;

maze0: Room
    sDesc = "Outside Maze"
    lDesc = {
        "You are in a small room, with exits north and south. The
         passage to the north has a small sign reading \"Behavior Lab Maze\";
         the room is filled with strange equipment, which you surmise is
         connected in some way to the maze. ";
         
        if (MazeRoom.controlOn)
            "The equipment is buzzing loudly. Standing
             near it makes you feel vaguely dizzy. ";
        
        if (!self.isSeen) {
            "\n\tYou sigh in resignation as you realize that you have reached
            the obligatory Adventure Game Maze, and ";
            if (mazeView.isSeen)
                "wish that this were a graphical adventure, so your trip to
                the maze viewing room had resulted in a map you could refer
                to now. Fortunately, you do recall noting that the passages
                in the maze all lead east-west or north-south. ";
            else
                "wonder what this maze's unique twist will be. Surely, it
                won't just be a boring old labyrinth... ";
        }
    }
    south = tunnel13
    north = maze1
;

mazeEquip: Decoration
    location = maze0
    noun = 'equipment' 'coil' 'coils' 'pipe' 'pipes' 'wire' 'wires'
    adjective = 'electric' 'electrical'
    sDesc = "equipment"
    lDesc = {
        "The equipment resembles some of the particle accelerators that
        you have seen. It has several huge electric coils, all arranged
        around a series of foot-diameter pipes. A tangled web of enormous
        wires connects various parts of the equipment together and other
        wires feed it power. ";
        
        if (MazeRoom.controlOn)
            "The equipment is buzzing loudly. Standing near it makes
            you feel vaguely dizzy. ";
    }
;

class MazeRoom: Room
    controlOn = true
    sDesc = "Lost in the Maze"
    lookAround(verbosity) = {
        self.dispBeginsDesc;
        self.statusRoot;
        self.dispEndsDesc;
        
        self.nrmLkAround(self.controlOn ? true : verbosity);
    }
    lDesc = {
        if (self.controlOn)
            "You can't quite seem to get your bearings. There are some
            passages leading away, but you can't quite tell how many or
            in which direction they leads. ";
        else {
            local cnt, tot, i;
            
            tot = 0;
            i = 1;
            while (i <= 4) {
                if (self.dirList[i]) 
                    ++tot;
                ++i;
            }

            "You are in a room in the maze; they all look alike.
            You can go ";
            
            i = 1;
            cnt = 0;
            while (i <= 4) {
                if (self.dirList[i]) {
                    if (cnt > 0) {
                        if (tot == 2) 
                            " and ";
                        else if (cnt + 1 == tot) 
                            ", and ";
                        else ", ";
                    }
                    cnt = cnt + 1;
                    
                    say(['north' 'south' 'east' 'west'][i]);
                }
                i = i + 1;
            }
            ". ";
        }
    }
    mazetravel( rm ) = {
        if (self.controlOn) {
            local r;
            
            "You can't figure out which direction is which; the more you
            stumble about, the more the room seems to spin around. After a
            few steps, you're not sure if you've actually gone anywhere,
            since all these rooms look alike...\b";
            
            /*
             *   We know we can only go one of four directions, but generate
             *   a random number up to 6; if we generate 5 or 6, we won't go
             *   anywhere, but we won't let on that this is the case.
             */
            r = rand(6);
            
            /*
             *   Note that we want to confuse the player in active-maze mode
             *   as much as possible, so we don't want any clues as to whether
             *   there was any travel in this direction or not.  So, return
             *   "self" rather than "nil," since we won't get any message if
             *   we return "nil," but we'll get the current room's message if
             *   we return "self;" since all the messages are the same, this
             *   won't provide any information.
             */
            if (r < 5) {
                r = self.dirList[r];
                if (r) 
                    return r;
                else return self;
            }
            else return self;
        } else {
            if (rm)
                return rm;
            else {
                "You can't go that way. ";
                return nil;
            }
        }
    }
    north = ( self.mazetravel( self.dirList[1] ))
    south = ( self.mazetravel( self.dirList[2] ))
    east = ( self.mazetravel( self.dirList[3] ))
    west = ( self.mazetravel( self.dirList[4] ))
    up = ( self.mazetravel( 0 ))
    down = ( self.mazetravel( 0 ))
    in = ( self.mazetravel( 0 ))
    out = ( self.mazetravel( 0 ))
    ne = ( self.mazetravel( 0 ))
    nw = ( self.mazetravel( 0 ))
    se = ( self.mazetravel( 0 ))
    sw = ( self.mazetravel( 0 ))
;

maze1: MazeRoom
    dirList = [ 0 maze0 maze2 0 ]
;

maze2: MazeRoom
    dirList = [ maze9 0 maze3 maze1 ]
;

maze3: MazeRoom
    dirList = [ maze8 0 maze4 maze2 ]
;

maze4: MazeRoom
    dirList = [ 0 0 maze5 maze3 ]
;

maze5: MazeRoom
    dirList = [ maze6 0 0 maze4 ]
;

maze6: MazeRoom
    dirList = [ 0 maze5 0 maze7 ]
;

maze7: MazeRoom
    dirList = [ maze30 0 maze9 maze8 ]
;

maze8: MazeRoom
    dirList = [ maze29 maze3 maze7 0 ]
;

maze9: MazeRoom
    dirList = [ 0 maze2 0 maze10 ]
;

maze10: MazeRoom
    dirList = [ 0 0 maze9 maze11 ]
;

maze11: MazeRoom
    dirList = [ 0 maze35 maze10 maze12 ]
;

maze12: MazeRoom
    dirList = [ 0 0 maze11 0 ]
;

maze13: MazeRoom
    dirList = [ maze24 maze18 0 0 ]
;

maze14: MazeRoom
    dirList = [ 0 maze17 0 maze15 ]
;

maze15: MazeRoom
    dirList = [ 0 maze16 maze14 0 ]
;

maze16: MazeRoom
    dirList = [ maze15 maze23 maze17 0 ]
;

maze17: MazeRoom
    dirList = [ maze14 maze22 maze18 maze16 ]
;

maze18: MazeRoom
    dirList = [ maze13 maze21 0 maze17 ]
;

maze19: MazeRoom
    dirList = [ 0 maze20 maze35 0 ]
;

maze20: MazeRoom
    dirList = [ maze19 0 0 maze21 ]
;

maze21: MazeRoom
    dirList = [ maze18 0 maze20 0 ]
;

maze22: MazeRoom
    dirList = [ maze17 0 0 0 ]
;

maze23: MazeRoom
    dirList = [ maze16 0 0 0 ]
;

maze24: MazeRoom
    dirList = [ 0 maze13 maze25 0 ]
;

maze25: MazeRoom
    dirList = [ maze33 0 maze26 maze24 ]
;

maze26: MazeRoom
    dirList = [ maze32 0 0 maze25 ]
;

maze27: MazeRoom
    dirList = [ maze31 0 0 0 ]
;

maze28: MazeRoom
    dirList = [ 0 0 maze29 0 ]
;

maze29: MazeRoom
    dirList = [ 0 maze8 maze30 maze28 ]
;

maze30: MazeRoom
    dirList = [ 0 maze7 0 maze29 ]
;

maze31: MazeRoom
    dirList = [ 0 maze27 0 maze32 ]
;

maze32: MazeRoom
    dirList = [ 0 maze26 maze31 0 ]
;

maze33: MazeRoom
    dirList = [ 0 maze25 0 maze34 ]
;

maze34: MazeRoom
    dirList = [ 0 0 maze33 mazeStart ]
;

maze35: MazeRoom
    dirList = [ maze11 0 0 maze19 ]
;

mazeStart: Room
    sDesc = "Start of Maze"
    lDesc = "You are at the start of the maze. A passage into the
        maze is to the east, and a heavy one-way door marked \"Exit\" is to the
        south. "
    east = maze34
    south = behaviorLab
;

mazeDoor: FixedItem
    location = mazeStart
    noun = 'door'
    adjective = 'exit' 'one-way' 'heavy'
    sDesc = "door"
    verDoOpen(actor) = { "No need; just walk on through. "; }
;

chuteRoom: Room
    sDesc = "Chute Room"
    lDesc = "You are in a small room with exits to the northwest
        and south. The bottom of a large chute opens into the room. "
    nw = pitBottom
    south = shipRoom
;

chute2: Fixture, Container
    location = chuteRoom
    noun = 'chute'
    sDesc = "chute"
    ioPutIn(actor, dobj) = {
        "This is the bottom of the chute; you can't put objects
        into it. ";
    }
;

pitTop: TunnelRoom
    sDesc = "Top of Pit"
    lDesc = {
        "You are in a large open area in the steam tunnels. In
         the center of the room is a large pit, around which is a protective
         railing. A steam tunnel is to the east. ";
        if (rope.tieItem == railing )
            "A rope is tied to the railing, and extends down into the pit. ";
    }
    east = tunnel12
    down = {
        if (rope.tieItem == railing) {
            "You climb down the rope...\b";
            return pitBottom;
        } else {
            "You'd probably break your neck if you tried to jump
            into the pit. ";
            return nil;
        }
    }
;

railing: FixedItem
    location = pitTop
    noun = 'rail' 'railing'
    sDesc = "rail"
    verIoTieTo(actor) = {}
    ioTieTo(actor, dobj) = {
        "You tie one end of the rope to the railing, and lower the other
        end into the pit. It appears to extend to the bottom of the pit. ";
        rope.tieItem = self;
        rope.moveInto(pitTop);
        rope.isFixed = true;
    }
;

rope: Item
    sDesc = "rope"
    isListed = (!self.isFixed)
    noun = 'rope'
    location = tunnelStorage
    lDesc = {
        if (self.tieItem) {
            "It's tied to "; self.tieItem.theDesc; ". ";
        } else pass lDesc;
    }
    verDoTieTo(actor, iobj) = {
        if (self.tieItem) {
            "It's already tied to "; self.tieItem.theDesc; "! ";
        }
    }
    doTake(actor) = {
        if (self.tieItem) {
            "(You untie it first, of course.) ";
            self.tieItem = nil;
            self.isFixed = nil;
        }
        pass doTake;
    }
    verDoClimb(actor) = {
        if (self.tieItem == nil)
            "Climbing down the rope in its present configuration would
            get you nowhere. ";
    }
    doClimb(actor) = {
        "%You% climb%s% down the rope...\b";
        actor.travelTo(pitBottom);
    }
    verDoClimbdown(actor) = {}
    doClimbdown(actor) = {self.doClimb(actor);}
;

pitBottom: Room
    sDesc = "Huge Cavern"
    lDesc = "You are in a huge and obviously artificial cavern. The cave
        has apparently been dug out over a long period of time; some parts
        look very old, and other areas look comparatively recent. A small bronze
        plaque affixed to one of the older walls reads, \"Great Undergraduate
     E  xcavation - 1982.\"
        From high above, a small opening in
        the ceiling casts a dim glow over the vast chamber. A rope extends
        from the opening above. Passages of various ages lead north, south,
        east, west, southeast, and southwest. "
    up = {
        "It's a long climb, but you somehow manage it.\b";
        return pitTop;
    }
    se = chuteRoom
    east = bioHall1
    south = bank
    north = cave
    sw = insOffice
    west = machineShop
;

pitPlaque: Fixture, ReadableItem
    location = pitBottom
    noun = 'plaque'
    adjective = 'bronze' 'small'
    sDesc = "bronze plaque"
    lDesc = "Great Undergraduate Excavation\n\t\t\t1982"
;

insOffice: Room
    sDesc = "Insurance Office"
    lDesc = {
        "You are in an insurance office. Like most insurance offices,
        the area is rather non-descript. The exit is northeast. ";
        if (!self.isSeen) {
            "\n\tAs you walk into the office, a large metallic robot, very
            much like the traditional sci-fi film robot, but wearing a dark
            polyester business suit, zips up to you. \"Hi, I'm Lloyd the
            Friendly Insurance Robot,\" he says in a mechanical British
            accent. \"You look like you could use some insurance! Here, let
            me prepare a policy for you.\"
            \n\tLloyd scurries around the room, gathering papers and
            studying charts, occasionally zipping up next to you and
            measuring your height and other dimensions, making all kinds
            of notes, and generally scampering about. After a few minutes,
            he comes up to you, showing you a piece of paper.
            \n\t\"I have the perfect policy for you. Just one dollar will
            buy you a hundred thousand worth of insurance!\" He watches
            you anxiously. ";
            
            notify( lloyd, #offerIns, 0 );
        }
    }
    ne = pitBottom
    out = pitBottom
;

/*
 *   Lloyd the Friendly Insurance Robot is a full-featured actor. He will
 *   initially just wait in his office until paid for a policy, but will
 *   thereafter follow the player around relentlessly.  Lloyd doesn't
 *   interact much, though; he just hangs around and does wacky things.
 */
lloyd: Actor
    location = insOffice
    noun = 'lloyd' 'him'
    sDesc = "Lloyd"
    aDesc = "Lloyd"
    theDesc = "Lloyd"
    isHim = true
    offering = true
    actorDesc = {
        if (self.offering)
            "Lloyd is here, offering you an insurance policy for only
            one dollar. ";
        else
            "Lloyd the Friendly Insurance Robot is here. ";
    }
    lDesc = "Lloyd the Friendly Insurance Robot is a strange combination
        of the traditional metallic robot and an insurance salesman; over his
        bulky metal frame is a polyester suit. "
    actorAction( v, d, p, i ) = {
        if (v == helloVerb)
            "\"Hello,\" Lloyd responds cheerfully. ";
        else if (v == followVerb && d == parserGetMe()) {
            if (self.offering)
                "\"Sorry, but I must stay here until I've made a sale.\" ";
            else
                "\"I will follow you,\" Lloyd says. \"That will allow me
                to pay any claim you make immediately should the need arise.
                It's just one of the ways we keep our overhead so low.\" ";
        } else
            "\"I'm sorry, that's not in your policy.\" ";
        exit;
    }
    verDoAskAbout(actor, iobj) = {}
    doAskAbout(actor, iobj) = {
        if (iobj == policy)
            "\"Your policy perfectly fits your needs, according to my
            calculations,\" Lloyd says. \"I'm afraid it's much too
            complicated to go into in any detail right now, but you have
            my assurances that you won't be disappointed.\" ";
        else
            "\"I'm afraid I don't know much about that,\" Lloyd says
            apologetically. ";
    }
    offerMsg = [
        'Lloyd waits patiently for you to make up your mind about the
         insurance policy.'
        'Lloyd watches you expectantly, hoping you will buy the insurance
         policy.'
        'Lloyd shows you the insurance policy again. "Only a dollar," he
         says.'
        'Lloyd looks at you intently. "What can I do to make you buy this
         insurance policy right now?" he asks rhetorically.'
        'Lloyd reviews the policy to make sure it looks just right, and
         offers it to you again.'
    ]
    offerIns = {
        if (self.location == parserGetMe().location) {
            "\b";
            say(self.offerMsg[rand( 5 )]);
        }
    }
    followMsg = [
        'Lloyd watches attentively to make sure you don\'t hurt yourself.'
        'Lloyd hums one of his favorite insurance songs.'
        'Lloyd gets out some actuarial tables and does some computations.'
        'Lloyd looks through his papers.'
        'Lloyd checks the area to make sure it\'s safe.'
    ]
    follow = {
        if (parserGetMe().location == machineStorage) 
            return;
        
        "\b";
        if (self.location != parserGetMe().location) {
            if (parserGetMe().location == railCar)
                "Lloyd hops into the railCar, showing remarkable agility
                for such a large mechanical device. ";
            else if (self.location == railCar)
                "Lloyd hops out of the railCar. ";
            else if (parserGetMe().location == pitTop 
                && self.location == pitBottom)
                "Amazingly, Lloyd scrambles up the rope and joins you. ";
            else if (parserGetMe().location == pitBottom 
                && self.location == pitTop)
                "Lloyd descends smoothly down the rope and joins you. ";
            else if ((parserGetMe().location == tunnel7 
                && self.location == tunnel12)
                || (parserGetMe().location == tunnel12 
                && self.location == tunnel7 ))
                "Somehow, Lloyd manages to squeeze through the crawl. ";
            else if (parserGetMe().location == quad)
                "Lloyd follows you. When he sees the apparent radioactive
                waste spill, he is quite alarmed. After a moment, though,
                he deduces that the spill is a staged part of the Ditch Day
                festivities, and he calms down. ";
            else if (parserGetMe().location == storage)
                "Lloyd enters the storage room. Upon seeing the guard, he
                rolls over and starts giving his insurance pitch. After a
                moment, he notices the guard is unconscious. \"It's just as
                well,\" Lloyd confides; \"security work is awfully risky, and
                his rates would have been quite high.\" ";
            else
                "Lloyd follows you. ";
                
            self.moveInto(parserGetMe().location);
        } else {
            say(self.followMsg[rand( 5 )]);
        }
    }
    verIoGiveTo(actor) = {}
    ioGiveTo(actor, dobj) = {
        if (dobj == dollar)
            self.doPay(actor);
        else if (dobj == darbCard)
            "Lloyd looks at you apologetically. \"So sorry, I must insist
            on cash,\" he says. ";
        else
            "Lloyd doesn't appear interested. ";
    }
    verDoPay(actor) = {}
    doPay(actor) = {
        if (!self.offering) {
            "Lloyd looks at you, confused. \"I've already sold you all
            the insurance you need!\" ";
        } else if (dollar.isIn(actor)) {
            "Lloyd graciously accepts the payment, and hands you a copy
            of your policy. \"You might wonder how we keep costs so low,\"
            Lloyd says. \"It's simple: we're highly automated, which keeps
            labor costs low; I run the whole company, which keeps the
            bureaucratic overhead low; and, most importantly, I follow you
            everywhere you go for the duration of the policy, ensuring that
            you're paid on the spot should anything happen, which means we
            don't have to waste money investigating claims!\" ";
            
            unnotify(self, #offerIns);
            notify(self, #follow, 0);
            dollar.moveInto(nil);
            policy.moveInto(actor);
            self.offering = nil;
            actor.isInsured = true;
        } else {
            "You don't have any money with which to pay Lloyd. ";
        }
    }
;

policy: ReadableItem
    location = lloyd
    noun = 'policy'
    adjective = 'insurance'
    sDesc = "insurance policy"
    aDesc = "an insurance policy"
    isCrawlable = true
    lDesc = {
        "The insurance policy lists the payment schedule for hundreds
        of types of injuries; the list is far too lengthy to go into in any
        detail here, but rest assured, ";
        if (lloyd.offering)
            "it looks like an excellent deal. ";
        else
            "you're highly satisified with what you
            got for your dollar. You feel extremely well protected. ";
    }
;

machineShop: Room
    sDesc = "Machine Shop"
    lDesc = "You are in the machine shop. It appears that this huge
        chamber was once used to build and maintain the equipment that
        was used to create the Great Undergraduate Excavation.  Though
        most of the equipment is gone now, one very large and strange
        machine dominates the center of the room.
        The exit is east, and a small passage leads north. "
    east = pitBottom
    out = pitBottom
    north = machineStorage
;

machine: FixedItem
    location = machineShop
    noun = 'machine'
    sDesc = "machine"
    lDesc = "The machine is unlike anything you've seen before; it's not
        at all clear what its purpose is. The only feature that looks like it
        might do anything useful is a large red button labelled \"DANGER!\" "
;

machineButton: ButtonItem
    location = machineShop
    adjective = 'red'
    sDesc = "red button"
    doPush(actor) = {
        "As you push the button, the machine starts making horrible noises
        and flinging huge metal rods in all directions. Enormous clouds of
        smoke rise from the machine as it flails about. After a few minutes
        of this behavior, you think to step back from the machine.  You're
        not fast enough, though; before you can escape it, a stray metal
        bar flings itself against your thumb, creating a sensation not unlike
        intense pain. ";
        
        if (lloyd.location == self.location) {
            "\n\t";
            if (self.isScored)
                "Lloyd looks at you apologetically. \"I don't want to sound
                patronizing, but you knew it would do that. I'm afraid I can't
                accept a claim for that injury, as it was effectively
                self-inflicted.\" He looks at you sadly. \"I do sympathize,
                though. That must be quite painful,\" he understates. ";
            else {
                self.isScored = true;
                "Lloyd rolls over and looks at your thumb. \"That looks
                awful,\" he says as you jump about, holding your thumb.
                Lloyd produces a thick pile of paper and starts leafing
                through it. \"Temporary loss of use of one thumb... ah,
                yes, here it is. That injury pays five dollars.\" Lloyd puts
                away his papers and produces a crisp new five dollar bill,
                which you accept (with your other hand). ";
                money.moveInto(parserGetMe());
            }
        }
    }
;

machineStorage: DarkRoom
    sDesc = "Storage Closet"
    lDesc = "You are in a small storage closet off the machine shop.
        The exit is south. "
    south = machineShop
    out = machineShop
;

happyGear: Treasure
    location = machineStorage
    noun = 'gear'
    adjective = 'mr.' 'mr' 'happy' 'mister'
    sDesc = "Mr.\ Happy Gear"
    theDesc = "Mr.\ Happy Gear"
    aDesc = "Mr.\ Happy Gear"
    lDesc = "It's an ordinary gear, about an inch in diameter; the only
        notable feature is that it has holes cut in such a manner that it
        looks like a happy face. "
;

bank: Room
    sDesc = "Bank"
    lDesc = "You're in what was once the Great Undergraduate Excavation's
        bank. It doesn't appear to get much use any more. The exit is north,
        and a small passage leads south. "
    north = pitBottom
    south = bankVault
    out = pitBottom
;

bankVault: Room
    sDesc = "Vault Room"
    lDesc = "The feature dominating this room is the bank's safe.
        The only exit is north. "
    north = bank
    out = bank
;

bankSafe: Fixture, OpenableContainer
    location = bankVault
    noun = 'safe' 'door' 'vault'
    isOpen = nil
    sDesc = "safe"
    lDesc = {
        if (self.isBlasted) {
            "The safe looks as though it has suffered some sort of
            intense trauma lately; the door is just barely hanging on
            its hinges, leaving the contents of the safe quite exposed. ";
            pass lDesc;
        } else {
            "The safe is huge, like the type you might find in a bank.
            The only notable features are a huge metal door (quite closed),
            and a large slot labelled \"Night Deposit Slot.\" ";
        }
    }
    doOpen(actor) = {
        if (self.isBlasted) 
            pass doOpen;
        else
            "You can't open such a secure safe without resorting
             to some sort of drastic action. ";
    }
;

darbCard: Treasure
    location = bankSafe
    noun = 'darbcard' 'card'
    adjective = 'darb'
    sDesc = "DarbCard"
;

bankSlot: Fixture, Container
    location = bankVault
    noun = 'slot'
    adjective = 'night' 'deposit'
    sDesc = "night deposit slot"
    lDesc = "The slot is very large, big enough to put a large sack of
        money into. Unfortunately, you won't have much luck extracting anything
        from the slot, since it has been carefully constructed to allow items
        to enter, but not leave. "
    ioPutIn(actor, dobj) = {
        "\^<< dobj.theDesc >> disappears into the deposit slot. ";
        dobj.moveInto( bankSafe );
    }
;

cave: Room
    sDesc = "Tunnel"
    lDesc = "You're in a north-south tunnel. The tunnel slopes steeply
        downward to the north. "
    north = railway1
    down = railway1
    south = pitBottom
    up = pitBottom
;

railway1: Room
    sDesc = "Subway Station"
    lDesc = "You're in a very large musty chamber deep underground. In the
        center of the room is a small rail car. Strangely, the car is sitting on
        the floor; there are no rails under the car, or, indeed, in the station
        at all. You look around, and notice about three meters up the east wall
        is a round tunnel. A passage leads south. "
    south = cave
    destination = railway2
    tunnelTime = 3
    east =
    {
        "The tunnel is too high up the wall. You won't be able to enter
        it without benefit of the railCar. ";
        return( nil );
    }
;

class RailTunnel: FixedItem
    noun = 'tunnel'
    sDesc = "tunnel"
    verDoEnter(actor) = {}
    doEnter(actor) = {
        railway1.east;
    }
;

RailTunnel1: RailTunnel
    location = railway1
;

railTunnel2: RailTunnel
    location = railway2
;

railway2: Room
    tunnelTime = 3
    destination = railway1
    sDesc = "Subway Station"
    lDesc = "You're in a subway station. About three meters up the west
        wall is a tunnel; a passage (at ground level) leads south. A railCar
        is sitting on the floor in the center of the room. "
    south = computerCenter
    west = {
        "The tunnel is too high up the wall. You won't be able to enter
        it without benefit of the railCar. ";
        return( nil );
    }
;

computerCenter: Room
    sDesc = "Computer Center"
    lDesc = "You're in a computer room. Unfortunately, all the equipment
        here is hopelessly out of date and doesn't interest you in the least.
        The exit is north. "
    north = railway2
;

compEquip: Decoration
    location = computerCenter
    noun = 'equipment'
    adjective = 'computer'
    sDesc = "computer equipment"
    lDesc = "The equipment is all very outdated. It's not the least bit
        interesting. "
;

randomBook: Treasure, ReadableItem
    location = computerCenter
    noun = 'book' 'digits'
    adjective = 'million' 'random' 'digits' 'numbers'
    sDesc = "book"
    lDesc = "The book is entitled \"A Million Random Digits.\" In flipping
        through the book, you find that it is, in fact, a million random digits,
        nicely tabulated and individually numbered from 0 to 999,999 (computer
        people always start numbering at 0). "
;

railTunnel: Room
    sDesc = "Tunnel"
    lDesc = "The tunnel is rather non-descript. "
;

railCar: Fixture, VehicularContainer
    location = railway1
    noun = 'car' 'railCar'
    adjective = 'rail'
    isDropLoc = true        // stuff dropped while on board ends up in railCar
    boardingPosture = sitting
    sDesc = "rail car"
    lDesc = {
        "This is a small rail car, big enough for a couple of people.
         It has a small control panel, which consists of a green button,
         a gauge, and a small hole labelled \"Coolant.\" ";
         if (funnel.location == railHole)
            "The hole seems to contain a funnel. ";
         railMeter.lDesc;
    }
    travel = {
        if (self.location != railTunnel) {
            "\bThe car rises to about three meters off the floor. It then
            starts to accelerate toward the tunnel, and plunges into the
            tunnel with a rush of air. ";
            self.destination = self.location.destination;
            self.tunnelTime = self.location.tunnelTime;
            self.moveInto(railTunnel);
        } else if (self.tunnelTime > 0) {
            "\bThe car races down the tunnel at terrifying speed. ";
            self.tunnelTime = self.tunnelTime - 1;
        } else {
            "\bThe car starts to decelerate sharply. After a few more seconds,
            it emerges into a station and glides to a halt. It slowly
            descends to the ground; once settled, the hum of its engine
            gradually disappears. ";
            self.isActive = nil;
            self.moveInto(self.destination);
            unnotify(self, #travel);
        }
    }
    checkUnboard = {
        if (self.isActive) {
            "Please wait until the railCar has come to a complete and
            final stop, and the captain has turned off the \"Fasten Seat
            Belt\" sign. (Actually, I made up the part about the \"Fasten
            Seat Belt\" sign, but you'll have to wait until the car
            has stopped nonetheless.) ";
            return nil;
        } else return true;
    }
    out = {
        if (self.checkUnboard) 
            pass out;
        else return( nil );
    }
    doUnboard(actor) = {
        if (self.checkUnboard) 
            pass doUnboard;
    }
;

railMeter: FixedItem
    location = railCar
    sDesc = "gauge"
    noun = 'gauge' 'meter'
    lDesc = {
        "The gauge is not labelled with any units you recognize, but
         the arc is divided into regions colored green, yellow, and red. The
         needle is currently in the ";
        if (railCar.isCooled) 
            "green"; 
        else "red";
        " section of the scale. ";
    }
;

railHole: Fixture, Container
    location = railCar
    sDesc = "hole"
    noun = 'hole'
    adjective = 'coolant'
    lDesc = {
        if (funnel.location == self)
            "A funnel is in the hole. ";
        else if (railCar.isCooled)
            "Wisps of water vapor rise from the hole. ";
        else
            "The hole is labelled \"Coolant.\" It's about an inch
            or so in diameter. ";
    }
    verIoPourIn(actor) = {}
    ioPourIn(actor, dobj) = { self.ioPutIn(actor, dobj); }
    ioPutIn(actor, dobj) = {
        if (dobj == funnel) {
            "A perfect fit! ";
            funnel.moveInto(self);
        } else if (dobj == ln2) {
            if (funnel.location == self) {
                if (railCar.isCooled)
                    "It's already full of liquid nitrogen! ";
                else {
                    "You carefully pour the liquid nitrogen into the
                    funnel.  It doesn't take very much before the tank
                    is full. ";
                    railCar.isCooled = true;
                }
            } else {
                "The hole is too small; most if not all of the liquid
                nitrogen you pour spills all around, rather than into
                the hole. ";
            }
        } else {
            "It won't fit in the hole. ";
        }
    }
;

railButton: ButtonItem
    location = railCar
    sDesc = "green button"
    adjective = 'green'
    doPush(actor) = {
        if (parserGetMe().location != railCar) {
            "Get in the railCar first. ";
        } else if (railCar.isBroken) {
            "Nothing happens. ";
        } else if (railCar.isActive) {
            "Nothing happens. Perhaps this is because you have
            already pushed the button quite recently. ";
        } else if (railCar.isCooled) {
            "A low-frequency hum sounds from within the railCar. After
            a few moments, it starts to levitate off the track. ";
            notify(railCar, #travel, 0);
            railCar.isActive = true;
        } else {
            "A low-frequency hum sounds from within the railCar. It grows
            in strength, and soon starts to vibrate the whole car. You smell
            the familiar odor of burning electronic components. Suddenly,
            a bright light flashes underneath the railCar, a cloud of thick
            black smoke rises, and the humming stops. It appears you have
            toasted the railCar. ";
            railCar.isBroken = true;
        }
    }
;

bioHall1: Room
    sDesc = "Hall"
    lDesc = "You are in an east-west hallway. A passage labelled \"Bio Lab\"
     leads south. "
    west = pitBottom
    east = bioHall2
    south = bioLab
;

bioLab: Room
    sDesc = "Bio Lab"
    lDesc = {
        "You are in the Biology Lab. All sorts of strange equipment is
        scattered around the room. A lab bench is in the center of the room,
        and on one wall is a cabinet (which is ";
        if (bioCabinet.isOpen) 
            "open"; 
        else "closed";
        "). A passage leads north. ";
    }
    north = bioHall1
;

bioEquipment: Decoration
    sDesc = "strange equipment"
    noun = 'equipment'
    adjective = 'strange'
    location = bioLab
    lDesc = "The equipment is entirely unfamiliar to you. "
;

bioBench: Fixture, Surface
    noun = 'bench'
    adjective = 'lab'
    sDesc = "lab bench"
    location = bioLab
    lDesc = {
        "The bench is topped with one of those strange black rubber surfaces
        that seemingly all scientific lab benches have. ";
        pass lDesc;
    }
;

funnel: Container
    location = bioBench
    noun = 'funnel'
    sDesc = "funnel"
    lDesc = {
        if (self.location == bottle || self.location == railHole) {
            "The funnel is stuck into "; self.location.aDesc; ". ";
        } else
            "It's a normal white plastic funnel, about six inches
            across at its wide end and about half an inch at its
            narrow end. ";
    }
    ioPutIn(actor, dobj) = {
        if (dobj == ln2) {
            if (self.location == railHole 
            || self.location == bottle)
                self.location.ioPutIn(actor, dobj);
            else
                "The liquid nitrogen pours through the funnel, lands
                nowhere in particular, and evaporates. ";
        } else {
            "It wouldn't accomplish anything to put ";
            dobj.theDesc; " into the funnel. ";
        }
    }
    verIoPourIn(actor) = {}
    ioPourIn(actor, dobj) = { self.ioPutIn(actor, dobj); }
;

bioHall2: Room
    sDesc = "Hall"
    lDesc = {
        "You are at the east end of an east-west hallway. A doorway
         leads east. ";
        if (!self.isSeen) {
            notify(bioCreature, #menace, 0);
        }
    }
    west = bioHall1
    east = {
        if (bioCreature.location == self) {
            if (slime.location == nil) {
                "The creature grabs you and pushes you back, depositing
                a huge glob of slime on you in the process. ";
                slime.moveInto(parserGetMe());
                slime.wearer = parserGetMe();
            } else "The creature won't let you pass. ";
            return nil;
        }
        else return bioOffice;
    }
;

slime: ClothingItem
    noun = 'glob' 'slime'
    sDesc = "glob of slime"
    doWear(actor) = {
        "No, thank you. ";
    }
;

bioCreature: Actor
    location = bioHall2
    sDesc = "creature"
    noun = 'creature'
    isHim = true
    lDesc = "It looks like the result of a biological experiment that
        failed (or succeeded, depending on who performed the experiment). "
    actorDesc = "An enormous creature is blocking the hallway to the east.
        He (please don't press for details as to how you know, but \"he\"
        is the appropriate pronoun here) appears to be part human, but
        exactly what part is not clear. The creature's leathery skin is
        a bright green, and is largely covered with a thick transluscent
        slime. "
    menaceMessage = [
        'The creature roars a huge roar in your general direction.'
        'The creature menaces you.'
        'In a tender moment, the creature produces a magazine and opens up
         the centerfold. He looks longingly at the picture. After a few
         moments, he notices you again, and puts away the magazine; as he\'s
         putting it away, you see that it\'s a copy of "Playmutant."'
        'The creature looks at you warily.'
        'The creature growls at you, showing his enormous pointy fangs.'
    ]
    menace = {
        if (self.location == parserGetMe().location) {
            "\b";
            say(self.menaceMessage[rand( 5 )]);
        }
    }
;

bioOffice: Room
    sDesc = "Bio Office"
    lDesc = "You are in the Biology Office. A large desk dominates the
     room. The exit is west. "
    west = bioHall2
;

bioDesk: Fixture, Surface
    noun = 'desk'
    adjective = 'large'
    location = bioOffice
    sDesc = "desk"
;

bioCabinet: Fixture, OpenableContainer
    noun = 'cabinet'
    location = bioLab
    sDesc = "cabinet"
    isOpen = nil
;

class ChemItem: Item
    location = bioCabinet
    noun = 'chemical'
    plural = 'chemicals'
    aDesc = { "some "; self.sDesc; }
    lDesc = {
        "It's a small lump of goo. The only identification is
         a label reading \""; self.sDesc; ".\" ";
    }
;
    
gfxq3: ChemItem
    noun = 'GF-XQ3'
    sDesc = "GF-XQ3"
;

gfxq9: ChemItem
    noun = 'GF-XQ9'
    sDesc = "GF-XQ9"
;

polyRed: ChemItem
    noun = 'red'
    adjective = 'poly'
    sDesc = "Poly Red"
;

polyBlue: ChemItem
    noun = 'blue'
    adjective = 'poly'
    sDesc = "Poly Blue"
;

compoundT99: ChemItem
    noun = 't99'
    adjective = 'compound'
    sDesc = "Compound T99"
;

compoundT30: ChemItem
    noun = 't30'
    adjective = 'compound'
    sDesc = "Compound T30"
;

cloneMaster: Container
    noun = 'master' 'cloneMaster'
    adjective = 'clone'
    location = bioBench
    sDesc = "CloneMaster"
    lDesc = {
        "The CloneMaster is a simple machine. It consists of a button
        marked \"Clone,\" and a small receptacle. ";
        cloneRecept.lDesc;
    }
    ioPutIn(actor, dobj) = { cloneRecept.ioPutIn(actor, dobj); }
    verDoTakeOut(actor, iobj) = { cloneRecept.verIoTakeOut(actor, iobj); }
;

cloneRecept: Fixture, Container
    location = cloneMaster
    noun = 'receptacle'
    sDesc = "receptacle"
    lDesc = {
        "The receptacle is somewhat like that of a kitchen blender. ";
        pass lDesc;
    }
;

cloneButton: ButtonItem
    location = cloneMaster
    sDesc = "clone button"
    adjective = 'clone'
    doPush(actor) = {
        if (slime.location == cloneRecept) {
            "The CloneMaster clicks and whirs for several seconds. ";
            if (gfxq3.location == cloneRecept &&
             polyBlue.location == cloneRecept &&
             compoundT99.location == cloneRecept
             and length(cloneRecept.contents) == 4) {
                "A monstrous female version (again, don't ask how you know
                 it's female) leaps from the tiny machine";
                if (parserGetMe().location == bioCreature.location) {
                    ". The two monsters look at each other with passion in
                    their mutant eyes. They run to each other with open arms,
                    seemingly in slow motion. They embrace, engage in some
                    mushy behavior, and then run away together to elope. ";
                    bioCreature.moveInto(nil);
                    unnotify(bioCreature, #menace);
                } else
                    ", looks around, and, seeing nothing of
                    interest, runs off. ";
            } else
                "An exact duplicate of the monstrous creature whose slime
                you placed in the CloneMaster leaps forth from the tiny
                machine. He looks at you menacingly for a moment, then
                runs off into the distance, never to be seen again. ";
                
            slime.moveInto(nil);
        } else "Nothing happens. ";
    }
;

omega: Treasure
    sDesc = "Great Seal of the Omega"
    noun = 'seal' 'omega' 'stamp'
    adjective = 'great' 'rubber'
    location = bioDesk
    lDesc = "The Great Seal is a large rubber stamp, about an inch and
     a half square. The stamp consists of a large circle that is filled by a
     giant capital omega, under which is the word \"Approved.\" Around the
     outside of the circle, the words \"The Great Seal of the Omega\" are
     inscribed. "
;

rope2: FixedItem
    sDesc = "rope"
    noun = 'rope'
    location = pitBottom
    lDesc = "The rope extends from the opening in the ceiling high above. "
    verDoTake(actor) = {
        "The rope seems to be tied to something from above (which would
        probably explain why it's extending up to the ceiling in the
        first place, you deduce in a rare moment of lucid thinking). ";
    }
    verDoClimbup(actor) = {}
    doClimbup(actor) = { self.doClimb(actor); }
    verDoClimb(actor) = {}
    doClimb(actor) = {
        "It's a long climb, but %you% somehow manage%s% it.\b";
        actor.travelTo(pitTop);
    }
;
