/*
 * WorldAct.t
 * Copyright (c) 1999 The Children of Dana
 * Written by David Haire
 *
 * This file contains the standard actor classes used in WorldClass.
 *
 */

/*
 * The general Actor class
 *
 * Set travelhook to a list of the form
 *
 *  [object1 &method1 object2 &method2 ... objectn &methodn]
 *
 * and the travelto method will call 
 *
 *  object1.(method1)(self, oldloc, oldloctype, newloc, newloctype)
 *  object2.(method2)(self, oldloc, oldloctype, newloc, newloctype)
 *  ...
 *  objectn.(methodn)(self, oldloc, oldloctype, newloc, newloctype)
 *
 * in order, where oldloc is the actor's location prior to the travel, and 
 *   newloc is the actor's location after the travel. oldloctype and 
 *   newloctype are the location types ('in', 'on', etc.) for the old and 
 *   new locations respectively. This works properly with obstacles too. 
 *   Note that these actually get called from inside the newlocation 
 *   method.
 */
class Actor: Fixture, Sensor, Topic, Container
  isactor = true
  position = 'standing' /* standing up by default */

  /*
   * Don't normally check for any deaths due to lack of food, drink, 
   *   sleep, etc.
   */
  starvationcheck = nil
  turnsuntilstarvation = 0
  dehydrationcheck = nil
  turnsuntildehydration = 0
  sleepcheck = nil
  turnsuntilsleep = 0

  /*
   * Give the Actor a default weight and bulk, to avoid problems with
   *   verifyinsertion.
   */
  weight = 1
  bulk = 1
  maxweight = 20
  maxbulk = 20

  /*
   * If the following flag is set, this actor knows about everything 
   *   the player actor (Me) knows about, and nothing more.  
   *
   * If the flag is not set, only things for which obj.isknownto(actor) 
   *   returns true will be known. This means you'll have to do lots of 
   *   explicit makeknownto's, which is almost never actually worth it.
   */
  knowsall = nil

  actordesc = { "\^<<self.subjadesc>> <<self.is>> here."; }

  /*
   * roomCheck is true if the verb is valid in the room. This is a 
   *   first pass.
   */
  roomCheck(v) = { return true; }

  fmtYou = { self.subjprodesc; }
  reflexivedesc = {
    if (self.isplural) "themselves";
    else "<<self.objprodesc(nil)>>self";
  }
  aamessage(v, d, p, i) = {
    "\^<<self.subjthedesc>> <<self.doesnt>> seem interested.";
  }
  actorAction(v, d, p, i) = {
    self.aamessage(v, d, p ,i);
    exit;
  }

  /*
   * No travelto hooks by default
   */
  travelhook = []

  /*
   * Move the actor into the given room.
   *
   * If the room parameter is actually an obstacle like a door, recurse
   *   on the destination behind the obstacle.
   *
   * If any other actors are present in the room the actor is moving 
   *   into, make the actor and everything the actor is carrying known 
   *   to those actors.
   */
  travelto(room) = {
    if (room = nil) {
      "You can't travel to nowhere!";     
      return;
    }
    else {
      local i;
      if (room.isobstacle) {
        local dest;
        dest := room.destination(self);
        if (dest <> nil) self.travelto(dest);
        return;
      }
      if (self.position <> 'standing') {
        "(\^<<self.subjprodesc>> <<standVerb.desc(self)>> up)\b";
        self.position := 'standing';
      }
      self.newlocation(room, 'in');
      for (i := 1; i <= length(global.actorlist); i++) {
        local o := global.actorlist[i];
        if (o = self) continue;
        if (o.cansee(self, nil, true))
          self.makeknownto(o);
        if (o.canseecontents(self, true, 'in'))
          self.makecontentsknownto(o);
      }
    }
  }

  /*
   * Do all the things we need to do when the actor is in a new
   *   location, but don't do travel-related things. This one also
   *   allows us to specify a location type.
   *
   * If the room parameter is actually an obstacle like a door, recurse
   *   on the destination behind the obstacle.
   *
   * Any travelhooks that exist for this actor are called after the
   *   actor is moved to the new location, but before the new location
   *   is described.
   */
  newlocation(room, loctype) = {
    local oldloc, oldloctype, i, l;
    if (room.isobstacle) {
      local dest;
      dest := room.destination(self);
      if (dest <> nil)
        self.newlocation(dest, loctype);
      return;
    }
    if (self.location <> nil)
      self.location.leave(self);
    oldloc := self.location;
    oldloctype := self.locationtype;
    self.moveto(room, loctype);
    l := self.travelhook;
    for (i := 1; i <= length(l); i += 2)
      l[i].(l[i + 1])(self, oldloc, oldloctype, room, loctype);
    room.enter(self);
  }     
    
  /*
   * Actors can hear things.
   */
  islistener(actor) = { return true; }

  /*
   * This method is called when the actor gets a speech command, like
   *
   *  floyd, say "hello"
   *
   * Player inherits this. This is called by strObj.doSay.
   */
  speech_handler(s) = {
    "\^<<self.subjthedesc>> <<sayVerb.desc(self)>> \"<<s>>\".";
  }

  /*
   * The following method is called by die() when the actor is killed.
   */
  diemessage = {
    "*** \^<<self.subjthedesc>> <<self.has>> died ***";
  }

  /*
   * Message methods for doTake.
   */
  weightexceeded(dobj, loctype) = {
    "\^<<self.possessivedesc>> load is too heavy.";
  }
  bulkexceeded(dobj, loctype) = {
    "\^<<self.subjthedesc>> can't carry any more.";
  }

  /*
   * Override default container messages (actors are containers, but you 
   *   can't put things in them or take things out of them...)
   */
  verDoLookin(actor) = {
    "\^<<actor.subjthedesc>> can't look inside <<self.objthedesc(actor)>>.";
  }
  verIoPutin(actor) = { 
    if (actor = self) 
      "\^<<actor.subjthedesc>> can't put things in <<self.objthedesc(actor)>>.";
    else
      self.verIoGiveto(actor); 
  }
  ioPutin(actor, dobj) = { self.ioGiveto(actor, dobj); }
  verIoTakeout(actor) = { 
    if (actor = self)
      "\^<<actor.subjthedesc>> can't take things out of <<self.objthedesc(actor)>>.";
    else
      self.verIoTakefrom(actor); 
  }
  ioTakeout(actor, dobj) = { self.ioTakefrom(actor, dobj); }

  /*
   * Default interaction messages. Actors can be shown things, asked 
   *   about things, or have things thrown at them. By default, 'throw 
   *   at' and 'throw to' work the same as 'give to'.
   */
  verIoShowto(actor) = {
    if (actor = self)
      "\^<<actor.subjthedesc>> can't show things to <<self.objthedesc(actor)>>.";
  }
  ioShowto(actor, dobj) = {
    "\^<<self.subjthedesc>> <<self.doesnt>> seem interested in <<dobj.objthedesc(actor)>>.";
  }
  verDoAskabout(actor, io) = {
    "\^<<self.subjthedesc>> <<self.doesnt>> have much to say about <<io.objthedesc(self)>>.";
  }
  verIoThrowat(actor) = { 
    if (actor = self)
      "\^<<actor.subjthedesc>> can't throw things at <<self.objthedesc(actor)>>.";
    else
      self.verIoGiveto(actor); 
  }
  ioThrowat(actor, dobj) = { self.ioGiveto(actor, dobj); }
  verIoThrowto(actor) = { 
    if (actor = self)
      "\^<<actor.subjthedesc>> can't throw things to <<self.objthedesc(actor)>>.";
    else
      self.verIoGiveto(actor); 
  }
  ioThrowto(actor, dobj) = { self.ioGiveto(actor, dobj); }

  /*
   * By default, things can be given to actors, but cannot be taken from them.
   */
  verIoTakefrom(actor) = {
    if (actor = self)
      "\^<<actor.subjthedesc>> can't take things from <<self.objthedesc(actor)>>.";
  }
  ioTakefrom(actor, dobj) = {
    "\^<<self.subjthedesc>> <<self.has>> <<dobj.objthedesc(actor)>> and won't let <<actor.objprodesc(self)>> have <<dobj.objprodesc(self)>>.";
  }
  acceptsput(actor, loctype) = { return true; }
  passgen(actor, obj, loctype, passmethod) = {
    "\^<<actor.youll>> have to convince <<self.objthedesc(nil)>> to give ";
    if (obj = nil) "that to <<actor.objprodesc(nil)>>";
    else "<<actor.objprodesc(nil)>> <<obj.objthedesc(nil)>>";
    " first.";
    return nil;
  }
  passcantouchin(actor, obj, loctype) = { 
    return self.passgen(actor, obj, loctype, &passcantouch);
  }
  passcantakein(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcantake);
  }

  /*
   * Generic message for 'inventory'.
   */
  inventory = {
    "\^<<self.subjthedesc>> <<inventoryVerb.desc(self)>> <<self.possessivedesc>> possessions. ";
  }
;

/*
 * itemActors are just like Actors except that they can be taken (i.e.,
 *   they are Items). The itemActor's isactor method is only true (and
 *   hence the actordesc is only used) when the actor's location is a Room.
 */
class itemActor: Item, Actor
  isactor = {
    if (isclass(self.location, Room))
      return true;
    else
      return nil;
  }
  isfixture = nil
;

/*
 * Follower
 *
 * This is a special object that can "shadow" the movements of a character
 *   as it moves from room to room. The purpose of a follower is to allow
 *   the player to follow an actor as it leaves a room by typing a "follow"
 *   command. Each actor that is to be followed must have its own follower
 *   object. The follower object should define all of the same vocabulary
 *   words (nouns and adjectives) as the actual actor to which it refers.
 *   The follower must also define the myactor property to be the Actor
 *   object that the follower follows. The follower will always stay one 
 *   room behind the character it follows; no commands are effective with a
 *   follower except for "follow."
 */
class Follower: Actor
  sdesc = { self.myactor.sdesc; }
  isfollower = true
  ldesc = { caps(); self.thedesc; " is no longer here. "; }
  actorAction(v, d, p, i) = { self.ldesc; exit; }
  actordesc = {}
  myactor = nil
  verDoFollow(actor) = {}
  doFollow(actor) = { actor.travelto(self.myactor.location); }
  dobjGen(a, v, i, p) = {
    if (v <> followVerb) {
      "\^<< self.myactor.thedesc >> is no longer here.";
      exit;
    }
  }
  iobjGen(a, v, d, p) = {
    "\^<< self.myactor.thedesc >> is no longer here.";
    exit;
  }
;

/*
 * Classes to inherit from to mark gender
 */
class Male: Thing
  isHim = true
  youre = {
    if (self.isplural) "they're";
    else "he's";
  }
  subjprodesc = {
    if (self.isplural) "they";
    else "he";
  }
  objprodesc(actor) = {
    if (actor = self) self.reflexivedesc;
    else {
      if (self.isplural) "them";
      else "him";
    }
  }
  possessivedesc = {
    if (self.isplural) "their";
    else "his";
  }
;
class Female: Thing
  isHer = true
  youre = {
    if (self.isplural) "they're";
    else "she's";
  }
  subjprodesc = {
    if (self.isplural) "they";
    else "she";
  }
  objprodesc(actor) = {
    if (actor = self) self.reflexivedesc;
    else {
      if (self.isplural) "them";
      else "her";
    }
  }
  possessivedesc = {
    if (self.isplural) "their";
    else "her";
  }
;

/*
 * realActor. A alightly more realistic actor, which can be given items to 
 *   carry. If takesall is set to true, the actor can be given any item. If 
 *   takesall is set to nil, the actor can only be given items that are in 
 *   the actor's takelist.
 */
class realActor: Actor
  takesall = true
  takelist = []
  verIoGiveto(actor) = {
    if (actor = self)
      "\^<<actor.subjthedesc>> can't give things to <<self.objthedesc(actor)>>.";
  }
  ioGiveto(actor, dobj) = {
    if (self.takesall or find(self.takelist, dobj)) {
      Outhide(true);
      self.ioPutX(self, dobj, 'in');
      Outhide(nil);
      if (dobj.isin(self)) {
        "\^<<actor.subjthedesc>> <<giveVerb.desc(actor)>> <<dobj.objthedesc(actor)>> to <<self.objthedesc(actor)>>.";
      }
      else self.ioPutX(self, dobj, 'in');
    }
    else
      "\^<<self.subjthedesc>> <<self.doesnt>> want <<dobj.objthedesc(self)>>.";
  }
;

/*
 * humanActor. An actor with slightly more human-like responses. These 
 *   actors can be asked about things.
 */
class humanActor: realActor
  actorAction(v, d, p, i) = {
    if (isclass(v, Systemverb)) {
      "\^<<self.subjthedesc>> says \"Do it yourself!\"'\n";
      exit;
    }
    if (v <> helloVerb) {
      self.aamessage(v, d, p ,i);
      exit;
    }
  }

  /*
   * Asking an actor about something is ok. Each actor has a property 
   *   called 'askme' which is a pointer to another method.
   */
  verIoAskabout(actor) = {}
  verDoAskabout(actor, io) = {
    local c;
    c := self.isatopic(io, nil, true);
    if (not c)
      self.isatopic(io, nil, nil);
  }

  doAskabout(a, io) = {

    /*
     * Look for property-pointer in actor.askme. If it is of type 
     *   "property-pointer", then we consider that this actor can 
     *   be asked. If what it points to is a "string" or a {method} 
     *   then we invoke it. If it's a 'string', we print it in 
     *   quotes.
     */
    if (datatype(self.askme) = 13) {
      switch (proptype(io, self.askme)) {
      case 3:
        "\^<<self.subjthedesc>> says \"<<io.(self.askme)>>\".\n";
        return;
      case 6:
      case 9:
        io.(self.askme);
        return;
      }
    }

    /*
     * If it is not, then we look for a generic method (askAnybody) 
     *   instead.
     */
    switch (proptype(io, &askAnybody)) {
    case 3:
      "\^<<self.subjthedesc>> says \"<<io.askAnybody(self)>>\".\n";
      return;
    case 6:
    case 9:
      io.askAnybody(self);
      return;
    }

    /*
     * If there was a bad pointer or the answer is not an 
     *   acceptable method, we bail out with a general message.
     */
    "\^<<self.subjthedesc>> <<self.doesnt>> have much to say about <<io.objthedesc(self)>>.";
  }
  verIoSayto(actor) = { }
  ioSayto(actor, dobj) = { dobj.doSayto(actor, self); }
;

/*
 * animalActor. An actor that cannot talk.
 */
class animalActor: realActor
  verDoAskabout(actor, io) = {
    "<<self.subjthedesc>> looks at <<actor.objthedesc(self)>>, puzzled.";
  }
  verIoSayto(actor) = { }
  ioSayto(actor, dobj) = { dobj.doSayto(actor, self); }
;

/*
 * Mover: New actor class which contains two methods: movevalid and
 *   getdestination. Both will be passed in a direction string (one of
 *   those contained in the dirstringlist). movevalid will check to see if
 *   the direction is passible. It does this by calling the verGo<method>
 *   with the actor to see if any text is printed. If it is, it will be
 *   assumed that the direction is obviously impassible (as in a brick wall
 *   offering no means of passage), and nil will be returned, or otherwise
 *   true to indicate that the direction is passible. Should you want a
 *   direction to be conditional passible, it uses a diroklist, which
 *   should contain of list of direction strings for those locations you
 *   wish to be considered "possible exits". getdestination checks with
 *   movevalid and the appropriate verGo<method> (if it finds one present)
 *   and passes out the destination if a valid destination exists. By
 *   calling movevalid you can determine whether a given direction has a
 *   possible exit. Then getdestination can be called to get the
 *   destination if the movement in that direction by the actor is valid.
 */
class Mover: Actor
  diroklist = []

  leavemessage(dir) = {
    "\n\^<<self.subjthedesc>> leave<<self.isplural ? "" : "s">> to the <<dir>>.";
  }
  entermessage(dir) = {
    "\n\^<<self.subjthedesc>> enter<<self.isplural ? "" : "s">> from the <<dir>>.";
  }
  attemptedleave(dir) = {
    "\n\^<,self.subjthedesc>> starts going to the <<dir>>. ";
  }

  movevalid(direction) = {
    local dir := find(global.dirstringlist, direction);
    local dest;

    /*
     * Check to see if the given direction is in the dirok list. If
     *   it is, then return true immediately.
     */
    if (find(self.diroklist, direction))
      return true;

    /*
     * Check to see if a verGo<dir> method exists, and if so call
     *   it to see if it outputs anything. If it does, the test
     *   fails.
     */
    if (defined(self.location, global.verdirlist[dir])) {
      Outhide(true);
      self.location.(global.verdirlist[dir])(self);
      if (Outhide(nil)) return nil;
    }

    /*
     * Check to see if a go<dir> is defined, then get the
     *   destination, remembering that it can be both a method or
     *   set to an object. Then check to see if destination is not
     *   nil, and if so output true.
     */
    if (defined(self.location, global.dirlist[dir])) {
      if (proptype(self.location, global.dirlist[dir])=6)
        dest := self.location.(global.dirlist[dir])(self);
      else
        dest := self.location.(global.dirlist[dir]);
      if (dest) return true; else return nil;
    }
    else {
      if (defined(self.location, global.olddirlist[dir]))
        return true;
      else
        return nil;
    }
  }

  getdestination(direction) = {
    local dir := find(global.dirstringlist, direction);

    /*
     * Check to see if a verGo<dir> method exists, and if so call
     *   it to see if it outputs anything. If it does, the test
     *   fails.
     */
    if (defined(self.location, global.verdirlist[dir])) {
      Outhide(true);
      self.location.(global.verdirlist[dir])(self);
      if (Outhide(nil)) return nil;
    }

    /*
     * Check to see if a go<dir> is defined, then get the
     *   destination, remembering that it can be both a method or
     *   set to an object. Then check to see if destination is not
     *   nil, and if so output true.
     */
    if (defined(self.location, global.dirlist[dir])) {
      if (proptype(self.location, global.dirlist[dir])=6)
        return self.location.(global.dirlist[dir])(self);
      else
        return self.location.(global.dirlist[dir]);
    }
    else {
      if (defined(self.location, global.olddirlist[dir]))
        return self.location.(global.olddirlist[dir]);
      else
        return nil;
    }
  }
;

/*
 * The player class is strictly for the main character in a game with only
 *   one player-controlled actor, as is typical.
 *
 * The isme property determines whether first-person syntax ('I', 'me',
 *   etc.) or third-person syntax ('You') is used.
 *
 * Note that the Player is assumed to be capable of speech, so it inherits 
 *   from humanActor. It also inherits from Mover to utilise the 'movevalid' 
 *   and 'getdestination' methods. This is to assist in games with multiple 
 *   player-controlled actors, all of whom should be defined as Players.
 */
class Player: Mover, humanActor
  incontentslisted(obj) = { return nil; }
  isplural = true                       /* remember, this is only syntactic! */
  isdetermined = true
  isme = nil
  is = {
    if (isme) "am";
    else "are";
  }
  isnt = {
    if (isme) "am not";
    else "aren't";
  }
  does = "do"
  doesnt = "don't"
  has = "have"
  doesnthave = "don't have"
  youre = {
    if (isme) "I'm";
    else "you're";
  }
  sdesc = "<<self.isme ? "I" : "you">>"
  thedesc = "<<self.isme ? "I" : "you">>"
  adesc = "<<self.isme ? "I" : "you">>"
  ldesc = {
    "\^<<self.subjthedesc>> <<self.is>> <<self.position>>";
    if (self.position = 'lying' or self.position = 'sitting')
      " down.";
    else
      ".";
  }
  subjsdesc = self.sdesc
  subjthedesc = self.thedesc
  subjadesc = self.adesc
  subjprodesc = "<<self.isme ? "I" : "you">>"
  objsdesc(actor) = "<<actor = self ? self.reflexivedesc : self.sdesc>>"
  objthedesc(actor) = "<<actor = self ? self.reflexivedesc : self.thedesc>>"
  objadesc(actor) = self.adesc
  objprodesc(actor) = {
    if (actor = self) self.reflexivedesc;
    else "<<self.isme ? "me" : "you">>";
  }
  possessivedesc = "<<self.isme ? "my" : "your">>"
  reflexivedesc = "<<self.isme ? "myself" : "yourself">>"

  actorAction(v, d, p, i) = {
    if (parserGetMe() <> self) {
      if (isclass(v, Systemverb)) {
        "\^<<self.subjthedesc>> says \"Do it yourself!\"'\n";
        exit;
      }
      if (v <> helloVerb) {
        self.aamessage(v, d, p ,i);
        exit;
      }
    }
  }
  putmessage(dobj, loctype) = { "Taken."; }
  roomAction(a, v, d, p, i) = {
    if (self.location <> nil)
      self.location.roomAction(a, v, d, p, i);
  }

  /*
   * Inventory.
   */
  inventory = {
    local tot, locn;
    caps();
    if (not self.cansee(self, nil, nil))
      return;
    locn := [];
    tot := listcontents(self, self, nil, nil, nil, nil, true, true, nil, locn);
    if (tot = 0)
      "\^<<self.subjthedesc>> <<self.doesnt>> appear to be carrying anything.";
  }

  /*
   * Check for starvation, dehydration, and death due to lack of sleep.
   *
   * starvationmessage gets passed the number of turns remaining until
   *   the actor dies of malnutrition. It should print messages based
   *   on the turn count to warn the player of the impending death.
   *   This method must at the very least tell the player the actor has
   *   dies when t = 0 because  starvationcheck calls die() when this
   *   is the case.
   *
   * dehydrationmessage and sleepmessage do the same for the
   *   other checks.
   *
   * When the player collapses from lack of sleep we call self.sleep.
   *   By default this doesn't do anything. It could change the turn
   *   counter, cause the player to be killed, make him lose some of
   *   his possessions, etc.
   */
  mealtime = 240
  starvationcheck = {
    local t;
    self.turnsuntilstarvation--;
    t := self.turnsuntilstarvation;
    self.starvationmessage(t);
    if (t < 1)
      die();
  }
  starvationmessage(t) = {
    switch (t) {
      case 0:
        P(); I();
        "\^<<self.subjthedesc>> simply can't go on any longer without food. \^<<self.subprodesc>> perish<<self.isplural ? "" : "es">> from lack of nutrition. ";
        break;
      case 5:
      case 10:
        P(); I();
        "\^<<self.subjthedesc>> can't go much longer without food. ";
        break;
      case 15:
      case 20:
      case 25:
        P(); I();
        "\^<<self.subjthedesc>> <<self.is>> feeling very hungry. \^<<self.subjprodesc>> better find some food soon.";
        break;
      case 30:
      case 35:
        P(); I();
        "\^<<self.subjthedesc>> <<self.is>> feeling a bit peckish. Perhaps it would be a good idea to find something to eat.";
        break;
    }
  }
  drinktime = 140
  dehydrationcheck = {
    local t;
    self.turnsuntildehydration--;
    t := self.turnsuntildehydration;
    self.dehydrationmessage(t);
    if (t < 1)
      die();
  }
  dehydrationmessage(t) = {
    switch (t) {
      case 0:
        P(); I();
        "\^<<self.subjthedesc>> simply can't go on any longer without water. \^<<self.subprodesc>> die<<self.isplural ? "" : "s">> of thirst.";
        break;
      case 5:
      case 10:
        P(); I();
        "\^<<self.subjthedesc>> can't go much longer without water. ";
        break;
      case 15:
      case 20:
      case 25:
        P(); I();
        "\^<<self.subjthedesc>> <<self.is>> feeling very thirsty. \^<<self.subjprodesc>> better find something to drink soon.";
        break;
      case 30:
      case 35:
        P(); I();
        "\^<<self.subjthedesc>> <<self.is>> feeling a bit thirsty. Perhaps it would be a good idea to find something to drink.";
        break;
    }
  }
  sleeptime = 440
  sleepcheck = {
    local t;
    self.turnsuntilsleep--;
    t := self.turnsuntilsleep;
    self.sleepmessage(t);
    if (t < 1) {
      self.sleep;
    }
  }
  sleep = {
    P(); I(); "\^<<self.subjthedesc>> wake<<self.isplural ? "" : "s">> up later with a headache.";
    self.turnsuntilsleep := self.sleeptime;
  }
  sleepmessage(t) = {
    switch (t) {
      case 0:
        P(); I();
        "\^<<self.subjthedesc>> pass<<self.isplural ? "" : "es">> out from exhaustion.";
        break;
      case 5:
      case 10:
        P(); I();
        "\^<<self.subjthedesc>> <<self.is>> so tired, <<self.subjprodesc>> can barely keep <<self.possessivedesc>> eyes open.";
        break;
      case 15:
      case 20:
      case 25:
        P(); I();
        "\^<<self.subjthedesc>> <<self.is>> feeling very sleepy. \^<<self.subjprodesc>> better find a place to rest soon.";
        break;
      case 30:
      case 35:
        P(); I();
        "\^<<self.subjthedesc>> <<self.is>> feeling a bit sleepy. Perhaps it would be a good idea to find a place to sleep.";
        break;
    }
  }

  /*
   * Normally, an Actor does not allow things to be removed from
   *   itself. This is to prevent the player from taking stuff that
   *   actors are carrying. However, we assume that an Actor will never
   *   try to take something from a Player unless this is a result of a
   *   player to actor command like
   *
   *  >salesperson, take credit card [from me]
   *
   * We want this to either work (i.e., get mapped to
   *
   *  >give credit card to saleperson
   *
   * or print a helpful eror message. Since the first option is quite
   *   difficult to get right, we'll take the easy way out and just
   *   print an informative message.
   */
  passgen(actor, obj, loctype, passmethod) = {
    "If you want to give another character access to <<self.possessivedesc>> belongings, you'll have to be more direct about it. For example, \"give item to actor\".";
    return nil;
  }
  passcantouchin(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcantouch);
  }
  passcantakein(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcantake);
  }

  verIoAskabout(actor) = { }
  verDoAskabout(actor, io) = {
    if (actor = self)
      "Talking to yourself is fine. Expecting answers is  weird!";
    else {
      local c;
      c := self.isatopic(io, nil, true);
      if (not c)
        self.isatopic(io, nil, nil);
    }
  }
  verIoSayto(actor) = { }
;

/*
 * Wanderer : An actor that can wander about at will. By default, a
 *   wanderer will automatically avoid an exit which prints a verGo<method>
 *   message, but you can easily set a room's diroklist to make the
 *   wanderers try to exit. As with Everywhere, we'll define a roomprop
 *   which can contain a property pointer to a property which must be set
 *   for any room the actor is to wander into. Thus you can limit an actor
 *   to only certain rooms. We'll also define a pauseturns variable, which
 *   indicates the number of turns the actor is to pause in each room for.
 *   The Wanderer implements an easily customizable wandering system. The
 *   wanderer will wander anywhere (unless it's roomprop is set, in which
 *   case it is limited to those specified rooms). It automatically
 *   distinguishes, with help from the Mover class, if a direction is
 *   possible, and should a direction be seemingly possible but not
 *   actually possible to that particular actor, it has a custom message as
 *   well. The responses when the actor enters, leaves, or attempts to
 *   leave the current room are contained in separate methods for ease of
 *   customisation. In the case of attemptedleave, the method is first
 *   called, and then the verGo<method>, which should print out a response
 *   as to why the actor could not go in that direction.
 *
 * If the method 'departure' is defined within the Wandered, this method will be 
 *   called before the Wanderer moves to a new location.
 *
 * If the method 'arrival' is defined within the Wanderer, this method will be 
 *   called after the Wanderer has moved to a new location.
 */
class Wanderer: Mover
  roomprop = nil
  pauseturns = 3
  pausectr = 0
  moveDaemon = {
    local tries;
    pausectr += global.turnspertick;
    if (pausectr <= pauseturns) return;
    for (tries:=1; tries<50; tries++) {
      local dno := rnd(8);
      local dir := global.dirstringlist[dno];
      local opp := global.dirstringlist[9 - dno];
      local seen, dest;
      if (not self.movevalid(dir)) return;
      seen := parserGetMe().cansee(self.location, self.locationtype, true);
      dest := self.getdestination(dir);
      if (dest = nil) {
        if (seen) {
          self.attemptedleave(dir);
          self.location.(global.verdirlist[dno])(self);
        }
        return;
      }
      else {
        if (not self.roomprop or (self.roomprop and dest.(self.roomprop))) {
          if (defined(self, &departure)) self.departure(self.location, self.locationtype);
          if (seen) self.leavemessage(dir);
          Outhide(true);
          self.travelto(dest);
          Outhide(nil);
          seen := parserGetMe().cansee(self.location, self.locationtype, true);
          if (seen) self.entermessage(opp);
          pausectr := 0;
          if (defined(self, &arrival)) self.arrival(self.location, self.locationtype);
          exitobj;
        }
      }
    }
  }
;

/*
 * trackActor: An actor that provides an implementation for track movement
 *   of an actor. The track should be provided in the derived actor's track
 *   list, in the form of strings of the direction to move in at each
 *   stage. As with the Wanderer class, pauseturns should be set to the
 *   number of turns the actor is to remain in each room. You then need
 *   only to add on default actor characteristics, and the actor will
 *   happily go around his specified track.
 *
 * As with the Wanderer class, the methods 'departure' and 'arrival' will, if 
 *   defined, be called before and after the trackActor has moved to a new 
 *   location.
 */
class trackActor: Mover
  track = []
  pauseturns = 3
  pausectr = 0
  trackctr = 1
  moveDaemon = {
    pausectr += global.turnspertick;
    if (pausectr <= pauseturns) return;
    if (length(self.track) = 0) {
      pausectr := 0;
      return;
    }
    else {
      local dir := self.track[trackctr];
      local dno := find(global.dirstringlist, dir);
      local opp := global.dirstringlist[9 - dno];
      local seen, dest;
      if (not self.movevalid(dir)) return;
      seen := parserGetMe().cansee(self.location, self.locationtype, true);
      dest := self.getdestination(dir);
      if (dest = nil) {
        if (seen) {
          self.attemptedleave(dir);
          self.location.(global.verdirlist[dno])(self);
        }
        return;
      }
      else {
        if (not self.roomprop or (self.roomprop and dest.(self.roomprop))) {
          if (defined(self, &departure)) self.departure(self.location, self.locationtype);
          if (seen) self.leavemessage(dir);
          Outhide(true);
          self.travelto(dest);
          Outhide(nil);
          seen := parserGetMe().cansee(self.location, self.locationtype, true);
          if (seen) self.entermessage(opp);
          pausectr := 0;
          trackctr += 1;
          if (trackctr > length(track)) trackctr := 1;
          if (defined(self, &arrival)) self.arrival(self.location, self.locationtype);
          exitobj;
        }
      }
    }
  }
;

/*
 * Controllable : An actor that the player can control during the game.
 *
 * As with the Wanderer class, the methods 'departure' and 'arrival' will, if 
 *   defined, be called before and after the Controllable has moved to a new 
 *   location.
 */
class Controllable: Mover
  actorAction(v, d, p, i) = {
    if (isclass(v, Travelverb)) {
      local dir := global.dirstringlist[find(global.dirlist, v.doprop)];
      local dno := find(global.dirstringlist, dir);
      local opp := global.dirstringlist[9 - dno];
      local seen, dest;
      if (not self.movevalid(dir)) return;
      seen := parserGetMe().cansee(self.location, self.locationtype, true);
      dest := self.getdestination(dir);
      if (dest = nil) {
        if (seen) {
          self.attemptedleave(dir);
          self.location.(global.verdirlist[dno])(self);
        }
        return;
      }
      else {
        if (not self.roomprop or (self.roomprop and dest.(self.roomprop))) {
          if (defined(self, &departure)) self.departure(self.location, self.locationtype);
          if (seen) self.leavemessage(dir);
          Outhide(true);
          self.travelto(dest);
          Outhide(nil);
          seen := parserGetMe().cansee(self.location, self.locationtype, true);
          if (seen) self.entermessage(opp);
          if (defined(self, &arrival)) self.arrival(self.location, self.locationtype);
          exitobj;
        }
      }
    }
  }
;

/*
 * talkingActor. A humanActor that it is possible to have conversations 
 *   with using the syntax "talk to x". This elicits a response and a list 
 *   of further conversation items, presented in a multiple-choice format.
 *
 * To use this, simply define a method and property as follows:
 * 
 *   x = ['opening', &method, 'response 1', condition1, &method1, ...]
 *   openmessage = &x
 *
 * 'opening' is the actor's initial response.
 * &method is a method that is called once a valid response has been 
 *   entered by the player, or nil.
 * 'response 1', ... are the possible further conversation items that will 
 *   be presented to the player as a response to what this actor said.
 * condition1, ... are the conditions to be checked to see if the 
 *   corresponding response should be presented as an option to the player, 
 *   or true if the response should always be available.
 * &method1, ... are the methods to be called when the player selects the 
 *   corresponding response. This can either be a simple string (in which 
 *   case the conversation ends), or a list in the same format as above (in 
 *   which case the conversation continues).
 *
 * openmessage should be a pointer to the method that will be called when 
 *   the player executes the 'talk to' command.
 *
 * This routine can also be used elsewhere, by calling the method 
 *   'talkative' and passing a pointer to the initial conversation list. 
 *   For example:
 *
 *  spade: Item
 *  ...
 *    ioAskabout(actor, dobj) = {
 *      if (dobj = fred)
 *        fred.talkative(&spadetalk);
 *      else pass ioAskabout;
 *    }
 *  ;
 */
class talkingActor: humanActor
  verDoTalk(actor) = { }
  doTalk(actor) = {
    if (length(self.(self.openmessage)) > 0) {
      self.talkative(self.openmessage);
    }
    else {
      "<<actor.subjthedesc>> <<actor.doesnt>> feel like  talking. ";
    }
  }
  openmessage = &greet
  greet = []
  talkative(lines) = {
    local TranslationList := [];
    local NumChoices := 0, TotNum;
    local n := 0, t, response, respnum, newlines;
    TotNum := ((length(self.(lines))-2)/3);
    "\b"; say (self.(lines)[1]);
    if (self.(lines)[2] = nil) t := 0;
    else t := proptype(self, self.(lines)[2]);
    if (TotNum <> 0) {
      "<hr>";
      NumChoices := 0;
      while (n <> TotNum) {
        n++;
        if (self.(lines)[(3*n) + 1]) {
          ++NumChoices;
          "\n<<cvtstr(NumChoices)>>) ";
          say (self.(lines)[3*n]);
          TranslationList += n;
        }
      }
      response := 0;
      while (response <= 0 or response > NumChoices) {
        "\n# ";
        response := cvtnum(inputline());
      }
      respnum := TranslationList[response];
      if (t = 6 or t = 9) {
        "\b"; self.(self.(lines)[2]);
      }
      newlines := (self.(lines)[3*(respnum)+2]);
      self.talkative(newlines);
    }
    else if (t = 6 or t = 9) {
      "\b"; self.(self.(lines)[2]);
    }
  }
;

ddialogVersion: versionTag
  id = "DDialog: v 1.0 February 1994.\n"
  author = 'Trevor Powell'
  func = 'directed dialog'
;

actorVersion : versionTag
  id="RActor: v 1.2 April 1994.\n"
  author='Jeff Laing'
  func='real actors'
;
