/*
 * WorldCls.t
 * Copyright (c) 1999 The Children of Dana
 * Written by David Haire
 *
 * This file contains the standard object classes used in WorldClass.
 *
 */

/*
 * Initialization. An object that can be used to perform special routines 
 *   at preinit() and init() time. The method preinit_phase is called from 
 *   within preinit(), and the method init_phase is called from within 
 *   init().
 */
class Initialization: object
  preinit_phase = { }
  init_phase = { }
;

/*
 * Thing. The basic TADS object class.
 */
class Thing: object

  /*
   * WorldClass is case-sensitive.
   */
  isEquivalent = { return self.isequivalent; }

  /*
   * On dynamic construction, move into location's contents list.
   */
  construct = {
    local loc, loctype;
    loc := self.location;
    loctype := self.locationtype;
    self.location := nil;
    self.locationtype := nil;
    self.moveto(loc, loctype);
  }

  /*
   * On dynamic destruction, move into nowhere.
   */
  destruct = { self.movein(nil); }

  /*
   * TADS provides visiblity handling for us, but we'll ignore it since
   *   we want to do our own stuff instead.
   */
  isVisible(vantage) = { return true; }

  /*
   * These shouldn't ever be called for non-actors, but we'll leave
   *   them in just in case.
   */
  aamessage(v, d, p, i) = {
    "\^<<self.subjthedesc>> <<self.isnt>> likely to carry out your whims.";
  }
  actorAction(v, d, p, i) = {
    self.aamessage(v, d, p ,i);
    exit;
  }

  /*
   * Parts that have this Thing as parent
   */
  parts = []

  location = nil                                    /* this thing is nowhere. */
  locationtype = 'in'          /* this thing is 'in' its location by default. */

  /*
   * Methods to set location and locationtype. These check to make sure
   *   they're not overwriting code. (E.g. for Floating's that have
   *   method locations.)
   */
  setlocation(loc) = {
    if (proptype(self, &location) <> 6)
      self.location := loc;
  }
  setlocationtype(loctype) = {
    if (proptype(self, &locationtype) <> 6) {
      self.locationtype := loctype;
      self.locationtypenum := find(global.loctypes, loctype);
    }
  }

  /*
   * isplural specifies whether this Thing uses singular or plural
   *   syntax. Note that the player ('you') is syntactically plural.
   */
  isplural = nil

  /*
   * isdetemined specified whether this Thing requires an article ('a'
   *   or 'the').
   */
  isdetermined = nil

  /*
   * Syntax strings.
   */
  is = "<<self.isplural ? "are" : "is">>"
  isnt = "<<self.isplural ? "aren't" : "isn't">>"
  does = "<<self.isplural ? "do" : "does">>"
  doesnt = "<<self.isplural ? "don't" : "doesn't">>"
  has = "<<self.isplural ? "have" : "has">>"
  doesnthave = "<<self.doesnt>> have"
  youll = "<<self.subjprodesc>>'ll"
  youre = "<<self.isplural ? "they're" : "it's">>"
  sdesc = "<<self.isplural ? "things" : "thing">>"
  multisdesc = "\(<<self.sdesc>>\)"
  ldesc = {
    if (self.isplural)
      "\^<<self.subjprodesc>> look like ordinary <<self.objsdesc(nil)>>.";
    else
      "\^<<self.subjprodesc>> looks like an ordinary <<self.objsdesc(nil)>>.";
  }
  thedesc = {
    if (self.isdetermined) self.sdesc;
    else "the <<self.sdesc>>";
  }
  adesc = {
    if (self.isdetermined or self.isplural) self.sdesc;
    else "a <<self.sdesc>>";
  }

  /*
   * tdesc is used for Things that put text in the status line.
   */
  tdesc = self.sdesc

  /*
   * Text printed in a contents listing. This includes extra properties
   *   that print things like "(being worn)", "(providing light)", etc.
   *
   * Note that objects that must start out with certain property
   *   pointers in the properties list (like Lightsources) have to be
   *   have this done in preinit. (We want to allow for multiple
   *   inheritance when more than one inherited class has a special
   *   property, so we can't do this with initmethod.)
   */
  properties = []                               /* list of property pointers. */
  listprops = {
    local i;
    for (i := length(self.properties); i > 0; i--) {
      Outhide(true);
      self.(self.properties[i]);
      if (Outhide(nil)) { " "; self.(self.properties[i]); }
    }
  }
  listdesc = { "<<self.subjadesc>><<self.listprops>>"; }
  listpluraldesc = { "<<self.pluraldesc>><<self.listprops>>"; }

  /*
   * The description types, with case specified.
   */
  subjsdesc = self.sdesc
  subjthedesc = self.thedesc
  subjadesc = self.adesc
  subjprodesc = "<<self.isplural ? "they" : "it">>"
  objsdesc(actor) = "<<actor = self ? self.reflexivedesc : self.sdesc>>"
  objthedesc(actor) = "<<actor = self ? self.reflexivedesc : self.thedesc>>"
  objadesc(actor) = self.adesc
  objprodesc(actor) = {
    if (actor = self) self.reflexivedesc;
    else "<<self.isplural ? "them" : "it">>";
  }
  possessivedesc = "<<self.isplural ? "their" : "its">>"
  reflexivedesc = "<<self.isplural ? "themselves" : "itself">>"

  /*
   * pluraldesc is only used for indistinguishable objects. For
   *   example, two indistinguishable coins may have  sdesc = "coin",
   *   but we want to tell the player that there are "two coins here".
   *   The pluraldesc prints the words "coins" in this case. The
   *   default is just tack on an s. For words like box we'll have to
   *   take care to override this.
   */
  pluraldesc = "<<self.sdesc>>s"

  /*
   * Sense descriptions for contents listings (and room descriptions).
   *   Note lack of subject. This is filled in with
   *   <<self.subjthedesc>>, "something in the bag", etc.
   */
  listlistendesc = "<<self.isnt>> making any noise."
  listsmelldesc = "<<self.doesnt>> smell unusual."
  listtastedesc = "<<tasteVerb.desc(self)>> normal."
  listtouchdesc = "<<self.doesnt>> feel unusual."

  /*
   * These get called when an actor says "smell <Thing>" or "taste
   *   <Thing>".
   */
  listendesc = "\^<<self.subjthedesc>> <<self.listlistendesc>>"
  smelldesc = "\^<<self.subjthedesc>> <<self.listsmelldesc>>"
  tastedesc = "\^<<self.subjthedesc>> <<self.listtastedesc>>"
  touchdesc = "\^<<self.subjthedesc>> <<self.listtouchdesc>>"

  /*
   * General properties.
   */
  isfixture = nil  /* not furniture we want to describe specially. */

  /*
   * Does the actor know about this thing?
   */
  knownto = []
  isknownto(actor) = {
    if (actor = self)
      return true;                           /* Actors know about themselves. */
    if (actor.knowsall and actor <> parserGetMe())
      return self.isknownto(parserGetMe());
    else {
      if (find(self.knownto, actor) <> nil)
        return true;
      else
        return nil;
    }
  }

  /*
   * Make this Thing known to the given actor.
   */
  makeknownto(actor) = {
    if (actor.knowsall and actor <> parserGetMe())
      return;
    if (not self.isknownto(actor) and actor.cansee(self, nil, true))
      self.knownto += actor;
  }

  /*
   * Make everything in this Thing known to the given actor.
   *
   * We assume that anything that's visible automatically becomes known
   *   to the actor. Things that aren't visible but (for example) make
   *   noise will have to be handled with special code.
   */
  makecontentsknownto(actor) = {
    local i, o;
    for (i := 1; i <= length(self.contents); i++) {
      o := self.contents[i];
      if (not o.isknownto(actor))
        if (actor.cansee(o, nil, true))
          o.makeknownto(actor);
      if (o.contents <> [])
        o.makecontentsknownto(actor);
    }
  }

  /*  
   * The XcontentslistedY functions determine whether the object given
   *   as the parameter will be listed in the contents list for this
   *   thing in room descriptions when the containment type is X and
   *   the sense is Y.
   */
  incontentslisted(obj) = { return true; }
  incontentslistedsmell(obj) = { return true; }
  incontentslistedsound(obj) = { return true; }
  incontentslistedtouch(obj) = { return true; }
  oncontentslisted(obj) = { return true; }
  oncontentslistedsmell(obj) = { return true; }
  oncontentslistedsound(obj) = { return true; }
  oncontentslistedtouch(obj) = { return true; }
  undercontentslisted(obj) = { return true; }
  undercontentslistedsmell(obj) = { return true; }
  undercontentslistedsound(obj) = { return true; }
  undercontentslistedtouch(obj) = { return true; }
  behindcontentslisted(obj) = { return true; }
  behindcontentslistedsmell(obj) = { return true; }
  behindcontentslistedsound(obj) = { return true; }
  behindcontentslistedtouch(obj) = { return true; }

  /*
   * islistable(actor) returns true if the object should be listed in
   *   room descriptions.
   *
   * islistablesmell, islistablesound, and islistabletouch are like
   *   islistable, but determine whether or not the Thing will be
   *   listed in the olfactory, audio, or touch contents lists.
   */
  islistable(actor) = { return nil; }
  islistablesmell(actor) = { return nil; }
  islistablesound(actor) = { return nil; }
  islistabletouch(actor) = { return nil; }

  /*
   * isnoticeable is for things like Parts that never want to be listed
   *   in visual contents listings. This differs from islistable in
   *   that islistable only applies when the contents listing is a room
   *   description.
   */
  isnoticeable(actor) = { return true; }

  /*
   * Each of the following methods returns a flattened contents list
   *   for this Thing. The list has the following syntax:
   *
   *  [ [ container loctype n c1 c2 .. cn-4 cn-3] ... ]
   *
   * This is a list of lists. Each sublist corresponds to a particular
   *   container.
   * E.g., calling
   *
   *  desk.seecont(parserGetMe(), ['in' 'on]);
   *
   * might produce the following list:
   *
   *  [ [ desk 'in' 5 key jar ]
   *    [ desk 'on' 6 box coin string ]
   *    [ jar  'in' 3 ]
   *    [ box  'in' 4 egg ] ]
   *
   * The first three entries in each sublist are the containing object,
   *   the location type, and the number of objects in the list
   *   (including the 3 header objects).
   *
   * The list is guaranteed to list containers before their contents.
   *
   * We have separate methods for contents flattening to make contents
   *   listing more efficient. By separating the two tasks of listing
   *   and flattening, we can make the composite operation more
   *   efficient.
   *
   * Note that there is usually no reason to call these methods. Only
   *   the contents lister really needs this kind of detailed contents
   *   information.
   */
  seecontX(actor, ltlist) = {
    local master := [];
    local tl;
    local i, j;
    local loctypeslen;
    local c;
    local cont;
    local o;
    local l;
    local lt;
    local len;
    local cs;

    if (self.contents = [])                     /* Doesn't contain anything. */
      return [];

    cs := [];
    loctypeslen := length(global.loctypes);
    cont := [];
    for (i := 1; i <= loctypeslen; i++) {
      if (ltlist <> nil and find(ltlist, global.loctypes[i]) = nil)
        cs += nil;
      else
        cs += self.passcanseein(actor, nil, global.loctypes[i]);
      cont += [[]];
    }
    c := self.contents;
    for (i := length(c); i > 0; i--) {
      o := c[i];
      lt := o.locationtypenum;
      if (not cs[lt])
        continue;
      cont[lt] += o;
    }
    for (i := 1; i <= loctypeslen; i++) {
      l := cont[i];
      len := length(l);
      if (len = 0)
        continue;
      tl := [] + self;
      tl += i;
      tl += len + 3;                            /* length, including header. */
      tl += l;
      master += [tl];
    }
    for (i := 1; i <= loctypeslen; i++) {
      l := cont[i];
      len := length(l);
      for (j := 1; j <= len; j++) {
        o := l[j];
        if (o.isvisible(actor) and o.isnoticeable(actor))
          master += o.seecontX(actor, nil);
      }
    }
    return master;
  }
  seecont(actor, ltlist) = {
    local l;
    Outhide(true);
    l := self.seecontX(actor, ltlist);
    Outhide(nil);
    return l;
  }
  hearcontX(actor, ltlist) = {
    local master := [];
    local tl;
    local i, j;
    local loctypeslen;
    local c;
    local cont;
    local o;
    local l;
    local lt;
    local len;
    local cs;

    if (self.contents = [])                      /* Doesn't contain anything. */
      return [];

    cs := [];
    loctypeslen := length(global.loctypes);
    cont := [];
    for (i := 1; i <= loctypeslen; i++) {
      if (ltlist <> nil and find(ltlist, global.loctypes[i]) = nil)
        cs += nil;
      else
        cs += self.passcanhearin(actor, nil, global.loctypes[i]);
      cont += [[]];
    }
    c := self.contents;
    for (i := length(c); i > 0; i--) {
      o := c[i];
      lt := o.locationtypenum;
      if (not cs[lt])
        continue;
      cont[lt] += o;
    }
    for (i := 1; i <= loctypeslen; i++) {
      l := cont[i];
      len := length(l);
      if (len = 0)
        continue;
      tl := [] + self;
      tl += i;
      tl += len + 3;                             /* length, including header. */
      tl += l;
      master += [tl];
    }
    for (i := 1; i <= loctypeslen; i++) {
      l := cont[i];
      len := length(l);
      for (j := 1; j <= len; j++) {
        o := l[j];
        master += o.hearcontX(actor, nil);
      }
    }
    return master;
  }
  hearcont(actor, ltlist) = {
    local l;
    Outhide(true);
    l := self.hearcontX(actor, ltlist);
    Outhide(nil);
    return l;
  }
  smellcontX(actor, ltlist) = {
    local master := [];
    local tl;
    local i, j;
    local loctypeslen;
    local c;
    local cont;
    local o;
    local l;
    local lt;
    local len;
    local cs;

    if (self.contents = [])                      /* Doesn't contain anything. */
      return [];

    cs := [];
    loctypeslen := length(global.loctypes);
    cont := [];
    for (i := 1; i <= loctypeslen; i++) {
      if (ltlist <> nil and find(ltlist, global.loctypes[i]) = nil)
        cs += nil;
      else
        cs += self.passcansmellin(actor, nil, global.loctypes[i]);
      cont += [[]];
    }
    c := self.contents;
    for (i := length(c); i > 0; i--) {
      o := c[i];
      lt := o.locationtypenum;
      if (not cs[lt])
        continue;
      cont[lt] += o;
    }
    for (i := 1; i <= loctypeslen; i++) {
      l := cont[i];
      len := length(l);
      if (len = 0)
        continue;
      tl := [] + self;
      tl += i;
      tl += len + 3;                            /* length, including header. */
      tl += l;
      master += [tl];
    }
    for (i := 1; i <= loctypeslen; i++) {
      l := cont[i];
      len := length(l);
      for (j := 1; j <= len; j++) {
        o := l[j];
        master += o.smellcontX(actor, nil);
      }
    }
    return master;
  }
  smellcont(actor, ltlist) = {
    local l;
    Outhide(true);
    l := self.smellcontX(actor, ltlist);
    Outhide(nil);
    return l;
  }
  touchcontX(actor, ltlist) = {
    local master := [];
    local tl;
    local i, j;
    local loctypeslen;
    local c;
    local cont;
    local o;
    local l;
    local lt;
    local len;
    local cs;

    if (self.contents = [])                      /* Doesn't contain anything. */
      return [];

    cs := [];
    loctypeslen := length(global.loctypes);
    cont := [];
    for (i := 1; i <= loctypeslen; i++) {
      if (ltlist <> nil and find(ltlist, global.loctypes[i]) = nil)
        cs += nil;
      else
        cs += self.passcantouchin(actor, nil, global.loctypes[i]);
      cont += [[]];
    }
    c := self.contents;
    for (i := length(c); i > 0; i--) {
      o := c[i];
      lt := o.locationtypenum;
      if (not cs[lt])
        continue;
      cont[lt] += o;
    }
    for (i := 1; i <= loctypeslen; i++) {
      l := cont[i];
      len := length(l);
      if (len = 0)
        continue;
      tl := [] + self;
      tl += i;
      tl += len + 3;                             /* length, including header. */
      tl += l;
      master += [tl];
    }
    for (i := 1; i <= loctypeslen; i++) {
      l := cont[i];
      len := length(l);
      for (j := 1; j <= len; j++) {
        o := l[j];
        master += o.touchcontX(actor, nil);
      }
    }
    return master;
  }
  touchcont(actor, ltlist) = {
    local l;
    Outhide(true);
    l := self.touchcontX(actor, ltlist);
    Outhide(nil);
    return l;
  }

  /*
   * doListcontents is for debugging only. It describes the location of
   *   every object contained in this Thing.
   */
  verDoListcontents(actor) = {}
  doListcontents(actor) = {
    local i;
    for (i := 1; i <= length(self.contents); i++) {
      self.contents[i].doLocate(actor);
      "\n";
    }
    for (i := 1; i <= length(self.contents); i++)
      self.contents[i].doListcontents(actor);
  }

  /*
   * Recursive contents enumeration.
   *
   * This function returns a list of all the contents of this Thing,
   *   and all the contents of the contents, etc. subject to the
   *   requirement that the containment types are in the list loctypes.
   *
   * loctypes is only used for the top level items. (If Z is (in, on,
   *   under, behind) X, and X is *in* Y, then Z is *in* Y too, so we
   *   don't need to check Z's loctype.
   */
  recursivecontents(actor, loctypes) = {
    local i, l, r := [];
    l := self.contents;
    for (i := 1; i <= length(l); i++)
      if (loctypes = nil or find(loctypes, l[i].locationtype) <> nil) {
        r += l[i];
        r += l[i].recursivecontents(actor, nil);
      }
    return r;
  }

  /*  
   * The topcontents function is just like recursivecontents, but
   *   doesn't recurse.
   */
  topcontents(actor, loctypes) = {
    local i, l, r := [];
    l := self.contents;
    for (i := 1; i <= length(l); i++)
      if (loctypes = nil or find(loctypes, l[i].locationtype) <> nil) {
        r += l[i];
      }
    return r;
  }

  /*
   * itemcontents is like recursivecontents, but only keeps items, and
   *   only recurses on objects that aren't Items. (If a container is
   *   an Item itself, we don't need to recurse, because the player
   *   would likely rather have the whole container than the individual
   *   contents.)
   */
  itemcontents(actor, loctypes) = {
    local i, l, r := [];
    l := self.contents;
    for (i := 1; i <= length(l); i++) {
      if (loctypes = nil or find(loctypes, l[i].locationtype) <> nil) {
        if (isclass(l[i], Item))
          r += l[i];
        else
          r += l[i].itemcontents(actor, nil);
      }
    }
    return r;
  }

  /*
   * Methods to deal with computing the weight and bulk of contents:
   *
   * - contentsX returns the total value of all contents with
   *   containment types given in loctypes, where the value of each
   *   contained Thing is given by Thing.(valueprop). Note that
   *   loctypes is only used for the top level of the recursion - lower
   *   down, all location types are considered.
   *
   *   (This makes sense -- consider calling this with loctypesequal to
   *   ['in' 'on']: if you have a phone booth witha mouse behind it and
   *   a table inside it, then the total weight inside the phone booth
   *   does not include the mouse, but it does include things which are
   *   under or behind the table.)
   *
   *   If loctypes is nil, we assume that all location types are
   *   allowed.
   *
   *   Contents without the value property defined are counted as
   *   zeroes.
   *
   * - contentsweight returns the total weight of all contents.
   * - contentsbulk returns the total weight of all contents.
   */
  contentsX(actor, valueprop, loctypes) = {
    local i, l, total := 0;
    l := self.contents;
    for (i := 1; i <= length(l); i++)
      if (loctypes = nil or find(loctypes, l[i].locationtype) <> nil) {
        if (defined(l[i], valueprop))
          total += l[i].(valueprop);
        total += l[i].contentsX(actor, valueprop, nil);
      }
    return total;
  }
  contentsweight(actor, loctypes) = {
    return self.contentsX(actor, &weight, loctypes);
  }
  contentsbulk(actor, loctypes) = {
    return self.contentsX(actor, &bulk, loctypes);
  }

  /*
   * By default, Things can't hold anything. (Use Holders for that.)
   *   These defaults get inherited by Holder - the default mapping
   *   from the specific max methods to the general ones (maxbulk and
   *   maxweight) are convenient - the programmer can use the old adv.t
   *   names maxbulk and maxweight as long as he doesn't want to make a
   *   distinction between, for example, the amount that can fit in the
   *   Thing and the amount that can fit on it.
   */
  maxbulk = 0
  maxweight = 0
  inmaxbulk = { return self.maxbulk; }
  onmaxbulk = { return self.maxbulk; }
  undermaxbulk = { return self.maxbulk; }
  behindmaxbulk = { return self.maxbulk; }
  inmaxweight = { return self.maxweight; }
  onmaxweight = { return self.maxweight; }
  undermaxweight = { return self.maxweight; }
  behindmaxweight = { return self.maxweight; }

  /*
   * Senses
   *
   * These just determine whether or not the the thing is inherently
   *   accessable in the various ways.
   *
   * This is very different from whether or not the thing is accessable
   *   at the given time and in its current location relative to the
   *   player. The canX methods determine the latter.
   *
   * If isvisible returns nil here, for example, it means that the
   *   object is simply invisible, and can never be seen regardless of
   *   where it is relative to the player.
   *
   * In order to maintain backward compatibility with old code, we
   *   check the verDo methods for the various senses here. This means
   *   that you can print explanations of why the object can't be
   *   accessed in the verDo methods as in old code. These message will
   *   be printed, however, in many more circumstances than before,
   *   since many verbs requires these kinds of access. For example,
   *   isvisible uses verDoInspect to check if the object is
   *   examinable. Things that are examinable are usually visible...
   */
  isvisible(actor) = {
    Outhide(true);
    self.verDoInspect(actor);
    if (Outhide(nil)) {
      self.verDoInspect(actor);
      return nil;
    }
    return true;
  }
  istouchable(actor) = {
    Outhide(true);
    self.verDoTouch(actor);
    if (Outhide(nil)) {
      self.verDoTouch(actor);
      return nil;
    }
    return true;
  }
  istakeable(actor, objl) = {
    Outhide(true);
    self.verDoTake(actor);
    if (Outhide(nil)) {
      self.verDoTake(actor);
      return nil;
    }
    return true;
  }
  issmellable(actor) = {
    Outhide(true);
    self.verDoSmell(actor);
    if (Outhide(nil)) {
      self.verDoSmell(actor);
      return nil;
    }
    return true;
  }
  isaudible(actor) = {
    Outhide(true);
    self.verDoListento(actor);
    if (Outhide(nil)) {
      self.verDoListento(actor);
      return nil;
    }
    return true;
  }
  islistener(actor) = { return nil; }

  /*
   * By default, we can't put things in, on, under, or behind Things.
   */
  acceptsput(actor, loctype) = {
    return nil;
  }
  
  /*
   * Sense path determination.
   *
   * These methods return true if there is a clear path from this Thing
   *   to the given desination object for a particular sense. To
   *   determine this, we walk up the containment hierarchy to find the
   *   lowest common ancestor between the thing and the object.
   *
   * We have a general method blocksreach that returns the first object
   *   in the hierarchy that blocks the given sense from passing
   *   through. (Nil, therefore, indicates that the sense can pass all
   *   the way through.)
   *  
   * Note that the Xpath messages should not include initial caps or
   *   final punctuation. When methods that get called indirectly by
   *   Xpath methods (like istakeable, or passcanseein, etc.) print
   *   things, the Xpath methods set a global flag global.canoutput
   *   which Thing.cantReach uses to properly format the explanation of
   *   why the action failed. This is a gross hack, but it works.
   *
   * Also note that blocksreach can print stuff - this happens when
   *   &sensein or &senseout print stuff when they fail. (At the
   *   moment, we don't allow these methods to print things when they
   *   succeed. They should also never change game state.)
   */
  blocksreach(obj, loctype, sensein, senseout, senseacross, outself, inobj) = {
    local selfparents := [], sploctype := [], splen;
    local objparents := [], oploctype := [], oplen;
    local p := nil, lt := nil;
    local i, ci;
    local across := true;
    local lca;
    local lcac_p, lcac_o;

    /*
     * Fail if obj = nil.
     */
    if (obj = nil) {
      "*** ERROR: <<self.sdesc>>.blocksreach called with obj = nil\n";
      return TOP;
    }

    /*
     * Find all parents of self.
     * If the containment ends (in nil) before reaching TOP, put
     *   [NIL TOP] at the end of the list.
     */
    p := self.location;
    lt := self.locationtype;
    while (p <> nil) {
      selfparents += p;
      sploctype += lt;
      lt := p.locationtype;
      p := p.location;
    }
    splen := length(selfparents);
    if (splen = 0 or selfparents[splen] <> TOP) {
      selfparents += [NIL TOP];
      sploctype += ['in' 'in'];
      splen += 2;
    }
  
    /*
     * Find all parents of object.
     * If the containment ends (in nil) before reaching TOP, put
     *   [NIL TOP] at the end of the list.
     */
    p := obj.location;
    lt := obj.locationtype;
    while (p <> nil) {
      objparents += p;
      oploctype += lt;
      lt := p.locationtype;
      p := p.location;
    }
    oplen := length(objparents);
    if (oplen = 0 or objparents[oplen] <> TOP) {
      objparents += [NIL TOP];
      oploctype += ['in' 'in'];
      oplen += 2;
    }

    /*
     * Print debug messages if global.debugreach is on.
     */
    if (global.debugreach) {
      local z;
      "\bblocksreach: <<self.sdesc>>: [ ";
      for (z := 1; z <= length(selfparents); z++)
        "(<<selfparents[z].sdesc>>) ";
      " ]\nblocksreach: <<obj.sdesc>>: [ ";
      for (z := 1; z <= length(objparents); z++)
        "(<<objparents[z].sdesc>>) ";
      " ]\n";       
    }   

    /*
     * Find first differing parent, starting from the ends of the
     *   lists and moving backwards. The element preceding the
     *   first differing element is the lowest common ancestor.
     *
     * If all the elements in both lists are the same, there are
     *   two cases:
     *
     * 1 There is nothing between the lowest common ancestor and
     *   one of the objects; i.e., one of the objects is contained
     *   directly in the lowest common ancestor and hence there is
     *   no differing parent.
     *
     * 2 One of the objects is contained in the other object.
     *
     * In both cases the immediate parent of the object with the
     *   short list is the lowest common ancestor. In case 2, we
     *   skip the sense ACROSS the common ancestor.
     */
    for (ci := 0; ; ci++) {
      if (ci >= splen and ci >= oplen) {
        ci--;
        lca := selfparents[splen-ci];
        break;
      }
      else if (ci >= splen or ci >= oplen) {
        if (ci >= oplen and selfparents[splen-ci] = obj or ci >= splen and objparents[oplen-ci] = self)
            across := nil;
        else
          ci--;
        if (ci >= oplen)
          lca := selfparents[splen-ci];
        else
          lca := objparents[oplen-ci];
        break;
      }
      if (selfparents[splen-ci] <> objparents[oplen-ci]) {
        ci--;
        lca := selfparents[splen-ci];
        break;
      }
    }

    /*
     * Determine the child of the lowest common ancestor on the
     *   self and object sides. These tell us how the objects are
     *   contained (in, on, under, behind) the lowest common
     *   ancestor.
     */
    if (splen-ci-1 < 1)
      lcac_p := self;
    else
      lcac_p := selfparents[splen-ci-1];
    if (oplen-ci-1 < 1)
      lcac_o := obj;
    else
      lcac_o := objparents[oplen-ci-1];
              
    /*
     * Run through the whole check if we're debugging.
     */
    if (global.debugreach) {
      "Least common ancestor is <<lca.sdesc>>\n";
      "Child of LCA (self) is <<lcac_p.sdesc>>\n";
      "Child of LCA (obj) is <<lcac_o.sdesc>>\n";
      "Reach trace:\b";
      if (outself) {
        " OUT [<<loctype>>] of [<<self.sdesc>>]\n";
        if (not self.(senseout)(self, obj, loctype))
          " (*)\n";
      }
      i := 1;
      while (i < splen-ci) {
        p := selfparents[i];
        if (p <> nil) {
          " OUT of [<<p.sdesc>>]\n";
          if (not p.(senseout)(self, obj,
              sploctype[i]))
            " (*)\n";
        }
        else
          " OUT of [nil] (*)\n";
        i++;      
      }
      if (across) {
        p := lca;
        if (p <> nil) {
          " ACROSS [<<p.sdesc>>]\n";
          if (not p.(senseacross)(self, obj, lcac_p.locationtype, lcac_o.locationtype))
            " (*)\n";
        }
        else
          " ACROSS [nil] (*)\n";
      }
      i := oplen-ci-1;
      while (i > 0) {
        p := objparents[i];
        if (p <> nil) {
          " IN [<<p.sdesc>>]\n";
          if (not p.(sensein)(self, obj, oploctype[i]))
            " (*)\n";
        }
        else
          " IN [nil] (*)\n";
        i--;
      }
      if (inobj) {
        " IN [<<loctype>>] of [<<obj.sdesc>>]\n";
        if (not obj.(sensein)(self, obj, loctype))
          " (*)\n";
      }
      "\bEnd of reach trace.\n";
    }

    /*
     * If we're to check that we can sense out of
     * ourselves, do so.
     */
    if (outself) {
      if (not self.(senseout)(self, obj, loctype))
        return self;
    }

    /*
     * Now we traverse the containment hierarchy up from ourselves
     *   to the lowest common ancestor (but not including the
     *   ancestor), making sure that each parent allows us to sense
     *   OUT.
     */
    i := 1;
    while (i < splen-ci) {
      p := selfparents[i];
      if (not p.(senseout)(self, obj, sploctype[i]))
        return p;
      i++;      
    }   

    /*
     * Now check to see that we can sense ACROSS the lowest common
     *   ancestor.
     */
    if (across) {
      p := lca;
      if (not p.(senseacross)(self, obj, lcac_p.locationtype, lcac_o.locationtype))
        return p;
    }
    
    /*
     * Now we traverse the containment hierarchy down from  the
     *   lowest common ancestor to the object, making sure that
     *   each container allows us to sense IN.
     *
     * The order here is crucial, because we want to return the
     *   first thing that blocks the sense on the containment path
     *   from the source to the destination.
     */
    i := oplen-ci-1;
    while (i > 0) {
      p := objparents[i];
      if (not p.(sensein)(self, obj, oploctype[i])) {
        return p;
      }
      i--;
    }

    /*
     * If we're to check that we can sense into  the object, do so.
     */
    if (inobj) {
      if (not obj.(sensein)(self, obj, loctype))
        return obj;
    }

    /*
     * Nothing blocks the sense!
     */
    return nil;
  }
  
  /*
   * Is there a line of sight to the object?
   */
  seepath(obj, loctype, silent) = {
    local b, i;
    local output;
    
    Outhide(true);
    b := self.blocksreach(obj, loctype, &passcanseein, &passcanseeout, &passcanseeacross, nil, nil);
    output := Outhide(nil);

    /*
     * Print output if there was any.
     */
    if (output and not silent)
      self.blocksreach(obj, loctype, &passcanseein, &passcanseeout, &passcanseeacross, nil, nil);
    if (b <> nil) {
      if (not silent and not output)
        "<<b.subjthedesc>> blocks <<self.possessivedesc>> view of <<obj.objthedesc(self)>>";
    }

    global.canoutput := output;
    return b ? nil : true;
  }

  /*
   * Is there a clear touch path to the object?
   */
  touchpath(obj, loctype, silent) = {
    local b, i;
    local output;

    Outhide(true);
    b := self.blocksreach(obj, loctype, &passcantouchin, &passcantouchout, &passcantouchacross, nil, nil);
    output := Outhide(nil);

    /*
     * Print output if there was any.
     */
    if (output and not silent)
      self.blocksreach(obj, loctype, &passcantouchin, &passcantouchout, &passcantouchacross, nil, nil);
    if (b <> nil) {
      if (not silent and not output)
        "<<self.subjprodesc>> can't reach <<obj.objthedesc(self)>> - <<b.subjthedesc>> <<b.is>> in the way";
    }

    global.canoutput := output;
    return b ? nil : true;
  }

  /*
   * Is there a clear take path to the object?
   */
  takepath(obj, loctype, silent) = {
    local b, i;
    local output;

    Outhide(true);
    b := self.blocksreach(obj, loctype, &passcantakein, &passcantakeout, &passcantakeacross, nil, nil);
    output := Outhide(nil);

    /*
     * Print output if there was any.
     */
    if (output and not silent)
      self.blocksreach(obj, loctype, &passcantakein, &passcantakeout, &passcantakeacross, nil, nil);
    if (b <> nil) {
      if (not silent and not output)
        "<<b.subjthedesc>> prevents <<self.objprodesc(nil)>> from taking <<obj.objthedesc(nil)>>";
    }

    global.canoutput := output;
    return b ? nil : true;
  }

  /*
   * Is there a clear smell path to the object?
   */
  smellpath(obj, loctype, silent) = {
    local b, i;
    local output;
    
    Outhide(true);
    b := self.blocksreach(obj, loctype, &passcansmellin, &passcansmellout, &passcansmellacross, nil, nil);
    output := Outhide(nil);

    /*
     * Print output if there was any.
     */
    if (output and not silent)
      self.blocksreach(obj, loctype, &passcansmellin, &passcansmellout, &passcansmellacross, nil, nil);
    if (b <> nil) {
      if (not silent and not output)
        "<<b.subjthedesc>> prevents <<self.objprodesc(nil)>> from smelling <<obj.objthedesc(self)>>";
    }

    global.canoutput := output;
    return b ? nil : true;
  }

  /*
   * Is there a clear hear path to the object?
   */
  hearpath(obj, loctype, silent) = {
    local b, i;
    local output;

    Outhide(true);
    b := self.blocksreach(obj, loctype, &passcanhearin, &passcanhearout, &passcanhearacross, nil, nil);
    output := Outhide(nil);

    /*
     * Print output if there was any.
     */
    if (output and not silent)
      self.blocksreach(obj, loctype, &passcanhearin, &passcanhearout, &passcanhearacross, nil, nil);
    if (b <> nil) {
      if (not silent and not output)
        "<<b.subjthedesc>> prevents <<self.objprodesc(nil)>> from hearing <<obj.objthedesc(self)>>";
    }

    global.canoutput := output;
    return b ? nil : true;
  }

  /*
   * Is there a clear speak-to path to the object?
   */
  speaktopath(obj, loctype, silent) = {
    local b, i;
    local output;

    Outhide(true);
    b := self.blocksreach(obj, loctype, &passcanspeaktoout, &passcanspeaktoin, &passcanspeaktoacross, nil, nil);
    output := Outhide(nil);

    /*
     * Print output if there was any.
     */
    if (output and not silent)
      self.blocksreach(obj, loctype, &passcanspeaktoout, &passcanspeaktoin, &passcanspeaktoacross, nil, nil);
    if (b <> nil) {
      if (not silent and not output)
        "<<b.subjthedesc>> prevents <<obj.objprodesc(nil)>> from hearing <<self.objthedesc(self)>>";
    }

    global.canoutput := output;
    return b ? nil : true;
  }

  /*
   * Is there a clear put-into path to the object?
   * This is really the same as taking (with the directions reversed)
   *   but we want to print different failure reasons.
   */
  putintopath(obj, loctype, silent) = {
    local b, i;
    local output;
    
    Outhide(true);
    b := self.blocksreach(obj, loctype, &passcantakeout, &passcantakein, &passcantakeacross, nil, true);
    output := Outhide(nil);

    /*
     * Print output if there was any.
     */
    if (output and not silent)
      self.blocksreach(obj, loctype, &passcantakeout, &passcantakein, &passcantakeacross, nil, true);
    if (b <> nil) {
      if (not silent and not output)
        "<<b.subjthedesc>> prevents <<self.objprodesc(nil)>> from putting things in <<obj.objthedesc(nil)>>";
    }

    global.canoutput := output;
    return b ? nil : true;
  }

  /*
   * Things can't sense - use a Sensor if you want these capabilities.
   *
   * Note that the canX messages should not include initial caps.
   */
  cansee(obj, loctype, silent) = {
    if (not silent)
      "<<self.subjthedesc>> can't see anything.";
    return nil;
  }
  cantouch(obj, loctype, silent) = {
    if (not silent)
      "<<self.subjthedesc>> can't touch anything.";
    return nil;
  }
  cantake(obj, loctype, silent) = {
    if (not silent)
      "<<self.subjthedesc>> can't take anything.";
    return nil;
  }
  cansmell(obj, loctype, silent) = {
    if (not silent)
      "<<self.subjthedesc>> can't smell anything.";
    return nil;
  }
  canhear(obj, loctype, silent) = {
    if (not silent)
      "<<self.subjthedesc>> can't hear anything.";
    return nil;
  }
  canspeakto(obj, loctype, silent) = {
    if (not silent)
      "<<self.subjthedesc>> can't say anything.";
    return nil;
  }
  canputinto(obj, loctype, silent) = {
    if (not silent)
      "<<self.subjthedesc>> can't put anything anywhere.";
    return nil;
  }

  /*
   * Can this Thing use obj as a topic?
   * I.e., as in
   *
   *  >ask troll about rabbit
   *
   * or
   *
   *  >tell troll about rabbit
   */
  istopic = nil
  isatopic(obj, loctype, silent) = {
    if (obj.istopic) {
      return true;
    }
    if (not silent)
      "something tells <<global.lastactor.subjthedesc>> <<obj.subjthedesc>> will not be a particularly productive topic.";
    return nil;
  }

  /*
   * Can this Thing sense the contents of the given object?
   * Although Things can't see, we define these here since they're in
   *   terms of the canX methods.
   *
   * If loctype is nil, we return the list of valid loctypes, or nil if
   *   no loctype is valid.
   */
  canXcontents(obj, silent, loctype, cansense, sensein, senseout, senseacross, s) = {
    local i;
    local selfloctype;
    local ok := [];
    local lt;
    local len := length(global.loctypes);
    local ret;

    /*
     * If self cannot sense obj at all, don't bother checking.
     */
    if (not self.(cansense)(obj, nil, silent))
      return nil;

    /*
     * Check that loctype is valid if it is not nil.
     */
    if (loctype <> nil)
      lt := find(global.loctypes, loctype);

    for (i := 1; i <= len; i++)
      ok += nil;

    /*
     * If self is contained in obj, check if it can sense across
     *   obj.
     *
     * If self is not contained in obj, check if it can sense into
     *   obj.
     */
    if (self.iscontained(obj, nil)) {
      selfloctype := self.containmenttype(obj);
      Outhide(true);
      for (i := 1; i <= len; i++)
        ok[i] := obj.(senseacross)(self, nil, selfloctype, global.loctypes[i]);
      if (Outhide(nil) and loctype <> nil) {
        if (not silent and not ok[lt]) {
          obj.(senseacross)(self, nil, selfloctype, loctype);
          return nil;
        }
      }
    }
    else {
      Outhide(true);
      for (i := 1; i <= len; i++)
        ok[i] := obj.(sensein)(self, obj, global.loctypes[i]);
      if (Outhide(nil) and loctype <> nil) {
        if (not silent and not ok[lt]) {
          obj.(sensein)(self, obj, loctype);
          return nil;
        }
      }
    }

    /*
     * Display a message if self cannot sense the contents of obj
     *   and return the valid loctypes.
     */
    if (loctype <> nil) {
      if (not silent and not ok[lt])
        "\^<<self.subjthedesc>> can't <<s>> <<obj.objthedesc(self)>>.";
      return ok[lt];
    }
    ret := [];
    for (i := 1; i <= length(ok); i++)
      if (ok[i])
        ret += global.loctypes[i];
    return ret;
  }

  /*
   * Can this Thing see the contents of obj?
   */
  canseecontents(obj, silent, loctype) = {
    return self.canXcontents(obj, silent, loctype, &cansee, &passcanseein, &passseeout, &passcanseeacross, 'see the contents of');
  }

  /*
   * Can this Thing touch the contents of obj?
   */
  cantouchcontents(obj, silent, loctype) = {
    return self.canXcontents(obj, silent, loctype, &cantouch, &passcantouchin, &passtouchout, &passcantouchacross, 'touch the contents of');
  }

  /*
   * Can this Thing take the contents of obj?
   */
  cantakecontents(obj, silent, loctype) = {
    return self.canXcontents(obj, silent, loctype, &cantake, &passcantakein, &passcantakeout, &passcantakeacross, 'take the contents of');
  }

  /*
   * Can this Thing smell the contents of obj?
   */
  cansmellcontents(obj, silent, loctype) = {
    return self.canXcontents(obj, silent, loctype, &cansmell, &passcansmellin, &passcansmellout, &passcansmellacross, 'smell the contents of');
  }

  /*
   * Can this Thing hear the contents of obj?
   */
  canhearcontents(obj, silent, loctype) = {
    return self.canXcontents(obj, silent, loctype, &canhear, &passcanhearin, &passcanhearout, &passcanhearacross, 'hear the contents of');
  }

  /*
   * Can this Thing speak to the contents of obj?
   */
  canspeaktocontents(obj, silent, loctype) = {
    return self.canXcontents(obj, silent, loctype, &canspeakto, &passcanspeaktoout, &passcanspeaktoin, &passcanspeaktoacross, 'speak to the contents of');
  }

  /*
   * Can this Thing put anything into the contents of obj?
   */
  canputintocontents(obj, silent, loctype) = {
    return self.canXcontents(obj, silent, loctype, &canputinto, &cantakeout, &cantakein, &passcantakeacross, 'put anything into the contents of');
  }
  
  /*
   * Psuedo-sense checks.  Things can't do anything with numbers or
   *   strings, so these always return nil.
   */
  canusealphanum(obj, loctype, silent) = {
    if (not silent)
      "that verb requires a double-quoted string or a number.";
    return nil;
  }
  canusealpha(obj, loctype, silent) = {
    if (not silent)
      "that verb requires a double-quoted string.";
    return nil;
  }
  canusenumber(obj, loctype, silent) = {
    if (not silent)
      "that verb requires a number.";
    return nil;
  }
  canusenumberorlist(obj, loctype, silent) = {
    if (not silent)
      "that verb requires a number or parenthesized list of numbers.";
    return nil;
  }

  /*
   * Sense passing
   *
   * These boolean-valued methods determine whether or not this thing 
   *   allows the various senses to pass through on the way out of and
   *   into it. The reason for the distinction between "out of" and
   *   "into" is that we want to be able to implement (for example)
   *   one-way mirrors.
   *
   * The obj parameter is the object that's trying to sense through us.
   *   The loctype parameter is the type of containment: 'in', 'on',
   *   'under', 'behind', etc. NOTE: The loctype parameter may be nil,
   *   in which case the method should return true if any loctype is
   *   allowed.
   *
   * The across versions determine whether or not something contained
   *   in the thing can sense other things contained in the thing. For
   *   example, you might not be able to see into a room from outside,
   *   or outside the room from inside, but you certainly might be able
   *   to see things on the same side of the room. While this may seem
   *   pointless, we put it to good use for dark rooms. Note that the
   *   across methods get two location types - one for self, and one
   *   for the object. These are not necessarily the same as
   *   self.locationtype and obj.locationtype, however - there may be
   *   other objects intervening between self and obj and the lowest
   *   common ancestor between them.
   */
  passcansee(actor, obj, loctype) = { return true; }
  passcanseein(actor, obj, loctype) = {
    return self.passcansee(actor, obj, loctype);
  }
  passcanseeout(actor, obj, loctype) = {
    return self.passcansee(actor, obj, loctype);
  }
  passcanseeacross(actor, obj, selfloctype, objloctype) = {
    return (selfloctype = objloctype and self.passcansee(actor, obj, nil));
  }
  passcantouch(actor, obj, loctype) = { return true; }
  passcantouchin(actor, obj, loctype) = {
    return self.passcantouch(actor, obj, loctype);
  }
  passcantouchout(actor, obj, loctype) = {
    return self.passcantouch(actor, obj, loctype);
  }
  passcantouchacross(actor, obj, selfloctype, objloctype) = {
    return (selfloctype = objloctype and self.passcantouch(actor, obj, nil));
  }
  passcantake(actor, obj, loctype) = { return true; }
  passcantakein(actor, obj, loctype) = {
    return self.passcantake(actor, obj, loctype);
  }
  passcantakeout(actor, obj, loctype) = {
    return self.passcantake(actor, obj, loctype);
  }
  passcantakeacross(actor, obj, selfloctype, objloctype) = {
    return (selfloctype = objloctype and self.passcantake(actor, obj, nil));
  }
  passcansmell(actor, obj, loctype) = { return true; }
  passcansmellin(actor, obj, loctype) = {
    return self.passcansmell(actor, obj, loctype);
  }
  passcansmellout(actor, obj, loctype) = {
    return self.passcansmell(actor, obj, loctype);
  }
  passcansmellacross(actor, obj, selfloctype, objloctype) = {
    return (selfloctype = objloctype and self.passcansmell(actor, obj, nil));
  }
  passcanhear(actor, obj, loctype) = { return true; }
  passcanhearin(actor, obj, loctype) = {
    return self.passcanhear(actor, obj, loctype);
  }
  passcanhearout(actor, obj, loctype) = {
    return self.passcanhear(actor, obj, loctype);
  }
  passcanhearacross(actor, obj, selfloctype, objloctype) = {
    return (selfloctype = objloctype and self.passcanhear(actor, obj, nil));
  }
  passcanspeakto(actor, obj, loctype) = { return true; }
  passcanspeaktoin(actor, obj, loctype) = {
    return self.passcanspeakto(actor, obj, loctype);
  }
  passcanspeaktoout(actor, obj, loctype) = {
    return self.passcanspeakto(actor, obj, loctype);
  }
  passcanspeaktoacross(actor, obj, selfloctype, objloctype) = {
    return (selfloctype = objloctype and self.passcanspeakto(actor, obj, nil));
  }

  /*
   * Move the Thing into another container.
   * Here we assume that the destination is a container. Calling this
   *   on a Thing that has a method location is OK, but the Thing's
   *   location won't be changed.
   *
   * lttm is a list of contents' locationtypes to move. Normally when
   *   we move something we don't want to move contents that are, for
   *   example, behind their containers. (For example, if the player
   *   takes a chair, and there is a mouse under the chair, the player
   *   shouldn't get the mouse, even though it is technically
   *   "contained in" the chair. Furthermore, once the chair is moved,
   *   we have to change the mouse's location to be the same as the
   *   chair's location was.)
   *
   * If lttm is nil, all location types are moved. (This is almost
   *   always wrong, however!)
   *
   * Returns true if the move actually happened, nil if the move got
   *   prevented for some reason.
   */
  moveinto(obj, loctype, lttm) = {
    local i, l, len := 0, oldloc, oldloctype;
    local a;

    /*
     * If we're already directly contained in the destination
     *   object, do nothing. We need this for Attachables.
     */
    if (self.location = obj and self.locationtype = loctype)
      return nil;
    
    /* Sanity check. */
    if (find(global.loctypes, loctype) = nil) {
      "\^<<self.subjthedesc>>.moveinto(<<obj.subjthedesc>>, <<loctype>>): loctype unknown\n";
      return nil;
    }

    oldloc := self.location;
    oldloctype := self.locationtype;
    
    /*
     * Notify container we're moving out of and container we're
     *   moving into that we're moving.
     */
    self.notifycontainers(obj, loctype);

    /*
     * Remove ourselves from parent's contents list. Remove our
     *   Parts from our previous location's contents list.
     */
    l := self.parts;
    len := length(l);
    if (self.location) {
      self.location.contents -= self;
      for (i := 1; i <= len; i++)
        self.location.contents -= l[i];
    }
    
    /*
     * Now update locations and locationtypes, and add our Parts to
     *   the new location's contents list.
     */
    self.setlocation(obj);
    if (obj) {
      obj.contents += self;
      self.setlocationtype(loctype);
      for (i := 1; i <= len; i++)
        obj.contents += l[i];
    }
    else {
      self.setlocationtype('in');
    }

    /*
     * Walk up the containment hierarchy. If we get an actor, make
     *   the object and all its contents known to the actor in case
     *   they aren't already. (This is to handle cases where the
     *   game will explicitly move something out of nil and into
     *   the actor.)
     */
    a := obj;
    while (a <> nil) {
      if (isclass(a, Actor) and not self.isknownto(a)) {
        self.makeknownto(a);
        self.makecontentsknownto(a);
      }
      a := a.location;
    }

    /*
     * Now fix locations and locationtypes of contents that do not
     *   get moved. If lttm is nil, all contents get moved.
     */
    if (lttm = nil)
      return true;
    l := self.contents;
    len := length(l);
    for (i := 1; i <= len; i++) {
      if (find(lttm, l[i].locationtype) = nil) {
        self.contents -= l[i];
        l[i].setlocation(oldloc);
        l[i].setlocationtype(oldloctype);
        l[i].location.contents += l[i];
      }
    }

    return true;
  }
  moveto(obj, loctype) = {
    return self.moveinto(obj, loctype, global.loctake);
  }
  movein(obj) = {
    return self.moveinto(obj, 'in', global.loctake);
  }
  moveon(obj) = {
    return self.moveinto(obj, 'on', global.loctake);
  }
  moveunder(obj) = {
    return self.moveinto(obj, 'under', global.loctake);
  }
  movebehind(obj) = {
    return self.moveinto(obj, 'behind', global.loctake);
  }

  /*
   * Notify new location that we're moving in and see if it approves. 
   *   Also notify current location we're moving out and see if it
   *   approves, and notify old and new locations about Part movements.
   */
  notifycontainers(obj, loctype) = {
    local i, l, len;
    if (self.location)
      self.location.movingout(self, obj, loctype);
    if (obj)
      obj.movingin(self, loctype);
    l := self.parts;
    len := length(self.parts);
    for (i := 1; i <= len; i++) {
      if (self.location)
        self.location.movingout(l[i], obj, loctype);
      if (obj)
        obj.movingin(l[i], loctype);
    }
    return true;
  }

  /*
   * movingout gets called whenever an object gets moved out of this
   *   Thing. The object's location and locationtype fields will still
   *   be set for the old location (this Thing) - the new location and
   *   locationtype are given by the tolocation and toloctype
   *   parameters.
   *
   * This routine gets notified about Parts too.
   *
   * WARNING: Some verbs that intuitively involve taking the direct
   *   object do not call moveinto. For example, shootVerb specifies
   *   &cantake as a requirement, but one hardly expects doShoot to
   *   actually call moveinto to move the gun into the player's
   *   inventory. So you need to be aware of these "transient grabs".
   */
  movingout(obj, tolocation, toloctype) = { }

  /*
   * movingin gets called whenever an object gets moved into this
   *   Thing. obj will still be in its old location and in its old
   *   location's contents list.
   *
   */
  movingin(obj, loctype) = { }

  /*
   * Return true if the Thing is contained in the object. Note that
   *   unlike the standard TADS isIn method, this doesn't care whether
   *   or not the containers pass visibility.The loctype parameter
   *   specifies the kind of containment ('in', 'on', 'under',
   *   'behind', etc.)
   *
   * This is a little tricky.  We say that A is contained in B if A is
   *   either directly contained in B or if A is in, on, under, behind,
   *   etc. something that is directly contained in B.
   *
   * For example, a key on a book which is behind a table is not
   *   contained in the table, but it is "contained" behind the table.
   *
   * This code also assumes that the location types are all orthogonal
   *   and transitive - i.e., that being "on" something does not have
   *   any effect on whether it's "under" (etc.) something else, and
   *   that if A is "on" B and B is "on" C, then A is "on" C.
   *
   * While this isn't really true in real life (for example, you can
   *   have a key on a table which is under an umbrella, but the key
   *   itself might not be under the umbrella), it's probably enough
   *   for the level of detail we require here.
   *
   * If loctype is nil, any kind of containment is OK.
   *
   * containmenttype returns the locationtype our ancestor that's
   *   contained in obj, or nil if no such ancestor exists.
   *
   * We have to handle the case "iscontainedX(nil ...)" specially,
   *   since technically everything has nil as an eventual parent. If
   *   we get to TOP before nil, the object is not "in" nil, otherwise
   *   it is.
   */
  iscontainedX(obj, loctype, boolean) = {
    local p := self;

    /*
     * Find parent (or self) of p such that p.location = obj.
     */
    while (p <> nil and p <> TOP) {
      if (p.location = obj)
        break;
      else
        p := p.location;
    }

    if (p = nil) {
      if (obj = nil)
        return true;
      else
        return nil;
    }
    else if (p = TOP) {
      if (obj = TOP)
        return true;
      else
        return nil;
    }
    else if (loctype = nil or p.locationtype = loctype) {
      if (boolean)
        return true;
      else
        return p.locationtype;
    }
    else
      return nil;
  }
  containmenttype(obj) = { return self.iscontainedX(obj, nil, nil); }
  iscontained(obj, loctype) = {
    return self.iscontainedX(obj, loctype, true);
  }
  isin(obj) = { return self.iscontained(obj, 'in'); }
  ison(obj) = { return self.iscontained(obj, 'on'); }
  isunder(obj) = { return self.iscontained(obj, 'under'); }
  isbehind(obj) = { return self.iscontained(obj, 'behind'); }
;

/*
 * more stuff for Thing, but the compiler can't handle such a big
 *   definition.
 */
modify Thing
  /*
   * The usual rigamarole - lots of verX methods that tell the player
   *   what he can't do.
   *
   * Many of the verX methods are unnecessary since the requirements in
   *   the verbs will catch a lot of stuff. For example, verDoPutX
   *   tells the player he can't put the Thing anywhere (Item overrides
   *   this) - but this should never get called because putVerb
   *   requires cantake, which Things will return nil for (because
   *   verDoTake prints something).
   */

  /*
   * Generic "you can't do that" message.
   */
  verDoX(actor, v) = {
    "\^<<actor.subjthedesc>> can't <<v.sdesc>> <<self.objthedesc(actor)>>.";
  }
  
  /*
   * Actions that take only direct objects.
   * These are in alphabetical order.
   */
  verDoBoard(actor) = { self.verDoX(actor, boardVerb); }
  verDoBreak(actor) = { self.verDoX(actor, breakVerb); }
  verDoClean(actor) = {
    "\^<<actor.possessivedesc>> efforts to clean <<self.objthedesc(actor)>> have little effect.";
  }
  verDoClimb(actor) = { self.verDoX(actor, climbVerb); }
  verDoClose(actor) = { self.verDoX(actor, closeVerb); }
  verDoDetach(actor) = { self.verDoX(actor, detachVerb); }
  verDoDrink(actor) = { self.verDoX(actor, drinkVerb); }
  verDoDrop(actor) = { self.verDoX(actor, dropVerb); }
  verDoEat(actor) = {
    "Now why would <<actor.subjthedesc>> want to eat <<self.objthedesc(actor)>>?";
  }
  verDoEmpty(actor) = { self.verDoX(actor, emptyVerb); }
  verDoEnter(actor) = {
    "\^<<actor.subjthedesc>> can't enter <<self.objthedesc(actor)>>.";
  }
  verDoExit(actor) = {
    "\^<<actor.subjthedesc>> can't exit <<self.objthedesc(actor)>>.";
  }
  verDoFasten(actor) = { self.verDoX(actor, fastenVerb); }
  verDoFill(actor) = { self.verDoX(actor, fillVerb); }
  verDoFlip(actor) = { self.verDoX(actor, flipVerb); }
  verDoFollow(actor) = { self.verDoX(actor, followVerb); }
  verDoGetin(actor) = { self.verDoX(actor, getinVerb); }
  verDoGetoff(actor) = { self.verDoX(actor, getoffVerb); }
  verDoGeton(actor) = { self.verDoX(actor, getonVerb); }
  verDoGetout(actor) = { self.verDoX(actor, getoutVerb); }
  verDoGetoutfrombehind(actor) = {
    self.verDoX(actor, getoutfrombehindVerb);
  }
  verDoGetoutfromunder(actor) = {
    self.verDoX(actor, getoutfromunderVerb);
  }
  verDoGetunder(actor) = { self.verDoX(actor, getunderVerb); }
  verDoGetbehind(actor) = { self.verDoX(actor, getbehindVerb); }
  verDoHit(actor) = { "There's no reason to get huffy!"; }  
  verDoInspect(actor) = {}
  verDoKick(actor) = { self.verDoHit(actor); }
  verDoKnockon(actor) = { self.verDoX(actor, knockonVerb); }
  verDoLiein(actor) = { self.verDoLieon(actor); }
  verDoLieon(actor) = { self.verDoX(actor, lieonVerb); }
  verDoListento(actor) = {}
  verDoLookbehind(actor) = {
    "Trying to look behind <<self.objthedesc(actor)>> doesn't gain <<actor.subjthedesc>> anything.";    
  }
  verDoLookin(actor) = {
    "\^<<actor.subjthedesc>> <<actor.has>> no way of looking inside <<self.objthedesc(actor)>>.";
  }
  verDoLookon(actor) = {
    "There's nothing on <<self.objthedesc(nil)>>.";
  }
  verDoLookthrough(actor) = {
    "\^<<actor.subjthedesc>> can't look through <<self.objthedesc(actor)>>.";
  }
  verDoLookunder(actor) = {
    "Trying to look under <<self.objthedesc(actor)>> doesn't gain <<actor.subjthedesc>> anything.";   
  }
  verDoMove(actor) = { self.verDoX(actor, moveVerb); }
  verDoMoveN(actor) = { self.verDoMove(actor); }
  verDoMoveE(actor) = { self.verDoMove(actor); }
  verDoMoveS(actor) = { self.verDoMove(actor); }
  verDoMoveW(actor) = { self.verDoMove(actor); }
  verDoMoveNE(actor) = { self.verDoMove(actor); }
  verDoMoveNW(actor) = { self.verDoMove(actor); }
  verDoMoveSE(actor) = { self.verDoMove(actor); }
  verDoMoveSW(actor) = { self.verDoMove(actor); }
  verDoOpen(actor) = { self.verDoX(actor, openVerb); }
  verDoPoke(actor) = {
    "Poking <<self.objthedesc(actor)>> doesn't seem to have much effect.";
  }
  verDoPull(actor) = {
    "Pulling on <<self.objthedesc(actor)>> doesn't seem to have much effect.";
  }
  verDoPush(actor) = {
    "Pushing <<self.objthedesc(actor)>> doesn't seem to have much effect.";
  }
  verDoRead(actor) = {
    "\^<<self.subjthedesc>> <<self.doesnt>> have any text on <<self.objprodesc(nil)>>.";
  }
  verDoScrew(actor) = { self.verDoX(actor, screwVerb); }
  verDoSearch(actor) = {
    if (global.searchisexamine)
      self.verDoInspect(actor);
  }
  verDoSitin(actor) = { self.verDoX(actor, sitinVerb); }
  verDoSiton(actor) = { self.verDoX(actor, sitonVerb); }
  verDoSmell(actor) = { }
  verDoStand(actor) = { "\^<<actor.subjthedesc>> can't stand here."; }
  verDoSwitch(actor) = { self.verDoX(actor, switchVerb); }
  verDoTake(actor) = { self.verDoX(actor, XtakeVerb); }
  verDoTaste(actor) = {}
  verDoTouch(actor) = {}
  verDoTurn(actor) = { self.verDoX(actor, turnVerb); }
  verDoTurnoff(actor) = { self.verDoX(actor, turnoffVerb); }
  verDoTurnon(actor) = { self.verDoX(actor, turnonVerb); }
  verDoUnfasten(actor) = { self.verDoX(actor, unfastenVerb); }
  verDoUnplug(actor) = { self.verDoX(actor, unplugVerb); }
  verDoUntie(actor) = { self.verDoX(actor, untieVerb); }
  verDoUnscrew(actor) = { self.verDoX(actor, unscrewVerb); }
  verDoUnwear(actor) = {
    "\^<<actor.subjthedesc>> <<actor.isnt>> wearing <<self.objthedesc(actor)>>.";
  }
  verDoWear(actor) = { self.verDoX(actor, wearVerb); }

  /*
   * Actions that take indirect objects
   * These are in alphabetical order.
   */
  verDoAskabout(actor, io) = {
    "\^<<actor.subjthedesc>> can't ask <<self.objthedesc(actor)>> about anything.";
  }
  verIoAskabout(actor) = {}
  verDoAskfor(actor, io) = {
    "\^<<actor.subjthedesc>> can't ask <<self.objthedesc(actor)>> for anything.";
  }
  verIoAskfor(actor) = {}
  verDoAttachto(actor, io) = {
    "\^<<actor.subjthedesc>> can't attach <<self.objthedesc(actor)>> to anything.";
  }
  verIoAttachto(actor) = {
    "\^<<actor.subjthedesc>> can't attach anything to <<self.objthedesc(actor)>>.";
  }
  verDoAttackwith(actor, io) = {
    "Attacking <<self.objthedesc(actor)>>, while perhaps satisfying, would achieve little.";
  }
  verIoAttackwith(actor) = {
    "\^<<actor.subjthedesc>> can't attack anything with <<self.objthedesc(actor)>>.";
  }
  verDoCleanwith(actor, io) = { self.verDoClean(actor); }
  verIoCleanwith(actor) = {
    "\^<<self.subjthedesc>> <<self.isnt>> useful for cleaning things.";
  }
  verDoDetachfrom(actor, io) = { self.verDoDetach(actor); }
  verIoDetachfrom(actor) = {
    "There's nothing attached to <<self.objthedesc(actor)>>.";
  }
  verDoDigwith(actor, io) = {
    "\^<<actor.subjthedesc>> can't dig in
     <<self.objthedesc(actor)>>.";
  }
  verIoDigwith(actor) = {
    "\^<<self.subjthedesc>> <<self.isnt>> useful for digging.";
  }
  verDoGiveto(actor, io) = {
    "\^<<actor.subjthedesc>> can't give <<self.objthedesc(actor)>> to anyone.";
  }
  verIoGiveto(actor) = {
    "\^<<actor.subjthedesc>> can't give things to <<self.objthedesc(actor)>>.";
  }
  verDoHitwith(actor, io) = { self.verDoHit(actor); }
  verIoHitwith(actor) = {
    "Hitting things with <<self.objthedesc(actor)>> isn't helpful.";
  }
  verDoLock(actor) = {
    "\^<<actor.subjthedesc>> can't lock <<self.objthedesc(actor)>>.";
  }
  verDoLockwith(actor, io) = { self.verDoLock(actor); }
  verIoLockwith(actor) = {
    "\^<<actor.subjthedesc>> can't lock things with <<self.objthedesc(actor)>>.";
  }
  verDoMovewith(actor, io) = { self.verDoMove(actor); }
  verIoMovewith(actor) = {
    "\^<<actor.subjthedesc>> can't move things with <<self.objthedesc(actor)>>.";
  }
  verDoPlugin(actor, io) = {
    "\^<<actor.subjthedesc>> can't plug <<self.objthedesc(actor)>> into anything.";
  }
  verIoPlugin(actor) = {
    "\^<<actor.subjthedesc>> can't plug anything into <<self.objthedesc(actor)>>.";
  }
  verDoPutX(actor, io, loctype) = {
    "\^<<actor.subjthedesc>> can't put <<self.objthedesc(actor)>> anywhere.";
  }
  verDoPutin(actor, io) = { self.verDoPutX(actor, io, 'in'); }
  verDoPuton(actor, io) = { self.verDoPutX(actor, io, 'on'); }
  verDoPutunder(actor, io) = { self.verDoPutX(actor, io, 'under'); }
  verDoPutbehind(actor, io) = { self.verDoPutX(actor, io, 'behind'); }
  verDoPutthrough(actor, io) = {
    "\^<<actor.subjthedesc>> can't put <<self.objthedesc(actor)>> anywhere.";
  }
  verIoPutin(actor) = {
    "\^<<actor.subjthedesc>> can't put anything in <<self.objthedesc(actor)>>.";
  }
  verIoPuton(actor) = {
    "\^<<actor.subjthedesc>> can't put anything on <<self.objthedesc(actor)>>.";
  }
  verIoPutunder(actor) = {
    "\^<<actor.subjthedesc>> can't put anything under <<self.objthedesc(actor)>>.";
  }
  verIoPutbehind(actor) = {
    "\^<<actor.subjthedesc>> can't put anything behind <<self.objthedesc(actor)>>.";
  }
  verIoPutthrough(actor) = {
    "\^<<actor.subjthedesc>> can't put anything through <<self.objthedesc(actor)>>.";
  }
  verDoSayTo(actor, io) = {
    "\^<<actor.subjthedesc>> can't say that.";
  }
  verIoSayTo(actor) = {
    "\^<<actor.subjthedesc>> <<actor.doesnt>> know how to talk to <<self.objthedesc(actor)>>";
  }
  verDoScrewwith(actor, io) = { self.verDoScrew(actor); }
  verIoScrewwith(actor) = {
    "\^<<actor.subjthedesc>> can't screw anything with <<self.objthedesc(actor)>>.";
  }
  verDoShootat(actor, io) = {
    "\^<<actor.subjthedesc>> can't shoot anything with <<self.objthedesc(actor)>>.";    
  }
  verIoShootat(actor) = {
    "Shooting <<self.objthedesc(actor)>> wouldn't help <<actor.subjthedesc>> any.";
  }
  verDoShootwith(actor, io) = {
    "Shooting <<self.objthedesc(actor)>> wouldn't help <<actor.subjthedesc>> any.";
  }
  verIoShootwith(actor) = {
    "\^<<actor.subjthedesc>> can't shoot anything with <<self.objthedesc(actor)>>.";    
  }
  verDoShowto(actor, io) = {}
  verIoShowto(actor) = {
    "\^<<actor.subjthedesc>> can't show things to <<self.objthedesc(actor)>>.";
  }
  verIoTakefrom(actor) = {
    "\^<<actor.subjthedesc>> can't take anything from <<self.objthedesc(actor)>>.";
  }
  verIoTakeoff(actor) = {
    "\^<<actor.subjthedesc>> can't take anything off <<self.objthedesc(actor)>>.";
  }
  verIoTakeout(actor) = {
    "\^<<actor.subjthedesc>> can't take anything out of <<self.objthedesc(actor)>>.";
  }
  verIoTakeunder(actor) = {
    "\^<<actor.subjthedesc>> can't take anything out from under <<self.objthedesc(actor)>>.";
  }
  verIoTakebehind(actor) = {
    "\^<<actor.subjthedesc>> can't take anything from behind <<self.objthedesc(actor)>>.";
  }
  verDoTellabout(actor, io) = {
    "\^<<actor.subjthedesc>> can't tell <<self.objthedesc(actor)>> about anything.";
  }
  verIoTellabout(actor) = {}
  verDoTellto(actor, io) = { io.verDoTellabout(actor, self); }
  verIoTellto(actor) = {}
  verDoTieto(actor, io) = { self.verDoX(actor, tieVerb); }
  verDoThrowX(actor, io) = {
    "\^<<actor.subjthedesc>> can't throw <<self.objthedesc(actor)>>.";
  }
  verDoThrowat(actor, io) = { self.verDoThrowX(actor, io); }
  verIoThrowat(actor) = {
    "Throwing things at <<self.objthedesc(actor)>> isn't helpful.";
  }
  verDoThrowbehind(actor, io) = { self.verDoThrowX(actor, io); }
  verIoThrowbehind(actor) = {
    "\^<<actor.subjthedesc>> can't throw things behind <<self.objthedesc(actor)>>.";
  }
  verDoThrowthrough(actor, io) = { self.verDoThrowX(actor, io); }
  verIoThrowthrough(actor) = {
    "\^<<actor.subjthedesc>> can't throw things through <<self.objthedesc(actor)>>.";
  }
  verDoThrowto(actor, io) = { self.verDoThrowX(actor, io); }
  verIoThrowto(actor) = {
    "\^<<actor.subjthedesc>> can't throw things to <<self.objthedesc(actor)>>.";
  }
  verDoThrowunder(actor, io) = { self.verDoThrowX(actor, io); }
  verIoThrowunder(actor) = {
    "\^<<actor.subjthedesc>> can't throw things under <<self.objthedesc(actor)>>.";
  }
  verDoTouchwith(actor, io) = { self.verDoTouch(actor); }
  verIoTouchwith(actor) = {}
  verDoTurnto(actor, io) = {
    "\^<<actor.subjthedesc>> can't turn <<self.objthedesc(actor)>> to anything.";
  }
  verIoTurnto(actor) = {
    "\^<<actor.subjthedesc>> can't turn anything to <<self.objthedesc(actor)>>.";
  }
  verIoTypeon(actor) = {
    "\^<<actor.subjthedesc>> can't type anything on <<self.objthedesc(actor)>>.";
  }
  verDoUnlock(actor) = {
    "\^<<actor.subjthedesc>> can't unlock <<self.objthedesc(actor)>>.";
  }
  verDoUnlockwith(actor, io) = { self.verDoUnlock(actor); }
  verIoUnlockwith(actor) = {
    "\^<<actor.subjthedesc>> can't unlock things with <<self.objthedesc(actor)>>.";
  }
  verDoUnplugfrom(actor, io) = {
    "\^<<actor.subjthedesc>> can't unplug <<self.objthedesc(actor)>> from anything.";
  }
  verIoUnplugfrom(actor) = {
    "\^<<actor.subjthedesc>> can't unplug anything from <<self.objthedesc(actor)>>.";
  }
  verDoUntiefrom(actor, io) = {
    "\^<<actor.subjthedesc>> can't untie <<self.objthedesc(actor)>> from anything.";
  }
  verIoUntiefrom(actor) = {
    "\^<<actor.subjthedesc>> can't untie anything from <<self.objthedesc(actor)>>.";
  }
  verDoUnscrewwith(actor, io) = { self.verDoUnscrew(actor); }
  verIoUnscrewwith(actor) = {
    "\^<<actor.subjthedesc>> can't unscrew anything with <<self.objthedesc(actor)>>.";
  }

  verDoLight(actor) = { self.verDoX(actor, lightVerb); }
  verDoUnlight(actor) = { self.verDoX(actor, unlightVerb); }
  verDoLightwith(actor, io) = { self.verDoLight(actor); }
  verIoLightwith(actor) = {
    "<<actor.subjthedesc>> can't light anything with <<self.objthedesc(actor)>>.";
  }

  /*
   * Default doX and ioX methods. These mainly redirect to other
   *   methods.
   */
  doInspect(actor) = { self.ldesc; }
  doLiein(actor) = { self.doLieon(actor); }
  doListento(actor) = { self.listendesc; }
  doSearch(actor) = {
    if (global.searchisexamine)
      self.doInspect(actor);
    else {
      "\^<<actor.subjthedesc>> <<searchVerb.desc(actor)>> <<self.objthedesc(actor)>>, but <<actor.doesnt>> find anything.";
    }
  }
  doSmell(actor) = { self.smelldesc; }
  doTaste(actor) = { self.tastedesc;  }
  doTouch(actor) = { self.touchdesc; }
  ioAskabout(actor, dobj) = { dobj.doAskabout(actor, self); }
  ioAskfor(actor, dobj) = { dobj.doAskfor(actor, self); }
  ioAttackwith(actor, dobj) = { dobj.doAttackwith(actor, self); }
  ioGiveto(actor, dobj) = { dobj.doGiveto(actor, self); }
  ioTellabout(actor, dobj) = { dobj.doTellabout(actor, self); }
  ioTellto(actor, dobj) = { self.doTellabout(actor, dobj); }
  ioTouchWith(actor, dobj) = {
    "Nothing unusual happens when <<actor.subjthedesc>> <<actor.isplural ? "touch" : "touches">> <<dobj.objthedesc(actor)>> with <<self.objthedesc(actor)>>.";
  }

  
  /*
   * Testing stuff
   */
  verDoGimme(actor) = {}
  doGimme(actor) = {
    "You cheater!";
    self.movein(actor);
  }
  verDoLocate(actor) = {}
  doLocate(actor) = {
    local loc, lt;
    self.sdesc; " <<upper(self.locationtype)>> ";
    loc := self.location;
    if (loc = nil) {
      "nil";
      return;
    }
    else
      loc.sdesc;
    lt := loc.locationtype;
    loc := loc.location;
    while (loc <> nil and loc <> TOP) {
      " <<upper(lt)>> <<loc.sdesc>>";
      lt := loc.locationtype;
      loc := loc.location;
    }
    if (loc = TOP)
      " IN <<TOP.subjthedesc>>";
  }
  verDoWarpto(actor) = {}
  doWarpto(actor) = {
    "You are transported!"; P();
    parserGetMe().travelto(self);
  }
  verDoKnow(actor) = {}
  doKnow(actor) = {
    "You get a sudden burst of omniscience!";
    self.makeknownto(actor);
  }
  verDoWeight(actor) = {}
  doWeight(actor) = {
    local tot;
    tot := self.contentsweight(actor, ['in' 'on']);
    if (isclass(self, Item))
      tot += self.weight;
    "\^<<self.subjthedesc>> weighs <<tot>> units.";
  }
  verDoBulk(actor) = {}
  doBulk(actor) = {
    local tot;
    tot := self.contentsbulk(actor, ['in' 'on']);
    if (isclass(self, Item))
      tot += self.bulk;
    "\^Bulk of <<self.objthedesc(nil)>> is <<tot>> units.";
  }

  iobjCheck(actor, verb, dobj, prep) = {
  }
  dobjCheck(actor, verb, dobj, prep) = {
  }

  iobjGen(actor, verb, dobj, prep) = {
    "\^<<actor.subjthedesc>> can't <<verb.sdesc>> ";
    if (dobj) "<<dobj.objthedesc(actor)>> "; else "anything ";
    if (prep) "<<prep.sdesc>>"; else "to ";
    " <<self.objthedesc(actor)>>.";
    abort;
  }
  dobjGen(actor, verb, dobj, prep) = {
    self.verDoX(actor, verb);
    abort;
  }
;

/*
 * Floating objects are objects that are in more than one place, either at
 *   a single time, or over the course of the game.
 *
 * Note that a Floating is never implicitly made known to an actor by the
 *   rest of WorldClass, because it never appears in contents lists. For
 *   this reason, Floatings are known to all actors by default - you can
 *   disable this by setting known to nil.
 */
class Floating: Thing
  known = true
  isknownto(actor) = {
    if (self.known)
      return true;
    else
      return inherited.isknownto(actor);
  }
  locationOK = true                                     /* location is method */
;

/*
 * An Everywhere is in every location.
 * It can be a container, but you must be careful here, since if multiple
 *   actors put things in it, they'll all have access to all the contained
 *   items. (I.e., even though an Everywhere may appear to be multiple
 *   copies of the same object, it isn't.)
 */
class Everywhere: Floating
  /*
   * If the actor's location has the following property, this method
   *   will override the Everywhere. Specifically, if the property is
   *   nil, the Everywhere won't appear in the location.
   */
  roomprop = nil
  
  /*
   * If the actor's location has the any of the following properties, 
   *   and this Everywhere actually appears in the location (according
   *   to roomprop above), this method will be called in place of this
   *   Everywhere's ldesc. This allows you to customize the description
   *   of an Everywhere in certain rooms just by defining a method in
   *   those rooms.
   */
  roomdescprop = nil
  roomlistenprop = nil
  roomsmellprop = nil
  roomtasteprop = nil
  roomtouchprop = nil

  lastlocation = nil
  location = {
    if (global.lastactor = nil)
      self.lastlocation := nil;
    else if (global.lastactor.location = self)
              ;                           /* self.lastlocation stays the same */
    else if (global.lastactor.location = nil)
      self.lastlocation := nil;
    else if (self.roomprop) {
      local cloc := global.lastactor.location;
      while (cloc.(self.roomprop) <> true and isclass(cloc, Nestedroom))
        cloc := cloc.location;
      if (cloc.(self.roomprop) <> true)
        self.lastlocation := nil;
      else {
        self.lastlocation := cloc;
      }
    }
    else
      self.lastlocation := global.lastactor.location;
    return self.lastlocation;
  }
  ldesc = {
    local actor := global.lastactor;
    if (self.roomdescprop and
       defined(actor.location, self.roomdescprop))
      actor.location.(self.roomdescprop);
    else
      "\^<<actor.subjthedesc>> <<actor.doesnt>> see anything unusual about <<self.objthedesc(actor)>>.";
  }
  listendesc = {
    local actor := global.lastactor;
    if (self.roomlistenprop and
       defined(actor.location, self.roomlistenprop))
      actor.location.(self.roomlistenprop);
    else
      return inherited.listendesc;
  }
  smelldesc = {
    local actor := global.lastactor;
    if (self.roomsmellprop and
       defined(actor.location, self.roomsmellprop))
      actor.location.(self.roomsmellprop);
    else
      return inherited.smelldesc;
  }
  tastedesc = {
    local actor := global.lastactor;
    if (self.roomtasteprop and
       defined(actor.location, self.roomtasteprop))
      actor.location.(self.roomtasteprop);
    else
      return inherited.tastedesc;
  }
  touchdesc = {
    local actor := global.lastactor;
    if (self.roomtouchprop and
       defined(actor.location, self.roomtouchprop))
      actor.location.(self.roomtouchprop);
    else
      return inherited.touchdesc;
  }
;

/*
 * Special classes and objects the parser uses.
 *
 * Note about strObj and numObj: the parser does not call the usual methods
 *   on the verb to check the validity of these objects, so the error
 *   messages for these will be different.
 */
class String: Thing
  value = ''           /* parser magically sets this to value of typed string */
  sdesc = "string"
  verDoTypeon(actor, io) = {}
  doTypeon(actor, io) = {
    "\"Tap, tap, tap, tap...\"";
  }
  doSynonym('Typeon') = 'Enteron' 'Enterin' 'Enterwith'
  verDoSave(actor) = {}
  doSave(actor) = {
    if (save(self.value))
      "Save failed.";
    else {
      "Saved.";
      global.lastsave := global.turns;
    }
    abort;
  }
  verDoRestore(actor) = {}
  doRestore(actor) = {
    mainRestore(self.value);
    abort;
  }
  verDoScript(actor) = {}
  doScript(actor) = {
    logging(self.value);
    "Writing script file.";
    abort;
  }
  verDoSay(actor) = {}
  doSay(actor) = {
    /*
     * If the actor has a method that tells us what to do when he
     *   says something, call it; otherwise, just say something
     *   cute.
     */
    if (defined(actor, &speech_handler))
      actor.speech_handler(self.value);
    else
      "Okay, \"<<self.value>>\".";
  }
  verDoSayto(actor, io) = { self.verDoSay(actor); }
  doSayto(actor, io) = { self.doSay(actor); }
;
class Number: Thing
  value = 0            /* parser magically sets this to value of typed number */
  sdesc = "number"
  verDoPush(actor) = {}
  doPush(actor) = {
    "You need to specify what you want to type the number on.";
  }
  verDoTypeon(actor, io) = {}
  doTypeon(actor, io) = {
    "\"Tap, tap, tap, tap...\"";
  }
  doSynonym('Typeon') = 'Enteron' 'Enterin' 'Enterwith' 
  verIoTurnto(actor) = {}
  ioTurnto(actor, dobj) = { dobj.doTurnto(actor, self); }
  verDoUndo(actor) = {}
  doUndo(actor) = { undoVerb.undocommands(actor, self.value); }
  verDoFootnote(actor) = {}
  doFootnote(actor) = {
    printnote(self.value);
  }
  verDoAutosave(actor) = {}
  doAutosave(actor) = {
    if (value > 1)
    {
      global.turnsbetweensaves := value;
      "\b*** Autosave is set to every ";
      say(value);
      " turns  ***\n";
    }
    else "\b*** Autosave value is too low ***\n";
  }
;
class Listobj: Thing
  value = []
  sdesc = "list of numbers"
  noun = 'intlist'
  isknownto(actor) = { return true; }
  verIoTurnto(actor) = {}
  ioTurnto(actor, dobj) = { dobj.doTurnto(actor, self); }
;
strObj: String;
numObj: Number;
listObj: Listobj;

/*
 * Decorations are just Things.
 */
class Decoration: Thing
;

/*
 * Unimportants are Things that aren't important, but that can be referred
 *   to.
 */
class Unimportant: Thing
  /*
   * By default, don't let any verbs but 'inspect' give much feedback.
   */
  dobjGen(a, v, i, p) = {
    if (v <> inspectVerb) {
      "\^<<self.subjthedesc>> <<self.isnt>> important.";
       exit;
    }
  }
  iobjGen(a, v, i, p) = {
    if (v <> inspectVerb) {
      "\^<<self.subjthedesc>> <<self.isnt>> important.";
       exit;
    }
  }
;

/*
 * A Distant is an item that is too far away to manipulate, but can be
 *   seen. The class uses dobjGen and iobjGen to prevent any verbs from
 *   being used on the object apart from those defined in allowedverbs; using 
 *   any other verb results in the message "It's too far away." Instances of 
 *   this class should provide the normal item properties: sdesc, ldesc, 
 *   location, and vocabulary.
 *
 * Note that you can add other verbs to the list of allowed verbs; just
 *   redefine allowedverbs.
 */
class Distant: Decoration
  allowedverbs = [inspectVerb throwVerb shootVerb]
  dobjGen(a, v, i, p) = {
    if (find(self.allowedverbs, v) = nil) {
      "\^<<self.subjthedesc>> <<self.is>> too far away.";
      exit;
    }
  }
  iobjGen(a, v, d, p) = { self.dobjGen(a, v, d, p); }
;

/*
 * Parts are things that are parts of other things. These mainly provide a
 *   way to have decorations that float around with another thing. The
 *   partof method specifies what object the Part is connected to.
 *
 * If you change the parent, always use the setpartof method, or the parts
 *   list of the parent(s) won't get updated properly.
 */
class Part: Floating
  partof = nil
  setpartof(p) = {
    if (self.partof <> nil)
      self.partof.parts -= self;
    if (p <> nil) {       
      self.partof := p;
      p.parts += self;
    }
  }
  
  /* don't ever list this Thing in contents listings */
  isnoticeable(actor) = { return nil; }
  
  location = {
    if (self.partof <> nil)
      return self.partof.location;
  } 
  locationtype = {
    if (self.partof <> nil)
      return self.partof.locationtype;
  }
  locationtypenum = {
    if (self.partof <> nil)
      return self.partof.locationtypenum;
  }
  isknownto(actor) = {
    if (self.partof = nil)
      return nil;
    else
      return self.partof.isknownto(actor);
  }
  moveinto(obj, loctype, lttm) =
  {
    self.setpartof(nil);
    inherited.moveinto(obj, loctype, lttm);
    self.setpartof(obj);
  }
;

/*
 * The method islistable(actor) returns true if the object should be listed
 *   in room decriptions.
 *
 * The Listable class makes this method always return true.
 */
class Listable: Thing
  islistable(actor) = { return true; }
;

/*
 * Listablesmells, Listedablesounds, and Listabletouch's get listed in the
 *   olfactory and audio contents listings (including room descriptions).
 */
class Listablesmell: Thing
  islistablesmell(actor) = { return true; }
;
class Listablesound: Thing
  islistablesound(actor) = { return true; }
;
class Listabletouch: Thing
  islistabletouch(actor) = { return true; }
;

/*
 * Fixtures are things like furniture that get listed separately (i.e., in
 *   a sentence of their own) in room descriptions.
 *
 * They mainly provide a way to have an object listed in a custom way. ("An
 *   enormous mahogany desk stands in the center of the room" vs. "There is
 *   an enormous mahogany desk here.")
 */
class Fixture: Listable, Thing
  isfixture = true
  heredesc = {
    if (self.isdetermined)
      "\^<<self.subjsdesc>> <<self.is>> here.";
    else
      "There <<self.is>> <<self.subjadesc>> here.";
  }
;

/*
 * A Topic is something that an be "asked about" or "told about".
 * By default, all Topics are known to all Actors.
 */
class Topic: Thing
  istopic = true
;

/*
 * Making all topics known seems like it would be a good idea, but is
 *   actually a terrible idea. The problem with this is that Items are also
 *   Topics, so they inherit this, which is bad.
 *
 * So we have a special class for Topics that are known from the start.
 */
class Knowntopic: Topic
  isknownto(actor) = { return true; }
;

/*
 * An Item is something an actor can carry.
 * By default it's a Topic.
 */
class Item: Listable, Topic, Thing
  weight = 0
  bulk = 1

  /*
   * We have to verify insertion here because verIo methods don't get
   *   dobj parameter.
   */
  verDoPutX(actor, io, loctype) = {
    if (defined(io, &verIoPutX))
      io.verIoPutX(actor, self, loctype);
  }
  verDoPutin(actor, io) = { self.verDoPutX(actor, io, 'in'); }
  verDoPuton(actor, io) = { self.verDoPutX(actor, io, 'on'); }
  verDoPutunder(actor, io) = { self.verDoPutX(actor, io, 'under'); }
  verDoPutbehind(actor, io) = { self.verDoPutX(actor, io, 'behind'); }

  /*
   * "put through" is a bit different since "through" isn't a location
   *   type.
   */
  verDoPutthrough(actor, io) = {
    if (defined(io, &verIoPutthrough))
      io.verIoPutthrough(actor);
  }
  
  /*
   * Take methods.
   *
   * If the actor has the object, but the object is not in the actor's
   *   "top level" possessions,  we interpret "take X" as "move X to my
   *   top level". Players often use this syntax to take things out of
   *   containers.
   *
   * Floor objects (like Ground) are a special case - things are
   *   considered to be on the ground if they're in the containing
   *   room.
   *
   * Clothing is a special case. The player may say "take off hat" or
   *   "remove hat". We want to map these to verDoUnwear.
   */
  verDoTakeX(actor, obj, loctype) = {
    if (obj <> nil)
      if (isclass(obj, Floor)) {
        obj := actor.location;
        if (obj)
          if (isclass(obj, Floor))
            obj := actor.location.location;
        loctype := nil;
      }
    if (obj <> nil)
      if (not self.iscontained(obj, loctype)) {
        if (obj.isactor)
          "\^<<obj.subjthedesc>> <<obj.doesnthave>> <<self.subjdesc>>";
        else if (loctype = nil)
          "\^<<self.subjthedesc>> <<self.isnt>> there";
        else
          "\^<<self.subjthedesc>> <<self.isnt>> <<loctype>> <<obj.objthedesc(actor)>>";
        if (self.isin(actor))
          "  - <<actor.subjthedesc>> <<actor.has>> <<self.objprodesc(actor)>>.";
        else
          ".";
        return;
      }
    if (self.isclothing and self.isin(actor) and self.isworn)
      self.verDoUnwear(actor);
  }
  verDoTake(actor) = { self.verDoTakeX(actor, nil, nil); }
  verDoTakefrom(actor, io) = { self.verDoTakeX(actor, io, nil); }
  verDoTakeoff(actor, io) = { self.verDoTakeX(actor, io, 'on'); }
  verDoTakeout(actor, io) = { self.verDoTakeX(actor, io, 'in'); }
  verDoTakeunder(actor, io) = { self.verDoTakeX(actor, io, 'under'); }
  verDoTakebehind(actor, io) = {
    self.verDoTakeX(actor, io, 'behind');
  }
  doTake(actor) = {
    local tot, m;
    if (self.isin(actor)) {
      if (self.isclothing and self.isworn) {
        self.doUnwear(actor);
        return;
      }
      else if (self.location = actor) {
        "\^<<actor.subjthedesc>> already <<actor.has>> <<self.objthedesc(actor)>>.";
        return;
      }
    }
    actor.ioPutX(actor, self, 'in');
  }
  doTakefrom(actor, io) = { self.doTake(actor); }
  doTakeoff(actor, io) = { self.doTake(actor); }
  doTakeout(actor, io) = { self.doTake(actor); }
  doTakeunder(actor, io) = { self.doTake(actor); }
  doTakebehind(actor, io) = { self.doTake(actor); }

  /*
   * Drop methods.
   */
  verDoDrop(actor) = {
    if (not self.isin(actor))
      "\^<<actor.subjthedesc>> <<actor.doesnthave>> <<self.objthedesc(actor)>>.";
  }
  doDrop(actor) = {
    actor.location.roomdrop(actor, self);
  }

  verDoGiveto(actor, io) = {}
  verDoShowto(actor, io) = {}
  verDoThrowat(actor, io) = {}
  verDoThrowbehind(actor, io) = {}
  verDoThrowthrough(actor, io) = {}
  verDoThrowto(actor, io) = {}
  verDoThrowunder(actor, io) = {}
;

/*
 * An Edible can be eaten. When the player eats an Edible, the time he can
 *   go before he must eat again is increased by the foodvalue of the item.
 *   This value defaults to actor.mealtime - the number of turns the player
 *   can go without eating at the beginning of the game.
 *
 * To disable starvation, make the actor's starvationcheck method nil.
 */
class Edible: Item
  foodvalue = { return global.lastactor.mealtime; }
  verDoEat(actor) = {}
  doEat(actor) =  {
    self.eatdesc(actor);
    self.addfoodvalue(actor);
    self.movein(nil);
  }
  addfoodvalue(actor) = {
    actor.turnsuntilstarvation += self.foodvalue;
  }
  eatdesc(actor) = "That was delicious!";
;

/*
 * Drinkables are just like Edibles, but they deal with dehydration.
 *
 * To disable dehydration, make the actor's dehydration method nil.
 */
class Drinkable: Item
  drinkvalue = { return global.lastactor.drinktime; }
  verDoDrink(actor) = {}
  doDrink(actor) = {
    self.drinkdesc(actor);
    self.adddrinkvalue(actor);
    self.movein(nil);
  }
  drinkdesc(actor) = {
    "\^<<self.subjthedesc>> quenches <<actor.possessivedesc>> thirst.";
  }
  adddrinkvalue(actor) = {
    actor.turnsuntildehydration += self.drinkvalue;
  }
;

/*
 * Note that the locationtype for Clothing that is being worn is
 *   'in'. Preinit guarantees this.
 */
class Clothing: Item
  isclothing = true
  isworn = nil
  wornprop = {
    if (self.isworn)
      "(being worn)";
  }
  moveinto(obj, loctype, lttm) = {
    if (self.isworn) {
      "(taking off <<self.objthedesc(nil)>> first)\n";
      Outhide(true);
      self.doUnwear(self.location);
      Outhide(nil);
    }
    return inherited.moveinto(obj, loctype, lttm);
  }
  verDoWear(actor) = {
    if (self.isworn and self.location = actor)
      "\^<<actor.youre>> already wearing <<self.objthedesc(actor)>>.";
  }
      doWear(actor) = {
    if (self.location <> actor) {
      "(taking <<self.objthedesc(actor)>> first)\n";
      if (not self.movein(actor))
        return;
    }
    self.wearmessage(actor);
    self.isworn := true;
    if (find(self.properties, &wornprop) = nil)
      self.properties += &wornprop;
  }
  verDoUnwear(actor) = {}
  doUnwear(actor) = {
    self.unwearmessage(actor);
    self.isworn := nil;
    self.properties -= &wornprop;
  }
  wearmessage(actor) = {
    "\^<<actor.subjthedesc>> <<actor.is>> now wearing <<self.objthedesc(actor)>>.";
  }
  unwearmessage(actor) = {
    "\^<<actor.subjthedesc>> <<unwearVerb.desc(actor)>> <<self.objthedesc(actor)>>.";
  }
;

/*
 * Lightsources are items that provide light.
 */
class Lightsource: Item
  islightsource = true
  ldesc = {
    "\^"; self.longdesc; ". ";
    if (self.islit)
      "\^<<self.subjprodesc>> <<self.litmessage>> ";
    if (self.isburnt)
      "\^<<self.subjprodesc>> <<self.deadmessage>> ";
  }
  longdesc = self.adesc
  islit = nil
  isburnt = nil
  burnsdown = nil
  burntime = 0
  canturnon = nil
  canrelight = true
  litprop = { if (self.islit) "(providing light)"; }
  verDoLight(actor) = {
    if (self.isburnt) {
      "\^<<self.subjthedesc>> <<self.deadmessage>> ";
    }
    else if (self.islit) {
      "\^<<self.subjthedesc>> <<self.islitmessage>> ";
    }
  }
  verDoUnlight(actor) = {
    if (not self.islit) {
      "\^<<self.subjthedesc>> <<self.notlitmessage>> ";
    }
  }
  doLight(actor) = {
    local waslit := actor.location.islit;
    self.islit := true;
    global.lightsources += self;
    "\^<<actor.subjthedesc>> <<self.vLight(actor)>> <<self.objthedesc(actor)>>";
    if (actor.location.islit and not waslit) {
      ", lighting the area.\b";
      actor.location.enter(actor);
    }
    else ". ";
    if (self.burnsdown)
      notify(self, &lightsout, 0);
  }
  doUnlight(actor) = {
    self.islit := nil;
    global.lightsources -= self;
    "\^<<actor.subjthedesc>> <<self.vUnlight(actor)>> <<self.objthedesc(actor)>>. ";
    if (self.burnsdown) {
      if (not self.canrelight) self.isburnt := true;
      unnotify(self, &lightsout);
    }
  }
  verDoTurnon(actor) = { 
    if (self.canturnon) self.verDoLight(actor);
    else inherited.verDoTurnon(actor);
  }
  verDoTurnoff(actor) = {
    if (self.canturnon) self.verDoUnlight(actor);
    else inherited.verDoTurnoff(actor);
  }
  doTurnon(actor) = {
    if (self.canturnon) self.doLight(actor);
    else inherited.doTurnon(actor);
  }
  doTurnoff(actor) = {
    if (self.canturnon) self.doUnlight(actor);
    else inherited.doTurnoff(actor);
  }
  lightsout = {
    self.burntime--;
    switch (self.burntime) {
    case 0:
      P(); I();
      "\^<<self.subjthedesc>> <<self.burnmessage>> ";
      self.islit := nil;
      global.lightsources -= self;
      self.isburnt := true;
      self.burntime := 0;
      break;
    case 5:
    case 10:
      P(); I();
      "\^<<self.subjthedesc>> <<self.flickermessage>> ";
      break;
    }
  }
  vLight(actor) = {
    if (self.canturnon) turnonVerb.desc(actor);
    else lightVerb.desc(actor);
  }
  vUnlight(actor) = {
    if (self.canturnon) turnoffVerb.desc(actor);
    else unlightVerb.desc(actor);
  }
  flickermessage = "flickers."
  burnmessage = "goes out!"
  litmessage = "is lit."
  islitmessage = "is already lit."
  notlitmessage = "is not lit."
  deadmessage = "is burned out."
;

/*
 * Readables are items that can be read. (such as books.)
 */
class Readable: Thing
  readdesc = { "\^<<self.prodesc>> reads, \"...\""; }
  verDoRead(actor) = {}
  doRead(actor) = { self.readdesc; }
;

/*
 * Buttons - can be pressed. Put action code in doPush.
 * Set touchpress = nil to disallow "touch button" = "press button".
 */
class Button: Thing
  sdesc = "button"
  touchpress = true
  verDoPush(actor) = {}
  verDoTouch(actor) = {
    if (self.touchpress)
      self.verDoPush(actor);
    else
      inherited.verDoTouch(actor);
  }
  doTouch(actor) = {
    if (self.touchpress)
      self.doPush(actor);
    else
      inherited.doTouch(actor);
  }
;

/*
 * Dials - set minsetting and maxsetting. dial.setting is current setting.
 */
class Dial: Thing
  sdesc = "dial"
  minsetting = 1
  maxsetting = 10
  setting = 1
  ldesc = { 
    "\^<<self.subjthedesc>> can be turned to settings numbered from <<self.minsetting>> to <<self.maxsetting>>."; P(); I(); "It's currently set to <<self.setting>>.";
  }
  verDoTurn(actor) = {}
  doTurn(actor) = { askio(toPrep); }
  verDoTurnto(actor, io) = {}
  doTurnto(actor, io) = {
    if (io = numObj) {
      if (numObj.value < self.minsetting or numObj.value > self.maxsetting) {
        "There's no such setting.";
      }
      else if (numObj.value <> self.setting) {
        self.setting := numObj.value;
        "Okay, it's now turned to <<self.setting>>.";
      }
      else {
        "It's already set to <<self.setting>>.";
      }
    }
    else
      "\^<<actor.subjthedesc>> can't turn
       <<self.objthedesc(actor)>> to that.";
  }
;

/*
 * Switches can be switched on or off. The isactive method returns the
 *   current value.
 *
 * Set pullable to nil to disallow the syntax "pull switch".
 * Set moveable to nil to disallow the syntax "move switch".
 */
class Switch: Thing
  isactive = nil
  pullable = true
  moveable = true
  sdesc = "switch"
  verDoSwitch(actor) = {}
  doSwitch(actor) = {
    self.isactive := not self.isactive;
    "Okay, <<self.subjthedesc>> is now switched ";
    if (self.isactive)
      "on";
    else
      "off";
    ".";
  }
  verDoFlip(actor) = {}
  doFlip(actor) = { self.doSwitch(actor); }
  verDoPull(actor) = {
    if (not self.pullable)
      "\^<<actor.subjprodesc>> can't pull <<self.objthedesc(actor)>>.";
  }
  doPull(actor) = { self.doSwitch(actor); }
  verDoMove(actor) = {
    if (not self.moveable)
      "\^<<actor.subjprodesc>> can't move <<self.objthedesc(actor)>>.";
  }
  verDoTurnon(actor) = {
    if (self.isactive)
      "\^<<self.subjprodesc>> <<self.is>> already turned on.";
  }
  doTurnon(actor) = {
    self.isactive := true;
    "Okay, <<self.subjthedesc>> <<self.is>> now turned on.";
  }
  verDoTurnoff(actor) = {
    if (not self.isactive)
      "\^<<self.subjthedesc>> <<self.is>> already turned off.";
  }
  doTurnoff(actor) = {
    self.isactive := nil;
    "Okay, <<self.subjthedesc>> <<self.is>> now turned off.";
  }
;

/*
 * A Transparent is something you can see into and out of. Be sure to put
 *   this on the left of a multiple inheritance list since the pass methods
 *   may be defined by other classes in the list (like Openable).
 */
class Transparent: Thing
  passcanseein(actor, obj, loctype) = { return true; }
  passcanseeout(actor, obj, loctype) = { return true; }
  passcanseeacross(actor, obj, loctype) = { return true; }
  verDoLookthrough(actor) = { }
  doLookthrough(actor) = { self.thrudesc; }
;

/*
 * A Holder has contents (things that are on, in, under, behind, ... it)
 *   Don't use Holders for game objects, though. Inherit from Container,
 *   Surface, Over, or Front instead.
 */
class Holder: Thing
  isholder = true
  iscontainer = nil
  issurface = nil
  isover = nil
  isfront = nil
  acceptsput(actor, loctype) = {
    switch (loctype) {
      case 'in':
        return self.iscontainer; break;
      case 'on':
        return self.issurface; break;
      case 'under':
        return self.isover; break;
      case 'behind':
        return self.isfront; break;
      default:
        return true; break;
    }
  }
  contents = []
  maxweight = -1                   /* can support an infinite amount of stuff */
  maxbulk = -1                        /* can hold an infinite amount of stuff */

  /*
   * Sense passing. See defaults in Thing for more info.
   */
  passcansee(actor, obj, loctype) = { return true; }
  passcantouch(actor, obj, loctype) = { return true; }
  passcantake(actor, obj, loctype) = { return true; }
  passcansmell(actor, obj, loctype) = { return true; }
  passcanhear(actor, obj, loctype) = { return true; }
  passcanspeakto(actor, obj, loctype) = { return true; }

  /*
   * Contents can be removed
   */
  verIoTakefrom(actor) = {}
  ioTakefrom(actor, dobj) = { dobj.doTake(actor); }

  /*
   * Look on, in, under, behind, etc.
   *
   * See if the actor can see into the container. If not, the canX
   *   method will explain why not.
   */
  doLookX(actor, loctype, emptyquiet) = {
    local tot, locn;
    if (not actor.canseecontents(self, nil, loctype))
      return;
    locn := [] + actor.location;
    tot := listcontents(self, actor, nil, nil, nil, loctype, true, true, nil, locn);
    if (tot = 0 and not emptyquiet)
      "There doesn't appear to be anything <<loctype>> <<self.objthedesc(nil)>>.";
  }

  /*
   * The general put in/on/under/behind method and accompanying
   *   verification method. The verify method is actually called by
   *   verDoPutX.
   */
  verIoPutX(actor, dobj, loctype) = {
    if (dobj = self) {
      "\^<<actor.subjthedesc>> can't put <<dobj.objthedesc(actor)>> <<loctype>> <<dobj.reflexivedesc>>!";
      return;
    }
    if (dobj.location = self and dobj.locationtype = loctype) {
      "\^<<dobj.subjthedesc>> <<dobj.is>> already <<loctype>> <<self.objthedesc(nil)>>.";
      return;
    }
  }

  /*
   * Verify that the object or list can be inserted into the container
   *   without exceeding the container's weight and bulk limits. Rather
   *   than just checking the max for the current container, it
   *   recursively  goes through the hierachy of containers to make
   *   sure that none of  the maximums are exceeded. This will continue
   *   until it reaches TOP (since Room is a container).
   * verifyinsertion ignores those containers with a max of -1. I may
   *   also change the default weightexceeded and bulkexceeded messages
   *   as well, since they won't really detail at the moment why
   *   exactly your load is too heavy or bulky.
   */
  verifyinsertion(actor, obj, loctype, loclist, valprop, check, msg) =
  {
    local i;
    local lt, temploc;
    local tot, m;
    local islist;

    if (datatype(obj) = 7)
      islist := true;
    else
      islist := nil;

    /* get weight of stuff in container */
    tot := self.(check)(actor, [loctype]);

    /* add weight of stuff to be put into container */
    if (islist)
      for (i := 1; i <= length(obj); i++) {
        tot += obj[i].(valprop);
        tot += obj[i].(check)(actor, global.loctake);
      }
    else {
      tot += obj.(valprop);
      tot += obj.(check)(actor, global.loctake);
    }
    temploc := self;
    lt := find(global.loctypes, loctype);
    while (isclass(temploc, Holder))
    {
      m := temploc.(loclist[lt]);
      if ((m <> -1) and (tot > m))
      {
        if (islist)
          self.(msg)(obj[1], loctype);
        else
          self.(msg)(obj, loctype);
        return nil;
      }
      lt := find (global.loctypes, temploc.locationtype);
      temploc := temploc.location;
    }
    return true;
  }
  
  /*
   * General put method. Note that dobj can be a list, in which case we
   *   want to know if all the objects in the list can be put in the
   *   container together. This is to handle Attachables.
   *
   * If dobj is a single object and has a doPutX method, we call it
   *   instead. (This is so objects can override the normal put
   *   behavior, not just the containers.)
   */
  ioPutX(actor, dobj, loctype) = {
    local i, islist;
    if (datatype(dobj) = 7)
      islist := true;
    else
      islist := nil;
    if (not islist)
      if (defined(dobj, &doPutX)) {
        dobj.doPutX(actor, self, loctype);
        return;
      }
    if (not self.verifyinsertion(actor, dobj, loctype, global.locmaxweight, &weight, &contentsweight, &weightexceeded))
      return;
    if (not self.verifyinsertion(actor, dobj, loctype, global.locmaxbulk, &bulk, &contentsbulk, &bulkexceeded))
      return;
    if (islist) {
      for (i := 1; i <= length(dobj); i++)
        dobj[i].moveinto(self, loctype, global.loctake);
      self.putmessage(dobj[1], loctype);
    }
    else {
      if (dobj.moveinto(self, loctype, global.loctake))
        self.putmessage(dobj, loctype);
    }
  }

  /*
   * Message methods for the user to customize as desired.
   */
  weightexceeded(dobj, loctype) = {
    "\^<<self.subjthedesc>> can't hold any more.";
  }
  bulkexceeded(dobj, loctype) = {
    "\^<<dobj.subjthedesc>> won't fit <<loctype>>
    <<self.objthedesc(nil)>>.";
  }
  putmessage(dobj, loctype) = { "Done."; }
;

/*
 * A Container has things in it.
 */
class Container: Holder
  iscontainer = true
  contdesc(actor) = { self.doLookin(actor); }
  verDoLookin(actor) = {}
  doLookin(actor) = { self.doLookX(actor, 'in', nil); }
  verIoPutin(actor) = {}
  ioPutin(actor, dobj) = { self.ioPutX(actor, dobj, 'in'); }
  verIoTakeout(actor) = {}
  ioTakeout(actor, dobj) = { dobj.doTake(actor); }
;

/*
 * A Qcontainer is just a container whose contents are not listed in a room
 *   description.
 */
class Qcontainer: Container
  incontentslisted(obj) = { return nil; }
;

/*
 * A Surface has things on it.
 */
class Surface: Holder
  issurface = true
  verDoLookon(actor) = {}
  doLookon(actor) = { self.doLookX(actor, 'on', nil); }
  verIoPuton(actor) = {}
  ioPuton(actor, dobj) = { self.ioPutX(actor, dobj, 'on'); }
  verIoTakeoff(actor) = {}
  ioTakeoff(actor, dobj) = { dobj.doTake(actor); }
;

/*
 * A Qsurface is just a surface whose contents are not listed in a room
 *   description.
 */
class Qsurface: Surface
  oncontentslisted(obj) = { return nil; }
;

/*
 * An Over has things under it.
 */
class Over: Holder
  isover = true
  verDoLookunder(actor) = {}
  doLookunder(actor) = { self.doLookX(actor, 'under', nil); }
  verIoPutunder(actor) = {}
  ioPutunder(actor, dobj) = { self.ioPutX(actor, dobj, 'under'); }
  verIoThrowunder(actor) = {}
  ioThrowunder(actor, dobj) = { self.ioPutunder(actor, dobj); }
  verIoTakeunder(actor) = {}
  ioTakeunder(actor, dobj) = { dobj.doTake(actor); }
;

/*
 * A Qover is just an Over whose contents are not listed in a room
 *   description.
 */
class Qover: Over
  undercontentslisted(obj) = { return nil; }
;

/*
 * A Front has things behind it.
 */
class Front: Holder
  isfront = true
  verDoLookbehind(actor) = {}
  doLookbehind(actor) = { self.doLookX(actor, 'behind', nil); }
  verIoPutbehind(actor) = {}
  ioPutbehind(actor, dobj) = { self.ioPutX(actor, dobj, 'behind'); }
  verIoThrowbehind(actor) = {}
  ioThrowbehind(actor, dobj) = { self.ioPutbehind(actor); }
  verIoTakebehind(actor) = {}
  ioTakebehind(actor, dobj) = { dobj.doTake(actor); }
;

/*
 * A Qfront is just a Front whose contents are not listed in a room
 *   description.
 */
class Qfront: Front
  behindcontentslisted(obj) = { return nil; }
;

/*
 * An Openable is a Container that can be either open or closed. When
 *   closed, it blocks senses.
 */
class Openable: Container
  isopen = nil
  ldesc = {
    "\^<<self.subjthedesc>> <<self.is>> currently ";
    if (self.isopen)
      "open.";
    else
      "closed.";
  }
  verDoOpen(actor) = {
    if (self.isopen)
      "\^<<self.subjthedesc>> <<self.is>> already open.";
  }
  doOpen(actor) = {
    local i;
    self.isopen := true;
    self.openmessage(actor);
    for (i := 1; i <= length(global.actorlist); i++) {
      local o := global.actorlist[i];
      if (o.canseecontents(self, true, 'in')) {
        self.makecontentsknownto(o);
      }
    }
  }
  verDoClose(actor) = {
    if (not self.isopen)
      "\^<<self.subjthedesc>> <<self.is>> already closed.";
  }
  doClose(actor) = {
    self.isopen := nil;
    self.closemessage(actor);
  }
  openmessage(actor) = {
    "Opened. ";
    self.doLookX(actor, 'in', nil);
  }
  closemessage(actor) = { "Closed."; }
  passgen(actor, obj, loctype, passmethod) = {
    if (loctype <> nil and loctype <> 'in')
      return inherited.(passmethod)(actor, obj, loctype);
    if (self.isopen)
      return true;
    else {
      "\^<<actor.youll>> have to open <<self.objthedesc(nil)>> first.";
      return nil;
    }   
  }
  passcanseein(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcansee);
  }
  passcanseeout(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcansee);
  }
  passcantouchin(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcantouch);
  }
  passcantouchout(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcantouch);
  }
  passcantakein(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcantake);
  }
  passcantakeout(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcantake);
  }
  passcansmellin(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcansmell);
  }
  passcansmellout(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcansmell);
  }
  passcanhearin(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcanhear);
  }
  passcanhearout(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcanhear);
  }
  passcanspeaktoin(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcanspeakto);
  }
  passcanspeaktoout(actor, obj, loctype) = {
    return self.passgen(actor, obj, loctype, &passcanspeakto);
  }
;

/*
 * A Qopenable is just an Openable whose contents are not listed in a room
 *   description.
 */
class Qopenable: Openable
  incontentslisted(obj) = { return nil; }
;

/*
 * A Lockable must be unlocked before it can be opened.
 * If key is not nil, the object it returns is the key required to unlock
 *   the Thing. If the key is nil, the Thing can be locked and unlocked
 *   without a key. To make a something impossible to unlock (or lock),
 *   just set its key to an object the player can never get.
 */
class Lockable: Openable
  islocked = true
  isopen = nil
  key = nil
  verDoOpen(actor) = {
    if (self.isopen)
      "\^<<self.subjthedesc>> <<self.is>> already open.";
    else if (self.islocked)
      "\^<<actor.youll>> have to unlock <<self.objthedesc(actor)>> first.";
  }
  verDoLock(actor) = {
    if (self.islocked)
      "\^<<self.subjthedesc>> <<self.is>> already locked.";
    else if (self.isopen)
      "\^<<actor.youll>> have to close <<self.objthedesc(actor)>> first.";
  }
  doLock(actor) = {
    if (self.key = nil)
      self.doLockwith(actor, nil);
    else
      askio(withPrep);
  }
  verDoLockwith(actor, io) = {
    if (self.islocked)
      "\^<<self.subjthedesc>> <<self.is>> already locked.";
    else if (self.key = nil)
      "\^<<actor.subjthedesc>> <<actor.doesnt>> need anything to lock <<self.objprodesc(actor)>>.";
    else if (io <> self.key)
      "\^<<io.subjthedesc>> <<io.doesnt>> fit the lock.";
    else if (self.isopen)
      "\^<<actor.youll>> have to close <<self.objprodesc(actor)>> first.";
  }
  doLockwith(actor, io) = {
    self.islocked := true;
    self.lockmessage(actor);
  }
  verDoUnlock(actor) = {
    if (not self.islocked)
      "\^<<self.subjthedesc>> <<self.is>> not locked.";
  }
  doUnlock(actor) = {
    if (self.key = nil)
      self.doUnlockwith(actor, nil);
    else
      askio(withPrep);
  }
  verDoUnlockwith(actor, io) = {
    if (not self.islocked)
      "\^<<self.subjthedesc>> <<self.is>> not locked.";
    else if (self.key = nil)
      "\^<<actor.subjthedesc>> <<actor.doesnt>> need anything to unlock <<self.objthedesc(actor)>>.";
    else if (io <> self.key)
      "\^<<io.subjthedesc>> <<io.doesnt>> fit the lock.";
  }
  doUnlockwith(actor, io) = {
    self.islocked := nil;
    self.unlockmessage(actor);
  }
  lockmessage(actor) = { "Locked."; }
  unlockmessage(actor) = { "Unlocked."; }
;

/*
 * A Qlockable is just a Lockable whose contents are not listed in a room
 *   description.
 */
class Qlockable: Lockable
  incontentslisted(obj) = { return nil; }
;

/*
 * A Key is something that can be used to lock and unlock things.
 */
class Key: Thing
  sdesc = "key"
  verIoLockwith(actor) = {}
  verIoUnlockwith(actor) = {}
  ioLockwith(actor, dobj) = { dobj.doLockwith(actor, self); }
  ioUnlockwith(actor, dobj) = { dobj.doUnlockwith(actor, self); }
;

/*
 * An Attachable can be attached to other things.
 *
 * tieable determines whether "tie" can be used instead of "attach".
 * plugable determines whether "plug...into" can be used instead of
 *   "attach".
 *
 * maxattachments specifies the maximum number of Attachpoints this thing
 *   can be attached to at once.
 *
 * attachesto lists the Attachpoints this thing can be attached to.
 *
 * attachedto lists the Attachpoints this thing is currently attached to.
 *
 * attachedtodesc prints a sentence of the form
 *  "The <Thing> is attached to ..."
 *
 * NOTE: Attachables are Items. For Attachables that can't be taken, you
 *   don't need this class; just override the relevant methods in a Thing.
 */
class Attachable: Item
  tieable = nil
  plugable = nil
  maxattachments = 1
  attachesto = []
  attachedto = []
  attachedprop = { "(<<self.listattachments>>)"; }
  ldesc = { self.attachedtodesc; }
  attachedtodesc = {      
    "\^<<self.subjthedesc>> <<self.is>> <<self.listattachments>>.";
  }
  listattachments = {
    local i;
    local len := length(self.attachedto);
    local attached;
    if (self.tieable)
      attached := 'tied';
    else if (self.plugable)
      attached := 'plugged in';
    else
      attached := 'attached';
    if (len = 0) {
      "not <<attached>> to anything";
      return;
    }
    "<<attached>> to ";
    for (i := 1; i <= len; i++) {
      self.attachedto[i].objthedesc(nil);
      if (i = len)
        break;
      else if (i = len - 1) {
        if (len > 2)
          ", and ";
        else
          " and ";
      }
      else
        ", ";
    }
  }
  verDoAttachto(actor, io) = {
    local attach, detaching;
    if (self.tieable) {
      attach := 'tie';
      detaching := 'untying';
    }
    else if (self.plugable) {
      attach := 'plug in';
      detaching := 'unplugging';
    }
    else {
      attach := 'attach';
      detaching := 'detaching';
    }
    if (find(self.attachesto, io) = nil)
      "\^<<actor.subjthedesc>> can't <<attach>> <<self.objthedesc(actor)>> to <<io.objthedesc(actor)>>.";
    else if (length(self.attachedto) >= self.maxattachments)
      "\^<<actor.subjthedesc>> can't <<attach>> <<self.objthedesc(actor)>> to <<io.objthedesc(actor)>> without <<detaching>> <<self.objprodesc(actor)>> from something else first.";
    else if (length(io.attachedto) >= io.maxattachments)
      "\^<<actor.subjthedesc>> can't <<attach>> <<io.objthedesc(actor)>> to <<self.objthedesc(actor)>> without <<detaching>> <<io.objprodesc(actor)>> from something else first.";
    else if (self.location <> io.location or self.locationtype <> io.locationtype) {

      /*
       * If this thing has to be attached to something in a
       *   different location, make sure we can put the thing
       *   in the new location - the location where the
       *   attachment point is.
       */
      if (not actor.canputinto(io.location, io.locationtype, true))
        actor.canputinto(io.location, io.locationtype, nil);
    }
  }

  /*
   * By default, things that are attached must be in the same location
   *   as their attachpoints. Aside from making general sense, this
   *   prevents an actor from attaching something to a fixed object and
   *   then walking off with the (still attached) item.
   */
  doAttachto(actor, io) = {
    if (self.location <> io.location or self.locationtype <> io.locationtype)
      if (not self.moveto(io.location, io.locationtype))
        return;
    self.attachedto += io;
    if (length(self.attachedto) = 1)
      self.properties += &attachedprop;
    io.attachedto += self;
    if (length(io.attachedto) = 1)
      io.properties += &attachedprop;
    self.attachedmessage(actor, io);
  }
  attachedmessage(actor, obj) = { "Done."; }
  verDoTieto(actor, io) = {
    if (self.tieable or io.tieable)
      self.verDoAttachto(actor, io);
    else
      inherited.verDoTieto(actor, io);
  }
  doTieto(actor, io) = {
    if (self.tieable or io.tieable)
      self.doAttachto(actor, io);
    else
      inherited.doTieto(actor, io);
  }
  verDoPlugin(actor, io) = {
    if (self.plugable or io.plugable)
      self.verDoAttachto(actor, io);
    else
      inherited.verDoPlugin(actor, io);
  }
  doPlugin(actor, io) = {
    if (self.plugable or io.plugable)
      self.doAttachto(actor, io);
    else
      inherited.doPlugin(actor, io);
  }
  verDoDetach(actor) = {
    if (length(self.attachedto) = 0) {
      local attached;
      if (self.tieable)
        attached := 'tied';
      else if (self.plugable)
        attached := 'plugged in';
      else
        attached := 'attached';
      "\^<<self.subjthedesc>> <<self.is>> not <<attached>> to anything.";
    }
    else if (length(self.attachedto) > 1) {
      local detach;
      if (self.tieable)
        detach := 'untie';
      else if (self.plugable)
        detach := 'unplug';
      else
        detach := 'detach';
      "\^<<actor.subjprodesc>> <<actor.isplural ? "need" : "needs">> to be more specific about what <<actor.objprodesc(nil)>> <<actor.isplural ? "want" : "wants">> to <<detach>> <<self.objthedesc(actor)>> from.";
    }
    else
      self.verDoDetachfrom(actor, self.attachedto[1]);
  }
  doDetach(actor) = { self.doDetachfrom(actor, self.attachedto[1]); }
  verDoDetachfrom(actor, io) = {
    if (find(self.attachedto, io) = nil) {
      local attached;
      if (self.tieable)
        attached := 'tied';
      else if (self.plugable)
        attached := 'plugged in';
      else
        attached := 'attached';
      "\^<<self.subjthedesc>> <<self.is>> not <<attached>> to <<io.objthedesc(nil)>>.";
    }
  }
  doDetachfrom(actor, io) = {
    self.attachedto -= io;
    if (length(self.attachedto) = 0)
      self.properties -= &attachedprop;
    io.attachedto -= self;
    if (length(io.attachedto) = 0)
      io.properties -= &attachedprop;
    self.detachedmessage(actor, io);
  }
  detachedmessage(actor, obj) = { "Done."; }
  verDoUntiefrom(actor, io) = {
    if (self.tieable or io.tieable)
      self.verDoDetachfrom(actor, io);
    else
      inherited.verDoUntiefrom(actor, io);
  }
  doUntiefrom(actor, io) = {
    if (self.tieable or io.tieable)
      self.doDetachfrom(actor, io);
    else
      inherited.doUntiefrom(actor, io);
  }
  verDoUntie(actor) = {
    if (self.tieable)
      self.verDoDetach(actor);
    else
      inherited.verDoUntie(actor);
  }
  doUntie(actor) = {
    if (self.tieable)
      self.doDetach(actor);
    else
      inherited.doUntie(actor);
  }
  verDoUnplugfrom(actor, io) = {
    if (self.plugable or io.plugable)
      self.verDoDetachfrom(actor, io);
    else
      inherited.verDoUnplugfrom(actor, io);
  }
  doUnplugfrom(actor, io) = {
    if (self.plugable or io.plugable)
      self.doDetachfrom(actor, io);
    else
      inherited.doUnplugfrom(actor, io);
  }
  verDoUnplug(actor) = {
    if (self.plugable)
      self.verDoDetach(actor);
    else
      inherited.verDoUnplug(actor);
  }
  doUnplug(actor) = {
    if (self.plugable)
      self.doDetach(actor);
    else
      inherited.doUnplug(actor);
  }

  /*
   * Construct a list of all things this thing is attached to, and all
   *   things the attached things are attached to, etc. to get a list
   *   of all the things this thing is connected to.
   */
  connectedto = {
    local i, j, l;
    l := [] + self;
    for (i := 1; i <= length(l); i++) {
      if (length(l[i].attachedto) > 0) {
        for (j := 1; j <= length(l[i].attachedto); j++) {
          if (find(l, l[i].attachedto[j]) = nil)
            l += attachedto[j];
        }
      }       
    }
    return cdr(l);
  }
  
  /*
   * When an Attachable is moved, it takes all its attachments with it.
   */
  moveinto(obj, loctype, lttm) = {
    local i, l;
    if (self.location = obj and self.locationtype = loctype)
      return nil;
    l := self.connectedto;
    inherited.moveinto(obj, loctype, lttm);
    for (i := 1; i <= length(l); i++)
      l[i].inheritedmoveinto(obj, loctype, lttm);
    return true;
  }
  inheritedmoveinto(obj, loctype, lttm) = {
    return inherited.moveinto(obj, loctype, lttm);
  }

  /*
   * When the player tries to put this thing in a container, we have to
   *   make sure that all its attachments will fit as well. To assure
   *   this, we put them in the container in one big lump.
   */
  doPutX(actor, io, loctype) = {
    local l;
    l := self.connectedto;
    l := ([] + self) + l;
    io.ioPutX(actor, l, loctype);
  }
  
  /*
   * Only takeable if not attached to any untakeable things, so we
   *   must verify that everything it is connected to is takable. As
   *   this method calls itself recursively, we pass a list of objects
   *   that we have already checked. This list is consulted to ensure
   *   that we do not check an object twice, thus avoiding infinite
   *   recursion.
   */
  istakeable(actor, objl) = {
    local i, l;
    local objlist;
    l := self.connectedto;
    for (i := 1; i <= length(l); i++) {
      if (find(objl, self) = nil) {
        local r;
        Outhide(true);
        objlist := objl + ([] + self);
        r := l[i].istakeable(actor, objlist);
        Outhide(nil);
        if (not r) {
          local detach;
          if (self.tieable)
            detach := 'untie';
          else if (self.plugable)
            detach := 'unplug';
          else
            detach := 'detach';
          "\^<<actor.youll>> have to <<detach>> <<self.objthedesc(actor)>> from <<l[i].objthedesc(actor)>> first.";
          return nil;
        }
      }
    }
    return true;
  }
;

/*
 * An Attachpoint can have things attached to it.
 * Attachpoints can be Items or plain old (fixed) Things.
 */
class Attachpoint: Thing
  maxattachments = 1
  attachedto = []

  /*
   * We have the Attachable handle everything. These methods just pass
   *   control along to it.
   */
  verIoAttachto(actor) = {}
  ioAttachto(actor, dobj) = { dobj.doAttachto(actor, self); }
  verIoDetachfrom(actor) = {}
  ioDetachfrom(actor, dobj) = { dobj.doDetachfrom(actor, self); }
  verIoTieto(actor) = { }
  ioTieto(actor, dobj) = { dobj.doTieto(actor, self); }
  verIoUntiefrom(actor) = { }
  ioUntiefrom(actor, dobj) = { dobj.doUntiefrom(actor, self); }
  verIoPlugin(actor) = { }
  ioPlugin(actor, dobj) = { dobj.doPlugin(actor, self); }
  verIoUnplugfrom(actor) = { }
  ioUnplugfrom(actor, dobj) = { dobj.doUnplugfrom(actor, self); }
;

/*
 * A Connection holds rooms, and knows how to pass (or block) senses
 *   between them. (At least, that's the theory.)
 */
class Connection: Container
;

/*
 * A Room holds items and actors.
 */
class Room: Container
  /*
   * Every room has walls, ground, and ceiling unless we say otherwise.
   *   If these are set to true, we'll use the standard floating
   *   decorations. Otherwise those decorations won't be reachable from
   *   the room.
   */
  walls = true
  ground = true
  ceiling = true

  /*
   * Show the status line.
   *
   * Check the system to see if this is an HTML-enabled run-time.
   * If so, generate an HTML-style banner; otherwise, generate an
   *   old-style banner.
   */
  statusLine =
  {
    if (systemInfo(__SYSINFO_SYSINFO) = true and systemInfo(__SYSINFO_HTML) = 1)
    {
      local scor := '';
      "<banner id=StatusLine height=previous border> <body bgcolor=statusbg text=statustext><b>";
      self.statusRoot;
      "</b><tab align=right><i>";
      if (global.datedisp)
        scor := scor + ' ' + showdate();
      if (global.timedisp)
        scor := scor + ' ' + showtime();
      if (global.scoredisp)
        scor := scor + ' ' + showscore(global.score, global.turns);
      say(scor); " ";
      "</i></banner>";
    }
    else
    {
      self.statusRoot;
      "\n\t";
    }
  }

  /*
   * Construct the basis of the status line.
   */
  statusRoot =
  {
    local stat, str;
    local i, toprint;
    stat := outcapture(true);
    self.banner;
    if (proptype(global, &statusline) = 6)
      global.statusline;
    str := outcapture(stat);

    /*
     * If the status line does not exceed the maximum allowable
     *   width, print it as usual. Otherwise, print as much as we
     *   can fit, followed by an ellipsis.
     */
    if (length(str) <= global.statuslinewidth) {
      say(str);
    }
    else {
      toprint := substr(str, 1, global.statuslinewidth - 4);
      toprint += ' ...';
      say(toprint);
    }
  }

  /*
   * Construct the string to be displayed in the status line for the
   *   player's location.
   */   
  banner = {
    local actor := global.lastactor;
    if (self.islit) self.tdesc;
    else "In the dark";
    if (actor and not isclass(self, Nestedroom)) {
      if (actor.location = self) {
        switch (actor.position) {
          case 'sitting':
            ", sitting down";
            break;
          case 'lying':
            ", lying down";
            break;
        }
      }
    }
  }

  /*
   * Describe the room.
   *
   * Print the room description if we're being verbose.
   */
  describe(actor, verbose) = {
    local l, o, i, len, dark, locn;
    dark := not actor.canseecontents(self, true, 'in');
    if (dark) {
      P(); I(); "It's too dark to see.\n";
    }
    if (verbose and not dark) {
      P();
      self.ldesc; "\n";
    }
    if (not dark) {
      locn := [] + self;
      listcontents(self, actor, nil, true, true, nil, true, true, nil, locn);
    }
  }
  lookaround(actor, verbose) = {
    self.banner;
    self.describe(actor, verbose);
  }

  /*
   * The following three functions are called by smellVerb,
   *   listentoVerb, and touchVerb. They're analagous lookaround above
   *   (hence the names), and like describe (which lookaround calls)
   *   are overridden in Nestedroom.
   *
   * Note that lookaround lists for all senses. These three only list
   *   for their particular senses.
   */
  smellaround(actor, verbose) = {
    local tot, locn;
    locn := [] + self;
    tot := listcontents(self, actor, nil, nil, nil, nil, nil, nil, &cansmell, locn);
    if (tot = 0)
      "\^<<actor.subjthedesc>> <<actor.doesnt>> smell anything unusual.";
  }
  listenaround(actor, verbose) = {
    local tot, locn;
    locn := [] + self;
    tot := listcontents(self, actor, nil, nil, nil, nil, nil, nil, &canhear, locn);
    if (tot = 0)
      "\^<<actor.subjthedesc>> <<actor.doesnt>> hear anything unusual.";
  }
  feelaround(actor, verbose) = {
    local tot, locn;
    locn := [] + self;
    tot := listcontents(self, actor, nil, nil, nil, nil, nil, nil, &cantouch, locn);
    if (tot = 0)
      "\^<<actor.subjthedesc>> <<actor.doesnt>> feel anything unusual.";
  }

  isseen = nil

  /*
   * Routines that are called when an actor enters or leaves the room.
   */
  enter(actor) = {
    local alreadyknown := self.isknownto(actor);

    /*
     * Check the system to see if this is an HTML-enabled run-time.
     * If so, change the background and text colors to those
     *   defined in the Room. If the colors are not defined, they
     *   will be reset to the default colors.
     */
    if (actor = parserGetMe() and systemInfo(__SYSINFO_SYSINFO) = true and systemInfo(__SYSINFO_HTML) = 1) {
      "<body ";
      if (defined(self, &bgcolor)) "bgcolor=<<self.bgcolor>>";
      if (defined(self, &textcolor)) " text=<<self.textcolor>>";
      ">";
    }
    if (self.islit) {
      local i;
      for (i := 1; i <= length(global.actorlist); i++) {
        local o := global.actorlist[i];
        self.makeknownto(o);
        self.makecontentsknownto(o);
      }
    }
    if (actor = parserGetMe()) {
      self.lookaround(actor, not alreadyknown or global.verbose);
      if (self.islit) self.isseen := true;
    }
  }
  leave(actor) = {
  }

  /*
   * This gets called when an actor drops something in the room.
   */
  roomdrop(actor, obj) = {
    self.ioPutX(actor, obj, 'in');
  }
  
  /*
   * Rooms should generally pass all senses across so that things in
   *   the same room can interact with each other.
   *
   * For connections between rooms that senses can pass through, use
   *   Connections or define smart sense passing methods in TOP.
   *
   * Don't set islit directly - it's a method that checks to see if any 
   *   light sources are in the room. Instead, call the darken and
   *   lighten methods.
   */
  ambientlight = true
  islit = {
    local i, l;
    if (self.ambientlight) return true;

    /*
     * If any of the lightsources can "see" the room, the room is
     *   lit.
     */
    l := global.lightsources;
    for (i := length(l); i > 0; i--) {
      if (l[i].seepath(self, nil, true))
        return true;
    }
        
    return nil;
  }
  darken = { self.ambientlight := nil; }
  lighten = { self.ambientlight := true; }
  passcanseeacross(actor, obj, selfloctype, objloctype) = {
    if (selfloctype <> objloctype)
      return nil;
      
    /*
     * Things that are themselves lit can see across the room. This
     *   is mainly to prevent infinite recursion in islit.
     */
    if (obj <> nil)
      if (obj.islit)
        return true;
          
    if (self.islit)
      return true;
    else {
      "\^<<actor.subjthedesc>> can't see anything - it's too dark.";
      return nil;
    }
  }

  verDoStand(actor) = {}
  doStand(actor) = {
    actor.position := 'standing';
    "\^<<actor.subjthedesc>> <<actor.is>> now standing.";
  }

  /*
   * The roomAction(actor, verb, directObject, preposition, 
   *   indirectObject) method is activated for each player command; by 
   *   default, all it does is call the room's location's roomAction 
   *   method if the room is inside another room.
   */
  roomAction(a, v, d, p, i) = {
    if (self.location) self.location.roomAction(a, v, d, p, i);
  }

;

/*
 * Darkroom - a Room that is not lit.
 */
class Darkroom: Room
  ambientlight = nil
;

/*
 * Outside - a Room with no walls or ceiling.
 */
class Outside: Room
  walls = nil
  ground = true
  ceiling = nil
;

/*
 * Nestedrooms are things an actor can be in, on, under, or behind. The
 *   actor is still in the main room, however, and if he looks around he'll
 *   see the containing room. Likewise, if he drops things, they'll wind up
 *   in the containing room, not in the nested room.
 *
 * This is useful for things like chairs that restrict the player's
 *   actions, but aren't really separate locations in their own right. For
 *   totally separate (but contained) locations, just put a room inside
 *   another room.
 *
 * We impose the restriction that actors can't touch or take anything in
 *   the containing room without getting up first. (This is right more
 *   often than not.) You can turn this off by setting reachsurroundings to
 *   true. Alternatively, you can make specific objects reachable by
 *   putting them in the reachable list. (Things in the reachable list can
 *   be containers too, in which case all the contained items are reachable
 *   as well, subject to the container's whims.)
 *
 * NOTE: Making a new Nestedroom class should only be attempted by a
 *   WorldClass expert! There are (unfortunately) many places in WorldClass
 *   where the class "Nestedroom" is hard-coded. You'll have to be sure to
 *   fix all these if you make your own Nestedroom substitute. Nestedroom
 *   bugs are very subtle and take a long time to find, so proceed with
 *   extreme caution!
 */
class Nestedroom: Room
  tdesc = {
    if (self.location <> nil)
      "<<self.location.tdesc>>, ";
    else
      caps();
    "<<global.lastactor.locationtype>> <<self.objsdesc(nil)>>";
  }
  describe(actor, verbose) = {
    self.location.describe(actor, verbose);
  }
  smellaround(actor, verbose) = {
    self.location.smellaround(actor, verbose);
  }
  listenaround(actor, verbose) = {
    self.location.listenaround(actor, verbose);
  }
  feelaround(actor, verbose) = {
    self.location.feelaround(actor, verbose);
  }
  reachsurroundings = nil
  reachable = []                      /* only used if reachsurroundings = nil */
  noexit = {
    local actor := global.lastactor;
    "\^<<actor.youll>> have to get <<global.locopposites[actor.locationtypenum]>> <<self.objthedesc(actor)>> first.";
    return nil;
  }
  
  /*
   * Rooms are Containers. Nestedrooms may not actually be able to have
   *   things in them, so we disable these methods we've inherited from
   *   Container.
   */
  verIoPutin(actor) = {
    "\^<<actor.subjthedesc>> can't put anything in <<self.objthedesc(actor)>>.";
  }
  verIoTakeout(actor) = {
    "\^<<actor.subjthedesc>> can't take anything out of <<self.objthedesc(actor)>>.";
  }

  /*
   * Sense passing
   */
  passmsg(actor, obj) = {
    "\^<<actor.youll>> have to get <<global.locopposites[actor.locationtypenum]>> <<self.objthedesc(actor)>> to reach <<obj.objthedesc(actor)>>.";
  } 
  passreachable(actor, obj, loctype) = {
    local i;
    if (self.reachsurroundings)
      return true;

    /*
     * If the object the actor is trying to reach is self, is the 
     *   location of self (rooms are containers), is in the 
     *   reachable list, or is contained in an item in the 
     *   reachable list, allow the sense to pass through.
     */
    if (obj = self or obj = self.location)
      return true;
    else for (i := length(self.reachable); i > 0; i--) {
      if (obj = self.reachable[i])
        return true;
      if (obj.iscontained(self.reachable[i], nil))
        return true;
    }
      
    self.passmsg(actor, obj);
    return nil;   
  }
  passcantouchout(actor, obj, loctype) = {
    return self.passreachable(actor, obj, loctype);
  }
  passcantakeout(actor, obj, loctype) = {
    return self.passreachable(actor, obj, loctype);
  }

  roomdrop(actor, obj) = { self.location.roomdrop(actor, obj); }
  
  islit = { return self.location.islit; } 
  darken = { self.location.darken; }
  lighten = { self.location.lighten; }
  enter(actor) = {
    if (self.location <> nil)
      self.location.enter(actor);
  }
  leave(actor) = {
    if (self.location <> nil)
      self.location.leave(actor);
  }
  ground = { return self.location.ground; }
  grounddesc = { return self.location.grounddesc; }
  ceiling = { return self.location.ceiling; }
  ceilingdesc = { return self.location.ceilingdesc; }
  walls = { return self.location.walls; }
  wallsdesc = { return self.location.wallsdesc; }
  
  /*
   * Subtle case: If an Actor is in the Nestedroom when the Nestedroom
   *   is moved into a new location, the contents of the location must
   *   be made known to the actor.
   */
  moveinto(obj, loctype, lttm) = {
    local o, i, a;

    /*
     * Move the Nestedroom. (We have to do this first so that the
     *   sense reach code will succeed for the Actor.)
     */
    if (inherited.moveinto(obj, loctype, lttm) = nil)
      return nil;

    /*
     * Find containing Room
     */
    o := obj;
    while (o <> nil and not isclass(o, Room))
      o := o.location;

    /*
     * If the destination is a Room, or is contained in a Room,
     *   make that Room's contents known to every actor in this
     *   Nestedroom.
     */
    if (o <> nil) {
      if (isclass(o, Room)) {
        for (i := 1; i <= length(self.contents); i++) {
          a := self.contents[i];
          if (isclass(a, Actor)) {
            o.makecontentsknownto(a);
            o.makeknownto(a);
          }
        }
      }
    }
  }
;

/*
 * Curtain: a nested room that is 'behind' another room, rather than
 *   simply being 'in' another room. For example, a curtain that the player
 *   could hide behind.
 *
 * The Curtain is defined as a behind-type nested room which the player
 *   can "hide behind", yet still see out of. Should another actor come
 *   into the outer room he would not be able to see the player.
 *
 * Only passcanseein, passcantouchin, and passcantakein were overridden.
 *   This is because sounds and smells should not be stopped by the
 *   curtain. As long as the entering NPC had code to call the appropriate
 *   canX methods, then sounds in the behind-room (such as the player
 *   carrying a cookoo clock), would be noticed, and appropriate action
 *   could be taken (such as the NPC finding the actor).
 */
class Curtain: Nestedroom
  tdesc = {
    if (self.location) self.location.tdesc;
    else caps();
    ", <<global.lastactor.position>> <<global.lastactor.locationtype>> <<self.objthedesc(nil)>>";
  }
  verDoExit(actor) = {
    if (not actor.iscontained(self, 'behind'))
      "\^<<actor.youre>> not behind <<self.objthedesc(actor)>>.";
  }
  doExit(actor) = {
    actor.moveto(self.location, self.locationtype);
    actor.position := 'standing';
    "\^<<actor.subjthedesc>> <<actor.isplural ? "get" : "gets">> out from behind <<self.subjthedesc>>. ";
  }
  verGoOut(actor) = { self.verDoExit(actor); }
  goOut(actor) = {
    actor.position := 'standing';
    "\^<<actor.subjthedesc>> <<actor.isplural ? "get" : "gets">> out from behind <<self.subjthedesc>>.\b";
    return self.location;
  }
  verDoGetbehind(actor) = {}
  doGetbehind(actor) = {
    actor.movebehind(self);
    actor.position := 'standing';
    "\^<<actor.youre>> now standing behind <<self.objthedesc(actor)>>.";
  }
  passcanseein(actor, obj, loctype) = { return nil; }
  passcantouchin(actor, obj, loctype) = { return nil; }
  passcantakein(actor, obj, loctype) = { return nil; }
;

/*
 * Entrylimiter: This class should only be used with nestedrooms, and
 *   redefines the verDoEnter to automatically make sure the nestedroom can
 *   hold our bulk weight. The original WorldClass only provides this kind
 *   of check for Items, and lets actors slip through the rules. This is
 *   implemented as a seperate class since the nestedroom objects (like
 *   Chair, Stool, etc.) define verDoEnter to empty {}. I didn't want to
 *   have to make the same modifications to each class, so leave you to
 *   define it seperately in your nestedroom's inheritance declaration, as
 *   in:
 *
 *   chair: Entrylimiter, Fixture, Chair
 *
 * Note that Entrylimiter should be at the left.
 */
class Entrylimiter: Nestedroom
  verDoEnter(actor) = {
    if (not self.verifyinsertion(self, actor, 'in', global.locmaxweight, &weight, &contentsweight, &weightexceeded))
      return;
    if (not self.verifyinsertion(self, actor, 'in', global.locmaxbulk, &bulk, &contentsbulk, &bulkexceeded))
      return;
  }
  weightexceeded(dobj, loctype) = {
    "\^<<self.subjthedesc>> can't hold <<dobj.possessivedesc>> weight.";
  }
;

/*
 * A Table is a table. It can have things on it and under it. The contdesc
 *   lists the contents. By default the ldesc just lists the contents.
 * Since we don't generally expect things to be under tables, the contdesc
 *   doesn't say "there's nothing under the table" when that is the case.
 *
 * If you want actors to be able to sit on the table, inherit from Chair as
 *   well.
 */
class Table: Surface, Over
  sdesc = "table"
  ldesc = { self.contdesc(global.lastactor); }
  contdesc(actor) = {
    self.doLookX(actor, 'on', nil);
    " ";
    self.doLookX(actor, 'under', true);
  }
;

/*
 * Chairs can be sat in and stood on.
 * By default you have to get out/off of the chair to reach anything in the
 *   containing room. You can turn this off by setting reachsurroundings to
 *   true, or by putting specific objects in the reachable list.
 *
 * Disable standing by setting standable = nil.
 *
 * NOTE: We map all the "enter" verbs to "enter ___" so that users only
 *   have to override verDoEnter and and doEnter not have to worry about
 *   the myriad other methods. Ditto for doExit and verDoExit.
 *
 * Note that if this Chair is standable, (ver)doGeton will be different
 *   from (ver)doEnter.
 */
class Chair: Nestedroom
  standable = nil
  sdesc = "chair"
  tdesc = {
    if (self.location)
      self.location.tdesc;
    else
      caps();
    ", <<global.lastactor.position>> <<global.lastactor.locationtype>> <<self.objthedesc(nil)>>";
  }
  verGoOut(actor) = { self.verGoUp(actor); }
  goOut(actor) = { return self.goUp(actor); }
  verGoUp(actor) = {}
  goUp(actor) = {
    if (defined(self, &up))
      self.up;
    else
      self.doGetoff(actor);
    return nil;
  }
  verDoEnter(actor) = {}
  verDoSiton(actor) = { self.verDoEnter(actor); }
  verDoSitin(actor) = { self.verDoEnter(actor); }
  verDoGetin(actor) = { self.verDoEnter(actor); }
  verDoGeton(actor) = { self.verDoEnter(actor); }
  verDoExit(actor) = {
    if (not actor.iscontained(self, 'in'))
      "\^<<actor.youre>> not in <<self.objthedesc(actor)>>.";
  }
  verDoGetoff(actor) = { self.verDoExit(actor); }
  verDoGetout(actor) = { self.verDoExit(actor); }
  verDoStand(actor) = {}
  doEnter(actor) = {
    if (actor.movein(self)) {
      actor.position := 'sitting';
      "\^<<actor.youre>> now sitting in <<self.objthedesc(actor)>>.";
    }
  }
  doSiton(actor) = { self.doEnter(actor); }
  doSitin(actor) = { self.doEnter(actor); }
  doGetin(actor) = { self.doEnter(actor); }
  doGeton(actor) = {
    if (not self.standable) {
      self.doEnter(actor);
      return;
    }
    if (actor.moveon(self)) {
      actor.position := 'standing';
      "\^<<actor.youre>> now standing on <<self.objthedesc(actor)>>.";
    }
  }
  doExit(actor) = {
    if (actor.moveto(self.location, self.locationtype)) {
      actor.position := 'standing';
      "\^<<actor.subjthedesc>> <<actor.is>> now standing.";
    } 
  }
  doGetout(actor) = { self.doExit(actor); }
  doStand(actor) = { self.doExit(actor); }
  doGetoff(actor) = { self.doExit(actor); }
;

/*
 * Stools can be sat on and stood on. The only real diference between
 *   stools and chairs is syntactical. You sit "in" chairs and sit "on"
 *   stools.
 *
 * By default you have to get off of the stool to reach anything in the
 *   containing room. You can turn this off by setting reachsurroundings to
 *   true, or by putting specific objects in the reachable list.
 *
 * Disable standing by setting standable = nil.
 *
 * NOTE: We map all the "enter" verbs to "enter ___" so that users only
 *   have to override verDoEnter and and doEnter not have to worry about
 *   the myriad other methods. Ditto for doExit and verDoExit.
 *
 * Note that if this Stool is standable, (ver)doGeton will be different
 *   from (ver)doEnter.
 */
class Stool: Nestedroom
  standable = true
  sdesc = "stool"
  tdesc = {
    if (self.location)
      self.location.tdesc;
    else
      caps();
    ", <<global.lastactor.position>> <<global.lastactor.locationtype>> <<self.objthedesc(nil)>>";
  }
  verGoUp(actor) = {}
  goUp(actor) = {
    if (defined(self, &up))
      self.up;
    else
      self.doGetoff(actor);
    return nil;
  }
  verDoEnter(actor) = {}
  verDoSiton(actor) = { self.verDoEnter(actor); }
  verDoSitin(actor) = { self.verDoEnter(actor); }
  verDoGetin(actor) = { self.verDoEnter(actor); }
  verDoGeton(actor) = { self.verDoEnter(actor); }
  verDoExit(actor) = {
    if (not actor.iscontained(self, 'on'))
      "\^<<actor.youre>> not on <<self.objthedesc(actor)>>.";
  }
  verDoGetoff(actor) = { self.verDoExit(actor); }
  verDoGetout(actor) = { self.verDoExit(actor); }
  verDoStand(actor) = {}
  doEnter(actor) = {
    if (actor.moveon(self)) {
      actor.position := 'sitting';
      "\^<<actor.youre>> now sitting on <<self.objthedesc(actor)>>.";
    }
  }
  doSitin(actor) = { self.doEnter(actor); }
  doGetin(actor) = { self.doEnter(actor); }
  doSiton(actor) = { self.doEnter(actor); }
  doGeton(actor) = {
    if (not self.standable) {
      self.doEnter(actor);
      return;
    }   
    if (actor.moveon(self)) {
      actor.position := 'standing';
      "\^<<actor.youre>> now standing on <<self.objthedesc(actor)>>.";
    }
  }
  doExit(actor) = {
    if (actor.moveto(self.location, self.locationtype)) {
      actor.position := 'standing';
      "\^<<actor.subjthedesc>> <<actor.is>> now standing.";
    }
  }
  doGetout(actor) = { self.doExit(actor); }
  doStand(actor) = { self.doExit(actor); }
  doGetoff(actor) = { self.doExit(actor); }
;

/*
 * A Ledge is any flat surface you can sit, stand, or lie on. A ledge can
 *   also hold things.
 *
 * As in Chair and Stool, we map many ver methods to verDoEnter, verDoExit,
 *   and doExit so only these two need to be overridden. However, since
 *   there are many ways to "enter" a ledge, you'll have to handle each of
 *   these methods separately.
 */
class Ledge: Surface, Nestedroom
  sdesc = "ledge"
  tdesc = {
    if (self.location)
      self.location.tdesc;
    else
      caps();
    ", <<global.lastactor.position>> <<global.lastactor.locationtype>> <<self.objthedesc(nil)>>";
  }
  verGoUp(actor) = {}
  goUp(actor) = {
    if (defined(self, &up))
      self.up;
    else
      self.doGetoff(actor);
    return nil;
  }
  verDoEnter(actor) = {}
  verDoSiton(actor) = { self.verDoEnter(actor); }
  verDoSitin(actor) = { self.verDoEnter(actor); }
  verDoLieon(actor) = { self.verDoEnter(actor); }
  verDoLiein(actor) = { self.verDoEnter(actor); }
  verDoGetin(actor) = { self.verDoEnter(actor); }
  verDoGeton(actor) = { self.verDoEnter(actor); }
  verDoExit(actor) = {
    if (not actor.iscontained(self, 'on'))
      "\^<<actor.youre>> not on <<self.objthedesc(actor)>>.";
  }
  verDoGetoff(actor) = { self.verDoExit(actor); }
  verDoGetout(actor) = { self.verDoExit(actor); }
  verDoStand(actor) = {}
  doSiton(actor) = {
    if (actor.moveon(self) or actor.iscontained(self, 'on')) {
      actor.position := 'sitting';
      "\^<<actor.youre>> now sitting on <<self.objthedesc(actor)>>.";
    }
  }
  doSitin(actor) = { self.doSiton(actor); }
  doGetin(actor) = { self.doSiton(actor); }
  doLieon(actor) = {
    if (actor.moveon(self) or actor.iscontained(self, 'on')) {
      actor.position := 'lying';
      "\^<<actor.youre>> now lying on <<self.objthedesc(actor)>>.";
    }
  }
  doLiein(actor) = { self.doLieon(actor); }
  doGeton(actor) = {
    if (actor.moveon(self) or actor.iscontained(self, 'on')) {
      actor.position := 'standing';
      "\^<<actor.youre>> now standing on <<self.objthedesc(actor)>>.";
    }
  }

  /*
   * Disallow "enter ledge" - ask the player what he really means.
   */
  doEnter(actor) = {
    "Please be more specific. \^<<actor.subjthedesc>> can sit, stand or lie on <<self.objthedesc(actor)>>.";
  }

  doExit(actor) = {
    if (actor.moveto(self.location, self.locationtype)) {
      actor.position := 'standing';
      "\^<<actor.subjthedesc>> <<actor.is>> now standing.";
    }
  }
  doGetout(actor) = { self.doExit(actor); }
  doStand(actor) = { self.doExit(actor); }
  doGetoff(actor) = { self.doExit(actor); }
;

/*
 * A Bed is a bed. You can sit, lie, or stand on it. It can also have
 *   things under it.
 */
class Bed: Ledge, Qover
  sdesc = "bed"
  tdesc = {
    if (self.location)
      self.location.tdesc;
    else
      caps();
    ", <<global.lastactor.position>> <<global.lastactor.locationtype>> <<self.objthedesc(nil)>>";
  }
;

/*
 * A Desk can have things on it or in it.
 *
 * Note that Openable and Qcontainer work as expected with Desks.
 * E.g.,
 *
 *  mydesk: Openable, Qcontainer, Desk;
 *
 * Defines a desk that does not list its "in" contents in room
 *   descriptions, and which must be open for its "in" contents to be
 *   visible.
 */
class Desk: Surface, Container
  sdesc = "desk"
  ldesc = { self.contdesc(global.lastactor); }
  contdesc(actor) = {
    self.doLookX(actor, 'on', nil);
  }
  verDoOpen(actor) = {}
  doOpen(actor) = { "Open."; }
  verDoClose(actor) = {}
  doClose(actor) = { "Closed."; }
;

/*
 * A Shelf can have things on it.
 */
class Shelf: Surface
  sdesc = "shelf"
  ldesc = { self.contdesc(global.lastactor); }
  contdesc(actor) = {
    self.doLookX(actor, 'on', nil);
  } 
;

/*
 * Obstacle and Door are old stuff from adv.t, updated slightly
 * but largely unchanged.
 */
class Obstacle: Thing
  isobstacle = true
;

/*
 * A Door is an obstacle that impedes progress when it is closed. When the
 *   door is open (isopen is true), the user ends up in the room specified
 *   in the destination property upon going through the door. Since a Door
 *   is an obstacle, use the door object for a direction property of the
 *   room containing the door. (The travelto Actor method handles this.)
 *
 * If noautoopen is not set to true, the door will automatically be opened
 *   when the player tries to walk through the door, unless the door is
 *   locked (islocked = true). If the door is locked, it can be unlocked
 *   simply by typing "unlock door", unless the key property is set, in
 *   which case the object specified in key must be used to unlock the
 *   door. Note that the door can only be relocked by the player under the
 *   circumstances that allow unlocking, plus the property islockable must
 *   be set true. By default, the door is closed; set isopen to true if the
 *   door is to start out open (and be sure to open the other side as
 *   well).
 *
 * otherside specifies the corresponding Door object in the destination
 *   room (doordest), if any. If otherside is specified, its isopen and
 *   islocked properties will be kept in sync automatically.
 */
class Door: Obstacle
  isdoor = true
  ldesc = { self.opendesc; }
  isopen = nil
  islocked = nil
  opendesc = {
    "\^<<self.subjprodesc>> <<self.is>> ";
    if (self.isopen) "open";
    else { "closed"; if (self.islocked) " and locked"; }
    ".";
  }
  setIsopen(setting) = {
    self.isopen := setting;
    if (self.otherside and self.otherside.isopen <> setting) 
      self.otherside.setIsopen(setting);
  }
  setIslocked(setting) = {
    self.islocked := setting;
    if (self.otherside and self.otherside.islocked <> setting) 
      self.otherside.setIslocked(setting);
  }
  destination(actor) = {
    if (self.isopen)
      return self.doordest;
    else if (not self.islocked and not self.noautoopen) {
      local err;
      "(Opening <<self.objthedesc(nil)>>)\n";
      err := execCommand(actor, openVerb, self, nil, nil, EC_HIDE_SUCCESS);
      if (err = EC_SUCCESS)
        return self.doordest;
      else
        return nil;
    }
    else {
      "\^<<actor.youll>> have to open <<self.objthedesc(actor)>> first.";
      setit(self);
      return nil;
    }
  }
  verDoOpen(actor) = {
    if (self.isopen)
      "\^<<self.subjthedesc>> <<self.is>> already open.";
    else if (self.islocked)
      "\^<<actor.youll>> have to unlock <<self.objthedesc(actor)>> first.";
  }
  doOpen(actor) = {
    self.setIsopen(true);
    self.openmessage(actor);
  }
  verDoClose(actor) = {
    if (not self.isopen)
      "\^<<self.subjthedesc>> <<self.is>> already closed.";
  }
  doClose(actor) = {
    self.setIsopen(nil);
    self.closemessage(actor);
  }
  verDoLock(actor) = {
    if (self.islocked)
      "\^<<self.subjthedesc>> <<self.is>> already locked.";
    else if (not self.islockable)
      "\^<<self.subjthedesc>> can't be locked.";
    else if (self.isopen)
      "\^<<actor.youll>> have to close <<self.objprodesc(actor)>> first.";
  }
  doLock(actor) = {
    if (self.key = nil)
      self.doLockwith(actor, nil);
    else
      askio(withPrep);
  }
  verDoUnlock(actor) = {
    if (not self.islocked)
      "\^<<self.subjthedesc>> <<self.is>> not locked.";
  }
  doUnlock(actor) = {
    if (self.key = nil)
      self.doUnlockwith(actor, nil);
    else
      askio(withPrep);
  }
  verDoLockwith(actor, io) = {
    if (self.islocked)
      "\^<<self.subjthedesc>> <<self.is>> already locked.";
    else if (not self.islockable)
      "\^<<self.subjthedesc>> can't be locked.";
    else if (self.key = nil)
      "\^<<actor.subjthedesc>> <<actor.doesnt>> need anything to lock <<self.objprodesc(actor)>>.";
    else if (io <> self.key)
      "\^<<io.subjthedesc>> <<io.doesnt>> fit the lock.";
    else if (self.isopen)
      "\^<<actor.youll>> have to close <<self.objprodesc(actor)>> first.";
  }
  doLockwith(actor, io) = {
    self.setIslocked(true);
    self.lockmessage(actor);
  }
  verDoUnlockwith(actor, io) = {
    if (not self.islocked)
      "\^<<self.subjthedesc>> <<self.is>> not locked.";
    else if (self.key = nil)
      "\^<<actor.subjthedesc>> <<actor.doesnt>> need anything to unlock <<self.objthedesc(actor)>>.";
    else if (io <> self.key)
      "\^<<io.subjthedesc>> <<io.doesnt>> fit the lock.";
  }
  doUnlockwith(actor, io) = {
    self.setIslocked(nil);
    self.unlockmessage(actor);
  }
  verDoKnockon(actor) = {
    if (self.isopen) "%You% can't knock on <<self.objthedesc(actor)>> - it is open! ";
  }
  doKnockon(actor) = {
    "%You% <<knockonVerb.desc(actor)>> <<self.objthedesc(actor)>>. "; 
  }
  verDoEnter(actor) = { }
  doEnter(actor) = {
    actor.travelto(self.destination);
  }
  doSynonym('Enter') = 'Gothrough'
  openmessage(actor) = { "Open."; }
  closemessage(actor) = { "Closed."; }
  lockmessage(actor) = { "Locked."; }
  unlockmessage(actor) = { "Unlocked."; }
;

/*
 * autoLockDoor: A Door that automatically locks when you close it.
 */
class autoLockDoor: Door
  isopen = nil
  islocked = true
  doClose(actor) = {
    self.closemessage;
    self.isopen := nil;
    if (self.otherside) self.otherside.isopen := nil;
    "There is an ominous <click> as the catch on <<self.objthedesc(actor)>> drops, locking it. ";
    self.islocked := true;
    if (self.otherside) self.otherside.islocked := true;
  }
;

/*
 * Keyhole : a keyhole in a door.
 *
 * The keyhole is a derivative of Part, and should have a door as it's
 *   partof object parent, and specify a noun. Just as doors must be
 *   defined for two locations, this class requires that both doors have a
 *   keyhole connected to them. It defines the full set of lock/unlock
 *   verbs, so a key can be placed in the lock and then used to unlock the
 *   lock/door. The keyhole can have 'unlock' applied to it. This will
 *   affect the door.
 *
 * The player can look through the keyhole, and view the room on the other
 *   side, so long as no object is in the keyhole. The description of the
 *   room beyond should be held in the 'thrudesc' of the keyhole.
 *
 * The player can place any single small object inside the keyhole. For
 *   simplicity, we shall ignore the possibility that the shape of a key
 *   might not fit into the lock, although because it is a container it
 *   will have a maximum bulk, which will partially do the job. Should a
 *   key (or other object) already be in on the opposite side of the lock,
 *   the player can push it out. The object will fall onto the floor in the
 *   other room.
 */
class Keyhole: Part, Container
  sdesc = "keyhole"
  ldesc = {
    "\^<<self.subjthedesc>> is currently ";
    if (self.contents <> []) {
      local o := self.contents[1];
      "blocked by <<o.objthedesc(global.lastactor)>>, which <<o.is>> in it.";
    }
    else {
      if (self.partof.otherside.parts[1].contents <> [])
        "blocked by an object placed in the other side of the keyhole.";
      else
        "empty.";
    }
  }
  verDoLookthrough(actor) = { }
  doLookthrough(actor) = {
    if (self.contents <> [])
      "\^<<self.contents[1].objthedesc(actor)>> is blocking <<actor.possessivedesc>> view through the keyhole.";
    else if (self.partof.otherside.parts[1].contents <> [])
      "Something on the other side is blocking <<actor.possessivedesc>> view through the keyhole.";
    else {
      local dest := self.partof.doordest;
      local oldloc := actor.location;
      local oldloctype := actor.locationtype;
      self.thrudesc;
    }
  }

  /*
   * Restrict the keyhole to holding only one object.
   */
  verDoPutin(actor,io) = {
    if (self.contents <> [])
      "\^<<actor.subjthedesc>> can't, because there is already something in <<self.objthedesc(actor)>>.";
  }
  doPutin(actor,io) = {
    if (self.partof.doordest.parts[1].contents <> []) {
      local spdd := self.partof.doordest;
      "As <<actor.subjthedesc>> <<putVerb.desc(actor)>> <<io.subjthedesc>> into <<self.objthedesc(actor)>>, <<actor.subjprodesc>> ";
      if (actor.isplural) "hear"; else "hears";
      " a soft thunk on the other side.";
      spdd.parts[1].contents[1].movein(spdd);
      Outhide(true);
      inherited.doPutin(actor,io);
      Outhide(nil);
    }
    else
      inherited.doPutin(actor,io);
  }

  /*
   * Redirect lock/unlock to the parent door.
   */
  verDoLock(actor) = { self.partof.verDoLock(actor); }
  doLock(actor) = { self.partof.doLock(actor); }
  verDoUnlock(actor) = { self.partof.verDoUnlock(actor); }
  doUnlock(actor) = { self.partof.doUnlock(actor); }
  verDoLockwith(actor,io) = { self.partof.verDoLockwith(actor,io); }
  doLockwith(actor,io) = { self.partof.doLockwith(actor,io); }
  verDoUnlockwith(actor,io) = {
    self.partof.verDoUnlockwith(actor,io);
  }
  doUnlockwith(actor,io) = { self.partof.doUnlockwith(actor,io); }
;

/*
 * A Floor is the floor or ground in a location. It handles things like
 *   "put ___ on floor" nicely.
 *
 * The ground is a special case of Floor. You only need to use Floor if you
 *   want to do something unusual with the ground, like making it a
 *   Listabletouch, for example, for an area where the floor is trembling
 *   or vibrating.
 *
 * NOTE: If you make your own special ground in a room using Floor, don't
 *   forget to set the room's ground property to the special ground object.
 *   If you don't do this, some normal ground funtions (like "lie on
 *   ground") won't work for your customized ground.
 *
 * The ground is in every room that does not have ground = nil. Putting
 *   stuff on the ground is equivalent to dropping it in the room. Getting
 *   on the ground is equivalent to standing up in the room. (This only
 *   makes sense if the player is currently in a Nestedroom.)
 *
 * The ground does not allow actors to sit, lie, or stand on it. Instead,
 *   it leaves the actor where he is and sets his position to 'sitting',
 *   'lying', or 'standing'.
 *
 * You can customize the ldesc by defining grounddesc in the Room. If this
 *   method exists, Ground will print it instead of the standard ldesc.
 *   (Everywhere provides this functionality generally.)
 */
class Floor: Everywhere, Ledge
  verDoLookon(actor) = {}
  doLookon(actor) = { actor.location.lookaround(actor, true); }
  verIoPuton(actor) = {}
  ioPuton(actor, dobj) = { dobj.doDrop(actor); }
  verIoTakeoff(actor) = { actor.location.verIoTakefrom(actor); }
  ioTakeoff(actor, dobj) = { actor.location.ioTakefrom(actor, dobj); }
  roomdrop(actor, obj) = { self.location.roomdrop(actor, obj); }
  verDoSiton(actor) = {}
  verDoLieon(actor) = {}
  doSiton(actor) = {
    actor.position := 'sitting';
    "\^<<actor.subjthedesc>> <<actor.is>> now sitting down.";
  }
  doLieon(actor) = {
    actor.position := 'lying';
    "\^<<actor.subjthedesc>> <<actor.is>> now lying down.";
  }

  /*
   * ...Geton will be called when the player says "get on ground" or
   *   "stand on ground". This usually means he wants to just be
   *   standing in the room - i.e., that he wants to get out of
   *   whatever nested room he's in.
   */
  verDoGeton(actor) = {
    if (actor.location = self.location)
      "\^<<actor.youre>> already on the ground. ";
    else actor.location.verDoGetoff(actor);
  }
  doGeton(actor) = { actor.location.doGetoff(actor); }
;

Ground: Floor
  reachsurroundings = true
  sdesc = "ground"
  noun = 'ground' 'floor'
  roomprop = &ground
  roomdescprop = &grounddesc
  roomlistenprop = &groundlisten
  roomsmellprop = &groundsmell
  roomtasteprop = &groundtaste
  roomtouchprop = &groundtouch
;

/*
 * The walls are purely decorative.
 * They're in every room which doesn't have walls = nil.
 */
Walls: Everywhere
  sdesc = "walls"
  noun = 'walls' 'wall'
  roomprop = &walls
  roomdescprop = &wallsdesc
  roomlistenprop = &wallslisten
  roomsmellprop = &wallssmell
  roomtasteprop = &wallstaste
  roomtouchprop = &wallstouch
;

/*
 * The ceiling is purely decorative.
 * It's in every room which doesn't have ceiling = nil.
 */
Ceiling: Everywhere
  sdesc = "ceiling"
  noun = 'ceiling'
  roomprop = &ceiling
  roomdescprop = &ceilingdesc
  roomlistenprop = &ceilinglisten
  roomsmellprop = &ceilingsmell
  roomtasteprop = &ceilingtaste
  roomtouchprop = &ceilingtouch
;

/*
 * A Sensor is a Thing that can sense other objects.
 * Actors are the most common types of Sensors.
 */
class Sensor: Thing 

  /* Can this thing see the object? */
  cansee(obj, loctype, silent) = {
    local i;
    if (not self.seepath(obj, loctype, silent))
      return nil;
    Outhide(true);
    i := obj.isvisible(self);
    global.canoutput := Outhide(nil);
    if (not i) {
      if (not silent) {
        if (global.canoutput)
          obj.isvisible(self);
        else
          "<<self.subjthedesc>> can't see <<obj.objthedesc(self)>>.";
      }
      return nil;
    }
    return true;      
  }

  /* Can this thing touch the object? */
  cantouch(obj, loctype, silent) = {
    local i;
    if (not self.touchpath(obj, loctype, silent))
      return nil;
    Outhide(true);
    i := obj.istouchable(self);
    global.canoutput := Outhide(nil);
    if (not i) {
      if (not silent) {
        if (global.canoutput)
          obj.istouchable(self);
        else
          "<<self.subjthedesc>> can't touch <<obj.objthedesc(self)>>.";
      }
      return nil;
    }
    return true;
  }
  
  /* Can this thing take the object? */
  cantake(obj, loctype, silent) = {
    local i;
    if (not self.takepath(obj, loctype, silent))
      return nil;
    Outhide(true);
    i := obj.istakeable(self, []);
    global.canoutput := Outhide(nil);
    if (not i) {
      if (not silent) {
        if (global.canoutput)
          obj.istakeable(self, []);
        else
          "<<self.subjthedesc>> can't take <<obj.objthedesc(self)>>.";
      }
      return nil;
    }
    return true;
  }

  /* Can this thing smell the object? */
  cansmell(obj, loctype, silent) = {
    local i;
    if (not self.smellpath(obj, loctype, silent))
      return nil;
    Outhide(true);
    i := obj.issmellable(self);
    global.canoutput := Outhide(nil);
    if (not i) {
      if (not silent) {
        if (global.canoutput)
          obj.issmellable(self);
        else
          "<<self.subjthedesc>> can't smell <<obj.objthedesc(self)>>.";
      }
      return nil;
    }
    return true;
  }
  
  /* Can this thing hear the object? */
  canhear(obj, loctype, silent) = {
    local i;
    if (not self.hearpath(obj, loctype, silent))
      return nil;
    Outhide(true);
    i := obj.isaudible(self);
    global.canoutput := Outhide(nil);
    if (not i) {
      if (not silent) {
        if (global.canoutput)
          obj.isaudible(self);
        else
          "<<self.subjthedesc>> can't hear <<obj.objthedesc(self)>>.";
      }
      return nil;
    }
    return true;
  }

  /* Can this thing speak to the object? */
  canspeakto(obj, loctype, silent) = {
    local i;
    if (not self.speaktopath(obj, loctype, silent))
      return nil;
    Outhide(true);
    i := obj.islistener(self);
    global.canoutput := Outhide(nil);
    if (not i) {
      if (not silent) {
        if (global.canoutput)
          obj.islistener(self);
        else
          "<<obj.subjthedesc>> can't hear <<self.objthedesc(obj)>>.";
      }
      return nil;
    }
    return true;
  }

  /* Can this thing put something into the object? */
  canputinto(obj, loctype, silent) = {
    local i;
    if (not self.putintopath(obj, loctype, silent))
      return nil;
    Outhide(true);
    i := obj.acceptsput(self, loctype);
    global.canoutput := Outhide(nil);
    if (not i) {
      if (not silent) {
        if (global.canoutput)
          obj.acceptsput(self, loctype);
        else
          "<<self.subjthedesc>> can't put anything <<loctype>> <<obj.objthedesc(self)>>.";
      }
      return nil;
    }
    return true;
  }

  /*
   * Pseudo-senses for use of strings and numbers. Things can only use
   *   an object as a string or number if that thing is a string or
   *   number.
   *
   * Note that these methods should not include initial caps or final
   *   punctuation.
   */
  canusealphanum(obj, loctype, silent) = {
    if (not isclass(obj, String) and not isclass(obj, Number)) {
      if (not silent)
        "that verb requires a double-quoted string or a number.";
      return nil;
    }
    else
      return true;
  }
  canusealpha(obj, loctype, silent) = {
    if (not isclass(obj, String)) {
      if (not silent)
        "that verb requires a double-quoted string.";
      return nil;
    }
    else
      return true;
  }
  canusenumber(obj, loctype, silent) = {
    if (not isclass(obj, Number)) {
      if (not silent)
        "that verb requires a number.";
      return nil;
    }
    else
      return true;
  }
  canusenumberorlist(obj, loctype, silent) = {
    if (not isclass(obj, Number) and not isclass(obj, Listobj)) {
      if (not silent)
        "that verb requires a number or a parenthesized list of numbers.";
      return nil;
    }
    else
      return true;
  }
;

/*
 * Breakable: A modifier that allows an object to be broken. The flag
 *   isbroken is used to determine the object's state.
 */
class Breakable: Thing
  ldesc = {
    if (self.isbroken) "\^<<self.subjthedesc>> <<self.isplural ? "are" : "is">> broken! ";
  }
  isbroken = nil
  verDoBreak(actor) = {
    if (self.isbroken) "\^<<self.subjthedesc>> <<self.isplural ? "are" : "is">> already broken! ";
  }
  doBreak(actor) = {
    "\^<<actor.subjthedesc>> <<breakVerb.desc(actor)>> <<self.objthedesc(actor)>>. ";
    self.isbroken := true;
  }
;
