/* input.c -- Read and tokenize the input file
   Copyright (c) 1993-1995 Eberhard Mattes

This file is part of emxdoc.

emxdoc is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

emxdoc is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emxdoc; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "emxdoc.h"
#include "cond.h"

#define COND_STACK_SIZE         8

struct cond
{
  int start_line;
  int true;
  int else_seen;
};

static struct cond cond_stack[COND_STACK_SIZE];
static int cond_sp;


#define ISARG(C) ((C) == '{' || (C) == '[')
#define ISARGW(C) (isspace (C) || ISARG (C))
#define ISENDW(C) (isspace (C) || (C) == 0)

#define SKIP1W(P,AT) do { (P) += (AT); if (isspace (*(P))) ++(P); } while (0)
#define SKIPW(P,AT) do { (P) += (AT); while (isspace (*(P))) ++(P); } while (0)

void read_line (void)
{
  uchar *p;
  int c1;

redo:
  ++line_no;
  if (fgets (input, sizeof (input), input_file) == NULL)
    {
      if (ferror (input_file))
        {
          perror (input_fname);
          exit (1);
        }
      end_of_file = TRUE;
      if (cond_sp >= 0)
        fatal ("%s:%d: Unterminated %cif",
               input_fname, cond_stack[cond_sp].start_line, escape);
      return;
    }
  p = strchr (input, '\n');
  if (p == NULL)
    fatal ("%s:%d: Line too long", input_fname, line_no);
  *p = 0;
  if (input[0] == escape)
    {
      p = input + 1;
      if (strncmp (p, "c", 1) == 0 && ISENDW (p[1]))
        goto redo;
      if (strncmp (p, "if", 2) == 0 && isspace (p[2]))
        {
          SKIPW (p, 2);
          c1 = condition (p);
          if (cond_sp + 1 >= COND_STACK_SIZE)
            fatal ("%s:%d: Conditional stack overflow", input_fname, line_no);
          ++cond_sp;
          cond_stack[cond_sp].true = c1;
          cond_stack[cond_sp].start_line = line_no;
          cond_stack[cond_sp].else_seen = FALSE;
          goto redo;
        }
      else if (strcmp (p, "else") == 0)
        {
          if (cond_sp < 0)
            fatal ("%s:%d: %celse without %cif",
                   input_fname, line_no, escape, escape);
          if (cond_stack[cond_sp].else_seen)
            fatal ("%s:%d: Multiple %celse for %cif in line %d",
                   input_fname, line_no, escape, escape,
                   cond_stack[cond_sp].start_line);
          cond_stack[cond_sp].else_seen = TRUE;
          cond_stack[cond_sp].true = !cond_stack[cond_sp].true;
          goto redo;
        }
      else if (strcmp (p, "endif") == 0)
        {
          if (cond_sp < 0)
            fatal ("%s:%d: %cendif without %cif",
                   input_fname, line_no, escape, escape);
          --cond_sp;
          goto redo;
        }
    }

  if (cond_sp >= 0 && !cond_stack[cond_sp].true)
    goto redo;

  p = input;
  while (isspace (*p))
    ++p;
  if (*p == 0)
    input[0] = 0;
}


void open_input (const char *name)
{

  line_no = 0; end_of_file = FALSE; cond_sp = -1;
  input_fname = name;
  input_file = fopen (input_fname, "rt");
  if (input_file == NULL)
    {
      perror (input_fname);
      exit (1);
    }
}


static void invalid_tag (void)
{
  fatal ("%s:%d: Invalid tag", input_fname, line_no);
}


int parse_tag (const uchar **ptr)
{
  const uchar *p;

  p = *ptr;
  if (*p != escape)
    return FALSE;
  ++p;                          /* Skip escape character */
  tg_flags = 0;
  switch (*p)
    {
    case '.':
      tg_tag = TAG_FULLSTOP;
      ++p;
      break;

    case 'b':
      if (strncmp (p, "bf", 2) == 0 && ISARGW (p[2]))
        {
          tg_tag = TAG_STYLE;
          tg_style = STYLE_BOLD;
          p += 2;
        }
      else if (strncmp (p, "break", 5) == 0 && ISENDW (p[5]))
        {
          tg_tag = TAG_BREAK;
          SKIP1W (p, 5);        /* Skip one space */
        }
      else if (strcmp (p, "bugs") == 0)
        tg_tag = TAG_BUGS;
      else
        invalid_tag ();
      break;

    case 'c':
      if (strncmp (p, "compat", 6) == 0 && isspace (p[6]))
        {
          tg_tag = TAG_COMPAT;
          p += 6;
        }
      else
        invalid_tag ();
      break;

    case 'd':
      if (strcmp (p, "description") == 0)
        tg_tag = TAG_DESCRIPTION;
      else
        invalid_tag ();
      break;

    case 'e':
      if (strncmp (p, "em", 2) == 0 && ISARGW (p[2]))
        {
          tg_tag = TAG_STYLE;
          tg_style = STYLE_EMPHASIZE;
          p += 2;
        }
      else if (strcmp (p, "enddescription") == 0)
        tg_tag = TAG_ENDDESCRIPTION;
      else if (strcmp (p, "endenumerate") == 0)
        tg_tag = TAG_ENDENUMERATE;
      else if (strcmp (p, "endexample") == 0)
        tg_tag = TAG_ENDEXAMPLE;
      else if (strcmp (p, "endheaders") == 0)
        tg_tag = TAG_ENDHEADERS;
      else if (strcmp (p, "endindent") == 0)
        tg_tag = TAG_ENDINDENT;
      else if (strcmp (p, "enditemize") == 0)
        tg_tag = TAG_ENDITEMIZE;
      else if (strcmp (p, "endlist") == 0)
        tg_tag = TAG_ENDLIST;
      else if (strcmp (p, "endprototype") == 0)
        tg_tag = TAG_ENDPROTOTYPE;
      else if (strcmp (p, "endsamplecode") == 0)
        tg_tag = TAG_ENDSAMPLECODE;
      else if (strcmp (p, "endverbatim") == 0)
        tg_tag = TAG_ENDVERBATIM;
      else if (strcmp (p, "endtypewriter") == 0)
        tg_tag = TAG_ENDTYPEWRITER;
      else if (strcmp (p, "endipf") == 0)
        tg_tag = TAG_ENDIPF;
      else if (strcmp (p, "endlatex") == 0)
        tg_tag = TAG_ENDLATEX;
      else if (strcmp (p, "endtext") == 0)
        tg_tag = TAG_ENDTEXT;
      else if (strcmp (p, "endtable") == 0)
        tg_tag = TAG_ENDTABLE;
      else if (strcmp (p, "enumerate") == 0)
        tg_tag = TAG_ENUMERATE;
      else if (strcmp (p, "errors") == 0)
        tg_tag = TAG_ERRORS;
      else if (strcmp (p, "example") == 0)
        tg_tag = TAG_EXAMPLE;
      else
        invalid_tag ();
      break;

    case 'f':
      if (strncmp (p, "format", 6) == 0 && isspace (p[6]))
        {
          tg_tag = TAG_FORMAT;
          SKIPW (p, 6);
        }
      else if (strncmp (p, "function", 8) == 0 && isspace (p[8]))
        {
          tg_tag = TAG_FUNCTION;
          SKIPW (p, 8);
        }
      else
        invalid_tag ();
      break;

    case 'h':
      if (p[1] >= '1' && p[1] <= '0' + SECTION_LEVELS)
        {
          tg_tag = TAG_HEADING;
          tg_level = p[1] - '0';
          tg_underline = (tg_level == 1 ? '=' : '-');
          p += 2;
          while (*p != 0 && !isspace (*p))
            switch (*p++)
              {
              case 'u':
                tg_flags |= HF_UNNUMBERED;
                break;
              default:
                invalid_tag ();
              }
          SKIPW (p, 0);
        }
      else if (p[1] == '-' || p[1] == '=')
        {
          tg_tag = TAG_HEADING;
          tg_level = 0;
          tg_underline = p[1];
          SKIPW (p, 2);
        }
      else if (strncmp (p, "hpt", 3) == 0 && ISARGW (p[3]))
        {
          tg_tag = TAG_HPT;
          p += 3;
        }
      else if (strcmp (p, "headers") == 0)
        tg_tag = TAG_HEADERS;
      else if (strcmp (p, "hints") == 0)
        tg_tag = TAG_HINTS;
      else
        invalid_tag ();
      break;

    case 'i':
      if (p[1] >= '1' && p[1] <= '2')
        {
          tg_tag = TAG_INDEX;
          tg_level = p[1] - '0';
          SKIPW (p, 2);
        }
      else if (strcmp (p, "implementation") == 0)
        tg_tag = TAG_IMPLEMENTATION;
      else if (strcmp (p, "ipf") == 0)
        tg_tag = TAG_IPF;
      else if (strcmp (p, "ipfminitoc") == 0)
        tg_tag = TAG_IPFMINITOC;
      else if (strncmp (p, "item", 4) == 0 && ISENDW (p[4]))
        {
          tg_tag = TAG_ITEM;
          SKIPW (p, 4);
        }
      else if (strncmp (p, "index", 5) == 0 && isspace (p[5]))
        {
          tg_tag = TAG_INDEX;
          tg_level = 0;
          SKIPW (p, 5);
        }
      else if (strcmp (p, "indent") == 0)
        tg_tag = TAG_INDENT;
      else if (strcmp (p, "itemize") == 0)
        tg_tag = TAG_ITEMIZE;
      else
        invalid_tag ();
      break;

    case 'k':
      if (strncmp (p, "keyword ", 8) == 0)
        {
          tg_tag = TAG_KEYWORD;
          SKIPW (p, 8);
        }
      else
        invalid_tag ();
      break;

    case 'l':
      if (strncmp (p, "label ", 6) == 0)
        {
          tg_tag = TAG_LABEL;
          SKIPW (p, 6);
        }
      else if (strncmp (p, "language", 8) == 0 && isspace (p[8]))
        {
          tg_tag = TAG_LANGUAGE;
          SKIPW (p, 8);
        }
      else if (strcmp (p, "latex") == 0)
        tg_tag = TAG_LATEX;
      else if (strcmp (p, "list") == 0)
        tg_tag = TAG_LIST;
      else
        invalid_tag ();
      break;

    case 'p':
      if (strcmp (p, "prototype") == 0)
        tg_tag = TAG_PROTOTYPE;
      else if (strncmp (p, "param", 5) == 0 && isspace (p[5]))
        {
          tg_tag = TAG_PARAM;
          SKIPW (p, 5);
        }
      else if (strncmp (p, "pa", 2) == 0 && ISARGW (p[2]))
        {
          tg_tag = TAG_STYLE;
          tg_style = STYLE_PARAM;
          p += 2;
        }
      else
        invalid_tag ();
      break;

    case 'r':
      if (strncmp (p, "ref", 3) == 0 && ISARGW (p[3]))
        {
          tg_tag = TAG_REF;
          p += 3;
        }
      else if (strncmp (p, "replace", 7) == 0 && isspace (p[7]))
        {
          tg_tag = TAG_REPLACE;
          SKIPW (p, 7);
        }
      else if (strcmp (p, "restrictions") == 0)
        tg_tag = TAG_RESTRICTIONS;
      else if (strcmp (p, "returnvalue") == 0)
        tg_tag = TAG_RETURNVALUE;
      else
        invalid_tag ();
      break;

    case 's':
      if (strncmp (p, "seealso", 7) == 0 && isspace (p[7]))
        {
          tg_tag = TAG_SEEALSO;
          p += 7;
        }
      else if (strcmp (p, "samplecode") == 0)
        tg_tag = TAG_SAMPLECODE;
      else if (strncmp (p, "samplefile", 10) == 0 && isspace (p[10]))
        {
          tg_tag = TAG_SAMPLEFILE;
          SKIPW (p, 10);
        }
      else if (strncmp (p, "set", 3) == 0 && isspace (p[3]))
        {
          tg_tag = TAG_SET;
          SKIPW (p, 3);
        }
      else if (strncmp (p, "sl", 2) == 0 && ISARGW (p[2]))
        {
          tg_tag = TAG_STYLE;
          tg_style = STYLE_SLANTED;
          p += 2;
        }
      else if (strncmp (p, "special", 7) == 0 && isspace (p[7]))
        {
          tg_tag = TAG_SPECIAL;
          SKIPW (p, 7);
        }
      else if (strncmp (p, "sy", 2) == 0 && ISARGW (p[2]))
        {
          tg_tag = TAG_STYLE;
          tg_style = STYLE_SYNTAX;
          p += 2;
        }
      else if (strncmp (p, "syntax", 6) == 0 && isspace (p[6]))
        {
          tg_tag = TAG_SYNTAX;
          SKIPW (p, 6);
        }
      else
        invalid_tag ();
      break;

    case 't':
      if (strncmp (p, "tt", 2) == 0 && ISARGW (p[2]))
        {
          tg_tag = TAG_STYLE;
          tg_style = STYLE_TTY;
          p += 2;
        }
      else if (strcmp (p, "toc") == 0)
        tg_tag = TAG_TOC;
      else if (strcmp (p, "text") == 0)
        tg_tag = TAG_TEXT;
      else if (strncmp (p, "table ", 6) == 0)
        {
          tg_tag = TAG_TABLE;
          SKIPW (p, 6);
        }
      else if (strncmp (p, "title ", 6) == 0)
        {
          tg_tag = TAG_TITLE;
          SKIPW (p, 6);
        }
      else if (strcmp (p, "typewriter") == 0)
        tg_tag = TAG_TYPEWRITER;
      else
        invalid_tag ();
      break;

    case 'u':
      if (strncmp (p, "ul", 2) == 0 && ISARGW (p[2]))
        {
          tg_tag = TAG_STYLE;
          tg_style = STYLE_UNDERLINE;
          p += 2;
        }
      else
        invalid_tag ();
      break;

    case 'v':
      if (strcmp (p, "verbatim") == 0)
        tg_tag = TAG_VERBATIM;
      else
        invalid_tag ();
      break;

    default:
      invalid_tag ();
    }
  *ptr = p;
  return TRUE;
}
