/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/print.c
 * Summary:
 *	Simple printf/vprintf implementations that are guaranteed not to
 *	use malloc. (Calling malloc while printing an error message might
 *	cause recursion).
 * Other notes:
 *	The following specifiers are implemented:
 *		%p	pointer, prints NULL, ILLEGAL or value (as hex)
 *		%d	signed integer
 *		%u	unsigned integer
 *		%s	string, prints "(null)" if NULL pointer passed
 *		%x	hexadecimal integer
 * Author      	Date		Notes
 * RWMJ		5/12/94		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include "bounds-lib.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

/* Note: 5/10/95. OSF/1 3.0 appears to allow reentrant calls to printf and
 * friends. So this isn't needed ...
 */
#if !defined(__osf__)

#if defined(printf)
#undef printf
#endif
#if defined(vprintf)
#undef vprintf
#endif

/*----------------------------------------------------------------------
 *	low level functions to manage a simple output buffer
 *----------------------------------------------------------------------*/

static char buffer [256], *next = buffer;

static inline void
printchar (char c)
{
  *next++ = c;
  if (c == '\n' || next - buffer == 256) {
    /* Flush output buffer. */
    write (2, buffer, next - buffer);
    next = buffer;
  }
}

/*----------------------------------------------------------------------
 *	low level functions to convert numbers to strings
 *----------------------------------------------------------------------*/

static inline char *
_ul_to_a (unsigned long u, char *buffer)
{
  if (u != 0) {
    unsigned long n = u % 10;
    buffer = _ul_to_a (u / 10, buffer);
    *buffer++ = n + '0';
  }
  return buffer;
}

static inline void
l_to_a (long i, char *buffer)
{
  if (i == 0) {
    *buffer++ = '0';
  } else {
    if (i < 0) {
    *buffer++ = '-';
    i = -i;
    }
    buffer = _ul_to_a ((unsigned long) i, buffer);
  }
  *buffer = 0;
}

static inline void
ul_to_a (unsigned long u, char *buffer)
{
  if (u == 0) {
    *buffer++ = '0';
  } else {
    buffer = _ul_to_a (u, buffer);
  }
  *buffer = 0;
}

/* Print a hex number. `width' is the number of digits (ie. number of
 * hex nybbles) to print.
 */
static inline void
xtoa (unsigned long u, char *buffer, int width)
{
  static char hexdigit[16] = {'0', '1', '2', '3', '4', '5',
			      '6', '7', '8', '9', 'a', 'b',
			      'c', 'd', 'e', 'f'};
  int i;
  unsigned d;
  unsigned long top_nybble_mask = 0xF << ((width - 1) * 4);

  for (i = width-1; i >= 1; --i) {
    if (u & top_nybble_mask)
      break;
    u <<= 4;
  }
  for (; i >= 0; --i) {
    d = u >> ((width - 1) * 4);
    u <<= 4;
    *buffer++ = hexdigit[d];
  }
  *buffer = 0;
  return;
}

/*----------------------------------------------------------------------
 *	like vprintf
 *----------------------------------------------------------------------*/

void
__bounds_vprintf (char *format, va_list args)
{
  char *p, *s;
  int i;
  int prec;
  char pchar;
  unsigned u;
  void *vp;
  char buffer [64];

  p = format;
  while (*p) {
    if (*p != '%') {
      printchar (*p);
      p++;
    } else {
      p++;
      prec = 0;
      pchar = ' ';
      if (*p == '0') pchar = *p++;	/* check for '%02d'    */
      while (*p >= '0' && *p <= '9')	/* calculate precision */
        prec = prec * 10 + *p++ - '0';
      switch (*p) {
      case 's':
	/* String. Print it, or print "(null)" if the pointer passed is
	 * NULL.
	 */
	s = va_arg (args, char *);
	if (s == NULL) {
          prec -= strlen("(null)");
          while (prec-- > 0) printchar(pchar);
	  __bounds_printf ("(null)");
	} else {
          prec -= strlen(s);
          while (prec-- > 0) printchar(pchar);
	  while (*s) {
	    printchar (*s);
	    ++s;
	  }
	}
	break;
      case 'd':
	/* Signed integer.
	 */
	i = va_arg (args, int);
	l_to_a (i, buffer);
        prec -= strlen(buffer);
        while (prec-- > 0) printchar(pchar);
	__bounds_printf ("%s", buffer);
	break;
      case 'u':
	u = va_arg (args, unsigned);
	ul_to_a (u, buffer);
        prec -= strlen(buffer);
        while (prec-- > 0) printchar(pchar);
	__bounds_printf ("%s", buffer);
	break;
      case 'p':
	/* Pointer. Print NULL, ILLEGAL or a representation of the pointer.
	 */
	vp = va_arg (args, void *);
	if (vp == NULL) {
          prec -= strlen("NULL");
          while (prec-- > 0) printchar(pchar);
	  __bounds_printf ("NULL");
	} else if (vp == ILLEGAL) {
          prec -= strlen("ILLEGAL");
          while (prec-- > 0) printchar(pchar);
	  __bounds_printf ("ILLEGAL");
	} else {
	  xtoa (PTR_TO_UNSIGNED (vp), buffer, sizeof (ptr_as_unsigned_t) * 2);
          prec -= strlen("0x") + strlen(buffer);
          while (prec-- > 0) printchar(pchar);
	  __bounds_printf ("0x%s", buffer);
	}
	break;
      case 'x':
	/* Hexadecimal integer.
	 */
	u = va_arg (args, unsigned);
	xtoa (u, buffer, sizeof (unsigned) * 2);
        prec -= strlen(buffer);
        while (prec-- > 0) printchar(pchar);
	__bounds_printf ("%s", buffer);
	break;
      case '%':
	printchar ('%'); break;
      case 0:
	printchar ('\\'); printchar ('0'); break;
      default:
	/* Unrecognized format specifier. Print character and advance va
	 * pointer along one to ignore that argument.
	 */
	printchar (*p);
	(void) (va_arg (args, int));
      }
      if (*p) p++;
    }
  }
}

/*----------------------------------------------------------------------
 *	like printf
 *----------------------------------------------------------------------*/

void
__bounds_printf (char *format, ...)
{
  va_list args;

  va_start (args, format);
  __bounds_vprintf (format, args);
  va_end (args);
}

#endif /* !defined (__osf__) */
