(*  :Title:	Discrete Fourier Transforms  *)

(*  :Authors:	Wally McClure, Brian Evans, James McClellan  *)

(*
    :Summary:	This file allows the computation of the forward and
		inverse discrete Fourier transform (DFT).
 *)

(*  :Context:	SignalProcessing`Digital`DFT`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*
    :History:   start			July 30, 1989
		redirection		October 2, 1989
		made into package	April 18, 1990
		made into release	May 25th, 1990
 *)

(*  :Keywords:	discrete Fourier transform  *)

(*
    :Source:	'Discrete-Time Signal Processing', Alan V. Oppenheim
		 	and Ronald W. Schafer
 *)

(*  :Warning:	*)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*  
    :Limitation: The DFT is calculated by first attempting to calculate the 
    		 DTFT of a function using the MyDTFT rule base.  At the end 
		 of the MyDTFT rule base, a call to the z-transform is made.  
		 If  the  function  does  not have a transform specified in 
		 either domain, then a nasty expression is returned.    
 *)

(*
    :Discussion:  The DFT is based on the DTFT.  A signal is made finite
		  in extent by multiplying it by ( Step[n] - Step[n - N] ),
		  where N is the number of points at which to evaluate the
		  DFT.
 *)

(*
    :Functions:	DFTransform
		InvDFTransform
		MultiDDFT
		MultiIDDFT
 *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ];


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Digital`DFT`",
	      "SignalProcessing`Digital`DTFT`",
	      "SignalProcessing`Digital`DSupport`",
	      "SignalProcessing`Digital`InvZTransform`",
	      "SignalProcessing`Digital`ZTransform`",
	      "SignalProcessing`Digital`ZSupport`",
	      "SignalProcessing`Support`TransSupport`",
	      "SignalProcessing`Support`ROC`",
	      "SignalProcessing`Support`SigProc`",
	      "SignalProcessing`Support`SupCode`" ]


(*  U S A G E     I N F O R M A T I O N  *)

DFTData::usage =
	"Data-tag for the symbolic DFT."

DFTransform::usage =
	"DFTransform[function, NumberOfPoints, TimeVariables, \
	FourierVariables, options] returns the DFT of function, \
	where function is a function of TimeVariables defined from \
	0 to NumberOfPoints - 1.  Note that only the first two arguments \
	are required and that DFTransform calls DTFTransform."

Finish::usage =
	"Finish is a data-tag for the DFT."

InvDFTransform::usage =
	"InvDFTransform[function, NumberOfPoints, FourierVariables, \
	TimeVariables, options] takes the inverse DFT of function, \
	where function is a function of FourierVariables defined from \
	0 to NumberOfPoints - 1.  Note that only the first two arguments \
	are required and that InvDFTransform calls InvDTFTransform."

KVariables::usage =
	"KVariables is a data-tag for the DFT."

Start::usage =
	"Start is a data-tag for the DFT."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin[ "`Private`" ]


(*  B E G I N     D F T  *)

(*  Evaluation of DFT operator  *)
Unprotect[DFT]
DFT/: TheFunction[ DFT[N_, n_, k_] [f_] ] := DFTransform[f, N, n, k]
Protect[DFT]

(*  Options for DFTransform  *)
DFTransform/: Options[ DFTransform ] := Options[ DTFTransform ]

(*  Extension of TheFunction to return the transform function.  *)
DFTData/: TheFunction[ DFTData[ trans_, var_ ] ] := trans

(*  Magnitude/Phase plot of a DFT object *)
mppoptions = {	Domain -> Discrete, DomainScale -> Linear,
		MagRangeScale -> Linear, PhaseRangeScale -> Degree,
		PlotRange -> All }

DFTData/: MagPhasePlot[ DFTData[trans_, KVariables[k_Symbol],
				Start[st_], Finish[end_] ] ] :=
	MagPhasePlot[ trans, {k, 0, end - st}, mppoptions ]

DFTData/: MagPhasePlot[ DFTData[trans_, KVariables[{k1_Symbol, k2_Symbol}],
				Start[{s1_, s2_}], Finish[{e1_, e2_}] ] ] :=
	MagPhasePlot[ trans, {k1, 0, e1 - s1}, {k2, 0, e2 - s2}, mppoptions ]

(*  This Code decides between a one-dimensional or multidimensional DFT. *)
DFTransform[ f_ ] :=
	Message[ Transform::novariables, "N (length)", GetVariables[f] ]

DFTransform[ f1_, N_ ] :=
	DFTransform[ f1, N, DummyVariables[Length[N], Global`n] ]

DFTransform[ f1_, N_, n_ ] :=
	DFTransform[ f1, N, n, DummyVariables[Length[n], Global`k] ]

DFTransform[ f1_, N_, n_, k_, options___ ] :=
	Block [	{begin, end, newfun, notenough, op, trans, vars, w, wvars},
		notenough = ( Length[k] < Length[n] );
		If [ notenough,
		     Message[Transform::notenough, "k (DFT index)"] ];
		vars = If [ notenough, DummyVariables[Length[n], Global`k], k ];
		wvars = DummyVariables[Length[n], w];

		op = ToList[options] ~Join~ Options[DFTransform];
		trans = If [ Length[n] > 1,  
			     MultiDDFT[f1, N, n, vars, wvars, op],
			     MyDFT[f1, N, n, vars, wvars, op] ];

		begin = Start[ If [AtomQ[n], 0, Table[0, {Length[n]}]] ];
		end = Finish[ N - 1 ]; 

		DFTData[FourierSimplify[trans], begin, end, KVariables[vars]] ]

(*  MultiDDFT --  multidimensional DFT  *)
MultiDDFT[ fun_, Num_, varin_, varout_, wvars_, op_ ] :=
	Block [	{F2 = fun, jj, length, num, var, w},
		length = Length[varin];
		For [ jj = 1, jj <= length, jj++,
		      var = varin[[jj]];
		      num = Num[[jj]];
		      w = wvars[[jj]];
		      F2 = MyDFT[F2, num, var, varout[[j]], w, op] ];
		F2 ]

(*  MyDFT  *)
MyDFT[ x_, N_, n_, k_, w_, op_ ] :=
	Block [	{newexpr = myDFT[x, n, k, w, L, True, FixUp[op]],
		 oldexpr = Null, ret, trace},

		trace = SameQ[ Replace[Dialogue, op], All ];
		While [ ! SameQ[newexpr, oldexpr],
			If [ trace, Print[newexpr]; Print[ "which becomes" ] ];
			oldexpr = newexpr;
			newexpr = MapAll[transform, oldexpr] ];

		oldexpr = TransformFixUp[ oldexpr, n, k, op, myDFT, False,
					  DFTransform, Null, Null ];

		newexpr = posttransform[oldexpr];
		While [ ! SameQ[newexpr, oldexpr],
			If [ trace, Print[newexpr]; Print[ "which becomes" ] ];
			oldexpr = newexpr;
			newexpr = posttransform[ oldexpr ] ];

		newexpr ]

(*  FixUp  *)
FixUp[op_] := { Dialogue -> Replace[Dialogue, op] }

(*  transform  *)
(* transform[myDFT[x_, rest__]] := Replace[myDFT[x, rest], DFTRules] *)
transform[x_] := x		(* no transform pairs, etc., encoded yet *)

(*  posttransform  *)
posttransform[x_] := x /. postrules

postrules = {

	myDFT[ x_, n_, k_, w_, L_, s_, op_ ] :>
		( TheFunction[DTFTransform[x Pulse[L, n], n, w, op]] /.
			w -> (2 Pi k / L) )

}

Format[ myDFT[ x_, n_, k_, w_, L_, s_, op_ ] ] :=
	SequenceForm[ ColumnForm[ { "DFT",
		         StringJoin["   ", ToString[L], ", ", ToString[n]] } ],
		      { x } ]


(*  E N D     D F T  *)


(*  B E G I N     I N V E R S E     D F T  *)

(*  Messages  *)
InvDFTransform::badlength = "Conflicting lengths in inverse DFT: `` != ``."

(*  Evaluation of inverse DFT operator  *)
Unprotect[InvDFT]
InvDFT/: TheFunction[ InvDFT[N_, k_, n_] [f_] ] := InvDFTransform[f, N, k, n]
Protect[InvDFT]

(*  Options for InvDFTransform  *)
InvDFTransform/: Options[ InvDFTransform ] :=
	{ Definition -> False } ~Join~ Options[ InvDTFTransform ]

(*  Choose whether to do the one or multi dimensional inverse DFT`s.  *)
InvDFTransform[ DFTData[x_, Start[st_], Finish[end_], KVariables[k_]] ] :=
	InvDFTransform[ x, end - st + 1, k ]

InvDFTransform[ f_ ] :=
	Message[ Transform::novariables, "N (length)", GetVariables[f] ]

InvDFTransform[ DFTData[x_, Start[st_], Finish[end_], KVariables[k_]], N_ ] :=
	If [ SameQ[ end - st + 1, N ],
	     InvDFTransform[ x, N, k ],
	     Message[ InvDFTransform::badlength, N, end - st + 1 ] ]

InvDFTransform[ f_, N_ ] :=
	InvDFTransform[ f, N, DummyVariables[Length[N], Global`k] ]

InvDFTransform[ f_, N_, k_ ] :=
	InvDFTransform[ f, N, k, DummyVariables[Length[k], Global`n] ]

InvDFTransform[ fun_, N_, k_, n_, options___ ]:=
	Block [	{notenough, nvars, op, w, wvars}, 
		notenough = ( Length[n] < Length[k] );
		If [ notenough, Message[Transform::notenough, "k (frequency)"]];
		nvars = If [ notenough,
			     DummyVariables[Length[n], Global`n],
			     n ];
		op = ToList[options] ~Join~ Options[DFTransform];
		wvars = DummyVariables[Length[n], w];

		If [ Length[k] > 1, 
		     MultiDInvDFT[ fun, N, k, nvars, wvars, op ],
		     MyInvDFT[ fun, N, k, nvars, wvars, op ] ] ]

(*   Multidimensional inverse DFT  *)
MultiDInvDFT[ fun_, N_, k_, n_, w_, op_ ] :=
	Block [	{F1 = fun, jj, length},

		length = Length[Num];
		For [ jj = 1, jj <= length, jj++,
		      F1 = MyInvDFT[ F1, N[[jj]], k[[jj]],
				     n[[jj]], w[[jj]], op ] ];

		F1 ]

(*  MyInvDFT  *)
MyInvDFT[ x_, N_, k_, n_, w_, op_ ] :=
	Block [	{newexpr, newx, oldexpr = Null, ret, trace},
	
		newx = x /. TrigToExpRules;
		newexpr = myinvDFT[newx, k, n, w, L, True, InvFixUp[op]];

		trace = SameQ[ Replace[Dialogue, op], All ];
		While [ ! SameQ[newexpr, oldexpr],
			If [ trace, Print[newexpr]; Print[ "which becomes" ] ];
			oldexpr = newexpr;
			newexpr = MapAll[invtransform, oldexpr] ];

		oldexpr = TransformFixUp[ oldexpr, k, n, op, myinvDFT, False,
					  DFTransform, Null, Null ];

		newexpr = postinvtransform[oldexpr];
		While [ ! SameQ[newexpr, oldexpr],
			If [ trace, Print[newexpr]; Print[ "which becomes" ] ];
			oldexpr = newexpr;
			newexpr = postinvtransform[ oldexpr ] ];

		newexpr ]

(*  FixUp  *)
InvFixUp[op_] := { Dialogue -> Replace[Dialogue, op],
		   Terms -> Replace[Terms, op],
		   Definition -> Replace[Definition, op] }

(*  invtransform  *)
(* invtransform[myinvDFT[x_, rest__]] :=
	Replace[myinvDFT[x, rest], InvDFTRules] *)
invtransform[x_] := x		(* no transform pairs, etc., encoded yet *)

(*  postinvtransform  *)
postinvtransform[x_] := x /. postinvrules

postinvrules = {

	myinvDFT[ x_, k_, n_, w_, L_, s_, op_ ] :>
		InvDTFTransform[x /. (k -> w L / (2 Pi)), w, n, op]

}

Format[ myinvDFT[ x_, k_, n_, w_, L_, s_, op_ ] ] :=
	SequenceForm[ ColumnForm[ { "DFT" Superscript[-1], 
		         StringJoin["   ", ToString[L], ", ", ToString[k]] } ],
		      { x } ]


(*  E N D     I N V E R S E     D F T  *)


(*  T R I G     R E W R I T E     R U L E S  *)

(* Trigonometric to Complex Exponential Simplification rules *)

TrigToExpRules = {
	Sin[ a__ ]	:> 	( Exp[ I a ] - Exp[ - I a ] ) / ( 2 I ),
	1/Sin[ a__ ]	:> 	1/( ( Exp[ I a ] - Exp[ - I a ] ) / ( 2 I ) ) ,
	Cos[ a__ ]	:>	( Exp[ I a ] + Exp[ - I a ] ) / 2,
	1/Cos[ a__ ]	:>	1/( ( Exp[ I a ] + Exp[ - I a ] ) / 2 ),
	Tan[ a__ ]	:>	Sin[ a ] / Cos[ a ],
	Cot[ a__ ]	:>	Cos[ a ] / Sin[ a ],
	Sec[ a__ ]	:>	1 / Cos[ a ],
	Csc[ a__ ]	:>	1 / Sin[ a ]
}

(* Exponential rewrite rules *)

ExpToTrigRules = {
	(a_. Exp[ Complex[0, b_] w_. ] + a_. Exp[ Complex[0, c_] w_. ] + x_.) :>
		2 a Cos[Abs[b] w] + x /;
		( b == -c ),

	(a_. Exp[ Complex[0, b_] w_. ] - a_. Exp[ Complex[0, c_] w_. ] + x_.) :>
		2 a Sin[b w] + x /;
		( b == -c ),

	(d_ - d_. Exp[ Complex[0, b_] c_. ]) :>
		-2 I d Exp[ I b c / 2 ] Sin[ b c / 2 ],

	(d_. Exp[ Complex[0, b_] c_. ] - d_) :>
		2 I d Exp[ I b c / 2 ] Sin[ b c / 2 ],

	(d_ + d_. Exp[ Complex[0, b_] c_. ]) :>
		2 d Exp[ I b c / 2 ] Cos[ b c / 2 ]
}

(*  E N D     T R I G     R E W R I T E     R U L E S  *)


(*  E N D     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ];


(*  H E L P     I N F O R M A T I O N  *)

Combine [ SPfunctions, { DFTransform, InvDFTransform } ]
Protect [ DFTransform, InvDFTransform ]


(*  E N D I N G     M E S S A G E  *)

Print[ "The discrete Fourier transform (DFT) rule bases are loaded." ]
Null
