(*  :Title:	Discrete-Time Fourier Transforms *)

(*  :Authors:	Wally McClure, Brian Evans, James McClellan  *)

(*
    :Summary:	This file will computes the forward and inverse
		discrete-time Fourier Transform of an input signal.
 *)

(*  :Context:	Signal Processing`Digital`DTFT`     *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*
    :History:	start			October 2, 1989
		made into package	April 18, 1990
		made into release	May 25, 1990
		debugged fully		September 6, 1990
 *)

(*  :Keywords:	discrete-time Fourier transform  *)

(*
    :Source:	{Discrete-Time Signal Processing} by A. V. Oppenheim and
		  Ronald W. Schafer.  Prentice Hall.  1989
 *)

(*
    :Warning:	If the input does not match any given transform, a nasty
		expression is returned. (But isn't this true for any
		invalid transform in the signal processing packages?)
 *)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*  :Limitation:  *)

(*
    :Discussion:  This  file  implements the forward and inverse DTFT.
		  The DTFTransform rule base (DTFTRules) consists of 5
		  property rules, 2  structure  rules  (upsampling and
		  downsampling),  13 lookup rules (for those functions
		  which  have  a  DTFT  but  not a z-transform), and 5
		  strategies --   distribute polynomials, expand terms
		  in cosine, sine,  and  exponential functions,  and a
		  call to  the  forward  z-transform rule base (if all
		  else fails).  The only functions supporting the for-
		  ward DTFT which depend  on the implementation of the
		  z-transform are ZtoDTFT and DTFTtoZ.

		  The InvDTFTransform is similarly arranged, having 11
		  lookup  rules,  6  property  rules, and 6 strategies
		  (including  a  call  to the inverse z-transform rule
		  base).  Only one rule in  InvDTFTRules  is dependent
		  on  the  implementation  of  the inverse z-transform
		  rule base InvZTransform.

	MyDTFT[f, n, w] determines the discrete-time Fourier transform
	of f with respect to n, where w is the frequency variable.

	MyInvDTFT[f, w, n] determines the inverse discrete-time Fourier
	transform of f with respect to w, where n is a "time" variable.

	These transform pairs are valid with respect to the DTFT formulas:

	    n
	(-1)                 w
	-----  <----->  - ------ CPulse[2 Pi, w + Pi]
	  n               2 I Pi

              n
	          (-1)
	- 2 I Pi  -----  <----->  w CPulse[2 Pi, w + Pi]
	            n
 *)

(*
    :Functions:	DTFTransform
		InvDTFTransform
		MultiDDTFT
		MultiDInvDTFT
 *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ];


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Digital`DTFT`",
	      "SignalProcessing`Digital`DSupport`",
	      "SignalProcessing`Digital`InvZTransform`",
	      "SignalProcessing`Digital`ZTransform`",
	      "SignalProcessing`Digital`ZSupport`",
	      "SignalProcessing`Support`TransSupport`",
	      "SignalProcessing`Support`ROC`",
	      "SignalProcessing`Support`SigProc`",
	      "SignalProcessing`Support`SupCode`" ]
											
(*  U S A G E     I N F O R M A T I O N  *)

DTFTData::usage =
	"DTFTData is a data-tag."

DTFTransform::usage =
	"DTFTransform[function, in variable(s), out variable(s), options] \
	returns the discrete-time Fourier transform of the input function. \
	Note that this transform may rely on the z-transform and the \
	substitution z = Exp[I w], which is not valid if the region of \
	convergence does not include the unit circle.  Also note that \
	only the first two arguments are required."

InvDTFTransform::usage =
	"InvDTFTransform[function, in variable(s), out variable(s), options] \
	return the inverse discrete-time Fourier transform (inverse DTFT) \
	of the input function. \
	The first two arguments are required. \
	By default, the inverse DTFT rule base will apply the definition \
	of the inverse DTFT when all other attempts to find the inverse \
	have failed."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin[ "`Private`" ]


(*  B E G I N     P A C K A G E  *)

(*  Generate a unique symbol with a Global context.  *)
unique[x_] :=
	Block [	{context, newsymbol},
		context = $Context;
		$Context = "Global`";
		newsymbol = Unique[x];
		$Context = context;
		newsymbol ]


(*  B E G I N     F O R W A R D     D T F T  *)

(*  Conversion from operator notation to function notation  *)
Unprotect[DTFT]
DTFT/: TheFunction [ DTFT[n_, w_][f_] ] := DTFTransform[f, n, w]
Protect[DTFT]

(*  Options for the DTFT  *)
DTFTransform/: Options[ DTFTransform ] :=
	{ Dialogue -> False } ~Join~ Options [ ZTransform ]

(*  Extension of TheFunction operator.  *)
DTFTData/: TheFunction [ DTFTData[ trans_, var_ ] ] := trans

(*  Magnitude/Phase plot of a DTFT object *)
mppoptions = {	Domain -> Continuous, DomainScale -> Linear,
		MagRangeScale -> Linear, PhaseRangeScale -> Degree,
		PlotRange -> All }

DTFTData/: MagPhasePlot[DTFTData[trans_, FVariables[w_Symbol]]] :=
	MagPhasePlot[ trans, {w, -Pi, Pi}, mppoptions ]

DTFTData/: MagPhasePlot[DTFTData[trans_, FVariables[{w1_Symbol, w2_Symbol}]]] :=
	MagPhasePlot[ trans, {w1, -Pi, Pi}, {w2, -Pi, Pi}, mppoptions ]

(*  Input a function and test to see if needs to be sent to	*)
(*  the multidimensional or one-dimensional Fourier transform.	*)
DTFTransform[ f_ ] :=
	Message[ Transform::novariables, "n (time)", GetVariables[f] ]
DTFTransform[ f1_, n_ ] :=
	DTFTransform [ f1, n, DummyVariables[Length[n], Global`w] ]
DTFTransform[ f1_, n_, w_, options___ ] :=
	Block [	{newtrans, op, trans, vars, zvars},
		vars = If [ Length[n] != Length[w],
			    DummyVariables[Length[n], Global`w],
			    w ];
		zvars = DummyVariables[Length[n], z];

		op = ToList[options] ~Join~ Options[DTFTransform];
		trans = If [ Length[n] > 1, 
			     MultiDDTFT[f1, n, vars, zvars, op],
			     MyDTFT[f1, n, vars, zvars, op] ];

		If [ InformUserQ[Replace[Dialogue, op]],
		     Scan[explain, trans, Infinity] ];

		newtrans = ReplaceRepeated[trans, SPSimplificationRules ];
		DTFTData[ Simplify[newtrans], FVariables[vars] ] ]

(* Multidimensional discrete-time Fourier transform *)
MultiDDTFT[ f1_, t_, w_, z_, oplist_ ]:= 
	Block [	{func = f1, i, len},
		len = Length[t];
		For [ i = 1, i <= len, i++,
		      func = MyDTFT[ func, t[[i]], w[[i]], z[[i]], oplist ] ];
		func ]

(*  One-dimensional discrete-time Fourier transform  *)
MyDTFT[ x_, n_, w_, z_, oplist_ ] :=
	Block [ {newexpr = myDTFT[x, n, w, z, FixUp[oplist], initDTFTstate[]],
		 oldexpr = Null, ret, trace},

		trace = SameQ[ Replace[Dialogue, oplist], All ];
		Off[ Power::infy ];
		While [ ! SameQ[newexpr, oldexpr], 
			If [ trace, Print[newexpr]; Print[ "which becomes" ] ];
			oldexpr = newexpr;
			newexpr = MapAll[transform, oldexpr] ];

		oldexpr = TransformFixUp[ oldexpr, n, w, oplist, myDTFT,
					  False, DTFTransform, Null, Null ];

		newexpr = posttransform[oldexpr];
		While [ ! SameQ[newexpr, oldexpr], 
			If [ trace, Print[newexpr]; Print[ "which becomes" ] ];
			oldexpr = newexpr;
			newexpr = posttransform[oldexpr] ];

		If [ ! FreeQ[{newexpr}, z],
		     newexpr = ( newexpr /. z -> Exp[I w] ) ];
		On[ Power::infy ];
		If [ trace, Print[newexpr] ];
		If [ SameQ[Head[newexpr], ZTransData],
		     TheFunction[newexpr],
		     newexpr ] ]

(*  Supporting Routines  *)

explain[ myDTFT[ x_, n_, rest__ ] ] :=
	Message[ Transform::incomplete,
		 "forward discrete-time Fourier transform", x, n ]

(*  FixUp --  removes redundancy and TransformLookup in options list  *)
FixUp[ op_ ] := { Dialogue -> Replace[Dialogue, op] }

posttransform[x_] := ( x /. postrules )

postrules = {
	derivative[x_, w_, 1] :>
		I D[ posttransform[x], w ],
	derivative[x_, w_, k_] :>
		I^k D[ posttransform[x], {w, k} ],
	DTFTtoZ[x_] :>
		Transform[ x, 0, Infinity ],
	DTFTtoZ[x_, w_, z_] :>
		Transform[ substituteforw[x, w, -I Log[z]], 0, Infinity ],
	myDTFT[ZTransData[x_, zrest__], rest__] :>
		x,
	substituteforw[x_, w_, wnew_] :>
		( posttransform[x] /. w -> wnew ),
	x_ :> x
}

transform[myDTFT[x_, rest__]] := Replace[myDTFT[x, rest], DTFTRules]
transform[x_] := x

ZtoDTFT[ SignalProcessing`Digital`ZTransform`Private`forwardz[x_, zrest__], n_, w_, z_, rest__ ] :=
	DTFTtoZ[ myDTFT[x, n, w, z, rest], w, z ]
ZtoDTFT[ x_, n_, w_, z_, rest__ ] := x


(*  Printed forms of supported routines for Dialogue -> All option  *)

Format[ myDTFT[ x_, n_, w_, z_, options_, flag_ ] ] :=
	SequenceForm[ ColumnForm[{"DTFT",
				  "     " ~StringJoin~ ToString[n]}],
		      { x } ]

Format[ derivative[x_, w_Symbol, k_] ] := 
	SequenceForm[ ColumnForm[{"D" Superscript[k],
				  "  " ~StringJoin~ ToString[w]}],
		      { x } ]

Format[	substituteforw[x_, w_, wnew_] ] :=
	SequenceForm[ {x}, Subscript[w -> wnew] ]

Format[	DTFTtoZ[x_, w_, z_] ] :=
	SequenceForm[ {x}, Subscript[w -> -I Log[z]] ]


(*  S T A T E  *)

initDTFTstate[] := True
nullDTFTstate[] := False


(*  B E G I N     D T F T     R U L E     B A S E  *)


DTFTRules = {

(*  A.	Lookup rules		*)
(*	1.  Pulse		*)
myDTFT[ Pulse[L_, n_ + n0_.], n_, w_, z_, options_, flag_ ] :>
	Exp[I w n0] Exp[- I w (L - 1) / 2] Dirichlet[L, w] /;
	FreeQ[{L, n0}, n],

myDTFT[ A_. Step[n_ + n0_.] - A_. Step[n_ + n1_.] + x_, n_, w_, z_, options_, flag_ ] :>
	myDTFT[ A Pulse[Abs[n1 - n0], n + Min[-n0, -n1]],
		n, w, z, options, flag ] +
	myDTFT[ x, n, w, z, options, flag ] /;
	FreeQ[{n0, n1}, n],

(*	2.  Step		*)
myDTFT[ Step[n_ + n0_.], n_, w_, z_, options_, flag_ ] :> 
	Block [	{l = unique["l"]},
		Exp[I w n0] / ( 1 - Exp[-I w] ) + 
		  Pi Exp[I w n0] Summation[l, -Infinity, Infinity, 1]
				       [Delta[w + 2 Pi l]] ] /;
	FreeQ[n0, n],

(*	3.  Summation forms	*)
myDTFT[ Summation[k_, -Infinity, Infinity, 1][x_],
		n_, w_, z_, options_, flag_ ] :>
	Block [ {dtft},
		dtft = myDTFT[x, n, w, z, options, flag];
		dtft / (1 - Exp[-I w]) +
		  Pi substituteforw[dtft, w, z]
		  Summation[k,-Infinity,Infinity,1][Delta[w - 2 Pi k]] ] /;
	FreeQ[k, n],

myDTFT[ Summation[k_, 0, N_, 1][a_. Exp[I 2 Pi k_ n_ / (N_ + 1)]],
		n_, w_, z_, options_, flag_ ] :>
	2 Pi Summation[k, -Infinity, Infinity,1][a Delta[w - 2 Pi k/(N+1)]] /;
	FreeQ[{k, c, N}, n],

(*	4.  Infinite extent cosine	*)
myDTFT[ Cos[w0_. n_ + b_.], n_, w_, z_, options_, flag_ ] :> 
	Block [	{l = unique["l"]},
		Pi Summation[l,-Infinity,Infinity,1]
			    [Exp[I b] Delta[w - w0 + 2 Pi l] +
			     Exp[- I b] Delta[w + w0 + 2 Pi l]] ] /;
	FreeQ[{b, w0}, n],

(*	5.  Infinite extent sine	*)
myDTFT[ Sin[w0_. n_ + b_.], n_, w_, z_, options_, flag_ ] :> 
	Block [	{l = unique["l"]},
		Pi Summation[l,-Infinity,Infinity,1]
			    [- I Exp[I b] Delta[w - w0 + 2 Pi l] +
			     I Exp[I b] Delta[w + w0 + 2 Pi l]] ] /;
	FreeQ[{b, w0}, n],

(*	6.  Constant function		*)
myDTFT[ 1, n_, w_, z_, options_, flag_ ] :> 
	Block [	{l = unique["l"]},
		2 Pi Summation[l,-Infinity,Infinity,1][Delta[w + 2 Pi l]] ],

(*	7.  Sinc and other 1/n		*)
myDTFT[ (-1)^(n_ + a_.) / ( n_ - 1 ), n_, w_, z_, options_, flag_ ] :>
	(-1)^a w Exp[- I w] CPulse[2 Pi, w + Pi] /;
	FreeQ[a, n],

myDTFT[ 1/n_, n_, w_, z_, options_, flag_ ] :> - 2 Pi I CStep[w],

myDTFT[ Sin[B_. n_] / n_, n_, w_, z_, options_, flag_ ] :> 
	Pi ( CStep[ w + B ] - CStep[ w - B ] ) /;
	FreeQ[B, n],

myDTFT[ Sinc[a_. n_ + b_.], n_, w_, z_, options_, flag_ ] :> 
	Exp[I b / a] Pi / a ( CStep[ w + a ] - CStep[ w - a ] ) /;
	FreeQ[{a,b}, n],

myDTFT[ (Sinc[a_. n_ + b_.])^2, n_, w_, z_, options_, flag_ ] :> 
	Exp[I b / a] (Pi / a)^2 ( Unit[-2][w + 2 a] - 2 Unit[-2][w] + Unit[-2][w - 2 a] ) /;
	FreeQ[{a,b}, n],


(*  B.	Properties		*)
(*	1.  Multiplication by constant  *)
myDTFT[ a_ x_, n_, w_, z_, options_, flag_ ] :>
	a myDTFT[x, n, w, z, options, flag] /;
	FreeQ[a, n],

(*	2.  Additivity property	*)
myDTFT[ x_ + y_, n_, w_, z_, options_, flag_ ] :> 
	myDTFT[x, n, w, z, options, flag] + myDTFT[y, n, w, z, options, flag],

(*	3.  Multiplication by n	*)
myDTFT[ n_^a_. b_, n_, w_, z_, options_, flag_ ] :>
	derivative[myDTFT[b, n, w, z, options, flag], w, a] /;
	FreeQ[a,n] && Implies[ NumberQ[a], IntegerQ[a] && a > 0 ],

(*	4.  Delay by m		*)
myDTFT[ f_. Step[n_ + m_], n_, w_, z_, options_, flag_ ] :>
	Exp[I m w] myDTFT[(f /. n->(n - m)) Step[n], n, w, z, options, flag] /;
	FreeQ[m, n],

(*	5.  Multiplication by exponential	*)
myDTFT[ a_. Exp[Complex[0, w0_] w1_. n_ + b_.], n_, w_, z_, options_, flag_ ] :>
	Exp[b] substituteforw[ myDTFT[a, n, w, z, options, flag],
			       w, w - w0 w1 ] /;
	FreeQ[{b, w0, w1}, n],

(*  C.	Structures		*)
(*	1.  Upsampling		*)
myDTFT[ Upsample[l_, n_] [ x_ ], n_, w_, z_, options_, flag_ ] :>
	ScaleAxis[l, w] [ myDTFT[x, n, w, z, options, flag] ],

(*	2.  Downsampling	*)
myDTFT[ Downsample[m_, n_] [ x_ ], n_, w_, z_, options_, flag_ ] :>
	Aliasby[m, w][ myDTFT[x, n, w, z, options, flag] ],

(*  D.	Strategies		*)
(*	1.  Multiply out terms in cosine, sin, and exponential functions.  *)
myDTFT[ Exp[a_] x_., n_, w_, z_, options_, flag_ ] :>
	myDTFT[ Exp[Expand[a]] x, n, w, options, flag ] /;
	! SameQ[Exp[a], Exp[Expand[a]]],

myDTFT[ Cos[a_] x_., n_, w_, z_, options_, flag_ ] :>
	myDTFT[ Cos[Expand[a]] x, n, w, options, flag ] /;
	! SameQ[Cos[a], Cos[Expand[a]]],

myDTFT[ Sin[a_] x_., n_, w_, z_, options_, flag_ ] :>
	myDTFT[ Sin[Expand[a]] x, n, w, options, flag ] /;
	! SameQ[Sin[a], Sin[Expand[a]]],

(*	2.  Expand out products like (n + 1)(n + 2) ...	*)
myDTFT[ x_, n_, w_, z_, options_, flag_ ] :>
	myDTFT[ Distribute[x], n, w, z, options, flag ] /;
	SameQ[Head[x], Times] && ! SameQ[x, Distribute[x]],

(*	3.  Use the z-transform	*)
myDTFT[ x_, n_, w_, z_, options_, True ] :> 
	Block [	{nullstate, ztrans},
		nullstate = nullDTFTstate[];
		ztrans = ZTransform[ x, n, z,
				     {Dialogue -> False} ~Join~ options ];
		If [ InformUserQ[Replace[Dialogue, options]],
		     Print[ "( after taking the z-transform of" ];
		     Print[ "  ", x ];
		     Print[ "  which yields" ];
		     Print[ "  ", ztrans ];
		     Print[ "  and adjusting the result )" ] ];
		If [ SameQ[Head[ztrans], ZTransData],
		     ( TheFunction[ztrans] /. z -> Exp[I w] ),
		     myDTFT[ MapAll[ ZtoDTFT[#1, n, w, z, options, nullstate]&,
				     ztrans ],
			     n, w, z, options, nullstate ] ] ]

}

(*  E N D     R U L E     B A S E  *)

(*  E N D     F O R W A R D     D T F T  *)


(*  B E G I N     I N V E R S E     D T F T  *)

InvDTFTransform::noncont =
	"Discrete operators and functions were converted into continuous ones."

(*  Conversion from operator notation to function notation  *)
Unprotect[InvDTFT]
InvDTFT/: TheFunction [ InvDTFT[w_, n_][f_] ] := InvDTFTransform[f, w, n]
Protect[InvDTFT]

(*  Give it options  *)
InvDTFTransform/: Options[InvDTFTransform] :=
	{ Terms -> False, Dialogue -> False, Definition -> True } ~Join~
	Options[InvZTransform]

(*  Input a function and test to see if needs to be sent to the    *)
(*  multidimensional or one-dimensional inverse Fourier transform. *)
InvDTFTransform[ DTFTData[ trans_, FVariables[vars_] ] ] :=
	InvDTFTransform [ trans, vars ]

InvDTFTransform[ f_ ] :=
	Message[ Transform::novariables, "w (frequency)", GetVariables[f] ]

InvDTFTransform[ f_, w_ ] :=
	InvDTFTransform[ f, w, DummyVariables[Length[w], Global`n] ]

InvDTFTransform[ f_, varin_, varout_, options___ ] :=
	Block [	{freqfun, invfun, op, vars, z, zvars},
		vars = If [ Length[varin] != Length[varout],
			    DummyVariables[Length[varin], Global`n],
			    varout ];

		zvars = DummyVariables[Length[varin], z];

		freqfun = ToContinuous[f];
		If [ ! SameQ[freqfun, f],
		     Message[InvDTFTransform::noncont] ];

		op = ToList[options] ~Join~ Options[InvDTFTransform];
		invfun = If [ Length[varin] > 1, 
			      MultiDInvDTFT[freqfun, varin, vars, zvars, op],
			      MyInvDTFT[freqfun, varin, vars, zvars, op] ];

		If [ InformUserQ[Replace[Dialogue, op]],
		     Scan[invexplain, invfun, Infinity] ];

		invfun ]

(*  Multidimensional inverse discrete-time Fourier transform  *)
MultiDInvDTFT[ f_, w_, n_, z_, op_ ] :=
	Block [	{func = f, i, len},
		len = Length[w];
		For [ i = 1, i <= len, i++,
		      func = MyInvDTFT[func, w[[i]], n[[i]], z[[i]], op] ];
		func ]				

(*  One-dimensional inverse discrete-time Fourier transform  *)
MyInvDTFT[ f_, w_, n_, z_, op_ ] :=
	Block [ {newexpr = myinvDTFT[f, w, n, z, InvFixUp[op], {True, True}],
		 oldexpr = Null, trace},
		trace = SameQ[ Replace[Dialogue, op], All ];
		While [ ! SameQ[newexpr, oldexpr], 
			If [ trace, Print[newexpr]; Print[ "which becomes" ] ];
			oldexpr = newexpr;
			newexpr = MapAll[findtransform, oldexpr] ];
		If [ trace, Print[newexpr] ];

		oldexpr = TransformFixUp[ oldexpr, w, n, op, myinvDTFT,
					  False, InvDTFTransform, Null, Null ];

		newexpr = postinvtransform[oldexpr];
		While [ ! SameQ[newexpr, oldexpr], 
			If [ trace, Print[newexpr]; Print[ "which becomes" ] ];
			oldexpr = newexpr;
			newexpr = postinvtransform[oldexpr] ];

		newexpr ]


(*  Supporting Routines  *)

invexplain[ myinvDTFT[ x_, w_, rest__ ] ] :=
	Message[ Transform::incomplete,
		 "inverse discrete-time Fourier transform", x, w ]

findtransform[myinvDTFT[x_, rest__]] :=
	Replace[myinvDTFT[x, rest], InvDTFTRules]
findtransform[x_] := x

(*  InvFixUp --  removes redundancy and TransformLookup in options list  *)
InvFixUp[ op_ ] := { Dialogue -> Replace[Dialogue, op],
		     Terms -> Replace[Terms, op],
		     Definition -> Replace[Definition, op] }

postinvtransform[x_] := ReplaceRepeated[x, postinvrules]

postinvrules = {
	substituteforn[x_, n_, newn_] :> ( x /. n -> newn ),

	(*	4.  Apply the definition		*)
	myinvDTFT[ a_ Pulse[l_, w_ + s_.], w_, n_, z_, options_, flag_ ] :>
		Integrate[ a, {w, -s, l - s} ] /;
		Replace[Definition, options] && N[(Pi > s) && (Pi < l - s)],

	myinvDTFT[ x_, w_, n_, z_, options_, flag_ ] :>
		Integrate[ x, {w, -Pi, Pi} ] /;
		Replace[Definition, options]
}


(*  Printed forms of supported routines for Dialogue -> All option  *)

Format[ myinvDTFT[ x_, w_, n_, z_, options_, flag_ ] ] :=
	SequenceForm[ ColumnForm[{"DTFT" Superscript[-1],
				  "     " ~StringJoin~ ToString[w]}],
		      { x } ]

Format[	substituteforn[x_, n_, newn_] ] :=
	SequenceForm[ {x}, Subscript[n -> newn] ]


(*  B E G I N     R U L E     B A S E  *)

InvDTFTRules = {

(*  A.	Lookup Rules		*)
(*	1.  Summation forms	*)
myinvDTFT[ Summation[k_,-Infinity,Infinity,1][Delta[w_ + w0_. + 2 Pi k_]],
		w_, n_, z_, options_, flag_ ] :>
	Exp[- I w0 n] / ( 2 Pi ) /;
	FreeQ[{k, w0}, w],

myinvDTFT[ Summation[k_,-Infinity,Infinity,1][Delta[w_ + w0_. + 2 Pi k_] +
		Delta[w_ + A_. - 2 Pi k_]], w_, n_, z_, options_, flag_ ] :>
	Cos[A n] / Pi /;
	( A == - w0 ) && FreeQ[{k, w0}, w],

myinvDTFT[ Summation[k_,-Infinity,Infinity,1][Delta[w_ + w0_. + 2 Pi k_] +
		b_ Delta[w_ + A_. - 2 Pi k_]], w_, n_, z_, options_, flag_ ] :>
	Sin[A n] / Pi /;
	( A == -w0 ) && b == -1 && FreeQ[{A, w0}, w],

myinvDTFT[ Summation[k_,-Infinity,Infinity,1][Delta[w_ - 2 Pi k_]],
		w_, n_, z_, options_, flag_ ] :>
	1 /;
	FreeQ[k, w],

myinvDTFT[ Summation[k_,-Infinity,Infinity,1][Delta[w_ - 2 Pi k_ / N_]],
		w_, n_, z_, options_, flag_ ] :>
	N / (2 Pi) Summation[k,-Infinity,Infinity,1][Delta[n - k N]] /;
	FreeQ[{k, N}, w],

(*	2.  Constant function	*)
myinvDTFT[ 1, w_, n_, z_, options_, flag_ ] :> Impulse[n],

(*	3.  Pulse in continuous frequency	*)
myinvDTFT[ CPulse[L_, w_ + w1_.], w_, n_, z_, options_, flag_ ] :>
	L Exp[I w1 n] Exp[- I L n / 2] Sinc[L n / 2] / ( 2 Pi ) /;
	FreeQ[{L, w1}, w],

myinvDTFT[ A_. CStep[w_ + w0_.] - A_. CStep[w_ + w1_.] + x_., w_, n_, z_, options_, flag_ ] :>
	myinvDTFT[ A CPulse[w0 - w1, w + w0], w, n, z, options, flag ] +
	myinvDTFT[ x, w, n, z, options, flag ] /;
	FreeQ[{w0, w1}, w],

(*	4.  Step function	*)
myinvDTFT[ 1 / ( 1 - Exp[- I w_] ) + Pi Summation[k_,-Infinity,Infinity,1][Pi Delta[w_ + 2 Pi k_]], w_, n_, z_, options_, flag_] :> 
	Step[n] /;
	FreeQ[k, w],

(* 	5.  Dirichlet kernel (aliased sinc)	*)
myinvDTFT[ Dirichlet[L_, w_ + w0_.], w_, n_, z_, options_, flag_ ] :>
	Exp[- I w0 n] Pulse[L, n + (L - 1)/2] /;
	FreeQ[{L, w0}, w],

(*	6.  Differentiator (high pass) filter	*)
myinvDTFT[ w_ CPulse[2 Pi, w_ + Pi], w_, n_, z_, options_, flag_ ] :>
	-2 I Pi (-1)^n / n,


(*  B.	Properties				*)
(*	1.  Multiplication by a constant	*)
myinvDTFT[ a_ x_, w_, n_, z_, options_, flag_ ] :>
	a myinvDTFT[x, w, n, z, options, flag] /;
	FreeQ[a, w],

(*	2.  Additivity				*)
myinvDTFT[ a_ + b_, w_, n_, z_, options_, flag_ ] :>
	myinvDTFT[a, w, n, z, options, flag] +
	myinvDTFT[b, w, n, z, options, flag],

(*	3.  Multiplication by a complex exp.	*)
myinvDTFT[ Exp[Complex[0, n1_] n0_. w_ + b_.] x_., w_, n_, z_, options_, flag_ ] :>
	Exp[b] substituteforn[ myinvDTFT[x, w, n, z, options, flag],
			       n, n + n0 n1 ] /;
	FreeQ[{b, n0, n1}, w],

(*	4.  Summation				*)
myinvDTFT[ x_. / ( 1 - Exp[- I w_] ) + Pi Summation[k_,-Infinity,Infinity,1][Delta[w_ + 2 Pi k_]], w_, n_, z_, options_, flag_ ] :>
	Summation[k,-Infinity,Infinity,1][myinvDTFT[x, w, n, z, options, flag]] /;
	FreeQ[k, w],

(*	5.  Upsampling		*)
myinvDTFT[ ScaleAxis[l_, w_] [ x_ ], w_, n_, z_, options_, flag_ ] :>
	Upsample[l, n] [ myinvDTFT[x, w, n, z, options, flag] ],

(*	6.  Downsampling	*)
myinvDTFT[ Aliasby[m_, w_] [ x_ ], w_, n_, z_, options_, flag_ ] :>
	Downsample[m, n][ myinvDTFT[x, w, n, z, options, flag] ],


(*  C.	Strategies					*)
(*	1.  Collect all subexpressions in terms of w	*)
myinvDTFT[ x_, w_, n_, z_, options_, {True, flag2_} ] :>
	myinvDTFT[ MyCollectAll[x, w], w, n, z, options, { False, flag2 } ],

(*	2.  Expand out products like (w + 1)(w + 2) ...	*)
myinvDTFT[ x_, w_, n_, z_, options_, flag_ ] :>
	myDTFT[ Distribute[x], n, w, z, options, flag ] /;
	SameQ[Head[x], Times] && ! SameQ[x, Distribute[x]],

(*	3.  Take inverse z-transform			*)
myinvDTFT[ x_, w_, n_, z_, options_, {flag1_, True} ] :>
	Block [	{newx, invztrans},
		newx = ( x /. w -> ( - I Log[z] ) ) /. SPSimplificationRules;
		invztrans = InvZTransform[ newx, z, n,
					   {Dialogue -> False} ~Join~ options ];
		If [ InformUserQ[Replace[Dialogue, options]],
		     Print[ "( after taking the inverse z-transform of" ];
		     Print[ "  ", x ];
		     Print[ "  which yields" ];
		     Print[ "  ", invztrans ];
		     Print[ "  and adjusting the result )" ] ];
		If [ InvalidInvZTransformQ[invztrans],
		     myinvDTFT[ invztrans, w, n, z, options, {flag1, False} ],
		     invztrans ] ],

(*	    Clean up expression returned from inverse z-transform	*)
myinvDTFT[SignalProcessing`Digital`InvZTransform`Private`myinvz[x_, z_, rest__], w_, n_, z_, options_, flag_ ] :>
	Block [	{dtftform},
		dtftform = ( x /. z -> Exp[I w] ) /. SPSimplificationRules;
		myinvDTFT[dtftform, w, n, z, options, flag] ]

}


(*  E N D     R U L E     B A S E  *)

(*  E N D     I N V E R S E     D T F T  *)


(*  E N D     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ];


(*  H E L P     I N F O R M A T I O N  *)

Combine [ SPfunctions, { DTFTransform, InvDTFTransform } ]
Protect [ DTFTransform, InvDTFTransform ]


(*  E N D I N G     M E S S A G E  *)

Print["The discrete-time Fourier transform (DTFT) rule bases are loaded."]
Null
