/****************************************************************************
**
**    genexp.c                        NQ                       Werner Nickel
**
**    Copyright 1992                            Mathematics Research Section
**                                           School of Mathematical Sciences 
**                                            Australian National University
**
**    This file contains an arithmetic for generator exponent strings.
**    There are the following functions :
**
**    void GEStrCopy( gs, hs ) ......... Copy the word gs to the word hs. It
**                                       is assumed that storage for hs has
**                                       already been allocated.
**    int  GEStrLength( gs )   ......... Return the length of the word gs,
**                                       i.e., the number of powers of
**                                       generators in gs.
**    word GEStrGen( g )       ......... Return a word for generator number g.
**    word GEStrMult( gs, hs ) ......... Return the product of the word gs
**                                       and the word hs and free gs and hs.
**    word GEStrPow( gs, pn )  ......... Return the word gs raised to the
**                                       power of (*pn). Free gs.
**    word GEStrConj( gs, hs ) ......... Return the conjugate of the word gs
**                                       by the word hs and free gs and hs.
**    word GEStrComm( gs, hs ) ......... Return the commutator [gs,hs] of the
**                                       words gs and hs. Free gs and hs.
**    word GEStrRel( gs, hs )  ......... Return the quotient gs * hs^-1 of 
**                                       the words gs and hs. Free gs and hs.
**    void GEStrInit()         ......... Initialize the evaluator with this
**                                       arithmetic module.
**    void GEStrPrint( gs )    ......... Print the word gs. The word gs is
**                                       not freed.
**
**    static word GEStrMultLocal( gs, hs )
**                                       The same as GEStrMult() but gs and
**                                       are not freed.
**    static word GEStrPowLocal( gs, pn )
**                                       The same as GEStrPow() but gs and
**                                       are not freed.
*/

#include "presentation.h"
#include "genexp.h"

void	GEStrCopy( gs, hs )
word	gs, hs;

{	while( gs->g != EOW ) *hs++ = *gs++;
	*hs = *gs;
}

int	GEStrLength( gs )
word	gs;

{	int	l = 0;

	while( gs->g != EOW ) { gs++; l++; }
	return l;
}

word	GEStrGen( g )
gen	g;

{	word	gs;

	gs = (word)Allocate( 2*sizeof( gpower ) );
	gs[0].g = g;
	gs[0].e = 1;
	gs[1].g = EOW;

	return gs;
}

static word	GEStrMultLocal( gs, hs )
word	gs, hs;

{	word	 fs, ffs, ggs, hhs;
	unsigned lg, lh;

	lg = GEStrLength(gs); lh = GEStrLength(hs);

	/* first perform free reduction. */
	hhs = hs;
	ggs = gs+lg-1;
	while( lg > 0 && lh > 0 && ggs->g == -hhs->g && ggs->e == hhs->e ) {
	    lg--; ggs--;
	    lh--; hhs++;
	}

	fs = (word)Allocate( (lg+lh+1) * sizeof(gpower) );
 
	/* Copy the first word except the last generator power. */
	ffs = fs;
	ggs = gs;
	while( lg > 1 ) { *ffs++ = *ggs++; lg--; }

	/* Now check if the last gen of g is the first gen of h. */
	if( lg > 0 && lh > 0 && ggs->g == -hhs->g ) {
	    if( ggs->e > hhs->e ) { ffs->g = ggs->g; ffs->e = ggs->e-hhs->e; }
	    else                  { ffs->g = hhs->g; ffs->e = hhs->e-ggs->e; }
	    ffs++;
	    lg--; ggs--;
	    lh--; hhs++;
	}
	else if( lg > 0 && lh > 0 && ggs->g == hhs->g ) {
	    ffs->g = ggs->g; ffs->e = ggs->e + hhs->e;
	    ffs++;
	    lg--; ggs--;
	    lh--; hhs++;
	}
	else if( lg > 0 ) *ffs++ = *ggs++;

	/* Copy the second word. */
	while( lh-- ) *ffs++ = *hhs++;
	ffs->g = EOW;
	ffs->e = 0;

	return fs;
}

word	GEStrMult( gs, hs )
word	gs, hs;

{	word	fs;

	fs = GEStrMultLocal( gs, hs );
	Free( (void *)gs ); Free( (void *)hs );

	return fs;
}

word	GEStrPowLocal( gs, pn )
word	gs;
int	*pn;

{	int	 e;
	unsigned lg;
	word	 hs, ggs;

	e = *pn;

	if( e < 0 ) {
	    lg = GEStrLength( gs );
	    ggs = (word)Allocate( (lg+1)*sizeof(gpower) );
	    ggs += lg;
	    ggs->g = EOW; ggs--;
	    while( lg-- ) {
		ggs->g = -gs->g;
		ggs->e = gs->e;
		ggs--; gs++;
	    }
	    e = -e;
	    gs = ggs+1;
	}
	else if( e > 0 ) {
	    lg = GEStrLength( gs );
	    ggs = (word)Allocate( (lg+1)*sizeof(gpower) );
	    GEStrCopy( gs, ggs );
	    gs = ggs;
	}
	if( e == 1 ) return gs;

	hs = (word)Allocate( sizeof(gpower) ); hs[0].g = EOW;
	while( e > 0 ) {
	    if( e % 2 == 1 ) {
		ggs = hs; hs = GEStrMultLocal( hs, gs ); Free((void *)ggs);
	    }
	    if( e > 1 ) {
		ggs = gs; gs = GEStrMultLocal( gs, gs ); Free((void *)ggs);
	    }
	    e = e / 2;
	}
	Free((void *)gs);

	return hs;
}

word	GEStrPow( gs, pn )
word	gs;
int	*pn;

{	word	fs;

	fs = GEStrPowLocal( gs, pn );
	Free( (void *)gs );

	return fs;
}

word	GEStrConj( gs, hs )
word	gs, hs;

{	word	ggs, hhs;
	int	e = -1;

	hhs = GEStrPowLocal( hs, &e );
	ggs = GEStrMultLocal( hhs, gs ); Free( (void *)hhs );
	hhs = GEStrMultLocal( ggs, hs ); Free( (void *)ggs );

	Free( (void *)gs ); Free( (void *)hs );
	return hhs;
}

word	GEStrComm( gs, hs )
word	gs, hs;

{	word	ggs, hhs;
	int	e = -1;

	ggs = GEStrMultLocal( hs, gs );
	hhs = GEStrPowLocal( ggs, &e );  Free( (void *)ggs );
	ggs = GEStrMultLocal( hhs, gs ); Free( (void *)hhs );
	hhs = GEStrMultLocal( ggs, hs ); Free( (void *)ggs );

	Free( (void *)gs ); Free( (void *)hs );
	return	hhs;
}

word	GEStrRel( gs, hs )
word	gs, hs;

{	word	ggs, hhs;
	int	e = -1;

	hhs = GEStrPowLocal( hs, &e );
	ggs = GEStrMultLocal( gs, hhs ); Free( (void *)hhs );

	Free( (void *)gs ); Free( (void *)hs );
	return ggs;
}

void	GEStrInit() {

	SetEvalFunc( TGEN,  (void *(*)())GEStrGen );
	SetEvalFunc( TMULT, (void *(*)())GEStrMult );
	SetEvalFunc( TPOW,  (void *(*)())GEStrPow );
	SetEvalFunc( TCONJ, (void *(*)())GEStrConj );
	SetEvalFunc( TCOMM, (void *(*)())GEStrComm );
	SetEvalFunc( TREL,  (void *(*)())GEStrRel );
	SetEvalFunc( TDRELL,(void *(*)())GEStrRel );
	SetEvalFunc( TDRELR,(void *(*)())GEStrRel );
}

void	GEStrPrint( gs )
word	gs;

{	if( gs->g != EOW)
	    if( gs->g > 0 ) {
		PrintGen( gs->g );
		if( gs->e > 1 ) printf( "^%d", gs->e );
	    }
	    else {
		PrintGen( -gs->g );
		printf( "^-%d", gs->e );
	    }
	else {
	    printf( "1" );
	    return;
	}
	gs++;
	    
	while( gs->g != EOW ) {
	    putchar( '*' );
	    if( gs->g > 0 ) {
		PrintGen( gs->g );
		if( gs->e > 1 ) printf( "^%d", gs->e );
	    }
	    else {
		PrintGen( -gs->g );
		printf( "^-%d", gs->e );
	    }
	    gs++;
	}
}
