#############################################################################
##
#A  graphlat.g                 	XGAP library               Susanne Keitemeier
##
#H  @(#)$Id: graphlat.g,v 1.5 1993/10/18 11:06:14 fceller Exp $
##
#Y  Copyright (C) 1993,  Lehrstuhl D fuer Mathematik,  RWTH, Aachen,  Germany
##
##  This file contains the non-interactive lattice program.
##
#H  $Log: graphlat.g,v $
#H  Revision 1.5  1993/10/18  11:06:14  fceller
#H  added fast updated
#H
#H  Revision 1.4  1993/10/05  12:33:26  fceller
#H  added '.isAlive'
#H
#H  Revision 1.4  1993/10/05  12:33:26  fceller
#H  added '.isAlive'
#H
#H  Revision 1.3  1993/08/13  13:37:04  fceller
#H  added resize
#H
#H  Revision 1.3  1993/08/13  13:37:04  fceller
#H  added resize
#H
#H  Revision 1.2  93/07/30  14:23:59  fceller
#H  added graphic lattice given by maximals
#H  
#H  Revision 1.1  1993/07/22  11:23:53  fceller
#H  Initial revision
##


#############################################################################
##

#V  GraphicLatticeOps . . . . . . . . . . . . . . . . . . . operations record
##
GraphicLatticeOps := Copy(GraphicSheetOps);
GraphicLatticeOps.name := "GraphicLatticeOps";


#############################################################################
##
#F  GraphicLatticeOps.Print( <S> )  . . . . . . . . . . . . . pretty printing
##
GraphicLatticeOps.Print := function( S )
    Print("GraphicLattice( ", S.group, ", ", S.width, ", ", S.height, " )");
end;


#############################################################################
## 
#F  Select( <S>, <list> ) . . . . . . . . . . . . . . . . select given groups
##
Select := function( S, l )
    local   i,  u,  c,  pos,  nos, rep, trn;
    
    S.operations.DeselectAll(S);
    nos := [];
    for u  in l  do
        c := Filtered( [1..Length(S.reps)], x -> u in S.classes[x] );
        if 1 <> Length(c)  then
            Error( "<u> does not lie in the subgroup lattice" );
        fi;
        rep := S.classes[c[1]].representative;
        trn := RightTransversal( S.group, Normalizer( S.group, rep ) );
        pos := Position( S.classes[ c[1]].representative^trn[pos], u );
        i := Position( S.idents, [ c[1], pos ] );
        Add( nos, PositionProperty( S.maximals, x -> x[1] = i ) );
    od;
    for i  in nos  do
        S.operations.ToggleSelection( S, S.vertices[i] );
    od;
    return nos;
    
end;


#############################################################################
##
#F  GraphicLatticeOps.ShowInfo( <S>, <obj> )  . . . . . . .  show object info
##
GraphicLatticeOps.ShowInfo := function( S, obj )
   local   U,  i,  str,  info, rep, trn;
    
   # destroy other text selectors flying around
   if IsBound(S.selector)  then
       Close(S.selector);
   fi;
   
   # get the group of <obj>
   i := S.idents[S.maximals[Position( S.vertices, obj )][1]];
  
   rep := S.classes[i[1]].representative;
   trn := RightTransversal( S.group, Normalizer( S.group, rep ) );
   U   := S.classes[i[1]].representative^trn[i[2]];

   # construct info texts
   info := [];
   
   # size
   str := String( "Size:", -14 );
   if IsBound(U.size)  then
       Append( str, String(U.size) );
   else
       Append( str, "unknown" );
   fi;
   Add( info, str );
   
   # index
   str := String( "Index:", -14 );
   if IsParent(U) and IsBound(U.index)  then
       Append( str, String(U.index) );
   else
       Append( str, "unknown" );
   fi;
   Add( info, str );
              
   # is abelian
   str := String( "IsAbelian:", -14 );
   if IsBound(U.isAbelian) then 
       Append( str, String(U.isAbelian) );
   else
       Append( str, "unknown" );
   fi; 
   Add( info, str );

   # is central
   str := String( "IsCentral:", -14 );
   if IsParent(U) and IsBound(U.isCentral) then 
       Append( str, String(U.isCentral) );
   else
       Append( str, "unknown" );
   fi; 
   Add( info, str );

   # is cyclic
   str := String( "IsCyclic:", -14 );
   if IsBound(U.isCyclic) then 
       Append( str, String(U.isCyclic) );
   else
       Append( str, "unknown" );
   fi; 
   Add( info, str );

   # is nilpotent
   str := String( "IsNilpotent:", -14 );
   if IsBound(U.isNilpotent) then 
       Append( str, String(U.isNilpotent) );
   else
       Append( str, "unknown" );
   fi; 
   Add( info, str );

   # is normal
   str := String( "IsNormal:", -14 );
   if IsParent(U) and IsBound(U.isNormal) then 
       Append( str, String(U.isNormal) );
   else
       Append( str, "unknown" );
   fi; 
   Add( info, str );

   # is perfect
   str := String( "IsPerfect:", -14 );
   if IsBound(U.isPerfect) then 
       Append( str, String(U.isPerfect) );
   else
       Append( str, "unknown" );
   fi; 
   Add( info, str );

   # is simple
   str := String( "IsSimple:", -14 );
   if IsBound(U.isSimple) then 
       Append( str, String(U.isSimple) );
   else
       Append( str, "unknown" );
   fi; 
   Add( info, str );

   # is solvable
   str := String( "IsSolvable:", -14 );
   if IsBound(U.isSolvable) then 
       Append( str, String(U.isSolvable) );
   else
       Append( str, "unknown" );
   fi;
   Add( info, str );

   # construct text selector
   S.selector := TextSelector( Concatenation( " Information about ",
                         obj.label.text ), info, [ "compute", "close" ] );
   Disable( S.selector, "compute" );
               
   # enable "compute" if a text is selected
   S.selector.textSelected := function( sel, tid )
       if tid = 0  then
           Disable( sel, "compute" );
       else
           Enable( sel, "compute" );
       fi;
   end;
             
   # compute entries
   S.selector.buttonPressed := function( sel, bt )

       if bt = "compute"  then
           info := Copy(S.selector.labels);
           if sel.selected = 1  then
               str := String( "Size:", -14 );
               info[1] := Concatenation(str, String(Size(U)));
           elif sel.selected = 2  then
               str := String( "Index:", -14 );
               info[2] := Concatenation(str, String(Index(S.group,U)));
           elif sel.selected = 3  then
               str := String( "IsAbelian:", -14 );
               info[3] := Concatenation(str, String(IsAbelian(U)));
           elif sel.selected = 4  then
               str := String( "IsCentral:", -14 );
               info[4] := Concatenation( str, String(IsCentral(S.group, U)));
           elif sel.selected = 5  then
               str := String( "IsCyclic:", -14 );
               info[5] := Concatenation(str, String(IsCyclic(U)));
           elif sel.selected = 6  then
               str := String( "IsNilpotent:", -14 );
               info[6] := Concatenation(str, String(IsNilpotent(U)));
           elif sel.selected = 7 then 
               str := String( "IsNormal:", -14 );
               info[7] := Concatenation(str, String(IsNormal(S.group, U)));
           elif sel.selected = 8  then
               str := String( "IsPerfect:", -14 );
               info[8] := Concatenation(str, String(IsPerfect(U)));
           elif sel.selected = 9  then
               str := String( "IsSimple:", -14 );
               info[9] := Concatenation(str, String(IsSimple(U)));
           elif sel.selected = 10  then
               str := String( "IsSolvable:", -14 );
               info[10] := Concatenation(str, String(IsSolvable(U)));
           fi;
           Relabel( sel, info );
           Disable( sel, "compute" );
       else
           Close(sel);
           Unbind(S.selector);
       fi;
   end;
   
end;


#############################################################################
##
#F  GraphicLatticeOps.HMove( <S>, <obj> ) . . . . . . . . . . .  move classes
##
GraphicLatticeOps.HMove := function( S, obj )
    
    SetTitle( S, "Select Destination");
    S.objectWantsToMove := obj; 

    # install temporary move function
    S.rightPointerButtonDown := function( S, x, y )   
        local    obj,  i,  j,  l,  diff,  olddiff;   
                             
        obj     := S.objectWantsToMove;
        diff    := x - obj.x;
        olddiff := diff;
        for i  in S.strips[obj.strip]  do
            l := i;
            j := 0;
            if S.vertices[l].x   + olddiff < 0 then
                diff := S.width  + olddiff;
            elif S.vertices[l].x + olddiff > S.width then
                diff := olddiff  - S.width;
            else 
                diff := olddiff;
            fi;
            while j < S.classLengths[S.idents[S.maximals[i][1]][1]]  do  
                Move( S.vertices[l], 
                      S.vertices[l].x + diff, 
                      S.vertices[l].y );
                l := l+1;
                j := j+1;
            od;
        od;
        S.rightPointerButtonDown := S.operations.RightMenu;
        S.leftPointerButtonDown  := S.operations.DragVertex;
        SetTitle( S, S.defaultTitle );
    end;
    S.leftPointerButtonDown := S.rightPointerButtonDown;
    
end;


#############################################################################
##
#F  GraphicLatticeOps.LeftMargin( <S>, <x> )  . . . . . . . . set left margin
##
GraphicLatticeOps.LeftMargin := function( S, x )
    local   obj;
    
    # make sure that the window is not getting to small
    if x > S.width + 100  then
        return;
    fi;

    # move all vertices to the left
    WcFastUpdate( S.id, true );
    for obj  in S.vertices  do
        obj.operations.MoveDelta( obj, -x, 0 );
    od;
    WcFastUpdate( S.id, false );

    # resize window
    S.operations.Resize( S, S.width-x, S.height );
    
end;


#############################################################################
##
#F  GraphicLatticeOps.TopMargin( <S>, <y> ) . . . . . . . . .  set top margin
##
GraphicLatticeOps.TopMargin := function( S, y )
    local   obj;
    
    # make sure that the window is not getting to small
    if y > S.height + 100  then
        return;
    fi;

    # move all vertices to the left
    WcFastUpdate( S.id, true );
    for obj  in S.vertices  do
        obj.operations.MoveDelta( obj, 0, -y );
    od;
    WcFastUpdate( S.id, false );

    # resize window
    S.operations.Resize( S, S.width, S.height-y );
    
end;


#############################################################################
##
#F  GraphicLatticeOps.RightMargin( <S>, <x> ) . . . . . . .  set right margin
##
GraphicLatticeOps.RightMargin := function( S, x )
    local   obj;
    
    # make sure that the window is not getting to small
    if x < 100  then
        return;
    fi;

    # resize window
    S.operations.Resize( S, x, S.height );
    
end;


#############################################################################
##
#F  GraphicLatticeOps.BottonMargin( <S>, <y> )  . . . . . . set botton margin
##
GraphicLatticeOps.BottonMargin := function( S, y )
    local   obj;
    
    # make sure that the window is not getting to small
    if y < 100  then
        return;
    fi;

    # resize window
    S.operations.Resize( S, S.width, y );
    
end;


#############################################################################
##
#F  GraphicLatticeOps.DoubleSize( <S> )	. . . . . . . . .  double size of <S>
##
GraphicLatticeOps.DoubleSize := function( S )
    local   obj;
    
    # double window
    S.operations.Resize( S, 2*S.width, 2*S.height );

    # and move objects to new position
    for obj  in S.vertices  do
        Move( obj, 2*(obj.x+1)-1, 2*(obj.y+1)-1 );
    od;
    
end;


#############################################################################
##
#F  GraphicLatticeOps.DragVertex( <S>, <x>, <y> ) . . . drag a representative
##
GraphicLatticeOps.DragVertex := function( S, x, y )
    local  pos,  obj,  d1,  d2,  oldx,  diff,  i,  j;
    
    pos := [x,y];
    for obj in S.objects do
        if IsBound(obj.isVertex) and obj.isVertex and pos in obj  then 
            i := Position( S.vertices, obj );
            
            # if the object is not a representative return
            if not S.maximals[i][1] in S.reps  then
                return;
            fi;
            
            # compute the moveable strip
            if 1 = obj.strip  then
                d1 := S.height;
            else
                d1 := List(S.strips[obj.strip-1], v -> S.vertices[v].y);
                d1 := Minimum(d1) - S.circle;
            fi;
            if Length(S.yCoordinates) = obj.strip  then
                d2 := 0;
            else
                d2 := List(S.strips[obj.strip+1], v -> S.vertices[v].y);
                d2 := Maximum(d2) + S.circle;
            fi;
            
            # drag representative
            oldx := obj.x;
            WcFastUpdate( S.id, true );
            Drag( S, x, y, 1, function( x, y )
                if y >= d1  then y := d1;  fi;
                if y <= d2  then y := d2;  fi;
                Move( obj, x, y );
            end );
            WcFastUpdate( S.id, false );
            
            # move class to representative
            diff := oldx - obj.x;
            for j  in [1..S.classLengths[S.idents[S.maximals[i][1]][1]]-1] do
                Move( S.vertices[i+j], S.vertices[i+j].x - diff, obj.y );
            od;
        fi;
    od;
end;


#############################################################################
##
#F  GraphicLatticeOps.YLevel( <S> ) . . . . . . . . . .  common y coordinates
##
GraphicLatticeOps.YLevel := function( S )
    local   i,  j,  a, obj;
    
    for i  in [ 1 .. Length(S.strips) ]  do
        a := 0;
        for j  in  S.strips[i]  do
            a := a + S.vertices[j].y;
        od;
        a := QuoInt( a, Length(S.strips[i]) );
        S.yCoordinates[i] := a;
    od;
    for obj  in S.vertices  do
        Move( obj, obj.x, S.yCoordinates[obj.strip] );
    od;

end;


#############################################################################
##
#F  GraphicLatticeOps.XLevel( <S> ) . . . . . . . . . .  common x coordinates
##
GraphicLatticeOps.XLevel := function( S )
    local   x,  obj,  strips,  l,  j,  diff,  pos;
    
    x := 0;
    if 0 < Length(S.selected)  then
        strips := [];
        for obj  in S.selected  do
            l := S.maximals[Position(S.vertices, obj )][1];
            if l in S.reps and not obj.strip in strips  then
                x := x + obj.x;
                Add( strips, obj.strip );
            else
                return;
            fi;
        od;
        x := QuoInt( x, Length(S.selected) );
        for obj  in S.selected  do
            pos  := Position( S.vertices, obj );
            diff := x - obj.x;
            l    := obj.ident[1];
            for j  in [ 0 .. S.classLengths[l]-1 ]  do
                Move( S.vertices[pos+j],
                      S.vertices[pos+j].x + diff,
                      S.vertices[pos+j].y );
            od;
        od;
    fi;
    
end;


#############################################################################
##
#F  GraphicLatticeOps.SwapObjects( <S> )  . . . . . . . . . . .  swap objects
##
GraphicLatticeOps.SwapObjects := function( S )
    local   classes,  class,  obj,  k,  oldx,  i;
    
    classes := List( S.reps, x -> [] );
    for obj  in S.selected  do
        k := obj.ident[1];
        Add( classes[k], obj );
    od;
    for class  in classes do
        if 1 < Length(class)  then
            oldx := class[1].x;
            for i  in [ 2 .. Length(class) ]  do
                Move( class[i-1], class[i].x, class[i].y );
            od;
            Move( class[Length(class)], oldx, class[1].y );
        fi;
    od;

end;


#############################################################################
##
#F  GraphicLatticeOps.ToggleSelection( <S>, <obj> ) .  toggle status of <obj>
##
GraphicLatticeOps.ToggleSelection := function( S, obj )
    local   k,  obj;
    
    k := Position( S.vertices, obj );
    if obj in S.selected  then
        if not S.maximals[k][1] in S.reps  then
            Unhighlight(obj);
        fi;
        Reshape( obj, 1 );
        S.selected := Filtered( S.selected, x -> x <> obj );
    else
        Add( S.selected, obj );
        Reshape( obj, 3 );
        if S.maximals[k][1] in S.reps  then
            Highlight(obj);
        fi;
    fi;
    
end;

                          
#############################################################################
##
#F  GraphicLatticeOps.DeselectAll( <S> )  . . . . . .  deselected all objects
##
GraphicLatticeOps.DeselectAll := function( S )
    local   k,  obj;
    
    for obj  in S.selected  do
        k := Position( S.vertices, obj );
        if not S.maximals[k][1] in S.reps  then
            Unhighlight(obj);
        fi;
        Reshape( obj, 1 );
    od;
    S.selected := [];
    
end;

                          
#############################################################################
##
#F  GraphicLatticeOps.RightMenu( <S>, <x>, <y> )  . . .  right button pressed
##
GraphicLatticeOps.RightMenu := function( S, x, y )
    local   res, pos, obj, o1, o2, h, ListY, Y, nr, i, j, k, diff, m,
            rep, trn, U;
  
    pos := [x,y];
    for obj in S.objects do
        if IsBound(obj.isVertex) and obj.isVertex then 
            if pos in obj then
                res := Query( S.rightPopupMenu, S );
                if res = "H-Move"  then
                    S.operations.HMove( S, obj );
                elif res = "Swap"  then
                    S.operations.SwapObjects(S);
                elif res = "Deselect All"  then
                    S.operations.DeselectAll(S);
                elif res = "Show Group Id"  then
                    m := obj.ident;
                    rep := S.classes[m[1]].representative;
                    trn := RightTransversal( S.group, 
                           Normalizer( S.group, rep ) );
                    U := S.classes[m[1]].representative^trn[m[2]];
                    Print( "#I  ", obj.label.text, " = ", U, "\n" );
                elif res = "X-Level"  then
                    S.operations.XLevel(S);
                elif res = "Y-Level"  then
                    S.operations.YLevel(S);
                elif res = "Toggle Selection"  then
                    S.operations.ToggleSelection( S, obj );
                elif res = "Information"  then
                    S.operations.ShowInfo( S, obj );
                fi;
                return;
            fi;
        fi;
    od;
    res := Query( S.rightPopupMenu2, S );
    if res = "Y-Level"  then
        S.operations.YLevel(S);
    elif res = "Deselect All"  then
        S.operations.DeselectAll(S);
    elif res = "Swap"  then
        S.operations.SwapObjects(S);
    elif res = "X-Level"  then
        S.operations.XLevel(S);
    elif res = "Left Margin"  then
	S.operations.LeftMargin( S, x );
    elif res = "Right Margin"  then
	S.operations.RightMargin( S, x );
    elif res = "Botton Margin"  then
	S.operations.BottonMargin( S, y );
    elif res = "Top Margin"  then
	S.operations.TopMargin( S, y );
    elif res = "Double Size"  then
        S.operations.DoubleSize(S);
    fi;
end;


#############################################################################
##
#F  GraphicLatticeOps.SortMaximals( <S> ) . . . . . . . . . sort the maximals
##
GraphicLatticeOps.SortMaximals := function( S )
    
    local   i,  j,  k,  l,	# loops
            minimals,           # number of minimal subgroups
            X1, X2,             # temps
            ord,                # order/class length
            max,                # new maximals
            L,                  # sorted connections (size)
            S1;                 # sorted connections (middle)
    
    # <S>.minimals contains the number of minimal subgroups
    minimals := [];
    for i  in [ 1 .. Length(S.reps) ]  do
        minimals[i] := [ S.maximals[S.reps[i]][2], 0 ];
        for j  in [ 1 .. Length(S.maximals) ]  do
            if S.reps[i] in S.maximals[j][3]  then
                minimals[i][2] := minimals[i][2] + 1;
            fi;
        od;
    od;
 
    # sort according to number of connections
    i := 2;
    L := [];
    while i < Length(minimals)  do
        X1  := [];
        X2  := [];
        ord := minimals[i][1];
        Add( X1, minimals[i][2] );
        Add( X2, S.reps[i] );
        i := i + 1;
        while minimals[i][1] = ord  do
            Add( X1, minimals[i][2] );
            Add( X2, S.reps[i] );
            i := i + 1;
        od;
        SortParallel(X1, X2);
        Add( L, X2 );
    od;

    # put the vertices with many connections nearer to the middle
    S1 := List( L, x -> [] );
    for i  in [ 1 .. Length(L) ]  do
        k := 0;
        l := 0;
        for j in  [ 1 .. Length(L[i]) ]  do
            if j mod 2 = 1  then
                S1[i][Length(L[i])-k] := L[i][j];
                k := k + 1;
            else
                S1[i][l+1] := L[i][j];
                l := l + 1;
            fi;
        od;      
    od;
    
    # sort <S>.maximals according to <S1>
    max := [S.maximals[1]];
    for i  in [ 1 .. Length(S1) ]  do
        for j  in [ 1 .. Length(S1[i]) ]  do
            ord := S.classLengths[S.idents[S.maximals[S1[i][j]][1]][1]];
            for k  in [ 1 .. ord ]  do
                Add( max, S.maximals[S1[i][j]+k-1] );
            od;
        od;
    od;
    Add( max, S.maximals[Length(S.maximals)] );
    S.maximals := max;
     
end;


#############################################################################
##
#F  GraphicLatticeOps.MakeVertices( <S> ) . . . . . . create list of vertices
##
GraphicLatticeOps.MakeVertices := function( S )
    local   i,  j,  k,  v;
    
    S.vertices := [];
    S.strips   := List( S.yCoordinates, x -> [] );
    for j in [ 1 .. Length(S.yCoordinates) ] do
        for i in [ 1 .. Length(S.maximals) ]  do
            if  S.maximals[i][2] = S.orders[j] then
                v := Vertex( S, S.xCoordinates[i], S.yCoordinates[j] );
                if S.prime  then 
                    k := Position( S.primeorders, S.orders[j] );
                else 
                    k := j;
                fi;
                v.ident := S.idents[S.maximals[i][1]];
                v.strip := k;
                Relabel( v, String(i) ); 
                if S.maximals[i][1] in S.reps  then
                    Add( S.strips[k], i );
                    Highlight(v);
                fi;
                S.vertices[i] := v;
            fi;
        od;
    od;

end;


#############################################################################
##
#F  GraphicLatticeOps.MakeConnections( <S> )  . . . . . .  create connections
##
GraphicLatticeOps.MakeConnections := function( S )
    local   i,  k,  j;  
    
    for i  in [ 1 .. Length(S.maximals) ]  do
        for j  in [ 1 .. Length(S.maximals[i][3]) ]  do
            k := S.maximals[i][3][j];
            k := PositionProperty( S.maximals, x -> x[1] = k );
            Connection( S.vertices[i], S.vertices[k] );
        od;
    od;
end;


###############################################################################
## 
#F  GraphicLatticeOps.MakeXFirstClass( <S>, <coor>, <n> ) . coors of 1. class
##
GraphicLatticeOps.MakeXFirstClass := function( S, coor, n )
    local  i,  j,  max,  x;
    
    i := PositionProperty( [1..Length(S.maximals)], 
                           x -> S.maximals[x][2] = S.r[n][2][1] );
    max := S.classLengths[S.idents[S.maximals[i][1]][1]];
    if 1 < Length(S.b)  then
        x := coor[n] - Int(max/2) * S.circle; 
    else
        x := coor[n];
    fi;
    j := 0;
    while j < max do
        if S.xCoordinates[i+j] = 0  then
            S.xCoordinates[i+j] := x + S.circle*j;
        fi;
        j := j+1;
    od;
end;
    

#############################################################################
##
#F  GraphicLatticeOps.MakeX( <S> )  . . . . . . . . . . .  make x coordinates
##
GraphicLatticeOps.MakeX := function( S )
    
    local   i,  j,  k,  l,	# loop variable
            ri,                 # local rows
            cl,                 # class length
            coor, coors,        # local x coordinate list
            x,                  # x coordinate
            n,                  # number of branches
            pos1,  pos2,        # positions
            tmp;
    
    # compute the layers <S>.l
    S.l := [];
    for i  in [ 1 .. Length(FactorsInt(Size(S.group))) ]  do
        S.l[i] := [];
    od;
    for i  in [ 2 .. Length(S.orderReps)-1 ]  do
        AddSet( S.l[Length(FactorsInt(S.orderReps[i])) ], S.orderReps[i] );
    od;
    S.l := Filtered( S.l, i -> 0 < Length(i) ); 
    
    # compute the branches <S>.b
    S.b := [];
    tmp := Maximum(List(S.l, x->Length(x)));
    for i in [ 1..tmp]  do
        S.b[i] := [];
        for j in [1..Length(S.l)] do
             if IsBound(S.l[j][i]) then
                 Add(S.b[i], S.l[j][i]);
             fi;
        od;
    od;
    S.b := Filtered( S.b, i -> 0 < Length(i) );
    
    # compute the row <S>.r of <S>.b and <S>.l
    S.r := [];
    ri  := [];
    for i  in [ 1 .. Length(S.l) ]  do
        for j  in [ 1 .. Length(S.b) ]  do
            tmp := Intersection( S.l[i], S.b[j] );
            if 0 < Length(tmp) then
                if Length(tmp) = 1 then
                    Add( ri, tmp );
                    Add( S.r, [ [i,j], tmp ] );
                else
                    for k  in [ 1 .. Length(tmp) ]  do
                        Add( ri, [tmp[k]] );
                        Add( S.r, [ [i,j], [tmp[k]] ] );
                    od;
                fi;
            fi;
        od;
    od;   
  
    # compute the x coordinates
    S.xCoordinates := List( S.maximals, x -> 0 );
    S.xCoordinates[1]                  := QuoInt( S.width, 2 );
    S.xCoordinates[Length(S.maximals)] := QuoInt( S.width, 2 );
   
    # divide x axis
    n := Length(S.b);
    if n = 2  then
        coors := [ QuoInt(S.width,3), 2*QuoInt(S.width,3) ];
    else
        coors := [ QuoInt(S.width,2) ];
        for j  in [ 1 .. QuoInt(n+1,2)-1 ]  do
            Add( coors, QuoInt( (2*j-1)*S.width, 2*n ) );
        od;
        for j  in [ QuoInt(n+1,2)+1 .. n ]  do
            Add( coors, QuoInt( (2*j-1)*S.width, 2*n ) );
        od;
    fi;
    coor := [];
    for i  in [ 1 .. Length(S.b) ]  do
        for j in [ 1 .. Length(S.b[i]) ]  do
            pos1 := S.noElements[ Position( S.orders, S.b[i][j]) ];
            pos2 := S.noClasses [ Position( S.orders, S.b[i][j]) ][1];
            if     1 = Length(coors)
               and 1 < pos1
               and 2*S.circle*pos1 < S.width
            then
                coor[ Position(ri,[S.b[i][j]]) ] :=
                  QuoInt(S.width,2) - QuoInt( pos1+pos2-1, 2 ) * S.circle;
            elif   1 = Length(coors)
               and 1 < pos1
               and pos1 > S.width/(2*S.circle) 
            then
                coor[ Position(ri,[S.b[i][j]]) ] :=
                  QuoInt(S.width,2) - QuoInt(pos1,2) * S.circle;
                
            elif 1 = pos1
               and j in [ 2..Length(S.b[i])-1 ]
               and S.noElements[ Position( S.orders, S.b[i][j+1]) ] =  1 
            then
                tmp := QuoInt(Maximum(S.noElements) + 1, 2);
                coor[ Position(ri,[S.b[i][j]]) ] := 
                  coors[i] + (-1)^(j+1) * tmp * S.circle;
            else
                coor[ Position(ri,[S.b[i][j]]) ] := coors[i];
            fi;
        od;
    od;
    for i  in [ 1.. Length(S.r) ]  do
        S.operations.MakeXFirstClass( S, coor, i );
    od;
    
    
    # set x coordinates of all other elements
    i := 1;
    while i < Length(S.maximals)  do
        if S.xCoordinates[i] = 0  then
            if     S.orderElements[S.maximals[i-1][1]]
                   <= S.orderElements[S.maximals[i][1]] 
               and S.maximals[i][1] in S.reps 
            then
                j := 1;
                while S.noClasses[j][2]<>S.orderElements[S.maximals[i][1]]  do
                    j := j + 1;
                od;
                if 2 * S.noElements[j] * S.circle > S.width  then
                    S.xCoordinates[i] := S.xCoordinates[i-1] + S.circle;  
                else
                    S.xCoordinates[i] := S.xCoordinates[i-1] + 2*S.circle;
                fi;
                if     S.xCoordinates[i] > S.width
                   and S.maximals[i][1] in S.reps
                then
                    S.xCoordinates[i] := S.circle;
                fi;
                cl := S.classLengths[S.idents[S.maximals[i][1]][1]];
                i  := i + 1;
                j  := 1;
                x  := S.xCoordinates[i-1]; 
                while j < cl  do
                    S.xCoordinates[i] := x + S.circle * j;
                    j := j + 1;
                    i := i + 1;
                od;
            else
                Error();
            fi;
        else
            i := i + 1;
        fi;
    od;
end;    
    

#############################################################################
##
#F  GraphicLatticeOps.MakeY( <S> )  . . . . . . . . . . .  make y-coordinates
##
GraphicLatticeOps.MakeY := function( S ) 
    local   L,  M,  i, pos, oldCoor;
       
    L := Length(S.orders);
    M := QuoInt(S.height-2*S.circle,L);
    S.yCoordinates := Reversed( Set( List( [1..L], i -> M*i ) ) );
    if S.prime  then
        oldCoor := ShallowCopy( S.yCoordinates );
        for i in [1..Length(S.primeorders)] do
            pos := Position( S.orders, S.primeorders[i] );
            S.yCoordinates[i] := oldCoor[pos];
        od;
    fi;
end;

    
#############################################################################
##
#F  GraphicLatticeOps.MakeMaximalSubgroups( <S> ) . . . . . . . . . . . local
##
GraphicLatticeOps.MakeMaximalSubgroups := function ( S )
    local   maxs,               # maximals (result)
            lat,                # lattice
            rel,                # maximal subgroup relation (for reps)
            sums,               # accumulated class lengths
            rep,                # representative of a class
            trn,                # transversal of normalizer of <rep>
            rep2,               # other representative
            nrm2,               # its normalizer
            trn2,               # its transversal
            max,                # maximal subgroups of <rep>
            i, k, l;            # loop variables

    # get the maximals relation
    lat := S.lattice;
    rel := lat.operations.MaximalSubgroups( lat );

    # mapping from pairs to numbers    
    sums := [ 0 ];
    for i  in [2..Length(lat.classes)]  do
        sums[i] := sums[i-1] + Size( lat.classes[i-1] );
    od;

    # loop over the conjugacy classes
    maxs := [];
    for i  in [1..Length(lat.classes)]  do

        # get the representative
        rep := lat.classes[i].representative;

        # enter the representative itself
        maxs[sums[i]+1] := [];
        maxs[sums[i]+1][1] := sums[i]+1;
        maxs[sums[i]+1][2] := Size( rep );
        maxs[sums[i]+1][3] := Set(List(rel[i],pair->sums[pair[1]]+pair[2]));

        # loop over the conjugates
        trn := RightTransversal( lat.group, Normalizer( lat.group, rep ) );
        for k  in [2..Length(trn)]  do

            # enter the conjugate
            maxs[sums[i]+k] := [];
            maxs[sums[i]+k][1] := sums[i]+k;
            maxs[sums[i]+k][2] := Size( rep );
            maxs[sums[i]+k][3] := [];

            # find the conjugated maximals
            for max  in rel[i]  do
                rep2 := lat.classes[max[1]].representative;
                nrm2 := Normalizer( lat.group, rep2 );
                trn2 := RightTransversal( lat.group, nrm2 );
                l := 1;
                while not trn2[max[2]] * trn[k] / trn2[l] in nrm2  do
                    l := l + 1;
                od;
                AddSet( maxs[sums[i]+k][3], sums[max[1]]+l );
            od;

        od;

    od;

    # assign the result
    S.maximals := maxs;
end;
 
 
#############################################################################
##
#F  GraphicLatticeOps.MakeLattice( <S>, <G> )  . . . . . compute full lattice
##
GraphicLatticeOps.MakeLattice := function( S, G )
    S.lattice := Lattice(G);
    return S.lattice.classes;
end;


#############################################################################
##
#V  NormalGraphicLatticeOps . . . . . . . . . operations for normal subgroups
##
NormalGraphicLatticeOps := Copy(GraphicLatticeOps);
NormalGraphicLatticeOps.name := "NormalGraphicLatticeOps";


#############################################################################
##
#F  NormalGraphicLatticeOps.MakeLattice( <S>, <G> ) . normal subgroup lattice
##
NormalGraphicLatticeOps.MakeLattice := function( S, G )
    return List( NormalSubgroups(G),
                 x -> ConjugacyClassSubgroups( G, x ) );
end;


#############################################################################
##
#F  NormalGraphicLatticeOps.MakeMaximalSubgroups( <S> )	. . . . . . . . local
##
NormalGraphicLatticeOps.MakeMaximalSubgroups := function( S )
    local   maximals,  inclusions,  i,  j;
    
    # compute the maximal subgroups
    maximals   := List( S.classes, x -> [] );
    inclusions := List( S.classes, x -> [] );
    for i  in [ 1 .. Length(S.classes) ]  do
        for j  in [ i, i-1 .. 1 ]  do
            if not j in inclusions[i]  then
                if IsSubgroup( Representative(S.classes[i]),
                               Representative(S.classes[j]) )
                then
                    AddSet( maximals[i], j );
                    AddSet( inclusions[i], j );
                    UniteSet( inclusions[i], inclusions[j] );
                fi;
            fi;
        od;
    od;
    S.maximals := List( [ 1 .. Length(maximals) ],
                        x -> [ x, Size(Representative(S.classes[x])),
                               maximals[x] ] );
end;


#############################################################################
##

#F  GraphicLattice( <G>, <x>, <y> )  . . . . . . . . . . .  display a lattice
##
GraphicLattice := function( arg )

    if IsGroup(arg[1])  then
        return GroupOps.GraphicLattice(arg);
    elif IsList(arg[1])  then
        return GraphicLatticeList(arg);
    fi;

end;


#############################################################################
##
#F  GroupOps.GraphicLattice( <G>, <x>, <y> ) . . . display a subgroup lattice
##
GroupOps.GraphicLattice := function( arglist )
    local   G,  S,  i,  j,  x,  y,  tmp,  def,  match;
    
    # check the arguments
    if 4 < Length(arglist) or Length(arglist) < 1  then
        Error( "usage: GraphicLattice( <G>, <x>, <y> )" );
    fi;
    if not IsGroup(arglist[1])  then
        Error( "<G> must be a group" );
    fi;
    G := arglist[1];
    
    # match a substring
    match := function( a, b )
        return a{[1..Length(b)]} = b;
    end;
    
    # get size
    if Length(arglist) = 1  then
        x := 1000;
        y := 600;
        i := 2;
    elif Length(arglist) >= 3 and IsInt(arglist[2])  then
        if not IsInt(arglist[3])  then
            Error( "usage: GraphicLattice( <G>, <x>, <y> )" );
        fi;
        x := arglist[2];
        y := arglist[3];
        i := 4;
    else
        x := 1000;
        y := 600;
        i := 2;
    fi;
    
    # set defaults and parse args
    def := rec();
    def.prime  := false;
    def.normal := false;
    while i <= Length(arglist)  do
        if IsString(arglist[i])  then
            if match( "Prime", arglist[i] )  then
                def.prime := true;
            elif match( "NormalSubgroups", arglist[i] )  then
                def.normal := true;
            else
                Error( "Valid Options:\n",
                       "    \"Prime\"\n",
                       "    \"NormalSubgroups\"\n",
                       "Illegal Parameter: \"", arglist[i], "\"" );
            fi;
        else
            Error( "Illegal Parameter: ", arglist[i] );
        fi;
        i := i + 1;
    od;
    
    # open a graphic sheet
    if def.normal  then
        if IsBound(G.name)  then
            tmp := Concatenation( "Normal Subgroups of ", G.name );
        else
            tmp := "Normal Subgroups";
        fi;
    else
        if IsBound(G.name)  then
            tmp := Concatenation( "Subgroup Lattice of ", G.name );
        else
            tmp := "Subgroup Lattice";
        fi;
    fi;
    S := GraphicSheet( tmp, x, y );
    S.circle       := 2*QuoInt( TINY_FONT[3]+4*(TINY_FONT[1]+TINY_FONT[2])+5, 3 );
    S.defaultTitle := tmp;
    S.prime        := def.prime;
    
    # change ops for normal subgroup lattice
    if def.normal  then
        S.operations := NormalGraphicLatticeOps;
    else
        S.operations := GraphicLatticeOps;
    fi;

    # store information in <S>
    S.group := G;
      
    # compute the lattice of <G>
    SetTitle( S, "Computing Lattice" );
    S.classes := S.operations.MakeLattice( S, G );
    
    # compute the maximals
    SetTitle( S, "Computing Maximal Subgroups" );
    S.orderReps := List(S.classes, C->Size(Representative(C)));
    S.orders    := Set(S.orderReps);
    S.idents    := [];
    S.reps      := [];
    for i  in [ 1 .. Length(S.classes) ]  do
        Add( S.reps, Length(S.idents)+1 );
        for j  in [ 1 .. Size(S.classes[i]) ]  do
            Add( S.idents, [i,j] );
        od;
    od;
    S.operations.MakeMaximalSubgroups( S );
    S.noClasses     := List( S.orders, x -> 
                             [ Number(S.orderReps,y->y=x),x ] );
    S.noElements    := List( S.orders, x -> Number( S.idents,
                               y -> S.orderReps[y[1]] = x ) );
    S.orderElements := List( S.idents, x -> S.orderReps[x[1]] );
    S.classLengths  := List( S.classes, Size );
    
    # compute the x-coordinates
    SetTitle( S, "Computing Coordinates" );
    S.operations.SortMaximals(S);
    if Length( S.orderReps ) > 2 then
        S.operations.MakeX(S);
        # compute the primeorders
        S.primeorders := [ 1 ];
        for i in [1 .. Length(S.r)] do
            Add( S.primeorders, S.r[i][2][1] );
        od;
        Add( S.primeorders, Size( S.group ) );
    else
        S.xCoordinates := [ QuoInt(S.width, 2), QuoInt(S.width, 2) ];
    fi;
      
    # compute the y-coordinates
    S.operations.MakeY(S);
    
    # draw lattice
    SetTitle( S, "Drawing" );
    S.operations.MakeVertices(S);
    S.operations.MakeConnections(S);
    SetTitle( S, S.defaultTitle );

    # and unbind unused information
    Unbind(S.xCoordinates);
    
    # no objects are selected at first
    S.selected := [];
    
    # add pointer action to <S>
    S.leftPointerButtonDown  := S.operations.DragVertex;
    S.rightPointerButtonDown := S.operations.RightMenu;
    
    # create right popup menu
    S.rightPopupMenu := PopupMenu( "Chose", [
        "Toggle Selection",
        "Deselect All" ,
        "Y-Level", 
        "X-Level",
        "================",
        "Show Group Id",
        "Information",
        "================",
        "Swap",
        "H-Move"
    ] );
    S.rightPopupMenu2 := PopupMenu( "Chose", [
        "Deselect All",
        "Y-Level",
        "X-Level",
        "Swap",
        "============",
        "Left Margin",
        "Right Margin",
        "Top Margin",
        "Botton Margin",
        "Double Size"
    ] );
    return S;
end;


#############################################################################
##
#V  GraphicLatticeListOps . . . . . . . . . . operations for maximals lattice
##
GraphicLatticeListOps := Copy(GraphicLatticeOps);
GraphicLatticeListOps.name := "GraphicLatticeListOps";


#############################################################################
##
#F  GraphicLatticeListOps.MakeX( <S> ) 	. . . . . . . . .  make x coordinates
##
GraphicLatticeListOps.MakeX := function( S )
    
    local   i,  j,  k,   	# loop variable
            diff;
 
    S.xCoordinates := [];
    k := 1;
    for i in [ 1 .. Length(S.orders) ] do
        diff := QuoInt( S.width, S.noElements[i] );
        S.xCoordinates[k] := QuoInt(S.width, 2) 
                             - QuoInt(S.noElements[i], 2 ) * diff;
        k := k + 1;
        for j in [ 2 .. S.noElements[i] ] do
            S.xCoordinates[k] := S.xCoordinates[k-1] + diff;
            k := k + 1;
        od;
    od;

end;


#############################################################################
##
#F  GraphicLatticeListOps.MakeY( <S> )  . . . . . . . . .  make y-coordinates
##
GraphicLatticeListOps.MakeY := function( S ) 
    local   L,  M;
       
    L := Length(S.orders);
    M := QuoInt(S.height-2*S.circle,L);
    S.yCoordinates := Reversed( Set( List( [1..L], i -> M*i ) ) );
end;


#############################################################################
##
#F  GraphicLatticeListOps.Print( <S> )  . . . . . . . . . . . pretty printing
##
GraphicLatticeListOps.Print := function( S )
    Print("GraphicLattice(",S.maximals,", ",S.width,", ",S.height," )");
end;


#############################################################################
##
#F  GraphicLatticeList( <L>, <x>, <y> )	. display a lattice given by maximals
##
GraphicLatticeList := function( arglist )
    local   L,  S,  i,  j,  x,  y,  tmp,  def,  match;
    
    # check the arguments
    if 4 < Length(arglist) or Length(arglist) < 1  then
        Error( "usage: GrLattice( <L>, <x>, <y> )" );
    fi;
    L := arglist[1];
    
    
    # match a substring
    match := function( a, b )
        return a{[1..Length(b)]} = b;
    end;
    
    # get size
    if Length(arglist) = 1  then
        x := 1000;
        y := 600;
    elif Length(arglist) >= 3 and IsInt(arglist[2])  then
        if not IsInt(arglist[3])  then
            Error( "usage: GrLattice( <L>, <x>, <y> )" );
        fi;
        x := arglist[2];
        y := arglist[3];
    else
        x := 1000;
        y := 600;
    fi;
    
    
    # open a graphic sheet
    S              := GraphicSheet( "Lattice", x, y );
    S.circle       := 25;
    S.defaultTitle := S.title;
    S.prime        := false;    
    S.maximals     := L;
    S.operations   := GraphicLatticeListOps;
      
    # compute the lattice of <G>
    SetTitle( S, "Computing Lattice" );
    
    # compute the orders
    S.orders := Set(List( S.maximals, x -> x[2] ));

    # compute the number of elements of one order
    S.noElements := List( S.orders, x -> Number( S.maximals, 
                                    y -> y[2] = x ) );
    S.reps         := List( L, x -> x[1] );
    S.classLengths := List( L, x -> 1 );
    S.idents       := List( L, x -> [ x[1], 1 ] );
    
    # compute the coordinates
    SetTitle( S, "Computing Coordinates" );
    S.operations.SortMaximals(S);
    S.operations.MakeX(S);
    S.operations.MakeY(S);
    
    # draw lattice
    SetTitle( S, "Drawing" );
    S.operations.MakeVertices(S);
    S.operations.MakeConnections(S);
    SetTitle( S, S.defaultTitle );

    # and unbind unused information
    Unbind(S.xCoordinates);
    
    # no objects are selected at first
    S.selected := [];
    
    #   create right popup menu
    S.rightPopupMenu := PopupMenu( "Chose", [
        "Toggle Selection",
        "Deselect All" ,
        "Y-Level", 
    ] );
    
    S.rightPopupMenu2 := PopupMenu( "Chose", [
       "Deselect All" ,
       "Y-Level", 
    ] );
         
    #  add pointer action to <S>
    S.leftPointerButtonDown  := S.operations.DragVertex;
    S.rightPointerButtonDown := S.operations.RightMenu;
    
    return S;
end;
