/****************************************************************************
**
*A  polynom.c                    GAP source                      Frank Celler
**
*H  @(#)$Id: polynom.c,v 3.17 1993/10/15 09:09:50 martin Rel $
**
*Y  Copyright 1990-1992,  Lehrstuhl D fuer Mathematik,  RWTH Aachen,  Germany
**
*H  $Log: polynom.c,v $
*H  Revision 3.17  1993/10/15  09:09:50  martin
*H  added 'register' to circumvent a MPW C compiler bug
*H
*H  Revision 3.16  1993/06/02  11:37:13  fceller
*H  added 'ProductsCoeffsMod', renamed 'ReduceModCoeffs' to 'ReduceCoeffsMod'
*H
*H  Revision 3.15  1993/05/05  11:11:43  fceller
*H  replace a wrong 'TYPE' by 'XType'
*H
*H  Revision 3.14  1993/03/01  10:33:13  fceller
*H  adapted to new list concept
*H
*H  Revision 3.13  1993/02/11  17:08:55  fceller
*H  removed creation of T_VECTOR of length 4
*H
*H  Revision 3.12  1993/02/04  18:20:32  martin
*H  fixed 'PowerModVecInt' from passing wrong degree to 'MultiplyVector'
*H
*H  Revision 3.11  1993/02/04  10:51:10  martin
*H  changed to use 'plist' interface
*H
*H  Revision 3.10  1993/01/06  17:53:39  fceller
*H  removed creation of T_VECFFE of length 4
*H
*H  Revision 3.9  1992/11/25  11:28:50  fceller
*H  fixed a bug in 'UnifyFFE'
*H
*H  Revision 3.8  1992/11/16  17:49:35  fceller
*H  added 'ReduceCoeffs' and 'ShiftedCoeffs'
*H
*H  Revision 3.7  1992/08/20  10:56:59  fceller
*H  pleased GCC
*H
*H  Revision 3.6  1992/08/17  11:18:46  fceller
*H  fixed handling of empty lists in 'AddCoeffs'
*H
*H  Revision 3.5  1992/07/06  08:09:09  fceller
*H  fixed a typo
*H
*H  Revision 3.4  1992/07/02  11:06:19  fceller
*H  add 'AddCoeffs'
*H
*H  Revision 3.3  1992/07/02  10:09:00  fceller
*H  add 'SumCoeffs'
*H
*H  Revision 3.2  1992/05/25  08:56:08  fceller
*H  Initial GAP 3.2 release
*/
#include        "system.h"              /* system dependent functions      */
#include        "gasman.h"              /* dynamic storage manager         */
#include        "scanner.h"             /* reading of tokens and printing  */
#include        "eval.h"                /* evaluator main dispatcher       */
#include        "integer.h"             /* arbitrary size integers         */
#include        "finfield.h"            /* finite field package            */
#include        "list.h"                /* generic list package            */
#include        "plist.h"               /* plain list package              */
#include        "vector.h"              /* vector package                  */
#include        "vecffe.h"              /* finite field vector package     */

#include        "polynom.h"             /* polynomial package              */


/****************************************************************************
**
*F  UnifiedFieldVecFFE( <hdL>, <hdR> )	. . . unify fields of <hdL> and <hdR>
**
**  Convert two finite field vectors into finite field vectors over  the same
**  finite field.  Signal an error if this conversion fails.
*/
TypHandle	UnifiedFieldVecFFE ( hdL,  hdR )
    TypHandle	    hdL;	        /* first finite field vector       */
    TypHandle       hdR;                /* second finite field vector      */
{
    TypHandle           hdFld;          /* handle of the field             */
    unsigned long       p;              /* characteristic                  */
    unsigned long       q;              /* size of common finite field     */
    unsigned long       q1;             /* size of field of row            */
    unsigned long       dl;             /* degree of <hdL>                 */
    unsigned long       dr;             /* degree of <hdR>                 */
    TypHandle           hdElm;          /* one row of the list             */
    TypFFE              v;              /* value of one element            */
    unsigned long       i, k;           /* loop variables                  */

    /* if <hdL> and <hdR> have already the same field return               */
    if ( FLD_VECFFE(hdL) == FLD_VECFFE(hdR) )
	return FLD_VECFFE(hdL);

    /* check the we know a common superfield of <hdL> and <hdR>            */
    p = CharVecFFE(hdL);
    if ( p != CharVecFFE(hdR) )
	return Error( "vectors have different characteristic", 0L, 0L );

    /* compute degree of superfield                                        */
    dl = DegreeVecFFE(hdL);
    dr = DegreeVecFFE(hdR);
    for ( k = dl;  dl % dr != 0;  dl += k )  ;

    /* make sure we can handle this field                                  */
    if ( (  2 <= p && 17 <= dl ) || (   3 <= p && 11 <= dl )
      || (  5 <= p &&  7 <= dl ) || (   7 <= p &&  6 <= dl )
      || ( 11 <= p &&  5 <= dl ) || (  17 <= p &&  4 <= dl )
      || ( 41 <= p &&  3 <= dl ) || ( 257 <= p &&  2 <= dl ) )
    {
	return Error( "common superfield is too large", 0L, 0L );
    }

    /* get a field that contains all elements                              */
    for ( q = 1, k = 1;  k <= dl;  k++ )  q *= p;
    if ( (SIZE_FF(FLD_VECFFE(hdL))-1) % (q-1) == 0 )
	hdFld = FLD_VECFFE(hdL);
    else if ( (SIZE_FF(FLD_VECFFE(hdR))-1) % (q-1) == 0 )
	hdFld = FLD_VECFFE(hdR);
    else
	hdFld = FLD_FFE( RootFiniteField( q ) );
    q = SIZE_FF(hdFld);

    /* convert <hdL> and <hdR>                                             */
    for ( i = 0;  i <= 1;  i++ )
    {
	hdElm = ( i == 0 ) ? hdL : hdR;
	if ( FLD_VECFFE(hdElm) != hdFld )
	{
	    q1 = SIZE_FF( FLD_VECFFE(hdElm) );
	    for ( k = LEN_VECFFE(hdElm);  0 < k;  k-- )
	    {
		v = VAL_VECFFE( hdElm, k );
		SET_VAL_VECFFE( hdElm, k, v==0 ? v : (v-1)*(q-1)/(q1-1)+1 );
	    }
	    SET_FLD_VECFFE( hdElm, hdFld );
	}
    }

    /* and return the common                                               */
    return FLD_VECFFE(hdL);
}


/****************************************************************************
**
*F  NormalizeCoeffs( <hdC>, <c> ) . . . . . . . . . . resize and retype <hdC>
*/
TypHandle	NormalizeCoeffs ( hdC,  c )
    TypHandle	    hdC;
    long            c;
{

    /* if <c> is zero convert <hdC> into a plain list                      */
    if ( c == 0 )
    {
	Resize( hdC,  SIZE_PLEN_PLIST(0) );
	Retype( hdC,  T_LIST );
	SET_LEN_PLIST( hdC, c );
    }

    /* if <hdC> is a finite field vector, resize <hdC>                     */
    else if ( TYPE(hdC) == T_VECFFE )
	SET_LEN_VECFFE( hdC, c );

    /* otherwise it is a plain list,  set length                           */
    else
	SET_LEN_PLIST( hdC, c );

    /* return <hdC>                                                        */
    return hdC;
}


/****************************************************************************
**
*F  AddVecFFE( <hdL>, <l>, <hdR>, <r>, <m> )  . . . . . . ff polynom addition
**
**  Add <m>-times the (finite field) coefficient vector <hdR>  to the (finite
**  field) coefficient vector <hdL>.  The  degree plus one of  the  result is
**  returned. Note that we will enlarge <hdL> if necessary, but we will *not*
**  shrink the vector, if if the degree of the result is less than <l>.
*/
long		AddVecFFE ( hdL, l, hdR, r, m )
    TypHandle       hdL;        /* left polynomial coeffs                  */
    TypHandle       hdR;        /* right polynomial coeffs                 */
    long            l;          /* degree plus one of left polynomial      */
    long            r;          /* degree plus one of right polynomial     */
    TypFFE          m;          /* multiple of <hdR> to add                */
{
    TypFFE        * ptL;        /* coeffs vectors of left polynomial       */
    TypFFE        * ptR;        /* coeffs vectors of right polynomial      */
    TypFFE        * ptE;        /* end of coeffs vector of sum             */
    TypFFE        * f;          /* finite field                            */
    TypFFE          t;          /* temp for FFEs                           */
    long            i;          /* loop variables                          */

    /* if <m> is trivial there is nothing to add                           */
    if ( m == 0 )
	return l;

    /* get the common finite field                                         */
    f = (TypFFE*) PTR( UnifiedFieldVecFFE( hdL, hdR ) );

    /* if <l> is less than <r> enlarge <hdL>                               */
    if ( l < r )
    {
	Resize( hdL, SIZE_PLEN_VECFFE(r) );
	SET_VAL_VECFFE( hdL, l+1, 0 );
	l = r;
    }

    /* set up all the pointers                                             */
    ptL = (TypFFE*)( PTR(hdL) + 1 );
    ptR = (TypFFE*)( PTR(hdR) + 1 );
    ptE = ptR + r;

    /* add <hdR> to <hdL>                                                  */
    if ( m == 1 )
    {
	while ( ptR < ptE )
	{
	    *ptL = SUM_FF( *ptL, *ptR, f );
	    ptL++;
	    ptR++;
	}
    }
    else
    {
	while ( ptR < ptE )
	{
	    t = PROD_FF( *ptR, m, f );
	    *ptL = SUM_FF( *ptL, t, f );
	    ptL++;
	    ptR++;
	}
    }

    /* compute the degree of <hdL>                                         */
    ptL = ((TypFFE*)( PTR(hdL) + 1 )) + (l-1);
    for ( i = l;  0 < i;  i--, ptL-- )
	if ( *ptL != 0 )
	    break;
    return i;
}


/****************************************************************************
**
*F  AddVector( <hdL>, <l>, <hdR>, <r>, <m> )  . . . . . . .  polynom addition
**
**  Add  <m>-times  the coefficient vector  <hdR>  to  the coefficient vector
**  <hdL>.  The degree plus one  of  the  result is  returned.  Note that  we
**  will enlarge <hdL> if necessary, but we will *not* shrink the vector,  if
**  if the degree of the result is less than <l>.
*/
long		AddVector ( hdL, l, hdR, r, hdM )
    TypHandle       hdL;        /* left polynomial coeffs                  */
    TypHandle       hdR;        /* right polynomial coeffs                 */
    long            l;          /* degree plus one of left polynomial      */
    long            r;          /* degree plus one of right polynomial     */
    TypHandle       hdM;        /* multiple of <hdR> to add                */
{
    TypHandle       hdLL;       /* one element of <hdL>                    */
    TypHandle       hdRR;       /* one element of <hdR>                    */
    TypHandle       hdAA;       /* sum of <hdLL> and <hdRR>                */
    long            i;          /* loop variables                          */

    /* if <m> is trivial there is nothing to add                           */
    if ( hdM == INT_TO_HD(0) )
	return l;

    /* if <l> is less than <r> enlarge <hdL>                               */
    if ( l < r )
    {
	Resize( hdL, SIZE_PLEN_PLIST(r) );
	for ( i = l+1;  i <= r;  i++ )
            SET_ELM_PLIST( hdL, i, INT_TO_HD(0) );
	l = r;
    }

    /* add <hdR> to <hdL>                                                  */
    for ( i = r;  0 < i;  i-- )
    {
	hdLL = ELM_PLIST( hdL, i );
	hdRR = ELM_PLIST( hdR, i );
	hdRR = PROD( hdRR, hdM );
	hdAA = SUM( hdLL, hdRR );
        SET_ELM_PLIST( hdL, i, hdAA );
    }

    /* compute the degree of <hdL>                                         */
    for ( i = l;  0 < i;  i-- )
	if ( EQ(ELM_PLIST(hdL, i), INT_TO_HD(0)) == HdFalse )
	    break;
    return i;
}


/****************************************************************************
**
*F  FunSumCoeffs( <hdCall> )  . . . . . . . . . . . . .  internal 'SumCoeffs'
**
**  'FunSumCoeffs' implements 'SumCoeffs( <l>, <r> )'
*/
TypHandle       FunSumCoeffs ( hdCall )
    TypHandle       hdCall;
{
    TypHandle       hdL,  hdR;
    long            l,    r;

    /* check arguments                                                     */
    if ( SIZE(hdCall) != 3 * SIZE_HD )
        return Error("usage: SumCoeffs( <l>, <r> )", 0L, 0L);
    hdL = EVAL(PTR(hdCall)[1]);
    hdR = EVAL(PTR(hdCall)[2]);

    /* if <l> or <r> is the empty list return a copy of the other list     */
    if ( IS_LIST(hdL) && LEN_LIST(hdL) == 0 )
        return Copy(hdR);
    if ( IS_LIST(hdR) && LEN_LIST(hdR) == 0 )
        return Copy(hdL);

    /* product of two finite field vectors                                 */
    if ( XType(hdL) == T_VECFFE && XType(hdR) == T_VECFFE )
    {

        /* make sure both vectors do have a non zero leading coefficient   */
        l = LEN_VECFFE(hdL);
        r = LEN_VECFFE(hdR);
        while ( 0 < l && VAL_VECFFE(hdL,l) == 0 )  l--;
        while ( 0 < r && VAL_VECFFE(hdR,r) == 0 )  r--;

	/* 'AddVecFFE' does all the work                                   */
	hdL = Copy(hdL);
	r   = AddVecFFE( hdL, l, hdR, r, 1 );

	/* normalize <hdL> and return                                      */
	return NormalizeCoeffs( hdL, r );
    }

    /* product of two integer/cyclotomics vectors                          */
    else if ( XType(hdL) == T_VECTOR && XType(hdR) == T_VECTOR )
    {
        l = LEN_LIST(hdL);
        r = LEN_LIST(hdR);

        /* make sure both vectors do have a non zero leading coefficient   */
        while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )  l--;
        while ( 0 < r && EQ(ELM_PLIST(hdR,r), INT_TO_HD(0)) == HdTrue )  r--;

	/* 'AddVector' does all the work                                   */
	hdL = Copy(hdL);
	r   = AddVector( hdL, l, hdR, r, INT_TO_HD(1) );

	/* normalize <hdL> and return                                      */
	return NormalizeCoeffs( hdL, r );
    }
    return Error( "<l> and <r> must be vectors", 0L, 0L );
}


/****************************************************************************
**
*F  FunAddCoeffs( <hdCall> )  . . . . . . . . . . . . .  internal 'AddCoeffs'
**
**  'FunAddCoeffs' implements 'AddCoeffs( <l>, <r> )'
*/
TypHandle       FunAddCoeffs ( hdCall )
    TypHandle       hdCall;
{
    TypHandle       hdL,  hdR,  hdM,  hdE;
    long            l,    r;
    TypFFE          m;

    /* check arguments                                                     */
    if ( 4 * SIZE_HD < SIZE(hdCall) || SIZE(hdCall) < 3 * SIZE_HD )
        return Error("usage: AddCoeffs( <l>, <r> )", 0L, 0L);
    hdL = EVAL(PTR(hdCall)[1]);
    hdR = EVAL(PTR(hdCall)[2]);
    if ( 4 * SIZE_HD == SIZE(hdCall) )
	hdM = EVAL(PTR(hdCall)[3]);
    else
	hdM = INT_TO_HD(1);

    /* if <hdR> is trivial there is nothing to add                         */
    if ( IS_LIST(hdR) && LEN_LIST(hdR) == 0 )
    {
	if ( XType(hdL) == T_VECFFE )
	{
	    l = LEN_VECFFE(hdL);
	    while ( 0 < l && VAL_VECFFE(hdL,l) == 0 )
		l--;
	    NormalizeCoeffs( hdL, l );
	    return HdVoid;
	}
	else if ( XType(hdL) == T_VECTOR )
	{
	    l = LEN_PLIST(hdL);
	    while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )
		l--;
	    NormalizeCoeffs( hdL, l );
	    return HdVoid;
	}
	else if ( IS_LIST(hdL) && LEN_LIST(hdL) == 0 )
	    return HdVoid;
	else
	    return Error( "<l> must be a vector", 0L, 0L );
    }

    /* if <hdL> is trivial enlarge <hdL> to contain at least one entry     */
    if ( IS_LIST(hdL) && LEN_LIST(hdL) == 0 )
    {
	if ( XType(hdR) == T_VECFFE )
	{
	    Retype( hdL, T_VECFFE );
	    SET_LEN_VECFFE( hdL, 1 );
	    SET_FLD_VECFFE( hdL, FLD_VECFFE(hdR) );
	    SET_VAL_VECFFE( hdL, 1, 0 );
	}
	else if ( XType(hdR) == T_VECTOR )
	{
	    Resize( hdL, SIZE_PLEN_PLIST(1) );
	    SET_LEN_PLIST( hdL, 1 );
	    SET_ELM_PLIST( hdL, 1, INT_TO_HD(0) );
	}
	else
	    return Error( "<l> must be a vector", 0L, 0L );
    }

    /* product of two finite field vectors                                 */
    if ( XType(hdL) == T_VECFFE && XType(hdR) == T_VECFFE )
    {

	/* if <hdM> is an integer compute the multiply of Z(p)^0           */
	if ( TYPE(hdM)==T_INT || TYPE(hdM)==T_INTPOS || TYPE(hdM)==T_INTNEG )
	{
	    if ( INT_TO_HD(1) != hdM )
	    {
		hdE = NewBag( T_FFE, SIZE_HD+sizeof(TypFFE) );
		SET_FLD_FFE( hdE, UnifiedFieldVecFFE(hdL,hdR) );
		SET_VAL_FFE( hdE, 1 );
		hdE = PROD( hdE, hdM );
		m   = VAL_FFE(hdE);
	    }
	    else
		m = 1;
	}

	/* if <hdM> is a finite field element check the field              */
	else if ( TYPE(hdM) == T_FFE )
	{
	    if ( UnifiedFieldVecFFE(hdL,hdR) == FLD_FFE(hdM) )
		m = VAL_FFE(hdM);
	    else
	    {
		hdR = PROD( hdR, hdM );
		m   = 1;
	    }
	}
	else
	    return Error( "<m> must be a finite field element", 0L, 0L );

        /* make sure both vectors do have a non zero leading coefficient   */
        l = LEN_VECFFE(hdL);
        r = LEN_VECFFE(hdR);
        while ( 0 < l && VAL_VECFFE(hdL,l) == 0 )  l--;
        while ( 0 < r && VAL_VECFFE(hdR,r) == 0 )  r--;

	/* 'AddVecFFE' does all the work                                   */
	r = AddVecFFE( hdL, l, hdR, r, m );

	/* normalize <hdL> and return                                      */
	NormalizeCoeffs( hdL, r );
	return HdVoid;
    }

    /* product of two integer/cyclotomics vectors                          */
    else if ( XType(hdL) == T_VECTOR && XType(hdR) == T_VECTOR )
    {

        /* make sure both vectors do have a non zero leading coefficient   */
        l = LEN_LIST(hdL);
        r = LEN_LIST(hdR);
        while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )  l--;
        while ( 0 < r && EQ(ELM_PLIST(hdR,r), INT_TO_HD(0)) == HdTrue )  r--;

	/* 'AddVector' does all the work                                   */
	r = AddVector( hdL, l, hdR, r, hdM );

	/* normalize <hdL> and return                                      */
	NormalizeCoeffs( hdL, r );
	return HdVoid;
    }
    return Error( "<l> and <r> must have a common field", 0L, 0L );
}


/****************************************************************************
**
*F  MultiplyVecFFE( <hdP>, <hdL>, <l>, <hdR>, <r> ) . . . poly multiplication
**
**  Multiply   two (finite field)  coefficient vectors  <hdL>   and <hdR> and
**  store the  result in  <hdP>.  <l>  (<r>)  gives the   degree plus  one of
**  <hdL> (<hdR>), the  result  must have  degree  <l>+<r>-2 because   we are
**  working over a field. The degree plus one of the result is returned.
*/
long		MultiplyVecFFE ( hdP, hdL, l, hdR, r )
    TypHandle	    hdP;	/* space for result of <hdL> * <hdR>       */
    TypHandle       hdL;        /* left polynomial coeffs                  */
    TypHandle       hdR;        /* right polynomial coeffs                 */
    long            l;          /* degree plus one of left polynomial      */
    long            r;          /* degree plus one of right polynomial     */
{
    TypFFE        * ptL;        /* coeffs vectors of left polynomial       */
    TypFFE        * ptLL;       /* coeffs vectors of left polynomial       */
    TypFFE        * ptEnd;      /* end of <ptL>                            */
    TypFFE        * ptR;        /* coeffs vectors of right polynomial      */
    TypFFE        * ptP;        /* coeffs vectors of product               */
    TypFFE        * ptPP;       /* coeffs vectors of product               */
    TypFFE        * f;          /* finite field                            */
    TypFFE          t;          /* temp for FFEs                           */
    long            i;          /* loop variable                           */

    /* if the one of the polynomials is trivial return the zero poly       */
    if ( l == 0 || r == 0 )
	return 0;

    /* get a common field of <hdL> and <hdR>                               */
    f = (TypFFE*) PTR( UnifiedFieldVecFFE( hdL, hdR ) );

    /* set common field in <hdP>                                           */
    SET_FLD_VECFFE( hdP, FLD_VECFFE(hdL) );

    /* chose larger vector as multiplicator                                */
    if ( l < r )
    {
	ptL = (TypFFE*)( PTR(hdL) + 1 );
	ptR = (TypFFE*)( PTR(hdR) + 1 );
    }
    else
    {
	ptR = (TypFFE*)( PTR(hdL) + 1 );
	ptL = (TypFFE*)( PTR(hdR) + 1 );
	i = l;  l = r;  r = i;
    }

    /* clear <hdP>                                                         */
    ptP = (TypFFE*)( PTR(hdP) + 1 );
    for ( ptPP = ptP, ptEnd = ptP + (l+r-1);  ptPP < ptEnd;  ptPP++ )
	*ptPP = 0;

    /* add <hdL> to <hdP> multiplied by the elements of <hdR>              */
    ptEnd = ptL + l;
    for ( i = 0;  i < r;  i++, ptR++, ptP++ )
    {
	if ( *ptR != 0 )
	{
	    for ( ptLL = ptL, ptPP = ptP;  ptLL < ptEnd;  ptLL++, ptPP++ )
		if ( *ptLL != 0 )
		{
		    t = PROD_FF( *ptLL, *ptR, f );
		    *ptPP = SUM_FF( *ptPP, t, f );
		}
	}
    }
    return l+r-1;
}


/****************************************************************************
**
*F  MultiplyVector( <hdP>, <hdL>, <l>, <hdR>, <r> ) . . . poly multiplication
**
**  Multiply two   (rational,   cyclotomic)  coefficient vectors   <hdL>  and
**  <hdR>  and store the result  in <hdP>.  <l> (<r>)  gives the degree  plus
**  one  of   <hdL> (<hdR>),  the result  must  have degree <l>+<r>-2 because
**  we are working  over   a field. The  degree  plus  one  of  the result is
**  returned.
*/
long		MultiplyVector ( hdP, hdL, l, hdR, r )
    TypHandle	    hdP;	/* space for result of <hdL> * <hdR>       */
    TypHandle       hdL;        /* left polynomial coeffs                  */
    TypHandle       hdR;        /* right polynomial coeffs                 */
    long            l;          /* degree plus one of left polynomial      */
    long            r;          /* degree plus one of right polynomial     */
{
    TypHandle       hdLL;       /* one element of <hdL>                    */
    TypHandle       hdRR;       /* one element of <hdR>                    */
    TypHandle       hdPP;       /* one element of <hdP>                    */
    TypHandle       hdTT;       /* temp element                            */
    long            i,  u,  k;  /* loop variables                          */

    /* if the one of the polynomials is of degree -1 return the zero poly  */
    if ( l == 0 || r == 0 )
	return 0;

    /* fold the product                                                    */
    for ( i = l+r;  1 < i;  i-- )
    {
	hdPP = INT_TO_HD( 0 );
	u = ( i-1 < l ) ? i-1 : l;
	for ( k = ( i-r < 1 ) ? 1 : i-r;  k <= u;  k++ )
	{
	    hdLL = ELM_PLIST( hdL, k   );
	    hdRR = ELM_PLIST( hdR, i-k );
	    hdTT = PROD( hdLL, hdRR );
	    hdPP = SUM( hdPP, hdTT );
	}
	SET_ELM_PLIST( hdP, i-1, hdPP );
    }
    return l+r-1;
}


/****************************************************************************
**
*F  FunProductCoeffs( <hdCall> )  . . . . . . . . .  internal 'ProductCoeffs'
**
**  'FunProductCoeffs' implements 'ProductCoeffs( <l>, <r> )'
*/
TypHandle       FunProductCoeffs ( hdCall )
    TypHandle       hdCall;
{
    TypHandle       hdL,  hdR,  hdP;
    long            l,  r;

    /* check arguments                                                     */
    if ( SIZE( hdCall ) != 3 * SIZE_HD )
        return Error( "usage: ProductCoeffs( <l>, <r> )", 0L, 0L );
    hdL = EVAL( PTR( hdCall )[ 1 ] );
    hdR = EVAL( PTR( hdCall )[ 2 ] );


    /* if <l> or <r> is the empty list return a copy of this list          */
    if ( IS_LIST(hdL) && LEN_LIST(hdL)==0 )
        return Copy(hdL);
    if ( IS_LIST(hdR) && LEN_LIST(hdR)==0 )
        return Copy(hdR);

    /* product of two finite field vectors                                 */
    if ( XType(hdL) == T_VECFFE && XType(hdR) == T_VECFFE )
    {

        /* make sure both vectors do have a non zero leading coefficient   */
        l = LEN_VECFFE(hdL);
        r = LEN_VECFFE(hdR);
        while ( 0 < l && VAL_VECFFE(hdL,l) == 0 )  l--;
        while ( 0 < r && VAL_VECFFE(hdR,r) == 0 )  r--;

	/* if one term is trivial return the zero polynomial               */
        if ( l == 0 || r == 0 )
        {
            hdP = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
            SET_LEN_PLIST( hdP, 0 );
	    return hdP;
        }

	/* the product of poly of deg l-1 and r-1 has deg l+r-1            */
	hdP = NewBag( T_VECFFE, SIZE_PLEN_VECFFE(l+r-1) );
	SET_FLD_VECFFE( hdP, FLD_VECFFE(hdL) );

	/* 'MultiplyVecFFE' does all the work                              */
	r = MultiplyVecFFE( hdP, hdL, l, hdR, r );
	return NormalizeCoeffs( hdP, r );
    }

    /* product of two integer/cyclotomics vectors                          */
    else if ( XType(hdL) == T_VECTOR && XType(hdR) == T_VECTOR )
    {
        /* make sure both vectors do have a non zero leading coefficient   */
        l = LEN_LIST( hdL );
        r = LEN_LIST( hdR );
        while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )  l--;
        while ( 0 < r && EQ(ELM_PLIST(hdR,r), INT_TO_HD(0)) == HdTrue )  r--;

	/* if one term is trivial return the zero polynomial               */
        if ( l == 0 || r == 0 )
        {
            hdP = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
            SET_LEN_PLIST( hdP, 0 );
	    return hdP;
        }

	/* the product of poly of deg l-1 and r-1 has deg l+r-1            */
	hdP  = NewBag( T_LIST, SIZE_PLEN_PLIST(l+r-1) );

	/* 'MultiplyVector' does all the hard work                         */
	r = MultiplyVector( hdP, hdL, l, hdR, r );
        return NormalizeCoeffs( hdP, r );
    }
    return Error( "<l> and <r> must have a common field", 0L, 0L );
}


/****************************************************************************
**
*F  MultiplyVectorMod( <hdP>, <hdL>, <l>, <hdR>, <r>, <hdN> ) . . . .  polmul
**
**  Multiply two   (rational,   cyclotomic)  coefficient vectors   <hdL>  and
**  <hdR>  and store the result  in <hdP>.  <l> (<r>)  gives the degree  plus
**  one  of   <hdL> (<hdR>),  reduce the result mod <hdN>.
*/
long		MultiplyVectorMod ( hdP, hdL, l, hdR, r, hdN )
    TypHandle	    hdP;	/* space for result of <hdL> * <hdR>       */
    TypHandle       hdL;        /* left polynomial coeffs                  */
    TypHandle       hdR;        /* right polynomial coeffs                 */
    long            l;          /* degree plus one of left polynomial      */
    long            r;          /* degree plus one of right polynomial     */
    TypHandle       hdN;        /* modulus                                 */
{
    TypHandle       hdLL;       /* one element of <hdL>                    */
    TypHandle       hdRR;       /* one element of <hdR>                    */
    TypHandle       hdPP;       /* one element of <hdP>                    */
    TypHandle       hdTT;       /* temp element                            */
    TypHandle       hdQ;        /* <hdP> / 2                               */
    long            i,  u,  k;  /* loop variables                          */

    /* if the one of the polynomials is of degree -1 return the zero poly  */
    if ( l == 0 || r == 0 )
	return 0;

    /* if <hdN> is zero return                                             */
    if ( hdN == INT_TO_HD(0) )
	return 0;

    /* if <hdP> is negative reduce into <hdP>/2 and <hdP>                  */
    if ( LT( hdN, INT_TO_HD(0) ) == HdTrue )
    {
	hdN = DIFF( INT_TO_HD(0), hdN );
	hdQ = SUM( hdN, INT_TO_HD(1) );
	hdQ = QuoInt( hdQ, INT_TO_HD(2) );
	hdQ = DIFF( hdQ, INT_TO_HD(1) );
    }
    else
	hdQ = 0;

    /* fold the product                                                    */
    for ( i = l+r;  1 < i;  i-- )
    {
	hdPP = INT_TO_HD( 0 );
	u = ( i-1 < l ) ? i-1 : l;
	for ( k = ( i-r < 1 ) ? 1 : i-r;  k <= u;  k++ )
	{
	    hdLL = ELM_PLIST( hdL, k   );
	    hdRR = ELM_PLIST( hdR, i-k );
	    hdTT = PROD( hdLL, hdRR );
	    hdPP = SUM( hdPP, hdTT );
	}
	hdPP = MOD( hdPP, hdN );
	if ( hdQ != 0 && LT( hdQ, hdPP ) == HdTrue )
	    hdPP = DIFF( hdPP, hdN );
	SET_ELM_PLIST( hdP, i-1, hdPP );
    }
    for ( i = l+r-1;  0 < i;  i-- )
	if ( EQ( ELM_PLIST( hdP, i ), INT_TO_HD(0) ) == HdFalse )
	    break;
    return i;
}


/****************************************************************************
**
*F  FunProductCoeffsMod( <hdCall> ) . . . . . . . internal 'ProductCoeffsMod'
**
**  'FunProductCoeffsMod' implements 'ProductCoeffsMod( <l>, <r>, <p> )'
*/
TypHandle       FunProductCoeffsMod ( hdCall )
    TypHandle       hdCall;
{
    TypHandle       hdP,  hdL,  hdR,  hdN;
    long            l,    r;

    /* check arguments                                                     */
    if ( SIZE( hdCall ) != 4 * SIZE_HD )
        return Error( "usage: ProductCoeffsMod( <l>, <r>, <n> )", 0L, 0L );
    hdL = EVAL( PTR(hdCall)[1] );
    hdR = EVAL( PTR(hdCall)[2] );
    hdN = EVAL( PTR(hdCall)[3] );

    /* check <l> and <r>                                                   */
    if ( IS_LIST(hdL) && LEN_LIST(hdL) == 0 )
        l = 0;
    else if ( XType(hdL) != T_VECTOR )
        return Error( "<l> must be a vector over the rationals", 0L, 0L );
    else
	l = LEN_PLIST(hdL);
    if ( IS_LIST(hdR) && LEN_LIST(hdR) == 0 )
	r = 0;
    else if ( XType(hdR) != T_VECTOR )
        return Error( "<r> must be a vector over the rationals", 0L, 0L );
    else
	r = LEN_PLIST(hdR);

    /* make sure both vectors do have a non zero leading coefficient       */
    while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )  l--;
    while ( 0 < r && EQ(ELM_PLIST(hdR,r), INT_TO_HD(0)) == HdTrue )  r--;

    /* compute the product of <hdL> and <hdR> mod <hdN>                    */
    if ( l == 0 || r == 0 )
    {
	hdP = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
	SET_LEN_PLIST( hdP, 0 );
	l = 0;
    }
    else
    {
	hdP = NewBag( T_LIST, SIZE_PLEN_PLIST(l+r-1) );
	l = MultiplyVectorMod( hdP, hdL, l, hdR, r, hdN );
    }
    NormalizeCoeffs( hdP, l );
    return hdP;
}


/****************************************************************************
**
*F  ReduceVecFFE( <hdL>, <l>, <hdR>, <r> )  . . . . . reduce polynomial <hdL>
**
**  Reduce  the  polynomial  <hdL> in place  by  the   polynomial <hdR>.  <l>
**  (<r>)  must be  the degree  plus 1 of  <hdL> (<hdR>). The degree plus one
**  of the result is returned.
**
**  As  this function should be as fast as possible  we are  *not*  using the
**  macros defined in "finfield.h".
*/
long		ReduceVecFFE ( hdL, l, hdR, r )
    TypHandle       hdL;        /* left polynomial coeffs                  */
    TypHandle       hdR;        /* right polynomial coeffs                 */
    long            l;          /* degree plus one of left polynomial      */
    long            r;          /* degree plus one of right polynomial     */
{
    TypFFE        * ptL;        /* coeffs vectors of left polynomial       */
    TypFFE        * ptLL;       /* coeffs vectors of left polynomial       */
    TypFFE        * ptR;        /* coeffs vectors of right polynomial      */
    TypFFE        * ptRR;       /* coeffs vectors of right polynomial      */
    TypFFE        * f;          /* the finite field                        */
    TypFFE          o;          /* size of <f> minus one                   */
    TypFFE          c,  q,  t;  /* temps for FFEs                          */
    register long   i,  k;      /* loop variables                          */

    /* <r> must be none zero                                               */
    if ( r == 0 )
	Error( "<r> must be non zero", 0L, 0L );

    /* if <l> is trivial return                                            */
    if ( l == 0 )
	return 0;

    /* get a common field                                                  */
    f = (TypFFE*) PTR( UnifiedFieldVecFFE( hdL, hdR ) );
    o = *f;

    /* set up all the pointers                                             */
    ptL = (TypFFE*)( PTR(hdL) + 1 );
    ptR = (TypFFE*)( PTR(hdR) + 1 ) + (r-1);

    /* get the leading coefficient of <hdR> * -1, <c> is never zero        */
    c = ( o%2 == 1 ? (*ptR) : ( (*ptR) <= o/2 ? (*ptR)+o/2 : (*ptR)-o/2 ) );

    /* compute the remainder                                               */
    for ( i = l-r;  0 <= i;  i-- )
    {

	/* if <ptLL> already ends with a zero, continue                    */
	ptLL = ptL + (i-1+r);
        if ( *ptLL == 0 )
	    continue;

        /* compute quotient of leading coefficients                        */
	q = ( c <= (*ptLL) ? (*ptLL)-c+1 : o-c+1+(*ptLL) );

        /* reduce <ptLL> by <q> * <ptRR>                                   */
	for ( k = r, ptRR = ptR;  0 < k;  k--, ptLL--, ptRR-- )
	{
	    if ( *ptRR == 0 )
		continue;
	    t = ( q-1 <= o-(*ptRR) ) ? q-1+(*ptRR) : q-1-(o-(*ptRR));
	    if ( *ptLL == 0 )
		*ptLL = t;
	    else if ( *ptLL <= t )
		*ptLL = PROD_FF( *ptLL, f[t-(*ptLL)+1], f );
	    else
		*ptLL = PROD_FF( t, f[(*ptLL)-t+1], f );
	}
    }

    /* compute new length                                                  */
    ptLL = ptL + l-1;
    for ( i = l;  0 < i;  i--, ptLL-- )
	if ( *ptLL != 0 )
	    break;
    return i;
}



/****************************************************************************
**
*F  ReduceVector( <hdL>, <l>, <hdR>, <r> )  . . . . . reduce polynomial <hdL>
**
**  Reduce  the  polynomial  <hdL> in place  by  the   polynomial <hdR>.  <l>
**  (<r>)  must be  the degree  plus 1 of  <hdL> (<hdR>). The degree plus one
**  of the result is returned.
*/
long		ReduceVector ( hdL, l, hdR, r )
    TypHandle       hdL;        /* left polynomial coeffs                  */
    TypHandle       hdR;        /* right polynomial coeffs                 */
    long            l;          /* degree plus one of left polynomial      */
    long            r;          /* degree plus one of right polynomial     */
{
    TypHandle       hdLL;       /* one element of <hdL>                    */
    TypHandle       hdRR;       /* one element of <hdR>                    */
    TypHandle       hdCC;       /* temp element                            */
    TypHandle       hdTT;       /* temp element                            */
    long            i,  k;      /* loop variables                          */

    /* <r> must be none zero                                               */
    if ( r == 0 )
	Error( "<r> must be non zero", 0L, 0L );

    /* if <l> is trivial return                                            */
    if ( l == 0 )
	return 0;

    /* compute the remainder                                               */
    for ( i = l-r;  0 <= i;  i-- )
    {
	hdRR = ELM_PLIST( hdR,   r );
	hdLL = ELM_PLIST( hdL, i+r );
	hdCC = QUO( hdLL, hdRR );
	for ( k = r;  0 < k;  k-- )
	{
	    hdRR = ELM_PLIST( hdR, k );
	    hdTT = PROD( hdCC, hdRR );
	    hdLL = ELM_PLIST( hdL, i+k );
	    hdLL = DIFF( hdLL, hdTT );
	    SET_ELM_PLIST( hdL, i+k, hdLL );
	}
    }
    for ( i = l;  0 < i;  i-- )
	if ( EQ( ELM_PLIST( hdL, i ), INT_TO_HD(0) ) == HdFalse )
	    break;
    return i;
}


/****************************************************************************
**
*F  FunRemainderCoeffs( <hdCall> )  . . . . . . .  internal 'RemainderCoeffs'
**
**  'FunRemainderCoeffs' implements 'RemainderCoeffs( <l>, <r> )'
*/
TypHandle       FunRemainderCoeffs ( hdCall )
    TypHandle       hdCall;
{
    TypHandle       hdL,  hdR,  hdP,  hdTT;
    TypFFE        * ptL,  * ptP;
    long            l,  r,  i;

    /* check arguments                                                     */
    if ( SIZE( hdCall ) != 3 * SIZE_HD )
        return Error( "usage: RemainderCoeffs( <l>, <r> )", 0L, 0L );
    hdL = EVAL( PTR( hdCall )[1] );
    hdR = EVAL( PTR( hdCall )[2] );

    /* if <l> is the empty list return a copy of this list                 */
    if ( IS_LIST(hdL) && LEN_LIST(hdL) == 0 )
        return Copy(hdL);

    /* if <r> is trivial there is nothing to reduce                        */
    if ( IS_LIST(hdR) && LEN_LIST(hdR) == 0 )
	return Error( "<r> must be non zero", 0L, 0L );

    /* remainder of two finite field vectors over the same field           */
    if ( XType(hdL) == T_VECFFE && XType(hdR) == T_VECFFE )
    {

        /* make sure both vectors do have a non zero leading coefficient   */
        l = LEN_VECFFE( hdL );
        r = LEN_VECFFE( hdR );
        while ( 0 < l && VAL_VECFFE(hdL,l) == 0 )  l--;
        while ( 0 < r && VAL_VECFFE(hdR,r) == 0 )  r--;

        /* <r> must be none zero                                           */
        if ( r == 0 )
            return Error( "<r> must be non zero", 0L, 0L );

	/* if <hdL> is trivial return the zero polynomial                  */
        if ( l == 0 )
        {
            hdP = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
            SET_LEN_PLIST( hdP, 0 );
            return hdP;
        }

	/* allocate the remainder and copy <l> into it                     */
	hdP = NewBag( T_VECFFE, SIZE_PLEN_VECFFE(l) );
	ptL = (TypFFE*)( PTR(hdL) + 1 );
	ptP = (TypFFE*)( PTR(hdP) + 1 );
	for ( i = l;  0 < i;  i-- )
	    *ptP++ = *ptL++;
	SET_FLD_VECFFE( hdP, FLD_VECFFE(hdL) );

	/* use 'ReduceVecFFE' to do the work                               */
	i = ReduceVecFFE( hdP, l, hdR, r );

	/* normalize <hdL> and return                                      */
	return NormalizeCoeffs( hdP, i );
    }

    /* remainder of two integers/cyclotomics vectors                       */
    else if ( XType( hdL ) == T_VECTOR && XType( hdR ) == T_VECTOR )
    {
        l = LEN_PLIST( hdL );
        r = LEN_PLIST( hdR );

        /* make sure both vectors do have a non zero leading coefficient   */
        while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )  l--;
        while ( 0 < r && EQ(ELM_PLIST(hdR,r), INT_TO_HD(0)) == HdTrue )  r--;

        /* <r> must be none zero                                           */
        if ( r == 0 )
            return Error( "<r> must be non zero", 0L, 0L );

	/* if <hdL> is trivial return the zero polynomial                  */
        if ( l == 0 )
        {
            hdP = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
            SET_LEN_PLIST( hdP, 0 );
	    return hdP;
        }

	/* allocate the remainder and copy <l> into it                     */
	hdP  = NewBag( T_LIST, SIZE_PLEN_PLIST(l) );
	for ( i = l;  0 < i;  i-- )
	{
	    hdTT = ELM_PLIST( hdL, i );
	    SET_ELM_PLIST( hdP, i, hdTT );
	}

	/* compute the remainder using 'ReduceVector'                      */
	i = ReduceVector( hdP, l, hdR, r );

	/* normalize <hdL> and return                                      */
	return NormalizeCoeffs( hdP, i );
    }
    return Error( "<l> and <r> must have a common field", 0L, 0L );
}


/****************************************************************************
**
*F  FunReduceCoeffs( <hdCall> ) . . . . . . . . . . . internal 'ReduceCoeffs'
**
**  'FunReduceCoeffs' implements 'ReduceCoeffs( <l>, <r> )'
*/
TypHandle       FunReduceCoeffs ( hdCall )
    TypHandle       hdCall;
{
    TypHandle       hdL,  hdR;
    long            l,  r,  i;

    /* check arguments                                                     */
    if ( SIZE( hdCall ) != 3 * SIZE_HD )
        return Error( "usage: ReduceCoeffs( <l>, <r> )", 0L, 0L );
    hdL = EVAL( PTR(hdCall)[1] );
    hdR = EVAL( PTR(hdCall)[2] );

    /* if <l> is the empty list return                                     */
    if ( IS_LIST(hdL) && LEN_LIST(hdL) == 0 )
        return HdVoid;

    /* if <r> is trivial there is nothing to reduce                        */
    if ( IS_LIST(hdR) && LEN_LIST(hdR) == 0 )
	return Error( "<r> must be non zero", 0L, 0L );

    /* remainder of two finite field vectors over the same field           */
    if ( XType(hdL) == T_VECFFE && XType(hdR) == T_VECFFE )
    {
        /* make sure both vectors do have a non zero leading coefficient   */
        l = LEN_VECFFE(hdL);
        r = LEN_VECFFE(hdR);
        while ( 0 < l && VAL_VECFFE(hdL,l) == 0 )  l--;
        while ( 0 < r && VAL_VECFFE(hdR,r) == 0 )  r--;

        /* <r> must be none zero                                           */
        if ( r == 0 )
            return Error( "<r> must be non zero", 0L, 0L );

	/* if <hdL> is trivial make it the zero polynomial                 */
        if ( l == 0 )
        {
	    NormalizeCoeffs( hdL, 0 );
            return HdVoid;
        }

	/* use 'ReduceVecFFE' to do the work                               */
	i = ReduceVecFFE( hdL, l, hdR, r );

	/* normalize <hdL> and return                                      */
	NormalizeCoeffs( hdL, i );
        return HdVoid;
    }

    /* remainder of two integers/cyclotomics vectors                       */
    else if ( XType(hdL) == T_VECTOR && XType(hdR) == T_VECTOR )
    {

        /* make sure both vectors do have a non zero leading coefficient   */
        l = LEN_PLIST(hdL);
        r = LEN_PLIST(hdR);
        while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )  l--;
        while ( 0 < r && EQ(ELM_PLIST(hdR,r), INT_TO_HD(0)) == HdTrue )  r--;

        /* <r> must be none zero                                           */
        if ( r == 0 )
            return Error( "<r> must be non zero", 0L, 0L );

	/* if <hdL> is trivial make it the zero polynomial                 */
        if ( l == 0 )
        {
	    NormalizeCoeffs( hdL, 0 );
	    return HdVoid;
        }

	/* compute the remainder using 'ReduceVector'                      */
	i = ReduceVector( hdL, l, hdR, r );

	/* normalize <hdL> and return                                      */
	NormalizeCoeffs( hdL, i );
        return HdVoid;
    }
    return Error( "<l> and <r> must have a common field", 0L, 0L );
}


/****************************************************************************
**
*F  ReduceVectorMod( <hdL>, <l>, <hdR>, <r>, <hdP> )  reduce polynomial <hdL>
**
**  Reduce   the  polynomial  <hdL> in place  by  the   polynomial <hdR>.  <l>
**  (<r>)  must be  the degree  plus 1 of  <hdL> (<hdR>). The degree plus one
**  of the result is returned. The reduction takes place modulo <hdP>
*/
long		ReduceVectorMod ( hdL, l, hdR, r, hdP )
    TypHandle       hdL;        /* left polynomial coeffs                  */
    TypHandle       hdR;        /* right polynomial coeffs                 */
    long            l;          /* degree plus one of left polynomial      */
    long            r;          /* degree plus one of right polynomial     */
    TypHandle       hdP;        /* an integer                              */
{
    TypHandle       hdLL;       /* one element of <hdL>                    */
    TypHandle       hdRR;       /* one element of <hdR>                    */
    TypHandle       hdCC;       /* temp element                            */
    TypHandle       hdTT;       /* temp element                            */
    TypHandle       hdQ;        /* <hdP> / 2                               */
    long            i,  k;      /* loop variables                          */

    /* catch a trivial case (<l> or <hdP> equal zero)                      */
    if ( l == 0 || EQ( hdP, INT_TO_HD(0) ) == HdTrue )
	return 0;

    /* if <hdP> is negative reduce into <hdP>/2 and <hdP>                  */
    if ( LT( hdP, INT_TO_HD(0) ) == HdTrue )
    {
	hdP = DIFF( INT_TO_HD(0), hdP );
	hdQ = SUM( hdP, INT_TO_HD(1) );
	hdQ = QuoInt( hdQ, INT_TO_HD(2) );
	hdQ = DIFF( hdQ, INT_TO_HD(1) );
    }
    else
	hdQ = 0;

    /* if <r> is zero or too big reduce modulo <hP>                        */
    if ( r == 0 || l < r )
    {
	for ( i = l;  0 < i;  i-- )
	{
	    hdTT = MOD( ELM_LIST(hdL,i), hdP );
	    if ( hdQ != 0 && LT( hdQ, hdTT ) == HdTrue )
		hdTT = DIFF( hdTT, hdP );
	    SET_ELM_PLIST( hdL, i, hdTT );
	}
    }

    /* otherwise, compute the remainder                                    */
    else
	for ( i = l-r;  0 <= i;  i-- )
	{
	    hdRR = ELM_LIST( hdR,   r );
	    hdLL = ELM_LIST( hdL, i+r );
	    hdCC = QUO( hdLL, hdRR );
	    hdCC = MOD( hdCC, hdP );
	    for ( k = r;  0 < k;  k-- )
	    {
		hdRR = ELM_LIST( hdR, k );
		hdTT = PROD( hdCC, hdRR );
		hdLL = ELM_LIST( hdL, i+k );
		hdLL = DIFF( hdLL, hdTT );
		hdLL = MOD( hdLL, hdP );
		if ( hdQ != 0 && LT( hdQ, hdLL ) == HdTrue )
		    hdLL = DIFF( hdLL, hdP );
		SET_ELM_PLIST( hdL, i+k, hdLL );
	    }
	}

    /* remove leading zeros                                                */
    for ( i = l;  0 < i;  i-- )
	if ( EQ( ELM_LIST(hdL,i), INT_TO_HD(0) ) == HdFalse )
	    break;

    /* and return the new length                                           */
    return i;
}


/****************************************************************************
**
*F  FunReduceCoeffsMod( <hdCall> )  . . . . . . .  internal 'ReduceCoeffsMod'
**
**  'FunReduceCoeffsMod' implements 'ReduceCoeffsMod( <l>, <r>, <p> )'
*/
TypHandle       FunReduceCoeffsMod ( hdCall )
    TypHandle       hdCall;
{
    TypHandle       hdL,  hdR,  hdP;
    long            l,    r;

    /* check arguments                                                     */
    if ( 4*SIZE_HD < SIZE(hdCall) || SIZE(hdCall) < 3*SIZE_HD )
        return Error( "usage: ReduceCoeffsMod( <l>, <r>, <p> )", 0L, 0L );
    hdL = EVAL( PTR(hdCall)[1] );
    if ( SIZE(hdCall) == 4 * SIZE_HD )
    {
	hdR = EVAL( PTR(hdCall)[2] );
	hdP = EVAL( PTR(hdCall)[3] );
    }
    else
    {
	hdR = 0;
	hdP = EVAL( PTR(hdCall)[2] );
    }

    /* if <l> is the empty list return                                     */
    if ( IS_LIST(hdL) && LEN_LIST(hdL) == 0 )
        return HdVoid;
    if ( XType(hdL) != T_VECTOR )
        return Error( "<l> must be a vector over the rationals", 0L, 0L );

    /* catch the special case that <hdR> is an empty list                  */
    if ( hdR == 0 || ( IS_LIST(hdR) && LEN_LIST(hdR) == 0 ) )
	r = 0;
    else if ( XType(hdR) != T_VECTOR )
        return Error( "<r> must be a vector over the rationals", 0L, 0L );
    else
	r = LEN_PLIST(hdR);

    /* remainder of two integers/cyclotomics vectors                       */
    l = LEN_PLIST(hdL);

    /* make sure both vectors do have a non zero leading coefficient       */
    while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )  l--;
    while ( 0 < r && EQ(ELM_PLIST(hdR,r), INT_TO_HD(0)) == HdTrue )  r--;

    /* reduce <hdL> by <hdR> and <hdP>                                     */
    if ( l != 0 )
	l = ReduceVectorMod( hdL, l, hdR, r, hdP );
    NormalizeCoeffs( hdL, l );
    return HdVoid;
}


/****************************************************************************
**
*F  PowerModVecInt( <hdG>, <g>, <exp>, <hdF>, <f> ) . . . power mod small int
*/
TypHandle	PowerModVecInt ( hdG,  g,  exp,  hdF,  f )
    TypHandle	    hdG;	/* polynomial to be powered                */
    long            g;          /* degree plus one of <hdG>                */
    long            exp;        /* exponent                                */
    TypHandle       hdF;        /* modulus                                 */
    long            f;          /* degree plus one of <hdF>                */
{
    TypHandle	    hdP;        /* result                                  */
    TypHandle       hdR1;       /* temporary storage for multiplication    */
    TypHandle       hdR2;       /* temporary storage for multiplication    */
    long            p;          /* length of <hdP>                         */
    int             isVecFFE;   /* vecffe or vector flag                   */
    unsigned long   i;          /* loop variable                           */

    /* if <g> and <f> are written over different fields, convert them      */
    isVecFFE = ( XType(hdG) == T_VECFFE );

    /* Copy <hdG> and reduce it modulo <hdF>                               */
    hdG = Copy(hdG);
    if ( isVecFFE )
    {
	if ( ReduceVecFFE( hdG, g, hdF, f ) == 0 )
	    return NormalizeCoeffs( hdG, 0 );
	Resize( hdG, SIZE_PLEN_VECFFE(2*f) );
    }
    else
    {
	if ( ReduceVector( hdG, g, hdF, f ) == 0 )
	    return NormalizeCoeffs( hdG, 0 );
	Resize( hdG, SIZE_PLEN_PLIST(2*f) );
    }

    /* allocate storage for multiplication                                 */
    hdR1 = Copy(hdG);
    if ( isVecFFE )
    {
	hdR2 = NewBag( T_VECFFE, SIZE_PLEN_VECFFE(2*f) );
	SET_FLD_VECFFE( hdR2, FLD_VECFFE(hdR1) );
    }
    else
	hdR2 = NewBag( T_LIST, SIZE_PLEN_PLIST(2*f) );

    /* compute the power using left to right repeated squaring             */
    i   = 1 << 31;
    p   = 0;
    hdP = 0;
    while ( 1 < i )
    {
	if ( hdP )
	{
	    if ( isVecFFE )
	    {
		p = MultiplyVecFFE( hdR2, hdP, p, hdP, p );
		p = ReduceVecFFE( hdR2, p, hdF, f );
	    }
	    else
	    {
		p = MultiplyVector( hdR2, hdP, p, hdP, p );
		p = ReduceVector( hdR2, p, hdF, f );
	    }
	    hdR1 = hdP;  hdP = hdR2;  hdR2 = hdR1;
	}
	i = i / 2;
	if ( i <= exp )
	{
	    if ( hdP )
	    {
		if ( isVecFFE )
		{
		    p = MultiplyVecFFE( hdR2, hdP, p, hdG, g );
		    p = ReduceVecFFE( hdR2, p, hdF, f );
		}
		else
		{
		    p = MultiplyVector( hdR2, hdP, p, hdG, g );
		    p = ReduceVector( hdR2, p, hdF, f );
		}
		hdR1 = hdP;  hdP = hdR2;  hdR2 = hdR1;
	    }
	    else
	    {
		hdP = hdR1;
		p = g;
	    }
	    exp = exp - i;
	}
    }

    /* resize result and return                                            */
    return NormalizeCoeffs( hdP, p );
}


/****************************************************************************
**
*F  PowerModVecLongInt( <hdG>, <g>, <hdE>, <hdF>, <f> ) .  power mod long int
*/
TypHandle	PowerModVecLongInt ( hdG,  g,  hdE,  hdF,  f )
    TypHandle	    hdG;	/* polynomial to be powered                */
    long            g;          /* degree plus one of <hdG>                */
    TypHandle       hdE;        /* exponent                                */
    TypHandle       hdF;        /* modulus                                 */
    long            f;          /* degree plus one of <hdF>                */
{
    TypHandle	    hdP;        /* result                                  */
    TypHandle       hdR1;       /* temporary storage for multiplication    */
    TypHandle       hdR2;       /* temporary storage for multiplication    */
    long            p;          /* length of <hdP>                         */
    TypDigit        e;          /* one digit of <hdE>                      */
    int             isVecFFE;   /* vecffe or vector flag                   */
    unsigned long   i;          /* loop variable                           */
    long            l;          /* loop variable                           */

    /* if <g> and <f> are written over different fields, convert them      */
    isVecFFE = ( XType(hdG) == T_VECFFE );

    /* Copy <hdG> and reduce it modulo <hdF>                               */
    hdG = Copy(hdG);
    if ( isVecFFE )
    {
	if ( ReduceVecFFE( hdG, g, hdF, f ) == 0 )
	    return NormalizeCoeffs( hdG, 0 );
	Resize( hdG, SIZE_PLEN_VECFFE(2*f) );
    }
    else
    {
	if ( ReduceVector( hdG, g, hdF, f ) == 0 )
	    return NormalizeCoeffs( hdG, 0 );
	Resize( hdG, SIZE_PLEN_PLIST(2*f) );
    }

    /* allocate storage for multiplication                                 */
    hdR1 = Copy(hdG);
    if ( isVecFFE )
    {
	hdR2 = NewBag( T_VECFFE, SIZE_PLEN_VECFFE(2*f) );
	SET_FLD_VECFFE( hdR2, FLD_VECFFE(hdR1) );
    }
    else
	hdR2 = NewBag( T_LIST, SIZE_PLEN_PLIST(2*f) );

    /* compute the power using left to right repeated squaring             */
    hdP = 0;
    p   = 0;
    for ( l = SIZE(hdE)/sizeof(TypDigit)-1;  0 <= l;  l-- )
    {
	i = 1 << (8*sizeof(TypDigit));
	e = ((TypDigit*) PTR(hdE))[l];
	while ( 1 < i )
	{
	    if ( hdP )
	    {
		if ( isVecFFE )
		{
		    p = MultiplyVecFFE( hdR2, hdP, p, hdP, p );
		    p = ReduceVecFFE( hdR2, p, hdF, f );
		}
		else
		{
		    p = MultiplyVector( hdR2, hdP, p, hdP, p );
		    p = ReduceVector( hdR2, p, hdF, f );
		}
		hdR1 = hdP;  hdP = hdR2;  hdR2 = hdR1;
	    }
	    i = i / 2;
	    if ( i <= e )
	    {
		if ( hdP )
		{
		    if ( isVecFFE )
		    {
			p = MultiplyVecFFE( hdR2, hdP, p, hdG, g );
			p = ReduceVecFFE( hdR2, p, hdF, f );
		    }
		    else
		    {
			p = MultiplyVector( hdR2, hdP, p, hdG, g );
			p = ReduceVector( hdR2, p, hdF, f );
		    }
		    hdR1 = hdP;  hdP = hdR2;  hdR2 = hdR1;
		}
		else
		{
		    hdP = hdR1;
		    p = g;
		}
		e = e - i;
	    }
	}
    }

    /* resize result and return                                            */
    return NormalizeCoeffs( hdP, p );
}


/****************************************************************************
**
*F  FunPowerModCoeffs( <hdCall> ) . . . .  internal function 'PowerModCoeffs'
*/
TypHandle	FunPowerModCoeffs ( hdCall )
    TypHandle	    hdCall;
{
    TypHandle       hdG,  hdE,  hdR,  hdP;
    long	    g,  r;

    /* check arguments                                                     */
    if ( SIZE( hdCall ) != 4 * SIZE_HD )
        return Error( "usage: PowerModCoeffs( <g>, <exp>, <r> )", 0L, 0L );
    hdG = EVAL( PTR(hdCall)[1] );
    hdE = EVAL( PTR(hdCall)[2] );
    hdR = EVAL( PTR(hdCall)[3] );

    /* if <l> is the empty list return a copy of this list                 */
    if ( IS_LIST(hdG) && LEN_LIST(hdG)==0 )
        return Copy(hdG);
    if ( IS_LIST(hdR) && LEN_LIST(hdR)==0 )
        return Error( "<r> must be non zero", 0L, 0L );
    if ( TYPE(hdE)!=T_INTPOS && ( TYPE(hdE)!=T_INT || HD_TO_INT(hdE)<1 ) )
	return Error( "<exp> must be a positive integer", 0L, 0L );

    /* power of a finite field vectors mod another over the same field     */
    if ( XType(hdG) == T_VECFFE && XType(hdR) == T_VECFFE )
    {

        /* make sure both vectors do have a non zero leading coefficient   */
        g = LEN_VECFFE(hdG);
        r = LEN_VECFFE(hdR);
        while ( 0 < g && VAL_VECFFE(hdG,g) == 0 )  g--;
        while ( 0 < r && VAL_VECFFE(hdR,r) == 0 )  r--;

        /* <r> must be none zero                                           */
        if ( r == 0 )
            return Error( "<r> must be non zero", 0L, 0L );

	/* if <hdG> is trivial return the zero polynomial                  */
        if ( g == 0 )
        {
            hdP = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
            SET_LEN_PLIST( hdP, 0 );
            return hdP;
        }

	/* use 'PowerModVecInt' or 'PowerModVecLongInt'                    */
	if ( TYPE(hdE) == T_INT )
	    return PowerModVecInt( hdG, g, HD_TO_INT(hdE), hdR, r );
	else
	    return PowerModVecLongInt( hdG, g, hdE, hdR, r );
    }

    /* power mod of two integers/cyclotomics vectors                       */
    else if ( XType( hdG ) == T_VECTOR && XType( hdR ) == T_VECTOR )
    {

        /* make sure both vectors do have a non zero leading coefficient   */
        g = LEN_PLIST(hdG);
        r = LEN_PLIST(hdR);
        while ( 0 < g && EQ(ELM_PLIST(hdG,g), INT_TO_HD(0)) == HdTrue )  g--;
        while ( 0 < r && EQ(ELM_PLIST(hdR,r), INT_TO_HD(0)) == HdTrue )  r--;

        /* <r> must be none zero                                           */
        if ( r == 0 )
            return Error( "<r> must be non zero", 0L, 0L );

	/* if <hdL> is trivial return the zero polynomial                  */
        if ( g == 0 )
        {
            hdP = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
            SET_LEN_PLIST( hdP, 0 );
	    return hdP;
        }

	/* use 'PowerModVecInt' or 'PowerModVecLongInt'                    */
	if ( TYPE(hdE) == T_INT )
	    return PowerModVecInt( hdG, g, HD_TO_INT(hdE), hdR, r );
	else
	    return PowerModVecLongInt( hdG, g, hdE, hdR, r );
    }
    return Error( "<g> and <r> must have a common field", 0L, 0L );
}


/****************************************************************************
**
*F  FunShiftedCoeffs( <hdCall> )  . . . . . internal function 'ShiftedCoeffs'
**
**  'FunShiftedCoeffs' implements 'ShiftedCoeffs( <l>, <n> )'
*/
TypHandle       FunShiftedCoeffs ( hdCall )
    TypHandle       hdCall;
{
    TypHandle       hdL,  hdN,  hdS;
    TypFFE        * ptL, *ptE, *ptS;
    long            l,  n,  i;

    /* check arguments                                                     */
    if ( SIZE( hdCall ) != 3 * SIZE_HD )
        return Error( "usage: ShiftedCoeffs( <l>, <n> )", 0L, 0L );
    hdL = EVAL( PTR(hdCall)[1] );
    hdN = EVAL( PTR(hdCall)[2] );
    if ( TYPE(hdN) != T_INT )
	return Error( "<n> must be an integer", 0L, 0L );
    n = HD_TO_INT(hdN);

    /* if <l> is the empty list return a copy                              */
    if ( IS_LIST(hdL) && LEN_LIST(hdL)==0 )
        return Copy(hdL);

    /* shift a finite field vector                                         */
    if ( XType(hdL) == T_VECFFE )
    {

        /* make sure the vectors has a non zero leading coefficient        */
        l = LEN_VECFFE(hdL);
        while ( 0 < l && VAL_VECFFE(hdL,l) == 0 )  l--;

	/* if <n> is negative shrink <hdL>                                 */
	if ( n < 0 )
	{

	    /* if <l> is not bigger than <n> <hdL> will shrink to nothing  */
	    if ( l <= -n )
	    {
		hdS = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
                SET_LEN_PLIST( hdS, 0 );
		return hdS;
	    }

	    /* allocate a new vector and copy the entries                  */
	    hdS = NewBag( T_VECFFE, SIZE_PLEN_VECFFE(l+n) );
            SET_FLD_VECFFE( hdS, FLD_VECFFE(hdL) );
	    ptL = ((TypFFE*)(PTR(hdL)+1))-n;
	    ptE = ((TypFFE*)(PTR(hdL)+1))+l;
	    ptS = (TypFFE*)(PTR(hdS)+1);
	    while ( ptL < ptE )
		*ptS++ = *ptL++;
	}

	/* if <n> is positive enlarge <hdL>                                */
	else if ( 0 < n )
	{
	    hdS = NewBag( T_VECFFE, SIZE_PLEN_VECFFE(l+n) );
            SET_FLD_VECFFE( hdS, FLD_VECFFE(hdL) );
	    ptL = (TypFFE*)(PTR(hdL)+1);
	    ptE = ((TypFFE*)(PTR(hdL)+1))+l;
	    ptS = ((TypFFE*)(PTR(hdS)+1))+n;
	    while ( ptL < ptE )
		*ptS++ = *ptL++;
	}

	/* if <n> is zero copy <hdL> and shrink if necessary               */
	else
	    hdS = NormalizeCoeffs( Copy(hdL), l );
	return hdS;
    }

    /* shift a vector                                                      */
    else if ( XType(hdL) == T_VECTOR )
    {

        /* make sure the vectors has a non zero leading coefficient        */
        l = LEN_PLIST(hdL);
        while ( 0 < l && EQ(ELM_PLIST(hdL,l), INT_TO_HD(0)) == HdTrue )  l--;

	/* if <n> is negative shrink <hdL>                                 */
	if ( n < 0 )
	{

	    /* if <l> is not bigger than <n> <hdL> will shrink to nothing  */
	    if ( l <= -n )
	    {
		hdS = NewBag( T_LIST, SIZE_PLEN_PLIST(0) );
		SET_LEN_PLIST( hdS, 0 );
		return hdS;
	    }

	    /* allocate a new vector and copy the entries                  */
	    hdS = NewBag( T_LIST, SIZE_PLEN_PLIST(l+n) );
	    SET_LEN_PLIST( hdS, l+n );
	    for ( i = l;  -n < i;  i-- )
		SET_ELM_PLIST( hdS, i+n, ELM_PLIST(hdL,i) );
	}

	/* if <n> is positive enlarge <hdL>                                */
	else if ( 0 < n )
	{
	    hdS = NewBag( T_LIST, SIZE_PLEN_PLIST(l+n) );
	    SET_LEN_PLIST( hdS, l+n );
	    for ( i = l;  0 < i;  i-- )
		SET_ELM_PLIST( hdS, i+n, ELM_PLIST(hdL,i) );
	    for ( i = n;  0 < i;  i-- )
		SET_ELM_PLIST( hdS, i, INT_TO_HD(0) );
	}

	/* if <n> is zero copy <hdL> and shrink if necessary               */
	else
	    hdS = NormalizeCoeffs( Copy(hdL), l );
	return hdS;
    }
    else
	return Error( "<l> must be a vector", 0L, 0L );
}


/****************************************************************************
**
*F  InitPolynom() . . . . . . . . . . . . . .  initialize the polynom package
*/
void            InitPolynom ()
{

    /* install the internal functions                                      */
    InstIntFunc( "ProductCoeffs",   	FunProductCoeffs    );
    InstIntFunc( "ProductCoeffsMod",	FunProductCoeffsMod );
    InstIntFunc( "RemainderCoeffs", 	FunRemainderCoeffs  );
    InstIntFunc( "ReduceCoeffs",    	FunReduceCoeffs     );
    InstIntFunc( "ReduceCoeffsMod", 	FunReduceCoeffsMod  );
    InstIntFunc( "PowerModCoeffs",  	FunPowerModCoeffs   );
    InstIntFunc( "SumCoeffs",       	FunSumCoeffs 	    );
    InstIntFunc( "AddCoeffs",       	FunAddCoeffs 	    );
    InstIntFunc( "ShiftedCoeffs",   	FunShiftedCoeffs    );
}


/****************************************************************************
**
*E  Emacs . . . . . . . . . . . . . . . . . . . . . . . local emacs variables
**
**  Local Variables:
**  mode:               outline
**  outline-regexp:     "*F\\|*V\\|*T\\|*E"
**  fill-column:        73
**  fill-prefix:        "**  "
**  eval:               (local-set-key "\t" 'c-indent-command)
**  eval:               (local-set-key ";"  'electric-c-semi )
**  eval:               (local-set-key "{"  'electric-c-brace)
**  eval:               (local-set-key "}"  'electric-c-brace)
**  eval:               (hide-body)
**  End:
*/



