#############################################################################
##
#A  grplatt.g                   GAP library                  Martin Schoenert
#A                                                            & Juergen Mnich
##
#H  @(#)$Id: grplatt.g,v 3.2 1993/07/30 11:13:57 martin Rel $
##
#Y  Copyright (C)  1993,  Lehrstuhl D fuer Mathematik,  RWTH Aachen,  Germany
##
##  This file contains the functions for conjugacy classes of subgroups.
##
#N  1993/07/29 martin the computation of the zuppos should be a function
##
#H  $Log: grplatt.g,v $
#H  Revision 3.2  1993/07/30  11:13:57  martin
#H  changed 'Lattice' so that it can be applied to other domains
#H
#H  Revision 3.1  1993/07/29  21:44:40  martin
#H  initial revision under RCS
#H
##


#############################################################################
##
#F  InfoLattice?(...) . . . . . . . . . . . information functions for lattice
##
if not IsBound(InfoLattice1)  then InfoLattice1 := Ignore;  fi;
if not IsBound(InfoLattice2)  then InfoLattice2 := Ignore;  fi;


#############################################################################
##
#F  IsConjugacyClassSubgroups(<C>)  . . . . . . . test for class of subgroups
##
IsConjugacyClassSubgroups := function ( C )
    return IsRec( C )
       and IsBound( C.isConjugacyClassSubgroups )
       and C.isConjugacyClassSubgroups;
end;


#############################################################################
##
#F  ConjugacyClassSubgroups(<G>,<H>)  . . . . . . create a class of subgroups
##
ConjugacyClassSubgroups := function ( G, H )

    # check the arguments
    if not IsGroup( G )  then
        Error("<G> must be a group");
    elif not IsSubgroup( G, H )  then
        Error("<H> must be a subgroup of <G>");
    fi;

    # create the class
    return G.operations.ConjugacyClassSubgroups( G, H );

end;

GroupOps.ConjugacyClassSubgroups := function ( G, H )
    local   C;

    # make the conjugacy class
    C := rec( );
    C.isDomain                  := true;
    C.isConjugacyClassSubgroups := true;

    # enter the identifying information
    C.group          := G;
    C.representative := H;

    # enter the operations record
    C.operations     := ConjugacyClassSubgroupsGroupOps;

    # return the conjugacy class
    return C;

end;


#############################################################################
##
#F  ConjugacyClassSubgroupsGroupOps . . . . . . operations record for classes
##
ConjugacyClassSubgroupsGroupOps := Copy( DomainOps );

ConjugacyClassSubgroupsGroupOps.Elements := function ( C )
    return Set( List( RightTransversal(C.group,
                                       Normalizer(C.group,C.representative)),
                      t -> C.representative^t ) );
end;

ConjugacyClassSubgroupsGroupOps.Size := function ( C )
    return Index( C.group, Normalizer( C.group, C.representative ) );
end;

ConjugacyClassSubgroupsGroupOps.\= := function ( C, D )
    local    isEql;
    if    IsRec( C )  and IsBound( C.isConjugacyClassSubgroups )
      and IsRec( D )  and IsBound( D.isConjugacyClassSubgroups )
      and C.group = D.group
    then
        isEql := Size( C ) = Size( D )
             and Size( C.representative ) = Size( D.representative )
             and RepresentativeOperation( C.group,
                                          D.representative,
                                          C.representative ) <> false;
    else
        isEql := DomainOps.\=( C, D );
    fi;
    return isEql;
end;

ConjugacyClassSubgroupsGroupOps.\in := function ( H, C )
    return Size( H ) = Size( C.representative )
       and RepresentativeOperation( C.group,
                                    H, C.representative ) <> false;
end;

ConjugacyClassSubgroupsGroupOps.Random := function ( C )
    return C.representative ^ Random( C.group );
end;

ConjugacyClassSubgroupsGroupOps.Print := function ( C )
    Print("ConjugacyClassSubgroups( ",C.group,", ",C.representative," )");
end;

ConjugacyClassGroupOps.\* := function ( C, D )
    if IsConjugacyClass( C )  then
        return Elements( C ) * D;
    elif IsConjugacyClass( D )  then
        return C * Elements( D );
    else
        Error("panic, neither <C> nor <D> is a class of subgroups");
    fi;
end;


#############################################################################
##
#F  ConjugacyClassesSubgroups(<G>)  . . . . . classes of subgroups of a group
##
ConjugacyClassesSubgroups := function ( G )

    # check the argument
    if not IsGroup( G )  then
        Error("<G> must be a group");
    fi;

    # compute the classes
    if not IsBound( G.conjugacyClassesSubgroups )  then
        G.conjugacyClassesSubgroups :=
          G.operations.ConjugacyClassesSubgroups( G );
    fi;

    # return the classes
    return G.conjugacyClassesSubgroups;

end;

GroupOps.ConjugacyClassesSubgroups := function ( G )
    return LatticeSubgroups( G ).classes;
end;


#############################################################################
##
#F  Lattice(<G>)  . . . . . . . . . . . . . . . . . . .  lattice of something
##
#N  1993/07/30 martin this should go somewhere else
##
Lattice := function ( G )
    if not IsBound( G.lattice )  then
        G.lattice := G.operations.Lattice( G );
    fi;
    return G.lattice;
end;

GroupOps.Lattice := function ( G )
    return LatticeSubgroups( G );
end;


#############################################################################
##
#F  LatticeSubgroups(<G>) . . . . . . . . . . . . . . .  lattice of subgroups
##
LatticeSubgroups := function ( G )

    # check the argument
    if not IsGroup( G )  then
        Error("<G> must be a group");
    fi;

    # compute the lattice of subgroups
    if not IsBound( G.latticeSubgroups )  then
        G.latticeSubgroups := G.operations.LatticeSubgroups( G );
    fi;

    # return the lattice of subgroups
    return G.latticeSubgroups;

end;

GroupOps.LatticeSubgroups := function ( G )
    local   lattice,            # lattice (result)
            classes,            # list of all classes (result)
            classesZ,           # zuppos blist of classes
            classesB,           # extend-by blist of classes
            nrSubs,             # number of subgroups
            layer,              # current layer (nr. of primes div. size)
            layerb,             # begin of previous layer
            layere,             # end of previous layer
            factors,            # factorization of <G>'s size
            ppowers,            # prime powers dividing <G>'s size
            elements,           # elements of <G>
            zuppos,             # generators of prime power order
            zupposPrime,        # corresponding prime
            zupposPower,        # index of power of generator
            perfect,            # classes of perfect subgroups of <G>
            perfectZ,           # zuppos blist of the perfect subgroups
            rep,                # representative of a class
            repE,               # elements of <rep>
            repZ,               # zuppos blist of <rep>
            repB,               # extend blist of <rep>
            new,                # new subgroup found
            newC,               # class of <new>
            newE,               # elements of <new>
            newZ,               # zuppos blist of <new>
            nrm,                # normalizer of <new>
            nrmE,               # elements of <nrm>
            nrmZ,               # zuppos blist of <nrm>
            con,                # one conjugate of <new> (not needed)
            conE,               # elements of <con>
            conZ,               # zuppos of <con>
            old,                # old representative of class in <classes>
            oldE,               # elements of <old> (not needed)
            oldZ,               # zuppos of <old>
            trn,                # transversal of <nrm> in <G>
            i, k, l, t;         # loop variables

    # compute the prime powers dividing <G>'s size (must include 1)
    factors := Factors( Size( G ) );
    ppowers := [ 1 ];
    for i  in Collected( factors )  do
        for k  in [1..i[2]]  do
            Add( ppowers, i[1]^k );
        od;
    od;

    # compute the elements of the group
    if IsBound( G.elements )  then
        elements := G.elements;
    else
        elements := Elements( G );
        Unbind( G.elements );
    fi;
    InfoLattice1( "#I  <G> has ", Length(elements), " elements, \c" );

    # compute a system of generators for the cyclic sgr. of prime power size
    zuppos := [];
    zupposPower := [];
    for t  in elements  do
        i := Order( G, t );
        if i in ppowers and not t in zupposPower  then
            Add( zuppos, t );
            for k  in [2..i-1]  do
                if GcdInt( i, k ) = 1  then
                    AddSet( zupposPower, t^k );
                fi;
            od;
        fi;
    od;
    IsSet( zuppos );
    InfoLattice1( Length(zuppos), " zuppos, \c" );

    # compute the prime corresponding to each zuppo and the index of power
    zupposPrime := [];
    zupposPower := [];
    for t  in zuppos  do
        i := SmallestRootInt( Order( G, t ) );
        Add( zupposPrime, i );
        k := 0;
        while k <> false  do
            k := k + 1;
            if GcdInt( i, k ) = 1  then
                l := Position( zuppos, t^(i*k) );
                if l <> false  then
                    Add( zupposPower, l );
                    k := false;
                fi;
            fi;
        od;
    od;
    InfoLattice1( "powers computed\n" );

    # get the perfect subgroups of <G>
    #N  1993/07/27 martin backward compatibility
    if IsBound( G.perfectSubgroups )  then
        perfect := G.perfectSubgroups;
    elif IsBound( G.representativesPerfectSubgroups )  then
        perfect := G.representativesPerfectSubgroups;
    elif IsBound( G.conjugacyClassesPerfectSubgroups )  then
        perfect := List(G.conjugacyClassesPerfectSubgroups,Representative);
    else
        perfect := RepresentativesPerfectSubgroups( G );
    fi;
    perfectZ := List( perfect, new -> BlistList( zuppos, Elements( new ) ) );
    InfoLattice1( "#I  <G> has ", Length(perfect),
                  " representatives of perfect subgroups\n" );

    # initialize the classes list
    classes  := [ ConjugacyClassSubgroups( G, TrivialSubgroup(G) ) ];
    classesZ := [ BlistList( zuppos, [G.identity] ) ];
    classesB := [ DifferenceBlist( BlistList(zuppos,zuppos), classesZ[1] ) ];
    nrSubs := 1;
    layerb := 1;
    layere := 1;

    # loop over the layers of group (except the group itself)
    for layer  in [1..Length(factors)-1]  do
        InfoLattice1( "#I  doing layer ", layer, ", ",
                      "previous layer has ", layere-layerb+1, " classes\n" );

        # extend representatives of the classes of the previous layer
        for i  in [layerb..layere]  do

            # get the representative, its elements and extend-by blist
            InfoLattice2( "#I    extending subgroup ", i, ", \c" );
            rep := classes[i].representative;
            repE := Elements( rep );
            repZ := classesZ[i];
            repB := classesB[i];
            InfoLattice2( "size = ", Length(repE), "\n" );

            # loop over the zuppos whose <p>-th power lies in <sub>
            for k  in [1..Length(zuppos)]  do
                if repB[k] and repZ[zupposPower[k]]  then

                    # add the new conjugacy class
                    new := Subgroup( Parent(G),
                                     Concatenation( rep.generators,
                                                    [ zuppos[k] ] ) );
                    newC := ConjugacyClassSubgroups( G, new );
                    Add( classes, newC );
                    InfoLattice2( "#I      found new class ",
                                  Length(classes), ", \c" );

                    # add the new zuppos blist
                    newE := Union( List( [0..zupposPrime[k]],
                                         l -> repE*zuppos[k]^l ) );
                    new.elements := newE;
                    new.size     := Length(newE);
                    newZ := BlistList( zuppos, newE );
                    if layer < Length(factors)-1  then
                        Add( classesZ, newZ );
                    fi;
                    InfoLattice2( "size = ", Length(newE), ", \c" );

                    # compute the normalizer
                    nrm := Normalizer( G, new );
                    # Unbind( new.normalizer );
                    new.normalizer := Subgroup( Parent(G), nrm.generators );
                    newC.size := Index( G, nrm );
                    InfoLattice2( "length = ", Index( G, nrm ), ", \c" );
                    nrSubs := nrSubs + Index( G, nrm );

                    # add the new extend-by blist (<nrm> - <new>)
                    if layer < Length(factors)-1  then
                        nrmE := Elements( nrm );
                        nrmZ := BlistList( zuppos, nrmE );
                        SubtractBlist( nrmZ, newZ );
                        Add( classesB, nrmZ );
                    fi;

                    # compute the transversal
                    trn := RightTransversal( G, nrm );
                    InfoLattice2( "tested \c" );

                    # loop over the conjugates of <new>
                    for t  in trn  do

                        # compute the zuppos blist of the conjugate
                        if t = G.identity  then
                            conE := newE;
                            conZ := newZ;
                        else
                            conE := OnSets( newE, t );
                            conZ := BlistList( zuppos, conE );
                        fi;

                        # loop over the already found classes
                        for l  in [i..layere]  do
                            old  := classes[l].representative;
                            oldZ := classesZ[l];

                            # test if the rep is a subgroup of <con>
                            if IsSubsetBlist( conZ, oldZ )  then

                                # don't extend <old> by the elements of <con>
                                SubtractBlist( classesB[l], conZ );

                            fi;

                        od;

                    od;

                    # now we are done with the new class
                    InfoLattice2( "inclusions\n" );

                fi; # if repB[k] and repZ[zupposPower[k]]  then ...
            od; # for k  in [1..Length(zuppos)]  do ...

            # remove the stuff we don't need any more
            Unbind( rep.elements );
            Unbind( classesZ[i] );
            Unbind( classesB[i] );

        od; # for i  in [layerb..layere]  do ...

        # add the classes of perfect subgroups
        for i  in [1..Length(perfect)]  do
            new := perfect[i];
            if IsPerfect(new) and Length(Factors(Size(new))) = layer  then

                # add the new conjugacy class
                newC := ConjugacyClassSubgroups( G, new );
                Add( classes, newC );
                InfoLattice2( "#I      found perfect class ",
                              Length(classes), ", \c" );

                # add the new zuppos blist
                newE := new.elements;
                newZ := perfectZ[i];
                if layer < Length(factors)-1  then
                    Add( classesZ, newZ );
                fi;
                InfoLattice2( "size = ", Length(newE), ", \c" );

                # compute the normalizer
                nrm := Normalizer( G, new );
                # Unbind( new.normalizer );
                new.normalizer := Subgroup( Parent(G), nrm.generators );
                newC.size := Index( G, nrm );
                InfoLattice2( "length = ", Index( G, nrm ), ", \c" );
                nrSubs := nrSubs + Index( G, nrm );

                # add the new extend-by blist (<nrm> - <new>)
                if layer < Length(factors)-1  then
                    nrmE := Elements( nrm );
                    nrmZ := BlistList( zuppos, nrmE );
                    SubtractBlist( nrmZ, newZ );
                    Add( classesB, nrmZ );
                fi;

                # compute the transversal
                trn := RightTransversal( G, nrm );
                InfoLattice2( "tested \c" );

                # loop over the conjugates of <new>
                for t  in trn  do

                    # compute the zuppos blist of the conjugate
                    if t = G.identity  then
                        conE := newE;
                        conZ := newZ;
                    else
                        conE := OnSets( newE, t );
                        conZ := BlistList( zuppos, conE );
                    fi;

                    # loop over the perfect classes
                    for l  in [i+1..Length(perfect)]  do
                        old := perfect[l];
                        oldZ := perfectZ[l];

                        # test if the rep is equal to <con>
                        if conZ = oldZ  then
                            perfect[l] := TrivialSubgroup( G );
                        fi;

                    od;

                od;

                # now we are done with the new class
                InfoLattice2( "inclusions \n" );

            fi; # if IsPerfect(new) and Length(Factors(Size(new))) = layer...
        od; # for new  in perfect  do

        # on to the next layer
        layerb := layere+1;
        layere := Length(classes);

    od; # for layer  in [1..Length(factors)-1]  do ...

    # add the whole group to the list of classes
    InfoLattice1( "#I  doing layer ", Length(factors), ", ",
                  "previous layer has ", layere-layerb+1, " classes\n" );
    newC := ConjugacyClassSubgroups( G, G );
    Add( classes, newC );
    newC.size := 1;
    InfoLattice2( "#I      found whole group, ",
                  "size = ", Size(G), ", ", "length = 1\n" );
    nrSubs := nrSubs + 1;

    # return the list of classes
    InfoLattice1( "#I  <G> has ", Length(classes), " classes, ",
                  "and ", nrSubs, " subgroups\n" );

    # sort the classes
    Sort( classes,
                  function ( c, d )
                     return Size(Representative(c)) < Size(Representative(d))
                        or (Size(Representative(c)) = Size(Representative(d))
                            and Size(c) < Size(d));
                   end );

    # create the lattice
    lattice := rec();
    lattice.isLattice  := true;
    lattice.classes    := classes;
    lattice.group      := G;
    lattice.printLevel := 0;
    lattice.operations := LatticeSubgroupsOps;

    # return the lattice
    return lattice;
end;


#############################################################################
##
#V  LatticeSubgroupOps  . . . . . . . operations record for subgroup lattices
##
LatticeSubgroupsOps := rec();

LatticeSubgroupsOps.Print := function ( L )
    local   i;
    for i  in [1..Length(L.classes)]  do
        PrintClassSubgroupLattice( L, i );
    od;
    Print("LatticeSubgroups( ",L.group," )");
end;

LatticeSubgroupsOps.SetPrintLevel := function ( L, printLevel )
    L.printLevel := printLevel;
end;

PrintClassSubgroupLattice := function ( L, i )
    local   printStuff,         # determines what to print
            rep,                # representative of the <i>-th class
            max,                # maximal subgroup
            min,                # minimal supergroup
            trn,                # transversal of the normalizer
            rep2,               # conjugated maximal subgroup representative
            nrm2,               # its normalizer
            trn2,               # its transversal
            k, l;               # loop variables

    # get the actor
    rep := L.classes[i].representative;

    # select which stuff to print
    if IsBound( L.printStuff )  then
        printStuff := L.printStuff;
    else
        if   not IsBound( L.printLevel )  or L.printLevel = 0  then
            printStuff := [];
        elif L.printLevel = 1  then
            printStuff := [ "class" ];
        elif L.printLevel = 2  then
            printStuff := [ "class", "rep", "repMax" ];
        elif L.printLevel = 3  then
            printStuff := [ "class", "rep", "repMax", "con" ];
        elif L.printLevel = 4  then
            printStuff := [ "class", "rep", "repMax",
                                     "con", "conMax" ];
        else
            printStuff := [ "class", "rep", "repMax", "repMin",
                                     "con", "conMax", "conMin" ];
        fi;
    fi;

    # print the class line
    if "class" in printStuff  then
        Print( "#I  class ",  i,         ", ",
                   "size ",   Size(rep), ", ",
                   "length ", Size(L.classes[i]), "\n" );
    fi;

    # print the representative
    if "rep" in printStuff  then
        Print( "#I    representative ", rep.generators, "\n" );
    fi;

    # print the maximals of the representative
    if "repMax" in printStuff  then
        Print( "#I      maximals \c" );
        for max  in L.operations.MaximalSubgroups( L )[i]  do
            Print( max, " \c" );
        od;
        Print( "\n" );
    fi;

    # print the minimals of the representative
    if "repMin" in printStuff  then
        Print("#I      minimals \c");
        for min  in L.operations.MinimalSupergroups( L )[i]  do
            Print( min, " \c" );
        od;
        Print( "\n" );
    fi;

    # loop over the conjugates
    if   "con"    in printStuff
      or "conMax" in printStuff
      or "conMin" in printStuff
    then

        # get the transversal
        trn := RightTransversal( L.group, Normalizer( L.group, rep ) );
        for k  in [2..Length(trn)]  do

            # print the conjugate
            if "con"  in printStuff  then
                Print( "#I    conjugate ", k, " by ", trn[k], " ",
                       "is ", OnTuples(rep.generators,trn[k]), "\n" );
            fi;

            # print the maximals of the conjugate
            if "conMax" in printStuff  then
                Print("#I      maximals \c");
                for max  in L.operations.MaximalSubgroups( L )[i]  do
                    rep2 := L.classes[max[1]].representative;
                    nrm2 := Normalizer( L.group, rep2 );
                    trn2 := RightTransversal( L.group, nrm2 );
                    l := 1;
                    while not trn2[max[2]] * trn[k] / trn2[l] in nrm2  do
                        l := l + 1;
                    od;
                    Print( [ max[1], l ], " \c" );
                od;
                Print( "\n" );
            fi;

            # print the minimals of the conjugate
            if "conMin" in printStuff  then
                Print("#I      minimals \c");
                for min in L.operations.MinimalSupergroups( L )[i]  do
                    rep2 := L.classes[min[1]].representative;
                    nrm2 := Normalizer( L.group, rep2 );
                    trn2 := RightTransversal( L.group, nrm2 );
                    l := 1;
                    while not trn2[min[2]] * trn[k] / trn2[l] in nrm2  do
                        l := l + 1;
                    od;
                    Print( [ min[1], l ], " \c" );
                od;
                Print( "\n" );
            fi;

        od;

    fi;
                  
end;

LatticeSubgroupsOps.MaximalSubgroups := function ( L )
    local   maxs,               # maximals as pair <class>, <conj> (result)
            mazs,               # their zuppos blist
            classes,            # list of all classes
            classesZ,           # zuppos blist of classes
            classes,            # classes of <G>
            classesZ,           # zuppo lists of <G>
            factors,            # factorization of <G>'s size
            ppowers,            # prime powers dividing <G>'s size
            elements,           # elements of <G>
            zuppos,             # generators of prime power order
            zupposPrime,        # corresponding prime
            zupposPower,        # index of power of generator
            rep,                # representative of a class
            repE,               # elements of <rep>
            repZ,               # zuppos blist of <rep>
            con,                # one conjugate of <rep> (not needed)
            conE,               # elements of <con>
            conZ,               # zuppos of <con>
            old,                # old representative of class in <classes>
            oldE,               # elements of <old> (not needed)
            oldZ,               # zuppos of <old>
            trn,                # transversal of <nrm> in <G>
            cnt,                # count for information messages
            i, k, l, t;         # loop variables

    # maybe we know it already
    if IsBound( L.maximalSubgroups )  then
        return L.maximalSubgroups;
    fi;

    # compute the prime powers dividing <G>'s size (must include 1)
    factors := Factors( Size( L.group ) );
    ppowers := [ 1 ];
    for i  in Collected( factors )  do
        for k  in [1..i[2]]  do
            Add( ppowers, i[1]^k );
        od;
    od;

    # compute the elements of the group
    if IsBound( L.group.elements )  then
        elements := L.group.elements;
    else
        elements := Elements( L.group );
        Unbind( L.group.elements );
    fi;
    InfoLattice1( "#I  <G> has ", Length(elements), " elements, \c" );

    # compute a system of generators for the cyclic sgr. of prime power size
    zuppos := [];
    zupposPower := [];
    for t  in elements  do
        i := Order( L.group, t );
        if i in ppowers and not t in zupposPower  then
            Add( zuppos, t );
            for k  in [2..i-1]  do
                if GcdInt( i, k ) = 1  then
                    AddSet( zupposPower, t^k );
                fi;
            od;
        fi;
    od;
    IsSet( zuppos );
    InfoLattice1( Length(zuppos), " zuppos, \n" );

    # compute the lattice, fetch the classes, zuppos, and representatives
    classes  := L.classes;
    classesZ := [];

    # initialize the maximals list (with whole group)
    InfoLattice1("#I  computing maximal relationship\n");
    maxs := List( classes, c -> [] );
    mazs := List( classes, c -> [] );

    # loop over all classes except the trivial ones
    for i  in [Length(classes),Length(classes)-1..1]  do

        # compute the zuppos blist for this subgroup
        rep  := classes[i].representative;
        InfoLattice2("#I    testing class ",i,", size ",Size(rep),", \c");
        if not IsBound( rep.elements )  then
            repE := Elements( rep );
            Unbind( rep.elements );
        else
            repE := rep.elements;
        fi;
        repZ := BlistList( zuppos, repE );
        classesZ[i] := repZ;

        # compute the transversal of the normalizer of the representative
        if not IsBound( rep.normalizer )  then
            trn := RightTransversal( L.group, Normalizer( L.group, rep ) );
            Unbind( rep.normalizer );
        else
            trn := RightTransversal( L.group, Normalizer( L.group, rep ) );
        fi;
        InfoLattice2("length ",Length(trn),", \c");
        cnt := 0;

        # loop over the conjugates of <rep>
        for k  in [1..Length(trn)]  do
            t := trn[k];

            # compute the zuppos blist of the conjugate
            if t = L.group.identity  then
                conE := repE;
                conZ := repZ;
            else
                conE := OnSets( repE, t );
                conZ := BlistList( zuppos, conE );
            fi;

            # loop over all other (larger classes)
            for l  in [i+1..Length(classes)]  do
                old  := classes[l].representative;
                oldZ := classesZ[l];

                # test if the rep is a subgroup of <con>
                if IsSubsetBlist( oldZ, conZ )
                  and ForAll( mazs[l], z -> not IsSubsetBlist( z, conZ ) )
                then
                    Add( maxs[l], [ i, k ] );
                    Add( mazs[l], conZ );
                    cnt := cnt + 1;
                fi;

            od;

        od;

        # add this to the table of marks
        InfoLattice2("maximals in ",cnt," representatives\n");

    od;

    # return the maximals list
    L.maximalSubgroups := maxs;
    return maxs;
end;

LatticeSubgroupsOps.MinimalSupergroups := function ( L )
    local   mins,               # minimals as pair <class>, <conj> (result)
            mizs,               # their zuppos blist
            classes,            # list of all classes
            classesZ,           # zuppos blist of classes
            classes,            # classes of <G>
            classesZ,           # zuppo lists of <G>
            factors,            # factorization of <G>'s size
            ppowers,            # prime powers dividing <G>'s size
            elements,           # elements of <G>
            zuppos,             # generators of prime power order
            zupposPrime,        # corresponding prime
            zupposPower,        # index of power of generator
            rep,                # representative of a class
            repE,               # elements of <rep>
            repZ,               # zuppos blist of <rep>
            con,                # one conjugate of <rep> (not needed)
            conE,               # elements of <con>
            conZ,               # zuppos of <con>
            old,                # old representative of class in <classes>
            oldE,               # elements of <old> (not needed)
            oldZ,               # zuppos of <old>
            trn,                # transversal of <nrm> in <G>
            cnt,                # count for information messages
            i, k, l, t;         # loop variables

    # maybe we know it already
    if IsBound( L.minimalSupergroups )  then
        return L.minimalSupergroups;
    fi;

    # compute the prime powers dividing <G>'s size (must include 1)
    factors := Factors( Size( L.group ) );
    ppowers := [ 1 ];
    for i  in Collected( factors )  do
        for k  in [1..i[2]]  do
            Add( ppowers, i[1]^k );
        od;
    od;

    # compute the elements of the group
    if IsBound( L.group.elements )  then
        elements := L.group.elements;
    else
        elements := Elements( L.group );
        Unbind( L.group.elements );
    fi;
    InfoLattice1( "#I  <G> has ", Length(elements), " elements, \c" );

    # compute a system of generators for the cyclic sgr. of prime power size
    zuppos := [];
    zupposPower := [];
    for t  in elements  do
        i := Order( L.group, t );
        if i in ppowers and not t in zupposPower  then
            Add( zuppos, t );
            for k  in [2..i-1]  do
                if GcdInt( i, k ) = 1  then
                    AddSet( zupposPower, t^k );
                fi;
            od;
        fi;
    od;
    IsSet( zuppos );
    InfoLattice1( Length(zuppos), " zuppos, \n" );

    # compute the lattice, fetch the classes, zuppos, and representatives
    classes  := L.classes;
    classesZ := [];

    # initialize the maximals list (with whole group)
    InfoLattice1("#I  computing maximal relationship\n");
    mins := List( classes, c -> [] );
    mizs := List( classes, c -> [] );

    # loop over all classes except the trivial ones
    for i  in [1..Length(classes)]  do

        # compute the zuppos blist for this subgroup
        rep  := classes[i].representative;
        InfoLattice2("#I    testing class ",i,", size ",Size(rep),", \c");
        if not IsBound( rep.elements )  then
            repE := Elements( rep );
            Unbind( rep.elements );
        else
            repE := rep.elements;
        fi;
        repZ := BlistList( zuppos, repE );
        classesZ[i] := repZ;

        # compute the transversal of the normalizer of the representative
        if not IsBound( rep.normalizer )  then
            trn := RightTransversal( L.group, Normalizer( L.group, rep ) );
            Unbind( rep.normalizer );
        else
            trn := RightTransversal( L.group, Normalizer( L.group, rep ) );
        fi;
        InfoLattice2("length ",Length(trn),", \c");
        cnt := 0;

        # loop over the conjugates of <rep>
        for k  in [1..Length(trn)]  do
            t := trn[k];

            # compute the zuppos blist of the conjugate
            if t = L.group.identity  then
                conE := repE;
                conZ := repZ;
            else
                conE := OnSets( repE, t );
                conZ := BlistList( zuppos, conE );
            fi;

            # loop over all other (larger classes)
            for l  in [1..i-1]  do
                old  := classes[l].representative;
                oldZ := classesZ[l];

                # test if the rep is a subgroup of <con>
                if IsSubsetBlist( conZ,oldZ )
                  and ForAll( mizs[l], z -> not IsSubsetBlist( conZ, z ) )
                then
                    Add( mins[l], [ i, k ] );
                    Add( mizs[l], conZ );
                    cnt := cnt + 1;
                fi;

            od;

        od;

        # add this to the table of marks
        InfoLattice2("minimals over ",cnt," representatives\n");

    od;

    # return the minimals list
    L.minimalSupergroups := mins;
    return mins;
end;


#############################################################################
##
#F  ConjugacyClassesPerfectSubgroups(<G>) . . .  classes of perfect subgroups
##
##  'ConjugacyClassesPerfectSubgroups' returns   the  conjugacy  classes   of
##  perfect subgroups of the finite group <G>.
##
ConjugacyClassesPerfectSubgroups := function ( G )

    # check the argument
    if not IsGroup( G )  then
        Error("<G> must be a group");
    fi;

    # compute the classes of perfect subgroups
    if not IsBound( G.conjugacyClassesPerfectSubgroups )  then
        G.conjugacyClassesPerfectSubgroups :=
          G.operations.ConjugacyClassesPerfectSubgroups( G );
    fi;

    # return the classes of perfect subgroups
    return G.conjugacyClassesPerfectSubgroups;

end;

GroupOps.ConjugacyClassesPerfectSubgroups := function ( G )
    local   classes,            # classes, result
            rep;                # representative

    # compute the 
    classes := [];
    for rep  in RepresentativesPerfectSubgroups( G )  do
        if not ForAny( classes, c -> rep in c )  then
            Add( classes, ConjugacyClassSubgroups( G, rep ) );
        fi;
    od;

    # return the classes
    return classes;

end;


#############################################################################
##
#F  RepresentativesPerfectSubgroups(<G>)  . . . . at least one representative
##
##  'RepresentativesPerfectSubgroups' returns  a list that contains  at least
##  one  subgroup in each conjugacy  class of perfect  subgroups of the group
##  <G>.
##
RepresentativesPerfectSubgroups := function ( G )

    # check the argument
    if not IsGroup( G )  then
        Error("<G> must be a group");
    fi;

    # compute a representative for each class
    if not IsBound( G.representativesPerfectSubgroups )  then
        G.representativesPerfectSubgroups :=
          G.operations.RepresentativesPerfectSubgroups( G );
    fi;

    # return the representatives
    return G.representativesPerfectSubgroups;

end;

GroupOps.RepresentativesPerfectSubgroups := function ( G )
    local   reps,               # representatives, result
            rep,                # one representative
            ders,               # derived series of <G>
            resd,               # solvable residuum of <G>
            perc,               # perfect group from the catalogue
            perf,               # converted into a real group
            subc,               # subgroup record of <perc>
            subf,               # corresponding subgroup of <perf>
            iso;                # isomorphism between <perf> and <resd>

    # compute the solvable residuum of <G>
    ders := DerivedSeries( G );
    resd := ders[Length(ders)];

    # easy case first
    if resd.generators = []  then
        return [];
    fi;

    # otherwise check through the perfect groups catalogue
    for perc  in PerfectGroupsCatalogue  do

        # make the perfect group (should fix up the catalogue)
        perf := Group( perc.generators, IdWord );
        perf.relators               := perc.relations;
        perf.isPerfect              := true;
        perf.size                   := perc.size;
        perf.PGantirelators         := perc.antirelations;
        perf.PGinvariants           := perc.grouptype;
        perf.PGinvariantsGenerators := perc.generatortype;
        perf.representativesPerfectSubgroups := [];
        for subc  in perc.subgroups  do
            subf := Subgroup( perf, subc.generators );
            subf.size      := subc.size;
            subf.isPerfect := true;
            Add( perf.representativesPerfectSubgroups, subf );
        od;

        # try to compute an isomorphism
        iso := IsomorphismPerfectGroup( perf, resd );

        # if this worked, them map the subgroups
        if iso <> false  then
            InfoLattice1("#I  identified type of solvable residuum\n");
            reps := [ resd ];
            for subf  in perf.representativesPerfectSubgroups  do
                rep := Image( iso, subf );
                rep.isPerfect := true;
                rep.size      := subf.size;
                Add( reps, rep );
            od;
            return reps;
        fi;

    od;

    # bad luck for the user
    Error("sorry, I cannot identify the solvable residuum of <G>");

end;


#############################################################################
##
#F  IsomorphismPerfectGroup(<P>,<R>)  . . . . .  try to construct isomorphism
##
##  'IsomorphismPerfectGroup' returns an  isomorphism from  the perfect group
##  <P> from the perfect groups  catalogue to the concrete  group <R>.  If no
##  such isomorphism exists, then 'IsomorphismPerfectGroup' returns 'false'.
##
IsomorphismPerfectGroupHelp := function ( P, R, imgs, C )
    local   iso,                # isomorphism, result
            rel,                # relator or antirelator
            inv,                # invariants of the next generator
            cls,                # possible class of next generator
            orb;                # orbit in <cls> under <C>

    # if we have choosen all images, test that this is a homorphism
    if Length(P.generators) = Length(imgs)  then

        # test if the mapping is a homomorphism
        for rel  in P.relators  do
            if MappedWord( rel, P.generators, imgs ) <> R.identity  then
                return false;
            fi;
        od;

        # test if the mapping is bijection
        for rel  in P.PGantirelators  do
            if MappedWord( rel, P.generators, imgs )  = R.identity  then
                return false;
            fi;
        od;

        # found the isomorphism
        iso := GroupHomomorphismByImages( P, R, P.generators, imgs );
        iso.isMapping      := true;
        iso.isHomomorphism := true;
        return iso;

    else

        # we have some information
        inv := P.PGinvariantsGenerators[Length(imgs)+1];

        # loop over the conjugacy classes
        for cls  in ConjugacyClasses(R)  do

            # only take those of the correct order and length
            if    Order( R, Representative(cls) ) = inv[1]
              and Size(cls) = inv[4]
            then

                # try the representatives under the operation of <C>
                for orb  in Orbits( C, Elements(cls) )  do

                    # only take those that have the correct length
                    if Length( orb ) = inv[2]  then

                        # try to extend the mapping
                        iso := IsomorphismPerfectGroupHelp( P, R,
                                   Concatenation( imgs, [orb[1]] ),
                                   Centralizer( C, orb[1] ) );
                        if iso <> false  then
                            return iso;
                        fi;

                    fi;

                od;

            fi;

        od;

        # no possible extension found
        return false;

    fi;

end;

IsomorphismPerfectGroup := function ( P, R )
    local   inv;                # invariants of the group <P>

    # compare the size of <P> and the group <R>
    if P.size <> Size(R)  then
        return false;
    fi;

    # test all the stored invariants
    for inv  in P.PGinvariants do

        # nr of elements of order inv[2] is inv[3]
        if inv[1] = 1
          and Sum(    Filtered( ConjugacyClasses( R ),
                                c -> Order(R,Representative(c)) = inv[2] ),
                      Size ) <> inv[3]
        then
            return false;

        # nr of <G>-classes of order inv[2] and size inv[3] is inv[4]
        # elif inv[1] = 2
        # then

        # nr of <R>-classes of order inv[2] and size inv[3] is inv[4]
        elif inv[1] = 3
          and Number( Filtered( ConjugacyClasses( R ),
                                c -> Order(R,Representative(c)) = inv[2] ),
                      c -> Size(c) = inv[3] ) <> inv[4]
        then
            return false;

        # nr of elms of order inv[2] with roots of order inv[3] is inv[4]
        # elif inv[1] = 4
        # then

        fi;

    od;

    # try to create an isomorphism
    return IsomorphismPerfectGroupHelp( P, R, [], R );

end;


#############################################################################
##
#F  NormalSubgroups(<G>)  . . . . . . . . . . . . normal subgroups of a group
##
NormalSubgroups := function ( G )

    # check that the argument is a group
    if not IsGroup( G )  then
        Error( "<G> must be a group" );
    fi;
    InfoGroup1( "#I  NormalSubgroups: ", GroupString(G,"G"), "\n" );

    # compute the normal subgroups
    if not IsBound( G.normalSubgroups )  then
        G.normalSubgroups := G.operations.NormalSubgroups( G );
    fi;

    # return the normal subgroups
    InfoGroup1("#I  NormalSubgroups: returns ",
                Length(G.normalSubgroups)," subgroups\n");
    return G.normalSubgroups;

end;

GroupOps.NormalSubgroups := function ( G )
    local   nrm;        # normal subgroups of <G>, result

    # compute the normal subgroup lattice above the trivial subgroup
    nrm := G.operations.NormalSubgroupsAbove(G,TrivialSubgroup(G),[]);

    # sort the normal subgroups according to their size
    Sort( nrm, function( a, b ) return Size( a ) < Size( b ); end );

    # and return it
    return nrm;

end;

GroupOps.NormalSubgroupsAbove := function ( G, N, avoid )
    local   R,          # normal subgroups above <N>, result
            C,          # one conjugacy class of <G>
            g,          # representative of a conjugacy class of <G>
            M;          # normal closure of <N> and <g>

    # initialize the list of normal subgroups
    InfoGroup1("#I    normal subgroup of order ",Size(N),"\n");
    R := [ N ];

    # make a shallow copy of avoid, because we are going to change it
    avoid := ShallowCopy( avoid );

    # for all representative that need not be avoided and do not ly in <N>
    for C  in ConjugacyClasses( G )  do
        g := Representative( C );

        if not g in avoid  and not g in N  then

            # compute the normal closure of <N> and <g> in <G>
            M := NormalClosure( G, Closure( N, g ) );
            if ForAll( avoid, rep -> not rep in M )  then
                Append( R, G.operations.NormalSubgroupsAbove(G,M,avoid) );
            fi;

            # from now on avoid this representative
            Add( avoid, g );
        fi;
    od;

    # return the list of normal subgroups
    return R;

end;


#############################################################################
##
#F  LatticeNormalSubgroups(<G>) . . . . . . . . . lattice of normal subgroups
##
##  'LatticeNormalSubgroups' is not yet implemented.
##


#############################################################################
##
#F  GroupOps.TableOfMarks(<G>)  . . . . . . . . . . . . make a table of marks
##
GroupOps.TableOfMarks := function ( G )
    local   tom,                # table of marks (result)
            mrk,                # marks for one class
            classes,            # list of all classes
            classesZ,           # zuppos blist of classes
            classes,            # classes of <G>
            classesZ,           # zuppo lists of <G>
            factors,            # factorization of <G>'s size
            ppowers,            # prime powers dividing <G>'s size
            elements,           # elements of <G>
            zuppos,             # generators of prime power order
            zupposPrime,        # corresponding prime
            zupposPower,        # index of power of generator
            rep,                # representative of a class
            repE,               # elements of <rep>
            repZ,               # zuppos blist of <rep>
            con,                # one conjugate of <rep> (not needed)
            conE,               # elements of <con>
            conZ,               # zuppos of <con>
            old,                # old representative of class in <classes>
            oldE,               # elements of <old> (not needed)
            oldZ,               # zuppos of <old>
            trn,                # transversal of <nrm> in <G>
            ind,                # index of <rep> in its normalizer
            i, k, l, t;         # loop variables

    # compute the prime powers dividing <G>'s size (must include 1)
    factors := Factors( Size( G ) );
    ppowers := [ 1 ];
    for i  in Collected( factors )  do
        for k  in [1..i[2]]  do
            Add( ppowers, i[1]^k );
        od;
    od;

    # compute the elements of the group
    if IsBound( G.elements )  then
        elements := G.elements;
    else
        elements := Elements( G );
        Unbind( G.elements );
    fi;
    InfoLattice1( "#I  <G> has ", Length(elements), " elements, \c" );

    # compute a system of generators for the cyclic sgr. of prime power size
    zuppos := [];
    zupposPower := [];
    for t  in elements  do
        i := Order( G, t );
        if i in ppowers and not t in zupposPower  then
            Add( zuppos, t );
            for k  in [2..i-1]  do
                if GcdInt( i, k ) = 1  then
                    AddSet( zupposPower, t^k );
                fi;
            od;
        fi;
    od;
    IsSet( zuppos );
    InfoLattice1( Length(zuppos), " zuppos, \n" );

    # compute the lattice, fetch the classes, zuppos, and representatives
    classes  := ConjugacyClassesSubgroups( G );
    classesZ := [];

    # initialize the table of marks
    InfoLattice1("#I  computing table of marks\n");
    tom := rec( subs := List( classes, c -> [] ),
                marks := List( classes, c -> [] ) );

    # loop over all classes except the trivial ones
    for i  in [1..Length(classes)]  do

        # compute the zuppos blist for this subgroup
        rep  := classes[i].representative;
        InfoLattice2("#I    testing class ",i,", size ",Size(rep),", \c");
        if not IsBound( rep.elements )  then
            repE := Elements( rep );
            Unbind( rep.elements );
        else
            repE := rep.elements;
        fi;
        repZ := BlistList( zuppos, repE );
        classesZ[i] := repZ;

        # compute the transversal of the normalizer of the representative
        if not IsBound( rep.normalizer )  then
            trn := RightTransversal( G, Normalizer( G, rep ) );
            ind := Index( Normalizer( G, rep ), rep );
            Unbind( rep.normalizer );
        else
            trn := RightTransversal( G, Normalizer( G, rep ) );
            ind := Index( Normalizer( G, rep ), rep );
        fi;
        InfoLattice2("length ",Length(trn),", \c");

        # set up the marking list
        mrk    := 0 * [1..Length(classes)];
        mrk[1] := Length(trn) * ind;
        mrk[i] := 1 * ind;

        # loop over the conjugates of <rep>
        for k  in [1..Length(trn)]  do
            t := trn[k];

            # compute the zuppos blist of the conjugate
            if t = G.identity  then
                conE := repE;
                conZ := repZ;
            else
                conE := OnSets( repE, t );
                conZ := BlistList( zuppos, conE );
            fi;

            # loop over all other (smaller classes)
            for l  in [2..i-1]  do
                old  := classes[l].representative;
                oldZ := classesZ[l];

                # test if the rep is a subgroup of <con>
                if IsSubsetBlist( conZ, oldZ )  then
                    mrk[l] := mrk[l] + ind;
                fi;

            od;

        od;

        # compress this line into the table of marks
        for l  in [1..i]  do
            if mrk[l] <> 0  then
                Add( tom.subs[i], l );
                Add( tom.marks[i], mrk[l] );
            fi;
        od;
        InfoLattice2("includes ",Length(tom.marks[i])," classes\n");

    od;

    # return the table of marks
    return tom;
end;


