#############################################################################
##
#F  DecodeGroup(p, <desc>) . . . . .  decode compact description of a p-group
##
##  Alice Niemeyer, E.A O'Brien and Colin Rhodes
##  School of Mathematical Sciences, ANU
##  Last revised February 1993
##
##  'DecodeGroup' creates a p-group from the description <desc>.
##
##  Each p-group is compactly described by a list of 4 integers.  The 
##  integers have the following meaning:
##       1st integer: The rank of the Frattini quotient of group
##       2nd integer: The number of pcp generators of the group 
##       3rd integer: The exponent-p class of the group
##       4th integer: A long integer encoding the pc-presentation
##
##  The pc-presentation is encoded in the following way. 
##  Let $G$ be a p-group described by a pc-presentation on 
##  $n$ generators,  $g_1, \ldots, g_n$, and with relations
##  $g_i^p = g_{i+1}^{e^{i,i}_{i+1}} \cdots g_n^{e^{i,i}_n}$ for 
##  $1 \le i \le n$ and the relations 
##  $[g_i, g_j]  = g_{i+1}^{e^{i,j}_{i+1}} \cdots g_n^{e^{i,j}_n}$ for 
##  $1 \le j < i \le n$. 
##  Since the group is a p-group, all exponents, $e^{i,j}_k$, lie in 
##  $0,..,p - 1$.  We order the exponents occuring on the right hand sides
##  of the relations as a sequence. We view the sequence as the p-adic 
##  expansion of an integer and use this integer to encode the presentation. 
##
DecodeGroup := function ( p, desc )
    local   grp,        # group described by 'desc', result
            gens,       # generators of the group
            rels,       # relators of the group
            exps,       # exponent bit string
            rhs,        # right hand sides of the relators
            i, k, l;    # loop variables

    # make the generators
    gens := [];
    for i  in [1..desc[2]]  do
        gens[i] := AbstractGenerator(ConcatenationString("a",StringInt(i)));
    od;

    # make the right hand sides of the relations
    exps := desc[4];
    rhs := [];
    for i  in [1..desc[2]]  do
        rhs[i] := [];
        for k  in [1..i-1]  do
            rhs[i][k] := gens[1]^0;
            for l  in [k..i-1]  do
                rhs[l][k] := rhs[l][k] * gens[i]^(exps mod p);
                exps := QuoInt( exps, p );
            od;
        od;
        rhs[i][i] := gens[1]^0;
    od;

    # make the relators
    rels := [];
    for i  in [1..desc[2]]  do
        Add( rels, gens[i]^p/rhs[i][i] );
        for k  in [1..i-1]  do
            Add( rels, Comm(gens[i],gens[k])/rhs[i][k] );
        od;
    od;

    # make the finitely presented group
    grp := Group( gens, gens[1]^0 );
    grp.relators := rels;

    # make the finite polycyclic group and enter rank and pclass
    grp := AgGroupFpGroup( grp );
    grp.rank               := desc[1];
    grp.pclass             := desc[3];
    grp.abstractGenerators := gens;
    grp.abstractRelators   := rels;

    # return the finite polycyclic group
    return grp;

end; #DecodeGroup
