;; Copyright (c) 1990-1994 The MITRE Corporation
;; 
;; Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;;   
;; The MITRE Corporation (MITRE) provides this software to you without
;; charge to use, copy, modify or enhance for any legitimate purpose
;; provided you reproduce MITRE's copyright notice in any copy or
;; derivative work of this software.
;; 
;; This software is the copyright work of MITRE.  No ownership or other
;; proprietary interest in this software is granted you other than what
;; is granted in this license.
;; 
;; Any modification or enhancement of this software must identify the
;; part of this software that was modified, by whom and when, and must
;; inherit this license including its warranty disclaimers.
;; 
;; MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;; OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;; OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;; FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;; SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;; SUCH DAMAGES.
;; 
;; You, at your expense, hereby indemnify and hold harmless MITRE, its
;; Board of Trustees, officers, agents and employees, from any and all
;; liability or damages to third parties, including attorneys' fees,
;; court costs, and other related costs and expenses, arising out of your
;; use of this software irrespective of the cause of said liability.
;; 
;; The export from the United States or the subsequent reexport of this
;; software is subject to compliance with United States export control
;; and munitions control restrictions.  You agree that in the event you
;; seek to export this software or any derivative work thereof, you
;; assume full responsibility for obtaining all necessary export licenses
;; and approvals and for assuring compliance with applicable reexport
;; restrictions.
;; 
;; 
;; COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994

(defun filename-basename ()
  (let ((fn (if (buffer-file-name)
		(file-name-nondirectory (buffer-file-name))
	      (buffer-name))))
    (substring
     fn
     0
     (string-match "\\." fn))))

(defun create-schemeweb-file ()
  (let ((path
	 (format "%s%s.sw"
		 (file-name-directory (buffer-file-name))
		 (filename-basename))))
    (and (file-exists-p path)
	 (or (y-or-n-p
	      (format "Overwrite file [%s]? " path))
	     (error "Schemeweb file already exists")))
    (write-file path)))

(defconst sw-special-characters '(?^ ?_ ?$ ?# ?%))


(defun sw-protect-special-characters (str)
  (let ((l nil)
	(i 0)
	(len (length str)))
    (while (< i len)
      (let ((ch (aref str i)))
	(if (memq ch sw-special-characters)
	    (setq l (cons ch (cons ?\\ l)))
	  (setq l (cons ch l)))
	(setq i (1+ i))))
    (concat (reverse l))))

(defun sw-omit-special-characters (str)
  (let ((l nil)
	(i 0)
	(len (length str)))
    (while (< i len)
      (let ((ch (aref str i)))
	(and (not (memq ch sw-special-characters))
	     (setq l (cons ch l)))
	(setq i (1+ i))))
    (concat (reverse l))))
  

(defvar sw-omit-def-form-pattern
  "^(def-\\(renamer\\|theory-ensemble-overloadings\\|parse-syntax\\|print-syntax\\|overloading\\)"
  "Defuns starting with a match to this pattern are removed from the Schemeweb file.")

(defvar sw-make-figure-pattern
  "^(def-\\(atomic-sort\\|constant\\|recursive-constant\\|quasi-constructor\\|theorem\\|inductor\\|schematic-macete\\)"
  "Figures are made for defuns starting with a match to this pattern.
The figure is constructed by xviewing the result of ")

(defconst sw-form-action-alist
  '(("def-renamer" 		      . sw-omit-def-form)
    ("def-theory-ensemble-overloadings" . sw-omit-def-form)
    ("def-parse-syntax"		      . sw-omit-def-form)
    ("def-print-syntax"		      . sw-omit-def-form)
    ("def-overloading"		      . sw-omit-def-form)
    ("def-imported-rewrite-rules"     . sw-omit-def-form)
    ("def-record-theory"	      . sw-omit-def-form)
    ("def-renamer"		      . sw-omit-def-form)
    ("def-transported-symbols"	      . sw-omit-def-form)
    ("def-atomic-sort"		      . sw-process-standard-def-form)
    ("def-constant"		      . sw-process-standard-def-form)
    ("def-recursive-constant"	      . sw-process-standard-def-form)
    ("def-quasi-constructor"	      . sw-process-standard-def-form)
    ("def-theorem"		      . sw-process-standard-def-form)
    ("def-inductor"		      . sw-process-standard-def-form)
    ("def-schematic-macete"	      . sw-process-standard-def-form)
    ("def-bnf"			      . sw-process-bnf-def-form)
    ("def-language"		      . sw-process-language-def-form)
    ("def-theory"		      . sw-process-theory-def-form)
    ("def-translation"		      . sw-process-translation-def-form)))

(defun sw-process-file ()
  (interactive)
  (if (not (y-or-n-p "Have you already loaded this file into Imps? "))
      (error "Please load the file into Imps.")
    (message "Starting Schemeweb on file...")
    (goto-char (point-min))
    (while (re-search-forward "^(def-" nil 1)
      (let* ((kind (next-symbol-string (match-beginning 0)))
	     (fun (cdr (assoc kind sw-form-action-alist))))
	(if (not (fboundp fun))
	    (goto-char (scan-sexps (match-beginning 0) 1))
	  (goto-char (match-beginning 0))
	  (funcall fun (point))))
      (sit-for 1))
    (message "Sent to Imps, retrieving tex output...")
    (sw-insert-imps-tex-output)
    (save-buffer)
    (message "Starting Schemeweb on file... done")))



(defun sw-omit-def-form (here)
  "Use latex iffalse to cause the current def-form to be omitted."
  (and
   (re-search-forward "^(def-" nil t)
   (progn
     (goto-char (match-beginning 0))
     (insert "\\iffalse%\n")
     (goto-char (scan-sexps (point) 1))
     (insert "\n\\fi%\n"))))
    
     

(defvar sw-omit-proofs t)

(defun sw-omit-trivial-def-forms ()
  (interactive)
  (goto-char (point-min))
  (while (re-search-forward sw-omit-def-form-pattern nil 1)
    (goto-char (match-beginning 0))
    (insert "\\iffalse%\n")
    (goto-char (scan-sexps (point) 1))
    (insert "\n\\fi%\n")))

(defun sw-omit-proofs ()
  (interactive)
  (goto-char (point-min))
  (save-restriction
    (while (re-search-forward "^(def-theorem" nil t)
      (let ((start (match-beginning 0))
	    (end   (scan-sexps (match-beginning 0) 1)))
	(and (search-forward "(proof" end t)
	     (goto-char (match-beginning 0))
	     (insert "\\iffalse%\n  ")
	     (goto-char (scan-sexps (match-beginning 0) 1))
	     (insert "\n\\fi%\n"))))))


(defvar sw-figure-index 0
  "Number to put in place of NN in /tmp/$USER-imps-NN.tex when requesting Imps construct 
the next figure.")

(defvar sw-figure-markers '()
  "Alist of figure indices and markers to insert resulting imps tex output.")


(defun sw-get-kind-name-text-and-theory-name (here)
  "Return list (kind name text theory-name) for the current standard def-form."
  (let ((parse-sexp-ignore-comments t)
	kind name text theory-name)
    (save-excursion 
      (goto-char here)
      (and
       (re-search-forward "^(def-" nil t)
       (progn
	 (goto-char (match-beginning 0))
	 (looking-at sw-make-figure-pattern))
       (let ((def-form-end (scan-sexps (match-beginning 0) 1)))
	 (setq kind (buffer-substring (match-beginning 1)(match-end 1)))
	 (goto-char (match-end 1))
	 (cond ((looking-at "\\s *\\((\\s *)\\)")
		(setq name "Anonymous"))
	       (t
		(setq name (next-symbol-string (point)))))
	 (forward-sexp 2)		;skip name and text
	 (backward-sexp 1)		;beginning of text
	 (and
	  (= (char-after (point)) 34)	; Ascii for double quote char
	  (setq	
	   text				;omit quotes 
	   (buffer-substring (1+ (point))
			     (1- (scan-sexps (point) 1)))))
	 (forward-sexp 1)		;skip contents
	 (save-excursion
	   (and (search-forward "(theory" def-form-end)
		(setq
		 theory-name
		 (sw-protect-special-characters (next-symbol-string (point))))))
	 (while (and (string= kind "theorem")
		     (not (last-list-item-p (point))))
	   (if (string= (next-sexp-as-string) "lemma")
	       (setq kind "lemma")
	     (forward-sexp 1)))
	 (list kind name text theory-name))))))

(defun sw-process-standard-def-form (here)
  "Replace the def-form following HERE with a latex theorem-like environment."
  (interactive "d")
  (let ((kind-name-text-and-theory-name 
	 (sw-get-kind-name-text-and-theory-name here))
	(new-marker (make-marker)))
    (and 
     kind-name-text-and-theory-name
     (let ((kind        (nth 0 kind-name-text-and-theory-name))
	   (name        (nth 1 kind-name-text-and-theory-name))
	   (text        (nth 2 kind-name-text-and-theory-name))
	   (theory-name (nth 3 kind-name-text-and-theory-name)))
       (if (not text)
	   (goto-char (scan-sexps here 1))
	 (insert
	  (format "
\\begin{%s}
{\\bf (%s)}
\\label{%s:%s}
Theory: %s

%%%% Contents of %s to be inserted here.

"
		  (sw-omit-special-characters kind)
		  (sw-protect-special-characters name)
		  (sw-omit-special-characters kind)
		  (sw-omit-special-characters name)
		  theory-name
		  (substitute-in-file-name
		   (format
		    "/tmp/$USER-imps-%d.tex"
		    sw-figure-index))))
	 (set-marker new-marker (point)(current-buffer))
	 (setq sw-figure-markers
	       (cons
		(cons new-marker sw-figure-index)
		sw-figure-markers))
	 (insert
	  (format
	   "

\\end{%s}
"
	   (sw-omit-special-characters kind)))
	 (sw-send-string theory-name text)
	 (re-search-forward "^(def-" nil t)
	 (goto-char (match-beginning 0))
	 (insert "
\\iffalse% 
")
	 (forward-sexp 1)
	 (insert "
\\fi% 
"
		 ))))))


(defun sw-process-theory-def-form (here)
  "Construct a latex environment and a figure of the axioms 
of the theory def-form after HERE."
  (interactive "d")
  (and
   (re-search-forward "^(def-" nil t)
   (progn
     (goto-char (match-beginning 0))
     (looking-at "^(def-theory"))
   (let* ((theory-start (point))
	  (theory-end (scan-sexps (point) 1))
	  (theory-end-marker
	   (set-marker
	    (make-marker)
	    theory-end)))
     (down-list 1)
     (forward-sexp 2)
     (backward-sexp 1)			;point at start of theory name
     (let ((theory-name (next-symbol-string (point)))
	   (fig-marker (make-marker))
	   (env-marker (make-marker)))
       (and
	(re-search-forward "(component-theories\\|(axioms" theory-end t)
	(let ((label
	       (format "fig:theory:%s" (sw-omit-special-characters theory-name))))
	  (goto-char here)
	  (insert
	   (format
	    "\n\\begin{figure}
\\begin{center}
\\fbox{\\begin{minipage}{4.5in}
%%%% Contents of %s to be inserted here.\n\n"
	    (substitute-in-file-name
	     (format "/tmp/$USER-imps-%d.tex" sw-figure-index))))
	  (set-marker fig-marker (point)(current-buffer))
	  (setq sw-figure-markers
		(cons
		 (cons fig-marker sw-figure-index)
		 sw-figure-markers))
	  (sw-send-theory-axioms theory-name)
	  (insert
	   (format
	    "\n\n\\end{minipage}}\n\\end{center}
\\caption{Components and axioms for %s}\n\\label{%s}
\\end{figure}\n\n"
	    (sw-protect-special-characters theory-name)
	    label))
	  (insert
	   (format "\n\\begin{theory}\n{\\bf (%s)}\n\\label{theory:%s} \n \n
%%%% Contents of %s to be inserted here.\n\n"
		   (sw-protect-special-characters theory-name)
		   (sw-omit-special-characters theory-name)
		   (substitute-in-file-name
		    (format "/tmp/$USER-imps-%d.tex" sw-figure-index))))
	  (set-marker env-marker (point) (current-buffer))
	  (setq sw-figure-markers
		(cons
		 (cons env-marker sw-figure-index)
		 sw-figure-markers))
	  (sw-send-theory-env theory-name label)
	  (insert
	   (format "\n\\end{theory}\n\n"))
	  (re-search-forward "^(def-" nil t)
	 (goto-char (match-beginning 0))
	 (insert "\n\\iffalse% \n")
	 (forward-sexp 1)
	 (insert "\n\\fi% \n")))))))
	       

(defun sw-send-theory-axioms (theory-name)
  (process-send-string
   tea-process
   (format
    "(xview-theory-for-figure '%s \"%s\")\n"
    theory-name  
    (substitute-in-file-name
     (format
      "/tmp/$USER-imps-%d.tex"
      sw-figure-index))))
  (setq sw-figure-index (1+ sw-figure-index)))


(defun sw-send-theory-env (theory-name label)
  (process-send-string
   tea-process
   (format
    "(xview-theory-for-env '%s \"%s\" \"%s\")\n"
    theory-name  
    (substitute-in-file-name
     (format
      "/tmp/$USER-imps-%d.tex"
      sw-figure-index))
    label))
  (setq sw-figure-index (1+ sw-figure-index)))



(defun sw-send-string (theory-name str)
  (process-send-string
   tea-process
   (format
    "(xview-figure '%s \"%s\" \"%s\")\n"
    theory-name str 
    (substitute-in-file-name
     (format
      "/tmp/$USER-imps-%d.tex"
      sw-figure-index))))
  (setq sw-figure-index (1+ sw-figure-index)))

(defun sw-process-bnf-def-form (here)
  "Create a latex environment for the BNF def form following HERE."
  (interactive "d")
  (and
   (re-search-forward "^(def-" nil t)
   (progn
     (goto-char (match-beginning 0))
     (looking-at "^(def-bnf"))
   (let ((bnf-start (point)))
     (down-list 1)
     (forward-sexp 2)
     (backward-sexp 1)			;point at start of BNF name
     (let ((bnf-name (next-symbol-string (point)))
	   (env-marker (make-marker)))
       (goto-char here)
       (insert
	(format
	 "\n\\begin{bnf}\n{\\bf (%s)}\n\\label{bnf:%s}\n \n 
%%%% Contents of %s to be inserted here.\n\n"
	 (sw-protect-special-characters bnf-name)
	 (sw-omit-special-characters bnf-name)
	 (substitute-in-file-name
	  (format "/tmp/$USER-imps-%d.tex" sw-figure-index))))
       (set-marker env-marker (point) (current-buffer))
       (setq sw-figure-markers
	     (cons
	      (cons env-marker sw-figure-index)
	      sw-figure-markers))
       (sw-send-bnf-env bnf-name)
       (insert
	(format "\n\\end{bnf}\n\n"))
       (re-search-forward "^(def-" nil t)
       (goto-char (match-beginning 0))
       (insert "\n\\iffalse% \n")
       (forward-sexp 1)
       (insert "\n\\fi% \n")))))



(defun sw-send-bnf-env (bnf-name)
  (process-send-string
   tea-process
   (format
    "(xview-bnf-for-env '%s \"%s\")\n"
    bnf-name  
    (substitute-in-file-name
     (format
      "/tmp/$USER-imps-%d.tex"
      sw-figure-index))))
  (setq sw-figure-index (1+ sw-figure-index)))




(defun sw-process-language-def-form (here)
  "Create a latex environment for the LANGUAGE def form following HERE."
  (interactive "d")
  (and
   (re-search-forward "^(def-" nil t)
   (progn
     (goto-char (match-beginning 0))
     (looking-at "^(def-language"))
   (let ((language-start (point)))
     (down-list 1)
     (forward-sexp 2)
     (backward-sexp 1)			;point at start of LANGUAGE name
     (let ((language-name (next-symbol-string (point)))
	   (env-marker (make-marker)))
       (goto-char here)
       (insert
	(format
	 "\n\\begin{signature}\n{\\bf (%s)} \n\\label{language:%s}\n \n 
%%%% Contents of %s to be inserted here.\n\n"
	 (sw-protect-special-characters language-name)
	 (sw-omit-special-characters language-name)
	 (substitute-in-file-name
	  (format "/tmp/$USER-imps-%d.tex" sw-figure-index))))
       (set-marker env-marker (point) (current-buffer))
       (setq sw-figure-markers
	     (cons
	      (cons env-marker sw-figure-index)
	      sw-figure-markers))
       (sw-send-language-env language-name)
       (insert
	(format "\n\\end{signature}\n\n"))
       (re-search-forward "^(def-" nil t)
       (goto-char (match-beginning 0))
       (insert "\n\\iffalse% \n")
       (forward-sexp 1)
       (insert "\n\\fi% \n")))))

(defun sw-send-language-env (language-name)
  (process-send-string
   tea-process
   (format
    "(xview-language-for-env '%s \"%s\")\n"
    language-name  
    (substitute-in-file-name
     (format
      "/tmp/$USER-imps-%d.tex"
      sw-figure-index))))
  (setq sw-figure-index (1+ sw-figure-index)))


(defun sw-process-translation-def-form (here)
  "Create a latex environment for the TRANSLATION def form following HERE."
  (interactive "d")
  (and
   (re-search-forward "^(def-" nil t)
   (progn
     (goto-char (match-beginning 0))
     (looking-at "^(def-translation"))
   (let ((translation-start (point)))
     (down-list 1)
     (forward-sexp 2)
     (backward-sexp 1)			;point at start of TRANSLATION name
     (let ((translation-name (next-symbol-string (point)))
	   (env-marker (make-marker)))
       (goto-char here)
       (insert
	(format
	 "\n\\begin{translation}\n{\\bf (%s)} \n\\label{translation:%s}\n \n 
%%%% Contents of %s to be inserted here.\n\n"
	 (sw-protect-special-characters translation-name)
	 (sw-omit-special-characters translation-name)
	 (substitute-in-file-name
	  (format "/tmp/$USER-imps-%d.tex" sw-figure-index))))
       (set-marker env-marker (point) (current-buffer))
       (setq sw-figure-markers
	     (cons
	      (cons env-marker sw-figure-index)
	      sw-figure-markers))
       (sw-send-translation-env translation-name)
       (insert
	(format "\n\\end{translation}\n\n"))
       (re-search-forward "^(def-" nil t)
       (goto-char (match-beginning 0))
       (insert "\n\\iffalse% \n")
       (forward-sexp 1)
       (insert "\n\\fi% \n")))))

(defun sw-send-translation-env (translation-name)
  (process-send-string
   tea-process
   (format
    "(xview-translation-for-env '%s \"%s\")\n"
    translation-name  
    (substitute-in-file-name
     (format
      "/tmp/$USER-imps-%d.tex"
      sw-figure-index))))
  (setq sw-figure-index (1+ sw-figure-index)))




(defun sw-insert-imps-tex-output ()
  (interactive)
  (let ((buff (current-buffer)))
    (mapcar
     (function
      (lambda (marker-index)
	(and
	 (eq buff (marker-buffer (car marker-index)))
	 (goto-char (marker-position (car marker-index)))
	 (insert-file-contents
	  (substitute-in-file-name
	   (format
	    "/tmp/$USER-imps-%d.tex"
	    (cdr marker-index))))
	 (set-marker (car marker-index) nil))))
     sw-figure-markers)
    (setq sw-figure-markers nil)))

(defun sw-make-envs-for-all-standard-def-forms (&optional arg)
  (interactive)
  (goto-char (point-min))
  (while (re-search-forward sw-make-figure-pattern nil t)
    (goto-char (match-beginning 0))
    (sw-process-standard-def-form (point))
    (sit-for 1))
  (or arg
      (message
       "Standard environments dispatched to Imps; please wait for retrieval..."))
  (sit-for 3)
  (sw-insert-imps-tex-output)
  (or arg
      (message "Figures retrieved from Imps; Schemeweb done.")))
  
(defun sw-make-figures-for-all-theory-def-forms (&optional arg)
  (interactive)
  (goto-char (point-min))
  (while (re-search-forward "^(def-theory" nil t)
    (goto-char (match-beginning 0))
    (sw-process-theory-def-form (point))
    (sit-for 1))
  (or arg
      (message
       "Theory figures dispatched to Imps; please wait for retrieval..."))
  (sit-for 3)
  (sw-insert-imps-tex-output)
  (or arg
      (message "Figures retrieved from Imps; Schemeweb done.")))
;; 
;; (defun sw-process-file ()
;;   (interactive)
;;   (if (not (y-or-n-p "Have you already loaded this file into Imps? "))
;;       (error "Please load the file into Imps.")
;;     (setq sw-figure-markers nil)
;;     (sw-make-figures-for-all-def-forms)
;;     (message "Figures dispatched to Imps.")
;;     (sit-for 3)
;;     (sw-insert-imps-tex-output)
;;     (save-buffer)))

;; (defun sw-process-file ()
;;   (interactive)
;;   (if (not (y-or-n-p "Have you already loaded this file into Imps? "))
;;       (error "Please load the file into Imps.")
;;     (message "Starting Schemeweb on file...")
;;     (sw-omit-trivial-def-forms)
;;     (sw-make-figures-for-all-theory-def-forms 'no-message)
;;     (sw-make-envs-for-all-standard-def-forms 'no-message)
;;     (save-buffer)
;;     (message "Starting Schemeweb on file... done")))

(defun sw-un-process-file ()
  (interactive)
  (if (y-or-n-p
       "Really undo Schemeweb on current buffer? ")
      (save-excursion
	(sw-undo-iffalses)
	(sw-undo-envs)
	(sw-undo-axioms)
	(message "Schemeweb undone"))
    (error "Schemeweb not undone")))

(defun sw-undo-iffalses ()
  (goto-char (point-min))
  (while (re-search-forward "\\\\iffalse%\\|\\\\fi%" nil t)
    (beginning-of-line)
    (kill-line)
    (delete-char 1)))

(defconst sw-envs
  '(constant theorem theory bnf signature translation lemma recursive-constant atomic-sort figure)
  "Symbols naming environments Schemeweb creates, and should delete to undo.")

(defun sw-undo-envs ()
  (goto-char (point-min))
  (while (re-search-forward "\\\\begin{\\([^}]+\\)}" nil t)
    (let ((start (match-beginning 0))
	  (kind (intern (buffer-substring (match-beginning 1) (match-end 1)))))
      (and (memq kind sw-envs)
	   (search-forward (format "\\end{%s}" kind))
	   (delete-region start (match-end 0))))))

(defun sw-undo-axioms ()
  (goto-char (point-min))
  (while (re-search-forward "^\\s-*;; Axioms---see Figure.*$" nil t)
    (delete-region (match-beginning 0)(1+ (match-end 0)))
    (re-search-forward "^\\s-*;; \\\\fi.*$" nil nil)
    (delete-region (match-beginning 0)(1+ (match-end 0)))))
		 
(defun hide-lemmas ()
  (goto-char (point-min))
  (while (search-forward "\\begin{lemma}" nil t)
    (comment-region
     (match-beginning 0)
     (progn
       (search-forward "\\end{lemma}")
       (match-end 0)))))

(defun unhide-lemmas ()
  (goto-char (point-min))
  (while (search-forward "\\begin{lemma}" nil t)
    (beginning-of-line)
    (let ((here (point)))
      (search-forward "\\end{lemma}")
      (save-restriction
	(narrow-to-region here (match-end 0))
	(goto-char here)
	(while (re-search-forward "^\\s<+\\s-*" nil t)
	  (replace-match "")
	  (end-of-line))))))
     
