;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald EXPRESSIONS)


; expressions are the most basic kind of IMPS object.  they are either simple,
; in which case they are called formal symbols, or complex, in which case they
; have a constructor and a list of components.  every expression has a sorting,
; which determines its suitability to being a component in more complex
; expressions.
;
; if an expression is a formal symbol, then it is either a variable or a
; constant.  otherwise, the constants and free and bound variables that it
; contains are computed from its components and constructor at the time it is
; made.  the newly-bound-variables are those which occur in free in a
; component, but not in the expression itself.  

(define-structure-type expression
  name							;name of the expression
  components						;list of immediate subexpressions
  constructor						;or nil if none
  qcs							;list of quasi-constructors 
  free-variables					;occurring in the expression
  bound-variables					;ditto
  newly-bound-variables					;ditto
  variables						;all of em
  constants						;ditto
  lead-constant						;leftmost formal-symbol if a constant
  var-name-conflict?					;'#t if any subexpr has two free 
							;variables with the same name 
  sorting						;indicating domain&range
  home							;smallest enclosing language
  height						;height as a tree
  hash							;a session-independent hash 
  pushed-not						;formula resulting by applying a negation
							;and pushing it through constructors
							;using de Morgan's laws.
  flushed-not						;existing negations in expr are flushed in
							;using de Morgan's laws (no new negation
							;pushed) 
  necessarily-undefined?				;contains UNDEFINED-OF-SORT in a position
							;that propagates
  hereditarily-prop-free?				;no subexpr is prop-sorted
							;means being
							;irreducible is
							;independent of context
  alpha-hash						;for hashing compound exprs
  alpha-root						;a unique representative of 
							;the alpha-equivalence class 
  descriptor-hash					;original descriptor-hash of an
							;expression before any GC.  

  (((print self port)
    (expression-print-method self port))
   ((name self)
    (or (expression-name self)
	(and (formula? self)
	     (theorem-name self))))
   ((base-sorts self)
    (base-sorts (expression-sorting self)))
   ((two-d-table-hash self) (expression-descriptor-hash self))))

(block
  (set (expression-name (stype-master expression-stype)) '#f)
  (set (expression-components (stype-master expression-stype)) nil)
  (set (expression-constructor (stype-master expression-stype)) nil)
  (set (expression-qcs (stype-master expression-stype)) (uncomputed))
  (set (expression-bound-variables (stype-master expression-stype)) nil)
  (set (expression-newly-bound-variables (stype-master expression-stype)) nil)
  (set (expression-variables (stype-master expression-stype)) (uncomputed))
  (set (expression-lead-constant (stype-master expression-stype)) 'no-lead-constant)
  (set (expression-var-name-conflict? (stype-master expression-stype)) '#f)
  (set (expression-hash (stype-master expression-stype)) (uncomputed))
  (set (expression-pushed-not (stype-master expression-stype)) '#f)
  (set (expression-flushed-not (stype-master expression-stype)) '#f)
  (set (expression-necessarily-undefined? (stype-master expression-stype)) (uncomputed))
  (set (expression-hereditarily-prop-free? (stype-master expression-stype)) (uncomputed))
  (set (expression-alpha-hash (stype-master expression-stype)) (uncomputed)))

(define (lead-constant? lc)
  (not (eq? lc 'no-lead-constant)))

(define (expression-second-lead-constant expr)
  (let ((comps (expression-components expr)))
    (if (< (length comps) 2)
	'#t						;'#t means component doesn't exist 
	(expression-lead-constant (nth comps 1)))))

(define use-string-form?
  (make-simple-switch 'use-string-form? boolean? '#f))

(define treat-qcs-specially?
  (make-simple-switch 'treat-qcs-specially? boolean? '#t))

(define (expression-print-method self port)
  (format port "#{IMPS-expression ~a: ~a}"
	  (object-hash self)
	  (imps-expression->string-procedure self)))

(define (compute-imps-hash expr)
  (let ((comps (expression-components expr))
	(constr (expression-constructor expr)))
    (cond ((not constr)
	   (string-hash (symbol->string (expression-name expr))))
	  ((null? comps)
	   (constructor-hash constr))
	  (else
	   (iterate iter ((hash (hash-combine-two-fixnums
				 (constructor-hash constr)
				 (imps-hash (car comps))))
			  (rest (cdr comps)))
	     (if (null? rest)
		 hash
		 (iter
		  (hash-combine-two-fixnums
		   hash
		   (imps-hash (car rest)))
		  (cdr rest))))))))

;;; The following gives a gc independent hash for each expression.
;;; Note, however, that the result of a gc might be to position a new
;;; expression exactly where an old expression was,  so that if its
;;; imps-hash is then computed it will equal the old one.   This would
;;; be embarrassing.  
;;;  (let ((hash+ (lambda (x y) (fx+ x (fixnum-ashl y 1)))))
;;;    (if (formal-symbol? expr)
;;;	(let ((n (name expr)))
;;;	  (if (symbol? n)
;;;	      ((*value t-implementation-env 'symbol-hash) n)
;;;	      (descriptor-hash n)))
;;;	(fixnum-abs
;;;	 (iterate loop ((sum 0) (comps (expression-components expr)))
;;;	   (if (null? comps) (hash+ ((*value t-implementation-env 'symbol-hash)
;;;				     (name (expression-constructor expr)))
;;;				    sum)
;;;	       (loop (hash+ sum (compute-imps-hash (car comps)))
;;;		     (cdr comps)))))))

(define (imps-hash expression)
  (let ((old (expression-hash expression)))
    (if (uncomputed? old)
	(block
	  (set (expression-hash expression) (compute-imps-hash expression))
	  (expression-hash expression))
	old)))

(define (alpha-hash-combine-exprs exprs)
  (iterate iter ((result 0)
		 (exprs exprs)
		 (i 0))
    (if (null? exprs)
	result
	(iter (fx+ result (fixnum-ashl (expression-alpha-hash (car exprs)) i))
	      (cdr exprs)
	      (fx+ i 1)))))

(define-integrable (hash-combine-exprs exprs)
  (iterate iter ((result 0)
		 (exprs exprs)
		 (i 0))
    (if (null? exprs)
	result
	(iter (fx+ result (fixnum-ashl (expression-descriptor-hash (car exprs)) i))
	      (cdr exprs)
	      (fx+ i 1)))))

;; (define hash-combine-exprs plain-hash-combine-exprs)

(define (variables expression)
  (imps-enforce expression? expression)
  (let ((vs (expression-variables expression)))
    (if (uncomputed? vs)
	(block
	  (set (expression-variables expression)
	       (set-union (expression-bound-variables expression)
			  (expression-free-variables expression)))
	  (expression-variables expression))
	vs)))

; an expression-list is a list of expressions.

(define (expression-list? e-l)
  (and (list? e-l)
       (every? expression? e-l)))

; home-language is a pleasant synonym for expression-home

(define home-language expression-home)

; a formula is an expression that expresses a proposition.  semantically, we
; require every formula to be either true or false.  

(define (formula? expression)
  (and (expression? expression)
       (eq? prop (expression-sorting expression))))

; a term is an expression that---if defined---denotes an individual, that is to
; say a member of some basic sort other than prop.  

(define (term? expression)
  (and (expression? expression)
       (not (formula? expression))
       (base-sort? (expression-sorting expression))))

; a predicate is an expression that, if applied to arguments, yields a formula.
; semantically, it must be defined for all possible arguments.  

(define (predicate? expression)
  (and (expression? expression)
       (let ((sort (expression-sorting expression)))
	 (and
	  (higher-sort? sort)
	  (eq? prop (higher-sort-range sort))))))

(define (UNARY-PREDICATE? expression)
  (and (predicate? expression)
       (equal? (length (higher-sort-domains (expression-sorting expression))) 1)))

(define (UNARY-PREDICATE-DOMAIN expression)
  (car (higher-sort-domains (expression-sorting expression))))

(define (BINARY-PREDICATE? expression)
  (and (predicate? expression)
       (equal? (length (higher-sort-domains (expression-sorting expression))) 2)))

; a predicator is, so to speak, a deferred formula.  that is, it is an
; expression that, when applied to arguments, yields either a formula or a less
; deferred predicator.  predicators are require to have the important semantic
; property of being defined for all possible arguments.  

(define (predicator? expression)
  (and (expression? expression)
       (let ((sort (expression-sorting expression)))
	 (and
	  (higher-sort? sort)
	  (prop-sorting? sort)))))

(define (formula-or-predicator? expression)
  (or (formula? expression)
      (predicator? expression)))

; a function is, so to speak, a deferred term.  that is, it is an
; expression that, when applied to arguments, yields either a term or a less
; deferred function.  functions are not required to be defined for all possible
; arguments.   

(define (function? expression)
  (and (expression? expression)
       (let ((sort (expression-sorting expression)))
	 (and
	  (higher-sort? sort)
	  (not (prop-sorting? sort))))))

(define (term-or-fn? expression)
  (or (term? expression)
      (function? expression)))

(define (arity expression)
  (let ((sort (expression-sorting expression)))
    (if (higher-sort? sort)
	(length (higher-sort-domains sort))
	'#f)))

; return either the list of domains of an expression of higher sort or NIL, if basic.  

(define (domain-sorts expression)
  (let ((sort (expression-sorting expression)))
    (if (higher-sort? sort)
	(higher-sort-domains sort)
	nil)))

(define-constant expression-domains domain-sorts)

; return either the list of the types to which the domains of an expression of
; higher sort belong, or alternatively NIL, if basic.  

(define (domain-types expression)
  (map type-of-sort (domain-sorts expression)))

(define (expression-range expression)
  (let ((sort (expression-sorting expression)))
    (if (higher-sort? sort)
	(higher-sort-range sort)
	nil)))

(define (range-type expression)
  (type-of-sort (expression-range expression)))

(define (expression-sorting-necessarily-within?  expression sort)
  (sort-necessarily-included?
   (expression-sorting expression)
   sort))

; return either the range of an expression of higher sort or the sorting, if basic.  

(define (range-sort expression)
  (let ((sort (expression-sorting expression)))
    (if (higher-sort? sort)
	(higher-sort-range sort)
	sort)))

(define (expression-type expr)
  (type-of-sort (expression-sorting expr)))

; the expression-category of an expression is the t-predicate above that holds
; true of it.  

(define (expression-category expression)
  (and
   (expression? expression)
   (cond ((formula? expression) formula?)
	 ((term? expression) term?)
	 ((function? expression) function?)
	 ((predicate? expression) predicate?)
	 ((predicator? expression) predicator?))))

; constants is a convenient way of getting the expression-constants of an
; expression or the language-constants of a language. 

(define (constants expression-or-langauge)
  (cond ((expression? expression-or-langauge)
	 (expression-constants expression-or-langauge))
	((language? expression-or-langauge)
	 (language-constants expression-or-langauge))
	(else (imps-error "constants: ~a neither an expression nor a language."
		     expression-or-langauge))))

; free-variables is a convenient way to get the expression-free-variables of an
; expression or the union of the free-variables in the expressions in some
; larger units, namely contexts (antecedents of sequents) and sequents.

(define (free-variables expression-context-or-sequent)
  (cond ((expression? expression-context-or-sequent)
	 (expression-free-variables expression-context-or-sequent))
	((context? expression-context-or-sequent)
	 (context-free-variables expression-context-or-sequent))
	((sequent? expression-context-or-sequent)
	 (sequent-free-variables expression-context-or-sequent))
	(else (imps-error "free-variables: ~a neither fish nor fowl."
			  expression-context-or-sequent))))

; bound-variables is a convenient way to get the expression-bound-variables of an
; expression or the union of the bound-variables in the expressions in some
; larger units, namely contexts (antecedents of sequents) and sequents.

(define (bound-variables expression-context-or-sequent)
  (cond ((expression? expression-context-or-sequent)
	 (expression-bound-variables expression-context-or-sequent))
	((context? expression-context-or-sequent)
	 (context-bound-variables expression-context-or-sequent))
	((sequent? expression-context-or-sequent)
	 (sequent-bound-variables expression-context-or-sequent))
	(else (imps-error "bound-variables: ~a neither fish nor fowl."
			  expression-context-or-sequent))))

; newly bound variables are those caused to be bound in an expression by its 
; binding-constructor	 

(define newly-bound-variables expression-newly-bound-variables)

; the formal symbols in an expression are its constants and variables.

(define (formal-symbols expression)
  (and (expression? expression)
       (append
	(constants expression)
	(variables expression))))

; an expression without free variables is closed. 

(define (closed? expression)
  (and (expression? expression)
       (null? (free-variables expression))))

; a sentence is a closed formula. 

(define (sentence? expression)
  (and (formula? expression)
       (closed? expression)))

; an atomic formula is one no immediate component of which is a formula.  in
; the presence of constructors like the definite description operator, this
; does not entail that no ultimate component is a formula.  

(define (atomic-formula? expression)
  (and (formula? expression)
       (not (any?
	     formula?
	     (expression-components expression)))))

;;;; a strong atomic formula is one no immediate component of which is prop sorted.  in
;;;; the presence of constructors like the definite description operator, this
;;;; does not entail that no ultimate component is a formula.  
;;;
;;;(define (strong-atomic-formula? expression)
;;;  (and (formula? expression)
;;;       (not (any?
;;;	     expression-of-category-prop?
;;;	     (expression-components expression)))))

; An expression is hereditarily prop-free if it is not a formula or predicator,
; and the same holds for all of its components.  This fact is stashed in a
; field of the expression, once computed, at Javier's request.  

(define (hereditarily-prop-free? expr)
  (let ((compute-it
	 (lambda (expr)
	   (and (term-or-fn? expr)
		(every?
		 hereditarily-prop-free?
		 (expression-components expr)))))
	(value (expression-hereditarily-prop-free? expr)))
    (if (uncomputed? value)
	(let ((value (compute-it expr)))
	  (set (expression-hereditarily-prop-free? expr) value)
	  value)
	value)))
    


;The subterms of an expression are those components which are in fact terms,
;together with the (recursive) subterms of its components.

(define (subterms expr)
  (set-union
   (separate-set term? (expression-components expr))
   (collect-set
    subterms
    (expression-components expr))))

; a possible symbol-form for a formal-symbol, etc., is a symbol, a fixnum, or a
; uniquely instantiated item of another numerical sort.  
; they are required to be uniquely instantiated, so that re-reading always
; yields the same expression.

(let* ((symbol-form-table (make-hash-table
			   numerical-object? tree-hash
			   numerical-= '#t  'symbol-form-table))
       (retrieve
	(lambda (key)
	  (table-entry symbol-form-table key)))
       (install
	(lambda (key)
	  (set (table-entry symbol-form-table key) key)
	  key)))

  (define (SEEK-SYMBOL-FORM sexp)
    (cond ((or (symbol? sexp)
	       (fixnum? sexp))
	   sexp)
	  ((possible-symbol-form? sexp)
	   (or (retrieve sexp)
	       (install sexp)))
	  (else '#f))))

(define (POSSIBLE-SYMBOL-FORM? sym-form)
  (or (symbol? sym-form)
      (numerical-object? sym-form)))			

; A canonical weak linear ordering on possible symbol forms.  Numbers are
; ordered by value, and preceed symbols.  Symbols are ordered lexicographically
; as strings.  

(define (COMPARE-POSSIBLE-SYMBOL-FORMS sf1 sf2)
  (cond ((eq? sf1 sf2) '#t)
	((and (numerical-object? sf1)
	      (numerical-object? sf2))
	 (numerical-< sf1 sf2))
	((numerical-object? sf1) '#t)
	((numerical-object? sf2) '#f)
	((and (symbol? sf1)
	      (symbol? sf2))
	 (string-less? (symbol->string sf1)
		       (symbol->string sf2)))
	(else (< (descriptor-hash sf1)
		 (descriptor-hash sf2)))))
	 

; A formal symbol is of KIND CONSTANT? or VARIABLE?.  Its SORTING is either
; symbol or a sequence of sortings.  If the symbol is an individual, its
; sorting is a symbol denoting the sort it is to belong to.  If it is a
; function, its sorting is a sequence giving the sortings of its arguments
; and the sorting of its value.  If it is a predicate, its sorting is a
; sequence giving the sortings of its arguments and the sort PROP of its
; result, a proposition.  Each symbol is expected to denote a sort in every
; language to which the symbol might belong.  SYMBOL-FORM is the symbol that
; should normally be assocated with the formal symbol in printing and
; reading.  It may be overridden by a particular language if the language has
; a conflict of symbol-forms. 

(define (MAKE-FORMAL-SYMBOL kind sorting symbol-form . home)
  ;;first check that the parameters make sense
  (imps-enforce possible-symbol-form? symbol-form)
  (or (eq? kind constant?)
      (eq? kind variable?)
      (imps-error "MAKE-FORMAL-SYMBOL: kind must be CONSTANT? or VARIABLE?, not ~S"
	     kind))
  (or (sorting-permissible? sorting (and home (car home)))
      (imps-error "MAKE-FORMAL-SYMBOL: impossible sorting ~A"
		  (or sorting
		      "--** sort symbol not readable in current language.")))

  ;;Now construct the formal symbol itself    
  (let ((expr (make-expression)))
    (set (expression-name expr) symbol-form)
    (set (expression-free-variables expr) (and (eq? kind variable?) (list expr)))
    (set (expression-constants expr) (and (eq? kind constant?) (list expr)))
    (if (eq? kind constant?)				;otherwise already 'no-lead-constant 
	(set (expression-lead-constant expr) expr))
    (set (expression-sorting expr) sorting)
    (set (expression-home expr) (and home (car home)))
    (set (expression-height expr) 0)
    (set (expression-hash expr)
	 (if (fixnum? symbol-form)
	     symbol-form
	     (hash-combine-two-fixnums
	      ((*value t-implementation-env 'string-hash)
	       (format nil "~s" (sort->list sorting)))
	      ((*value t-implementation-env 'string-hash)
	       (format nil "~s" symbol-form)))))
    (set (expression-alpha-root expr) expr)
    (set (expression-alpha-hash expr)
	 (if (eq? kind constant?)			;ensure hash respects 
	     (descriptor-hash expr)			;change of variable name 
	     (sort-alpha-hash sorting)))
    (set (expression-descriptor-hash expr)		;ensure this hash is permanent
	 (descriptor-hash expr))
    expr))

;A formal-symbol is just an expression without a constructor.

(define (FORMAL-SYMBOL? expr)
  (and (expression? expr)
       (null? (expression-constructor expr))))

; A variable or constant is just a formal symbol that is its own free-variable
; or expression-constant.  

(define (VARIABLE? expr)
  (and (formal-symbol? expr)
       (eq? expr (car (expression-free-variables expr)))))

(define (CONSTANT? expr)
  (and (formal-symbol? expr)
       (eq? expr (car (expression-constants expr))))) 

(define (RATIONAL-CONSTANT? x)
  (and (constant? x)
       (rational? (name x))))

(define (NONNEGATIVE-INTEGER-CONSTANT? x)
  (and (constant? x)
       (non-negative-integer? (name x))))

(define (INTEGER-CONSTANT? x)
  (and (constant? x)
       (integer? (name x))))

(define (FLOATING-POINT-CONSTANT? x)
  (and (constant? x)
       (float? (name x))))

(define (NUMERICAL-CONSTANT? x)
  (and (constant? x)
       (numerical-object? (name x))))

(define (FUNCTION-CONSTANT? x)
  (and (constant? x)
       (function? x)))


; A path is a list of integers, used to select a subexpression from an
; expression.  The first integer indicates the component to choose from the
; components of the top level; the remainder of the list is used in recursive
; calls.  Note that this function, which uses nth, is 0-based.   

(define (FOLLOW-PATH expr path)
  (if (null? path)
      expr
      (follow-path
       (nth (expression-components expr) (car path))
       (cdr path))))
	
; HOST-AND-PATH-AFTER-STEP returns two values, namely the result of applying
; the first integer in the path to the given host, and the remainder of the
; path. 

(define (HOST-AND-PATH-AFTER-STEP expr path)
  (if (null? path)
      (imps-error "PATH-AND-HOST-AFTER-STEP: trying to follow null path.")
      (return
       (nth (expression-components expr) (car path))
       (cdr path))))

; The set of variables bound at least once between HOST and the expression at
; the end of PATH.  The newly-bound-variables of HOST are included when the
; path is non-null, but those of (follow-path host path) are not necessarily.    

(define (BOUND-VARIABLES-ON-PATH host path)
  (iterate iter ((host host)
		 (path path)
		 (pbvs nil))
    (if (null? path)
	pbvs
	(receive (new-host new-path)
	  (host-and-path-after-step host path)
	  (iter
	   new-host
	   new-path
	   (set-union pbvs (expression-newly-bound-variables host)))))))
	   
	
    
  

; A test for being a (possibly improper) subexpression

(define (SUBEXPRESSION? exp1 exp2)
  (or (eq? exp1 exp2) 
      (proper-subexpression? exp1 exp2)))

(define (PROPER-SUBEXPRESSION? exp1 exp2)
  (cond ((constant? exp1)
	 (memq? exp1 (constants exp2)))
	((variable? exp1)
	 (memq? exp1 (variables exp2)))
	((< (expression-height exp1)
	    (expression-height exp2))
	 (any?
	  (lambda (expr) (subexpression? exp1 expr))
	  (expression-components exp2)))
	(else '#f)))

; Return a set of paths.  P is in the set iff
; (follow-path expr P) => subexp.  If depth-bound is non-negative, decrement it
; on recursive calls and stop as soon as it reaches 0.  For a complete search,
; start with depth = -1.  

(define (PATHS-TO-OCCURRENCES expr subexp depth-bound)
  (cond
   ((=0? depth-bound) nil)
   ((alpha-equivalent? subexp expr) (list nil))
   ((> (expression-height subexp)
       (expression-height expr))
    nil)
   (else
    (do ((components (expression-components expr) (cdr components))
	 (index 0 (1+ index))
	 (paths nil
		(append!
		 (map!
		  (lambda (path)
		    (cons index path))
		  (paths-to-occurrences (car components) subexp (-1+ depth-bound)))
		 paths)))
	((null? components)
	 paths)))))


(define (SORTED-PATHS-TO-OCCURRENCES expr subexp depth-bound)
  (sort-paths! (paths-to-occurrences expr subexp depth-bound)))
       
; Return the expression that differs from HOST in having REPLACEMENT at the end
; of PATH.  

(define (SUBSTITUTION-AT-PATH host replacement path)
  (if
   (> (length path)
      (expression-height host))
   (imps-error "SUBSTITUTION-AT-PATH: path ~S too long for host ~S." path host)
   (iterate iter ((host host)
		  (path path))
     (if (null? path)
	 replacement
	 (let ((comps (expression-components host))
	       (i (car path)))
	   (apply (expression-constructor host)
		  (replace-nth-in-list comps i
				       (iter (nth comps i) (cdr path)))))))))

	
(define (sap-accumulate-nils lopl already offset)
  (cond ((null? lopl) already)
	((memq? '() (car lopl))
	 (sap-accumulate-nils
	  (cdr lopl)
	  (cons offset already)
	  (1+ offset)))
	(else
	 (sap-accumulate-nils
	  (cdr lopl)
	  already
	  (1+ offset)))))

(define (sap-filter-lopl lopl n)
  (map
   (lambda (pl)
     (iterate iter ((pl pl)
		    (new '()))
       (cond ((null? pl) new)
	     ((null? (car pl))
	      (imps-error "sap-filter-lopl: why didn't sap-accumulate-nils win? ~S." lopl))
	     ((= (caar pl) n)
	      (iter (cdr pl)
		    (cons (cdar pl) new)))
	     (else
	      (iter (cdr pl) new)))))
   lopl))

(define (SUBSTITUTIONS-AT-PATHS host replacement-list list-of-path-lists)
  (let ((indices  (sap-accumulate-nils list-of-path-lists '() 0)))
    (cond ((null? indices)
	   (if (expression-constructor host)
	       (apply (expression-constructor host)
		      (iterate iter ((comps (expression-components host))
				     (rev-new-comps '())
				     (offset 0))
			(if (null? comps)
			    (reverse! rev-new-comps)
			    (iter (cdr comps)
				  (cons
				   (substitutions-at-paths
				    (car comps)
				    replacement-list
				    (sap-filter-lopl list-of-path-lists offset))
				   rev-new-comps)
				  (1+ offset)))))
	       host))
	  ((null? (cdr indices))
	   (let ((index (car indices)))
	     (if (< index (length replacement-list))
		 (nth replacement-list (car indices))
		 (imps-error "SUBSTITUTIONS-AT-PATHS: index ~S out of range ~S."
			     index (length replacement-list)))))
	  (else
	   (imps-error "SUBSTITUTIONS-AT-PATHS: too many applicable indices ~S~_in list of path-lists ~S~_~S~_~S.~&."
		       indices list-of-path-lists host replacement-list)))))

;finds paths to subexpressions of expr which satisfy pred.

(define (PATHS-TO-SATISFACTION expr pred depth-bound)
  (cond
   ((=0? depth-bound) nil)
   (else
    (do ((components (expression-components expr) (cdr components))
	 (index 0 (1+ index))
	 (paths (if (pred expr)
		    (list '())
		    '())
		(append!
		 (map!
		  (lambda (path)
		    (cons index path))
		  (paths-to-satisfaction (car components) pred (-1+ depth-bound)))
		 paths)))
	((null? components)
	 paths)))))

(define (SUBEXPRESSION-SATISFIES? pred expr)
  (if (pred expr)
      '#t
      (any? (lambda (x) (subexpression-satisfies? pred x))
	    (expression-components expr))))

(define (EXPRESSION-OF-CATEGORY-IND? expr)
  (eq? (sort-category (expression-sorting expr)) ind))

(define (EXPRESSION-OF-CATEGORY-PROP? expr)
  (eq? (sort-category (expression-sorting expr)) prop))

(define (MAKE-SUBEXPRESSION-GENERATOR expr)
  (let ((path '())
	(already-seen '()))
    (lambda ()
      (iterate iter ()
	(cond ((fail? path) '#f)
	      ((not (memq? (follow-path expr path) already-seen))
	       (let ((v (follow-path expr path)))
		 (set already-seen (cons (follow-path expr path) already-seen))
		 (set path (path-successor! expr path))
		 v))
	      (else
	       (set path (path-successor! expr path))
	       (iter)))))))

(define (path-successor! expr path)
  (if (not (null? (expression-components (follow-path expr path))))
      (append-item-to-last-cdr! path 0)
      (iterate iter ((path path))
	(receive (last shortened)
	  (last-&-all-but-last! path)
	  (cond ((null? path) (fail))
		((< (1+ last)
		    (length
		     (expression-components (follow-path expr shortened))))
		 (append-item-to-last-cdr! shortened (1+ last)))
		(else (iter shortened)))))))
	    
(define (make-expression-table . maybe-id)
  (apply make-hash-table expression? expression-descriptor-hash eq? '#f maybe-id))
	    
      
    
	
    
  
