;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald (inferences context-entailment))


;;; Context-based simplifification for formulas

(define (CONTEXT-ENTAILS? context formula)
  (or (context-trivially-entails? context formula)
      (truth?
       (context-insistently-simplify
	context
	formula
	(context-simplification-persistence)))))

(define (CONTEXT-INSISTENTLY-SIMPLIFY context expr persist)
  (if
   (and (formula? expr)
	(context-trivially-entails? context expr))
   truth
   (iterate iter ((expr expr))
     (let ((simplified (context-simplify context expr persist)))
       (if (or (eq? simplified expr)
	       (alpha-equivalent? simplified expr)) 
	   expr
	   (iter simplified))))))
	
(define (CONTEXT-IMMEDIATELY-ENTAILS? context formula)
  (truth? (context-simplify context formula (context-low-simplification-persistence))))

(define (CONTEXT-SIMPLIFY-VIRGIN context expr persist)
  ((prune-simplification context-insistently-simplify) context expr persist))

(define (context-trivially-entails? context expr)
  (let ((fn
	 (lambda (context expr)
	   (cond ((truth? expr))
		 ((context-absurd? context))
		 ((mem? alpha-equivalent? expr (context-assumptions context)))
		 ((context-contains-entry? context expr))
		 ((convergence? expr)
		  (let ((term (convergence-term expr)))
		    (or (context-match-check-definedness? context term)
			(context-trivially-entails-defined? context term))))
		 ((convergence-in-sort? expr)
		  (context-trivially-entails-defined-in-sort?
		   context
		   (convergence-term expr)
		   (convergence-sort expr)))
		 ((quasi-equation? expr)
		  (context-trivially-entails-quasi-equal?
		   context
		   (quasi-equation-lhs expr)
		   (quasi-equation-rhs expr)))
		 ((equation? expr)
		  (context-trivially-entails-equal? 
		   context
		   (expression-lhs expr)
		   (expression-rhs expr)))
		 ;; ((conjunction? expr)
		 ;;  (every?
		 ;;   (lambda (conjunct)
		 ;;     (context-trivially-entails? context conjunct))
		 ;;   (ultimate-conjuncts (list expr))))
		 (else '#f)))))
    (or (fn context expr)
	(fn context (recursive-constructor-simplify expr)))))	
	


;; Flags/switches --
;; 

(define dont-work-on-definedness
  (make-simple-switch 'dont-work-on-definedness boolean? '#f))


;; The global variable undischarged-minor-premises is "bind"-ed to accumulate
;; undischarged minor premises when simplify-with-minor-premises in
;; $INFERENCES/syllogistic-inference.t is called.  

(lset undischarged-minor-premises nil) 
(define accumulate-undischarged-minor-premises?
  (make-simple-switch 'accumulate-undischarged-minor-premises?
		      boolean?
		      '#f))

(define simplify-non-virgins?
  (make-simple-switch 'simplify-non-virgins?
		      boolean?
		      '#t))

;; (SIMPLIFY-WITH-PARTIALLY-SIMPLIFIED-FORM context expr proc) executes proc
;; (which is assumed to be a simplification procedure) on context and expr1,
;; the transitive partially simplified form of expr in context.  The result is
;; stored as the new partially simplified form for both expr and expr1 unless
;; there are undischarged-minor-premises.  

(define (SIMPLIFY-WITH-PARTIALLY-SIMPLIFIED-FORM context expr proc)
  (if (or (simplify-non-virgins?)
	  (not ((sensor partially-simplified-form) context expr)))
      (let ((expr1 (transitive-partially-simplified-form context expr)))
	(let ((expr2 (proc context expr1)))
	  (imps-enforce expression? expr2)
	  (if (null? undischarged-minor-premises)
	      (block
		(if (not (alpha-equivalent? expr expr2))
		    (set (partially-simplified-form context expr) expr2))
		(if (not (alpha-equivalent? expr1 expr2))
		    (set (partially-simplified-form context expr1) expr2))))
	  expr2))
      (transitive-partially-simplified-form context expr)))

;; If proc is a procedure of the three args (context expr persistence),
;; and syntactic-restriction? is a predicate, then
;; make-simplification-procedure returns a procedure which calls proc on the
;; partially simplified form, if the syntactic restriction holds true of the
;; latter.  

(define (make-simplification-procedure proc syntactic-restriction?)
  (lambda (context expr persist)
    (if (or (context-expr-defined? context expr)
	    (context-expr-defined? context (expression-alpha-root expr)))
	(equate-to-alpha-root context expr))
    (context-simplify-installer
     context
     expr
     (simplify-with-partially-simplified-form
      context
      expr
      (object (lambda (context expr)
		(if (and (not (negative? persist))
			 (syntactic-restriction? expr))
		    (proc context expr persist)
		    expr))
	((print self port)
	 (format port "#{Procedure ~D ~S}"
		 (object-hash self)
		 proc)))))))

(define (PRUNE-SIMPLIFICATION proc)
  (lambda (context expr . rest)
    (bind (((simplify-non-virgins?) '#f))
      (apply proc context expr rest))))

(define (CONTEXT-SIMPLIFY-TOP-LEVEL context expr persist)
  (or (and (formula? expr)
	   (seek-truth-value context expr persist))
      (let ((expr (context-recursively-seek-formal-symbols context expr)))
	(context-simplify-1
	 context
	 expr
	 persist))))

(define CONTEXT-SIMPLIFY
  (make-simplification-procedure
   context-simplify-top-level
   (always '#t)))

(define (context-simplify-installer context expr simplified)
  (cond ((alpha-equivalent? expr simplified))
	(undischarged-minor-premises)
	((eq? simplified truth)
	 (add-context-entry context expr))
	((eq? simplified falsehood)
	 (add-context-entry context
			    (push-not expr)))
	((and (term-or-fn? expr)
	      (or (context-trivially-entails-defined? context expr)
		  (context-trivially-entails-defined? context simplified)))
	 (add-context-entry context
			    (equality expr simplified)))
	((term-or-fn? expr)
	 (add-context-quasi-equation context expr simplified))
	((formula? expr)
	 (add-context-entry context
			    (biconditional expr simplified))))
  simplified)

(define (CONTEXT-SIMPLIFY-1-INTERNAL context expr persist)
  (if (formal-symbol? expr)
      expr
      (or (and (formula? expr)
	       (seek-truth-value context expr persist))
	  (simplify-by-transforms context expr persist))))

(define CONTEXT-SIMPLIFY-1
  (make-simplification-procedure
   context-simplify-1-internal
   (always '#t)))


(lset *simplify-by-transforms-tracer* (make-table '*simplify-by-transforms-tracer*))

(define (SIMPLIFY-BY-TRANSFORMS context expr persist)
  (if (table-entry *simplify-by-transforms-tracer* expr)
      expr
      (bind (((table-entry *simplify-by-transforms-tracer* expr) '#t))
	(receive (simplified reqs any?)
	  (insistently-apply-matching-transforms-in-context context expr persist)
	  (cond ((and (not any?)
		      (alpha-equivalent? simplified expr))
		 (simplify-logically context expr persist)) ;(-1+ persist)
		((alpha-equivalent? simplified expr) expr)
		((necessarily-false? simplified) falsehood)
		((necessarily-undefined? simplified)
		 (undefined (expression-sorting simplified)))
		((tc-refute-requirements context reqs persist)
		 (undefined-or-falselike simplified))
		((tc-prove-requirements context reqs persist)
		 simplified)
		((truth? simplified)
		 (conjunction-simplifier
		  (map
		   (lambda (req)
		     (context-simplify context req (-1+ persist)))
		   reqs)))
		(else expr))))))

(define (SEEK-TRUTH-VALUE context expr persist)
  (cond
   ((context-absurd? context) truth)			;context absurd
   ((truth-value? expr) expr)				;truth value already
   ((context-trivially-entails? context expr)		;expr known in context
    truth)						
   ((context-contains-entry? context (push-not expr))	;negation known in context
    falsehood)						
   ((context-universal-instance? context expr persist) truth)
   (else '#f)))

(define (SIMPLIFY-LOGICALLY-INTERNAL context expr persist)
  (or
   (and (formula? expr)
	(seek-truth-value context expr persist))
   (let ((transform
	  (or					
	   (constructor-logical-transform		;Use it even if disabled, for instance 
	    (expression-quasi-constructor expr))	;in the case of lambda-application 
	   (constructor-logical-transform		
	    (expression-constructor expr)))))
     (if transform
	 (transform context expr persist)
	 (context-apply-constructor-simplifier context expr persist)))))

(define SIMPLIFY-LOGICALLY
  (make-simplification-procedure
   simplify-logically-internal
   (always '#t)))

(define (CONTEXT-APPLY-CONSTRUCTOR-SIMPLIFIER context expr persist)
  (if (formal-symbol? expr)
      expr
      (receive (simp constr comps)
	(constructor-simplifier-&-constructor-&-components expr)
	(if (and (binding-constructor? constr)
		 (not (quasi-constructor? constr)))
	    (let ((vars (binding-variables expr))
		  (body (binding-body expr)))
	      (simp
	       (cons
		(context-simplify-1
		 (requantification-filter-context expr context)
		 body
		 persist)
		vars)))
	    (simp
	     (simplify-components-in-context
	      constr
	      comps
	      context
	      persist
	      context-simplify-1))))))

(define (SIMPLIFY-COMPONENTS-IN-CONTEXT constructor components context persist rec-call)
  (let ((incrementer (constructor-lc-incrementer constructor))
	(len (length components))
	(components (copy-list components)))
    (iterate iter ((i 0))
      (if (= i len)
	  components
	  (let* ((current-comp (nth components i))
		 (local-context
		  (context-add-assumptions context (incrementer components i)))
		 (simp (rec-call local-context current-comp persist)))
	    (set (nth components i) simp)
	    (iter (1+ i)))))))

(define (CONTEXT-SEEK-FORMAL-SYMBOL context expr)
  (let ((root (context-q-root context expr)))
    (if (or (formal-symbol? root)
	    (any?
	     (lambda (qc)
	       (quasi-constructor-constantlike?
		(quasi-constructor-if-enabled qc)))
	     (expression-quasi-constructors root))) 
	root
	'#f)))

(define (CONTEXT-RECURSIVELY-SEEK-FORMAL-SYMBOLS context expr)
  (iterate iter ((context context)
		 (exoscopes nil)
		 (expr expr))
    (if (quasi-constructor-constantlike?
	 (or (expression-quasi-constructor expr)
	     (expression-constructor expr)))
	expr
	(let ((new (context-seek-formal-symbol context expr)))
	  (cond ((and (expression? new)
		      (alpha-equivalent? new expr))
		 expr)
		((memq? new exoscopes)
		 expr)
		(new)
		(else
		 (if (formal-symbol? expr)
		     expr
		     (receive (simp constr comps)
		       (constructor-simplifier-&-constructor-&-components expr)
		       (if (and (binding-constructor? constr)
				(not (quasi-constructor? constr)))
			   (simp
			    (cons
			     (iter (requantification-filter-context expr context)
				   (append (expression-newly-bound-variables expr) exoscopes)
				   (binding-body expr))
			     (binding-variables expr)))
			   (simp
			    (map
			     (lambda (c)
			       (iter context exoscopes c))
			     comps)))))))))))

(define simplify-quasi-constructors-messily?
  (make-simple-switch 'simplify-quasi-constructors-messily?
		      boolean?
		      '#f))

(define (CONSTRUCTOR-SIMPLIFIER-&-CONSTRUCTOR-&-COMPONENTS expr)
  (receive (q-constr q-comps)
    (expression-quasi-constructor-&-components expr)
    (cond ((constructor-simplifier q-constr)
	   =>
	   (lambda (simp)
	     (return simp q-constr q-comps)))
	  ((and (not (simplify-quasi-constructors-messily?))
		(quasi-constructor-if-enabled q-constr))
	   (return (lambda (q-comps)
		     (substitutions-at-paths
		      expr
		      q-comps
		      (quasi-constructor-paths-to-quasi-components q-constr)))
		   q-constr
		   q-comps))
	  ((expression-constructor expr)
	   =>
	   (lambda (constr)
	     (return
	      (or (constructor-simplifier constr)
		  (lambda (comps) (apply constr comps)))
	      constr
	      (expression-components expr))))
	  (else
	   (return
	    (lambda (comps) (ignore comps) expr)
	    nil
	    nil)))))

(define (LAMBDA-APPLICATION-LOGICAL-TRANSFORM context expr persist)
  (let ((operator
	 (cond ((constructor-logical-transform		
		 (expression-quasi-constructor (operator expr)))
		=>
		(lambda (transform)
		  (transform context (operator expr) persist)))
	       (else (operator expr))))
	(args (arguments expr)))
    
    (let ((reqs
	   ;;
	   ;; Formerly:
	   ;;
	   ;; (substitution-definedness-conditions
	   ;; (targets-and-replacements->subst
	   ;;  (binding-variables operator)
	   ;;  args))
	   ;;  
	   (restricted-substitution-definedness-conditions 
	    (targets-and-replacements->subst
	     (binding-variables operator)
	     args)
	    (exposed-variables (binding-body operator)))))
      (if (tc-refute-requirements context reqs persist)
	  (undefined-or-falselike expr)
	  (let ((naive
		 (context-simplify-1
		  context
		  (beta-reduce-recklessly (apply apply-operator operator args))
		  persist)))
	    (cond ((falsehood? naive) falsehood)
		  ((necessarily-undefined? naive)
		   (undefined (expression-sorting naive)))
		  ((tc-prove-requirements context reqs persist) naive)
		  ((truth? naive)
		   (conjunction-simplifier
		    (map
		     (lambda (req)
		       (context-simplify context req (-1+ persist)))
		     reqs)))
		  (else (apply apply-operator
			       operator
			       (map (lambda (arg)
				      (context-simplify context arg persist))
				    args)))))))))

(define (TYPE-PREFERRED-QC-TRANSFORM context expr persist)
  (enforce lambda-expression? expr)
  (let ((old-vars (binding-variables expr))
	(make-new-vars
	 (lambda (vars)
	   (iterate loop ((vars vars)
			  (new-vars '())
			  (big-avoid-vars (set-union vars (context-variables context)))) 
	     (if (null? vars)
		 (reverse! new-vars)
		 (let* ((var (car vars))
			(new-var (new-variable
				  (type-of-sort (expression-sorting var))
				  (expression-name var)
				  big-avoid-vars)))
		   (loop (cdr vars)
			 (cons new-var new-vars)
			 (cons new-var big-avoid-vars))))))))
    (let ((new-vars (make-new-vars old-vars)))
      (context-apply-constructor-simplifier
       context
       (apply imps-lambda
	      (apply-substitution
	       (targets-and-replacements->subst old-vars new-vars)
	       (binding-body expr))
	      new-vars)
       persist))))

(define (EQUATE-TO-TYPE-QC-TRANSFORM context expr persist)
  (enforce lambda-expression? expr)
  (context-simplify-installer
   context expr 
   (type-preferred-qc-transform context expr persist))
  (context-apply-constructor-simplifier
   context expr persist))
 
		 
		  

(define CONTEXT-SIMPLIFY-LAMBDA-APPLICATION
  LAMBDA-APPLICATION-LOGICAL-TRANSFORM)

;;; Formerly
;;; (make-simplification-procedure
;;;   lambda-application-logical-transform
;;;   lambda-application-inverse)


(define (QUASI-EQUALITY-LOGICAL-TRANSFORM context expr persist)
  (destructure (((lhs rhs) (expression-quasi-components expr)))
    (cond ((context-trivially-entails-quasi-equal? context lhs rhs) truth)
	  ((or (context-trivially-entails-defined? context lhs)
	       (context-trivially-entails-defined? context rhs))
	   (equality-logical-transform context (equality lhs rhs) persist))
	  (else
	   (quasi-equality-simplifier
	    (list 
	     (context-simplify-1 context lhs persist)
	     (context-simplify-1 context rhs persist)))))))

(define (STRENGTHENED-QUASI-EQUALITY-LOGICAL-TRANSFORM context expr persist)
  (destructure (((lhs rhs) (expression-quasi-components expr)))
    (quasi-equality-logical-transform-1 context lhs rhs persist)))

(define (quasi-equality-logical-transform-1 context lhs rhs persist)
  (cond ((context-trivially-entails-quasi-equal? context lhs rhs) truth)
	((or (context-immediately-entails-defined? context lhs)
	     (context-immediately-entails-defined? context rhs))
	 (equality-logical-transform context (equality lhs rhs) persist))
	((and (context-entails-undefined? context lhs 0)
	      (context-entails-undefined? context rhs 0))
	 truth)
	(else
	 (quasi-equality-simplifier
	  (list 
	   (context-simplify-1 context lhs persist)
	   (context-simplify-1 context rhs persist))))))

(define CONTEXT-SIMPLIFY-QUASI-EQUALITY
  strengthened-QUASI-EQUALITY-LOGICAL-TRANSFORM)

(define (EQUALITY-LOGICAL-TRANSFORM context expr persist)
  (cond
   ((context-trivially-entails-equal?
     context (expression-lhs expr) (expression-rhs expr))
    truth)
   ((context-entails-unequal?
     context (expression-lhs expr) (expression-rhs expr) persist)
    falsehood)
   (else
    (let ((lhs 
	   (context-simplify-1 context (expression-lhs expr) (-1+ persist)))
	  (rhs
	   (context-simplify-1 context (expression-rhs expr) (-1+ persist))))	
      (cond ((formula? lhs)				;really should be a biconditional 
	     (context-simplify context
			       (biconditional lhs rhs)
			       (-1+ persist)))
	    ((context-immediately-entails-equal? context lhs rhs) truth) 
	    (else 	       
	     (equality-simplifier (list lhs rhs))))))))

(define CONTEXT-SIMPLIFY-EQUALITY
  EQUALITY-LOGICAL-TRANSFORM)

(define (SORT-CONVERGENCE-LOGICAL-TRANSFORM context expr persist)
  (if (convergence-in-sort? expr)
      (let ((term (car (expression-components expr)))
	    (sort (expression-sorting (cadr (expression-components expr)))))
	(cond ((context-immediately-entails-defined-in-sort? context term sort)
	       truth)
	      ((sorts-necessarily-disjoint? (expression-sorting term) sort)
	       falsehood)
	      ((context-entails-undefined? context term persist)
	       falsehood)
	      ((or (sort-necessarily-included? (expression-sorting term) sort)
		   (theory-coercion-everywhere-defined? (context-theory context)
							(expression-sorting term)
							sort))
	       (context-simplify-convergence context (is-defined term) persist)) 
	      (else
	       (let ((simp
		      (theory-context-simplify-convergence-in-sort context expr persist)))
		 (if (not (convergence-in-sort? simp))
		     (context-simplify context simp persist)
		     (is-defined-in-sort-simplifier
		      (list
		       (convergence-term simp)
		       (cadr (expression-components expr)))))))))
      (context-simplify-1 context expr (-1+ persist))))


(define context-simplify-sort-convergence 
  SORT-CONVERGENCE-LOGICAL-TRANSFORM)

;;; Formerly:
;;; 
;;; (make-simplification-procedure
;;;    sort-convergence-logical-transform
;;;    convergence-in-sort?)

(define (CONVERGENCE-LOGICAL-TRANSFORM context expr persist)
  (if (convergence? expr)
      (let ((term (car (expression-components expr))))
	(cond ((context-trivially-entails-defined? context term)
	       truth)
	      ((context-entails-undefined? context term persist)
	       falsehood)
	      ((convergence-simplify-conditional-term context term persist))
	      (else
	       (let ((simp
		      (theory-context-simplify-convergence context expr persist)))
		 (if (not (convergence? simp))
		     (context-simplify context simp persist)
		     (is-defined-simplifier (expression-components simp)))))))
      (context-simplify-1 context expr (-1+ persist))))

(define context-simplify-convergence
  CONVERGENCE-LOGICAL-TRANSFORM)

(define (convergence-simplify-conditional-term context term persist)
  (and
   (conditional-term? term)
   (let ((incrementer (constructor-lc-incrementer if-term))
	 (components (expression-components term)))
     (if (every? (lambda (n)				;ensure consequent & 
		   (context-entails-defined?		;alternative defined
		    (context-add-assumptions
		     context
		     (incrementer components n))
		    (nth components n)	
		    persist))
		 '(1 2))
	 truth
	 (let ((tests
		(map (lambda (n)			;check whether consequent & 
		       (context-entails-undefined?	;alternative UNdefined
			(context-add-assumptions
			 context
			 (incrementer components n))
			(nth components n)	
			persist))
		     '(1 2))))
	   (cond ((every? true? tests) falsehood)

		 ;; jt and farmer changed this. There were two problems previously:
		 ;; (a) (- 2 i) was (- 3 i)
		 ;; (b) The first term of the conjunction depends on
		 ;; the value of (index-of-any tests).
                 ;; 
                 ;; The code below should do the following.  Assume TERM has
                 ;; the form if(a,b,c) and either b or c is undefined.  If b
                 ;; is undefined, (not(a) and #(c)) is simplified and returned.
                 ;; Otherwise, (a and #(b)) is simplified and returned.

		 ((index-of-any tests)
		  =>
		  (lambda (i)
		    (context-simplify-1
		     context
		     (conjunction (if (= i 0)
				      (negation (nth components 0))
				      (nth components 0))
				  (is-defined
				   (nth components (- 2 i))))
		     persist)))
		 (else '#f)))))))

(define (iota-logical-transform context expr persist)
  (if (iota-expression? expr)
      (let ((i-var (car (binding-variables expr)))
	    (conjuncts (ultimate-conjuncts (list (binding-body expr)))))
	(cond ((any? necessarily-false? conjuncts)
	       (undefined (expression-sorting i-var)))
	      ((any
		(lambda (c)
		  (and (equation? c)
		       (or (and (eq? (expression-lhs c) i-var)
				(not (memq? i-var (expression-free-variables (expression-rhs c))))
				(expression-rhs c))
			   (and (eq? (expression-rhs c) i-var)
				(not (memq? i-var (expression-free-variables (expression-lhs c))))
				(expression-lhs c)))))
		conjuncts)
	       =>
	       (lambda (term)
		 (context-simplify-1
		  context
		  (if-term
		   (conjunction-simplifier
		    (cons (defined-in term (expression-sorting i-var))
			  (let ((subst (list (cons i-var term))))
			    (map
			     (lambda (c)
			       (apply-substitution-fastidiously subst c))
			     conjuncts))))
		   term
		   (undefined (expression-sorting i-var)))
		  persist)))
	      (else (context-apply-constructor-simplifier context expr persist))))
      (context-simplify-1 context expr persist)))

(define (FORSOME-LOGICAL-TRANSFORM context expr persist)
  (if (not (existential? expr))
      (context-apply-constructor-simplifier context expr persist)
      (cond ((application? (binding-body expr))
	     (forsome-application-transform context expr persist))
	    ((or (equation? (binding-body expr))
		 (conjunction? (binding-body expr)))
	     (forsome-conjunction-transform context expr persist))
	    (else (context-apply-constructor-simplifier context expr persist)))))

;;; The job of this procedure is to find, within the body of an existential,
;;; any equations involving variables of quantification.  When the other side
;;; of the eqn does *not* contain any variables of quantification free, we can
;;; delete that variable, and plug in its value throughout the rest of the
;;; body.  Note that we must include the definedness in sort of the value. 

(define (forsome-conjunction-transform context expr persist)
  (let ((nbvs (expression-newly-bound-variables expr))
	(filtered-context (requantification-filter-context expr context)))    
    (iterate iter ((remaining (ultimate-conjuncts (list (binding-body expr))))
		   (kept-conjuncts '())
		   (kept-vars nbvs)
		   (subst the-empty-substitution))
      (cond ((null? remaining)
	     (forsome-simplifier
	      (cons
	       (conjunction-simplifier
		(reverse!
		 (map!
		  (lambda (kc)
		    (context-simplify-1
		     filtered-context
		     (apply-substitution subst kc)
		     persist))
		  kept-conjuncts)))
	       kept-vars)))
	    ((let ((c (car remaining)))
	       (and (equation? c)
		    (memq? (expression-lhs c) kept-vars)
		    (null-intersection? (expression-free-variables (expression-rhs c))
					kept-vars)
		    (expression-lhs c)))
	     =>
	     (lambda (var)
	       (let ((c (car remaining)))
		 (iter
		  (cdr remaining)
		  (cons
		   (defined-in (expression-rhs c) (expression-sorting var))
		   kept-conjuncts)
		  (delete-set-element var kept-vars)
		  (add-subst-component
		   (make-subst-component
		    var
		    (context-simplify-1
		     filtered-context
		     (expression-rhs c)
		     persist))
		   subst)))))

	    ((let ((c (car remaining)))
	       (and (equation? c)
		    (memq? (expression-rhs c) kept-vars)
		    (null-intersection? (expression-free-variables (expression-lhs c))
					kept-vars)
		    (expression-rhs c)))
	     =>
	     (lambda (var)
	       (let ((c (car remaining)))
		 (iter
		  (cdr remaining)
		  (cons
		   (defined-in (expression-lhs c) (expression-sorting var))
		   kept-conjuncts)
		  (delete-set-element var kept-vars)
		  (add-subst-component
		   (make-subst-component
		    var
		    (context-simplify-1
		     filtered-context
		     (expression-lhs c)
		     persist))
		   subst)))))

	    (else
	     (iter
	      (cdr remaining)
	      (cons (car remaining) kept-conjuncts)
	      kept-vars
	      subst))))))

(define (forsome-application-transform context expr persist)
  (let ((body (binding-body expr))
	(newly-bound (binding-variables expr)))
    (iterate iter ((args (arguments body))
		   (domains (expression-domains (operator body)))
		   (subst the-empty-substitution))
      (cond ((null? args)
	     (quantifier-trivial-variable-simplifier
	      forsome
	      (cons
	       (context-simplify-1
		(requantification-filter-context expr context)
		(apply-substitution subst body)
		persist)
	       (map
		(lambda (v)
		  (apply-substitution subst v))
		newly-bound))))
	    ((let ((arg (car args)))
	       (and (memq? arg newly-bound)
		    (sort-necessarily-included?
		     (car domains)
		     (expression-sorting arg))
		    arg))
	     =>
	     (lambda (var)
	       (iter (cdr args)
		     (cdr domains)
		     (add-subst-component
		      (make-subst-component
		       var
		       (new-variable (car domains) (name var) (variables body)))
		      subst))))
	    (else
	     (iter (cdr args)
		   (cdr domains)
		   subst))))))

(define (if-logical-transform context expr persist)
  (if (conditional-term? expr)
      (destructure (((test conseq alt) (expression-components expr)))
	(let ((test-1 (context-simplify-1 context test persist)))
	  (cond ((truth? test-1)
		 (context-simplify-1 context conseq persist))
		((falsehood? test-1)
		 (context-simplify-1 context alt persist))
		(else
		 (if-term-simplifier
		  (simplify-components-in-context
		   if-term
		   (list test-1 conseq alt)
		   context
		   persist
		   context-simplify-1))))))))
	
(define (CONTEXT-TRIVIALLY-ENTAILS-QUASI-EQUAL? context term1 term2)
  (and
   (sorts-may-overlap? (expression-sorting term1)
		       (expression-sorting term2))
   (or (context-exprs-quasi-equal? context term1 term2)
       (alpha-equivalent? term1 term2)
       (let ((constr1 (expression-constructor term1))
	     (constr2 (expression-constructor term2)))
	 (and
	  constr1
	  (not (binding-constructor? constr1))
	  (eq? constr1 constr2)
	  (let ((comps1 (expression-components term1))
		(comps2 (expression-components term2)))
	    (and (= (length comps1) (length comps2))
		 (every?
		  (lambda (arg1 arg2)
		    (context-trivially-entails-quasi-equal? context arg1 arg2))
		  comps1
		  comps2))))))))

(define (CONTEXT-TRIVIALLY-ENTAILS-EQUAL? context term1 term2)
  (or (context-exprs-equal? context term1 term2)
      (and (or (context-trivially-entails-defined? context term1)
	       (context-trivially-entails-defined? context term2))
	   (context-trivially-entails-quasi-equal? context term1 term2))))

(define (CONTEXT-IMMEDIATELY-ENTAILS-QUASI-EQUAL? context term1 term2)
  (or (context-trivially-entails-quasi-equal? context term1 term2)
      (let ((constr1 (expression-constructor term1))
	    (constr2 (expression-constructor term2)))
	(and constr1
	     (eq? constr1 constr2)
	     (if (binding-constructor? constr1)
		 (and (equal? (binding-variables term1)
			      (binding-variables term2))
		      (context-immediately-entails-quasi-equal?
		       (rebinding-filter-context term1 context)
		       (binding-body term1)
		       (binding-body term2)))
		 (let ((comps1 (expression-components term1))
		       (comps2 (expression-components term2)))
		   (and (= (length comps1)
			   (length comps2))
			(compare-components-in-context?
			 context
			 constr1 
			 comps1
			 comps2
			 (context-low-simplification-persistence)))))))))

(define (compare-components-in-context? context constructor comps1 comps2 persist)
  (let ((incrementer (constructor-lc-incrementer constructor))
	(len (length comps1)))
    (iterate iter ((i 0))
      (if (= i len)
	  '#t				;succeed!!
	  (let ((comp1 (nth comps1 i))
		(comp2 (nth comps2 i)))
	    (if (sorts-may-overlap? (expression-sorting comp1)
				    (expression-sorting comp2))
		(let ((local-context
		       ;;
		       ;; NB. since the incrementers work from left to right, and we 
		       ;; may assume that all previous components are already known
		       ;; quasi-equal, we may apply the incrementer to comps1 alone
		       ;; without controversy. 
		       ;; 
		       (context-add-assumptions context (incrementer comps1 i))))
		  (if (truth?
		       (quasi-equality-logical-transform-1 local-context comp1 comp2 persist))
		      (iter (1+ i))
		      ;;
		      ;; Otherwise, we fail.
		      ;; 
		      '#f))
		;;
		;; Otherwise, we fail.
		;; 
		'#f))))))

(define (context-immediately-entails-equal? context term1 term2)
  (or (context-trivially-entails-equal? context term1 term2)
      (and (or (context-immediately-entails-defined? context term1)
	       (context-immediately-entails-defined? context term2))
	   (context-immediately-entails-quasi-equal? context term1 term2))))

(define (CONTEXT-ENTAILS-EQUAL? context term1 term2 persist)
  (truth? (context-simplify-equality context (equality term1 term2) persist)))

;;; Is any kind of enrichment appropriate here?
;;; 

(define (CONTEXT-ENTAILS-UNEQUAL? context term1 term2 persist)
  (ignore persist)
  (or (context-absurd?
       (context-add-assumption context (equality-simplifier (list term1 term2))))
      (let ((theory (context-theory context)))
	(and
	 (theory? theory)
	 (or (theory-exclude-term-coercion? theory term1 (expression-sorting term2))
	     (theory-exclude-term-coercion? theory term2 (expression-sorting term1))
	     (and (constant? term1)
		  (constant? term2)
		  (theory-constants-distinct? theory term1 term2)))))))
      
(define (CONTEXT-ENTAILS-DEFINED? context term persist)
  (truth? (context-simplify-convergence context (is-defined term) persist)))
  
(define (CONTEXT-TRIVIALLY-ENTAILS-DEFINED? context term)
  (or (context-expr-defined? context term)
      (theory-entails-defined? (context-theory context) term)))

(define (CONTEXT-IMMEDIATELY-ENTAILS-DEFINED? context term)
  (context-entails-defined? context term 0))

(define (CONTEXT-ENTAILS-UNDEFINED? context term persist)
  (if (necessarily-defined? term)
      '#f
      (or (necessarily-undefined? term)
	  (context-contains-entry? context (negation (is-defined term)))
	  (theory-context-entails-undefined? context term persist)
	  (and						;for if-term ...
	   (eq? (expression-constructor term) if-term)
	   (let ((incrementer (constructor-lc-incrementer if-term))
		 (components (expression-components term)))
	     (every?					;ensure 
	      (lambda (n)				;consequent & alternative undefined
		(context-entails-undefined?
		 (context-add-assumptions
		  context
		  (incrementer components n))
		 (nth components n)	
		 persist))
	      '(1 2)))))))

(define (CONTEXT-TRIVIALLY-ENTAILS-DEFINED-IN-SORT? context term sorting)
  (let ((theory (context-theory context)))
    (let ((coercion-ok?
	   (lambda (s)
	     (or (sort-necessarily-included? s sorting)
		 (theory-coercion-everywhere-defined? theory s sorting)))))
      (or (any?
	   coercion-ok?
	   (context-term-defined-sorts context term))
	  (and (coercion-ok? (expression-sorting term))
	       (context-trivially-entails-defined? context term))))))

(define (CONTEXT-IMMEDIATELY-ENTAILS-DEFINED-IN-SORT? context term sorting)
  (or
   (context-trivially-entails-defined-in-sort? context term sorting)
   (let ((critical-pairs
	  (theory-critical-subterms-and-sorts
	   (context-theory context)
	   term
	   sorting)))
     (every?
      (lambda (pair)
	(destructure (((term . sorting) pair))
	  (context-trivially-entails-defined-in-sort? context term sorting)))
      critical-pairs))))

(define (CONTEXT-ENTAILS-DEFINED-IN-SORT? context term sorting persist)
  (and (eq? (expression-type term) (type-of-sort sorting)) ;;added by JT
       (truth?
	(context-simplify-sort-convergence context (defined-in term sorting) persist))))

; When (CONTEXT-ENTAILS-SUBSTITUTION-DEFINED? c s) is met, then c
; entails that t[s] is defined whenever t is.

(define (CONTEXT-ENTAILS-SUBSTITUTION-DEFINED? context subst)
  (and (succeed? subst)
       (or (context-trivially-entails-substitution-defined? context subst)
	   (every?
	    (lambda (f)
	      (context-immediately-entails? context f))
	    (substitution-definedness-conditions subst)))))

(define (SUBSTITUTION-DEFINEDNESS-CONDITIONS subst)
  (make-set
   (map-substitution
    subst-comp->definedness-condition
    subst)))

(define (SUBST-COMP->DEFINEDNESS-CONDITION subst-comp)
  (let ((replace (replacement subst-comp))
	(s (expression-sorting (target subst-comp))))
    (if (not (sorts-may-overlap?
	      (expression-sorting replace)
	      s))
	(imps-error "subst-comp->definedness-condition: sorting mismatch ~S ~S"
		    (target subst-comp) replace))
    (constructor-simplify
     (defined-in replace s))))

(define (CONTEXT-TRIVIALLY-ENTAILS-SUBSTITUTION-DEFINED? context subst)
  (and (succeed? subst)
       (every?
	(lambda (component)
	  (destructure (((target . repl) component))
	    (context-trivially-entails-defined-in-sort?
	     context
	     repl
	     (expression-sorting target))))
	subst)))

(define (context-match-check-definedness? context expr)
  (let ((classes (context-q-classes context))
	(exoscopes (context-free-variables context)))
    (catch found
      (context-walk-matches-to-constructor-and-first-lead
       (lambda (e)
	 (and (q-seek-defined? classes e)
	      (context-trivially-entails-substitution-defined?
	       context
	       (match-under-exoscopes expr e exoscopes))
	      (found '#t)))
       context
       expr)
      '#f)))

;(define context-strong-substitution-definedness?
;  (make-simple-switch 'context-strong-substitution-definedness? boolean? '#t))
;  bind (((context-strong-substitution-definedness?) '#f))

(define (CONTEXT-STRONGLY-ENTAILS-SUBSTITUTION-DEFINED? context subst persist)
    (and (succeed? subst)
	 (or (context-trivially-entails-substitution-defined? context subst)
	     (every?
	      (lambda (f)
		(truth? (context-simplify context f persist)))
	      (substitution-definedness-conditions subst)))))

(define context-do-match?
  (make-simple-switch 'context-do-match? boolean? '#t))

(define (CONTEXT-MATCH context expr pattern persist)
  (let ((substitution-defined?
	 (lambda (subst)
	   (context-strongly-entails-substitution-defined?
	    context
	    subst
	    (subtract1 persist)))))

    (if (or (< persist 0)
	    (not (context-do-match?)))
	(fail)

	(bind (((context-do-match?) '#f))
	  (let ((subst
		 (match-under-exoscopes expr pattern
					(context-free-variables context))))
	    (if (substitution-defined? subst)
		subst
		(fail)))))))


;;; The use of the following switch is an ad-hoc device to prevent circularity
;;; in context-universal-instance?.  The selection of the persistence
;;; (currently 1) in the call to context-match is also somewhat ad hoc.  

(define call-context-universal-instance?
  (make-simple-switch 'call-context-universal-instance? boolean? '#t))

(define (CONTEXT-UNIVERSAL-INSTANCE? context formula persist)
  (if (call-context-universal-instance?)
      (and (not-negative? persist)
	   (bind (((call-context-universal-instance?) '#f))
	     (trivial-backchaining-opportunity context formula)))
      '#f))

;; (context-any-matching-entry?
;; 	      (lambda (pattern)
;; 		(or (alpha-equivalent? formula pattern)
;; 		    (succeed? (context-match context formula pattern 1))))
;; 	      context
;; 	      formula)

(define (tc-prove-requirements context reqs persist)	   
  (every?
   (lambda (req)
     (let ((simplified-req
	    (context-simplify context req (-1+ persist))))
       (cond ((truth? simplified-req))
	     ((accumulate-undischarged-minor-premises?)
	      (push undischarged-minor-premises
		    (build-sequent context simplified-req))
	      '#t)
	     (else '#f))))
   reqs))

(define (tc-refute-requirements context reqs persist)
  (ignore persist)
  (any?
   (lambda (req)
     (if (dont-work-on-definedness)
	 (falsehood? (recursive-constructor-simplify req))
	 (bind (((dont-work-on-definedness) '#t))
	   (context-immediately-entails? context (push-not req)))))
   reqs))

(define (tc-vigorously-prove-requirements context reqs)	   
  (every?
   (lambda (req)
     (cond ((context-entails? context req))
	   ((accumulate-undischarged-minor-premises?)
	    (push undischarged-minor-premises
		  (build-sequent
		   context
		   (context-simplify context req (context-simplification-persistence))))
	    '#t)
	   (else '#f)))
   reqs))
