;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald RULES)


; A rule is an object which, when called on a list of sequents (and '#fs),
; returns either an inference extending the list or else FAIL. 

(define-predicate RULE?)

; Many rules are generated from other objects, such as a schematic inference, a
; "special rule" or a procedure like CONTRAPOSITION-RULE or FORCE-SUBSTITUTION.

(define-operation (RULE-GENERATOR rule)
  (if (rule? rule)
      rule
      '#f))

(define-operation (RULE-MAJOR-PREMISES-PROC rule hyps)(list (car hyps)))

; Some rules are sound relative to some theories but not others.
; RULE-SOUNDNESS-PREDICATE is an operation that returns a predicate of
; theories.  It promises to return true for a particular theory iff the rule is
; sound in that theory.  The examples of soundness predicates envisaged are:
; 1.  always, for logically sound rules;
; 2.  (lambda (theory) (sub-theory? fixed-theory theory)), for a rule that
;     depends for its soundness on the axioms of FIXED-THEORY.

(define-operation (RULE-SOUNDNESS-PREDICATE rule)
  (imps-error "RULE-SOUNDNESS-PREDICATE: method undefined."))

; For objects that generate rules, the operation ->RULE performs the coercion.
; By default, it is the identity on any object that is already a rule.

(define-operation (->rule generator)
  (if (rule? generator)
      generator
      (imps-error "->rule: Don't know how to coerce ~S into a rule." generator)))

; Some basic rules are given first:

(define NODE-SIMPLIFICATION
  (object 
      (lambda (sequents)
	(let* ((conclusion (last sequents))
	       (context (sequent-context conclusion))
	       (assertion (sequent-assertion conclusion))
	       (simplified-assertion
		(syllogistic-inference-simplify context assertion)))
	  (if (eq? assertion simplified-assertion)
	      '#f
	      (build-inference
	       node-simplification
	       (if (truth? simplified-assertion)
		   nil
		   (list (build-sequent context simplified-assertion)))
	       conclusion))))
    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) (if (simplify-quasi-constructors-messily?)
		     'insistent-simplification
		     'simplification))))

;;;(define WEAK-NODE-SIMPLIFICATION
;;;  (object 
;;;      (lambda (sequents)
;;; 	(let* ((conclusion (last sequents))
;;; 	       (context (sequent-context conclusion))	 
;;; 	       (assertion (sequent-assertion conclusion))
;;; 	       (simplified-assertion
;;; 		((prune-simplification context-insistently-simplify)
;;;		 context
;;;		 assertion
;;;		 (context-simplification-persistence))))
;;; 	  (if (eq? assertion simplified-assertion)
;;; 	      '#f
;;; 	      (build-inference
;;; 	       weak-node-simplification
;;; 	       (if (truth? simplified-assertion)
;;; 		   nil
;;; 		   (list (build-sequent context simplified-assertion)))
;;; 	       conclusion))))
;;;    ((rule? self) '#t)
;;;    ((rule-soundness-predicate self) (always '#t))
;;;    ((name self) 'weak-simplification)))

;; (define WEAK-NODE-SIMPLIFICATION node-simplification)

;;;(define BETA-REDUCTION-RULE
;;;  (object 
;;;      (lambda (sequents)
;;;	(let* ((conclusion (last sequents))
;;;	       (context (sequent-context conclusion))
;;;	       (assertion (sequent-assertion conclusion))
;;;	       (beta-reduced-assertion
;;;		(context-beta-reduce context assertion)))
;;;	  (if (eq? assertion beta-reduced-assertion)
;;;	      '#f
;;;	      (build-inference
;;;	       beta-reduction-rule
;;;	       (if (truth? beta-reduced-assertion)
;;;		   nil
;;;		   (list (build-sequent context beta-reduced-assertion)))
;;;	       conclusion))))
;;;    ((rule? self) '#t)
;;;    ((rule-soundness-predicate self) (always '#t))
;;;    ((name self) (if (simplify-quasi-constructors-messily?)
;;;		     'insistent-beta-reduction
;;;		     'beta-reduction))))

;;;(define SEQUENT-ENTAILMENT-RULE
;;;  (object
;;;      (lambda (sequents)
;;;	(or (null? (cdr sequents))
;;;	    (imps-error "sequent-entailment->inference: Too many sequents ~S" sequents))
;;;	(let ((conc (car sequents)))
;;;	  (if (sequent-entailed? conc)
;;;	      (build-inference sequent-entailment-rule nil conc)
;;;	      '#f)))	  
;;;    ((rule? self) '#t)
;;;    ((rule-soundness-predicate self) (always '#t))
;;;    ((name self) 'sequent-entailment-rule)))

(define NODE-SIMPLIFICATION-WITH-MINOR-PREMISES
  (object 
      (lambda (sequents)
	(let ((conclusion (last sequents)))
	  (let ((context (sequent-context conclusion)) 
		(assertion (sequent-assertion conclusion)))
	    (receive (simplified-assertion minor-premises)
	      (simplify-with-minor-premises context assertion)
	      (if (eq? assertion simplified-assertion)
		  '#f
		  (build-inference
		   node-simplification-with-minor-premises
		   (if (truth? simplified-assertion)
		       minor-premises
		       (cons (build-sequent context simplified-assertion)
			     minor-premises))
		   conclusion))))))
    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) 'simplification-with-minor-premises)))
  

;;;(define TAUTOLOGY-RULE
;;;  (object
;;;      (lambda (sequents)
;;;	(or (null? (cdr sequents))
;;;	    (imps-error "TAUTOLOGY-RULE: Too many sequents ~S" sequents))
;;;	(let ((conc (car sequents)))
;;;	  (if (tautologically-entails? (sequent-context conc) (sequent-assertion conc))
;;;	      (build-inference tautology-rule nil conc)
;;;	      '#f)))	  
;;;    ((rule? self) '#t)
;;;    ((rule-soundness-predicate self) (always '#t))
;;;    ((name self) 'tautology)))

(define WEAKENING-RULE 
  (object
      (lambda (sequents)
	(or (null? (cddr sequents))
	    (imps-error "WEAKENING-RULE: Too many sequents ~S" sequents))
	(let ((conc (last sequents))
	      (hyp (car sequents)))
	  (if (and (not (eq? conc hyp))
		   (sequent-entails-sequent? hyp conc))
	      (build-inference weakening-rule (list hyp) conc)
	      '#f)))
    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) 'weakening))) 

(define ASSERTION-STRENGTHENING-RULE WEAKENING-RULE)

;;; Formerly said:
;;    (object
;;       (lambda (sequents)
;; 	(or (fx= (length sequents) 2)
;; 	    (imps-error
;; 	     "assertion-strengthening-rule: requires exactly 2 sequents"))
;; 	(let ((conc (cadr sequents))
;; 	      (hyp (car sequents)))
;; 	  (if (and (eq? (sequent-context conc)
;; 			(sequent-context hyp))
;; 		   (not (eq? conc hyp))
;; 		   (subset? (ultimate-conjuncts (list (sequent-assertion conc)))
;; 			    (ultimate-conjuncts (list (sequent-assertion hyp)))))
;; 	      (build-inference assertion-strengthening-rule (list hyp) conc)
;; 	      '#f)))
;;     ((rule? self) '#t)
;;     ((rule-soundness-predicate self) (always '#t))
;;     ((name self) 'assertion-strengthening-rule))

(let ((theorem-rules (make-table 'theorem-rules)))
  (define (THEOREM-DISCHARGE-RULE theorem)
    (or (table-entry theorem-rules theorem)
	(labels
	    ((rule 
	      (object
		  (lambda (sequents)
		    (if (not (null? (cddr sequents))) ;more than two seqs not ok
			(imps-error
			 "theorem-discharge-rule : Too many sequents ~S" sequents))
		    (let ((conclusion (last sequents))
			  (given-premise (and (not (null? (cdr sequents)))
					      (car sequents))))
		      (theorem-discharge-rule-aux theorem conclusion given-premise rule)))
		((rule? self) '#t)
		((rule-soundness-predicate self)
		 (lambda (theory)
		   (theory-theorem? theory theorem))
		 ;;
		 ;;Formerly:
		 ;;
		 ;;
		 ;;(any?
		 ;;   (lambda (thm)
		 ;;    (eq? theorem (theorem-formula thm)))
		 ;;  (theory-theorems theory))
		 )
		((name self) 'theorem-assumption))))
	  (set (table-entry theorem-rules theorem) rule)
	  rule))))

(define (theorem-discharge-rule-aux theorem conclusion given-premise rule)
  (let ((premise
	 (build-sequent
	  (context-add-assumption (sequent-context conclusion) theorem)
	  (sequent-assertion conclusion))))
    (cond  ((eq? premise conclusion)			;no action?
	    '#f)
	   ((not given-premise)				;generate premise
	    (build-inference rule
			     (list premise)
			     conclusion))
	   ((sequent-entails-sequent?			;for use as recognizer, 
	     given-premise premise)			;check whether given-premise 
	    (build-inference rule			;suffices
			     (list given-premise) 
			     conclusion))
	   (else '#f))))				;otherwise fail

(define (UNFOLD-DEFINED-CONSTANT-RULE constant definition paths)
    (let ((defining-expr (if (definition? definition)
			     (definition-defining-expr definition)
			     (nth (recursive-definition-rec-defining-expr-list definition)
				  (pos eq?
				       constant 
				       (recursive-definition-constant-list definition))))))
  (labels
      ((rule
	(object
	 (lambda (sequents)
	   (let* ((major-premise (if (null? (cdr sequents))
				     '#f
				     (car sequents)))
		  (conclusion (last sequents))
		  (assertion (sequent-assertion conclusion))
		  (context (sequent-context conclusion)))
	     (if (not (null? (cddr sequents)))
		 (imps-error "UNFOLD-DEFINED-CONSTANT-RULE: too many hypotheses ~S"
			     (reverse (cdr (reverse sequents)))))
	     (let ((hypothesis-assertion
		    (expand-constant-occurrences constant defining-expr assertion paths)))
	       (cond ((eq? hypothesis-assertion assertion)
		      (fail))
		     ((truth? hypothesis-assertion)
		      (build-inference rule '#f conclusion))
		     ((and (not (null? major-premise))
			   (sequent-entails-sequent?
			    major-premise
			    (build-sequent context hypothesis-assertion)))
		      (build-inference rule (list major-premise) conclusion))
		     ((not (null? major-premise)) (fail))
		     (else
		      (build-inference rule
				       (list
					(build-sequent
					 context
					 hypothesis-assertion))
				       conclusion))))))
	 ((rule? self) '#t)
	 ((rule-generator self) unfold-defined-constant-rule) 
	 ((rule-soundness-predicate self) (always '#t))
	 ((name self) 'defined-constant-unfolding))))
  rule)))

;;;(define (UNFOLD-CONSTANT-DEFINITION-RULE constant definition path)
;;;    (let ((defining-expr (if (definition? definition)
;;;			     (definition-defining-expr definition)
;;;			     (nth (recursive-definition-rec-defining-expr-list definition)
;;;				  (pos eq?
;;;				       constant 
;;;				       (recursive-definition-constant-list definition))))))
;;;  (labels
;;;      ((rule
;;;	(object
;;;      (lambda (sequents)
;;;
;;;	(let* ((major-premise (if (null? (cdr sequents))
;;;				  nil
;;;				  (car sequents)))
;;;	       (conclusion (last sequents))
;;;	       (assertion (sequent-assertion conclusion))
;;;	       (context (sequent-context conclusion)))
;;;	  (if (not (null? (cddr sequents)))
;;;	      (imps-error "UNFOLD-CONSTANT-DEFINITION-RULE: too many hypotheses ~S"
;;;			  (reverse (cdr (reverse sequents)))))
;;;	  (let ((hypothesis-assertion
;;;		 (expand-constant constant defining-expr assertion path)))
;;;	    (cond ((eq? hypothesis-assertion assertion)
;;;		   (fail))
;;;		  ((truth? hypothesis-assertion)
;;;		   (build-inference rule
;;;				    nil
;;;				    conclusion))
;;;		  ((and (not (null? major-premise))
;;;			(sequent-entails-sequent?
;;;			 major-premise
;;;			 (build-sequent context hypothesis-assertion)))
;;;		   (build-inference rule
;;;				    (list major-premise)
;;;				    conclusion))
;;;		  ((not (null? major-premise)) (fail))
;;;		  (else
;;;		   (build-inference rule
;;;				    (list
;;;				     (build-sequent
;;;				      context
;;;				      hypothesis-assertion))
;;;				    conclusion))))))
;;;    ((rule? self) '#t)
;;;    ((rule-generator self) unfold-constant-definition-rule) 
;;;    ((rule-soundness-predicate self) (always '#t))
;;;    ((name self) 'unfold-constant-definition-rule))))
;;;  rule)))
;;;
;;;(define (UNFOLD-RECURSIVE-DEFINITION-RULE recursive-definition n)
;;;  (labels
;;;      ((rule
;;;	(object
;;;      (lambda (sequents)
;;;
;;;	(let* ((major-premise (if (null? (cdr sequents))
;;;				  nil
;;;				  (car sequents)))
;;;	       (conclusion (last sequents))
;;;	       (assertion (sequent-assertion conclusion))
;;;	       (context (sequent-context conclusion)))
;;;	  (if (not (null? (cddr sequents)))
;;;	      (imps-error "unfold-recursive-definition-rule: too many hypotheses ~S"
;;;			  (reverse (cdr (reverse sequents)))))
;;;	  (let* ((path (nth (reverse
;;;			     (paths-to-occurrences
;;;			      assertion
;;;			      (recursive-definition-constant recursive-definition) -1))
;;;			     n))
;;;		 (hypothesis-assertion
;;;		  (unfold-recursive-definition recursive-definition assertion path)))
;;;	    (cond ((eq? hypothesis-assertion assertion)
;;;		   (fail))
;;;		  ((truth? hypothesis-assertion)
;;;		   (build-inference rule
;;;				    nil
;;;				    conclusion))
;;;		  ((and (not (null? major-premise))
;;;			(sequent-entails-sequent?
;;;			 major-premise
;;;			 (build-sequent context hypothesis-assertion)))
;;;		   (build-inference rule
;;;				    (list major-premise)
;;;				    conclusion))
;;;		  ((not (null? major-premise)) (fail))
;;;		  (else
;;;		   (build-inference rule
;;;				    (list
;;;				     (build-sequent
;;;				      context
;;;				      hypothesis-assertion))
;;;				    conclusion))))))
;;;    ((rule? self) '#t)
;;;    ((rule-soundness-predicate self) (always '#t))
;;;    ((name self) 'unfold-definition-rule))))
;;;  rule))


(define (CONTRAPOSITION-RULE context-formula)
  (labels
      ((self
	(object
	    (lambda (sequents)
	      (or (null? (cdr sequents))
		  (imps-error "contraposition->inference: Too many sequents ~S" sequents))
	      (let ((conc (car sequents)))
		(if (mem? alpha-equivalent? context-formula (sequent-assumptions conc))
		    (let ((prem
			   (build-sequent
			    (context-add-assumption
			     (context-omit-assumptions
			      (sequent-context conc)
			      (set-separate
			       (lambda (assumption)
				 (alpha-equivalent? context-formula assumption))
			       (context-assumptions (sequent-context conc))))
			     ;; gently!!
			     (push-not (sequent-assertion conc)))
			    ;; gently!!
			    (push-not context-formula))))
		      (build-inference self (list prem) conc))
		    '#f)))	  
	  ((rule? self) '#t)
	  ((rule-soundness-predicate self) (always '#t))
	  ((name self) 'contraposition)
	  ((rule-generator self) contraposition-rule))))
    self))

;;;(define (BACKCHAIN-RULE context-formula)
;;;  (labels
;;;      ((self
;;;	(object
;;;	    (lambda (sequents)
;;;	      (or (null? (cddr sequents))
;;;		  (imps-error "backchain-rule: Too many sequents ~S" sequents))
;;;	      (let ((new-hyp (backchain-through-sequent
;;;			      (last sequents)
;;;			      context-formula)))
;;;				     
;;;		(and new-hyp
;;;		     (build-inference self (list new-hyp) (last sequents)))))
;;;
;;;	  ((rule? self) '#t)
;;;	  ((rule-soundness-predicate self) (always '#t))
;;;	  ((name self) 'backchain-inference)
;;;	  ((rule-generator self) backchain-rule))))
;;;    self))

(define (BACKCHAIN-RULE context-formula)
  (labels
      ((self
	(object
	    (lambda (sequents)
	      (or (null? (cddr sequents))
		  (imps-error "backchain-rule: Too many sequents ~S" sequents))
	      (let ((hyps (backchain-through-sequent
			   (last sequents)
			   context-formula)))
				     
		(and hyps
		     (build-inference self hyps (last sequents)))))

	  ((rule? self) '#t)
	  ((rule-soundness-predicate self) (always '#t))
	  ((name self) 'backchain-inference)
	  ((rule-generator self) backchain-rule))))
    self))

(define (RAISE-CONDITIONALS paths)
  (labels
      ((self
	(object
	    (lambda (sequents)
	      (or (null? (cddr sequents))
		  (imps-error "RAISE-CONDITIONALS: too many sequents ~s" sequents))
	      (let* ((conc (last sequents))
		     (hyp (and (cdr sequents)
			       (car sequents)))
		     (assertion (sequent-assertion conc))
		     (new-assertion
		      (raise-conditionals-make-assertion assertion (sort-paths paths))))
		(cond ((eq? assertion new-assertion) '#f)
		      ((not hyp)
		       (build-inference
			self
			(list (build-sequent 
			       (sequent-context conc)
			       new-assertion))
			conc))
		      ((sequent-entails-sequent?
			hyp (build-sequent 
			     (sequent-context conc)
			     new-assertion))
		       (build-inference self (list hyp) conc))
		      (else '#f))))

	  ((rule? self) '#t)
	  ((rule-soundness-predicate self) (always '#t))
	  ((name self) 'raise-conditional-inference)
	  ((rule-generator self) raise-conditionals))))
    self))

(define (RAISE-CONDITIONALS-MAKE-ASSERTION assertion paths)
  (let* ((quads (map
		 (lambda (path) 
		   (frontpath-lastformula-backpath-conditional assertion path))
		 paths))
	 (good-quads (remove-bad-quadruples quads)))
    (iterate loop ((quads good-quads) (assertion assertion))
      (if (null? quads)
	  assertion
	  (let* ((front-path (nth (car quads) 0))
		 (last-formula (nth (car quads) 1))
		 (back-paths (nth (car quads) 2))
		 (components (expression-components (nth (car quads) 3)))
		 (test (nth components 0))
		 (conseq (nth components 1))
		 (altern (nth components 2))
		 (new-conseq (iterate loop ((new-conseq last-formula) 
					    (back-paths back-paths))
			       (if (null? back-paths)
				   new-conseq
				   (loop (substitution-at-path 
					  new-conseq
					  conseq 
					  (car back-paths))
					 (cdr back-paths)))))
		 (new-altern (iterate loop ((new-altern last-formula) 
					    (back-paths back-paths))
			       (if (null? back-paths)
				   new-altern
				   (loop (substitution-at-path 
					  new-altern
					  altern 
					  (car back-paths))
					 (cdr back-paths))))))
	    (loop (cdr quads)
		  (substitution-at-path
		   assertion
		   (if-form test new-conseq new-altern)
		   front-path)))))))

(define (FRONTPATH-LASTFORMULA-BACKPATH-CONDITIONAL assertion path)
  (iterate loop ((path path)
		 (rev '())
		 (host assertion)
		 (last assertion)
		 (front-rev '())
		 (back-rev '()))
    (cond ((and (null? path)
		(conditional? host))
	   (list (reverse front-rev) last (reverse back-rev) host))
	  ((null? path)
	   (imps-error "FRONTPATH-LASTFORMULA-BACKPATH-CONDITIONAL: ~S ~A."
		        path "is not a path to a conditional"))
	  (else
	   (receive (new-host new-path)
	     (host-and-path-after-step host path)
	     (let ((new-rev (cons (car path) rev)))
	       (if (formula? new-host)
		   (loop new-path new-rev new-host new-host new-rev '())
		   (let ((new-back-rev (cons (car path) back-rev)))
		     (loop new-path new-rev new-host last front-rev new-back-rev)))))))))

(define (REMOVE-BAD-QUADRUPLES quads)
  (let ((new-quads (remove-bad-quadruples-aux quads)))
    (iterate loop ((quads new-quads) (good-quads '()))
      (if (null? quads)
	  good-quads
	  (let* ((first-quad (car quads))
		 (first-front-path (nth first-quad 0))
		 (first-last-formula (nth first-quad 1))
		 (first-back-paths (nth first-quad 2))
		 (first-conditional (nth first-quad 3))
		 (new-back-paths 
		  (set-separate
		   (lambda (back-path)
		     (null-intersection? 
		      (bound-variables-on-path first-last-formula back-path)
		      (free-variables first-conditional)))
		   first-back-paths)))
	    (if (null? new-back-paths)
		(loop (cdr quads) good-quads)
		(loop (cdr quads) 
		      (cons (list first-front-path 
				  first-last-formula
				  new-back-paths 
				  first-conditional)
			    good-quads))))))))

(define (REMOVE-BAD-QUADRUPLES-AUX quads)
  (iterate loop1 ((quads quads) (good-quads '()))
    (if (null? quads)
	good-quads
	(let* ((first-quad (car quads))
	       (first-front-path (nth first-quad 0))
	       (first-last-formula (nth first-quad 1))
	       (first-back-path (nth first-quad 2))
	       (first-conditional (nth first-quad 3)))
	  (if (eq? (length quads) 1)
	      (cons (list first-front-path 
			  first-last-formula
			  (list first-back-path)
			  first-conditional)
		    good-quads)
	      (iterate loop2 ((qs (cdr quads))
			      (bad-qs '()) 
			      (back-paths (list first-back-path)))
		(if (null? qs)
		    (loop1 (set-diff quads (cons first-quad bad-qs))
			   (cons (list first-front-path 
				       first-last-formula 
				       back-paths 
				       first-conditional)
				 good-quads))
		    (let* ((q (car qs))
			   (front-path (nth q 0))
			   (back-path (nth q 2))
			   (conditional (nth q 3)))
		      (cond ((and (equal? first-front-path front-path)
				  (eq? first-conditional conditional))
			     (loop2 (cdr qs) 
				    (cons q bad-qs) 
				    (cons back-path back-paths)))
			    ((or (and (equal? first-front-path front-path)
				      (not (eq? first-conditional conditional)))
				 (and (not (equal? first-front-path front-path))
				      (path-extends? first-front-path front-path)))
			     (loop2 (cdr qs) 
				    (cons q bad-qs) 
				    back-paths))
			    ((and (not (equal? first-front-path front-path))
				  (path-extends? front-path first-front-path))
			     (loop1 (set-diff quads (cons first-quad bad-qs)) good-quads))
			    (else
			     (loop2 (cdr qs) bad-qs back-paths)))))))))))

;;;(define (CONDITIONAL-TERM->FORMULA path)
;;;  (labels
;;;      ((self
;;;	(object
;;;	    (lambda (sequents)
;;;	      (or (null? (cddr sequents))
;;;		  (imps-error "conditional-term->formula: Too many sequents ~S" sequents))
;;;	      (let* ((conc (last sequents))
;;;		     (hyp (and (cdr sequents)
;;;			       (car sequents)))
;;;		     (new-assertion
;;;		      (conditional-term->formula-make-assertion
;;;		       (sequent-assertion conc)
;;;		       path)))
;;;		(cond ((not new-assertion) '#f)
;;;		      ((not hyp)
;;;		       (build-inference
;;;			self
;;;			(list (build-sequent 
;;;			       (sequent-context conc)
;;;			       new-assertion))
;;;			conc))
;;;		      ((sequent-entails-sequent?
;;;			hyp (build-sequent 
;;;			     (sequent-context conc)
;;;			     new-assertion))
;;;		       (build-inference self (list hyp) conc))
;;;		      (else '#f))))
;;;
;;;	  ((rule? self) '#t)
;;;	  ((rule-soundness-predicate self) (always '#t))
;;;	  ((name self) 'raise-conditional)
;;;	  ((rule-generator self) conditional-term->formula))))
;;;    self))
;;;
;;;(define (conditional-term->formula-make-assertion assertion path)
;;;  (let ((ct (follow-path assertion path)))
;;;    (or (conditional-term? ct)
;;;	(imps-error "conditional-term->formula: Bad path ~S leads to ~S" path ct))
;;;    (receive (path-to-last-formula last-formula)
;;;      (iterate iter ((path path)
;;;		     (rev  nil)
;;;		     (host assertion)
;;;		     (last assertion)
;;;		     (last-rev nil))
;;;	(if (null? path)
;;;	    (return (reverse! last-rev) last)
;;;	    (receive (new-host new-path)
;;;	      (host-and-path-after-step host path)
;;;	      (let ((new-rev (cons (car path) rev)))
;;;		(if (formula? new-host)
;;;		    (iter new-path new-rev new-host new-host new-rev)
;;;		    (iter new-path new-rev new-host last last-rev))))))
;;;      (let* ((components (expression-components ct))
;;;	     (path-rest (do ((path path (cdr path))
;;;			     (part path-to-last-formula (cdr part)))
;;;			    ((null? part)
;;;			     path)))
;;;	     (pbvs (bound-variables-on-path last-formula path-rest)))
;;;	(if (non-null-intersection? pbvs (free-variables ct))
;;;	    '#f
;;;	    (let ((test (nth components 0))
;;;		  (conseq (nth components 1))
;;;		  (alt (nth components 2)))
;;;	      (substitution-at-path
;;;	       assertion
;;;	       (if-form test
;;;			(substitution-at-path last-formula conseq path-rest)
;;;			(substitution-at-path last-formula alt path-rest))
;;;	       path-to-last-formula)))))))

(define (PARITY-AT-PATH formula path)
  (iterate iter ((formula formula)
		 (path path)
		 (so-far 1))
    (cond ((zero? so-far) 0)
	  ((null? path) so-far)
	  ((not (formula? formula)) 0)
	  (else
	   (receive (new-formula new-path)
	     (host-and-path-after-step formula path)
	     (iter new-formula new-path
		   (fx* so-far
			((constructor-parity (expression-constructor formula))
			 (car path)))))))))


(define (PATH-TO-POSITIVE-LOCATION? expr path)
  (< 0 (parity-at-path expr path)))

(define (VIRTUAL-PATH-TO-POSITIVE-LOCATION? expr virtual-path)
  (every?
   (lambda (path) (< 0 (parity-at-path expr path)))
   (expand-virtual-path expr virtual-path)))

;;;(define (FORCE-SUBSTITUTION path premise-expr conclusion-expr)
;;;  (labels
;;;      ((soi
;;;	(object
;;;	    (lambda (seqs)
;;;	      (or (sequent? (car seqs))
;;;		  (sequent? (last seqs))
;;;		  (imps-error "force-substitution: neither premise nor conclusion given ~S" seqs))
;;;	      (or (= (length seqs) 2)
;;;		  (imps-error "force-substitution: wrong number of sequents in ~S." seqs))
;;;	      (let ((premise
;;;		     (or (car seqs)
;;;			 (build-sequent
;;;			  (sequent-context (last seqs))
;;;			  (substitution-at-path (sequent-assertion (last seqs))
;;;						premise-expr path))))
;;;		    (conclusion
;;;		     (or (last seqs)
;;;			 (build-sequent
;;;			  (sequent-context (car seqs))
;;;			  (substitution-at-path (sequent-assertion (car seqs))
;;;						conclusion-expr path)))))
;;;		(or (and (eq? conclusion-expr
;;;			      (follow-path (sequent-assertion conclusion) path))
;;;			 (eq? premise-expr
;;;			      (follow-path (sequent-assertion premise) path)))
;;;		    (imps-error "force-substitution: mismatch ~S ~S ~S ~S ~S"
;;;				premise premise-expr conclusion
;;;				conclusion-expr path))
;;;		(let* ((minor-premise-context
;;;			(local-context-at-path (sequent-context conclusion)
;;;					       (sequent-assertion conclusion)
;;;					       path))
;;;		       (parity (parity-at-path (sequent-assertion conclusion) path))
;;;		       (minor-premise
;;;			(build-sequent
;;;			 minor-premise-context
;;;			 (xcond ((not (formula? premise-expr))
;;;				 (quasi-equality  conclusion-expr premise-expr))
;;;				;;order changed by JT, 11-5-89, 15:16:12 EST 1989
;;;				((fx= 0 parity)
;;;				 (biconditional conclusion-expr premise-expr))
;;;				((fx= 1 parity)
;;;				 (implication premise-expr conclusion-expr))
;;;				((fx= -1 parity)
;;;				 (implication conclusion-expr premise-expr))))))
;;;
;;;		  (build-inference soi (list premise minor-premise) conclusion))))
;;;	  ((rule? soi) '#t)
;;;	  ((rule-soundness-predicate soi) (always '#t))
;;;	  ((name soi) 'force-substitution)
;;;	  ((rule-generator soi) force-substitution))))
;;;    soi))
;;;

(define (FORCE-SUBSTITUTION premise-exprs conclusion-exprs paths)
  (labels
      ((soi
	(object
	    (lambda (seqs)
	      (if (null? paths) 
		  (fail)
		  (block
		    (or (sequent? (car seqs))
			(sequent? (last seqs))
			(imps-error "force-substitution: neither premise nor conclusion given ~S" seqs))
		    (or (= (length seqs) 2)
			(imps-error "force-substitution: wrong number of sequents in ~S." seqs))
		    
		    (let* ((premise
			    (or (car seqs)
				(build-sequent
				 (sequent-context (last seqs))
				 (substitutions-at-paths (sequent-assertion (last seqs))
							 premise-exprs
							 (map list paths)))))
			   (conclusion
			    (or (last seqs)
				(build-sequent
				 (sequent-context (car seqs))
				 (substitutions-at-paths (sequent-assertion (car seqs))
							 conclusion-exprs
							 (map list paths))))))
		      (if (and conclusion-exprs premise-exprs)
			  (or (and (every?
				    (lambda (conclusion-expr path)
				      (alpha-equivalent? conclusion-expr
							 (follow-path (sequent-assertion conclusion) path)))
				    conclusion-exprs paths)
			
				   (every?
				    (lambda (premise-expr path)
				      (alpha-equivalent? premise-expr
							 (follow-path (sequent-assertion premise) path)))

				    premise-exprs paths))
			      (imps-error "force-substitution: mismatch between targets and replacements")))
		      (let* ((conclusion-exprs
			      (if conclusion-exprs
				  conclusion-exprs
				  (map (lambda (path)
					 (follow-path (sequent-assertion conclusion) path))
				       paths)))

			     (premise-exprs
			      (if premise-exprs
				  premise-exprs
				  (map (lambda (path)
					 (follow-path (sequent-assertion premise) path))
				       paths)))
			     (minor-premise-contexts
			      (map (lambda (path)
				     (local-context-at-path (sequent-context conclusion)
							    (sequent-assertion conclusion)
							    path))
				   paths))
			     
			     (minor-premise-common-context
			      (build-context
			       (sequent-theory conclusion)
			       (big-cap (map context-assumptions minor-premise-contexts))))
			     
			     (parities
			      (map
			       (lambda (path)
				 (parity-at-path (sequent-assertion conclusion) path))
			       paths))
			     (minor-premise-assertions
			      (map
			       (lambda (premise-expr conclusion-expr parity)
				 (xcond ((not (formula? premise-expr))
					 (quasi-equality  conclusion-expr premise-expr))
					((fx= 0 parity)
					 (biconditional conclusion-expr premise-expr))
					((fx= 1 parity)
					 (implication premise-expr conclusion-expr))
					((fx= -1 parity)
					 (implication conclusion-expr premise-expr))))
			       premise-exprs conclusion-exprs parities))
			     (minor-premises
			      (make-set
			       (map (lambda (minor-premise-assertion)
				      (build-sequent
				       minor-premise-common-context
				       minor-premise-assertion))
				    minor-premise-assertions))))
			(build-inference soi (cons premise minor-premises) conclusion))))))
	  ((rule? soi) '#t)
	  ((rule-soundness-predicate soi) (always '#t))
	  ((name soi) 'force-substitution)
	  ((rule-generator soi) force-substitution))))
    soi))

(define FALSEHOOD-ELIMINATION
  (object
      (lambda (seqs)
	(let* ((conclusion (last seqs))
	       (premise	   (if (cdr seqs)
			       (car seqs)
			       '#f))
	       (expected (build-sequent (sequent-context conclusion) falsehood)))
	  (cond ((not conclusion) '#f)
		((not premise)
		 (build-inference
		  falsehood-elimination
		  (list expected) conclusion))
		((sequent-entails-sequent? premise expected)
		 (build-inference
		  falsehood-elimination
		  (list premise) conclusion))
		(else '#f))))

    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) 'falsehood-elimination)))

(define NEGATION-ELIMINATION
  (object
      (lambda (seqs)
	(let* ((conclusion (last seqs))
	       (premise	   (if (cdr seqs)
			       (car seqs)
			       '#f)))
	  (cond ((or (not conclusion)
		     (not premise))
		 '#f)
		((and (eq? (sequent-assertion conclusion) falsehood)
		      (negation? (sequent-assertion premise))
		      (eq? (sequent-context conclusion)
			   (context-add-assumption
			    (sequent-context premise)
			    (negation-body (sequent-assertion premise)))))
		 (build-inference negation-elimination
				  (list premise)
				  conclusion))
		(else '#f))))

    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) 'negation-elimination)))

;; Already defined in special-rules.t
;;(define disjunction-elimination-rule
;;  (object
;;      (lambda (seqs)
;;	(let* ((conclusion (last seqs))
;;	       (major-premise (if (cdr seqs)
;;				  (car seqs)
;;				  '#f)))
;;	  (cond ((or (not conclusion)
;;		     (not major-premise))
;;		 '#f)
;;		((and (disjunction? (sequent-assertion major-premise))
;;		      (context-entails-context? (sequent-context conclusion)
;;						(sequent-context major-premise)))
;;		 (let ((context (sequent-context conclusion))
;;		       (assertion (sequent-assertion conclusion))
;;		       (disjuncts (expression-components (sequent-assertion major-premise))))
;;		   (build-inference
;;		    disjunction-elimination-rule
;;		    (cons major-premise
;;			  (map
;;			   (lambda (d)
;;			     (build-sequent (context-add-assumption context d) assertion))
;;			   disjuncts))
;;		    conclusion)))
;;		(else '#f))))
;;
;;    ((rule? self) '#t)
;;    ((rule-soundness-predicate self) (always '#t))
;;    ((name self) 'disjunction-elimination)))
;;
;;(define disjunction-elimination disjunction-elimination-rule)

(define IMPLICATION-ELIMINATION-RULE 
  (object
      (lambda (seqs)
	(let ((conclusion (last seqs))
	      (premise (if (cdr seqs)
			   (car seqs)
			   '#f)))
	  (cond ((or (not conclusion)
		     (not premise))
		 '#f)
		((implication? (sequent-assertion premise))
		 (let ((p (implication-antecedent (sequent-assertion premise)))
		       (q (implication-consequent (sequent-assertion premise)))
		       (gamma (sequent-context premise)))
		   (let ((gamma+p (add-set-element p (context-assumptions gamma))))
		     (if (and
			  (every?
			   (lambda (a)
			     (mem?
			      alpha-equivalent?
			      a
			      (context-assumptions (sequent-context conclusion))))
			   gamma+p)
			  (alpha-equivalent? q (sequent-assertion conclusion))
			  ;; Formerly also allowed:
			  ;; (context-entails? (context-add-assumption
			  ;;   (sequent-context conclusion) q)
			  ;;    (sequent-assertion conclusion))
			  ) 
			 (build-inference implication-elimination-rule
					  (list premise)
					  conclusion)
			 '#f))))
		(else '#f))))

    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) 'incorporate-antecedent)))

(define implication-elimination implication-elimination-rule)

;;;(define (IMPLICATION-ELIMINATION-GENERATOR context-assumption)
;;;  (object
;;;      (lambda (seqs)
;;;	(and
;;;	 (null? (cdr seqs))				;only conclusion
;;;	 (let ((conclusion (car seqs)))
;;;	   (let ((context (sequent-context conclusion))
;;;		 (assertion (sequent-assertion conclusion)))
;;;	     (if (memq? context-assumption (context-assumptions context))
;;;		 (implication-elimination-rule
;;;		  (list (build-sequent (context-omit-assumption context context-assumption)
;;;				       (implication context-assumption assertion))
;;;			conclusion))
;;;		 '#f)))))
;;;    ((rule? self) '#t)
;;;    ((rule-soundness-predicate self) (always '#t))
;;;    ((name self) 'implication-elimination-rule)
;;;    ((rule-generator self) implication-elimination-generator)))

;;(define (disjunction-elimination-generator or-formula)
;;  (if (disjunction? or-formula)
;;      (object
;;	  (lambda (seqs)
;;	    (and
;;	     (null? (cdr seqs))				;only conclusion
;;	     (disjunction-elimination-rule
;;	      (list (build-sequent (sequent-context (car seqs))
;;				   or-formula)
;;		    (car seqs)))))
;;	
;;
;;	((rule? self) '#t)
;;	((rule-soundness-predicate self) (always '#t))
;;	((name self) 'implication-elimination-rule)
;;	((rule-generator self) implication-elimination-generator))
;;      (imps-error "disjunction-elimination-generator: Non-disjunction ~S" or-formula)))


(define EXTENSIONALITY
  (object
      (lambda (seqs)
	(let* ((conclusion (last seqs))			;assumed to be present
	       (assertion (and conclusion (sequent-assertion conclusion))))
	  (if (or (not conclusion)
		  (not (equation? assertion))
		  (not (higher-sort? (expression-sorting (expression-lhs assertion)))))
	      '#f
	      (let* ((f (expression-lhs assertion))
		     (g (expression-rhs assertion))
		     (vars (sorts->new-variables 
			    (common-enclosing-sort-list (domain-sorts f) (domain-sorts g))
			    'x
			    (free-variables conclusion))))
		    
		(cond ((predicator? f)
		       (build-inference 
			extensionality
			(list
			 (build-sequent
			  (sequent-context conclusion)
			  (apply forall (equality (apply apply-operator f vars)
						  (apply apply-operator g vars))
				 vars)))
			conclusion))
		      ((function? f)
		       (build-inference 
			extensionality
			(list
			 (build-sequent
			  (sequent-context conclusion)
			  (apply forall (quasi-equality (apply apply-operator f vars)
							(apply apply-operator g vars))
				 vars)))
			conclusion))
		      (else '#f))))))

    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) 'extensionality)))

(define INVERSE-EXTENSIONALITY
  (object
      (lambda (seqs)
	(let* ((conclusion (last seqs))			;assumed to be present
	       (assertion (and conclusion (sequent-assertion conclusion))))
	  (if (or (not conclusion)
		  (not (negated-equation? assertion))
		  (not (higher-sort?
			(expression-sorting
			 (expression-lhs
			  (negation-body assertion))))))
	      '#f
	      (let* ((f (expression-lhs (negation-body assertion)))
		     (g (expression-rhs (negation-body assertion)))
		     (vars (sorts->new-variables 
			    (common-enclosing-sort-list (domain-sorts f) (domain-sorts g))
			    'x
			    (free-variables conclusion))))
		    
		(cond ((predicator? f)
		       (build-inference 
			inverse-extensionality
			(list
			 (build-sequent
			  (sequent-context conclusion)
			  (apply forsome (negation
					  (equality (apply apply-operator f vars)
						    (apply apply-operator g vars)))
				 vars)))
			conclusion))
		      ((function? f)
		       (build-inference 
			inverse-extensionality
			(list
			 (build-sequent
			  (sequent-context conclusion)
			  (apply forsome (negation
					  (quasi-equality (apply apply-operator f vars)
							  (apply apply-operator g vars)))
				 vars)))
			conclusion))
		      (else '#f))))))

    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) 'inverse-extensionality)))


;;;(define (OBSOLETE-IOTA-RULE virtual-path)
;;;  (labels
;;;      ((self
;;;	(object
;;;	    (lambda (sequents)
;;;	      (or (null? (cddr sequents))
;;;		  (imps-error "OBSOLETE-IOTA-RULE: Too many sequents ~S" sequents))
;;;	      (let* ((conc (last sequents))
;;;		     (hyp (and (cdr sequents)
;;;			       (car sequents)))
;;;		     (new-assertion
;;;		      (obsolete-iota-rule-make-assertion
;;;		       (sequent-assertion conc)
;;;		       virtual-path)))
;;;		(cond ((not new-assertion) '#f)
;;;		      ((not hyp)
;;;		       (build-inference
;;;			self
;;;			(list
;;;			 (build-sequent 
;;;			       (sequent-context conc)
;;;			       new-assertion))
;;;			conc))
;;;		      ((sequent-entails-sequent?
;;;			hyp (build-sequent 
;;;			     (sequent-context conc)
;;;			     new-assertion))
;;;		       (build-inference self (list hyp) conc))
;;;		      (else '#f))))
;;;
;;;	  ((rule? self) '#t)
;;;	  ((rule-soundness-predicate self) (always '#t))
;;;	  ((name self) 'iota-elimination-2)
;;;	  ((rule-generator self) obsolete-iota-rule))))
;;;    self))
;;;
;;;(define (OBSOLETE-IOTA-RULE-MAKE-ASSERTION assertion virtual-path)
;;;    (iota-rule-path-check assertion virtual-path)
;;;    (let* ((iota-expr (follow-virtual-path assertion virtual-path))
;;;	   (iota-body (binding-body iota-expr))
;;;	   (iota-var (car (binding-variables iota-expr)))
;;;	   (new-iota-var1 (new-variable 
;;;			   (expression-sorting iota-var)
;;;			   'y%iota 
;;;			   (variables assertion)))
;;;	   (new-iota-var2 (new-variable 
;;;			   (expression-sorting iota-var)
;;;			   'z%iota 
;;;			   (variables assertion)))
;;;	   (new-iota-body1 (apply-substitution
;;;			   (one-component-subst iota-var new-iota-var1)
;;;			   iota-body))
;;;	   (new-iota-body2 (apply-substitution
;;;			   (one-component-subst iota-var new-iota-var2)
;;;			   iota-body))
;;;	   (new-assertion 
;;;	    (substitution-at-virtual-path assertion new-iota-var2 virtual-path))
;;;	   (assertion1 (forsome
;;;			(conjunction
;;;			 new-iota-body1
;;;			 (forall
;;;			  (implication
;;;			   new-iota-body2
;;;			   (equality new-iota-var2 new-iota-var1))
;;;			  new-iota-var2))
;;;			new-iota-var1))
;;;	   (assertion2 (forall
;;;			(implication
;;;			 new-iota-body2
;;;			 new-assertion)
;;;			new-iota-var2)))
;;;      (conjunction assertion1 assertion2)))

(define (IOTA-RULE virtual-path)
  (labels
      ((self
	(object
	    (lambda (sequents)
	      (or (null? (cddr sequents))
		  (imps-error "IOTA-RULE: ~A ~S."
			      "Too many sequents" sequents))
	      (let* ((conc (last sequents))
		     (hyp (and (cdr sequents)
			       (car sequents)))
		     (new-assertion
		      (iota-rule-make-assertion
		       (sequent-assertion conc)
		       virtual-path)))
		(cond ((not new-assertion) '#f)
		      ((not hyp)
		       (build-inference
			self
			(list
			 (build-sequent 
			       (sequent-context conc)
			       new-assertion))
			conc))
		      ((sequent-entails-sequent?
			hyp (build-sequent 
			     (sequent-context conc)
			     new-assertion))
		       (build-inference self (list hyp) conc))
		      (else '#f))))

	  ((rule? self) '#t)
	  ((rule-soundness-predicate self) (always '#t))
	  ((name self) 'iota-elimination-1)
	  ((rule-generator self) iota-rule))))
    self))

(define (IOTA-RULE-MAKE-ASSERTION assertion virtual-path)
    (iota-rule-path-check assertion virtual-path)
    (let* ((iota-expr (follow-virtual-path assertion virtual-path))
	   (iota-body (binding-body iota-expr))
	   (iota-var (car (binding-variables iota-expr)))
	   (new-iota-var1 (new-variable 
			   (expression-sorting iota-var)
			   'y%iota
			   (variables assertion)))
	   (new-iota-var2 (new-variable 
			   (expression-sorting iota-var)
			   'z%iota
			   (variables assertion)))
	   (new-iota-body1 (apply-substitution
			   (one-component-subst iota-var new-iota-var1)
			   iota-body))
	   (new-iota-body2 (apply-substitution
			   (one-component-subst iota-var new-iota-var2)
			   iota-body))
	   (new-assertion (substitution-at-virtual-path assertion new-iota-var1 virtual-path)))
      (forsome
       (conjunction
	new-iota-body1
	(forall
	 (implication
	  new-iota-body2
	  (equality new-iota-var2 new-iota-var1))
	 new-iota-var2)
	new-assertion)
       new-iota-var1)))

(define (IOTA-RULE-PATH-CHECK assertion virtual-path)
  (or (iota-elimination-path-check 
       assertion 
       (car (expand-virtual-path assertion virtual-path)))
      (imps-error "IOTA-RULE-PATH-CHECK: bad path or formula")))

(define (IOTA-ELIMINATION-PATH-CHECK assertion path)
  (if (atomic-formula? assertion)
      (receive (new-host new-path) (host-and-path-after-step assertion path)
                                   ; Step through atomic formula
	(iterate loop ((host new-host) (path new-path))
	  (cond ((and (iota-expression? host)
		      (null? path))
		 '#T)
		((and (application? host)
		      (expression-of-category-ind? host)
		      (not (null? path)))
		 (receive (new-host new-path) (host-and-path-after-step host path)
			  (loop new-host new-path)))
		(else '#f))))
      '#f))

;;;(define (OBSOLETE-INVERSE-IOTA-RULE virtual-path)
;;;  (labels
;;;      ((self
;;;	(object
;;;	    (lambda (sequents)
;;;	      (or (null? (cddr sequents))
;;;		  (imps-error "OBSOLETE-INVERSE-IOTA-RULE: Too many sequents ~S" sequents))
;;;	      (let* ((conc (last sequents))
;;;		     (hyp (and (cdr sequents)
;;;			       (car sequents)))
;;;		     (new-assertion
;;;		      (obsolete-inverse-iota-rule-make-assertion
;;;		       (sequent-assertion conc)
;;;		       virtual-path)))
;;;		(cond ((not new-assertion) '#f)
;;;		      ((not hyp)
;;;		       (build-inference
;;;			self
;;;			(list
;;;			 (build-sequent 
;;;			       (sequent-context conc)
;;;			       new-assertion))
;;;			conc))
;;;		      ((sequent-entails-sequent?
;;;			hyp (build-sequent 
;;;			     (sequent-context conc)
;;;			     new-assertion))
;;;		       (build-inference self (list hyp) conc))
;;;		      (else '#f))))
;;;
;;;	  ((rule? self) '#t)
;;;	  ((rule-soundness-predicate self) (always '#t))
;;;	  ((name self) 'inverse-iota-elimination-2)
;;;	  ((rule-generator self) obsolete-inverse-iota-rule))))
;;;    self))
;;;
;;;(define (OBSOLETE-INVERSE-IOTA-RULE-MAKE-ASSERTION assertion virtual-path)
;;;    (inverse-iota-rule-path-check assertion virtual-path)
;;;    (let* ((iota-expr (follow-virtual-path assertion virtual-path))
;;;	   (iota-body (binding-body iota-expr))
;;;	   (iota-var (car (binding-variables iota-expr)))
;;;	   (new-iota-var1 (new-variable 
;;;			   (expression-sorting iota-var)
;;;			   'y%iota
;;;			   (variables assertion)))
;;;	   (new-iota-var2 (new-variable 
;;;			   (expression-sorting iota-var)
;;;			   'z%iota
;;;			   (variables assertion)))
;;;	   (new-iota-body1 (apply-substitution
;;;			   (one-component-subst iota-var new-iota-var1)
;;;			   iota-body))
;;;	   (new-iota-body2 (apply-substitution
;;;			   (one-component-subst iota-var new-iota-var2)
;;;			   iota-body))
;;;	   (new-assertion 
;;;	    (substitution-at-virtual-path assertion new-iota-var2 virtual-path))
;;;	   (assertion1 (forsome
;;;			(conjunction
;;;			 new-iota-body1
;;;			 (forall
;;;			  (implication
;;;			   new-iota-body2
;;;			   (equality new-iota-var2 new-iota-var1))
;;;			  new-iota-var2))
;;;			new-iota-var1))
;;;	   (assertion2 (forall
;;;			(implication
;;;			 new-iota-body2
;;;			 new-assertion)
;;;			new-iota-var2)))
;;;      (implication assertion1 assertion2)))

(define (INVERSE-IOTA-RULE virtual-path)
  (labels
      ((self
	(object
	    (lambda (sequents)
	      (or (null? (cddr sequents))
		  (imps-error "INVERSE-IOTA-RULE: ~A ~S."
			      "Too many sequents" sequents))
	      (let* ((conc (last sequents))
		     (hyp (and (cdr sequents)
			       (car sequents)))
		     (new-assertion
		      (inverse-iota-rule-make-assertion
		       (sequent-assertion conc)
		       virtual-path)))
		(cond ((not new-assertion) '#f)
		      ((not hyp)
		       (build-inference
			self
			(list
			 (build-sequent 
			       (sequent-context conc)
			       new-assertion))
			conc))
		      ((sequent-entails-sequent?
			hyp (build-sequent 
			     (sequent-context conc)
			     new-assertion))
		       (build-inference self (list hyp) conc))
		      (else '#f))))

	  ((rule? self) '#t)
	  ((rule-soundness-predicate self) (always '#t))
	  ((name self) 'inverse-iota-elimination-1)
	  ((rule-generator self) inverse-iota-rule))))
    self))

(define (INVERSE-IOTA-RULE-MAKE-ASSERTION assertion virtual-path)
    (inverse-iota-rule-path-check assertion virtual-path)
    (let* ((iota-expr (follow-virtual-path assertion virtual-path))
	   (iota-body (binding-body iota-expr))
	   (iota-var (car (binding-variables iota-expr)))
	   (new-iota-var1 (new-variable 
			   (expression-sorting iota-var)
			   'y%iota
			   (variables assertion)))
	   (new-iota-var2 (new-variable 
			   (expression-sorting iota-var)
			   'z%iota
			   (variables assertion)))
	   (new-iota-body1 (apply-substitution
			   (one-component-subst iota-var new-iota-var1)
			   iota-body))
	   (new-iota-body2 (apply-substitution
			   (one-component-subst iota-var new-iota-var2)
			   iota-body))
	   (new-assertion (substitution-at-virtual-path assertion new-iota-var1 virtual-path))
	   (assertion1 (forsome
			(conjunction
			 new-iota-body1
			 (forall
			  (implication
			   new-iota-body2
			   (equality new-iota-var2 new-iota-var1))
			  new-iota-var2))
			new-iota-var1))
	   (assertion2 (forsome
			(conjunction new-iota-body1 new-assertion)
			new-iota-var1)))
      (implication assertion1 assertion2)))

(define (INVERSE-IOTA-RULE-PATH-CHECK assertion virtual-path)
  (or (inverse-iota-elimination-path-check 
       assertion 
       (car (expand-virtual-path assertion virtual-path)))
      (imps-error "INVERSE-IOTA-RULE-PATH-CHECK: bad path or formula")))

(define (INVERSE-IOTA-ELIMINATION-PATH-CHECK assertion path)
  (if (negated-atomic-formula? assertion)
       (receive (new-host new-path) (host-and-path-after-step assertion path)
                                    ; Step through negation and atomic formula
         (receive (new-host new-path) (host-and-path-after-step new-host new-path)
	   (iterate loop ((host new-host) (path new-path))
	     (cond ((and (iota-expression? host)
			 (null? path))
		    '#T)
		   ((and (application? host)
			 (expression-of-category-ind? host)
			 (not (null? path)))
		    (receive (new-host new-path) (host-and-path-after-step host path)
			     (loop new-host new-path)))
		   (else '#f)))))
      '#f))
	       
(define IOTA-RULE-GENERATOR iota-rule)
(define INVERSE-IOTA-RULE-GENERATOR inverse-iota-rule)

(define (CHOICE-PRINCIPLE-RECOGNIZER expr)
  (and (existential? expr)
       (= (length (binding-variables expr)) 1)
       (universal? (binding-body expr))
       (let* ((body (binding-body expr))
	      (existential-var (car (binding-variables expr)))
	      (existential-sort (expression-sorting existential-var))
	      (universal-variables (binding-variables body))
	      (universal-sorts (map expression-sorting universal-variables)))
	 (and (higher-sort? existential-sort)
	      (let ((existential-domains (higher-sort-domains existential-sort)))
		(and (= (length universal-variables) (length existential-domains))
		     (every? sorts-may-overlap? existential-domains universal-sorts)
		     (let ((application (apply
					 apply-operator
					 existential-var
					 universal-variables)))
			 (= (length (virtual-paths-to-occurrences body application -1))
			    (length (virtual-paths-to-occurrences body existential-var -1))))))))))

(define CHOICE-PRINCIPLE
  (object 
      (lambda (sequents)
	(let* ((conclusion (last sequents))
	       (context (sequent-context conclusion))
	       (assertion (sequent-assertion conclusion)))
	  (if (not (choice-principle-recognizer assertion))
	      '#f
	      (let* ((op (car (binding-variables assertion)))
		     (new-var (new-variable
			       (enforce
				sort?
				(expression-range (car (binding-variables assertion))))
			       (concatenate-symbol
				'y_
				(name op))
			       (set-union
				(sequent-free-variables conclusion)
				(sequent-bound-variables conclusion))))
		     (universal (enforce universal? (binding-body assertion)))
		     (matrix (binding-body universal))
		     (uni-vars (binding-variables universal))
		     (term (apply apply-operator op uni-vars))
		     (new-matrix
		      (substitutions-at-paths
		       matrix
		       (list new-var)
		       (list (paths-to-occurrences matrix term -1)))))
		(if (memq? op (expression-free-variables new-matrix))
		    '#f
		    (build-inference
		     choice-principle
		     (list (build-sequent
			    context
			    (apply
			     forall
			     (forsome
			      new-matrix
			      new-var)
			     uni-vars)))
		     conclusion))))))
    ((rule? self) '#t)
    ((rule-soundness-predicate self) (always '#t))
    ((name self) 'choice-principle)))
