;% Copyright (c) 1990-1994 The MITRE Corporation
;% 
;% Authors: W. M. Farmer, J. D. Guttman, F. J. Thayer
;%   
;% The MITRE Corporation (MITRE) provides this software to you without
;% charge to use, copy, modify or enhance for any legitimate purpose
;% provided you reproduce MITRE's copyright notice in any copy or
;% derivative work of this software.
;% 
;% This software is the copyright work of MITRE.  No ownership or other
;% proprietary interest in this software is granted you other than what
;% is granted in this license.
;% 
;% Any modification or enhancement of this software must identify the
;% part of this software that was modified, by whom and when, and must
;% inherit this license including its warranty disclaimers.
;% 
;% MITRE IS PROVIDING THE PRODUCT "AS IS" AND MAKES NO WARRANTY, EXPRESS
;% OR IMPLIED, AS TO THE ACCURACY, CAPABILITY, EFFICIENCY OR FUNCTIONING
;% OF THIS SOFTWARE AND DOCUMENTATION.  IN NO EVENT WILL MITRE BE LIABLE
;% FOR ANY GENERAL, CONSEQUENTIAL, INDIRECT, INCIDENTAL, EXEMPLARY OR
;% SPECIAL DAMAGES, EVEN IF MITRE HAS BEEN ADVISED OF THE POSSIBILITY OF
;% SUCH DAMAGES.
;% 
;% You, at your expense, hereby indemnify and hold harmless MITRE, its
;% Board of Trustees, officers, agents and employees, from any and all
;% liability or damages to third parties, including attorneys' fees,
;% court costs, and other related costs and expenses, arising out of your
;% use of this software irrespective of the cause of said liability.
;% 
;% The export from the United States or the subsequent reexport of this
;% software is subject to compliance with United States export control
;% and munitions control restrictions.  You agree that in the event you
;% seek to export this software or any derivative work thereof, you
;% assume full responsibility for obtaining all necessary export licenses
;% and approvals and for assuring compliance with applicable reexport
;% restrictions.
;% 
;% 
;% COPYRIGHT NOTICE INSERTED: Mon Apr 11 11:42:27 EDT 1994


(herald SPECIAL-INFERENCES)


; A special inference must behave roughly like a schematic inference.  I.e.,
; in case it succeeds, it must return three objects, namely the inference (itself),
; the list of hypotheses, and the conclusion.  Its inputs are
; The intended major premise and conclusion (if they exist).  The non-null
; inputs must appear among the outputs as a hypothesis and the conclusion ---
; however, there may be additional "minor premises" generated by the inference.  For
; instance, the universal instantiation inference would have as as instance:
; 
;   C => (forall (((x y z) sort)) (A x y z))   C => r#   C=> s#   C => t# 
;   ---------------------------------------------------------------------
;   			      C => (A r s t)
;
; In this inference, the number of minor premises  is determined not by the inference but
; by the major formula (forall (((x y z) sort)) (A x y z)).
;
; There need not be a major premise --- in the inference
;
;   C => P   C=> Q   C=> R
;   ---------------------- ,
;      C => (and P Q R)
;
; there is no major premise.
;
; Note that in some cases, e.g. universal instantiation, the major premise
; supplies the pattern, while in others, e.g. existential generalization, the
; conclusion *must* be used.
; 
;   C => (A r r s t)   C => r#   C=> s#   C => t# 
;   ----------------------------------------------
;     C => (forsome (((x y z) sort)) (A r x y z))
;
; Unfortunately, in some cases it appears that *both* the major premise and the
; conclusion must be given in order to construct the instance.  Existential
; elimination and disjunction elimination are examples:
;
;  
;   C => (or Q R S)   C, Q => P   C, R => P   C, S => P 
;   ----------------------------------------------------
;                          C => P
;
; Hence the idea of a special inference generator.  This is a procedure with
; two arguments MAJOR-PREMISE and CONCLUSION.  These are expected to be
; sequents.  The procedure may ignore either of its arguments.  Its
; responsibility is to generate three arguments, namely '#t, if
; successful -- otherwise '#f, the list of hypotheses, and the conclusion.
; All contexts given in its sequent arguments are treated as unadorned.   



(define-structure-type SPECIAL-INFERENCE
  name
  generator
  sideformula-condition
  comment 
	 
  (((print self port)
    (format port "#{IMPS-special-inference~_~A}"
	    (special-inference-name self)))
   ((name self) (special-inference-name self))
   ((->rule sp-inf)
    (object 
	(lambda (sequents)
	  (special-inference->rule sp-inf sequents))
      ((rule? self) '#t)
      ((rule-soundness-predicate self) (always '#t))
      ((rule-generator self) sp-inf)
      ((name self) (name sp-inf))))))
  
(set (special-inference-sideformula-condition (stype-master special-inference-stype))
     (always '#t))

(define (BUILD-SPECIAL-INFERENCE name generator sideformula-condition comment)
  (let ((sr (make-special-inference)))
    (set (special-inference-name sr) name)
    (set (special-inference-generator sr) generator)
    (if sideformula-condition
	(set (special-inference-sideformula-condition sr) sideformula-condition))
    (if comment 
	(set (special-inference-comment sr) comment))
    sr))

(define (APPLY-SPECIAL-INFERENCE inference major-premise conclusion)  
  (receive (ok? hypotheses conclusion)
    ((special-inference-generator inference)
     major-premise conclusion)
    (if (and ok?					;inference succeeded
	     ((special-inference-sideformula-condition inference)	;sideformulas ok?
	      major-premise conclusion))
	(return inference hypotheses conclusion)
	(return '#f nil nil))))

(define (SPECIAL-INFERENCE->RULE inference sequents)
  (let ((major-premise (car sequents))
	(conclusion (last sequents)))
    (receive (rule hyps conc)
      (apply-special-inference inference major-premise conclusion)
      (if rule
	  (build-inference rule hyps conc)
	  nil))))

(define (UNIVERSAL-INSTANTIATION-GENERATOR major-premise conclusion)
  (let ((gen-subst					;match to find terms
	 (lambda (universal instance)			
	   (match instance (binding-body universal))))
	(build-minors					;build seqs asserting 
	 (lambda (subst)				;terms defined
	   (let ((context (sequent-context conclusion))
		 (nbvs (binding-variables (sequent-assertion major-premise))))
	     (map
	      (lambda (var)
		(build-sequent
		 context
		 (defined-in
		   (apply-substitution subst var)
		   (expression-sorting var))))
	      nbvs)))))
    (cond
     ((not (and major-premise conclusion))		;need both items
      (return '#f nil nil))
     ((not (universal?					;check operator
	    (sequent-assertion major-premise)))
      (return '#f nil nil))
     ((not (context-entails-context?			;allow application of 
	    (sequent-context conclusion)		;weakening to premise
	    (sequent-context major-premise)))
      (return '#f nil nil))
     (else
      (let ((subst (gen-subst (sequent-assertion major-premise)
			      (sequent-assertion conclusion))))
	(if (succeed? subst)
	    (return
	     '#t
	     (cons major-premise (build-minors subst))
	     conclusion)
	    (return '#f nil nil)))))))

(define UNIVERSAL-INSTANTIATION
  (build-special-inference
   'universal-instantiation
   universal-instantiation-generator
   nil "Rule to infer body[terms/vars] from (forall vars body)
together with minor premisess asserting that the terms are defined."))

(define (UNIVERSAL-INSTANTIATION-BUILD-MAJOR-PREMISE conclusion vars)
  (let ((context (sequent-context conclusion))
	(assertion (sequent-assertion conclusion)))
    (build-sequent context (apply forall assertion vars))))

(define (UNIVERSAL-INSTANTIATION-BUILD-CONCLUSION major-premise)
  (let ((context (sequent-context major-premise))
	(assertion (sequent-assertion major-premise))
	(avoid-vars (context-free-variables (sequent-context major-premise))))
    (let ((clean-substitution
	   (lambda (vars)
	     (map
	      (lambda (var)
		(make-subst-component
		 var 
		 (new-variable
		  (expression-sorting var)
		  (expression-name var)
		  avoid-vars)))
	      vars))))
      (build-sequent
       context
       (apply-substitution (clean-substitution (binding-variables assertion))
			   (binding-body assertion))))))
  

(define (EXISTENTIAL-GENERALIZATION-GENERATOR major-premise conclusion)
  (let ((gen-subst					;match to find terms
	 (lambda (existential instance)			;(these are the assertions)
	   (match instance (binding-body existential))))
	(build-minors					;build seqs asserting 
	 (lambda (subst)				;terms defined
	   (let ((context (sequent-context conclusion))
		 (nbvs (binding-variables (sequent-assertion conclusion))))
	     (map
	      (lambda (var)
		(build-sequent
		 context
		 (defined-in
		   (apply-substitution subst var)
		   (expression-sorting var))))
	      nbvs)))))
	     
    (cond
     ((not (and major-premise conclusion))		;need both items
      (return '#f nil nil))
     ((not (existential?				;check operator
	    (sequent-assertion conclusion)))
      (return '#f nil nil))
     ((not (context-entails-context?			;allow application of 
	    (sequent-context conclusion)		;weakening to premise
	    (sequent-context major-premise)))
      (return '#f nil nil))
     (else
      (let ((subst (gen-subst (sequent-assertion conclusion)
			      (sequent-assertion major-premise))))
	(if (succeed? subst)
	    (return
	     '#t
	     (cons major-premise (build-minors subst))
	     conclusion)
	    (return '#f nil nil)))))))

(define EXISTENTIAL-GENERALIZATION
  (build-special-inference
   'existential-generalization
   existential-generalization-generator
   nil
   "Rule to infer (FORSOME VARS BODY) from BODY[TERMS/VARS]
together with minor premisess asserting that the terms are defined."))

(define (EXISTENTIAL-ELIMINATION-SIDEFORMULA-CONDITION major-premise conclusion)
  (let ((vars (newly-bound-variables (sequent-assertion major-premise))))
    (null-intersection?
     vars
     (free-variables (sequent-context conclusion)))))

(define (EXISTENTIAL-ELIMINATION-GENERATOR major-premise conclusion)
  (cond
   ((not (and major-premise conclusion))		;need both items
    (return '#f nil nil))
   ((not (existential?					;check operator
	  (sequent-assertion major-premise)))
    (return '#f nil nil))
   ((not (context-entails-context?			;allow application of 
	  (sequent-context conclusion)			;weakening to premise
	  (sequent-context major-premise)))
    (return '#f nil nil))
   (else
    (return '#t
	    (list
	     major-premise
	     (build-sequent
	      (context-add-assumption
	       (sequent-context conclusion)
	       (existential-matrix (sequent-assertion major-premise) nil))
	      (sequent-assertion conclusion)))
	    conclusion))))

(define EXISTENTIAL-ELIMINATION
  (build-special-inference
   'existential-elimination
   existential-elimination-generator
   existential-elimination-sideformula-condition
   "Rule to infer P from  (FORSOME VARS BODY) and the minor premise BODY => P"))

(define (DISJUNCTION-ELIMINATION-GENERATOR major-premise conclusion)
  (cond
   ((not (and major-premise conclusion))		;need both items
    (return '#f nil nil))
   ((not (disjunction?					;check operator
	  (sequent-assertion major-premise)))
    (return '#f nil nil))
   ((not (context-entails-context?			;allow application of 
	  (sequent-context conclusion)			;weakening to premise
	  (sequent-context major-premise)))
    (return '#f nil nil))
   (else
    (let ((disjuncts (expression-components (sequent-assertion major-premise)))
	  (consequent (sequent-assertion conclusion)))
      (return '#t
	      (cons 
	       major-premise
	       (map
		(lambda (d)
		  (build-sequent (context-add-assumption (sequent-context conclusion) d)
				 consequent))
		disjuncts))
	      conclusion)))))

(define DISJUNCTION-ELIMINATION
  (build-special-inference
   'disjunction-elimination
   disjunction-elimination-generator
   nil
   "Rule to infer P from  (OR components) and the minor premises C => P for all components C."))

(define (CUT-RULE-GENERATOR major-premise conclusion)
  (if (and (alpha-equivalent? (sequent-assertion major-premise)
			      (sequent-assertion conclusion))
	   (every?
	    (lambda (c-a)
	      (any?
	       (lambda (p-a)
		 (alpha-equivalent? c-a p-a))
	       (sequent-assumptions major-premise)))
	    (sequent-assumptions conclusion)))
      (let ((discharged-assumptions
	     (list-difference (sequent-assumptions major-premise)
			      (sequent-assumptions conclusion)
			      alpha-equivalent?))
	    (conclusion-context (sequent-context conclusion)))
	(return
	 '#t
	 (cons
	  major-premise
	  (map
	   (lambda (discharged)
	     (build-sequent conclusion-context discharged))
	   discharged-assumptions))
	 conclusion))
      (return '#f nil nil)))

(define CUT-RULE
  (build-special-inference
   'cut
   cut-rule-generator
   nil
   "Rule to infer C=>P from  C,D=>P and minor premises C=>d for all assumptions d in D.")) 
