/******************************************************************************
  lat_enum.c
******************************************************************************/
#include "kant.h"
#include "conv.e"
#include "mat.h"
#include "lattice.h"

#include <math.h>

static	lattice		lat;
static	integer_small 	rank;
static	matrix 		chol;
static	t_handle 		chol_ring;

static	lat_enum_env 	env;
static	lat_elt 	act_coefs;

static	t_logical		lbound_flag;
static	t_logical		coef_bounds_user_flag;
static	t_logical		ref_vector_flag;

/* private functions */
static t_logical lat_enum_kernel_short P_((void));
static t_logical lat_enum_kernel_long P_((void));
static lat_enum_kernel_short_up P_((void));
static t_logical lat_enum_kernel_long_up P_((void));

t_logical
lat_enum WITH_2_ARGS (
	lattice,	llat,
	lat_enum_env,	eenv
)
/******************************************************************************
 
Description:	Looks for a vector v which satiesfies the conditions in env.
 
Calling sequence:
 
	Success = lat_enum(lat,env)

	t_logical		success		= TRUE  : a vector which satiesfies
						  the conditions in env 
						  is returned in 
						  lat_enum_act_coefs(env).
					  FALSE : no vector was found
	lattice		lat		= t_handle of lattice
	lat_enum_env	env		= t_handle of enumeration environment		
 
History:

	92-04-10 JS	renaming of const
	92-03-18 KW	minor changes
	92-03-15 KW	user_coefs_bounds
	92-03-12 KW	lbound
	92-03-10 KW	reference vector 
	92-02-24 KW    	written
 
******************************************************************************/
{
	block_declarations;

	integer_small	i,j;
	lat_elt		tmp_coefs;

	t_real		tempu;
	t_real		temp_length;
	t_real		temp_ubound;
	t_real		cnst;

	t_logical		up_flag;

	lat 		= llat;
	env 		= eenv;

	lat_chol_calc(lat);

	rank		= lat_rank(lat);
	chol		= lat_chol(lat);
	chol_ring 	= lat_chol_ring(lat);

	act_coefs	= lat_enum_act_coefs(env);

	tmp_coefs	= MEM_NH;

/*
**	lbound_flag gets TRUE if lat_enum_lbound(env) is specified else FALSE.
**	In the first case we convert lat_enum_lbound(env) to double.
*/
	if (lat_enum_lbound(env))
	{
		lbound_flag = TRUE;
		lat_enum_double_lbound(env)
		= conv_real_to_double(chol_ring,lat_enum_lbound(env));
	}
	else lbound_flag = FALSE;    

/*
**	coef_bounds_user_flag gets TRUE if any coef_user_flag is set else FALSE.
*/
	for (i=1;(i<=rank)
		 && (!lat_enum_coef_ubound_user_flag_is_set(env,i))
		 && (!lat_enum_coef_lbound_user_flag_is_set(env,i));i++);
	coef_bounds_user_flag = (i > rank) ? FALSE : TRUE;

/*
**	ref_vector_flag gets TRUE if a reference vector is specified else FALSE.
**	In the first case we convert the entries of the reference vector to double.
*/
	ref_vector_flag = !lat_enum_ref_vector_is_zero(env);
	if ((ref_vector_flag) && lat_enum_precision_is_short(env))
	{
		for (i=1;i<=rank;i++)
		{
			lat_enum_double_ref_vector_coef(env,i)
			= conv_real_to_double(chol_ring,lat_enum_ref_vector_coef(env,i));
		}
	}

/*
**	Set enumeration strategy.
*/

	up_flag = lat_enum_strategy_is_up(env) && !ref_vector_flag
		&& !coef_bounds_user_flag && !lbound_flag;

/*
**	Control routines.
*/

	if (lat_enum_request_is_shortest(env))
	{
/*
**		Shortest vector.
*/
		lat_enum_status_set_new(env);

		temp_ubound = (lat_enum_ubound(env))
			    ? real_incref(lat_enum_ubound(env))
			    : MEM_NH;

		if (!ref_vector_flag)
		{
/*
**			Shortest vector && no reference vector.
**
**			Look for a basis vector b[i] (1 <= i <= rank) with
**			(1) lat_enum_lbound <= length(b[i]) <= lat_enum_ubound
**			(2) length(b[i]) = min { length(b[k]) | 1 <= k <= rank }
**			Remark: i=0 is possible.
*/
			for (i=0,j=1;j<=rank;j++) 
			{
				tempu = (lat_gram_is_over_z(lat))
				      ? conv_int_to_real(chol_ring,mat_elt(lat_gram(lat),j,j))
				      : conv_real_to_real(mat_elt(lat_gram(lat),j,j),lat_gram_ring(lat),chol_ring);

				if ( ( !lat_enum_ubound(env)
					|| (real_compare(chol_ring,tempu,lat_enum_ubound(env)) != 1) )
				  && ( !lat_enum_lbound(env)
					|| (real_compare(chol_ring,lat_enum_lbound(env),tempu) != 1) ) )
				{
					if (i == 0) i = j;
					else
					{
						if (ring_elt_compare(lat_gram_ring(lat),
							mat_elt(lat_gram(lat),j,j),mat_elt(lat_gram(lat),i,i)) == -1) i = j;
					}
				}
				real_delete(&tempu);				
			}			    
/*
**			If we find a vector which satisfies (1) and (2)
**			we use it as temporary solution.
*/
			if (i)
			{
				lat_elt_alloc(tmp_coefs,rank);
				for(j=1;j<=rank;j++) if (i != j) lat_elt_coef(tmp_coefs,j) = 0;
				lat_elt_coef(tmp_coefs,i) = -1;
								
				if (lat_enum_ubound(env)) real_delete(&lat_enum_ubound(env));
				lat_enum_ubound(env) = (lat_gram_is_over_z(lat))
						     ? conv_int_to_real(chol_ring,mat_elt(lat_gram(lat),i,i))
						     : conv_real_to_real(mat_elt(lat_gram(lat),i,i),lat_gram_ring(lat),chol_ring);
				temp_length = real_incref(lat_enum_ubound(env));
			}
			else
			{
/*
**				length(b[k]) < lbound (1 <= k <= rank) and ubound = NIL !!
**				What next ???? ??? ?? ?
*/
				if (!lat_enum_ubound(env)) return(FALSE);
			}

			if (lat_enum_precision_is_short(env))
			{
/*
**				Shortest vector && no reference vector && short precision
*/
				if (!tmp_coefs)  
				{
/*
**					We did not find a basis vector b[i] which satisfies (1) and (2).
**					Therefore we try to get a temporary solution using
**					lat_enum_ubound(env) as upper bound for the kernel enumeration routine.
*/
					lat_enum_double_ubound(env)
					= conv_real_to_double(chol_ring,lat_enum_ubound(env));
					if ( (up_flag) 	? !lat_enum_kernel_short_up()
							: !lat_enum_kernel_short())
					{
/*
**						Restore lat_enum_ubound(env)
*/
						real_delete(&lat_enum_ubound(env));
						lat_enum_ubound(env) = temp_ubound;
						return(FALSE);
					}
					lat_elt_alloc(tmp_coefs,rank);
					for (j=1;j<=rank;j++)
					{
						lat_elt_coef(tmp_coefs,j)
						= integer_incref(lat_elt_coef(act_coefs,j));
					}
					temp_length
					= conv_double_to_real(chol_ring,lat_enum_double_act_length(env));
/*
**					We use the length of the vector we found as new
**					upper bound for enumeration.
*/
					lat_enum_double_ubound(env) = lat_enum_double_act_length(env);
				}
				else
				{
					lat_enum_double_ubound(env)
					= conv_real_to_double(chol_ring,lat_enum_ubound(env));
				}

/*
**				If the gram matrix of lat is over Z we will avoid to find vectors of
**				the length we already have realized by the temporary solution.
*/
				if (lat_gram_is_over_z(lat)) lat_enum_double_ubound(env) -= 0.5;

				while((up_flag) ? lat_enum_kernel_short_up()
						: lat_enum_kernel_short())
				{
					if (lat_enum_double_act_length(env) < lat_enum_double_ubound(env))
					{
/*
**						We find a new temporary solution which yields a new
**						upper bound for enumeration.
*/
						for (j=1;j<=rank;j++)
						{
							integer_delref(lat_elt_coef(tmp_coefs,j));
							lat_elt_coef(tmp_coefs,j) = integer_incref(lat_elt_coef(act_coefs,j));
						}
						real_delete(&temp_length);
						temp_length
						= conv_double_to_real(chol_ring,lat_enum_double_act_length(env));

						lat_enum_double_ubound(env)
						= lat_enum_double_act_length(env);
						if (lat_gram_is_over_z(lat)) lat_enum_double_ubound(env) -= 0.5;
					}
				}
			}
			else
			{
/*
**				Shortest vector && no reference vector && long precision
**
**				Corresponding routine for long arithmetic.
*/
				cnst = conv_double_to_real(chol_ring,0.5);

				if (!tmp_coefs)   
				{
					if ( (up_flag) 	? !lat_enum_kernel_long_up()
							: !lat_enum_kernel_long())
					{
						real_delete(&lat_enum_ubound(env));
						lat_enum_ubound(env) = temp_ubound;
						return(FALSE);
					}
					lat_elt_alloc(tmp_coefs,rank);
					for (j=1;j<=rank;j++)
					{
						lat_elt_coef(tmp_coefs,j) = integer_incref(lat_elt_coef(act_coefs,j));
					}
					temp_length = real_incref(lat_enum_act_length(env));
					real_delete(&lat_enum_ubound(env));
					lat_enum_ubound(env) = real_incref(lat_enum_act_length(env));
				}

				if (lat_gram_is_over_z(lat))
				{
					tempu = lat_enum_ubound(env);
					lat_enum_ubound(env) = real_subtract(chol_ring,tempu,cnst);
					real_delete(&tempu);
				}

				while((up_flag) ? lat_enum_kernel_long_up()
						: lat_enum_kernel_long())
				{
					if (real_compare(lat_chol_ring(lat),
						lat_enum_act_length(env), lat_enum_ubound(env)) == -1)
					{
						for (j=1;j<=rank;j++)
						{
							integer_delref(lat_elt_coef(tmp_coefs,j));
							lat_elt_coef(tmp_coefs,j)
							= integer_incref(lat_elt_coef(act_coefs,j));
						}
						real_delete(&temp_length);
						temp_length = real_incref(lat_enum_act_length(env));

						real_delete(&lat_enum_ubound(env));
						lat_enum_ubound(env) = (lat_gram_is_over_z(lat))
								     ? real_subtract(chol_ring,lat_enum_act_length(env),cnst)
								     : real_incref(lat_enum_act_length(env));
					}
				}
				real_delete(&cnst);
			}

/*
**			Shortest vector && no reference vector
*/

/*
**			Restore lat_enum_ubound(env)
*/
			real_delete(&lat_enum_ubound(env));
			lat_enum_ubound(env) = temp_ubound;

/*
**			We can now return our 'temporary' solution.
**			First the length.
*/
			real_delete(&lat_enum_act_length(env));
			lat_enum_act_length(env) = temp_length;
		}
		else
		{
/*
**			Shortest vector && no reference vector
*/

/*
**			We look for the shortest basis vector and
**			calculate its length.
*/
			for (i=1,j=2;j<=rank;j++) 
			{
				if (ring_elt_compare(lat_gram_ring(lat),
					mat_elt(lat_gram(lat),j,j),mat_elt(lat_gram(lat),i,i)) == 1) i = j;
			}

			tempu = (lat_gram_is_over_z(lat))
			      ? conv_int_to_real(chol_ring,mat_elt(lat_gram(lat),i,i))
			      : conv_real_to_real(mat_elt(lat_gram(lat),i,i),lat_gram_ring(lat),chol_ring);

/*
**			Let L denote the length of the shortest basis vector.
**			If L satisfies
**			
**			(3) lat_enum_lbound(env) <= L <= lat_enum_ubound(env)
**
**			we use L as new upper bound for enumeration.
*/
			if (!lbound_flag
				|| (real_compare(chol_ring,tempu,lat_enum_lbound(env)) != 1))
			{
				if (lat_enum_ubound(env))
				{
					if (real_compare(chol_ring,tempu,lat_enum_ubound(env)) != 1)
					{
						real_delete(&lat_enum_ubound(env));
						lat_enum_ubound(env) = real_incref(tempu);
					}
				}
				else
				{
					lat_enum_ubound(env) = real_incref(tempu);
				}
			}
			real_delete(&tempu);

			if (!lat_enum_ubound(env))
			{
/*
**				length(b[i]) > lat_enum_ubound(env) (1 <= i <= rank)
**				and lat_enum_ubound = NIL !!!!
**				What next ? ?? ??? ???? ?????
*/
				return(FALSE);
			}
			if (lat_enum_precision_is_short(env))
			{
/*
**				Shortest vector .. reference vector .. short precision
**
**
**				We first need a temporary solution.
*/
				lat_enum_double_ubound(env)
				= conv_real_to_double(chol_ring,lat_enum_ubound(env));
				if (lat_enum_kernel_short())
				{
					lat_elt_alloc(tmp_coefs,rank);
					for (j=1;j<=rank;j++)
					{
						lat_elt_coef(tmp_coefs,j) = integer_incref(lat_elt_coef(act_coefs,j));
					}
				}
				else
				{
					real_delete(&lat_enum_ubound(env));
					lat_enum_ubound(env) = temp_ubound;
					return(FALSE);
				}
				lat_enum_double_ubound(env) = lat_enum_double_act_length(env);

/*
**				Same strategy as above
*/
				while (lat_enum_kernel_short())
				{
					if (lat_enum_double_act_length(env) < lat_enum_double_ubound(env))
					{
						for (j=1;j<=rank;j++)
						{
							integer_delref(lat_elt_coef(tmp_coefs,j));
							lat_elt_coef(tmp_coefs,j) = integer_incref(lat_elt_coef(act_coefs,j));
						}
						lat_enum_double_ubound(env) = lat_enum_double_act_length(env);
					}
				}
				real_delete(&lat_enum_ubound(env));
				lat_enum_ubound(env)
				= conv_double_to_real(chol_ring,lat_enum_double_ubound(env));
			}
			else
			{
/*
**				Shortest vector && reference vector && long precision
*/
				if (lat_enum_kernel_long())
				{
					lat_elt_alloc(tmp_coefs,rank);
					for (j=1;j<=rank;j++)
					{
						lat_elt_coef(tmp_coefs,j) = integer_incref(lat_elt_coef(act_coefs,j));
					}
				}
				else 
				{
					real_delete(&lat_enum_ubound(env));
					lat_enum_ubound(env) = temp_ubound;
					return(FALSE);
				}
				real_delete(&lat_enum_ubound(env));        
				lat_enum_ubound(env) = real_incref(lat_enum_act_length(env));

				while (lat_enum_kernel_long())
				{
					if (real_compare(chol_ring,
						lat_enum_act_length(env),lat_enum_ubound(env)) == -1)
					{
						for (j=1;j<=rank;j++)
						{
							integer_delref(lat_elt_coef(tmp_coefs,j));
							lat_elt_coef(tmp_coefs,j) = integer_incref(lat_elt_coef(act_coefs,j));
						}
						real_delete(&lat_enum_ubound(env));
						lat_enum_ubound(env) = real_incref(lat_enum_act_length(env));
					}
				}					
			}
/*
**			Shortest vector && reference vector
*/
			real_delete(&lat_enum_act_length(env));
			lat_enum_act_length(env) = lat_enum_ubound(env);
			lat_enum_ubound(env) = temp_ubound;
		}
/*
**		Shortest vector
**
**		Return the coefs of our temporary solution.
*/
		for (j=1;j<=lat_rank(lat);j++)
		{
			integer_delref(lat_elt_coef(act_coefs,j));
			lat_elt_coef(act_coefs,j) = integer_incref(lat_elt_coef(tmp_coefs,j));
		}
		lat_elt_delete(lat,&tmp_coefs);

/*
**		Our work is finished.
*/
		return(TRUE);
	}
	else
	{
/*
**		Next vector
*/
		if (lat_enum_status_is_done(env))
		{
/*
**			Enumeration is done.
*/
			return(FALSE);
		}
	
		if (lat_enum_precision_is_short(env))
		{
/*
**			Next vector .. short precision
*/
			lat_enum_double_ubound(env)
			= conv_real_to_double(chol_ring,lat_enum_ubound(env));

			if ( (up_flag) 	? lat_enum_kernel_short_up()
					: lat_enum_kernel_short())
			{
				real_delete(&lat_enum_act_length(env));
				lat_enum_act_length(env)
				= conv_double_to_real(chol_ring,lat_enum_double_act_length(env));
				return(TRUE);
			}
			else
			{
				return(FALSE);
			}
		}
		else
		{
/*
**			Next vector .. long precision
*/
			return((up_flag) ? lat_enum_kernel_long_up()
					 : lat_enum_kernel_long());
		}			
	}
}



/*
**
**
**	internal * internal * internal * internal * internal * internal
**
**
*/



/*
**	The following routine is exactly described in Pohst/Zassenhaus, p.190.
*/

static t_logical        
lat_enum_kernel_short WITH_NO_ARGS()
{
	block_declarations;

	integer_small	i,j;
	double		tempx,tempy,temp1,temp2;

	if (!lat_enum_status_is_new(env))
	{
		i = 1;		
		goto s3;
	}

	lat_enum_counter(env) = 0;
	i = rank;
	lat_enum_double_tmp_bound(env,i) = lat_enum_double_part_length(env, i) = 0.;

       	for (j=1;j<=rank;j++)
	{		
		integer_delref(lat_elt_coef(act_coefs,j));
		integer_delref(lat_enum_coef_ubound(env,j));
	}

s2:	if (lat_enum_double_ubound(env)<lat_enum_double_tmp_bound(env,i)) goto s4;

	tempx = lat_enum_double_part_length(env,i);
	if (ref_vector_flag) tempx += lat_enum_double_ref_vector_coef(env,i);

	tempy = sqrt((lat_enum_double_ubound(env)
			- lat_enum_double_tmp_bound(env,i)) / dmat_elt(lat_double_chol(lat),i,i));

	temp1 = (int)(ceil (-tempy-tempx));
	temp2 = (int)(floor( tempy-tempx));

/*
**	Test coeffient bounds provided by user
*/

	if (lat_enum_coef_lbound_user_flag_is_set(env,i)
	 && (lat_enum_coef_lbound_user(env,i) > temp1))
		temp1 = lat_enum_coef_lbound_user(env,i);

	if (lat_enum_coef_ubound_user_flag_is_set(env,i)
	 && (lat_enum_coef_ubound_user(env,i) < temp2))
		temp2 = lat_enum_coef_ubound_user(env,i);

	lat_elt_coef(act_coefs,i)   = temp1-1;
	lat_enum_coef_ubound(env,i) = temp2;

/*
**	Increase actual coef
*/

s3:	lat_elt_coef(act_coefs,i)++;

	if (lat_elt_coef(act_coefs,i) <= lat_enum_coef_ubound(env,i)) goto s5;

s4:	i++;

	if (i<=rank) goto s3;

	lat_enum_status_set_done(env);
	return(FALSE);

s5:	if (i == 1)
	{
		if (!ref_vector_flag)
		{
			for (j=1;(j<=rank) && (!lat_elt_coef(act_coefs,j));j++);
/*
**			Avoid to return (0,...,0)
*/

			if (j > rank)
			{
				lat_enum_status_set_done(env);
				return(FALSE);
			}
		}

/*
**		Calculate the length
*/

		tempx
		= (double)(lat_elt_coef(act_coefs,1)) + lat_enum_double_part_length(env,1);
		if (ref_vector_flag) tempx += lat_enum_double_ref_vector_coef(env,1);
		lat_enum_double_act_length(env) = lat_enum_double_tmp_bound(env,1)
							+  dmat_elt(lat_double_chol(lat),1,1) * tempx * tempx;

		if (lbound_flag
		 && (lat_enum_double_lbound(env) > lat_enum_double_act_length(env))) goto s3;

		lat_enum_status_set_busy(env);
		lat_enum_counter(env)++;

		return(TRUE);
	}			
	else
	{
		i--;

		lat_enum_double_part_length(env,i) = 0.;

		for(j=i+1;j<=rank;j++)
		{
			tempx = lat_elt_coef(act_coefs,j);
			if (ref_vector_flag) tempx += lat_enum_double_ref_vector_coef(env,j);			
			lat_enum_double_part_length(env,i)
			+= dmat_elt(lat_double_chol(lat),i,j) * tempx;
		}

		tempx = lat_elt_coef(act_coefs,i+1) + lat_enum_double_part_length(env,i+1);
		if (ref_vector_flag) tempx += lat_enum_double_ref_vector_coef(env,i+1);
		lat_enum_double_tmp_bound(env,i) = lat_enum_double_tmp_bound(env,i+1)
							+ dmat_elt(lat_double_chol(lat),i+1,i+1) * tempx * tempx;

		goto s2;
	}
}


static t_logical
lat_enum_kernel_long WITH_NO_ARGS()
{
	block_declarations;

	integer_small	i,j;
	integer_big	tempa,tempb;
	t_real		tempt,tempu,tempv,tempw,tempx,tempy,tempz;

	if (!lat_enum_status_is_new(env))
	{
		i = 1;		
		goto s3;
	}

	lat_enum_counter(env) = 0;

	i = rank;

	real_delete(&lat_enum_tmp_bound(env,i));
	lat_enum_tmp_bound(env,i) = ring_zero(chol_ring);

	real_delete(&lat_enum_part_length(env,i));
	lat_enum_part_length(env,i) = ring_zero(chol_ring);

s2:	if (real_compare(chol_ring,
			lat_enum_ubound(env),lat_enum_tmp_bound(env,i)) == -1) goto s4;

	integer_delref(lat_elt_coef(act_coefs,i));
	integer_delref(lat_enum_coef_ubound(env,i));

	tempt = (ref_vector_flag)
	      ? real_add(chol_ring,
				lat_enum_part_length(env,i),lat_enum_ref_vector_coef(env,i))
	      : real_incref(lat_enum_part_length(env,i));
	tempu = real_subtract(chol_ring,
				lat_enum_ubound(env),lat_enum_tmp_bound(env,i));
	tempv = real_divide(chol_ring,tempu,mat_elt(chol,i,i));
	tempw = real_sqrt(chol_ring,tempv);
	tempx = real_subtract(chol_ring,tempw,tempt);
	tempy = real_negate(chol_ring,tempw);
	tempz = real_subtract(chol_ring,tempy,tempt);

	tempa = conv_real_to_int_ceil(chol_ring,tempz);
	tempb = conv_real_to_int_floor(chol_ring,tempx);

/*
**	Test coeffient bounds provided by user
*/

	if (lat_enum_coef_lbound_user_flag_is_set(env,i)
	 && (integer_compare(lat_enum_coef_lbound_user(env,i),tempa) == 1))
	{
		integer_delref(tempa);
		tempa = integer_incref(lat_enum_coef_lbound_user(env,i));
	}

	if (lat_enum_coef_ubound_user_flag_is_set(env,i)
	 && (integer_compare(lat_enum_coef_ubound_user(env,i),tempb) == -1))
	{
		integer_delref(tempb);
		tempb = integer_incref(lat_enum_coef_ubound_user(env,i));
	}

	lat_elt_coef(act_coefs,i)   = integer_subtract(tempa,1);
	lat_enum_coef_ubound(env,i) = integer_incref(tempb);

	real_delete(&tempt);
	real_delete(&tempu);
	real_delete(&tempv);
	real_delete(&tempw);
	real_delete(&tempx);
	real_delete(&tempy);
       	real_delete(&tempz);

	integer_delref(tempa);
	integer_delref(tempb);

s3:	tempa = lat_elt_coef(act_coefs,i);
	lat_elt_coef(act_coefs,i) = integer_add(tempa,1);
	integer_delref(tempa);

	if (integer_compare(lat_elt_coef(act_coefs,i),
				lat_enum_coef_ubound(env,i)) != 1)
	{		
		real_delete(&lat_enum_act_real_coef(env,i));
		lat_enum_act_real_coef(env,i)
		= conv_int_to_real(chol_ring,lat_elt_coef(act_coefs,i));
		goto s5;		
	}

s4:	i++;

	if (i<=rank) goto s3;

	lat_enum_status_set_done(env);
	return(FALSE);

s5:	if (i == 1)
	{
		if (!ref_vector_flag)
		{
			for (j=1;(j<=rank) && (!lat_elt_coef(act_coefs,j));j++);
/*
**			Avoid to return (0,...,0)
*/
			if (j > rank)
			{
				lat_enum_status_set_done(env);
				return(FALSE);
			}
		}

/*
**		Calculate the length
*/

		real_delete(&lat_enum_act_length(env));

		tempt = (ref_vector_flag)
			? real_add(chol_ring,
					lat_enum_part_length(env,1),lat_enum_ref_vector_coef(env,1))
			: real_incref(lat_enum_part_length(env,1));
		tempu = real_add(chol_ring,lat_enum_act_real_coef(env,1),tempt);
		tempv = real_mult(chol_ring,tempu,tempu);
		tempw = real_mult(chol_ring,mat_elt(chol,1,1),tempv);

		lat_enum_act_length(env)
		= real_add(chol_ring,tempw,lat_enum_tmp_bound(env,1));

		real_delete(&tempt);
		real_delete(&tempu);
		real_delete(&tempv);
		real_delete(&tempw);

		if (lbound_flag
		 && (real_compare(chol_ring,
				lat_enum_lbound(env),lat_enum_act_length(env)) == 1)) goto s3;

		lat_enum_status_set_busy(env);
		lat_enum_counter(env)++;

		return(TRUE);
	}
	else
       	{
		i--;

		real_delete(&lat_enum_part_length(env,i));
		tempu = ring_zero(chol_ring);

		for (j=i+1;j<=rank;j++)
		{
			tempt = (ref_vector_flag)
				? real_add(chol_ring,
					lat_enum_act_real_coef(env,j),lat_enum_ref_vector_coef(env,j))
				: real_incref(lat_enum_act_real_coef(env,j));
			tempv = tempu;
			tempw = real_mult(chol_ring,mat_elt(chol,i,j),tempt);
			tempu = real_add(chol_ring,tempu,tempw);

			real_delete(&tempt);
			real_delete(&tempv);
			real_delete(&tempw);
		}
		lat_enum_part_length(env,i) = tempu;
		
		real_delete(&lat_enum_tmp_bound(env,i));

		tempt = (ref_vector_flag)
			? real_add(chol_ring,
					lat_enum_part_length(env,i+1),lat_enum_ref_vector_coef(env,i+1))
			: real_incref(lat_enum_part_length(env,i+1));
		tempu = real_add(chol_ring,lat_enum_act_real_coef(env,i+1),tempt);
		tempv = real_mult(chol_ring,tempu,tempu);
		tempw = real_mult(chol_ring,tempv,mat_elt(chol,i+1,i+1));
		lat_enum_tmp_bound(env,i)
		= real_add(chol_ring,lat_enum_tmp_bound(env,i+1),tempw);

		real_delete(&tempt);
		real_delete(&tempu);
		real_delete(&tempv);
		real_delete(&tempw);
	}
	goto s2;
}


static
lat_enum_kernel_short_up WITH_NO_ARGS()
{
	block_declarations;

	integer_small	i,j,k;
	double		tempx,tempy;

	if (!lat_enum_status_is_new(env))
	{
		i = 1;		
		goto s3;
	}

	lat_enum_counter(env) = 0;
	i = rank;
	lat_enum_double_tmp_bound(env,i) = lat_enum_double_part_length(env, i) = 0.;

       	for (j=1;j<=rank;j++)
	{		
		integer_delref(lat_elt_coef(act_coefs,j));
		integer_delref(lat_enum_coef_lbound(env,j));
		integer_delref(lat_enum_coef_ubound(env,j));
	}

s2:	if (lat_enum_double_ubound(env)<lat_enum_double_tmp_bound(env,i)) goto s4;

	tempx = lat_enum_double_part_length(env,i);
	tempy = sqrt((lat_enum_double_ubound(env)
			- lat_enum_double_tmp_bound(env,i)) / dmat_elt(lat_double_chol(lat),i,i));

	lat_enum_coef_lbound(env,i) = (int)(ceil (-tempy-tempx));
	lat_enum_coef_ubound(env,i) = (int)(floor( tempy-tempx));

	if (lat_enum_coef_lbound(env,i) > lat_enum_coef_ubound(env,i)) goto s4;

	if (lat_enum_coef_ubound(env,i) < 0)
	{
		lat_elt_coef(act_coefs,i) = lat_enum_coef_ubound(env,i);
	}
	else
	{
		if (lat_enum_coef_lbound(env,i) > 0)
		{
			lat_elt_coef(act_coefs,i) = lat_enum_coef_lbound(env,i);
		}
		else
		{
			lat_elt_coef(act_coefs,i) = 0;
			for (j=i+1,k=0;(j<=rank) && (!lat_elt_coef(act_coefs,j));j++) k++;
			if (i+k == rank)
			{
				lat_enum_coef_ubound(env,i) = 0;
				if (i == 1) goto s3;
			}
		}
	}
	goto s5;

/*
**	Modify actual coef (0,1,-1,2,-2,.....)
*/

s3:	if (lat_elt_coef(act_coefs,i) > 0)
	{
		lat_elt_coef(act_coefs,i) = -lat_elt_coef(act_coefs,i);
		if (lat_elt_coef(act_coefs,i) >= lat_enum_coef_lbound(env,i)) goto s5;
		lat_elt_coef(act_coefs,i) = -lat_elt_coef(act_coefs,i) + 1;	
		if (lat_elt_coef(act_coefs,i) <= lat_enum_coef_ubound(env,i)) goto s5;
	}
	else
	{
		lat_elt_coef(act_coefs,i) = -lat_elt_coef(act_coefs,i) + 1;	
		if (lat_elt_coef(act_coefs,i) <= lat_enum_coef_ubound(env,i)) goto s5;
		lat_elt_coef(act_coefs,i) = -lat_elt_coef(act_coefs,i);
		if (lat_elt_coef(act_coefs,i) >= lat_enum_coef_lbound(env,i)) goto s5;
	}

s4:	i++;

	if (i<=rank) goto s3;

	lat_enum_status_set_done(env);
	return(FALSE);

s5:	if (i == 1)
	{
/*
**		Calculate the length
*/

		tempx
		= (double)(lat_elt_coef(act_coefs,1)) + lat_enum_double_part_length(env,1);
		lat_enum_double_act_length(env) = lat_enum_double_tmp_bound(env,1)
							+  dmat_elt(lat_double_chol(lat),1,1) * tempx * tempx;

		lat_enum_status_set_busy(env);
		lat_enum_counter(env)++;

		return(TRUE);
	}			
	else
	{
		i--;

		lat_enum_double_part_length(env,i) = 0.;

		for(j=i+1;j<=rank;j++)
		{
			tempx = lat_elt_coef(act_coefs,j);
			lat_enum_double_part_length(env,i)
			+= dmat_elt(lat_double_chol(lat),i,j) * tempx;
		}

		tempx = lat_elt_coef(act_coefs,i+1) + lat_enum_double_part_length(env,i+1);
		lat_enum_double_tmp_bound(env,i) = lat_enum_double_tmp_bound(env,i+1)
							+ dmat_elt(lat_double_chol(lat),i+1,i+1) * tempx * tempx;

		goto s2;
	}
}

static t_logical
lat_enum_kernel_long_up WITH_NO_ARGS()
{
	block_declarations;

	integer_small	i,j,k;
	integer_big	tempa,tempb;
	t_real		tempu,tempv,tempw,tempx,tempy,tempz;

	if (!lat_enum_status_is_new(env))
	{
		i = 1;		
		goto s3;
	}

	lat_enum_counter(env) = 0;

	i = rank;

	real_delete(&lat_enum_tmp_bound(env,i));
	lat_enum_tmp_bound(env,i) = ring_zero(chol_ring);

	real_delete(&lat_enum_part_length(env,i));
	lat_enum_part_length(env,i) = ring_zero(chol_ring);

s2:	if (real_compare(chol_ring,
			lat_enum_ubound(env),lat_enum_tmp_bound(env,i)) == -1) goto s4;

	integer_delref(lat_elt_coef(act_coefs,i));
	integer_delref(lat_enum_coef_ubound(env,i));
	integer_delref(lat_enum_coef_lbound(env,i));

	tempu = real_subtract(chol_ring,
				lat_enum_ubound(env),lat_enum_tmp_bound(env,i));
	tempv = real_divide(chol_ring,tempu,mat_elt(chol,i,i));
	tempw = real_sqrt(chol_ring,tempv);
	tempx = real_subtract(chol_ring,tempw,lat_enum_part_length(env,i));
	tempy = real_negate(chol_ring,tempw);
	tempz = real_subtract(chol_ring,tempy,lat_enum_part_length(env,i));

	lat_enum_coef_lbound(env,i) = conv_real_to_int_ceil(chol_ring,tempz);
	lat_enum_coef_ubound(env,i) = conv_real_to_int_floor(chol_ring,tempx);

	real_delete(&tempu);
	real_delete(&tempv);
	real_delete(&tempw);
	real_delete(&tempx);
	real_delete(&tempy);
       	real_delete(&tempz);

	if (integer_compare(lat_enum_coef_lbound(env,i),
				lat_enum_coef_ubound(env,i)) == 1) goto s4;

	if (integer_compare(lat_enum_coef_ubound(env,i),0) == -1)
	{
		lat_elt_coef(act_coefs,i) = integer_incref(lat_enum_coef_ubound(env,i));
	}
	else
	{
		if (integer_compare(lat_enum_coef_lbound(env,i),0) == 1)
		{
			lat_elt_coef(act_coefs,i) = integer_incref(lat_enum_coef_lbound(env,i));
		}
		else
		{
			lat_elt_coef(act_coefs,i) = 0;
			for (j=i+1,k=0;(j<=rank) && (!lat_elt_coef(act_coefs,j));j++) k++;
			if (i+k == rank)
			{
				integer_delref(lat_enum_coef_ubound(env,i));
				lat_enum_coef_ubound(env,i) = 0;
				if (i == 1) goto s3;
			}
		}
	}
	goto s5;

/*
**	Modify actual coef (0,1,-1,2,-2,.....)
*/

s3:	if (integer_compare(lat_elt_coef(act_coefs,i),0) == 1)
	{
		tempa = lat_elt_coef(act_coefs,i);
		lat_elt_coef(act_coefs,i) = integer_negate(tempa);
		integer_delref(tempa);
		if (integer_compare(lat_elt_coef(act_coefs,i),
				lat_enum_coef_lbound(env,i)) != -1) goto s5;

		tempa = lat_elt_coef(act_coefs,i);
		tempb = integer_negate(tempa);
		lat_elt_coef(act_coefs,i) = integer_add(tempb,1);
		integer_delref(tempa);
		integer_delref(tempb);
		if (integer_compare(lat_elt_coef(act_coefs,i),
				lat_enum_coef_ubound(env,i)) != 1) goto s5;
	}
	else
	{
		tempa = lat_elt_coef(act_coefs,i);
		tempb = integer_negate(tempa);
		lat_elt_coef(act_coefs,i) = integer_add(tempb,1);
		integer_delref(tempa);
		integer_delref(tempb);
		if (integer_compare(lat_elt_coef(act_coefs,i),
				lat_enum_coef_ubound(env,i)) != 1) goto s5;

		tempa = lat_elt_coef(act_coefs,i);
		lat_elt_coef(act_coefs,i) = integer_negate(tempa);
		integer_delref(tempa);
		if (integer_compare(lat_elt_coef(act_coefs,i),
				lat_enum_coef_lbound(env,i)) != -1) goto s5;
	}

s4:	i++;

	if (i<=rank) goto s3;

	lat_enum_status_set_done(env);
	return(FALSE);

s5:	real_delete(&lat_enum_act_real_coef(env,i));
	lat_enum_act_real_coef(env,i)
		= conv_int_to_real(chol_ring,lat_elt_coef(act_coefs,i));

	if (i == 1)
	{
/*
**		Calculate the length
*/

		real_delete(&lat_enum_act_length(env));

		tempu = real_add(chol_ring,
				lat_enum_act_real_coef(env,1),lat_enum_part_length(env,1));
		tempv = real_mult(chol_ring,tempu,tempu);
		tempw = real_mult(chol_ring,mat_elt(chol,1,1),tempv);

		lat_enum_act_length(env)
		= real_add(chol_ring,tempw,lat_enum_tmp_bound(env,1));

		real_delete(&tempu);
		real_delete(&tempv);
		real_delete(&tempw);

		lat_enum_status_set_busy(env);
		lat_enum_counter(env)++;

		return(TRUE);
	}
	else
       	{
		i--;

		real_delete(&lat_enum_part_length(env,i));
		tempu = ring_zero(chol_ring);

		for (j=i+1;j<=rank;j++)
		{
			tempv = tempu;
			tempw = real_mult(chol_ring,
					mat_elt(chol,i,j),lat_enum_act_real_coef(env,j));
			tempu = real_add(chol_ring,tempu,tempw);

			real_delete(&tempv);
			real_delete(&tempw);
		}
		lat_enum_part_length(env,i) = tempu;
		
		real_delete(&lat_enum_tmp_bound(env,i));

		tempu = real_add(chol_ring,
				lat_enum_act_real_coef(env,i+1),lat_enum_part_length(env,i+1));
		tempv = real_mult(chol_ring,tempu,tempu);
		tempw = real_mult(chol_ring,tempv,mat_elt(chol,i+1,i+1));
		lat_enum_tmp_bound(env,i)
		= real_add(chol_ring,lat_enum_tmp_bound(env,i+1),tempw);

		real_delete(&tempu);
		real_delete(&tempv);
		real_delete(&tempw);
	}
	goto s2;
}

