/******************************************************************************
  order_basis_check.c
******************************************************************************/
 
#include "kant.h" 
 
t_logical
order_basis_check WITH_3_ARGS(
        order,                  ord,
	dyn_arr_handle,		elts,
        order *,                ordnew
)
/*******************************************************************************
                   
Description:	
                                    
        Checks whether the algebraic numbers submitted in elts are a 
        basis of an order. If so, the new order is returned in ordnew.
 
        This function checks whether the elements elts are Z-linearly
        independent and whether their Z-module is multiplicatively closed.
        
        At the moment, ord must be given over Z.
        The elements submitted must be given via their coefficients.
  
Calling sequence:  

	if (order_basis_check(ord, elts, &ord1)) ...

	ord	=	t_handle of order
        elts    = 	dyn. array with algebraic numbers of ord
        ord1    =       new order
 
  
History:
 
	92-09-07 JS     first version
 
*******************************************************************************/
{
	block_declarations;
 
        integer_big     den, cden, x1, x2, det, det2;
        integer_small   n, i, j, ncols, nc;
        order           Z;
        vector          vec, vecd;
        matrix          trans, mat, mat2, mat3, hnf, matlarge;
        anf_elt         omega, a1, a2, a3, a4, a5;
        t_logical         answer;
 
        order_must_be_over_z(ord);
 
        Z = order_coef_order(ord);
        n = order_rel_degree(ord);
 
        if (dyn_arr_curr_length(elts) != n) return 0;
 
/*
   first simplification and common denominator
*/
 
        vec  = vec_new(n);
        vecd = vec_new(n);
        cden = 1;
 
        for (i=1; i<=n; i++)
        {
                omega = anf_elt_simplify(ord, dyn_arr_element(elts, i-1)); 
                den   = anf_elt_is_integer(omega) ? 1 : anf_elt_den(omega);
                x1    = cden;
                cden  = integer_lcm(den, x1);
                vec_entry(vec, i) = omega;
 
                integer_delref(x1);
        }       
         
/*
    The columns of trans contain all elts, multiplied by the
    common denominator cden
*/
        trans = mat_new(n, n);
        for (i=1; i<=n; i++)
        {               
                a1 = vec_entry(vec, i);
                                    
                if (anf_elt_is_integer(a1))
                {
                        omega = integer_mult(cden, a1);
                        vec_entry(vecd, i) = integer_incref(cden);
                }
                else
                {
	                x1    = integer_div(cden, anf_elt_den(a1));
                        vec_entry(vecd, i) = x1;
	                anf_elt_alloc(omega, n);
        	        for (j=1; j<=n; j++)
                        {
                	        anf_elt_coef(omega, j) = 
				integer_mult(anf_elt_coef(a1, j), x1);
                        }
                }
                anf_elt_to_mat_order_col(ord, omega, trans, i);
                anf_elt_delete(ord, &omega);
        }
        hnf = mat_ring_hnf_col(Z, trans);
        det = 1;
        for (i=1; i<=n; i++)
        {
                x2  = det;
                det = integer_mult(x2, mat_elt(hnf, i, i));
                integer_delref(x2);
        }
        mat_delref(Z, &hnf);

	if (anf_print_level > 2) 
		cay_print("order_basis_check: Determinant is %d.\n",  det); 
 
        if (det == 0) 
        {
                mat_delref(Z, &trans);
	        vec_delete(ord, &vec);
	        vec_delete(ord, &vecd);
                integer_delref(cden);
                return 0;
        }
                          
        ncols = (n*(n+1))/2 + n;
        mat2  = mat_new(n, ncols);
        mat_ring_create_zero_sub(Z, &mat2);
        matlarge  = mat_ring_insert(Z, mat2, trans, 1, 1);
        mat_delref(Z, &mat2);
                
/*  all possible products */             

        nc = n;
        for (i=1; i<=n; i++)
        {
                omega  = vec_entry(vec, i);
                for (j=i; j<=n; j++)
                {
	                a1 = anf_elt_mult(ord, omega, vec_entry(vec, j));
                        a2 = anf_elt_mult(ord, a1, cden);
                        a5 = anf_elt_simplify(ord, a2);
                        if (!anf_elt_is_integer(a5) && (anf_elt_den(a5) != 1))
                        {     
			        if (anf_print_level > 2) 
                                printf("Denominator occured when multiplying elt %d by %d.\n", i, j);

	                        anf_elt_delete(ord, &a1);
	                        anf_elt_delete(ord, &a2);
	                        anf_elt_delete(ord, &a5);
			        vec_delete(ord, &vec);
			        vec_delete(ord, &vecd);
			        mat_delref(Z, &trans);
			        mat_delref(Z, &matlarge);
			        integer_delref(cden);
			        integer_delref(det);
                                return FALSE;
                        }

                        anf_elt_to_mat_order_col(ord, a5, matlarge, ++nc);  
                }
        }
 
/*  now get dependencies */
 
        mat2 = mat_ring_hnf_col_upper(Z, matlarge);
        mat3 = mat_ring_submat(Z, mat2, 1, 1, n, n);
        det2 = mat_ring_det(Z, mat3);
        if (anf_print_level > 2) 
			cay_print("\n ... and new determinant: %d\n",  det2); 
 
        if (integer_compare(det2, det) == 0)
        {
                *ordnew = order_trans(ord, trans, cden);
                answer = 1;
        }
        else
        {
                answer = 0;
        }

        vec_delete(ord, &vec);
        vec_delete(ord, &vecd);
        mat_delref(Z, &trans);
        mat_delref(Z, &mat2);
        mat_delref(Z, &mat3);
        mat_delref(Z, &matlarge);
        integer_delref(cden);
        integer_delref(det);
        integer_delref(det2);
 
        return answer;
}
