/******************************************************************************
  order_norm_equation_im2.c
******************************************************************************/
 
#include "kant.h"        

t_handle
order_norm_equation_im2 WITH_2_ARGS(
	order,		ord,
	t_int,		k
)
/*******************************************************************************
 
Description:
                                     
	We search for all algebraic numbers alpha in ord with
	| N(alpha) | = | k |. The function returns a dynamic array
	where we stored all non-associated results.

	ord must be defined over a (totally) complex quadratic
	number field!!!
  
Calling sequence:
 
        order   	ord:    order
	t_int		k:	norm
 
History:

	93-01-26 CO     first version

*******************************************************************************/
{       
	block_declarations;

        t_handle            R,Z;
        lattice             lat,lll_lat;
        lat_enum_env        lat_env,lll_lat_env; 
        lat_elt             lat_vec;                         
        matrix              trans,inv_trans;
        anf_elt             alpha;
	t_real		bound;
	integer_big	hlp1, hlp2;
	t_handle	results;
	t_int		new, len;

    order_reals_assure(ord);
    R = order_reals(ord);

    results = dyn_arr_alloc( 0 );

/*    In this special case ( complex quadratic number field )   */
/*    we know:                                                  */
/*      2* | N(alpha) |  =  2* | alpha |^2  =  T2( alpha )      */
/*    and because of this we use lat_enum with			*/
/*      lower bound  =  2* | k |  =  upper bound.		*/

    order_lat (ord, &lat, &lat_env);

    lll_lat = lat_lll_reduce(lat,&trans,&inv_trans);
    lll_lat_env = lat_enum_create(lll_lat);

    lat_enum_request_set_next(lll_lat_env);
    lat_enum_strategy_set_down(lll_lat_env);          
    
    hlp1 = integer_abs( k );
    hlp2 = integer_mult( hlp1, 2 );
    bound = conv_int_to_real( R, hlp2 );

    lat_enum_lbound(lll_lat_env) = real_incref( bound );
    lat_enum_ubound(lll_lat_env) = real_incref( bound );

    lat_enum_status_set_new(lll_lat_env);
 
    while (lat_enum(lll_lat, lll_lat_env))
    {    

        lat_vec  = lat_elt_move (lll_lat,lat_enum_act_coefs(lll_lat_env),trans);
        alpha    = lat_elt_to_anf_elt (lat, lat_vec, ord);

/*    If ord is maximal and its discriminant is less than -4	*/
/*    lat_enum will not return us any algebraic number being	*/
/*    associated to another number we got before. In all other	*/
/*    cases we have to care for associated elements.		*/

	if( order_is_maximal( ord ))
	{
	    order_disc_assure( ord );

	    if( integer_compare( order_disc( ord ), -4 ) >= 0 )
		new = _order_norm_equation_im2_comp_ass( ord, alpha, results );
	    else
		new = 1;
	}
	else
	    new = _order_norm_equation_im2_comp_ass( ord, alpha, results );

	if( new )
	{
	    len = dyn_arr_curr_length( results );
	    dyn_arr_assure_space( results, len + 1, 1 );
	    dyn_arr_element( results, len ) = anf_elt_incref( alpha );
	    dyn_arr_curr_length( results )++;
	}

        lat_elt_delete (lll_lat,&lat_vec);
        anf_elt_delete(ord, &alpha);
    }

/* kill	*/

    real_delete( &bound );
    integer_delref( hlp1 );
    integer_delref( hlp2 );

    Z = m_z_str_incref(structure_z);
    mat_delref(Z,&trans);
    mat_delref(Z,&inv_trans);
    ring_delete(&Z);	

    lat_enum_delete(lll_lat, &lll_lat_env);
    lat_delete(&lll_lat);
    lat_enum_delete(lat, &lat_env);
    lat_delete(&lat); 

    return( results );
}


t_int
_order_norm_equation_im2_comp_ass WITH_3_ARGS(
	order,		ord,
	anf_elt,	alpha,
	t_handle,	results
)
/*******************************************************************************
 
Description:
                                     
	We compare alpha to all algebraic numbers being stored in the
	dynamic array 'results'. If alpha is associated to one of them,
	we return  0, else  1.
  
Calling sequence:
 
        order   	ord:    	order
	anf_elt		alpha:		algebraic number in ord
	t_handle	results:	set of algebraic numbers in ord
 
History:

	93-01-26 CO     first version

*******************************************************************************/
{       
	block_declarations;

	t_int		i, new, len;
	anf_elt		beta, beta1;
	integer_big	normabs, den;

	new = 1;
	len = dyn_arr_curr_length( results );

	for( i = 0; ( i < len ) && ( new ); i++ )
	{
		beta = anf_div( ord, alpha, dyn_arr_element( results, i )); 
		beta1 = anf_elt_simplify( ord, beta );

/*****  perhaps beta1 has norm 1, but it is not in ord !  *****/

		if( integer_compare( anf_elt_den( beta1 ), 1 ) == 0 )
		{
			anf_elt_norm_abs( ord, beta1, &normabs, &den );

			if( integer_compare( den, normabs ) == 0 )
				new = 0;

			integer_delref( normabs );
			integer_delref( den );
		}
	        anf_elt_delete( ord, &beta1 );
	        anf_elt_delete( ord, &beta );
	}
	return( new );
}
