#include "defs.h"
#include "integer.e"
#include "inthdl.e"
#include "intbig.h"

t_int
i_miller_rabin WITH_2_ARGS(
       integer_big,	n,
       t_int,	limit
)
/*
Attempt to prove that n is composite using the Miller-Rabin test,
taking up to limit attempts. The return value is -1 if n is definitely
not prime, and 1 if n is probably prime. 
*/
{
    block_declarations;

    integer_big		n1, d, x, temp;
    t_int	s, count, a, i;

    /*
    The algorithm comes from D.E. Knuth, The Art of Computer Programming,
    Vol II, Seminumerical Algorithms, Section 4.5.4, Algorithm P. 

    John Brownie, August 1989; grossly modified by Allan Steel, 4/2/91.
    */
    

    DEBUG_INTEGER(("+miller_rabin", 2, n, limit));

    if (n == 2)
	return 1;

    n1 = integer_add(n, -1);

    /*
    Find s s.t. n1 = 2^s * d, d odd.
    */

    integer_shift_right_maximal(n1, &d, &s);

    for (count = 0; count < limit; count++)
    {
	/*
	Find (single) base a such that 1 < a < n.
	*/

	if (integer_is_single(n))
	    a = (integer_long_random(ZETA) % (n - 1)) + 1;
	else
	{
	    do
		a = integer_long_random(ZETA) + 1;
	    while (a == BETA);
	}


	x = modint_exp(n, a, d);

	if (x != 1)
	{
	    for (i = 0; i < s; i++)
	    {
		if (integer_compare(x, n1) == 0)
		    goto NEXT;
		if (i < s)
		{
		    temp = x;
		    x = modint_exp(n, x, 2);
		    integer_delref(temp);
		}
	    }

	    integer_delref(n1);
	    integer_delref(d);
	    integer_delref(x);
	    DEBUG_INTEGER(("-miller_rabin - composite", 0));
	    return -1;
	}

    NEXT:
	integer_delref(x);
    }

    integer_delref(n1);
    integer_delref(d);

    DEBUG_INTEGER(("-miller_rabin - probably prime", 0));
    return 1;
}
