/******************************************************************************
  anf_elt_factorize_md.c                                                           
******************************************************************************/
 
#include "kant.h"
  
#define MINEXT    20
#define FBBOUND   50

t_logical
anf_elt_factorize_md WITH_2_ARGS(
	order,		ord,
	anf_elt,	alpha
)
/*******************************************************************************

Description: 
 
        Tries to factorize the principal ideal given by alpha into a
        factor basis of the order ord. If the factor basis has not been
        initialized yet this will be done automatically.
 
        Still missing:
 
        (2) enlarging of vec and mat only if norm decomposable
        (3) powers of ideals in ideal structure (Cohen)
        (4) Better handling if status != 0

Calling sequence: 
	
	if (anf_elt_factorize_md(ord, alpha)) ...
                               
	order		ord    = t_handle of the order 
	anf_elt		alpha  = t_handle of algebraic number
 
History:
                    
        92-09-09 JS     anf_elt_norm_abs
        92-09-06 MD     new calling sequence for order_fac_basis_create.
 	92-05-15 JS 	if alpha is a unit no factor basis is necessary
 	92-05-05 JS 	first version
  
*******************************************************************************/
{
	block_declarations;
  
        integer_small   idcnt, pcnt, rcnt, actl, i, j, degree, rindex, pexp, xpo;
        integer_small   numid;
        integer_big     norm, den, temp, p;
        anf_ideal       id, idpower, idtemp;
        anf_elt         alphan, rel;
        t_handle          ideals;
        matrix          mat, mat1, mat2;
        vector          vec, vec1;
        t_handle          Z;

/*  
    Maybe we have alpha (or -alpha) already?
    (This definitely could be done faster...)
*/
	rcnt  = order_relation_count(ord);


        if (rcnt > 0)
	{
	        alphan = anf_elt_negate(ord, alpha);
	        for (i=1; i<=order_relation_count(ord); i++)
	        {
	        	rel = vec_entry(order_relation_numbers(ord), i);
		        if (anf_elt_equal(ord, alpha,  rel) ||
		            anf_elt_equal(ord, alphan, rel))
	                {
	                        anf_elt_delete(ord, &alphan);
	                        return FALSE;
	                }
	        }
	        anf_elt_delete(ord, &alphan);
        }
	
 
/*
    at the moment only status = 0 is supported
*/
        if (order_relation_status(ord))
        {
          if (anf_print_level > 2)
          puts("anf_elt_factorize_md: status !=0 => new relation matrix.");
          order_relation_delete(ord);
        }
           
        Z = m_z_str_incref(structure_z);
 
/*
    we first need the norm of alpha
*/
        anf_elt_norm_abs (ord, alpha, &temp, &den);
 
/*
    the norm should be an integer
*/
        if (den != 1)
                error_internal("anf_elt_factorize_md: No denominator supported!");
        norm = integer_abs(temp);
        integer_delref(temp);
 
        if (anf_print_level > 4)
        {
                printf("Algebraic number: "); anf_elt_write(ord, alpha);
                printf(" with norm "); integer_write(norm); puts("");
        }
 
/*
    assuring existence of factor basis
*/
 
        if (!order_fac_basis(ord) && norm != 1)
                order_fac_basis_create_md (ord, FBBOUND,FALSE);
 
/*
    assuring correct length of relation matrix etc.
*/
                                   
	idcnt = order_fac_basis_ideals_count(ord);
	pcnt  = order_fac_basis_len(ord);
        mat   = order_relation_mat(ord);
        vec   = order_relation_numbers(ord);
 
        if ( !mat && idcnt)
        {
 	        if (anf_print_level > 2) 
		   printf("Creating relation matrix (%d by %d)...\n", idcnt, MINEXT);
                mat = mat_new(idcnt, MINEXT);
                mat_ring_create_zero_sub(Z, &mat);
        }
 
        if (!vec)
        {
                vec  = vec_new(MINEXT);
        } 
 
        if (mat && rcnt >= (actl=mat_col(mat)))
        {
 	        if (anf_print_level > 2) 
		   printf("Enlarging relation matrix (now %d)...\n", actl+MINEXT);
                mat2 = mat_new(idcnt, actl + MINEXT);
                mat1 = mat;
                mat_ring_create_zero_sub(Z, &mat2);
                mat  = mat_ring_insert(Z, mat2, mat1, 1, 1);
 
                mat_delref(Z, &mat1);
                mat_delref(Z, &mat2);
        }
         
        if (rcnt >= (actl=vec_length(vec)))
        {
                vec1 = vec;
                vec  = vec_new(actl + MINEXT);
                for (i=1; i<=rcnt; ++i)
                        vec_entry(vec, i) = anf_elt_incref(vec_entry(vec1, i));
 
                vec_delete(ord, &vec1);
        }
 
        order_relation_mat(ord)     = mat;
        order_relation_numbers(ord) = vec;
 
/*
    Now we can start to decompose!
    First the easy case: The number might be a unit.
*/ 
 
        if (norm == 1)
        {
                rcnt++;
                vec_entry(vec, rcnt)        = anf_elt_incref(alpha);
                order_relation_count(ord)   = rcnt;
	        ring_delref(&Z);
                return TRUE;
        }
 
/*
    We first decompose the norm.
    Here vec1 holds the valuations of the primes of the factor basis.
*/      

        vec1 = vec_new(pcnt);
        for (j=1; j<=pcnt; ++j)
	{                                             
                p = order_fac_basis_prime(ord, j);
	        vec_entry(vec1, j) = integer_prime_val_div(&norm, p);
        }  
 
/*
    Do we have any chance?
*/
	if (norm != 1)
        {
 	        if (anf_print_level > 4) puts("Norm could not be decomposed.");
                vec_delete(Z, &vec1);
	        ring_delref(&Z);
                return FALSE;
        }
        if (anf_print_level > 4) 
	{
	     	puts("Norm decomposition:");
                mat_anf_write(Z, vec1);
        }
 
/*
    Now we "decompose" alpha.
    We simply check the maximal power of the ideals id such that alpha
    still is contained in them.
*/      
                   
        rcnt++;
        rindex = 0;
        for (i=1; i<=pcnt; i++)
        {        

                /* this loop runs over all primes */
                ideals = order_fac_basis_ideals(ord, i);
                numid  = m_poly_z_faclst_len(ideals);
                pexp   = vec_entry(vec1, i);
 

                for (j=1; j<=numid; ++j)
                {

	                /* this loop runs over all prime ideals over prime #i */
                        rindex++;
                        id 	= m_poly_z_faclst_factor(ideals, j-1);
                        idpower = anf_ideal_incref(id);
                        degree 	= anf_ideal_degree(id); 
                        xpo     = 0;
                                             
 
                        /* let's check it */
                        while (pexp >= degree)
                        {                   
                                if (anf_elt_in_ideal(ord, alpha, idpower))
                                {
                                        pexp -= degree; xpo++;


                                        if (pexp >= degree)
                                        {                                     
                                                idtemp  = idpower;
                                                idpower = anf_ideal_mult(ord, id, idtemp);
                                                anf_ideal_delete(ord, &idtemp);
                                        }
                                }
                                else
                                {
                                        break;
                                }
                        }
 
                        /* we are done for a particular ideal */
                        mat_elt(mat, rindex, rcnt) = xpo;
                        anf_ideal_delete(ord, &idpower);
                } 
                /* preparing for the next prime. pexp must have been zeroed */
                if (pexp)
                {
                        /* it is not: alpha does not yield a relation. */
 
                        for (j=1; j<=rindex; j++) mat_elt(mat, j, rcnt) = 0;    

	                vec_delete(Z, &vec1);
		        ring_delref(&Z);
                	return FALSE;
                }
        }
 
/*    Here we have a new relation. */
 

        vec_entry(vec, rcnt)        = anf_elt_incref(alpha);
        order_relation_count(ord)   = rcnt;

        if (anf_print_level > 5) 
	{
	     	puts("Relation found! New relation matrix:");
                mat_anf_write(Z, mat);
                puts("");
        }
 
        vec_delete(Z, &vec1);
        ring_delref(&Z);
 
        return TRUE;
}
