/*******************************************************************************
-----------------------------------------------------------------------------

    ACHTUNG:
    Zur Vermeidung von zu grossen Kompatibilitaetsproblemen duerfen
    Aenderungen in den Strukturdefinitionen nur nach gemeinsamer
    Eroerterung bzw. Einigung zwischen der Cayley- und der Kant-Gruppe
    durchgefuehrt werden.
    Sydney, 25.9.1992             John Cannon, Johannes Gf. Schmettow

-----------------------------------------------------------------------------

    ATTENTION:

    To avoid major problems of incompatibility, changes to the structure
    definitions should only be made after common discussion and agreement
    between the Cayley and Kant groups.

    Sydney, 25/9/1992             John Cannon, Johannes Ct. Schmettow

-----------------------------------------------------------------------------
 
lattice.h

This header file includes structure definitions for lattices
and the "enumeration environment" of a lattice.
  
Notation (l = lattice, e = lat_enum_env):
 
------------------------------------------------------------------------------
| lattice.h                     |     [1]      |    [2]       |    [3]       |
------------------------------------------------------------------------------
| lat_rank(l)                   |      m       |     k        |     k        |
| lat_basis(l)                  |      B       |     B        |     -        |
| lat_gram(l)                   |      A       |     A        |     A        |
| lat_chol(l)                   |      Q       |     Q        |     Q        |
| lat_disc(l)                   |      -       |    d(.)      |    d(.)      |
| lat_succ_min(l, i)            |      -       |    M_i       |    M_i       |
| lat_succ_min_vec(l, i)        |      -       |     -        |    y_i       |
|                               |              |              |              |
| lat_enum_ubound(e)            |      C       |     C        |     C        |
| lat_enum_lbound(e)            |      C'      |     -        |     -        |
| lat_enum_ref_vector(e)        |      -       |  \tilde v    |     -        |
| lat_enum_coef_ubounds_user(e) |      -       |     -        |     -        |
| lat_enum_coef_lbounds_user(e) |      -       |     -        |     -        |
| lat_enum_act_real_coef(e, i)  |     x_i      |    z_i       |    x_i       |
| lat_enum_act_length(e)        |     Q(x)     |    Q(z)      |    Q(x)      |
| lat_enum_coef_lbound(e, i)    |      -       |     -        |     -        |
| lat_enum_coef_ubound(e, i)    |   UB(x_i)    |  UB(z_i)     |  UB(x_i)     | 
| lat_enum_part_length(e, i)    |     U_i      |    S_i       |    U_i       |
| lat_enum_tmp_bound(e, i)      |     T_i      |    T_i       |    T_i       |
------------------------------------------------------------------------------

 
Bibliography:
 
   [1] U. Fincke and M. Pohst, "Improved methods for calculating vectors
       of short length in a lattice, including a complexity analysis",
       Math. Comp. 44 (1985), 463-471.
   [2] H. W. Lenstra Jr., M.E. Pohst, "Computational algebraic number theory", 
       Birkhaeuser, to appear 1992, 28-30.
   [3] M. Pohst and H. Zassenhaus, "Algorithmic Algebraic Number Theory", 
       Cambridge Univ. Press 1989, 186-191.
 

History:
        92-09-17 JS        Move to Cayley V4
	92-04-01 KW 	   third version
	92-01-20 JS 	   second version
	91-12-31 JS 	   first version
  
*******************************************************************************/

#ifndef INCLUDED_LATTICE_H
#define INCLUDED_LATTICE_H
 
#include "integer.e"
#include "mat.e"
#include "real.e"
#include "dyn_arr.h"
#include "dmat.h"                 

/*******************************************************************************
/
/  Introducing new types
/
*******************************************************************************/
 
typedef t_handle  lattice;
typedef t_handle  lat_elt;
typedef t_handle  lat_enum_env;
 
/*
    temporarily (until update of block_type.h)
*/
 
#ifndef LATTICE_BLOCK_TYPE
 
#define LATTICE_BLOCK_TYPE       19567
#define LAT_ELT_BLOCK_TYPE       19568
#define LAT_ENUM_ENV_BLOCK_TYPE  19569
 
#endif  /* LATTICE_BLOCK_TYPE */

  
/*******************************************************************************
/
/  Structure definition for lattices
/
*******************************************************************************/
  
typedef struct
	{
		t_block_header	trbl_hdr;
                                      
                integer_small   rank;
 
		matrix          basis;
                t_handle          basis_ring;
 
                matrix          gram;
                t_handle          gram_ring;

                matrix          chol;
                t_handle          chol_ring;

		t_dmat		double_chol;

                t_handle          disc; 
                t_handle          disc_ring;
 
                dyn_arr_handle  succ_mins;
                dyn_arr_handle  succ_mins_vecs;
	}
	lattice_struct;
 
  
/*******************************************************************************
/
/   Macros for accessing structure items of lattices
/
*******************************************************************************/
 
/*
    general access macro for table element (internal use)
*/
  
#define	lat_access(h)		((lattice_struct *) mem_access(h))
 
 
/*
    rank of lattice
*/
 
#define lat_rank(h)		(lat_access(h) -> rank)
  
 
/*
    basis of lattice
    coefficient ring of basis
    Is the matrix known?
    Is the basis given over Z?
*/
 
#define lat_basis(h)		(lat_access(h) -> basis)
#define lat_basis_ring(h)	(lat_access(h) -> basis_ring)
#define lat_basis_known(h)      (lat_basis(h) != MEM_NH)
#define lat_basis_is_over_z(h)  (ring_type(lat_basis_ring(h)) == RING_Z)
 
   
/*
    Gram matrix
    coefficient ring of Gram matrix
    Is the matrix known?
    Is the Gram matrix given over Z?
*/
  
#define lat_gram(h)		(lat_access(h) -> gram)
#define lat_gram_ring(h)	(lat_access(h) -> gram_ring)
#define lat_gram_known(h)       (lat_gram(h) != MEM_NH)
#define lat_gram_is_over_z(h)   (ring_type(lat_gram_ring(h)) == RING_Z)
 
 
/*
    Matrix of Cholesky decomposition
    coefficient ring of Cholesky matrix (should be real)
    Is the matrix known?
    Is the Cholesky matrix given over Z?

*/
 
#define lat_chol(h)		(lat_access(h) -> chol)
#define lat_chol_ring(h)	(lat_access(h) -> chol_ring)
#define lat_chol_known(h)       (lat_chol(h) != MEM_NH)
#define lat_chol_is_over_z(h)   (ring_type(lat_chol_ring(h)) == RING_Z)

/*
    double-Matrix of Cholesky decomposition
*/

#define lat_double_chol(h)	(lat_access(h) -> double_chol)
         
/*
    Lattice discriminant
    Ring over which discriminant is defined (at the moment this is the
             same ring as the coefficient ring of the Cholesky matrix)
    Is the Discriminant known?
    ... assure its existence
*/
 
#define lat_disc(h)		(lat_access(h) -> disc)
#define lat_disc_ring(h)	(lat_access(h) -> disc_ring)
#define lat_disc_known(h)       (lat_disc(h) != MEM_NH)
#define lat_disc_assure(h)	if (!lat_disc_known(h)) lat_disc_calc(h)
 
 
/*
    Table of successive minima (internal use)
    Table of vectors which represent successive minima (internal use)
*/
 
#define lat_succ_mins(h)        (lat_access(h) -> succ_mins)
#define lat_succ_mins_vecs(h)   (lat_access(h) -> succ_mins_vecs)
 
 
/*
    accessing the successive minima:
       ... a certain minimum
       ... accessing number of minima given
       ... setting number of minima and allocating/assuring space
*/

#define lat_succ_min(h, i)	(dyn_arr_element(lat_succ_mins(h), (i)-1))
 
#define lat_succ_min_count(h) 	\
		(lat_succ_mins(h) == MEM_NH \
		? 0 \
		: (dyn_arr_curr_length(lat_succ_mins(h))) \
		)
 
#define lat_succ_min_count_set(h, len) \
		((lat_succ_mins(h) == MEM_NH \
		? lat_succ_mins(h) = dyn_arr_alloc((len)) \
		: dyn_arr_assure_space_fun(lat_succ_mins(h), (len), 1) \
		), dyn_arr_curr_length(lat_succ_mins(h)) = (len))
 

/*
    accessing the vectors of the successive minima:
       ... a certain vector
       ... accessing number of vectors given
       ... setting number of vectors and allocating/assuring space
*/

#define lat_succ_min_vec(h, i)	(dyn_arr_element(lat_succ_mins_vecs(h), (i)-1))
 
#define lat_succ_min_vec_count(h) 	\
		(lat_succ_mins_vecs(h) == MEM_NH \
		? 0 \
		: (dyn_arr_curr_length(lat_succ_mins_vecs(h))) \
		)
 
#define lat_succ_min_vec_count_set(h, len) \
		((lat_succ_mins_vecs(h) == MEM_NH \
		? (void) (lat_succ_mins_vecs(h) = dyn_arr_alloc((len))) \
		: (void) (dyn_arr_assure_space_fun(lat_succ_mins_vecs(h), (len), 1)) \
		), (void) (dyn_arr_curr_length(lat_succ_mins_vecs(h)) = (len)))
 

  
/*******************************************************************************
/
/  Structure definition for lattices elements
/
*******************************************************************************/
  
typedef struct
	{
		t_block_header	trbl_hdr;
		integer_big     lat_elt_coefs[VARIABLE_LENGTH];
	} 	
	lat_elt_struct;
 
/*******************************************************************************
/
/   Macros for accessing structure items of lattice elements
/
*******************************************************************************/

/*   
    length of header (internal use)
    total size of space required by the number (internal use)
*/
  
#define LAT_ELT_HDR_LEN			BLOCK_HEADER_WORDS
#define lat_elt_required_space(k)	((k) + LAT_ELT_HDR_LEN)
   
/*
    accessing 
       ... a certain place within the structure (internal use)
       ... a certain component
*/
  
#define lat_elt_access(e)    \
			((lat_elt_struct *) mem_access((e)))
 
#define lat_elt_coef(e, i)   (lat_elt_access(e) -> lat_elt_coefs[(i)-1])

  
/*
    allocating space
*/
 
#define lat_elt_alloc(e, k)	\
	{ \
		e = (lat_elt) mem_alloc_words_zero(lat_elt_required_space(k)); \
		block_init(e, LAT_ELT_BLOCK_TYPE); \
	}
  
/*                                
    lat_elt-/reference counter maintenance
       - reassigning a lattice element
       - increment reference counter 
       - has the element no references? (internal use)
       - has the element other references? (internal use)
*/
  
#define lat_elt_reassign(lat, a, b)	\
    { lat_elt _lat_elt_temp = a; a = b; lat_elt_delete(lat, &_lat_elt_temp ); }
 
#define lat_elt_incref(a) 		((lat_elt) block_incref(a))
#define lat_elt_has_no_refs(a) 		block_has_no_refs(a)
#define lat_elt_has_other_refs(a)	block_has_other_refs(a)
 
 
/*******************************************************************************
/
/  Structure definition for enumeration environments of lattices
/
*******************************************************************************/
  
typedef struct
	{
		t_block_header	trbl_hdr;
 
                t_real          ubound;
                t_real          lbound;

		double		double_ubound;
		double		double_lbound;
 
                dyn_arr_handle  ref_vector;
		t_dmat		double_ref_vector;
 
                dyn_arr_handle  coef_ubounds_user;
                dyn_arr_handle  coef_lbounds_user;
                dyn_arr_handle  coef_bounds_user_flags;
 
                integer_small   request;
		integer_small	strategy;
 
                integer_small   status;
		integer_small	precision;
 
                lat_elt         act_coefs;
                t_real          act_length;
 
		dyn_arr_handle  act_real_coefs;
                dyn_arr_handle  coef_lbounds;
                dyn_arr_handle  coef_ubounds;
                dyn_arr_handle	part_lengths;
                dyn_arr_handle  tmp_bounds;

		double		double_act_length;		

		t_dmat		double_tmp_bounds;
		t_dmat		double_part_lengths;

		integer_small	counter;
	}
	lat_enum_env_struct;
 
  
/*******************************************************************************
/
/   Macros for accessing structure items of enumeration environments
/
/   All real values have to refer to the real field of the Cholesky matrix 
/   of the lattice under consideration.
/
*******************************************************************************/
 
/*
    general access macro for table element (internal use)
*/

#define	lat_enum_access(h)		((lat_enum_env_struct *) mem_access(h))

  
/*
    Upper bound 
    Lower bound
       ... is it zero?
    Reference vector (for shortest or nearest vector)
       ... is it zero (i.e. are we looking for shortest vectors)?
       ... accessing a coefficient
*/
 
#define lat_enum_ubound(h)		(lat_enum_access(h) -> ubound)
#define lat_enum_lbound(h)		(lat_enum_access(h) -> lbound)
#define lat_enum_lbound_is_zero(h)	(lat_enum_lbound(h) == MEM_NH)
#define lat_enum_ref_vector(h)		(lat_enum_access(h) -> ref_vector)
#define lat_enum_ref_vector_is_zero(h)	(lat_enum_ref_vector(h) == MEM_NH)
#define lat_enum_ref_vector_coef(h, i)  (dyn_arr_element \
						(lat_enum_ref_vector(h), (i)-1))

#define lat_enum_double_ref_vector(h)		(lat_enum_access(h) -> double_ref_vector)
#define lat_enum_double_ref_vector_coef(h,i)    (dmat_entry(lat_enum_double_ref_vector(h),i))


/*
    double Upper bound
    double Lower bound
*/

#define lat_enum_double_ubound(h)	(lat_enum_access(h) -> double_ubound)
#define lat_enum_double_lbound(h)	(lat_enum_access(h) -> double_lbound)       


 
/*
    Coefficient bounds, provided by user
*/
 
#define lat_enum_coef_ubounds_user(h)	        \
				(lat_enum_access(h) -> coef_ubounds_user)
#define lat_enum_coef_lbounds_user(h)	        \
				(lat_enum_access(h) -> coef_lbounds_user)
#define lat_enum_coef_ubound_user(h, i)         	\
	(dyn_arr_element(lat_enum_coef_ubounds_user(h), (i)-1))
#define lat_enum_coef_lbound_user(h, i)         	\
	(dyn_arr_element(lat_enum_coef_lbounds_user(h), (i)-1))

#define lat_enum_coef_bounds_user_flags(h)	\
				(lat_enum_access(h) -> coef_bounds_user_flags)
#define lat_enum_coef_bounds_user_flag(h, i)	\
	(dyn_arr_element(lat_enum_coef_bounds_user_flags(h), (i)-1))
#define lat_enum_coef_ubound_user_flag_set(h, i)	\
	(lat_enum_coef_bounds_user_flag(h, i) |= 1)
#define lat_enum_coef_lbound_user_flag_set(h, i)	\
	(lat_enum_coef_bounds_user_flag(h, i) |= 2)
#define lat_enum_coef_ubound_user_flag_is_set(h, i)		\
	(lat_enum_coef_bounds_user_flag(h, i) & 1)
#define lat_enum_coef_lbound_user_flag_is_set(h, i)		\
	(lat_enum_coef_bounds_user_flag(h, i) & 2)
#define lat_enum_coef_ubound_user_flag_clear(h, i)	\
	(lat_enum_coef_bounds_user_flag(h, i) &= 2)
#define lat_enum_coef_lbound_user_flag_clear(h, i)	\
	(lat_enum_coef_bounds_user_flag(h, i) &= 1)

 
/*
    Accessing the request flag 
       ... "shortest vector"
       ... "next vector"
       ... setting to "shortest vector"
       ... setting to "next vector"
*/
 
#define lat_enum_request(h)       	(lat_enum_access(h) -> request)
#define lat_enum_request_is_shortest(h) (lat_enum_request(h) == 1)
#define lat_enum_request_is_next(h)     (lat_enum_request(h) == 2)
#define lat_enum_request_set_shortest(h) (lat_enum_request(h) = 1)
#define lat_enum_request_set_next(h)     (lat_enum_request(h) = 2)

/*
    Accessing the strategy flag
*/


#define lat_enum_strategy(h)		(lat_enum_access(h) -> strategy)
#define lat_enum_strategy_is_down(h)	(lat_enum_strategy(h) == 1)
#define lat_enum_strategy_is_up(h)	(lat_enum_strategy(h) == 2)
#define lat_enum_strategy_set_down(h)	(lat_enum_strategy(h) = 1)
#define lat_enum_strategy_set_up(h)	(lat_enum_strategy(h) = 2)
 
/*
    Flag for enumeration routines
*/
 
#define lat_enum_status(h)         	(lat_enum_access(h) -> status)
#define lat_enum_status_is_new(h)	(lat_enum_status(h) == -1)
#define lat_enum_status_is_done(h)	(lat_enum_status(h) == 0)
#define lat_enum_status_is_busy(h)	(lat_enum_status(h) == 1)
#define lat_enum_status_set_new(h)	(lat_enum_status(h) = -1) 
#define lat_enum_status_set_done(h)	(lat_enum_status(h) = 0)
#define lat_enum_status_set_busy(h)	(lat_enum_status(h) = 1)

                                            
/*
    Flag for double arithmetic
*/

#define lat_enum_precision(h)			(lat_enum_access(h) -> precision)
#define lat_enum_precision_is_short(h)		(lat_enum_precision(h) == 1)
#define lat_enum_precision_set_short(h)		(lat_enum_precision(h) = 1)
#define lat_enum_precision_set_long(h)		(lat_enum_precision(h) = 0)


/*
    Accessing the actual vector
       ... its square length
       ... its double square length
*/
 
#define lat_enum_act_coefs(h)         	(lat_enum_access(h) -> act_coefs)
#define lat_enum_act_length(h)         	(lat_enum_access(h) -> act_length)
#define lat_enum_double_act_length(h)	(lat_enum_access(h) -> double_act_length) 
 
/*
    Internal auxiliary arrays 
*/

#define lat_enum_act_real_coefs(h)     	(lat_enum_access(h) -> act_real_coefs) 
#define lat_enum_coef_lbounds(h)  	(lat_enum_access(h) -> coef_lbounds)
#define lat_enum_coef_ubounds(h)  	(lat_enum_access(h) -> coef_ubounds)
#define lat_enum_part_lengths(h)  	(lat_enum_access(h) -> part_lengths)
#define lat_enum_tmp_bounds(h)  	(lat_enum_access(h) -> tmp_bounds)

#define lat_enum_act_real_coef(h, i)    (dyn_arr_element \
					     (lat_enum_act_real_coefs(h), (i)-1))
#define lat_enum_coef_lbound(h, i)      (dyn_arr_element \
					     (lat_enum_coef_lbounds(h), (i)-1))
#define lat_enum_coef_ubound(h, i)      (dyn_arr_element \
					     (lat_enum_coef_ubounds(h), (i)-1))
#define lat_enum_part_length(h, i)      (dyn_arr_element \
					     (lat_enum_part_lengths(h), (i)-1))
#define lat_enum_tmp_bound(h, i)       	(dyn_arr_element \
					     (lat_enum_tmp_bounds(h), (i)-1))


#define lat_enum_double_tmp_bounds(h)  	(lat_enum_access(h) -> double_tmp_bounds)
#define lat_enum_double_part_lengths(h)  (lat_enum_access(h) -> double_part_lengths)

#define lat_enum_double_tmp_bound(h, i)    (dmat_entry(lat_enum_double_tmp_bounds(h),i))
#define lat_enum_double_part_length(h, i)  (dmat_entry(lat_enum_double_part_lengths(h),i))

/*

*/
  
#define lat_enum_counter(h)		(lat_enum_access(h) -> counter)

/*******************************************************************************
/
/   End
/
*******************************************************************************/
 
#endif /* INCLUDED_LATTICE_H */
