#ifndef _POLY_H_
#define _POLY_H_

#include "defs.h"
#include "gen_ring.h"
#include "ring.h"
#include "poly.e"




/*
Polynomials in multiple variables are stored in a canonical form based on
a total ordering of the variables: the variable with the smallest number is
called the PRINCIPAL variable and is stored "outermost".
*/


/* 
** Since a t_poly is a disguised t_handle, when we delete a t_poly variable
** we should set it to something which is obviously not valid.
*/

#define NULL_POLYNOMIAL  integer_BETA

/*
** Generic Ring access macros. 
*/
#define m_poly_gen_prms(ctx)         (ctx->gprim_fns)
#define m_poly_ctx_elt_add(c)     (m_gringp_elt_add(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_coerce(c)  (m_gringp_elt_coerce(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_delete(c)  (m_gringp_elt_delete(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_equal(c)   (m_gringp_elt_equal(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_gcd(c)     (m_gringp_elt_gcd(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_incref(c)  (m_gringp_elt_incref(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_mult(c)    (m_gringp_elt_mult(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_negate(c)  (m_gringp_elt_negate(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_power(c)   (m_gringp_elt_power(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_print(c)   (m_gringp_elt_print(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_quotrem(c) (m_gringp_elt_quotrem(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_slash(c)   (m_gringp_elt_slash(m_poly_gen_prms(c)))
#define m_poly_ctx_elt_subtract(c)        (m_gringp_elt_subtract(m_poly_gen_prms(c)))
#define m_poly_ctx_is_zero(c)     (m_gringp_is_zero(m_poly_gen_prms(c)))
#define m_poly_ctx_is_one(c)     (m_gringp_is_one(m_poly_gen_prms(c)))
#define m_poly_ctx_is_minus_one(c)     (m_gringp_is_minus_one(m_poly_gen_prms(c)))
 
/*
** Context access macros. 
*/
#define m_poly_context_cring(c)        	(c.cring)
#define m_poly_ctx_cring(c)        	(c->cring)
#define m_poly_ctx_quotient(c)   	(c->quotient)
#define m_poly_ctx_ideal(c)   		(c->ideal)



/*
The header for a block containing a polynomial has the following structure:
*/

typedef
    struct
	{
	    t_block_header		block_header;
	    t_int		poly_prnc_var;
	    t_int		poly_lst_prnc_var;
	    t_int		poly_ntrms;
	}
    poly_header;

/*
Its length in units of one t_int is:
*/

#define	POLY_HDR_LEN	MEM_BYTES_TO_WORDS( sizeof( poly_header ) )

/*
Each term of a polynomial consists of a coefficient c and an exponent e:
the term is  c * x ** e  where x is the principal variable.

Coefficients may be any polynomial, in "t_poly" form, provided that the
principal variable of the coefficient is one greater than that of the polynomial
which contains the coefficient.

Exponents are small integers.
*/

/*
A term has the following structure:
*/

typedef
    struct
	{
	    t_poly	poly_cft;
	    t_int	poly_xpt;
	}
    poly_termtype;

/*
Its length in units of one t_int is:
*/

#define	POLY_TERM_LEN	MEM_BYTES_TO_WORDS( sizeof( poly_termtype ) )

/*
The overall structure of a polynomial block is as follows:
*/

typedef
    struct
	{
	    poly_header		poly_hdr;
	    poly_termtype	poly_trm[ VARIABLE_LENGTH ];
	}
    poly_struct;

typedef		poly_struct	*t_polyp;


/*
The following macros convert between the disguised and undisguised forms of the
t_handle for true polynomials:
*/

#ifdef POSH
#define m_poly_poly_to_handle(p)  ((p) - integer_BETA)
#define m_poly_handle_to_poly(h)  (integer_BETA + (h))
#else
#define m_poly_poly_to_handle(p)  (integer_BETA - (p))
#define m_poly_handle_to_poly(h)  (integer_BETA - (h))
#endif

/* 
** The following macro gives a t_polyp from a t_poly
** We ought to make this conversion as soon as we know it isn't
** constant.
*/

#define m_poly_to_ptr(apoly) ((t_polyp)  mem_access(m_poly_poly_to_handle(apoly)) )
#define m_poly_hdl_to_ptr(aph) ((t_polyp)  mem_access(aph) )

/*
** The following macros access the header and the terms.  Note that they take
** a t_poly_ptr as argument.
*/

#define m_polyp_least_pvar(p)		(p->poly_hdr.poly_lst_prnc_var)
#define m_polyp_princvar(p)		(p->poly_hdr.poly_prnc_var )
#define m_polyp_nterms(p)              (p->poly_hdr.poly_ntrms )
 
#define m_polyp_coefft(p,i)		( p->poly_trm[i].poly_cft )
#define m_polyp_expt(p,i)            	( p->poly_trm[i].poly_xpt )
 

/*
The following macros access the header and the terms.  Note that they take
a poly_handle not a t_poly as argument.
*/


#define	m_poly_access(ph)		( ( (poly_struct *) mem_access(ph) ) -> poly_hdr )

#define m_poly_least_pvar(ph)	( m_poly_access(ph).poly_lst_prnc_var)
#define m_poly_princvar(ph)	( m_poly_access(ph).poly_prnc_var )
#define m_poly_nterms(ph)		( m_poly_access(ph).poly_ntrms )

#define m_poly_term(ph,i)	(((poly_struct *) mem_access(ph) ) -> poly_trm[i] )

#define	m_poly_coefft(ph,i)	( m_poly_term(ph,i).poly_cft )
#define	m_poly_expt(ph,i)		( m_poly_term(ph,i).poly_xpt )

/*
The following gives the maximum number of terms which a polynomial block has
room to contain without extension:
*/

#define	m_poly_maxterms(ph)	\
	( ( mem_words(ph) - POLY_HDR_LEN ) / POLY_TERM_LEN )

/*
The following gives the number of bh_ints required to store a polynomial with
n terms:
*/

#define m_poly_required_size(n)	( POLY_HDR_LEN + (n) * POLY_TERM_LEN )

/*
The following allocates space for a new polynomial
*/

extern t_void f_poly_create_empty MEM_P_4(t_handle *, t_int, t_int, t_int);
#define m_poly_create_empty(ph,prvar, lstvar,Nterms)\
	 f_poly_create_empty MEM_CALL_4(ph,prvar, lstvar,Nterms)

#define m_poly_u_create_empty(ph, Nterms)\
		f_poly_create_empty MEM_CALL_4(ph, 1, 1,Nterms)


extern t_handle f_poly_array_alloc MEM_P_1(t_int);
#define poly_array_alloc(length) f_poly_array_alloc MEM_CALL_1(length)
/*
The following discards space after the n-th term:
*/

#define m_poly_red(ph,n)	\
	mem_reduce_words( ph, m_poly_required_size( n ) )


/*
** macros to check for univariate polynomials
*/

#define m_poly_univariate(aph)      (m_poly_princvar(aph) == m_poly_least_pvar(aph))
#define m_polyp_univariate(ap)      (m_polyp_princvar(ap)==m_polyp_least_pvar(ap))
#define m_poly_not_univariate(aph)  (m_poly_princvar(aph) != m_poly_least_pvar(aph))
#define m_polyp_not_univariate(ap)  (m_polyp_princvar(ap)!=m_polyp_least_pvar(ap))

/*
** macros to distinguish constant from non-constant polynomials
** In fact, they mean very little as a polynomial ring over a polynomial 
** ring would have "constants" whose constant term is a polynomial.
** 
** Try not to use these macros.
*/

#define m_poly_not_const( apoly ) \
	( ( ! m_poly_is_small_int( apoly ) ) && \
	  ( block_type( m_poly_poly_to_handle( apoly ) ) == POLY_BLOCK_TYPE ) )

#define m_poly_const( apoly ) \
	( m_poly_is_small_int( apoly ) || \
	  ( block_type( m_poly_poly_to_handle( apoly ) ) != POLY_BLOCK_TYPE ) )


/*
Auxilliary reference count checks
*/

#define m_poly_has_other_refs(h)	block_has_other_refs(h)

#define m_poly_u_zm_is_monic( pdig, apoly ) \
	(    ( apoly == 1 ) \
	  || ( m_poly_coefft( m_poly_poly_to_handle( apoly ), m_poly_nterms( m_poly_poly_to_handle( apoly )) - 1 ) == 1 ) \
	)

/*
** Structure of the block containing a polynomial ring.
*/

typedef struct
	{
		t_ring_common	ring_hdr;

		t_handle	coeff_ring;
		integer_small	nvars;
		integer_small	first_var;
		t_handle	named_ring;
		t_handle	previous_ring;
	}
	t_poly_table;

typedef	t_poly_table	*t_pring_ptr;

/*
** Access to the information in the polynomial ring.
*/

#define m_pring_access(h)		((t_pring_ptr) mem_access(h))

#define m_poly_coeff_ring_ptr(p)	(p -> coeff_ring)
#define m_poly_coeff_ring(h)	(m_poly_coeff_ring_ptr(m_pring_access(h)))

#define m_poly_coeff_ring_zero(h)	(ring_zero(m_poly_coeff_ring(h)))
/* ring zero is the function in variety_ring */

#define m_poly_nvars_ptr(p)	(p->nvars)
#define m_poly_nvars(h)		(m_poly_nvars_ptr(m_pring_access(h)))

#define m_poly_first_var_ptr(p)	(p->first_var)
#define m_poly_first_var(h)	(m_poly_first_var_ptr(m_pring_access(h)))

#define m_poly_named_ring(h)	(m_pring_access(h) -> named_ring)
#define m_poly_prev_ring(h)	(m_pring_access(h) -> previous_ring)

#define m_poly_lower_ring(h) 	(m_poly_prev_ring(h)?m_poly_prev_ring(h):m_poly_coeff_ring(h))


/*
** A polynomial is either a ring element, or a disguised t_handle to a block.
*/


/*
** The next type represents the t_handle (in undisguised form) of a "true"
** polynomial:
*/


/*
** The following macro distinguishes the small integer case from all
** other cases:
*/

#define m_poly_is_small_int(p)	integer_is_single(p)

/*
 *  Handy macros
 */
 
#define m_poly_z_handle_incref( pring, p )  block_incref( p )
 
#define m_poly_z_incref( pring, p ) \
    ( m_poly_is_small_int( p ) || \
      block_incref( m_poly_poly_to_handle( p ) ), \
      p )
 
#define m_modpoly_handle_incref m_poly_z_handle_incref
#define m_poly_zm_handle_incref m_poly_z_handle_incref
#define m_modpoly_incref    m_poly_z_incref
#define m_poly_zm_incref    m_poly_z_incref
 
#define m_poly_gf_incref    m_poly_z_incref
 
#define m_poly_z_delref(pring, p)   \
    ((void) ( m_poly_is_const( p ) \
        ? (integer_delref( p ), 1 ) \
        :(poly_z_handle_delref( pring, m_poly_poly_to_handle(p)), 1 ) \
      ) )
 
#define m_modpoly_delref m_poly_z_delref
#define m_poly_zm_delref m_poly_z_delref
 
#define m_poly_gf_delref(K, p) m_poly_z_delref(K, p)
 
#define m_poly_str_incref(h) (block_incref(h))

/*
** The following macro distinguishes between a true polynomial and a ring
** element.
*/

#define m_poly_is_const(p) \
    (  m_poly_is_small_int(p) \
    || block_type( m_poly_poly_to_handle( p )) != POLY_BLOCK_TYPE \
    )

#include "poly_protos.h"

#endif /* _POLY_H_ */
