#include <LiDIA/polynomial.h>

// BASE_POLYNOMIAL FUNCTIONS

/**
 ** member functions
 **/
template < class T > 
void base_polynomial < T >::set_degree(lidia_size_t d)
{
  debug_handler_l("base_polynomial", "in member - function "
                  "set_degree(lidia_size_t)",  5);
  
  if (d < 0)
    lidia_error_handler("void base_polynomial < T >",
			"set_degree(lidia_size_t d):: d must be non-negative");
  
  if (d == deg)
    return;

  T *tmp = coeff;
  coeff = new T [d + 1];

  lidia_size_t minimum = ( d < deg)? d : deg;

  for (register lidia_size_t i = 0; i <= minimum; i++)
    coeff[i] = tmp[i];
  
  deg = d;
  delete[] tmp;    
}

template < class T > 
void base_polynomial < T >::remove_leading_zeros()
{
  debug_handler_c("base_polynomial", "remove_leading_zeros ()", -1, 
		  cout<<"degree is "<<deg<<endl<<flush);
  T c, *ap, *np;
  lidia_size_t d, i;

  d = deg;
  ap = coeff + d;
  while (d >= 0){
    if (!ap->is_zero())
      break;
    else
      d--, ap--;
  }

  if (d == -1){
    deg = 0;
    delete[] coeff;
    coeff = new T[1];
    coeff[0].assign_zero();
  }
  else if (d != deg){
    debug_handler_c("base_polynomial","remove_leading_zeros()", -1,
		    cout << "degree is "<<d<<endl<<flush);
    deg = d;
    np = new T[d + 1];
    for (i = 0; i <= d; i++){
      debug_handler_c("base_polynomial","remove_leading_zeros()", -1,
		      cout << "assign for "<<i<<endl<<flush);
      np[i] = coeff[i];
    }
    delete[] coeff;
    coeff = np;
  }
}

/**
 ** assignment
 **/

template < class T > 
base_polynomial < T > & base_polynomial < T >::operator = (const base_polynomial < T > &a)
{
  debug_handler_l("base_polynomial","operator =(const base_polynomial <T> &)", -1);
  if (deg != a.deg){
    delete[] coeff;
    deg = a.deg;
    coeff = new T[deg + 1];
  }
  for (register lidia_size_t i = 0; i <= deg; i++)
    coeff[i] = a.coeff[i];
  return *this;
}

/**
 ** operator overloading
 **/

template < class T > 
T base_polynomial < T >::operator() (T value) const
{
  T result = 0;
  if (deg == 0){
    result = coeff[0];
  }
  else{
    result = coeff[deg];
    for (lidia_size_t i = deg - 1; i >= 0; i--){
      result *= value;
      result += coeff[i];
    }
  }
  return result;
}

template < class T > 
base_polynomial < T > operator + (const base_polynomial < T > &a,
			     const base_polynomial < T > &b)
{
  T *ap, *bp, *cp;
  lidia_size_t deg_a = a.deg, deg_b = b.deg;
  lidia_size_t i, min_deg_ab, max_deg_ab;

  debug_handler_l("base_polynomial", "operator +(...)", -1);
  if (deg_a > deg_b){
    max_deg_ab = deg_a;
    min_deg_ab = deg_b;
  }
  else{
    max_deg_ab = deg_b;
    min_deg_ab = deg_a;
  }

  base_polynomial < T > c;
  c.set_degree(max_deg_ab);

  ap = a.coeff;
  bp = b.coeff;
  cp = c.coeff;

  for (i = min_deg_ab + 1; i; i--, ap++, bp++, cp++)
    *cp = (*ap) + (*bp);

  if (deg_a > min_deg_ab)
    for (i = deg_a - min_deg_ab; i; i--, cp++, ap++)
      *cp = *ap;
  else if (deg_b > min_deg_ab)
    for (i = deg_b - min_deg_ab; i; i--, cp++, bp++)
      *cp = *bp;
  else
    c.remove_leading_zeros();
  return c;
}

template < class T >
base_polynomial < T > operator - (const base_polynomial < T > &a,
			     const base_polynomial < T > &b)
{
  T *ap, *bp, *cp;
  lidia_size_t deg_a = a.deg, deg_b = b.deg;
  lidia_size_t i, min_deg_ab, max_deg_ab;

  debug_handler_l("base_polynomial", "operator -(...)", -1);
  if (deg_a > deg_b){
    max_deg_ab = deg_a;
    min_deg_ab = deg_b;
  }
  else{
    max_deg_ab = deg_b;
    min_deg_ab = deg_a;
  }

  base_polynomial < T > c;
  c.set_degree(max_deg_ab);

  ap = a.coeff;
  bp = b.coeff;
  cp = c.coeff;
  for (i = min_deg_ab + 1; i; i--, ap++, bp++, cp++)
    *cp = (*ap) - (*bp);

  if (deg_a > min_deg_ab)
    for (i = deg_a - min_deg_ab; i; i--, cp++, ap++)
      *cp = *ap;
  else if (deg_b > min_deg_ab)
    for (i = deg_b - min_deg_ab; i; i--, cp++, bp++)
      *cp = -(*bp);
  else
    c.remove_leading_zeros();
  return c;
}

template < class T >
base_polynomial < T > operator * (const base_polynomial < T > &a,
			     const base_polynomial < T > &b)
{
  debug_handler_l("base_polynomial", "operator *(poly,poly)", -1);
  T *ap, *bp, *cp;
  lidia_size_t deg_a = a.deg, deg_b = b.deg;
 
  lidia_size_t i, j, deg_ab = deg_a + deg_b;

  base_polynomial < T > c;
  c.set_degree(deg_ab);

  for (i = deg_ab + 1, cp = c.coeff; i; i--, cp++)
    cp->assign_zero();

  for (i = 0, ap = a.coeff; i <= deg_a; i++, ap++)
    for (j = deg_b + 1, bp = b.coeff, cp = c.coeff + i; j; j--, bp++, cp++)
      *cp += (*ap) * (*bp);
  c.remove_leading_zeros();
  return c;
}

template < class T >
void power(base_polynomial < T > & c, 
	   const base_polynomial < T > & a, const bigint & b)
{
  bigint exponent;
  base_polynomial < T > multiplier;
  if (b.is_negative())
    c.assign_zero();
  else if (b.is_zero() || a.is_one())
    c.assign_one();
  else{
    exponent.assign(b);
    multiplier = a;
    c.assign_one();
    while (exponent.is_gt_zero()){
      if (!exponent.is_even())
	c *= multiplier;
      multiplier *= multiplier;
      exponent.divide_by_2();
    }
  }
}

/**
 ** functions
 **/

template < class T >
base_polynomial < T > derivation(const base_polynomial < T > &a)
{
  lidia_size_t d = a.deg;
  if (d == 0){
    a.coeff[0].assign_zero();
    return a;
  }
  base_polynomial < T > c;
  c.set_degree(d - 1);
  for (lidia_size_t i = 1; i <= d; i++)
    c.coeff[i - 1] = a.coeff[i] * T(i);
  return c;
}

/**
 ** input / output
 **/

template < class T > 
istream & operator >> (istream & s, base_polynomial < T > &a)
{
  debug_handler_l("base polynomial", "in operator "
		  "istream & operator >>(istream &, base_polynomial < T > &)",
		  1);
  
  register lidia_size_t n = 0; 
  lidia_size_t sz = 8;
  char c;
  
  T *vbuf = new T[sz];
  memory_handler(vbuf, "base_polynomial", "operator >> :: "
		 "out of memory");
  s >> ws >> c;
  if (c != '['){
    // lidia_error_handler("base_polynomial", "operator>> :: '[' expected.");
    s.putback(c);
    return a.read_verbose(s);
  }
  s >> ws >> c;
  while (c != ']'){
    s.putback(c);
    s >> vbuf[n];
    n++;

    if (n == sz){
      debug_handler_l("base_polynomial", "operator>> :: "
		      "doubling input size", 1);
	  
      lidia_size_t osz = sz - 1;
	  
      sz *= 2;
	  
      T *nbuf = new T[sz];
      memory_handler(nbuf, "base_polynomial", "operator >> :: "
		     "out of memory");
	  
      a.copy_data(nbuf, vbuf, osz);
      delete[] vbuf;
      vbuf = nbuf;
    }
    s >> ws >> c;
  }

  if (n < 1){
    a.assign_zero();
    return s;
  }

  n--;
  a.set_degree(n);
  for (register lidia_size_t i = 0; i <= n; i++)
    a.coeff[n-i] = vbuf[i];
  delete[] vbuf;

  a.remove_leading_zeros();
  return s;
}

template < class T > 
istream & base_polynomial < T >::read_verbose(istream & s)
{
  debug_handler_l("base polynomial", "in member-function "
		  "istream & read_verbose(istream &)", 1);

  register lidia_size_t n = 0;
  lidia_size_t sz;
  char c;
  
  set_degree(8);
  for(;n <= deg; n++)
    coeff[n] = 0;

  char variable = 0;
  T coeff_tmp = 1;
  T tmp;

  // Read a monomial, i.e. "x^k" or "- x^k"
  // or "a*x^k" or "a x^k" or "x^k*a" or "x^k a"

  do{
    c = s.get();
  } while (isspace(c) && c != '\n');
  while (c!='\n' && c != EOF){
    sz = 0;		// Assume we read coeffizient of x^0;
    if (c=='+'){
      coeff_tmp = 1;
      do{
	c = s.get();
      } while (isspace(c) && c != '\n');
    }
    if (c=='-'){
      coeff_tmp = -1;
      do{
	c = s.get();
      } while (isspace(c) && c != '\n');
    }
#ifdef POLYREAD_DEBUG
    cout << "\n 1) Now looking at " << c;
#endif
    if (c>='1' && c<='9' || c == '('){
      s.putback(c);
      s >> tmp;
      coeff_tmp *= tmp;
      do{
	c = s.get();
#ifdef POLYREAD_DEBUG
	cout << ", looking at "<< c;
#endif
      } while (isspace(c) && c != '\n');
      if (c == '*')
	do{
	  c = s.get();
	} while (isspace(c) && c != '\n');
#ifdef POLYREAD_DEBUG
      cout << "\n coeff_tmp is now "<<coeff_tmp;
      cout << ", looking at "<< c;
#endif
    }
#ifdef POLYREAD_DEBUG
    cout << "\n 2) Now looking at " << c;
#endif
    if ((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z')){
      if (variable == 0)
	variable = c;
      else if (variable != c)
	lidia_error_handler_c("base_polynomial", "member function "
			      "read_verbose: The given string is not "
			      "recognized to be a univariate polynomial",
			      cout << "Variable name seemed to be "<<variable;
			      cout << " and now you use "<<c<<endl);
      do{
	c = s.get();
      } while (isspace(c) && c != '\n');
#ifdef POLYREAD_DEBUG
      cout << "\n 3) Now looking at " << c;
#endif

      if (c != '^') sz = 1;
      else {
	s >> sz;
#ifdef POLYREAD_DEBUG
	cout << "sz ist "<<sz;
#endif
	do {
	  c = s.get();
#ifdef POLYREAD_DEBUG
	  cout << "\n4') Looking at "<<c;
#endif
	} while (isspace(c) && c != '\n');
#ifdef POLYREAD_DEBUG
	cout << "\n 5) Now looking at " << c;
#endif
      }
      if (c=='*'){
	s >> tmp;
	coeff_tmp *= tmp;
	do{
	  c = s.get();
	} while (isspace(c) && c != '\n');
      }
#ifdef POLYREAD_DEBUG
      cout << "\n 6) Now looking at " << c;
#endif

      if (c>='1' && c<='9'|| c == '('){
	s.putback(c);
	s >> tmp;
#ifdef POLYREAD_DEBUG
	cout <<"\n Old coeff_tmp: "<<coeff_tmp;
#endif
	coeff_tmp *= tmp;
#ifdef POLYREAD_DEBUG
	cout <<"\n New coeff_tmp: "<<coeff_tmp;
#endif
	do{
	  c = s.get();
	} while (isspace(c) && c != '\n');
      }
    }

    if (c!='+' && c!='-' && c!='\n' && c!=EOF)
	lidia_error_handler_c("base_polynomial", "member function "
			      "read_verbose: The given string is not "
			      "recognized as univariate polynomial",
			      cout << "Found '"<<c;
			      cout <<"' instead of '+' or '-'"<<endl);
    if (sz >= n){
      set_degree(sz);
      for(;n <= deg; n++)
	coeff[n] = 0;
    }
    coeff[sz]+=coeff_tmp;
    
#ifdef POLYREAD_DEBUG
    cout << "\nSuccessfully read next monomial; Poly now is "<<(*this);
    cout << "\n Now looking at " << c;
#endif
  }
  remove_leading_zeros();
  return s;
}

/* print polynom */
template < class T > 
ostream & operator << (ostream & s, const base_polynomial < T > &a)
{
  debug_handler_l("base_polynomial","operator <<(...)", -1);
  lidia_size_t d = a.deg;

  if (d == 0)
    s << a.coeff[0];
  else if (d == 1){
    if (a.coeff[1] == T(1))
      s << "x";
    else 
      s << a.coeff[1] << " * x";
    if (a.coeff[0] != T(0))
      s << "+ " << a.coeff[0];
  }
  else {
    if (a.coeff[d] == T(1))
      s << "x^" << d;
    else
      s << a.coeff[d] << " * x^" << d;
    for (register lidia_size_t i = d - 1; i > 1; i--)
      if (a.coeff[i] == T(1))
	s << " + x^" << i;
      else if (a.coeff[i] != T(0))
	s << " + " << a.coeff[i] << " * x^" << i;
    if (a.coeff[1] == T(1))
      s << " + x";
    else if (a.coeff[1] != T(0))
      s << " + " << a.coeff[1] << " * x";
    if (a.coeff[0] != T(0))
      s << " + " << a.coeff[0];
  }
  return s;
}

// template < class T > 
// ostream & operator << (ostream & s, const base_polynomial < T > &a)
// {
//   lidia_size_t i, d = a.deg;

//   s << "[";
//   for (i = 0; i <= d; i++){
//     s << " ";
//     s << a.coeff[i];
//   }
//   s << " ]";
//   return s;
// }


// POLYNOMIAL FUNCTIONS

/**
 ** assignment
 **/

template < class T > 
polynomial < T > & polynomial < T >::operator = (const base_polynomial < T > &a)
{
  debug_handler_l("polynomial","operator =(const base_polynomial <T> &)", -1);
  if (deg != a.degree()){
    delete[] coeff;
    deg = a.degree();
    coeff = new T[deg + 1];
  }
  for (register lidia_size_t i = 0; i <= deg; i++)
    coeff[i] = a[i];
  return *this;
}
