//
// LiDIA - a library for computational number theory
//   Copyright (c) 1994, 1995 by the LiDIA Group
//
// File        : alg_number_appl.c (based on bigrational_appl.c,
//				    based on the GNU Integer Test) 
// Author      : Stefan Neis
//

#include <assert.h>
#include <stdlib.h>
#include <fstream.h>

#if defined(HAVE_MAC_DIRS) || defined(__MWERKS__)
#include <LiDIA:alg_number.h>
#include <LiDIA:timer.h>
#else
#include <LiDIA/alg_number.h>
#include <LiDIA/timer.h>
#endif

ifstream in("alg_appl_input");

void identitytest(const module& a, const module& b, const module& c)
{
  assert( (a + b) ==  (b + a));
  assert( (a & b) ==  (b & a));
  assert( (a * b) ==  (b * a));

  debug_handler_l("module_appl::identitytest","assertions 1 to 3 succeded",3);

  assert( (a + (b + c)) ==  ((a + b) + c));
  assert( (a + (b + c)) ==  ((a + c) + b));
  assert( (a * (b * c)) ==  ((a * b) * c));
  assert( (a * (b * c)) ==  ((c * a) * b));
  assert( (a & (b & c)) ==  ((a & b) & c));
  assert( (a & (b & c)) ==  ((c & a) & b));

  debug_handler_l("module_appl::identitytest","assertions 4 to 9 succeded",3);

  debug_handler_c("module_appl::identitytest",
		  "provisional result for assertion 4",0,
		  cout << "a:\n"<< a <<"\nb:\n" << b <<"c:\n"<<c;
		  cout <<"\nb+c: \n"<<b+c<< "\na*(b+c)\n"<<a*(b+c)<<"\na*b\n";
		  cout <<a*b <<"\na*c\n"<<a*c<<"\nSumme: \n"<<(a*b)+(a*c));
  assert( (a * (b + c)) ==  ((a * b) + (a * c)));
  assert( (a * (b & c)) ==  ((a * b) & (a * c)));
  assert( (a & (b + c)) ==  ((a & b) + (a & c)));

  debug_handler_l("module_appl::identitytest",
		  "assertions 10 to 12 succeded",3);

  debug_handler_c("module_appl::identitytest","preparing assertion 11",3,
		  cout<<a<<b<<(a&b)<<(a&b)+b);
  assert( ((a & b) + b) ==  b);
  assert( ((a + b) & b) ==  b);
//  assert( ((a * b) & b) ==  ((a&b)*b));
//  assert( ((a * b) & a) ==  (a*(a&b)));
  assert( ((a & b) & a) ==  (a & b));

  debug_handler_l("module_appl::identitytest",
		  "assertions 13 to 17 succeded",3);

  assert( ((a * b) / b) == a);
  assert( ((a * b) / a) == b);
  assert( ((a / b) * b) == a);
  assert( ((b / a) * a) == b);
  assert( ((a / b) / a) ==  inverse(b));

  debug_handler_l("module_appl::identitytest",
		  "assertions 18 to 20 succeded",3);

  module d;

  add(d, b, a);  assert( (a + b) ==  d);
  intersect(d, a , b);  assert( (a & b) ==  d);
  multiply(d, a, b); assert( d ==  (b * a));
  d.assign(b); multiply(d, a, d); assert( d ==  (a * b));
  intersect(d, a, b); assert( (a & b) ==  d);
  divide(d, a, b); assert( (a / b) ==  d);
  add(d, b, c); multiply(d, a, d);  assert(d ==  ((a * b) + (a * c)));
  d = b; d.invert();  assert( ((a / b) / a) == d);
  debug_handler_l("module_appl::identitytest",
		  "prozedural versions completed successful",3);
}


void utiltest( module& a)
{
  module b, c;

  square(b, a);
  multiply(c, a, a);
  assert(b == c);

  module x = alg_number(1), y = alg_number(1);

  for (int i = 0; i < 10; ++i)
  {
    power(b, a, i);
    assert(b == x);
    x *= a;
    y = a * y;
    assert(y == x);
  }
  x.assign_one();
  assert(x.is_one());
}


void accumtest( module& a,  module& b,  module& c)
{
  module x = a;
  x *= b;
  assert(x == (b * a));
  x += c;
  assert(x == (c+(b * a)));
  x &= a;
  assert(x == ((c & a) + ((b * a) & a)));
  x /= b;
  assert(x == ((c & a)/b + ((b * a)& a)/b));

  x.assign(a);
  assert(x == a);
  multiply(x, x, b);
  assert(x == (b * a));
  add(x, x, c);
  assert(x == (c+(b * a)));
  intersect(x, x, a);
  assert(x == ((c + (b * a))&a));
  divide(x, x, b);
  assert(x == (((c + (b * a))&a)/b));
}

// void anothertest()
// {
//   module pow64;
//   power(pow64, module(bigint(2)), 64);
  
//   bigint s64 = 1;
//   s64 <<= 64;

//   assert(alg_number(s64) == pow64);
//   assert(!(alg_number(s64) != pow64));
  
//   bigint s32 = s64 >> 32;
//   assert(!(pow64 == alg_number(s32)));
//   assert(pow64 != alg_number(s32));

//   debug_handler_l("module_appl::anothertest",
// 		  "simple tests succeded, starting identitytest",4);
//   identitytest(module(s64), module(s32), pow64);
//   debug_handler_l("module_appl::anothertest",
// 		  "identitytest succeded, starting accumtest",4);
//   accumtest(pow64, alg_number(s32), pow64);
//   debug_handler_l("module_appl::anothertest",
// 		  "accumtest succeded, starting utiltest",4);
//   utiltest(alg_number(s32));
// }

// void iotest()
// {
//   module result;

//   cout << "\nenter an module: ";
//   cin >> result;
//   cout << "module = " << result << "\n";
// }


void all_test(const bigint & ii)
{
    module a,b;

    do{
	a.randomize(ii);
	a *= a.denominator();	// simplified testing
    } while (a*a.denominator()==(*a.which_order()));

    do{
	b.randomize(ii);
	b *= b.denominator();	// simplified testing
    } while (b*b.denominator()==(*b.which_order()));

    cout << "\n\n Test with \na = " << a << "\nb = " << b<<endl;

    utiltest(a);
    identitytest(a, b, b);
    identitytest(a, a, b);

    accumtest(a, a, b);
    accumtest(b, a, b);

    utiltest(a);
    utiltest(b);
}


int main()
{
  order O;
  cout << "Order:\n"<<flush;
  in >> O;

  cout << "Maximalordnung berechnen dauert:"<<flush;
  timer T;
  T.start_timer();
  order ext = O.maximize();
  T.stop_timer();
  cout << T <<  endl << flush;

  current_order = &ext;
  
// The following 3 lines lead to an internal compiler error,
// if gcc-2.6.2 is used:
  module one(alg_number(bigint(1)),alg_number(bigint(0)));
  cout << "one = " << one << "\n";
  assert(one == one + alg_number(bigint(1)));

  module n(alg_number(bigint(0)),alg_number(bigint(0)));
  assert(n.is_zero());
  debug_handler_l("module_appl",
		  "simple tests succeded, starting anothertest",5);
//  anothertest();
//  debug_handler_l("module_appl","anothertest succeded, starting iotest",5);
//  iotest();
  debug_handler_l("module_appl","simple tests succeded, starting random tests",5);

  bigint i=129099121;

  for(int j=0; j<10; j++)
        all_test(i);

  cout << "\nEnd of test\n";
  return 0;
}
