% SOFTSOL.RED  F.J.Wright, FJW@Maths.QMW.AC.UK, London  13/3/92

% This file provides an alternative interface called SOFT_SOLVE
% to the SOLVE package, with identical input specification.
% The difference is that when the errors listed below occur,
% SOFT_SOLVE returns the corresponding error number as its value
% instead of failing with a hard REDUCE error.  This allows a
% program calling SOLVE to trap the error and take appropriate action.
% An error return can be detected by using NUMBERP (or FIXP),
% because if SOFT_SOLVE returns normally then its value is always a
% list, as for SOLVE.  (Conversely, LISTP defined at the end of this
% file could be used.)  See SOFTSOLV.TST for examples of use.

% The error numbers returned by SOFT_SOLVE and the corresponding
% error message that would be output by SOLVE are the following,
% collected from the SOLVE source code:

%  1  =  "SOLVE called with no equations"
%  2  =  "zero multiplicity"
%  3  =  "SOLVE called with no variables"
%  4  =  "Non linear equation solving not yet implemented"
%  5  =  "SOLVE given inconsistent equations"
%  6  =  "SOLVE given singular equations"
%  7  =  "Sub error in glnrsolve"

% If any other error number, such as 0, should occur then it
% corresponds to some other non-specific error.


fluid'(soft!-rerror!-number);

symbolic procedure soft_solveeval u;
   % Based on definition of solveeval in SOLVE.RED.
   % Catch errors from solve, which always returns a
   % list if successful, and return an error number
   % instead of failing with a hard REDUCE error.
   begin scalar !*redefmsg;
         % to avoid PSL messages about redefinition of rerror.
      integer soft!-rerror!-number;   % defaults to 0 rather than nil
      copyd('rerror, 'soft!-rerror); % redefine rerror
      u := errorset!*(list('solveeval, mkquote u), nil);
      copyd('rerror, 'rerror!%);    % restore rerror
      return if errorp u then soft!-rerror!-number else car u
   end;

put('soft_solve, 'psopfn, 'soft_solveeval);

symbolic procedure rerror!%(packagename,number,message);
   % This is precisely the definition of rerror in RLISP.RED,
   % but redefining it here makes sure it is loaded,
   % and also avoids the need to save it.
   rederr message;

symbolic procedure soft!-rerror(packagename,number,message);
   << soft!-rerror!-number := number; error1() >>;

% Cannot use ERROR to return an error number, because in some cases
% e.g. SOLVE errors 5 and 6, there are 2 levels of errorset,
% and the outer level replaces the error number by 0.

% Note that to simply turn off printing of error messages by rederr
% and hence rerror the following suffices:
% !*protfg := nil;  % this is global
% without the need to redefine rerror, but the errorset would still
% be necessary to catch the error return.


% Algebraic mode list predicate:
% =============================

symbolic procedure evallistp u;
   eqcar(u, 'list);

put('listp, 'boolfn, 'evallistp);
flag('(listp), 'boolean);

end;


% SOTFSOLV.TST  F.J.Wright, FJW@Maths.QMW.AC.UK, London  13/3/92

% Test a few of the errors that are caught by soft_solve.
% =======================================================

% Assumes that you have already done either (a)

% on comp;  % not essential
% in "softsolv.red"

% or (b)

% faslout "softsolv";
% in "softsolv.red"
% faslend;
% and then
% load_package softsolv;

% or something equivalent for your system.


clear x;  on errcont;

% A solvable equation - no error:
solve(x**2 + x + 1, x);
soft_solve(x**2 + x + 1, x);

% No equations:
solve();
soft_solve();

% No variables:
solve(1);
soft_solve(1);

% Inconsistent equations in 1 variable:
solve({x=1,x=2}, x);
soft_solve({x=1,x=2}, x);

% Inconsistent equations in 2 variables:
solve({x+y=1,x+y=2}, {x,y});
soft_solve({x+y=1,x+y=2}, {x,y});

% Singular equations:
solve({x+y=1,x+y=1}, {x,y});
soft_solve({x+y=1,x+y=1}, {x,y});

off solvesingular;  % default is on

solve({x+y=1,x+y=1}, {x,y});
soft_solve({x+y=1,x+y=1}, {x,y});

% Use within an algebraic procedure:
algebraic procedure soft_solve_test(eqns, vars);
   begin scalar soln;
      soln := soft_solve(eqns, vars);
      if numberp soln then
         write "Equations not solvable"
      else return soln
   end;

on solvesingular;
soft_solve_test({x+y=1,x+y=1}, {x,y});
off solvesingular;
soft_solve_test({x+y=1,x+y=1}, {x,y});

end;
