%
%               SOLVEINX.RED for REDUCE 3.4 and REDUCE 3.4.1
%
% Adaption of SOLVE to allow inconsistent systems
% of linear equations to be treated.
%
% Authors: David Hartley and Robin W Tucker
%          School of Physics and Materials
%          Lancaster University
%          Lancaster    LA1 4YB
%          United Kingdom
%
%          email: D.Hartley@lancaster.ac.uk
%                 R.W.Tucker@lancaster.ac.uk
%
% Date:   14 September 1992
%

% ======================================================================


module solveinx;


% ----------------------------------------------------------------------


load!-package 'solve;    % needs solve package to run

fluid '(!*solvesingular !*solveinconsistent vars!*);

global '(inconsistent!*); % Flag to show if equations were inconsistent

flag('(inconsistent!*),'share);

switch solveinconsistent; % New switch

!*solveinconsistent := t; % Default value


% ----------------------------------------------------------------------


symbolic procedure solvelnrsys(u,v);
   % This is hook to general solve package. u is a list of polynomials
   % (s.f.'s) linear in the kernels of list v. Result is a tagged
   % standard form for the solutions.
   % If the system is inconsistent, then the second element of
   % the list returned is NIL, so that !*solvelist2solveeqlist
   % gives expr = 0, rather than var = expr.
   list list(car w,cadr w and v,1) where w = glnrsolve(u,v);


% ----------------------------------------------------------------------


symbolic procedure glnrsolve(u,v);

% glnrsolve(u: list of sf's, v: list of kernels)
% -> (xprs: list of sq's,
%     flg: boolean)
%
% Adapted from E Schruefer's glnrsolve.
% The equations u must be ordered with respect to the kernels v
% If flg is t then xprs is an ordered list of solutions for v.
% If flg is nil then xprs is a list of consistency conditions.
   begin scalar arbvars,sgn,x,y,cnds;
     inconsistent!* := nil;
     while u and null x do
       <<x := !*sf2ex(car u,v);
         if null x then u := cdr u
         else if inconsistency!-chk x then
           <<cnds := car u . cnds; x := nil; u := cdr u>>; >>;
     if null u then       % no consistent non-zero equations
       if cnds then (inconsistent!* := t) and   % inconsistent equations
         if null !*solveinconsistent then 
           rerror(solve,5,"SOLVE given inconsistent equations")
         else % bump up length of vars!* to fool !*solvelist2solveeqlist
              % in REDUCE 3.4.1.
           <<if getd 'not_imag_num and length cnds > 1
               then vars!* := t . vars!*;
             return list(for each j in cnds collect !*f2q j,nil)>>
       else               % all equations were zero
         return list(for each j in v collect !*f2q makearbcomplex(),t);
     u := cdr u;
     for each j in u do
       if y := extmult(!*sf2ex(j,v),x) then
          if inconsistency!-chk y then
            cnds := numr cancel(lc y ./ lc x) . cnds
          else
            x := y;
     if cnds then (inconsistent!* := t) and
       if null !*solveinconsistent then
         rerror(solve,5,"SOLVE given inconsistent equations")
       else % bump up length of vars!* to fool !*solvelist2solveeqlist
            % in REDUCE 3.4.1.
         <<if getd 'not_imag_num and length cnds > 1
             then vars!* := t . vars!*;
           return list(for each j in cnds collect !*f2q j,nil)>>;
     arbvars := for each j in setdiff(v,lpow x) collect
                    j . makearbcomplex();
     if arbvars and null !*solvesingular
       then rerror(solve,6,"SOLVE given singular equations");
     if null red x then return
        list(for each j in v collect
               if y := atsoc(j,arbvars) then !*f2q cdr y else nil ./ 1,
             t);
     sgn := evenp length lpow x;
     return list(for each j in v collect
                   if y := atsoc(j,arbvars) then !*f2q cdr y
                   else mkglsol(j,x,sgn := not sgn,arbvars),
                 t);
   end;


% ----------------------------------------------------------------------


endmodule;


end;


