#define IS_DEFINITION
#include "upshot.h"



/* UTILITY FUNCTIONS */

/*****************************************************************************/
/* ITOA: converts integers into their corresponding ascii values so that they*/
/*       may be used as strings.                                             */
/*****************************************************************************/
void itoa(i,s)
     int i;
     char s[];
{
  int j = 0; 
  int first,last,temp;

  do 
    {
      s[j++] = i %10 + '0'; 
    }
  while ((i /= 10) > 0); 
  s[j]='\0';
                                           /*reverses the strings*/
  for(first = 0, last = strlen(s)-1; first<last; first++,last--)
    temp=s[first], s[first]=s[last], s[last]=temp;

}

/*****************************************************************************/
/* TIME TO PIXELS:  function returns the pixel value of an event given its   */
/*                  time                                                     */
/*****************************************************************************/
#ifndef time_to_pixels
int time_to_pixels(time)
     unsigned long time;
{
  int x;
  x = ((time - left_time) / interval); 
  return x;
}
#endif

/****************************************************************************/
/* HELP MESSAGE */
/****************************************************************************/
void helpMessage()
{
  printf("upshot understands all standard Xt command line options.\n");
  printf("Additional options are as follows: \n\n");
  printf("OPTION         SPECIFICATION        VALID RANGE\n");
  printf("-l             log file name        string up to 250 charaters\n");
  printf("-logfile       log file name        string up to 250 charaters\n");
  printf("-s             state file name      string up to 250 charaters\n");
  printf("-statefile     state file name      string up to 250 charaters\n");
  printf("-ch            canvas height        480 to 1000\n");
  printf("-cheight       canvas height        480 to 1000\n");
  printf("-t             graph tasks          no value necessary\n");
  printf("-tasks         graph tasks          no value necessary\n");

  }

/****************************************************************************/
/* COMPUTE INTERVAL:  computes number of seconds per pixel;  may change     */
/*                    the value of the distance between markers             */
/****************************************************************************/
void compute_interval(pixmap_secs,interval)
     int pixmap_secs;
     float *interval;
{
  *interval = pixmap_secs /pixmapwidth;
}

/****************************************************************************/
/* COMPUTE INCRMNT: obtains most appropriate number of time line divisions  */
/****************************************************************************/
void compute_incrmnt(canvas_secs,sec_incrmnt)
     int canvas_secs;
     int *sec_incrmnt;
{
  int choices[15];      /*array containing increment possibilities */
  int i,j;
  int loop;

  choices[0]=5;  
  choices[1]=10;
  choices[2]=20;
  choices[3]=50;
  choices[4]=100;
  choices[5]=500;
  choices[6]=1000;
  choices[7]=2000;
  choices[8]=5000;
  choices[9]=10000;
  choices[10]=50000;
  choices[11]=100000;
  choices[12]=500000;
  choices[13]=1000000;
  choices[14]=5000000;

  /** Search through choices array for amount that produces less than
      ten time line divisions **/

 *sec_incrmnt = 0;
  loop = FALSE;
  i = 0;
  j = 1;
  while (i < 15)
    {  
      /*check bottom case possibility*/
      if((canvas_secs/(choices[14]*j)) > 10)
	j++;
      
      else   /* must produce between 2 and 10 time line divisions */
	if(((canvas_secs/(choices[i]*j)) > 2) && 
	   ((canvas_secs/(choices[i]*j)) < 11))  
	  {
	    *sec_incrmnt = choices[i]*j;
	    i = 15;
	  }
	else
	  {
	    i++;
	    if((i == 15) && (*sec_incrmnt == 0))  /* assignment not made yet */
	      {
		i = 0;
		j++;
	      }
	  }
    }

}

/*****************************************************************************/
/* GET PIX INCRMNT: obtains the number of pixels between markers             */
/*****************************************************************************/
void get_pix_incrmnt(sec_incrmnt,interval,pix_incrmnt)
     int sec_incrmnt;
     float interval;
     int *pix_incrmnt;
{
  *pix_incrmnt = sec_incrmnt/interval;
}


/****************************************************************************/
/* REARRANGE UNITS: assigns the time line either micro, milli or simply     */
/*                  seconds according to the total number of microseconds   */
/*                  being displayed and converts the screens begininng time */
/*                  to the appropriate time unit                            */
/*                  The key here is that time is always kept in microsecs   */
/*                  so that when canvas_secs and pixmap_secs are referenced */
/*                  there's no ambiguity as to the amount of time           */
/****************************************************************************/
void rearrange_units()
{

  /*NOTE: units_name,marker_tag,begin_label,canvas_secs,sec_incrmnt are all 
    globaly defined */

  if (canvas_secs < 10000)      /*if scale is microseconds*/
    {
      strcpy(units_name,"Microseconds");
      marker_tag = sec_incrmnt;
    }
  else if((canvas_secs >= 10000) && (canvas_secs < 10000000))
    {                                      /*if scale is milliseconds*/
      strcpy(units_name,"Milliseconds");
      marker_tag = (sec_incrmnt / 1000);
      begin_label = (begin_label / 1000);
    }
  else if(canvas_secs >= 10000000)        /*if scale is seconds*/
    {
      strcpy(units_name,"Seconds");
      marker_tag = (sec_incrmnt / 1000000);
      begin_label = (begin_label / 1000000);
    }


}

/*****************************************************************************/
/* FIND PROC LINE: function returns the pixel value of the corresponding     */
/*                 process id                                                */
/*****************************************************************************/
int find_proc_line(id)
     int id;
{
  int y,j,done;

  j = 0;
  done = FALSE;
  while((j < procTaskTotal) && (!done))
    {
      if (id == procTable[j].id)
	{ 
	  y = procTable[j].proc_line;               /*y coordinate of box*/
	  done = TRUE;
	}
      else
	j++;
    }

  return y;
}

/*****************************************************************************/
/* FIND_PROC_ID:  function returns proc/task id corresponding to the given   */
/*                line                                                       */
/*****************************************************************************/
int find_proc_id(line)
     int line;
{
  int i,done,proc_line;
  int id = -1;           

  i = 0;
  done = FALSE;
  
  while((i < procTaskTotal) && (!done))
    {
      proc_line = procTable[i].proc_line;
      
      if((line > proc_line - EVBOX_WIDTH/2) && 
	 ( line < proc_line + EVBOX_WIDTH/2))
	{
	  id = procTable[i].id;
	  done = TRUE;
	}
      else
	i++;
    }

  return(id);  /*if id == -1 it means the line didn't match any of the procs*/
}

/****************************************************************************/
/* FREE STATE QUEUE: puts the state queue (except for the head) in the      *
/*                   available queue so that the nodes may be reused        */
/****************************************************************************/
void FreeStateQueue()
{
  struct stateNode *tail;  /* will point at the tail of the states queue */

  tail = states->next;

  while(tail->next != NULL)
    tail = tail->next;     

  tail->next = available;
  available = states->next;

}

/****************************************************************************/
/* GET STATE NODE: is a function that manages the creation of state nodes by*/
/*                 maintaining a queue of available, unsused nodes.  Every  */
/*                 time a state-node is requested it will check the queue   */
/*                 for an available node first and return a pointer to it,  */
/*                 or create a new one in case the queue is empty.          */
/*                 In addition it will fill the node in with the given data */
/*                 so it is ready to use upon returning...                  */
/****************************************************************************/
struct stateNode *GetStateNode(start,end,color,name)
     int start,end;
     char color[],name[];
{
  struct stateNode *nextNode;

  if(available == NULL)   /* checks if queue is empty */
    nextNode = ((struct stateNode *) malloc(sizeof(struct stateNode)));

  else
    {
      nextNode = available;
      available = nextNode->next;
    }


  /* fill the node with given data */
  nextNode->start = start;
  nextNode->end = end;
  nextNode->color = ConvertColor(color);
  strcpy(nextNode->name,name);

  nextNode->next = NULL;

  return(nextNode);

}











/*****************************************************************************/
/* EVENT FORMAT DATA:  creates an event-format-data node, fills it in with   */
/*                     the given data (event & format) and returns the filled*/
/*                     node to the calling routine                           */
/*****************************************************************************/
struct eventFormatData *GetEventFormatNode(event,format)
     int event;
     char format[];
{
  struct eventFormatData *node;

  node = (struct eventFormatData *) malloc(sizeof(struct eventFormatData));
  node->event = event;
  strcpy(node->format,format);
  node->next = NULL;

  return(node);
}


/* FILE READING ROUTINES */

/****************************************************************************/
/* READ LOG HEADER:  reads in the header information to initialize globals  */
/*                   and set up for the graphics initialization             */
/****************************************************************************/
int ReadLogHeader(fname,keyString)
     char fname[],keyString[];  /*keyString is used to create the event key */
{
  FILE *seeker;     /* logfile pointer */
  long position;    /* tells the byte position of seeker from the begining 
		       of file*/
  
  int i;
  int event,proc_id,task,i_data,cycle;
  unsigned long time;  
  char c_data[BUFLEN],num[6];
  int finished = 0;                 /*flag used to mark end of process */
  
  struct eventFormatData *formatNode,*formatLast; /*used in event format link*/


  /* Initialize parameters to -1 to make sure they receive accurate values
     from the file's header */

  entries =  procTaskTotal = start_time = -1;
  event_types = 0;
  header = 0;
  strcpy(keyString,"EVENT KEY\n\n");

  /* Begin event data format link by creating a header node and attaching it
     to the dataFormat pointer.  This node will then be initialized to be
     ready for use */

  dataFormat = GetEventFormatNode(-1," ");
  formatLast = dataFormat;

  if(!logfileP && (logfileP = fopen(fname,"r")) == NULL)
    {
      printf("ERROR: Cannot open logfile -->` %s ' \n",fname);
       return(FALSE);
    }

  else
    {
      while(!finished)
	{
	  /** READ DATA FROM LOG FILE: numeric vals are read in first 
	    and then the string data is read with fgets **/
	  
	  if(fscanf(logfileP,"%d %d %d %d %lu %lu",&event,&proc_id,&task,&i_data,
		    &cycle,&time) == EOF)
	    finished = TRUE;
	  else    
	    if(fgets(c_data,BUFLEN,logfileP) == NULL)  /* read string in */
	      finished = TRUE;

	  if(event > 0)        /* stop once a non header entry is found */
	    finished = TRUE;
	  else
	    {
	      header = ftell(logfileP);  /* keeps track of the file stream, its 
					  last value will indicate the bytes 
					  covered by the file header */

	      switch(event)
	      {
	      case -1:     /*string data for the Creator and Date,*/
		break;     /*currently not used by Upshot*/

	      case -2:
		entries = i_data;
		break;

	      case -3:
		if(!data.tasks)             /* records process total only */
		  procTaskTotal = i_data;
		break;

	      case -4:    
		if(data.tasks)              /* records tasks total only */
		  procTaskTotal = i_data;
		break;     

	      case -5:
		event_types = i_data;
		break;

	      case -6:  
		start_time = time;   /* start time of first buffer */
		fileBeginTime = time;  /* records file initial time */

		/*
		  if(start_time < 9999999)
		  timefactor = 1000;
		  */
		break; 

	      case -7:     /* long data gives time of last event in the file*/
		fileEndTime = time;
		break;     

	      case -8:     /*integer data gives total number of clock cycles*/ 
		break;     /*currently not used by Upshot: assumes  1 cycle*/

	      case -9:     
		/* Construct event key entry and add it to the keyString */
		itoa(i_data,num);
		strcat(keyString,num);
		strcat(keyString,":");
		strcat(keyString,c_data);
		strcat(keyString,"\n");

		break;     
	      
	      case -10:
		if(strchr(c_data,'%') != NULL)
		  {
		    /* enter event print format in data format link */
		    formatNode = GetEventFormatNode(i_data,c_data);
		
		    formatLast->next = formatNode;
		    formatLast = formatNode;
		  }
		break;

	      case -11:
		cycle_time = time;
		if(start_time == -1) /* makes sure start time has been read */
		  {
		    printf("ERROR: Starting time must be logged before  \n");
		    printf("       cycle time for accurate processing...\n");
		    return(FALSE);
		  }

		else
		  time_gap = cycle_time - start_time;

		break;

		case -13:
		    /* State definition */
		    StateAddDefinitionFromEvent( task, i_data, c_data );
		break;

	      } /* switch */

	    } /* else */

	} /* while */

      return(TRUE);

    } /* else */


}

/****************************************************************************/
/* FILL PROC TABLE:  reads the logfile to obtain the proc/task ids.  It then*/
/*                  assigns one line per processor/task                     */ 
/****************************************************************************/ 
void fill_procTable()
{
  int done = FALSE, recorded = FALSE;
  int i,j = 0;
  int linegap;   /* distance between processor/task lines */

  for(j=0;j < PROCESS_MAX; j++)   /* initialization */
    procTable[j].id = -1;

  linegap = (canvas_height - 25)/procTaskTotal;

  /** RECORD ENTRY IN PROC TABLE: contains process number and its 
    corresponding Y-coordinate value on the canvas. 
    The reference array takes into consideration that
    proc ids might not be in numerical order or start
    at 0 or 1 **/

  i = -1;
  while(!done)
    {
      ++i;
      j = 0;
      recorded = FALSE;
      while((!recorded) && (j < procTaskTotal))
	{
	  if(procTable[j].id == buffer[i].id)
	    recorded = TRUE;

	  else
	    if(procTable[j].id == -1) 
	      {
		procTable[j].id = buffer[i].id;
		if(j == 0)
		  procTable[j].proc_line = EVBOX_WIDTH;
		else
		  procTable[j].proc_line = procTable[j-1].proc_line + linegap;
		                           
		recorded = TRUE;
	      }
	  j++;
      
	}  /* while */
      
      if(procTable[procTaskTotal -1].id != -1)
	done = TRUE;
  
    } /* 2nd while */

}

/****************************************************************************/
/* READ PG: reads up to EVENT MAX number of events into the buffer          */
/****************************************************************************/
void ReadPg(fname,pgPtr)
     struct page *pgPtr;    /* points to page to be loaded in buffer */
     char fname[];
{
  FILE *fd;
  long position;    /* tells the position of fd from the begining of file*/
  struct page *pgNeighbor;

  int i;
  int event,proc_id,task,i_data,cycle;
  unsigned long time;  
  char buf[BUFLEN];
  char c_data[CDATALEN];
  int finished = FALSE;
  long aproxLow;
  char ct[80];        /* used in throw away read */
  register struct eventData *ev;

  /* Initialize buffer entries */
#ifdef FOO
  /* All of these will be overwritten by the reading of the file */
  for(i = 0; i < EVENT_MAX; i++)
    {
      buffer[i].event     = -1;
      buffer[i].id        = -1;
      buffer[i].time      = -1;
      buffer[i].i_data    = -1;
      buffer[i].c_data[0] = 0;
    }
#endif


  if (!logfileP) {
      fprintf( stderr, "Logfile not open\n" );
      exit(0);
      }
  if(pgPtr->low < 0)   /* means pg hasn't been initialized */
    {
      pgNeighbor = pgPtr;
      --pgNeighbor;

      /* recursive call to find nearest previously initialized neighbor.  Once
	 found initalize all neighbors in between.  The pgPtr can then define
	 its low mark to be the previous pg's high mark */

      if(pgNeighbor->low < 0)   
	ReadPg(fname,pgNeighbor);

      pgPtr->low = pgNeighbor->high;
    }

  fseek(logfileP,pgPtr->low,0);  /*positions file pointer at next event to be read*/

  i = 0;
  while(i < EVENT_MAX)
    {
      ev = buffer + i;
#ifndef FOO
    if (ReadRecord( logfileP, 
		   &event, &proc_id, &task, &i_data, &cycle, &time, ev->c_data ) ==
	EOF) break;
#else
      if(fscanf(logfileP,"%d %d %d %d %lu %lu",&event,&proc_id,&task,&i_data,
		&cycle,&time) == EOF)
	  break;         /* No record to process */
      else    
        {
	  if(fgets(buf,BUFLEN,logfileP) == NULL)  /* read string in */
	    finished = TRUE;  /* Really a no-op, since a null string is a
				 legit value */
        }
      /* if (finished) break; */
#endif
      /* position = ftell(logfileP); */
     
      if(i == 0)
	start_time = time;     /* initializes buffer's start time */

      /** PLACE LOGFILE DATA IN BUFFER **/
      ev->event = event;

      if(data.tasks)
	ev->id = task;
      else
	ev->id = proc_id;

      ev->i_data    = i_data;
      ev->state_off = 0;

#ifdef FOO
      strncpy(ev->c_data, buf, CDATALEN);
      ev->c_data[CDATALEN - 1] = '\0';
#endif

      /** Adjust time for events that may have happened after
	the clock cycle rolled over **/
      if (cycle > 0)                         /*can only handle 1 clock cycle */
	ev->time = (time + time_gap);  /* (might need timefactor) */
      
      else                                   /*clock cycle hasn't rolled over*/
	ev->time = (time - start_time); /* (might need timefactor) */

      i++;

    } /* while */

  end_time = buffer[i-1].time;   /* initializes buffer's ending time */

  /* Position is either EOF (since failed on EOF read) or current location
     (since failed on i < EVENT_MAX) */
  position = ftell( logfileP);
  pgPtr->high = position;
  pgPtr->eventTotal = i - 1;


}

/****************************************************************************/
/* READ LOGFILE: carries out the first reading of the logfile; creates and  */
/*               initializes the the number of pages required.              */
/****************************************************************************/
void ReadLogfile(fname)
     char fname[];         /* logfile's name */
{
  struct page *pgPtr;
  int i;

    
  /* determine total number or pages */
  if((entries %  EVENT_MAX) == 0)
    pgTotal = entries / EVENT_MAX;
  else
    pgTotal = (entries / EVENT_MAX) + 1;

  /* allocate & initialize pages */
  firstPg = ((struct page *) malloc(sizeof(struct page) * pgTotal));
  pgPtr = firstPg;

  i = 0;
  while(i < pgTotal)
    {
      pgPtr->number = i;
      pgPtr->low = -1;
      pgPtr->high = -1;
      pgPtr->eventTotal = -1;

      pgPtr++;
      i++;
    }

  firstPg->low = header;
  ReadPg(fname,firstPg);

/* make a correspondence between proc/task ids and their line coordinates */
  fill_procTable();

}

/****************************************************************************/
/* READ STATE FILE: obtains definition of the different states from the user*/
/*                  and places it in the state_ref array and the states     */
/*                  structure                                               */
/****************************************************************************/
void ReadStatefile(fname)
     char fname[];
{
  int i;
  char name[50],color[50],message[350];
  int id,start,end;
  FILE *fp;
  struct stateNode *nextNode, *last;


  if((fp = fopen(fname,"r")) == NULL)
    {
      sprintf(message,"ERROR: cannot open state file `%s'",fname);
      ErrorBox(message);
      StatefileLoaded = FALSE;
    }
  else
    {

      for(i = 0; i < MAX_EVTYPES; i++)    /* fill state reference array */
	state_reference[i] = -1;
      initState_ref = TRUE;


      last = states;
      i = 1;
      while(fscanf(fp,"%d %d %d %s %s\n",&id,&start,&end,color,name) != EOF)
	{

	  nextNode = GetStateNode(start,end,color,name); 
	                      /* This is a function that manages state nodes.
				 It returns the next node available or creates
				 a new one if necessary */

	  last->next = nextNode;  /* hook it up to the states queue */
	  last = nextNode;

	  /** place the corresponding state events in state_reference array
	    for use in accessing state info later on **/

	  state_reference[start]=i; /* note that the relationship is made by */
	  state_reference[end] = i; /* associating the vector's index to the */
                                    /* event# and the entry to the corresp.  */
                                    /* state id */

	  histohelp[i].start = start;  /* The histohelp array of structures */
	  histohelp[i].end = end;      /* is needed for Histogram().  DPN   */
	  strcpy(histohelp[i].color, color);

	  i++;

	}

      StatefileLoaded = TRUE; /* indicates file has been read in successfully
				 and needs not be read in again */
      DefineStateKey(i-1);
      fclose(fp);
    }

}




















/* PLOTING ROUTINES */

/*****************************************************************************/
/* PLOT BOXES: pins event description boxes on corresponding processor line  */
/*             by searching through the log table for all those events that  */
/*             fall within the period of time being displayed.               */
/*****************************************************************************/
void plot_boxes()
{
  int i;                   /*array index of an event to be printed to screen*/
  int x,y;                 /* x and y coordinates of the box to be printed*/
  int j,done;
 

  /*Set the font for the little box numbers and plot them */
  change_font(smallFontId);  

  for(i = next_event; i<last_displayed; i++)
    {
      y = find_proc_line(buffer[i].id);
      x = time_to_pixels(buffer[i].time) + 8;
      draw_box(buffer[i].event,x,y);     /*for 1st event box to be seen */

    }
  change_font(bigFontId);       
}
#ifdef FOO
/*****************************************************************************/
/* PLOT STATES: searches the log table for events and keeps track of them in */
/*              an array that records the development of a state and sends it*/
/*              to be graphed (by draw_state) once it has been completed.    */
/*              The array is used as sort of checklist together with the     */
/*              information provided by each state info structure            */
/*                                                                           */
/* In the loop each event undergoes the following process:                   */
/*                                                                           */
/*     1) It is read from the buffer                                         */
/*     2) next it is identified with a state (this info is in state_reference*/
/*        array)                                                             */
/*     3) the corresponding state (pointed by state_info) is accessed for    */
/*        reference in the states queue (ie to check whether the event is    */
/*	  of start type or end type)                                         */
/*     4) the events existence is then recorded in the state_check table     */
/*        at the corresponding proccessor number which in this case matches  */
/*        the table's index                                                  */
/*                                                                           */
/*     Once an entry in the state_check table contains two events            */
/*     and a legal time entry it is sent to be graphed by draw_state.        */
/*                                                                           */
/*     Thus reading in a event should result in establishing a pointer to    */
/*     its corresponding state and updating the state_check table (keeps     */
/*     track of states by processor)                                         */
/*****************************************************************************/
void plot_states()  
     
{
  int i,j,k;
  int startX,endX,y;            /* x and y coordinates of state bar */
  int event,state_number,id;
  struct stateNode *state;       /*pointer to state structure being examined*/
  int pr = FALSE;

  if(!colorScreen)
    XSetFillStyle(display,gc,FillTiled);

/* NOTE: "next_event" is the buffer's index of next event to be referenced, 
   it indicates where to start checking for states specially helpful when a 
   shift draw has occured */

  for(i = next_event; i<= last_displayed; i++)
    {
      /* (1) */
      event = buffer[i].event;

      if (event < MAX_EVTYPES)
	{
	  /* (2) */
	  state_number = state_reference[event];

	  /* (3) */
	  state = states;       /* points at the header node */
	  if (state_number > 0)
	    {
	      for(k = 1; k <= state_number; k++)
		state = state->next;

	      /* (4) */
	      id = buffer[i].id;

	      if(state_check[id].event == -1)  /* NO EVENTS PRESENT */
		{
		  if(event == state->start)  /* if starting event record it */
		    {
		      state_check[id].event = event;
		      state_check[id].time = buffer[i].time;
		    }


		  else   /* must be an end event without a matching start*/
		    {
		      startX = time_to_pixels(left_time) + 8;
		      endX = time_to_pixels(buffer[i].time) + 8;
		      y = find_proc_line(buffer[i].id);
		      draw_state(startX,endX,y,state->color,state_number,0);
 		    }
		}

	   
	      else   /* EVENT ALREADY PRESENT */
		if(state_check[id].event > -1)
		  {
		    if(event == state->end)
		      {                   
			startX = time_to_pixels(state_check[id].time)+8;
			endX = time_to_pixels(buffer[i].time)+8;
			y = find_proc_line(buffer[i].id);
			draw_state(startX,endX,y,state->color,state_number,0);

			state_check[id].event = -1; 
			state_check[id].time = -1;   

			/*clear mem only if state fully contained in buffer*
		        if(state_check[id].time > left_time)
			  {
			    state_check[id].event = -1; 
			    state_check[id].time = -1;   
			  }
			else * record the ending event and its time *
			  {
			    state_check[id].event = event;
			    state_check[id].time = buffer[i].time;
			  }
			*/

		      }
		    else  /* must be a strayed start event */
		      /* makes present event the start-event */
		      state_check[id].time = buffer[i].time;
		  }
	      
      
	      /* Notice that an end-state will be recorded in the state_check 
		 array only if a start-state has preceeded it */
	    }
	}
    }



  /* To accomodate for those states whose end-event or start-event does not 
     fall within the current time span, an final check is made for those 
     processors that may have pending states to display */


  for(i = 0; i < PROCESS_MAX; i++)
    if(state_check[i].event > -1)
      {
	event = state_check[i].event;
	state_number = state_reference[event];
	state = states;
	if (state_number > 0)
	  {
	    for(k = 1; k <= state_number; k++)
	      state = state->next;


	    if (event == state->start)  /* starting event w/out matching end */
	      {
		startX = time_to_pixels(state_check[i].time)+8;
		endX = time_to_pixels(right_time)+8;
	      }


	    else   /* must be an ending event w/out a matching start*/
	      {
		startX = time_to_pixels(left_time)+8;
		endX = time_to_pixels(state_check[i].time)+8;
	      }

	    y = find_proc_line(i);

	    draw_state(startX,endX,y,state->color,state_number,0);
/*
printf("%d) %d %lu\n",i,state_check[i].event,state_check[i].time);
*/
	  }
      }
/*
printf("------------------------------\n");
*/

  if(!colorScreen)
    XSetFillStyle(display,gc,FillSolid);

  ShowStateKey();

}

#endif















/* DRAWING ROUTINES */

/*****************************************************************************/
/* DRAW FRAME: draws x&y axis and labels process lines                       */
/*****************************************************************************/
void draw_frame()
{
  int start_x,start_y;
  int end_x,end_y;

  int line_gap;           /*pixel distance between processes lines*/
  char s[MAX_STR_LEN];    /*string used to hold the process's number*/
  int i,j;
  int y;                  /*y-coordinate of the process line*/
  int len;                /*length of the string to be drawn */ 
  ApplicationData data;

  /* X axis */
  start_y = end_y = (canvas_height - Y_OFFSET);
  start_x = X_OFFSET;
  end_x = (CANVAS_WIDTH - X_OFFSET);
  drawline(w,start_x,start_y,end_x,end_y,XOR,BLACK);

  /* draw Y axis */
  start_y = 0;
  end_y = (canvas_height - Y_OFFSET);
  start_x = end_x = X_OFFSET;
  drawline(w,start_x,start_y,end_x,end_y,XOR,BLACK);

/*
  for(i = 0; i< procTaskTotal;i++) 
    { 
      itoa(procTable[i].id,s);    convert process-id to a string 
      len = strlen(s);

      drawstring(w,0,(procTable[i].proc_line+5),s,len);   
 
    }
*/
}


/*****************************************************************************/
/* DRAW PROCESS LINES:  divides the window into the apropriate no of         */
/*                      processor lines and labels the Y axis                */
/*****************************************************************************/
void draw_process_lines()
{
  int i;
  int graphEnd;         /* the right_time value in pixels */

  graphEnd = time_to_pixels(right_time);

  for(i = 0; i< procTaskTotal;i++) 
    {
      drawline(canvas_pixmap,0,procTable[i].proc_line,graphEnd,
	       procTable[i].proc_line,XOR,BLACK);
/*
      if(drawMini)
	drawline(miniW,0,i*MINI_PROCWIDTH+5,CANVAS_WIDTH,i*MINI_PROCWIDTH+5,
		 XOR,BLACK);
*/
    }

  endLine();
}

/***************************************************************************/
/* DRAW_SCALE: draws the x-axis and its scale                               */
/***************************************************************************/
 void draw_scale(begin_label,interval,marker_tag,pix_incrmnt,units_name)
     int begin_label;               /*start time in correct time unit*/
     float interval;                 /*number of seconds per pixel */
     int marker_tag;                 /* the present time units tag */
     int pix_incrmnt;                /*number of pixels between markers*/
     char units_name[];              /*string holding the unit of time */

{ 
  int start_x,end_x,y;               /*x axis line coordinates*/
  char s[2];                         /*string holding the slash ("|")*/
  char  marker[10];                 /*contains names of the time line markers*/
  int stringlength, stringlength2; /*length of strings*/
  int i,j;
  int graphEnd;                     /* the equivalent of right_time in pixels*/
  
  start_x = 0;   
  end_x = pixmapwidth;
  y = canvas_height - Y_OFFSET;

  drawline(canvas_pixmap,start_x,y,end_x,y,XOR,BLACK); /* pixmap's X axis */

  strcpy(s,"|");
  stringlength = strlen(s);

  j = begin_label;        /*initializes j to first time to appear on screen*/

  graphEnd = time_to_pixels(right_time);
  for(i = 8;i<= (graphEnd + 2); i=(i+pix_incrmnt))
    {
      itoa(j,marker);                  /*convert numeric time to a string*/
      stringlength2=strlen(marker);    /*get the lenght of the time string*/
      
      /* i is the distance from the y axis where the label and its value will
	 be placed */
      drawstring(canvas_pixmap,i,canvas_height-2,marker,stringlength2);
      drawstring(canvas_pixmap,i,(canvas_height-Y_OFFSET),s,stringlength);
      j = j + marker_tag;     /*j equals the numeric value of the next time*/
    }

}

/*****************************************************************************/
/* ACQUIRE BNDRY DATA:  performs a backwards search of the buffer (starting  */
/*                     with next_event to be graphed) for possible unresolved*/
/*                     states.  It searches for starting events whose time   */
/*                     stamp is less than left time.                         */
/*****************************************************************************/
acquireBndryData()

{
  int i,k;
  int found,index;
  int event,state_number,id;
  struct stateNode *state;


  /* first flush the state_check array */
  for (i = 0; i < PROCESS_MAX; i++)
    {
      state_check[i].event = -1;
      state_check[i].time = -1;
    }

  for(i = 0; i < PROCESS_MAX; i++)
    {
      if((id = procTable[i].id) > -1)
	{
	  found = FALSE;
	  index = (next_event - 1);
	  while((index >= 0) && (!found))
	    {
	      if(buffer[index].id == id)
		{
		  found = TRUE;
		  event = buffer[index].event;/*read event number from buffer*/

		  if(event < MAX_EVTYPES)
		    {
		      /* find what state the event belongs to */
		      state_number = state_reference[event];

		      state = states;  /* state now points to the state link */
		      if(state_number > 0) 
			{
			  /* arrange for state to point to correct state node*/
			  for(k = 1; k <= state_number; k++)
			    state = state->next;    
			
			  if(event == state->start) 
			    {
			      /* record only starting events */
			      state_check[id].event = event;
			      state_check[id].time = 100;
			    }
			}
		    }
		}

	      index--;

	    }  /* end of while */
	  
	  if(!found) /*no events were found for the process/task */
	    {
	      /* there is a possibility that the last event for the process or
		 task at hand is contained in a previous page.  If this is the
		 case its existence would have been recorded in the bndry data
		 contained in bndry_check */

	      if(bndry_check[id].event > -1)
		{
		  state_check[id].event = bndry_check[id].event;
		  state_check[id].time = bndry_check[id].time;
		}
	    }

	}  /* end of if(id .... > -1) */

    } /* end of for */

}

/*****************************************************************************/
/* NEXT DRAW:  determines the chunck of the buffer that will be graphed      */
/*             next and flushes the state_check and the bndry stack when     */
/*             necessary, it then calls the plotting routines                */
/*****************************************************************************/
void next_draw(left_time,checkLeft)
     unsigned long left_time; /*time identified with the left end of pixmap*/
     int checkLeft;           /* indicates whether to obtain the graph's left
				 boundary or not */
{
  int i;
  int done = FALSE;
  struct bndryEntry *ptr;
  int safePixmapwidth;


  /* the very first thing is to check if the canvas is pixmap is too big
     for the given time span... if it is then arrange for the pixmapwidth
     to be shorten by half.  The pixmapwidth will be reset to its real
     value after the draw*/
/*
  safePixmapwidth = pixmapwidth;
  while(canvas_secs < CANVAS_WIDTH)
    pixmapwidth = pixmapwidth /2;
*/


  /* given left time compute RIGHTmost TIME of the graph */

  if((left_time + pixmap_secs) <= end_time) /*left_time limit is end_time*/
    right_time = pixmap_secs + left_time;
  else
    right_time = end_time;
/*
printf("%d <- -> %d\n",left_time,right_time);
*/

  /*find first and last events that belong is present time span*/
  i = 0;
  while(!done)
    if (buffer[i].time >= left_time)
      {
	next_event = i;      /* buffer's index of next event to be displayed */
	    done = TRUE;
      }
    else
      i++;

  i = next_event;
  done = FALSE;
  while((!done) && (buffer[i].time >= 0))
    {
      if (buffer[i].time >= right_time)
	{
	  last_displayed = i;
	  done = TRUE;
	}
      else
	{
	  i++;
	}
    }



  /*NOW, for the actual drawing*/
  draw_scale(begin_label,interval,marker_tag,pix_incrmnt,
	     units_name);

  draw_process_lines();

  if(draw_states)
    {
      if(checkLeft)
	{
	  if(left_time > 0) 
	    acquireBndryData(); /* searches for left boundary of the 
				   present time span */
	  
	  else        /* at begining of buffer need to get pg bndry data */
	    {
	      for(i = 0; i < PROCESS_MAX; i++)
		if(bndry_check[i].event > -1)
		  {
		    state_check[i] = bndry_check[i];
		  }
	    }

	}

      
      /* UPDATE time on state_check table */
      for(i = 0; i < PROCESS_MAX; i++)
	if(state_check[i].event > -1)
	  state_check[i].time = left_time;

      plot_states2();
    }

  if(draw_events)
    plot_boxes();
  
}






















/* SPECIALTY ROUTINES */

/****************************************************************************/
/* ZOOM PROC: arranges the scale and time line variables to readjust the    */
/*               present screen (uses global variables)                     */
/****************************************************************************/
void zoom_proc(Zoomout,zoomstep)
     int Zoomout;     /* boolean indicates whether zooming IN or OUT */
     int zoomstep;    /* magnitude of magnification */
{
  float percent = 0;

  draw_watch(TRUE);

  CopyPixmapToCanvas();    /* makes sure view time is defined */

  /* Determine new amount of seconds for canvas and pixmap*/
  if(Zoomout)
    {
      pixmap_secs = (pixmap_secs * zoomstep) + 100;
      bufferwidth = bufferwidth / zoomstep;
      zoomfactor = (zoomfactor * 1.0) / (zoomstep * 1.0);

      if (bufferwidth < pixmapwidth)
        bufferwidth = pixmapwidth;
    }

  else /*Zoomin*/
    {
      pixmap_secs = (pixmap_secs / zoomstep) + 100;
      bufferwidth = bufferwidth * zoomstep;
      zoomfactor = (zoomfactor * 1.0) * (zoomstep * 1.0);
    }

  canvas_secs = pixmap_secs / CANVAS_TOTAL;

  /* Free old pixmap and create and initialize new one */
  free_pixmap();
  set_up_pixmap();

  /* Get new pixel-sec associations and get ready for redrawing */
  compute_interval(pixmap_secs,&interval);
  compute_incrmnt(canvas_secs,&sec_incrmnt);

  get_pix_incrmnt(sec_incrmnt,interval,&pix_incrmnt);
  begin_label = view_time + (start_time - fileBeginTime);

  rearrange_units();

  /* in zooming we would like to start displaying the blown up (or reduced)
     picture at the portion of time that last appeared on the canvas */
  left_time = view_time;
  next_draw(left_time,FALSE);
  label_units();

  /* Copies stuff from the pixmap to the screen */
  percent = ((view_time * 1.0) / interval)/(bufferwidth * 1.0);
  XawScrollbarSetThumb(Cscroll,0.0,-1.0);
  CanvasScroll(Cscroll,NULL,NULL);

  draw_watch(FALSE);

}


/****************************************************************************/
/* RESET GRAPH: restores the graph to the original picture by acquiring the */
/*             correct zoomstep and calling zoomProc to arrange for a redraw*/
/****************************************************************************/
ResetGraph()
{
  int zoomout = TRUE, zoomin = FALSE;
  int zoomstep = 0;

  if(zoomfactor < 1)             /* nonlocal references to the zoomstep and */
    {                            /* zoomfactor globals */
      zoomstep = 1/zoomfactor; 
      zoom_proc(zoomin,zoomstep);
    }
  else
    if(zoomfactor > 1)
      {
	zoomstep = zoomfactor;
	zoom_proc(zoomout,zoomstep);
      }
  /* Nothing happens when zoomfactor == 1 */
}

/****************************************************************************/
/* GET PG BNDRY DATA:  obtains bndry data left on state check array from the*/
/*                     previous page and places it in the bndry_check array */
/****************************************************************************/
getPgBndryData()
{
  int i;

  for(i = 0; i < PROCESS_MAX; i++)   /* first flush bndry_check */
    {
      bndry_check[i].event = -1;
      bndry_check[i].time = -1;
    }
    

  for(i = 0; i < PROCESS_MAX; i++)
    if(state_check[i].event > -1)
      {
        bndry_check[i] = state_check[i];
      }
}


/****************************************************************************/
/* SHIFT PG:  organizes the loading of the next file's page into the buffer */
/****************************************************************************/
void ShiftPg(logfile,pgPtr)
     char logfile[];
     struct page *pgPtr;

{
  int i;
  
  draw_watch(TRUE);

  /* acquire bndry data left on state check table from previous pg (except for
     the first page, ofcourse!) and place it on the bndry_check array  */
  if(pgPtr->number > 0)
    getPgBndryData();

  /* flush state_check */
  for(i = 0; i < PROCESS_MAX; i++)
    {
      state_check[i].event = -1;
      state_check[i].time = -1;
    }

  /* flush buffer */
  for(i = 0; i< EVENT_MAX; i++)
    buffer[i].time = -1;  

  /* load desired page */
  ReadPg(logfile,pgPtr);

  if(end_time >= canvas_secs)  /*means the view has not been zoomed out too far
				drawing may proceed */
    {
      /* since the buffer's width in pixels changes with the amount of time 
	 covered in it it is necessary to recompute bufferwidth */

      bufferwidth = end_time / interval;

      /* setup pixmap for it*/
      free_pixmap();
      set_up_pixmap();

      /* Since the file's view has been shifted the begin_label (marks the 
	 graph's leftmost time mark) must be updated to reflect the shift.  
	 This is carried out by defining the new value of begin_label and 
	 calling rearrange units to put it in the correct time units ( seconds,
	 milliseconds, etc)*/

      begin_label = start_time - fileBeginTime;
      rearrange_units();

      left_time = 0;
      next_draw(left_time,TRUE);   /* TRUE indicates that the left bndry of the
				      graph should be obtained */

      CopyPixmapToCanvas();
    }
  
  else
    ErrorBox("ERROR:  view has been zoomed out too far!");

  draw_watch(FALSE);

}

/****************************************************************************/
/* SHIFT DRAWING:  shifts the view of the file either to the left or right  */
/*                 depending on the "leftshift" parameter.                  */
/*                 (REM:  begin_label keeps track of the graph's first event*/
/*                        time, while view_time & left_time are defined with*/
/*                        relation to the buffer's start_time)              */
/****************************************************************************/
void shiftDrawing(leftshift)
     int leftshift;      /*indicates whether to switch view to right or left*/

{
  int i;

  draw_watch(TRUE);

  /* Free old pixmap, create and initialize new one */
  free_pixmap();
  set_up_pixmap();

  if (leftshift)                  /* LEFT */
    {
      while(view_time < left_time)
	{
	  left_time = (left_time + canvas_secs) - pixmap_secs;
	  if(left_time < 0)
	    left_time = 0;
	}
      begin_label = left_time + (start_time - fileBeginTime);
      
    }

      
  else    /* moving the view to the RIGHT */
    {
      while(view_time > (left_time + pixmap_secs))
	left_time = left_time + (pixmap_secs - canvas_secs);

      begin_label = left_time + (start_time - fileBeginTime);
    }

  /* rearrange units is called with the sole purpose of changing the time 
     label to the present time units */
  rearrange_units();    
  next_draw(left_time,leftshift);

  draw_watch(FALSE);
}

/*****************************************************************************/
/* SET STATE DEF:  adds a state node to the states queue in order to define a*/
/*                 a new state                                               */
/*****************************************************************************/
void SetStateDef()
{
  int stateCounter,i;
  int start,end;
  char name[50],color[50];
  struct stateNode *nextNode,*tail;
  int allGood = TRUE;

  /* gets data from StateDef dialogs */
  CheckStateDef(&start,&end,color,name,&allGood);  

  if(allGood)     /* all acquired data is valid */
    {
      StateAddDefinition( start, end, color, name );
      ShowStateKey();
      /* Update the state info on the display */
      FindStateInfo();
      DisplayStates();
    }
}

/*****************************************************************************/
/*SHOW EVENT DATA: given aproximate x & y coordinates of an event box find   */
/*                its entry in the buffer and display its corresponding log  */
/*                data                                                       */
/*****************************************************************************/
ShowEventData(x,y,whichbutton)
     int x,y,whichbutton;
{
  unsigned long time = 0;
  int procTaskLine = 0;
  int id,i,index,done = FALSE;
  long view_span,realTime;
  int halfBoxSecs;
  char idData[3],eventData[4],timeData[15],cData[BUFLEN];
  struct eventFormatData *last;


  /* convert x coordinate to time equivalent */
  x = x - X_OFFSET - 8;          /*pixel shift due to space for Y-axis plus 8 
				   pixel offset to accomodate for event boxes*/

  time = x * interval;           /* time on canvas */

  CopyPixmapToCanvas();          /* make sure view_time is up to date */
  time = time + view_time;       /* time on buffer */


  /* determine which process/task the event belongs to */
  id = find_proc_id(y);

  /* view span determines the rightmost time seen on the screen */
  view_span = view_time + ((CANVAS_WIDTH - X_OFFSET + 5) * interval);
  halfBoxSecs = ((EVBOX_WIDTH/2) * interval);


  /* search the buffer for the entry that matches the given time and id most
     closely; begin search at first state graphed on pixmap */

  index = -1;
  i = next_event;  
  while((buffer[i].time < view_span) && (!done))
    { 
      if(buffer[i].id == id)
	{
	  if(time == buffer[i].time)
	    {
	      index = i;
	      done = TRUE;
	    }
	  else   /* coordinates correspond to time covered by box */
	    {
	      if((time > buffer[i].time - halfBoxSecs) &&  
		 (time < buffer[i].time + halfBoxSecs))
		index = i;
	    }
	}
      i++;
    }
  
  if(index > -1)  /* create event data box only if an event matching the give
		     coordinates was found */
    {
      /* convert event data to its ascii equivalent and send it to 
	 EventDataBox to be graphed.  Special steps need to be taken to
	 fit integer data into its corresponding event format */

      realTime = buffer[index].time + start_time - fileBeginTime;
      
      /* search event-data-format link for corresponding format */
      last = dataFormat->next;   /*remember first node in link is dummy node*/
      while(!done && (last != NULL))
	{
	  if(last->event == buffer[index].event)
	    done = TRUE;
	  else
	    last = last->next;
	}

      if(last != NULL)
	/* sprintf puts uses integer data along with the given format
	   to create the cData string.  If error occurs sprintf will 
	   return a negative number, otherwise positive */

	sprintf(cData,last->format,buffer[index].i_data);

      else
	itoa(buffer[index].i_data,cData);

      strcat(cData,buffer[index].c_data);
      EventDataBox(x,y,buffer[index].id,buffer[index].event,realTime,
		   cData,data.tasks,whichbutton);
	

    }
}


/*****************************************************************************/
/* START_PROC: initializes and directs the data display process              */
/*****************************************************************************/
void start_proc()
{
  int i;

  get_pix_incrmnt(sec_incrmnt,interval,&pix_incrmnt);
  begin_label = 0;      /* begin_label defined to be used by rearrange_units */
  rearrange_units();

  /* Initialize state_check array  (in case states will be drawn) */
  for(i = 0; i < PROCESS_MAX; i++)
    {
      state_check[i].event = -1;
      state_check[i].time = -1;
    }

  /* Draw graph unto pixmap */
  left_time = 0;

  drawMini = TRUE;
  next_draw(left_time,FALSE);
  drawMini = FALSE;

  label_units();

  /* Copy graph from pixmap to the canvas */
  CopyPixmapToCanvas();

}








/****************************************************************************/
/*                          UPSHOT MAIN PROGRAM                             */
/****************************************************************************/ 
main(argc, argv)
     int argc;
     char *argv[];
{
  int j = 0,k,l;
  char keyString[200];
  int frame_depth;

  draw_states = FALSE;  /* initialization of important globals */
  draw_events = TRUE;
  initState_ref = FALSE;
  StatefileLoaded = FALSE;
  available = NULL;
  canvas_height = 1;   
  start_time = 0;
  end_time = 0;
  zoomfactor = 1.000;
  logfileP   = 0;
  num_children = 0;  /* initially how many child processes upshot has.  DPN */


  /** Initialize GRAPHICS SETUP:  sets up basic graphics components 
      including the application's shell and interpretation of graphics
      components in the command line **/
  init_graphics(argc,argv);
  
  /* exit with help message if it was specified in the command line */
  if(data.help)
    {
      helpMessage();
      exit(1);
    }


  /* READ LOGFILE */
  states = GetStateNode(-1,-1,"black","NULL"); /* initialize states queue*/
  if(!ReadLogHeader(data.logfile,keyString))
    {
      helpMessage();
      exit(1);
    }

  /* Setup the entry buffer and allocate space */
  nevents = EVENT_MAX;
  if (entries < nevents) nevents = entries;
  
  buffer = (struct eventData *)malloc( nevents * sizeof(struct eventData) );
  if (!buffer) {
      fprintf( stderr, "Could not allocate event buffer\n" );
      exit(0);
      }

  /** Define Canvas height to fit all processes/tasks **/
  canvas_height = (procTaskTotal+2) * (EVBOX_WIDTH + EVBOX_WIDTH/2); 
  if(canvas_height < data.canvas_height) /* min height has been predefined */ 
    canvas_height = data.canvas_height;

  ReadLogfile(data.logfile);
  
  /** Set up all drawing equipment:  defines remaining components of the 
      graphics setup **/
  set_up_equipment(data.logfile,data.statefile,keyString);
  change_font(bigFontId);

  draw_watch(TRUE);  /* change cursor shape to indicate processing */

  /* READ STATE FILE (if it applies) must be done after init_graphics so that
     the colormap may be referenced in determining the states color values */  

  /* states = GetStateNode(-1,-1,"black","NULL"); *//* initialize states queue*/
  if(strcmp(data.statefile,"") != 0)
    {
      ReadStatefile(data.statefile);

      if(StatefileLoaded)
	{
	  draw_states = TRUE;
	  draw_events = FALSE;
	}

      }
  else if (nstates > 1) {
      /* States defined in the log file */
      StateFromStateDef();
      draw_states = TRUE;
      draw_events = FALSE;
      }

  /** Establish the number of seconds visualized on the canvas **/
  view_time = 0;

  pixmap_secs = buffer[firstPg->eventTotal].time; /* change to end time */
  canvas_secs = pixmap_secs/CANVAS_TOTAL;

  create_pixmap();

  compute_interval(pixmap_secs,&interval);
  compute_incrmnt(canvas_secs,&sec_incrmnt);

  /* initilization of bufferwidth for the file's first page */
  bufferwidth = end_time/interval;

  /** Initial draw and get ready for user input **/
  draw_frame();

  start_proc();

  draw_watch(FALSE);

  main_loop();
}

#include <ctype.h>

/* ReadRecord - since we read LOTS of records, and we want to do it
   fast, this code reads them KNOWING the format */
int ReadRecord( fp, event, proc_id, task, i_data, cycle, time, c_data )
FILE          *fp;
int           *event, *proc_id, *task, *i_data;
unsigned long *cycle, *time;
char          *c_data; 
{
/* Format is %d %d %d %d %lu %lu %s\n */
int c, k, isign;
int l, v[4];
unsigned long ul, uv[2];
  /** Create and initialize pixmap **/
c = getc( fp );
for (k=0; k<4; k++) {
    while (c != EOF && isspace(c)) c = getc(fp);
    if (c == '-') {
	isign = 1;
	l     = 0;
	}
    else {
	isign = 0;
	l = c - '0'; 
	}
    c = getc(fp);
    /* u * 10 == (u * (8+2)) == (u*8 + u*2) == (u << 3) + (u << 1); this
       is faster on SPARC's without integer multiply */
    while (c != EOF && isdigit(c)) { l = l * 10 + (c - '0'); c = getc(fp); }
    v[k] = (isign) ? -l : l;
    }
for (k=0; k<2; k++) {
    while (c != EOF && isspace(c)) c = getc(fp);
    ul = c - '0'; 
    c  = getc(fp);
    while (c != EOF && isdigit(c)) { ul = ul * 10 + (c - '0'); c = getc(fp); }
    uv[k] = ul;
    }
if (c == EOF) return EOF;
*event   = v[0];
*proc_id = v[1];
*task    = v[2];
*i_data  = v[3];
*cycle   = uv[0];
*time    = uv[1];
/* Get the string.  Get only the leading characters */
while (c != EOF && c != '\n' && isspace(c)) c = getc(fp);
if (c == '\n') {
    c_data[0] = 0;
    return 1;
    }
/* There is a string to read.  The old code read the space as part of the
   data; we do that here for compatibility */
c_data[0] = ' ';
c_data[1] = (char) c;
for (k=2; k<12; k++) {
    c = getc(fp);
    if (c == EOF || c == '\n') break;
    c_data[k] = (char)c;
    }
c_data[k] = 0;
while (c != EOF && c != '\n') c = getc(fp);
return 1;
}

/* This routine adds a state to the state definition.  The values can come
   from either the state file, a user-definition, or the logfile */
StateAddDefinitionFromEvent( start, end, str )
int  start, end;
char *str;
{
char *color, *name;
int  stateCounter;

while (*str && *str == ' ') str++;
color = str;
while (*str && *str != ' ') str++;
*str++ = 0;
while (*str && *str == ' ') str++;
name   = str;

stateCounter = StateKeyAddToQueue( start, end, color, name );
}

StateAddDefinition( start, end, color, name )
int start, end;
char *color, *name;
{
int              stateCounter,i;
struct stateNode *nextNode, *tail;

stateCounter = StateKeyAddToQueue( start, end, color, name );

/* add the state's entry to the state key */
DefineStateKey(stateCounter+1);

/* Note that this does not set histohelp[] array! */
#ifdef SET_HIST
histohelp[stateCounter+1].start = start;  /* The histohelp array of structures */
histohelp[stateCounter+1].end = end;      /* is needed for Histogram().  DPN   */
strcpy(histohelp[stateCounter+1].color, color);
#endif      
}

int StateKeyAddToQueue( start, end, color, name )
int start, end;
char *color, *name;
{
struct stateNode *nextNode, *tail;
int              stateCounter, i;

nextNode = GetStateNode(start,end,color,name);

/* Hook node to the end of the states queue */
stateCounter = 0;
tail = states;
while(tail->next != NULL)
    {
    tail = tail->next;
    stateCounter++;
    }
nstates = stateCounter;

tail->next = nextNode;

if(!initState_ref) /*make sure the state_ref array has been initialized*/
    {
    for(i = 0; i < MAX_EVTYPES; i++)
	state_reference[i] = -1;
    initState_ref = TRUE;
    }

/* register state in the state_reference arrary */
state_reference[start] = stateCounter+1;
state_reference[end] = stateCounter+1;

return stateCounter;
}

StateFromStateDef()
{
struct stateNode *nextNode, *tail;
int              stateCounter, i;

stateCounter = 0;
tail = states;
while(tail->next != NULL)
    {
    tail = tail->next;
    stateCounter++;
    }
DefineStateKey(stateCounter);
StatefileLoaded = TRUE;
}
