/* main part of gs-preprocessor */

static struct parameter *PR = 0;		/* parameter structure */
static int n_PR = 0, max_PR = 0;

static struct place *PL = 0, *cur_PL;		/* place structure */
static int n_PL = 0, max_PL = 0;

static struct transition *TR = 0, *cur_TR;	/* transition structure */
static int n_TR = 0, max_TR = 0;

static struct result *RS = 0;			/* result struture */
static int n_RS = 0,  max_RS = 0;

static int error_cnt = 0;			/* error counter */

static int result_type;				/* result type */

#define EPS	    1e-12			/* for FP-zero tests */



main ()
{
    yyparse();					/* parse the input */
    check();					/* sanity check */

    if (error_cnt)
	exit(1);				/* exit in case of errors */

    build_place();				/* emitt simulator structures */
    build_trans();
    build_param();
    build_tfunc();
    build_rslts();
    build_ffunc();

    exit(0);					/* done! */
}

yyerror (s)					/* error printer		 */
    char *s;
{
    error_cnt++;
    fprintf (stderr, "%s, line %d: %s\n", src_file, yylineno, s);
}

struct parameter *find_param (s)		/* find a parameter */
    register char *s;
{
    register int i;
    register struct parameter *p;

    for (i = n_PR,  p = PR; i--; p++)
	if (!strcmp(s, p->name)) {
	    free(s);
	    return p;
	}

    if (n_PR <= max_PR) {			/* need more space */
	max_PR += 20 + max_PR / 2;
	if (PR)
	    PR = p = (struct parameter *) realloc (PR,  max_PR * sizeof(struct parameter));
	else
	    PR = p = (struct parameter *) malloc (max_PR * sizeof(struct parameter));
    }

    p += n_PR++;
    p->name = s;
    p->defined = 0;
    p->used = 0;
    p->max_used = 0;
    p->ind = 0;

    return p;
}

add_param (s, v)				/* got a new parameter */
    register char *s;
    double v;
{
    register struct parameter *p = find_param(s);

    if(p->defined)
	yyerror("Parameter allready defined");
    p->defined = 1;
    p->value = v;
}

struct place *find_place(s)				/* locate a place */
    register char *s;
{
    register int    i;
    register struct place *p;

    for (i = n_PL, p = PL; i--; p++)			/* see if place is allready defined */
	if (!strcmp(s, p->name)) {
	    free(s);
	    return p;
	}

    if (n_PL >= max_PL) {			/* need more space */
	max_PL += 50 + max_PL / 2;
	if (PL)
	    PL = (struct place *) realloc(PL, sizeof(struct place) * max_PL);
	else
	    PL = (struct place *) malloc(sizeof(struct place) * max_PL);
    }

    p = &PL[n_PL++];
    p->name = s;
    p->defined = 0;
    p->input_ptr = 0;
    p->inhib_ptr = 0;
    p->result_ptr = 0;
    p->n_input = 0;
    p->n_output = 0;
    return p;
}

add_place (s)					/* got a new place */
    register char *s;
{
    register int    i;
    register struct place *p = find_place(s);

    if (p->defined) {
	yyerror("Place allready defined");
	return;
    }

    cur_PL = p;
    p->defined = 1;
    p->mdef = 0;
    p->marks = 0;
}

chg_marks (d)					/* change number of marks in a place */
    double d;
{
    register int i = d + 0.5;			/* round to an integer */

    if (i < 0 || cur_PL->mdef)
	yyerror ("illegal or multiple marking-specification");
    else {
	cur_PL->mdef = 1;
	cur_PL->marks = i;
    }
}

chg_p_marks (s)					/* change number of marks in a place */
    char *s;
{
    register struct parameter *p = find_param(s);	/* locate parameter */

    if (cur_PL->mdef || (p->used && p->type))
	yyerror ("Marking specification problem");
    else {
	if (p->used >= p->max_used) {
	    p->type = 0;
	    p->max_used += 10 + p->max_used / 2;
	    if (p->ind)
		p->ind = (int *) realloc (p->ind, p->max_used * sizeof(int));
	    else
		p->ind = (int *) malloc (p->max_used * sizeof(int));
	}
	cur_PL->mdef = 1;
	p->ind[p->used++] = cur_PL - PL;
    }
}

struct transition *find_transition(s)
    register char *s;
{
    register int    i;
    register struct transition *t;

    for (i = n_TR, t = TR; i--; t++)		/* see if transition is allready defined */
	if (!strcmp(s, t->name)) {
	    free(s);
	    return t;
	}

    if (n_TR >= max_TR) {			/* need more space */
	max_TR += 50 + max_TR / 2;
	if (TR)
	    TR = (struct transition *) realloc(TR, sizeof(struct transition) * max_TR);
	else
	    TR = (struct transition *) malloc(sizeof(struct transition) * max_TR);
    }

    t = &TR[n_TR++];
    t->name = s;				/* set up default transition structure */
    t->def = 0;
    t->ddef = 0;
    t->delay = 1.0;
    t->depdef = 0;
    t->depend = 1;
    t->n_input = 0;
    t->n_output = 0;
    t->n_inhib = 0;
    t->input_ptr = 0;
    t->output_ptr = 0;
    t->inhib_ptr = 0;

    return t;
}

add_transition (s)				/* got a new transition */
    char *s;
{
    register struct transition *t = find_transition(s);

    if (t->def)
	yyerror("Transition multiply defined");
    t->def = 1;
    cur_TR = t;
}

set_type (t)					/* define the transition type */
    int t;
{
    cur_TR->type = t;
}

chg_depend (d)					/* change the transition delay */
    double d;
{
    if (cur_TR->depdef || d < 0.0)
	yyerror("dependency definition problem");
    else {
	cur_TR->depdef = 1;
	cur_TR->depend = d;
	if (cur_TR->depend != 1)
	    yyerror("Sorry, only depency = 1 supported");
    }
}

chg_rate (d)					/* change the transition rate */
    double d;
{
    if (cur_TR->ddef || d <= EPS) 
	yyerror("rate definition problem");
    else {
	cur_TR->ddef = 1;
	cur_TR->delay = 1.0 / d;
    }
}

chg_p_rate (s)					/* change the transition rate parameter */
    char *s;
{
    register struct parameter *p = find_param(s);	/* locate parameter */

    if (cur_TR->ddef || (p->used && !p->type)) 
	yyerror("rate definition problem");
    else {
	if (p->used >= p->max_used) {
	    p->type = 1;
	    p->max_used += 10 + p->max_used / 2;
	    if (p->ind)
		p->ind = (int *) realloc (p->ind, p->max_used * sizeof(int));
	    else
		p->ind = (int *) malloc (p->max_used * sizeof(int));
	}
	p->ind[p->used++] = cur_TR - TR;
	cur_TR->ddef = 1;
    }
}

add_input (s)					/* add input to a transition */
    char *s;
{
    register struct list_item *t;
    register struct place *p = find_place(s);

    t = (struct list_item *) malloc(sizeof(struct list_item));
    t->ind = p - PL;
    t->next = cur_TR->input_ptr;
    cur_TR->input_ptr = t;
    cur_TR->n_input++;

    t = (struct list_item *) malloc(sizeof(struct list_item));
    t->ind = cur_TR - TR;
    t->next = p->input_ptr;
    p->input_ptr = t;
    p->n_input++;
}

add_output (s)					/* add output to a transition */
    char *s;
{
    register struct list_item *t;
    register struct place *p = find_place(s);

    t = (struct list_item *) malloc(sizeof(struct list_item));
    p->n_output++;
    t->ind = p - PL;
    t->next = cur_TR->output_ptr;
    cur_TR->output_ptr = t;
    cur_TR->n_output++;
}

add_inhib (s)					/* add inhib to a transition */
    char *s;
{
    register struct list_item *t;
    register struct place *p = find_place(s);

    t = (struct list_item *) malloc(sizeof(struct list_item));
    t->ind = p - PL;
    t->next = cur_TR->inhib_ptr;
    cur_TR->inhib_ptr = t;
    cur_TR->n_inhib++;

    t = (struct list_item *) malloc(sizeof(struct list_item));
    t->ind = cur_TR - TR;
    t->next = p->inhib_ptr;
    p->inhib_ptr = t;
}

check()						/* sanity check */
/*
 * Make sure that:
 *
 *  0. See that parameters are defined & used
 *  1. All places are defined
 *  2. All places have at least one in/out arc
 *  3. All transitions have at least one input and one output
 *     (I can't think of any sensible transition lacking either)
 *  4. Transition type matches given parameters
 *     (can't specify a reate for an immediate transition)
 */
{
    register int    i, j, k, *ip;
    register double d;
    register struct place *p;
    register struct transition *t;
    register struct parameter *pr;
    char            buf[80];

    if (n_PL < 1 || n_TR < 1)
	yyerror ("Need at least 1 place/transition");

    for(i = n_PR, pr = PR; i--; pr++) {		/* check parameters and insert values */
	if (!pr->defined) {
	    sprintf(buf, "Undefined parameter '%s'", pr->name);
	    yyerror(buf);
	}
	if (pr->type) {				/* rate-parameter */
	    d = pr->value;
	    if (d <= EPS)
		yyerror ("Negative rate parameter");
	    else {
		d = 1.0 / d;
		for (j = pr->used, ip = pr->ind; j--; ip++)
		    TR[*ip].delay = d;
	    }
	} else {				/* marking parameter */
	    d = pr->value;
	    if (d <= -EPS)
		yyerror ("Negative marking parameter");
	    else {
		k = d + 0.5;
		for (j = pr->used, ip = pr->ind; j--; ip++)
		    PL[*ip].marks = k;
	    }
	}
    }

    for (i = n_PL, p = PL; i--; p++) {
	if (!p->defined) {
	    sprintf(buf, "Undefined place '%s'", p->name);
	    yyerror(buf);
	}
	if (p->n_input < 1 || p->n_output < 1) {
	    sprintf(buf, "Warning: place '%s' has no input/output arc", p->name);
	    yyerror(buf);
	    error_cnt--;
	}
    }

    for (i = n_TR, t = TR; i--; t++) {
	if (t->n_input < 1 || t->n_output < 1) {
	    sprintf(buf, "Warning: transition '%s' has no input/output", t->name);
	    yyerror(buf);
	    error_cnt--;
	}
    }
}

build_place ()					/* produce place structures for simulator */
/*
 * Note: the corresponding structure definition is supplied in "gs.h"
 *
 */
{
    register int    i, j;
    printf("#define N_PLACES %d\n\n", n_PL);

    printf("struct place PLACE[N_PLACES] = {\n");
    for (i = 0; i < n_PL; i++) {
	if (i)
	    printf(",\n");
	printf("    {\"%s\", %d, %d, 0, 0.0, 0}", PL[i].name, PL[i].marks, PL[i].marks);
    }
    printf("\n};\n\n");
}

build_tfunc ()					/* produce fire-to macros structures for simulator */
/*
 * Note: the corresponding structure definition is supplied in "gs.h"
 *
 */
{
    register int    i, j;
    register struct list_item *t;

    for (i = 0; i < n_PL; i++) {
	printf("#define FIRE_TO%d ", i);
	if (PL[i].n_input < 1)
	    printf("0.0\n");
	else {
	    printf("(");
	    for(t = PL[i].input_ptr, j = 0; t; t = t->next)
		printf("TRANSITION[%d].fire_cnt%s", t->ind, (t->next) ? 
			((!(++j % 3)) ? " +\\\n    " : " + ") : ")\n");
	}
    }
    printf("\n");
}

build_trans()					/* build transition structure */
{
    register int    i, j;
    register struct list_item *t;
    
    printf("#define N_TRANSITIONS %d\n\n", n_TR);

    for (i = 0; i < n_TR; i++)
	printf ("void fire_T%d();\n", i);

    printf("\nstruct transition TRANSITION[N_TRANSITIONS] = {\n");
    for (i = 0; i < n_TR; i++) {
	if (i)
	    printf(",\n");

	j = 0;					/* count # of conditions that prevent fire */
	for (t = TR[i].input_ptr; t; t = t->next)
	    j += PL[t->ind].marks <= 0;
	for (t = TR[i].inhib_ptr; t; t = t->next)
	    j += PL[t->ind].marks > 0;
	
	printf("    {\"%s\", %d, %d, %d, %.12e, 0, 0, 0, 0, 0, 0.0, 0.0, 0.0, fire_T%d}",
	    TR[i].name, TR[i].type, j, j, TR[i].delay, i);
    }
    printf("\n};\n\n");
}

build_ffunc ()					/* build fire functions */
{
    register int    i, j;
    register struct list_item *t, *q;
    register struct result *r;

    static char *eq_type[5] = {"ENQUEUE_I_TR(", "enqueue_e_tr(&",  "enqueue_d_tr(&",  "enqueue_n_tr(&",  "enqueue_b_tr(&"};

    for (i = 0; i < n_PL; i++) {
	printf ("#define REMOVE_M_PL%d\tPLACE[%d].H[PLACE[%d].cur_mrk].prob += clock - PLACE[%d].t_last;\\\n", i, i, i, i);
	printf ("\t\t\tPLACE[%d].t_last = clock;\\\n", i);
	printf ("\t\t\tif (!(--PLACE[%d].cur_mrk)) {\\\n", i);
	for (t = PL[i].input_ptr; t; t = t->next)
	    printf("\t\t\t   if (!(TRANSITION[%d].ena_cnt++)) DEQUEUE_%c_TR(TRANSITION[%d]);\\\n", t->ind,
		(TR[t->ind].type == TTY_IMM) ? 'I' : 'X', t->ind);
	for (t = PL[i].inhib_ptr; t; t = t->next)
	    printf("\t\t\t   if (!(--TRANSITION[%d].ena_cnt)) %sTRANSITION[%d]);\\\n", t->ind, eq_type[TR[t->ind].type],  t->ind);
	printf ("\t\t\t}\n");

	printf ("#define ADD_M_PL%d\tPLACE[%d].H[PLACE[%d].cur_mrk].prob += clock - PLACE[%d].t_last;\\\n", i, i, i, i);
	printf ("\t\t\tPLACE[%d].t_last = clock;\\\n", i);
	printf ("\t\t\tif (!(PLACE[%d].cur_mrk++)) {\\\n", i);
	for (t = PL[i].inhib_ptr; t; t = t->next)
	    printf("\t\t\t   if (!(TRANSITION[%d].ena_cnt++)) DEQUEUE_%c_TR(TRANSITION[%d]);\\\n", t->ind,
		(TR[t->ind].type == TTY_IMM) ? 'I' : 'X', t->ind);
	for (t = PL[i].input_ptr; t; t = t->next)
	    printf("\t\t\t   if (!(--TRANSITION[%d].ena_cnt)) %sTRANSITION[%d]);\\\n", t->ind, eq_type[TR[t->ind].type],  t->ind);
	printf ("\t\t\t}\\\n");
	printf ("\t\t\tif (PLACE[%d].cur_mrk >= PLACE[%d].max_mrk) add_H_space (&PLACE[%d])\n\n", i, i, i);
    }

    for (i = 0; i < n_TR; i++) {
	for (j = n_RS, r = RS; j--; r++)
	    r->flg = 0;				/* default: assume result is not affected */

	printf("void fire_T%d()\n{\n", i);
	printf("    TRANSITION[%d].fire_cnt++;\n", i);
	for (t = TR[i].input_ptr; t; t = t->next){
	    printf("    REMOVE_M_PL%d;\n", t->ind);
	    for (q = PL[t->ind].result_ptr; q; q = q->next)
		RS[q->ind].flg = 1;		/* mark results that depend on this place */
	}
	for (t = TR[i].output_ptr; t; t = t->next) {
	    printf("    ADD_M_PL%d;\n", t->ind);
	    for (q = PL[t->ind].result_ptr; q; q = q->next)
		RS[q->ind].flg = 1;		/* mark results that depend on this place */
	}

	for (j = 0; j < n_RS; j++)
	    if (RS[j].flg)
		printf("    rs_func%d();\n", j);

	if (TR[i].type != TTY_IMM) {
	    printf("    if(!TRANSITION[%d].ena_cnt) {\n", i);
	    printf("\tDEQUEUE_X_TR(TRANSITION[%d]);\n", i);
	    printf("\t%sTRANSITION[%d]);\n", eq_type[TR[i].type], i);
	    printf("    }\n");
	}
	printf("}\n\n");
    }

    for (i = 0; i < n_PL; i++)
	printf("#undef REMOVE_M_PL%d\n#undef ADD_M_PL%d\n", i, i);
}

build_param()					/* dump the parameter structure */
{
    register int i, j, k, *ip, i_cnt, d_cnt;
    register struct parameter *p;

    printf("#define N_PARAMETERS %d\n\n", n_PR);

    if (!n_PR) {				/* no parameters, dump dummy to keep compiler happy */
	printf("struct parameter PARAMETER[1];\n");
	printf("double *d_param[1];\n");
	printf("unsigned short *i_param[1];\n");
	return;
    }

    printf("struct parameter PARAMETER[N_PARAMETERS] = {");
    i_cnt = 0;
    d_cnt = 0;
    for (i = n_PR, p =PR; i--; p++) {
	if (p->type) {
	    printf("\n    {\"%s\", 1, %d, %d, %.10e}", p->name, p->used, d_cnt, p->value);
	    d_cnt += p->used;
	} else {
	    printf("\n    {\"%s\", 0, %d, %d, %.10e}", p->name, p->used, i_cnt, p->value);
	    i_cnt += p->used;
	}
	if(i)
	    printf (",");
    }
    printf("\n};\n");

    if(d_cnt) {
	printf("double *d_param[%d] = {\n    ", d_cnt);
	for (i = n_PR, p =PR, k = 0; i--; p++)
	    if (p->type) {
		for (j = p->used, ip = p->ind; j--; ip++)
		    printf ("&(TRANSITION[%d].delay)%s", *ip,
			    (--d_cnt) ? ((++k & 3) ? ", " : ",\n    ") : "};\n");
	    }
    } else
	printf("double *d_param[1];\n");

    if(i_cnt) {
	printf("unsigned short *i_param[%d] = {\n    ", i_cnt);
	for (i = n_PR, p =PR, k = 0; i--; p++)
	    if (!p->type) {
		for (j = p->used, ip = p->ind; j--; ip++)
		    printf ("&(PLACE[%d].ini_mrk)%s", *ip,
			    (--i_cnt) ? ((++k & 3) ? ", " : ",\n    ") : "};\n");
	    }
    } else
	printf("unsigned short *i_param[1];\n\n");
}


add_result (s, b)					/* got a new result def */
    register char *s, *b;
{
    register int    i;
    register struct result *r;

    for (i = n_RS, r = RS; i--; r++)		/* see if result is allready defined */
	if (!strcmp(s, r->name)) {
	    free(s);
	    free(b);
	    yyerror("Result multiply defined");
	    return;
	}

    if (n_RS >= max_RS) {			/* need more space */
	max_RS += 50 + max_RS / 2;
	if (RS)
	    RS = (struct result *) realloc(RS, sizeof(struct result) * max_RS);
	else
	    RS = (struct result *) malloc(sizeof(struct result) * max_RS);
    }

    r = &RS[n_RS++];
    r->name = s;				/* set up default transition structure */
    r->body = b;
    r->type = result_type;
}

char *concat (s1, s2)
    char *s1, *s2;
{
    register char *s;
    s = (char *) malloc (6 + strlen(s1) + strlen(s2));
    strcpy(s, s1);
    strcat(s, s2);
    free(s1);
    free(s2);
    return s;
}

ini_result()					/* prepare for a result definition */
{
    result_type = 0;
}

char *do_prob(d, s)
    char *s, *d;
{
    char *s1;

    if (result_type && result_type != 1)
	yyerror("Dimension mismatch");
    result_type = 1;

    s1 = (char *) malloc(strlen(s) + 50);
    sprintf(s1, "    if (%s) t %c= %s;\n", s, it_opr, d);
    free(s);
    free(d);
    return s1;
}

char *do_exp(d, s)
    char *s, *d;
{
    char *s1;

    if (result_type && result_type != 1)
	yyerror("Dimension mismatch");
    result_type = 1;

    s1 = (char *) malloc(strlen(s) + 50);
    sprintf(s1, "    t %c= %s * %s;\n", it_opr, d, s);
    free(s);
    free(d);
    return s1;
}

char *do_cexp(d, s, c)
    char *s, *c, *d;
{
    char *s1;

    if (result_type && result_type != 1)
	yyerror("Dimension mismatch");
    result_type = 1;

    s1 = (char *) malloc(strlen(s) + 50 + strlen(c));
    sprintf(s1, "    if (%s) t %c= %s * %s;\n", c, it_opr, d, s);
    free(s);
    free(c);
    free(d);
    return s1;
}

char *do_dwell(c, s)
    char *c, *s;
{
    char *s1;
    int  i;

    if (result_type && result_type != 2)
	yyerror("Dimension mismatch");
    result_type = 2;

    s1 = (char *) malloc (100);
    i = find_place(s) - PL;

					/* Note: that 10^-60 prevents div by 0 and nothing else */
    sprintf(s1, "    t %c= %s * acc_dwt(&PLACE[%d]) / (1e-60 + FIRE_TO%d);\n", it_opr, c, i, i);
    free(c);
    return s1;
}

char *do_fire(c, s)
    char *c, *s;
{
    char *s1;
    int  i;

    if (result_type && result_type != 3)
	yyerror("Dimension mismatch");
    result_type = 3;

    s1 = (char *) malloc (100);
    i = find_transition(s) - TR;
    sprintf(s1, "    t %c= %s * TRANSITION[%d].fire_cnt;\n", it_opr, c, i);
    free(c);
    return s1;
}

char *get_val (d)
    double d;
{
    char *s;
    double rint();

    s = (char *) malloc(50);
    if (d == rint(d))
	sprintf(s, "%.1f", d);
    else
	sprintf(s, "%.10e", d);

    return s;
}

char *get_para(s)
    char *s;
{
    char *s1;
    s1 = (char *) malloc(50);
    sprintf(s1, "PARAMETER[%d].value", find_param(s) - PR);
    return s1;
}

char *get_place (s)
    char *s;
{
    register struct place *p = find_place(s);
    register struct list_item *t = (struct list_item *) malloc (sizeof (struct list_item));

    char *s1 = (char *) malloc (30);
    sprintf (s1, "PLACE[%d].cur_mrk", (int) (p - PL));
    t->ind = n_RS;
    t->next = p->result_ptr;
    p->result_ptr = t;
    return s1;
}

char *c_negate (s)
    char *s;
{
    char *s1 = (char *) malloc (strlen(s) + 5);
    sprintf (s1, "!(%s)", s);
    free (s);
    return s1;
}

char *do_paren (s)
    char *s;
{
    char *s1 = (char *) malloc (strlen(s) + 5);
    sprintf (s1, "(%s)", s);
    free (s);
    return s1;
}

char *do_and (s1, s2)
    char *s1, *s2;
{
    char *s = (char *) malloc (strlen(s1) + 10 + strlen(s2));
    sprintf (s, "(%s && %s)", s1, s2);
    free (s1);
    free (s2);
    return s;
}

char *do_or (s1, s2)
    char *s1, *s2;
{
    char *s = (char *) malloc (strlen(s1) + 10 + strlen(s2));
    sprintf (s, "(%s || %s)", s1, s2);
    free (s1);
    free (s2);
    return s;
}

char *do_cmp (s1, s2, d)
    char *s1, *s2;
    double d;
{
    char *s = (char *) malloc (strlen(s1) + 20 + strlen(s2));
    sprintf (s, "(%s %s %d)", s1, s2, (int) (d + 0.5));
    free (s1);
    free (s2);
    return s;
}

build_rslts ()					/* build result functions */
{
    register struct result *r;
    register int i;

    printf("#define N_RESULTS %d\n\n", n_RS);

    if (n_RS < 1) {
	printf("struct result RESULT[1];\n\n");	/* dummy for compilier happiness */
	return;
    }

    for (i = n_RS; i--;)
	printf ("void rs_func%d();\n", i);

    printf("struct result RESULT[N_RESULTS] = {\n");
    for (i = 0,  r = RS; i < n_RS; r++, i++)
	printf ("    {\"%s\", %d, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, rs_func%d}%s",
		r->name, r->type - 1, i, (i + 1 >= n_RS) ? "\n};\n\n" : ",\n");

    for (i = 0,  r = RS; i < n_RS; r++, i++) {
	if (r->type == 1) {
	    printf ("void rs_func%d()\n{\n    register double t = 0.0;\n%s", i, r->body);
	    printf ("    RESULT[%d].acc += RESULT[%d].cur * (clock - RESULT[%d].t_last);\n", i, i, i);
	    printf ("    RESULT[%d].cur = t;\n", i);
	    printf ("    RESULT[%d].t_last = clock;\n}\n\n", i);
	} else if (r->type == 2) {
	    printf ("void rs_func%d()\n{\n    register double t = 0.0;\n%s", i, r->body);
	    printf ("    RESULT[%d].sum += t;\n", i);
	    printf ("    RESULT[%d].sum_sq += t * t;\n}\n\n", i);
	} else if (r->type == 3) {
	    printf ("void rs_func%d()\n{\n    register double t = 0.0;\n%s", i, r->body);
	    printf ("    RESULT[%d].sum += t /= clock;\n", i);
	    printf ("    RESULT[%d].sum_sq += t * t;\n}\n\n", i);
	}
    }
}
