/*
 * Handle expansion of meta charaters
 *
 * Copyright 1999 Bjrn Ekwall <bj0rn@blox.se>
 *
 * "kernelversion" idea from the Debian release via:
 *	Wichert Akkerman <wakkerma@cs.leidenuniv.nl>
 *
 * Use wordexp(): idea from Tim Waugh <tim@cyberelk.demon.co.uk>
 *
 * This file is part of the Linux modutils.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <ctype.h>
#if HAVE_WORDEXP
#include <wordexp.h>
#elif HAVE_GLOB
#include <glob.h>
#endif
#include "util.h"

static int glob_it(char *pt, GLOB_LIST *g)
{
#if HAVE_WORDEXP
	wordexp_t w;

	memset(&w, 0, sizeof(w));
	if (wordexp(pt, &w, WRDE_UNDEF)) {
		/*
		error("wordexp %s failed", pt);
		*/
		return -1;
	}
	/* else */
	g->pathc = w.we_wordc;
	g->pathv = w.we_wordv;

	return 0;
#elif HAVE_GLOB /* but not wordexp */
	glob_t w;

	memset(&w, 0, sizeof(w));
	if (glob(pt, GLOB_NOSORT, NULL, &w)) {
		/*
		error("glob %s failed", pt);
		*/
		return -1;
	}
	/* else */
	if (w.gl_pathc && strpbrk(w.gl_pathv[0], SHELL_META)) {
		globfree(&w);
		return -1;
	}
	g->pathc = w.gl_pathc;
	g->pathv = w.gl_pathv;

	return 0;
#else /* Neither wordexp nor glob */
	return -1;
#endif
}

/*
 * Expand the string (including meta-character) to a list of matches
 *
 * Return 0 if OK else -1
 */
int meta_expand(char *pt, GLOB_LIST *g, char *base_dir, char *version)
{
	char *p;
	char tmpline[PATH_MAX + 1];

	g->pathc = 0;
	g->pathv = NULL;

	/*
	 * Take care of version dependent expansions
	 * Needed for forced version handling
	 */
	if ((p = strchr(pt, '`')) != NULL) {
		do {
			char tmp[PATH_MAX + 1];
			char *s;

			for (s = p + 1; isspace(*s); ++s)
				;

			if (strncmp(s, "uname -r", 8) == 0) {
				while (*s && (*s != '`'))
					++s;
				if (*s == '`') {
					*p = '\0';
					sprintf(tmp, "%s%s%s",
						     pt,
						     version,
						     s + 1);
				}
				strcpy(tmpline, tmp);
				pt = tmpline;
			} else if (strncmp(s, "kernelversion", 13) == 0) {
				while (*s && (*s != '`'))
					++s;
				if (*s == '`') {
					int n;
					char *k;

					*p = '\0';
					for (n = 0, k = version; *k; ++k) {
						if (*k == '.' && ++n == 2)
							break;
					}
					sprintf(tmp, "%s%.*s%s",
						     pt,
						     k - version,
						     version,
						     s + 1);
					strcpy(tmpline, tmp);
					pt = tmpline;
				}
			} else
				break;
		} while ((p = strchr(pt, '`')) != NULL);
	}
	
	/*
	 * Any remaining meta-chars?
	 */
	if (!strpbrk(pt, SHELL_META)) {
		char tmp[PATH_MAX + 1];

		/* no meta-chars, no back quotes */
		sprintf(tmp, "%s%s", (base_dir ? base_dir : ""), pt);

		if ((p = strtok(tmp, " \t\n")) != NULL) {
			while (p) {
				g->pathv = (char **)xrealloc(g->pathv,
					   (g->pathc + 2) * sizeof(char *));
				g->pathv[g->pathc++] = xstrdup(p);
				p = strtok(NULL, " \t\n");
			}
		}
	} else { /* meta-chars, no back quotes */
		FILE *fin;
		int len = 0;
		char cmd[PATH_MAX + 11];
		char *line = NULL;
		char *tmp;

		if (glob_it(pt, g) == 0)
			return 0;

		tmpline[PATH_MAX] = '\0';

		sprintf(cmd, "/bin/echo %s%s", (base_dir ? base_dir : ""), pt);
		if ((fin = popen(cmd, "r")) == NULL) {
			error("Can't execute: %s", cmd);
			return -1;
		}
		/* else */

		while (fgets(tmpline, PATH_MAX, fin) != NULL) {
			int l = strlen(tmpline);

			line = (char *)xrealloc(line, len + l + 1);
			line[len] = '\0';
			strcat(line + len, tmpline);
			len += l;
		}
		pclose(fin);

		if ((tmp = strtok(line, " \t\n")) != NULL) {
			do {
				g->pathv = (char **)xrealloc(g->pathv,
					   (g->pathc + 2) * sizeof(char *));
				g->pathv[g->pathc++] = xstrdup(tmp);
			} while ((tmp = strtok(NULL, " \t\n")));
		}
		free(line);
	}

	if (g->pathc)
		g->pathv[g->pathc] = NULL;

	return 0;
}
