/*    imaginary.sql.iMsqlConnection
 *    from the Imaginary Java Class Library
 *    Copyright (c) 1996 George Reese
 *    created by George Reese (borg@imaginary.com) 960311
 *    an mSQL implementation of the JDBC specification Connection interface
 */

package imaginary.sql;

import java.sql.CallableStatement;
import java.sql.DatabaseMetaData;
import java.sql.Driver;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import imaginary.sql.msql.Msql;
import imaginary.sql.msql.MsqlException;
import imaginary.sql.msql.MsqlResult;

public class iMsqlConnection implements java.sql.Connection {
  /**
   * Darryl Collins' Msql object from his mSQLJava package
   */
  private Msql msql = null;
  /**
   * The JDBC driver for this connection
   */
  private Driver driver;
  /**
   * The URL for this connection
   */
  private String url;
  /**
   * The user name used to connect to the database
   */
  private String user;
  /**
   * Catalog string that has no meaning to mSQL
   */
  private String catalog;
  /**
   * Transaction isolation level, meaningless to mSQL
   */
  private int isolation;
  
  /**
   * Constructs a new JDBC draft specification connection object for the
   * mSQL database.  Creates an instance of Darryl Collins' mSQL class
   * and uses it to connect to the named database on the named host.
   * @exception SQLException raised in the event of connection failure
   * @param host the IP address of the host on which the mSQL server resides
   * @param root true if this is a root connection, false if otherwise
   * @param database the database to which a connection should be made
   * @param u the URL used to connect to this database
   * @param d the driver that instantiated this connection
   */
  public iMsqlConnection(String host, boolean root, String user,
			 String database, String u, Driver d)
       throws SQLException {
    msql = new Msql();
    if( user == null || user == "" ) user = "nobody";
    try {
      msql.Connect(host, user, root);
      msql.SelectDB(database);
      
    }
    catch( MsqlException e ) {
      msql = null;
      throw new SQLException("mSQL exception: " + e.getMessage());
    }
   url = u;
   this.user = user;
   driver = d;
  }
  
  /**
   * JDBC draft specification method for returning a SQL statement
   * object.
   * @see java.sql.Connection#createStatement
   * @exception SQLException thrown in the event the creation fails
   * @return a new statement object
   */
  public Statement createStatement() throws SQLException {
    return new iMsqlStatement(this);
  }

  /**
   * JDBC draft specification method for returning an SQL pre-compiled
   * statement.  Note that mSQL does not support such things, so this
   * thing just throws an exception.
   * @see java.sql.Connection#prepareStatement
   * @exception SQLException gets thrown any time this is called
   */
  public PreparedStatement prepareStatement(String sql)
       throws SQLException {
    throw new SQLException("mSQL does not support prepared statements.");
  }

  /**
   * JDBC draft specification method for creating a stored procedure call.
   * Note that mSQL does not support stored procedures, so this method
   * throws an exception.
   * @see java.sql.Connection#prepareCall
   * @exception SQLException gets thrown any time this is called
   */
  public CallableStatement prepareCall(String sql)
       throws SQLException {
    throw new SQLException("mSQL does not support stored procedures.");
  }

  /**
   * JDBC draft specification method for converting generic SQL into
   * mSQL specific SQL.  Such a task would be mind-boggling, given the fact
   * that mSQL supports such a small subset of ANSI SQL.  This method therefore
   * simply returns the original and hopes for the best.
   * @see java.sql.Connection#nativeSQL
   * @param sql the query which just gets thrown right back out
   * @return the mSQL SQL string
   */
  public String nativeSQL(String sql) throws SQLException {
    return sql;
  }

  /**
   * JDBC draft specification method for setting auto-commit.  Since
   * mSQL has no transaction management, this connection object acts
   * exactly as if auto-commit were set.  So this method does nothing.
   * @see java.sql.Connection#setAutoCommit
   * @param b this does nothing
   */
  public void setAutoCommit(boolean b) throws SQLException {
  }

  /**
   * JDBC draft specification method for committing a transaction.  Since
   * mSQL has no support for commits or rollbacks, this method does nothing.
   * NOTE: Should I throw an exception here?
   * @see java.sql.Connection#commit
   */
  public void commit() throws SQLException {
  }

  /**
   * JDBC draft specification method for rolling back a transaction.  Since
   * mSQL has no support for rollbacks, this method throws an exception.
   * @see java.sql.Connection#rollback
   * @exception SQLException gets thrown if this ever gets called
   */
  public void rollback() throws SQLException {
    throw new SQLException("mSQL exception: mSQL does not support rollbacks.");
  }

  /**
   * JDBC draft specification method for closing the database connection.
   * @see java.sql.Connection#close
   * @exception SQLException thrown in the event of an MsqlException
   */
  public void close() throws SQLException {
    try {
      msql.Close();
      msql = null;
    }
    catch( MsqlException e ) {
      throw new SQLException("mSQL exception: " + e.getMessage());
    }
  }

  /**
   * JDBC draft specification method for letting others know the connection
   * status.
   * @see java.sql.Connection#isClosed
   * @exception SQLException who knows why Sun thought this needed an exception
   * @return true if the connection is closed, false otherwise
   */
  public boolean isClosed() throws SQLException {
    return (msql == null);
  }

  /**
   * JDBC draft specification method to return information about
   * the database.
   * @see java.sql.Connection#getMetData
   * @exception SQLException thrown if an error occurs reading meta-data
   * @return a DatabaseMetaData object with database info
   */
  public DatabaseMetaData getMetaData()
       throws SQLException {
    return null; //Worry about this later
  }

  /**
   * JDBC draft specification method to put the connection in read-only mode.
   * mSQL does not support read-only mode, so this method does nothing.
   * NOTE: Should an exceptionbe thrown here?
   * @see java.sql.Connection#setReadOnly
   * @param b dummy parameter than has no meaning to mSQL
   */
  public void setReadOnly(boolean b) throws SQLException {
  }

  /**
   * JDBC draft specification method to return the read-only status of the
   * connection.  Since mSQL has no such status, this always returns false.
   * @see java.sql.Connection#isReadOnly
   * @return always false
   */
  public boolean isReadOnly() throws SQLException {
    return false;
  }

  /**
   * JDBC draft specification method to select a sub-space of the target
   * database.  It basically has no menaing to mSQL.
   * @see java.sql.Connection#setCatalog
   * @param str the catalog
   */
  public void setCatalog(String str) throws SQLException {
    catalog = str;
  }

  /**
   * JDBC draft specification method to return the catalog.  This has no
   * meaning to mSQL.
   * @see java.sql.Connection#getCatalog
   * @return the catalog name
   */
  public String getCatalog() throws SQLException {
    return catalog;
  }

  /**
   * JDBC draft specification method for setting the transaction isolation
   * level for the connection.  This has no meaning to mSQL.
   * @see java.sql.Connection#setTransactionIsolation
   * @param x the isolation level
   */
  public void setTransactionIsolation(int x)
       throws SQLException {
    isolation = x;
  }

  /**
   * JDBC draft specification method for retrieving the transaction
   * isolation level for the connection.  This information has no meaning
   * to mSQL.
   * @see java.sql.Connection#getTransactionIsolation
   * @return the transaction isolation level
   */
  public int getTransactionIsolation() throws SQLException {
    return isolation;
  }

  /**
   * JDBC draft specification method for disabling of closing of statements
   * and result sets.  This has no meaning to mSQL.
   * @see java.sql.Connection#disableAutoClose
   */
  public void disableAutoClose() throws SQLException {
  }

  /**
   * JDBC draft specification method for retrieving a chain of warnings
   * related to the connection.
   * @see java.sql.Connection#getWarnings
   * @return the chain of warnings for this connection
   */
  public SQLWarning getWarnings() throws SQLException {
    return null;
  }

  /**
   * JDBC draft specification method for clearing the warning chain.
   * @see java.sql.Connection#clearWarnings
   */
  public void clearWarnings() throws SQLException {
  }

  /**
   * Executes an mSQL statement.
   * @param sql the statement to be executed
   * @return MsqlResult with the results of the statement
   */
  public MsqlResult executeMsql(String sql)
       throws SQLException {
    MsqlResult result;

    try {
      result = msql.Query(sql);
    }
    catch( MsqlException e ) {
      throw new SQLException("mSQL exception: " + e.getMessage());
    }
    return result;
  }
  
  /**
   * Gives the URL used to connect to the database.
   * @return the URL
   */
  public final String getURL() {
    return url;
  }

  /**
   * Returns the user name for the connection.
   * @return the user name used to connect to the database
   */
  public final String getUser() {
    return user;
  }
}
