/*    imaginary.sql.iMsqlMetaResult
 *    from the Imaginary Java Class Library
 *    Copyright (c) 1996 George Reese
 *    created by George Reese (borg@imaginary.com) 960313
 *    mSQL implementation of the JDBC ResultSetMetaData interface
 */

package imaginary.sql;

import java.sql.SQLException;
import java.sql.Types;
import imaginary.sql.msql.MsqlFieldDesc;
import imaginary.sql.msql.MsqlResult;

/**
 * This class provides the mSQL implementation of the JDBC ResultSetMetaData
 * interface.
 *
 * @see java.sql.ResultSetMetaData
 * @version 1.3, 96/04/20
 * @author George Reese
 */
public class iMsqlMetaResult implements java.sql.ResultSetMetaData {
  /**
   * The result this meta-data exists for.
   */
  private MsqlResult msql;

  /**
   * Constructs a new instance of the iMsqlMetaResult class for the
   * named MsqlResult object.
   * @param result the MsqlResult object
   */
  public iMsqlMetaResult(MsqlResult result) {
    msql = result;
  }
  
  /**
   * Returns the number of columns in this result set.
   * @see java.sqlResultSetMetaData#getColumnCount
   * @return number of columns
   */
  public int getColumnCount() throws SQLException {
    return msql.NumFields();
  }

  /**
   * I have no idea what this method does.
   * @see java.sql.ResultSetMetaData#isAutoIncrement
   * @return ???
   */
  public boolean isAutoIncrement(int column) throws SQLException { 
    return false;
  }

  /**
   * Tells anything wanting to know that mSQL is case sensitive on
   * column names.
   * @see java.sql.ResultSetMetaData#isCaseSensitive
   * @return true, meaning mSQL is case sensitive
   */
  public boolean isCaseSensitive(int column) throws SQLException {
    return true;
  }

  /**
   * I have no clue what this means.
   * @see java.sql.ResultSetMetaData#isSearchable
   * @return ???
   */
  public boolean isSearchable(int column) throws SQLException {
    return true;
  }

  /**
   * Since there is no way in an mSQL database to do currency as such,
   * this will always return false.
   * @see java.sql.ResultSetMetaData#isCurrency
   * @return false
   */
  public boolean isCurrency(int column) throws SQLException {
    return false;
  }

  /**
   * Determines if the column in question is nullable.
   * @see java.sql.ResultSetMetaData#isNullable
   * @exception SQLException thrown if the column is out of range
   * @return true if the column can accept null values, otherwise false
   */
  public int isNullable(int column) throws SQLException{
    MsqlFieldDesc f[];

    f = msql.ListFields();
    try {
      return ((f[column-1].NonNull()) ? columnNoNulls : columnNullable);
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * Since all mSQL integer types are signed, this returns true.
   * @see java.sql.ResultSetMetaData#isSigned
   * @return true
   */
  public boolean isSigned(int column) throws SQLException {
    return true;
  }

  /**
   * Gives the display size for this column.
   * @see java.sql.ResultSetMetaData#getColumnDisplaySize
   */
  public int getColumnDisplaySize(int column) throws SQLException {
    MsqlFieldDesc f[];

    f = msql.ListFields();
    try {
      return f[column-1].FieldLength();
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * This returns table_name.column_name.
   * @see java.sql.ResultSetMetaData#getColumnLabel
   * @param column the column whose label is wanted
   * @return the display value for the field
   */
  public String getColumnLabel(int column)
       throws SQLException {
    return (getTableName(column) + "." + getColumnName(column));
  }

  /**
   * The name of the requested column.
   * @see java.sql.ResultSetMetaData#getColumnName
   * @exception thrown when the column desired is bad
   * @param column the column whose name is wanted
   * @return the name of the requested column
   */
  public String getColumnName(int column)
       throws SQLException {
    MsqlFieldDesc f[];

    f = msql.ListFields();
    try {
      return f[column-1].FieldName();
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * This does not apply to mSQL.
   */
  public String getSchemaName(int column)
       throws SQLException {
    return "";
  }

  /**
   * Pizza pizza on this one.
   */
  public int getPrecision(int column)
       throws SQLException {
    throw new SQLException("mSQL has no clue what the precision is.");
  }

  /**
   * More pizza with wombles this time
   */
  public int getScale(int column) throws SQLException {
    throw new SQLException("mSQL has no clue what the scale is.");
  }

  /**
   * Gives the name of the table to which this column belongs.
   * @see java.sql.ResultSetMetaData#getTableName
   * @param column the column of the field this information is needed for
   * @return the table name
   */
  public String getTableName(int column)
       throws SQLException {
    MsqlFieldDesc f[];

    f = msql.ListFields();
    try {
      return f[column-1].TableName();
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
  }

  /**
   * mSQL does not support this
   */
  public String getCatalogName(int column)
       throws SQLException {
    throw new SQLException("mSQL does not support catalogging.");
  }

  /**
   * Gives the column type using the types in java.sql.Types.
   * @see java.sqlTypes
   * @see java.sql.ResultSetMetaData#getColumnType
   * @exception SQLException thrown for any number of reasons
   * @param column the column type information is needed on
   * @return the type as listed in java.sql.Types
   */
  public int getColumnType(int column) throws SQLException {
    MsqlFieldDesc f[];

    f = msql.ListFields();
    try {
      switch(f[column-1].FieldType()) {
      case 1:
	return Types.INTEGER;

      case 2:
	return Types.CHAR;

      case 3:
	return Types.REAL;

      case 5:
	return Types.NULL;
      }
    }
    catch( Exception e ) {
      throw new SQLException(e.getMessage());
    }
    throw new SQLException("Unknown data type.");
  }

  /**
   * Gives the column type as a string.
   * @see java.sql.ResultSetMetaData#getColumnTypeName
   * @exception SQLException thrown randomly for fun
   * @param column the column for which the type name is wanted
   * @return the name of the column type
   */
  public String getColumnTypeName(int column)
       throws SQLException {
    switch(getColumnType(column)) {
    case Types.INTEGER:
      return "INT";

    case Types.CHAR:
      return "CHAR";

    case Types.REAL:
      return "REAL";

    default:
      return "NULL";
    }
  }

  /* These last methods have no bearing on mSQL */
  public boolean isReadOnly(int column) throws SQLException {
    return false;
  }

  public boolean isWritable(int column) throws SQLException {
    return true;
  }

  public boolean isDefinitelyWritable(int column) throws SQLException {
    return true;
  }
}
    


   
